/*
 * Copyright (C) 2014 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include "glue.h"

static int
glue_nibble(char c)
{
	if ('0' <= c && c <= '9') {
		return c - '0';
	} else if ('A' <= c && c <= 'F') {
		return c - 'A' + 10;
	} else if ('a' <= c && c <= 'f') {
		return c - 'a' + 10;
	} else {
		return -1;
	}
}

static int
glue_hex(int fd, unsigned char *cp)
{
	char c1;
	char c0;
	int val1;
	int val0;
	int ret;

	ret = read(fd, &c1, sizeof(c1));
	if (ret != 1) return -1;
	ret = read(fd, &c0, sizeof(c0));
	if (ret != 1) return -1;

	val1 = glue_nibble(c1);
	if (val1 < 0) return -1;
	val0 = glue_nibble(c0);
	if (val0 < 0) return -1;

	*cp = (val1 << 4) | (val0 << 0);
	return 1;
}

static int
glue_hex_read(const char *path, void *_rom, int romsize)
{
	unsigned char *rom = _rom;
	int fd;
	int len;
	unsigned int start;
	int ret;

	fd = open(path, O_RDONLY);
	if (fd < 0) return -1;

	start = 0;
	len = 0;

	for (;;) {
		char s;
		unsigned char count;
		unsigned int offset;
		unsigned char type;
		unsigned char byte[256];
		unsigned char checksum;
		char eoln;
		unsigned int addr;
		int x;

		/*
		 * Read record.
		 */
		/* Read ':'. */
		ret = read(fd, &s, 1);
		if (ret == 0) {
			/* End record missing. */
	fail:		;
			len = -1;
			break;
		}
		if (s != ':') {
			/* Bad start code. */
			goto fail;
		}

		/* Read number of bytes. */
		ret = glue_hex(fd, &count);
		if (ret <= 0) {
			goto fail;
		}

		/* Read address. */
		offset = 0;
		for (x = 0; x < 2; x++) {
			unsigned char tmp;

			ret = glue_hex(fd, &tmp);
			if (ret <= 0) {
				goto fail;
			}
			offset <<= 8;
			offset |= tmp;
		}

		/* Read record type. */
		ret = glue_hex(fd, &type);
		if (ret <= 0) {
			goto fail;
		}

		/* Read bytes. */
		for (x = 0; x < count; x++) {
			ret = glue_hex(fd, &byte[x]);
			if (ret <= 0) {
				goto fail;
			}
		}

		/* Read checksum. */
		ret = glue_hex(fd, &checksum);
		if (ret <= 0) {
			goto fail;
		}

		/* Read EOLN. */
		ret = read(fd, &eoln, 1);
		if (ret <= 0) {
			goto fail;
		}
		if (eoln == '\r') {
			ret = read(fd, &eoln, 1);
			if (ret <= 0) {
				goto fail;
			}
		}
		if (eoln != '\n') {
			goto fail;
		}

		/*
		 * Check record.
		 */
		/* FIXME */

		/*
		 * Process record.
		 */
		switch (type) {
		case 0:
			/* Data */
			/* Write record info flash. */
			addr = start + offset;

			addr &= romsize - 1;

			if (romsize < addr + count) {
				goto fail;
			}

			memcpy(rom + addr, byte, count);

			if (len < addr + count) {
				len = addr + count;
			}
			break;
		case 1:
			/* EOF */
			goto done;
		case 2:
			/* Extended Segment Address */
			if (count != 2) {
				goto fail;
			}
			start = (byte[0] << 8) | (byte[1] << 0);
			start <<= 4;
			break;
		case 3:
			/* Start Segment Address */
			if (count != 2) {
				goto fail;
			}
			/* Not used... */
			break;
		case 4:
			/* Extended Linear Address */
			if (count != 2) {
				goto fail;
			}
			start = (byte[0] << 8) | (byte[1] << 0);
			start <<= 16;
			break;
		case 5:
			/* Start Linear Address */
			if (count != 4) {
				goto fail;
			}
			/* Not used... */
			break;
		default:
			goto fail;
		}
	}

done:	;
	ret = close(fd);
	assert(0 <= fd);

	return len;
}

static int
glue_s19_read(const char *path, void *_rom, int romsize)
{
	unsigned char *rom = _rom;
	int fd;
	int len;
	int ret;

	fd = open(path, O_RDONLY);
	if (fd < 0) return -1;

	len = 0;

	for (;;) {
		char s;
		char type;
		unsigned char count;
		unsigned int alen;
		unsigned int addr;
		unsigned char byte[256];
		unsigned char checksum;
		char eoln;
		int x;

		/*
		 * Read line.
		 */
		/* Read 'S'. */
		ret = read(fd, &s, 1);
		if (ret == 0) {
			break;
		}
		if (ret < 0) {
	fail:		;
			len = -1;
			break;
		}
		if (s != 'S') {
			goto fail;
		}

		/* Read record type. */
		ret = read(fd, &type, 1);
		if (ret <= 0) {
			goto fail;
		}
		if (type < '0' || '9' < type || type == '4' || type == '6') {
			goto fail;
		}

		/* Read length. */
		ret = glue_hex(fd, &count);
		if (ret <= 0) {
			goto fail;
		}

		switch (type) {
		case '0':
		case '1':
		case '5':
		case '9': alen = 2; break;

		case '2':
		case '8': alen = 3; break;

		case '3':
		case '7': alen = 4; break;

		default:
			assert(0); /* Cannot happen. */
		}
		count -= alen;
		count -= 1;

		/* Read address. */
		addr = 0;
		for (x = 0; x < alen; x++) {
			unsigned char tmp;

			ret = glue_hex(fd, &tmp);
			if (ret <= 0) {
				goto fail;
			}
			addr <<= 8;
			addr |= tmp;
		}
		/* Read bytes. */
		for (x = 0; x < count; x++) {
			ret = glue_hex(fd, &byte[x]);
			if (ret <= 0) {
				goto fail;
			}
		}

		/* Read checksum. */
		ret = glue_hex(fd, &checksum);
		if (ret <= 0) {
			goto fail;
		}

		/* Read EOLN. */
		ret = read(fd, &eoln, 1);
		if (ret <= 0) {
			goto fail;
		}
		if (eoln == '\r') {
			ret = read(fd, &eoln, 1);
			if (ret <= 0) {
				goto fail;
			}
		}
		if (eoln != '\n') {
			goto fail;
		}

		/*
		 * Write line into flash.
		 */
		if (type == '1'
		 || type == '2'
		 || type == '3') {
			if (romsize < addr + count) {
				goto fail;
			}

			memcpy(rom + addr, byte, count);

			if (len < addr + count) {
				len = addr + count;
			}
		}
	}

	ret = close(fd);
	assert(0 <= ret);

	return len;
}

static int
glue_bin_read(const char *path, void *rom, int romsize)
{
	int fd;
	int len;
	int ret;

	fd = open(path, O_RDONLY);
	if (fd < 0) return -1;

	len = read(fd, rom, romsize);
	
	ret = close(fd);
	assert(0 <= ret);

	return len;
}

int
glue_rom_read(const char *path, void *_rom, int romsize)
{
	int ret;

	memset(_rom, 0xff, romsize);

	ret = glue_hex_read(path, _rom, romsize);
	if (0 < ret) return ret;
	ret = glue_s19_read(path, _rom, romsize);
	if (0 < ret) return ret;
	ret = glue_bin_read(path, _rom, romsize);
	if (0 < ret) return ret;

	return -1;
}
