/*
  Copyright (C) 2001-2004 Stephane Magnenat & Luc-Olivier de Charrière
  for any question or comment contact us at <stephane at magnenat dot net> or <NuageBleu at gmail dot com>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#ifndef FORMATABLESTRING_H
#define FORMATABLESTRING_H

#include <string>
#include <sstream>

namespace GAGCore {
	/*!
	* string that can be used for argument substitution.
	* Example :
	* FormatableString fs("Hello %0");
	* cout << fs.arg("World");
	*/
	class FormatableString : public std::string
	{
		private:
			/*!
			* Next argument to be replaced.
			*/
			int argLevel;
		
			/*!
			* Replace the next argument by replacement.
			*/
			void proceedReplace(const std::string &replacement);
			
		public:
			
			FormatableString() : std::string(), argLevel(0) { }
			/*!
			* Creates a new FormatableString with format string set to s.
			* \param s A string with indicators for argument substitution.
			* Each indicator is the % symbol followed by a number. The number
			* is the index of the corresponding argument (starting at %0).
			*/
			FormatableString(const std::string &s)
		: std::string(s), argLevel(0) { }
			
			/*!
			* Replace the next arg by an int value.
			* \param value Value used to replace the current argument.
			* \param fieldWidth min width of the displayed number
			* \param base Radix of the number (8, 10 or 16)
			* \param fillChar Character used to pad the number to reach fieldWidth
			* \see arg(const T& value)
			*/
			FormatableString &arg(int value, int fieldWidth = 0, int base = 10, char fillChar = ' ');
			
			/*!
			* Replace the next arg by an int value.
			* \param value Value used to replace the current argument.
			* \param fieldWidth min width of the displayed number
			* \param base Radix of the number (8, 10 or 16)
			* \param fillChar Character used to pad the number to reach fieldWidth
			* \see arg(const T& value)
			*/
			FormatableString &arg(unsigned value, int fieldWidth = 0, int base = 10, char fillChar = ' ');
			
			/*!
			* Replace the next arg by a float value.
			* \param value Value used to replace the current argument.
			* \param fieldWidth min width of the displayed number.
			* \param precision Number of digits displayed.
			* \param fillChar Character used to pad the number to reach fieldWidth.
			* \see arg(const T& value)
			*/
			FormatableString &arg(float value, int fieldWidth = 0, int precision = 6, char fillChar = ' ');
			
			/*!
			* Replace the next arg by a value that can be passed to an ostringstream.
			* The first call to arg replace %0, the second %1, and so on.
			* \param value Value used to replace the current argument.
			*/
			template <typename T> FormatableString &arg(const T& value)
			{
				// transform value into std::string
				std::ostringstream oss;
				oss << value;
			
				proceedReplace(oss.str());
	
				// return reference to this so that .arg can proceed further
				return *this;
			}
			
			/*!
			* Affects a new value to the format string and reset the arguments
			* counter.
			* \param str New format string.
			*/
			FormatableString& operator=(const std::string& str) ;
	};
}

#endif // FORMATABLESTRING_H //
