#include "config.h"

#include <EEPROM.h>

#include "product.h"
#include "panstamp.h"
#include "regtable.h"
#include "eeprom.h"
#include "commands.h"
#include "fade.h"
#include "rgbled.h"

#ifdef ENABLE_IR_SEND
# include <IRremote.h>
extern IRsend irsend;
#endif

extern RGBLED rgbLed;
extern byte last_ir_type;
extern unsigned long last_ir_value;

extern byte ir_send_type;
extern unsigned long ir_send_value;

extern byte current_ir_reg;
extern byte learn_ir;

extern void setBlink( int );

/**
 * Declaration of common callback functions
 */
DECLARE_COMMON_CALLBACKS()

/**
 * Definition of common registers
 */
DEFINE_COMMON_REGISTERS();

const void updRGBlevels( byte rId );
const void setRGBlevel(byte rId, byte *levels);
const void updIRrecv( byte rId );
const void updInternalTemp( byte rId );
const void setPowerOnState(byte rId, byte *state);
const void updtRepeaterCfg(byte rId);
const void setRepeaterCfg(byte rId, byte *config);
const void setRGBsend(byte rId, byte *bytes);

/*
 * Definition of custom registers
 */
// RGB levels
static byte dtRGBlevel[4];
REGISTER regRGBlevels(dtRGBlevel, sizeof(dtRGBlevel), &updRGBlevels, &setRGBlevel);

#ifdef ENABLE_IR_RECV
// IR recv
static byte dtIRrecv[5];
REGISTER regIRrecv(dtIRrecv, sizeof(dtIRrecv), &updIRrecv, NULL);
#else
REGISTER regIRrecv(0, 0, NULL, NULL);
#endif

// internal temperature
static byte dtInternalTemp[4];
REGISTER regInternalTemp(dtInternalTemp, sizeof(dtInternalTemp), &updInternalTemp, NULL);

// power on state
static byte dtPowerOnState[5];
REGISTER regPowerOnState(dtPowerOnState, sizeof(dtPowerOnState), NULL, &setPowerOnState);

// command
static byte dtCommand[10];
REGISTER regCommand(dtCommand, sizeof(dtCommand), NULL, &setCommand);

#ifdef ENABLE_REPEATER
// Repeater mode configuration register
byte dtRepeaterCfg[1];      // Maximum hop count. 0 to disable repeater
REGISTER regRepeaterCfg(dtRepeaterCfg, sizeof(dtRepeaterCfg), &updtRepeaterCfg, &setRepeaterCfg);
#else
REGISTER regRepeaterCfg(0, 0, NULL, NULL);
#endif

#ifdef ENABLE_BRIGHTNESS
// brightness
static byte dtBrightness[2];
REGISTER regBrightness(dtBrightness, sizeof(dtBrightness), &updtBrightness, NULL);
#else
REGISTER regBrightness(0, 0, NULL, NULL);
#endif

#ifdef ENABLE_DMX
// dmx config
static byte dtDMX[1];
REGISTER regDMX(dtDMX, sizeof(dtDMX), NULL, &setDMX);
#else
REGISTER regDMX(0, 0, NULL, NULL);
#endif

#ifdef ENABLE_LED_POWER
// led power
static byte dtLedPower[2];
REGISTER regLedPower(dtLedPower, sizeof(dtLedPower), &updLedPower, NULL);
#else
REGISTER regLedPower(0, 0, NULL, NULL);
#endif

#ifdef ENABLE_IR_SEND
// IR send
static byte dtIRsend[9];
REGISTER regIRsend(dtIRsend, sizeof(dtIRsend), NULL, &setdIRsend );
#else
REGISTER regIRsend(0, 0, NULL, NULL);
#endif


/**
 * Initialize table of registers
 */
DECLARE_REGISTERS_START()
        &regRGBlevels,
        &regIRrecv,
        &regInternalTemp,
        &regPowerOnState,
        &regCommand,
        &regRepeaterCfg,
        &regBrightness,
        &regDMX,
        &regLedPower,
DECLARE_REGISTERS_END()

/**
 * Definition of common getter/setter callback functions
 */
DEFINE_COMMON_CALLBACKS()

/**
 * Definition of custom getter/setter callback functions
 */

/**
 * setRGBlevel
 *
 * Set RGB color levels
 *
 * 'rId'     Register ID
 * 'levels'  New RGB levels
 */
const void setRGBlevel(byte rId, byte *levels)
{
  // Update register
  memcpy(dtRGBlevel, levels, sizeof(dtRGBlevel));

  // Control RGB LED
  rgbLed.setColor(levels);
}

const void updRGBlevels( byte rId )
{
  const byte *levels = rgbLed.getColors();

  setRGBlevel(REGI_RGBLEVELS, const_cast<byte *>(levels));
}

#ifdef ENABLE_IR_RECV
const void updIRrecv( byte rId )
{
  dtIRrecv[0] =  last_ir_type;

  dtIRrecv[1] = (last_ir_value >> 24) & 0xFF;
  dtIRrecv[2] = (last_ir_value >> 16) & 0xFF;
  dtIRrecv[3] = (last_ir_value >>  8) & 0xFF;
  dtIRrecv[4] =  last_ir_value        & 0xFF;
}
#endif

const void updInternalTemp( byte rId )
{
  //regInternalTemp.setRegValue(panstamp.getInternalTemp());
  long temp = panstamp.getInternalTemp();
  dtInternalTemp[0] = (temp >> 24) & 0xFF;
  dtInternalTemp[1] = (temp >> 16) & 0xFF;
  dtInternalTemp[2] = (temp >>  8) & 0xFF;
  dtInternalTemp[3] =  temp        & 0xFF;
}

const void setPowerOnState(byte rId, byte *state)
{
  // Update register
  memcpy(dtPowerOnState, state, sizeof(dtPowerOnState));

  // write to eeprom
  for( byte i= 0; i < regTable[REGI_POWER_ON_STATE]->length; ++i )
    EEPROM.write(EEPROM_POWER_ON_STATE + i, state[i]);
}

struct CmdOn
{
  union {
    byte level[sizeof(fade.to)];
    struct{
      byte r;
      byte g;
      byte b;
      byte w;
      byte w2;
    };
  };
  byte secs;
};
struct CmdOnForTimer
{
  byte _secs[2];
  //inline unsigned int secs() const { return _secs[0] << 8 | _secs[1]; };
  inline unsigned int secs(){ union {unsigned int secs; byte to[2]; } tmp; tmp.to[0] = _secs[1]; tmp.to[1] = _secs[0]; return tmp.secs; };
};
//inline void copySwapped2( byte *from, byte *to ) { to[0] = from[1]; to[1] = from[0]; };
//inline void copySwapped4( byte *from, byte *to ) { to[0] = from[3]; to[1] = from[2]; to[2] = from[1]; to[3] = from[0]; };
struct CmdSetIR
{
  byte reg;
  byte bytes[4];
  //inline unsigned long irValue(){ unsigned long value; copySwapped4( bytes, (byte*)&value); return value; };
  inline unsigned long irValue(){ union {unsigned long value; byte to[4]; } tmp; tmp.to[0] = bytes[3]; tmp.to[1] = bytes[2]; tmp.to[2] = bytes[1]; tmp.to[3] = bytes[0]; return tmp.value; };
};
struct CmdSendIR
{
  byte type;
  byte bytes[4];
  //inline unsigned long irValue(){ unsigned long value; copySwapped4( bytes, (byte*)&value); return value; };
  inline unsigned long irValue(){ union {unsigned long value; byte to[4]; } tmp; tmp.to[0] = bytes[3]; tmp.to[1] = bytes[2]; tmp.to[2] = bytes[1]; tmp.to[3] = bytes[0]; return tmp.value; };
};
struct CmdFade
{
  byte reg;
  fadeStep fade_step;
};
struct Cmd
{
  union {
    struct{
      byte cmd;
      union {
        CmdOn cmd_on;
        CmdOnForTimer cmd_on_for_timer;
        CmdSetIR cmd_set_ir;
        CmdFade cmd_fade;
        CmdSendIR cmd_send_ir;
        byte bytes[9];
      };
    };
    byte raw[10];
  };
};

static byte saved[sizeof(fade.to)] = { 0xff, 0xff, 0xff, 0xff };
const void
setCommand(byte rId,byte *command)
{
  // Update register
  memcpy(dtCommand, command, sizeof(dtCommand));

  fadeStop();
  fade_end = fade_start;
  Cmd *cmd = (Cmd *)command;
  if( learn_ir )
    {
      if( cmd->cmd == CMD_NOOP )
        learn_ir = 0;
      else
        switch( learn_ir )
          {
            case 1:
              memcpy(ir_cmds[current_ir_reg].cmd, command, sizeof(dtCommand));
              for( byte i = sizeof(irCmd)*current_ir_reg; i < sizeof(irCmd)*(current_ir_reg+1); ++i )
                EEPROM.write(EEPROM_CONFIG_IR+i, ((byte*)ir_cmds)[i]);
              learn_ir = 0;
              break;
            case 2:
              memcpy(ir_cmds[current_ir_reg].cmd, command, sizeof(dtCommand));
              setBlink( 750 );
              learn_ir = 3;
              break;
            case 3:
              learn_ir = 0;
              break;
            default:
              learn_ir = 0;
              dtCommand[0] = RET_ERR;
          }
      if( !learn_ir )
        setBlink( 0 );
    }
  else
  switch( cmd->cmd ) {
    case CMD_On:
      if( cmd->cmd_on.r != 0 || cmd->cmd_on.g != 0 || cmd->cmd_on.b != 0 ||cmd->cmd_on.w != 0 )
        fadeTo(cmd->cmd_on.secs, cmd->cmd_on.r,cmd->cmd_on.g,cmd->cmd_on.b,cmd->cmd_on.w);
      else switch( regTable[REGI_POWER_ON_STATE]->value[0] & _SOFT_POWER_ON_MASK )
        {
          case SOFT_POWER_ON_STATE_COLOR:
            fadeTo( cmd->cmd_on.secs, regTable[REGI_POWER_ON_STATE]->value[1], regTable[REGI_POWER_ON_STATE]->value[2],  regTable[REGI_POWER_ON_STATE]->value[3],  regTable[REGI_POWER_ON_STATE]->value[4] );
            break;
          case SOFT_POWER_ON_STATE_AUTOSAVE:
            fadeTo( cmd->cmd_on.secs, saved[0], saved[1], saved[2], saved[3]  );
            break;
          default:
            fadeTo( cmd->cmd_on.secs, 255, 255, 255, 255 );
            break;
        }
      break;
    case CMD_Off:
      if( !rgbLed.isOff() ) memcpy( saved, rgbLed.getColors(), sizeof(fade.to) );
      fadeTo( cmd->cmd_on_for_timer.secs(), 0, 0, 0 );
      //fadeTo( cmd->cmd_on.secs, 0, 0, 0 );
      break;
    case CMD_DimDown:
      rgbLed.dimDown();
      getRegister(REGI_RGBLEVELS)->getData();
      break;
    case CMD_DimUp:
      rgbLed.dimUp();
      getRegister(REGI_RGBLEVELS)->getData();
      break;
    case CMD_OnForTimer:
      if( !rgbLed.isOff() ) memcpy( saved, rgbLed.getColors(), sizeof(fade.to) );
      if( rgbLed.isOff() ) rgbLed.setColor(255,255,255); // FIXME: use configured on state
      fadeTo( 0, 0, 0, 0 );
      fade.start = millis() + (unsigned long)cmd->cmd_on_for_timer.secs() * 1000;
      break;
    case CMD_Toggle:
      if( rgbLed.isOff() )
        {
          switch( regTable[REGI_POWER_ON_STATE]->value[0] & _SOFT_POWER_ON_MASK )
            {
              case SOFT_POWER_ON_STATE_COLOR:
                fadeTo( cmd->cmd_on.secs, regTable[REGI_POWER_ON_STATE]->value[1], regTable[REGI_POWER_ON_STATE]->value[2],  regTable[REGI_POWER_ON_STATE]->value[3],  regTable[REGI_POWER_ON_STATE]->value[4] );
                break;
              case SOFT_POWER_ON_STATE_AUTOSAVE:
                fadeTo( cmd->bytes[0], saved[0], saved[1], saved[2], saved[3]  );
                break;
              default:
                fadeTo( cmd->bytes[0], 255, 255, 255, 255 );
                break;
            }
        }
      else
        {
          memcpy( saved, rgbLed.getColors(), sizeof(fade.to) );
          fadeTo( 1, 0, 0, 0, 0 );
        }
      break;
    case CMD_GetIR:
      current_ir_reg = cmd->cmd_set_ir.reg & 0xf;
      dtCommand[1] = current_ir_reg;
      copySwapped4( (byte *)&(ir_cmds[current_ir_reg].ir_cmd), dtCommand+2 );
      getRegister(REGI_COMMAND)->getData();
      memcpy(dtCommand, ir_cmds[current_ir_reg].cmd, sizeof(dtCommand));
      break;
    case CMD_SetIR:
      learn_ir = 1;
      setBlink( 250 );
      current_ir_reg = cmd->cmd_set_ir.reg & 0xf;
      ir_cmds[current_ir_reg].ir_cmd = cmd->cmd_set_ir.irValue();
      dtCommand[1] = current_ir_reg;
      break;
    case CMD_LearnIR:
      learn_ir = 2;
      setBlink( 250 );
      current_ir_reg = cmd->cmd_set_ir.reg & 0xf;
      dtCommand[1] = current_ir_reg;
      break;
    case CMD_GetFade:
      dtCommand[1] = cmd->cmd_fade.reg & 0xf;
      memcpy(dtCommand+2, &(fade_steps[dtCommand[1]]), sizeof(dtCommand)-2);
      break;
    case CMD_SetFade:
      dtCommand[1] = cmd->cmd_fade.reg & 0xf;
      fade_steps[dtCommand[1]] = cmd->cmd_fade.fade_step;
      for( byte i = sizeof(fadeStep)*dtCommand[1]; i < sizeof(fadeStep)*(dtCommand[1]+1); ++i )
        EEPROM.write(EEPROM_CONFIG_FADE+i, ((byte*)fade_steps)[i]);
      break;
    case CMD_StartFade:
      fade_start = cmd->bytes[0] & 0xf;
      fade_end = cmd->bytes[1] & 0xf;
      dtCommand[1] = fade_start;
      dtCommand[2] = fade_end;
      current_fade_reg = fade_start;
      fadeTo(fade_steps[current_fade_reg].secs, fade_steps[current_fade_reg].rgb[0],fade_steps[current_fade_reg].rgb[1],fade_steps[current_fade_reg].rgb[2]);
      break;
#ifdef ENABLE_IR_SEND
    case CMD_SendIR:
      switch( cmd->cmd_send_ir.type ) {
        case NEC:
        case SONY:
          ir_send_type = cmd->cmd_send_ir.type;
          ir_send_value = cmd->cmd_send_ir.irValue();
          break;
        default:
          dtCommand[0] = RET_ERR;
          break;
      }
      break;
#endif
    case CMD_RESET:
      panstamp.enterSystemState( SYSTATE_RESTART );
      break;
    default:
      dtCommand[0] = RET_ERR;
      break;
    }
}

#ifdef ENABLE_REPEATER
/**
 * updtRepeaterCfg
 *
 * Update configuration of the repeater mode
 *
 * 'rId'  Register ID
 */
const void
updtRepeaterCfg(byte rId)
{
  dtRepeaterCfg[0] = maxRepeaterHop & 0x0F;
}

/**
 * setRepeaterCfg
 *
 * Set repeater configuration
 *
 * 'rId'      Register ID
 * 'config'   New configuration
 */
const void
setRepeaterCfg(byte rId, byte *config)
{
  // Update register
  dtRepeaterCfg[0] = config[0];

  // Update repeater config values
  maxRepeaterHop = dtRepeaterCfg[0] & 0x0F;

  // Save config in EEPROM
  EEPROM.write(EEPROM_CONFIG_REPEATER, dtRepeaterCfg[0]);

  panstamp.enableRepeater(maxRepeaterHop);
}
#endif

#ifdef ENABLE_BRIGHTNESS
const void
updtBrightness(byte rId)
{
  int result = analogRead(BRI_PIN);

  regTable[rId]->value[0] = (result >> 8) & 0xFF;
  regTable[rId]->value[1] = result & 0xFF;
}
#endif

#ifdef ENABLE_DMX
const void
setDMX(byte rId,byte *dmx)
{
  dtDMX[0] = dmx[0];

  EEPROM.write(EEPROM_CONFIG_DMX, dtDMX[0]);
}
#endif

#ifdef ENABLE_LED_POWER
const void
updLedPower(byte rId)
{
  unsigned int value = analogRead(LED_POWER_PIN);
  regTable[rId]->value[0] = (value >> 8) & 0xFF;
  regTable[rId]->value[1] = value & 0xFF;
}
#endif

#ifdef ENABLE_IR_SEND
const void
setdIRsend(byte rId,byte *bytes)
{
}
#endif
