/*  BatrachiansEngine.cpp - Main engine

    batrachians - A fly-eating frog game.
    Copyright (C) 2004-2024 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
    02110-1301, USA.
*/

#include "BatrachiansEngine.h"

#include <flatzebra/PixmapLoadError.h>

#include <assert.h>
#include <iostream>
#include <sstream>
#include <algorithm>

using namespace std;
using namespace flatzebra;


///////////////////////////////////////////////////////////////////////////////


const int PAUL = 12;


static const string pkgPixmapDir = GameEngine::getDirPathFromEnv(PKGPIXMAPDIR, "PKGPIXMAPDIR");


ostream &
operator << (ostream &out, const RCouple &c)
{
    return out << '(' << c.x << ", " << c.y << ')';
}

inline
string
doubleToString(double x)
{
    char temp[512];
    snprintf(temp, sizeof(temp), "%f", x);
    return temp;
}

inline
string
exprToString(const char *name, double value)
{
    return string("{") + name + " = " + doubleToString(value) + "} ";
}

inline
string
exprToString(const char *name, int value)
{
    return string("{") + name + " = " + doubleToString(value) + "} ";
}

inline
string
exprToString(const char *name, size_t value)
{
    return string("{") + name + " = " + doubleToString(value) + "} ";
}

inline
string
exprToString(const char *name, RCouple value)
{
    return string("{") + name + " = ("
                + doubleToString(value.x)
                + ", " + doubleToString(value.y) + ")} ";
}

inline
string
exprToString(const char *name, Couple value)
{
    return string("{") + name + " = ("
                + doubleToString(value.x)
                + ", " + doubleToString(value.y) + ")} ";
}

#define SHOW(expr) (exprToString(#expr, expr))


static void
removeNulls(RSpriteList &sl)
{
    RSpriteList::iterator it = remove(sl.begin(), sl.end(), (RSprite *) NULL);
    sl.erase(it, sl.end());
}

static void
deleteSprite(RSprite *p)
{
    delete p;
}

template <class Container>
static void
deleteSprites(Container &c)
{
    for_each(c.begin(), c.end(), deleteSprite);
    c.clear();
}


enum { SCRNWID = 800, SCRNHT = 540 };


const double BatrachiansEngine::PI = 4.0 * atan(1.0);

int BatrachiansEngine::LilyPad::y = SCRNHT - 50;
int BatrachiansEngine::LilyPad::height = 8;

static const double g = 1.4;  // vertical gravitational acceleration


BatrachiansEngine::BatrachiansEngine(const string &windowManagerCaption,
                                        size_t _maxNumFlies,
                                        bool _useSound,
                                        bool fullScreen,
                                        bool processActiveEvent,
                                        bool useAcceleratedRendering)
  : GameEngine(Couple(SCRNWID, SCRNHT),
                windowManagerCaption,
                fullScreen,
                processActiveEvent,
                useAcceleratedRendering),  // may throw string exception
    firstTime(true),
    tickCount(0),
    gameOver(true),
    gameStartTime(0),
    totalPauseTime(0),
    pauseStartTime(0),

    blackColor(mapRGBA(0x00, 0x00, 0x00)),

    lilyPadTexture(NULL),
    lilyPadSurfacePos(),
    leftLilyPad( SCRNWID * 14 / 100, SCRNWID * 32 / 100),
    rightLilyPad(SCRNWID * 59 / 100, SCRNWID * 27 / 100),

    skyTexture(NULL),
    waterTexture(NULL),
    cloudTextures(),

    starPA(2),
    stars(),

    frogPA(8),

    userFrog(NULL),

    compFrog(NULL),

    splashPA(2),

    tonguePA(6),

    crosshairsPA(1),
    crosshairs(NULL),

    fly0PA(4),
    fly1PA(4),
    fly2PA(4),
    fly3PA(4),
    flies(),
    maxNumFlies(_maxNumFlies),
    ticksBeforeNextFly(1),

    userDigitPA(10),
    computerDigitPA(10),
    scoreSprites(),

    controller(),

    theSoundMixer(NULL),
    useSound(_useSound),
    sounds(),
    fontDim(getFontDimensions())
{
    skyTexture = createTextureFromFile(pkgPixmapDir + "sky.xpm");
    if (skyTexture == NULL)
        throw PixmapLoadError(PixmapLoadError::UNKNOWN, NULL);
    waterTexture = createTextureFromFile(pkgPixmapDir + "water.xpm");
    if (waterTexture == NULL)
        throw PixmapLoadError(PixmapLoadError::UNKNOWN, NULL);


    /*  Lily pads:
    */
    lilyPadTexture = createTextureFromFile(pkgPixmapDir + "lilypads0.xpm");
    if (lilyPadTexture == NULL)
        throw PixmapLoadError(PixmapLoadError::UNKNOWN, NULL);

    Couple size = getTextureSize(lilyPadTexture);
    lilyPadSurfacePos = Couple((SCRNWID - size.x) / 2, SCRNHT - size.y);
    int yWater = getYWater();


    /* Clouds:
    */
    cloudTextures[0] = createTextureFromFile(pkgPixmapDir + "cloud0_0.xpm");
    cloudTextures[1] = createTextureFromFile(pkgPixmapDir + "cloud1_0.xpm");


    /*  Stars:
    */
    loadPixmap("star0.xpm", starPA, 0);
    loadPixmap("star1.xpm", starPA, 1);
    for (int i = 20; i > 0; i--)
        stars.push_back(new RSprite(starPA,
                        RCouple(rand() % (SCRNWID - 100) + 50,
                                rand() % (yWater - 60) + 30),
                        RCouple(), RCouple(), RCouple(), RCouple()));
    changeStarColors();


    /*  User and computer frogs:
    */
    loadPixmap("frog0_l_stand.xpm", frogPA, FROG_L_STAND);
    loadPixmap("frog0_l_swim.xpm",  frogPA, FROG_L_SWIM);
    loadPixmap("frog0_r_stand.xpm", frogPA, FROG_R_STAND);
    loadPixmap("frog0_r_swim.xpm",  frogPA, FROG_R_SWIM);
    loadPixmap("frog1_l_stand.xpm", frogPA, 4 + FROG_L_STAND);
    loadPixmap("frog1_l_swim.xpm",  frogPA, 4 + FROG_L_SWIM);
    loadPixmap("frog1_r_stand.xpm", frogPA, 4 + FROG_R_STAND);
    loadPixmap("frog1_r_swim.xpm",  frogPA, 4 + FROG_R_SWIM);

    Couple frogSize = frogPA.getImageSize();

    RSprite *userFrogSprite = new RSprite(frogPA,
                    RCouple(), RCouple(), RCouple(0, +2), RCouple(), frogSize);

    RSprite *computerFrog = new RSprite(frogPA,
                    RCouple(), RCouple(), RCouple(0, +2), RCouple(), frogSize);


    loadPixmap("splash0.xpm", splashPA, 0);
    loadPixmap("splash1.xpm", splashPA, 1);


    loadPixmap("tongue0_r2.xpm", tonguePA, 5);

    RCouple collBoxPos(0, 0);
    RCouple collBoxSize = tonguePA.getImageSize() + RCouple(0, 2);
    RSprite *userTongue = new RSprite(tonguePA,
                            RCouple(), RCouple(), RCouple(),
                            collBoxPos, collBoxSize);
    SDL_Color userFrogColor = mapRGBA(0xb5, 0x18, 0x5a);

    userFrog = new Frog(userFrogSprite, userTongue, &userDigitPA, userFrogColor);

    RSprite *computerTongue = new RSprite(tonguePA,
                            RCouple(), RCouple(), RCouple(),
                            collBoxPos, collBoxSize);

    SDL_Color computerFrogColor = mapRGBA(0xbd, 0xae, 0xce);

    compFrog = new Frog(computerFrog, computerTongue, &computerDigitPA, computerFrogColor);
                        // ownership of RSprite objects passed to Frog objects


    /*  Flies:
    */
    loadPixmap("fly0_r0.xpm", fly0PA, 2);
    loadPixmap("fly1_r0.xpm", fly1PA, 2);
    loadPixmap("fly2_r0.xpm", fly2PA, 2);
    loadPixmap("fly3_r0.xpm", fly3PA, 2);


    /*  Crosshairs:
    */
    loadPixmap("crosshairs.xpm", crosshairsPA, 0);
    crosshairs = new RSprite(crosshairsPA,
                        RCouple(), RCouple(), RCouple(), RCouple(), RCouple());


    /* Digits:
    */
    loadPixmap("digit_user_0.xpm", userDigitPA, 0);
    loadPixmap("digit_user_1.xpm", userDigitPA, 1);
    loadPixmap("digit_user_2.xpm", userDigitPA, 2);
    loadPixmap("digit_user_3.xpm", userDigitPA, 3);
    loadPixmap("digit_user_4.xpm", userDigitPA, 4);
    loadPixmap("digit_user_5.xpm", userDigitPA, 5);
    loadPixmap("digit_user_6.xpm", userDigitPA, 6);
    loadPixmap("digit_user_7.xpm", userDigitPA, 7);
    loadPixmap("digit_user_8.xpm", userDigitPA, 8);
    loadPixmap("digit_user_9.xpm", userDigitPA, 9);

    loadPixmap("digit_computer_0.xpm", computerDigitPA, 0);
    loadPixmap("digit_computer_1.xpm", computerDigitPA, 1);
    loadPixmap("digit_computer_2.xpm", computerDigitPA, 2);
    loadPixmap("digit_computer_3.xpm", computerDigitPA, 3);
    loadPixmap("digit_computer_4.xpm", computerDigitPA, 4);
    loadPixmap("digit_computer_5.xpm", computerDigitPA, 5);
    loadPixmap("digit_computer_6.xpm", computerDigitPA, 6);
    loadPixmap("digit_computer_7.xpm", computerDigitPA, 7);
    loadPixmap("digit_computer_8.xpm", computerDigitPA, 8);
    loadPixmap("digit_computer_9.xpm", computerDigitPA, 9);


    /*  Sound effects:
    */
    if (useSound)
    {
        try
        {
            theSoundMixer = NULL;
            theSoundMixer = new SoundMixer(16);  // may throw string
        }
        catch (const SoundMixer::Error &)
        {
            return;
        }

        static const string d = getDirPathFromEnv(PKGSOUNDDIR, "PKGSOUNDDIR");

        try
        {
            sounds.gameStarts.init(d + "game-starts.wav");
            sounds.frogJumps.init(d + "frog-jumps.wav");
            sounds.flyEaten.init(d + "fly-eaten.wav");
            sounds.tongueOut.init(d + "tongue-out.wav");
            sounds.splash.init(d + "splash.wav");
            sounds.gameEnds.init(d + "game-ends.wav");
        }
        catch (const SoundMixer::Error &e)
        {
            throw e.what();
        }
    }


    initGame();
}


BatrachiansEngine::~BatrachiansEngine()
{
    delete theSoundMixer;
    delete crosshairs;
    delete compFrog;
    delete userFrog;
    SDL_DestroyTexture(cloudTextures[0]);
    SDL_DestroyTexture(cloudTextures[1]);
    SDL_DestroyTexture(waterTexture);
    SDL_DestroyTexture(skyTexture);
    SDL_DestroyTexture(lilyPadTexture);
}


///////////////////////////////////////////////////////////////////////////////


void
BatrachiansEngine::loadPixmap(const char *filePath,
                              PixmapArray &pa,
                              size_t index)
{
    GameEngine::loadPixmap(pkgPixmapDir + filePath, pa, index);
}


/*virtual*/
void
BatrachiansEngine::processKey(SDL_Keycode keysym, bool pressed)
{
    controller.processKey(keysym, pressed);
}


/*virtual*/
void
BatrachiansEngine::processActivation(bool appActive)
{
    if (!appActive)  // if pause starting:
        pauseStartTime = SDL_GetTicks();  // remember time when pause started
    else  // pause ending:
        totalPauseTime += SDL_GetTicks() - pauseStartTime;  // accumulate time spent in pauses
}


/*virtual*/
bool
BatrachiansEngine::tick()
{
    if (controller.isQuitRequested())
        return false;

    tickCount++;

    if (controller.isFullScreenToggleRequested())
        setFullScreenMode(!inFullScreenMode());  // ignore failure

    if (!gameOver)
    {
        moveFrog(*userFrog->frogSprite, *userFrog->tongueSprite, userFrog->splashSprite, true);
        moveFrog(*compFrog->frogSprite, *compFrog->tongueSprite, compFrog->splashSprite, false);
    }

    moveFlies();
    animateTemporarySprites(scoreSprites);


    // Make stars sparkle (they are only visible towards the end):
    if (tickCount % (3 * FPS) == 1)
        changeStarColors();


    bool timeIsUp = draw();

    if (!gameOver && timeIsUp)
    {
        gameOver = true;
        initFrogPositions();
        playSoundEffect(sounds.gameEnds);
    }

    if (!gameOver)
    {
        detectCollisions();

        controlCrosshairs();
        controlUserFrogJump();
        controlUserFrogTongue();
        controlComputerFrogJump();
        controlComputerFrogTongue();
    }
    else
    {
        if (controller.isStartRequested())
        {
            firstTime = false;
            initGame();
            gameOver = false;

            playSoundEffect(sounds.gameStarts);
        }
    }

    controller.update();

    return true;
}


void
BatrachiansEngine::initFrogPositions()
{
    Couple frogSize = frogPA.getImageSize();
    userFrog->frogSprite->setPos(RCouple(leftLilyPad.xMiddle() - frogSize.x / 2,
                                                LilyPad::y - frogSize.y));
    userFrog->frogSprite->setSpeed(RCouple());
    userFrog->frogSprite->setAccel(RCouple());
    userFrog->frogSprite->currentPixmapIndex = FROG_R_STAND;

    compFrog->frogSprite->currentPixmapIndex = 4 + FROG_L_STAND;
    compFrog->frogSprite->setPos(RCouple(rightLilyPad.xMiddle() - frogSize.x / 2,
                                                LilyPad::y - frogSize.y));
    compFrog->frogSprite->setSpeed(RCouple());
    compFrog->frogSprite->setAccel(RCouple());
}


void
BatrachiansEngine::initGame()
{
    initFrogPositions();

    userFrog->score = 0;
    compFrog->score = 0;

    compFrog->setTicksBeforeNextJump();

    delete userFrog->splashSprite;
    userFrog->splashSprite = NULL;
    delete compFrog->splashSprite;
    compFrog->splashSprite = NULL;

    userFrog->tongueSprite->currentPixmapIndex = 5;
    userFrog->tongueTicksLeft = 0;

    compFrog->tongueSprite->currentPixmapIndex = 5;
    compFrog->tongueTicksLeft = 0;

    Couple crosshairsSize = crosshairsPA.getImageSize();
    crosshairs->setPos(RCouple(
                            userFrog->frogSprite->getCenterPos().x - crosshairsSize.x / 2,
                        SCRNHT / 2 - crosshairsSize.y / 2));

    setTicksBeforeNextFly();

    deleteSprites(flies);

    gameStartTime = SDL_GetTicks();
    totalPauseTime = 0;
    pauseStartTime = 0;
}


void
BatrachiansEngine::moveFrog(RSprite &aFrog, RSprite &itsTongue,
                                        RSprite *&itsSplash, bool isUser)
{
    RSprite *frog = &aFrog;

    if (frog->getSpeed().isZero())
        return;

    bool swimming = isFrogSwimming(*frog);

    if (swimming && itsSplash != NULL)
    {
        animateSplash(itsSplash);
        return;  // wait for splash animation to end before swimming to a pad
    }

    RCouple &pos = frog->getPos();
    RCouple &speed = frog->getSpeed();
    RCouple &accel = frog->getAccel();
    Couple frogSize = frog->getSize();

    frog->addAccelToSpeed();
    frog->addSpeedToPos();

    RCouple lrPos = frog->getLowerRightPos();

    const int N = 15;
    const int M = 10;

    RCouple left(pos.x + N, lrPos.y + 1);
    RCouple right(lrPos.x - 1 - N, lrPos.y + 1);
    bool onLeftPad = (leftLilyPad.isPointOverPad(left)
                        || leftLilyPad.isPointOverPad(right));
    bool onRightPad = (rightLilyPad.isPointOverPad(left)
                        || rightLilyPad.isPointOverPad(right));

    if (swimming)
        assert(speed.isNonZero());

    size_t piOffset = (isUser ? 0 : 4);

    if (!swimming && lrPos.y > LilyPad::y)
    {
        /*  The frog's bottom is now lower than the surface of the
            lily pad.
            If one of the bottom corners of the frog is over a lily pad,
            then the frog stops moving and now stands still on the pad.
            If the whole bottom of the frog is over water, then the frog
            is allowed to go a little lower, and then it changes state
            to the "swimming" pose.
        */

        if (onLeftPad || onRightPad)
        {
            // Frog is over a lily pad.
            pos.y = LilyPad::y - frogSize.y;
            speed.zero();
            compFrog->setTicksBeforeNextJump();
        }
        else
        {
            if (lrPos.y > LilyPad::y + LilyPad::height)
            {
                pos.y = LilyPad::y + LilyPad::height - frogSize.y / 2 + 1;

                // Choose a horiz. direction towards the nearest pad:
                int dx = 0;
                if (lrPos.x < leftLilyPad.xRight())
                    dx = +1;
                else if (pos.x > rightLilyPad.x)
                    dx = -1;
                else
                {
                    double start = frog->getCenterPos().x;
                    double toLeft = fabs(leftLilyPad.xRight() - 1 - start);
                    double toRight = fabs(rightLilyPad.x - start);
                    dx = (toLeft <= toRight ? -1 : +1);
                }
                assert(dx != 0);

                speed = RCouple(dx * 1.5, 0);
                accel.zero();
                frog->currentPixmapIndex = piOffset + FROG_R_SWIM;
                assert(isFrogSwimming(*frog));
                swimming = true;

                // Show a splash:
                assert(itsSplash == NULL);
                itsSplash = new RSprite(splashPA,
                                        RCouple(frog->getPos()),
                                        RCouple(), RCouple(),
                                        RCouple(), RCouple());
                itsSplash->currentPixmapIndex = 0;
                itsSplash->values = new long[1];
                itsSplash->setTimeToLive(35);

                // Hide tongue:
                if (isUser)
                    userFrog->tongueTicksLeft = 0;
                else
                    compFrog->tongueTicksLeft = 0;

                playSoundEffect(sounds.splash);
            }
        }
    }
    else if (swimming)
    {
        if (onRightPad)
        {
            if (pos.x > rightLilyPad.x)
            {
                pos = RCouple(rightLilyPad.xRight() + M, LilyPad::y) - frogSize;
                frog->currentPixmapIndex = piOffset + FROG_L_STAND;
            }
            else
            {
                pos = RCouple(rightLilyPad.x - M, LilyPad::y - frogSize.y);
                frog->currentPixmapIndex = piOffset + FROG_R_STAND;
            }
        }
        else if (onLeftPad)
        {
            if (pos.x < leftLilyPad.x)
            {
                pos = RCouple(leftLilyPad.x - M, LilyPad::y - frogSize.y);
                frog->currentPixmapIndex = piOffset + FROG_R_STAND;
            }
            else
            {
                pos = RCouple(leftLilyPad.xRight() + M, LilyPad::y) - frogSize;
                frog->currentPixmapIndex = piOffset + FROG_L_STAND;
            }
        }

        if (onLeftPad || onRightPad)
        {
            speed.zero();
            compFrog->setTicksBeforeNextJump();
        }
    }

    if (!swimming)
    {
        if (pos.x < 0)
        {
            pos.x = 0;
            speed.x = 0;
        }
        if (lrPos.x > SCRNWID)
        {
            pos.x = SCRNWID - frogSize.x;
            speed.x = 0;
        }

        setTonguePosition(*frog, piOffset, itsTongue);
    }

    assert(frog->currentPixmapIndex < frogPA.getNumImages());
}


void
BatrachiansEngine::setTonguePosition(const RSprite &frog,
                                        size_t piOffset,
                                        RSprite &tongue)
{
    Couple frogSize = frog.getSize();
    Couple tongueSize = tongue.getSize();
    int yTongue = (frogSize.y - tongueSize.y) / 2;

    if (frog.currentPixmapIndex == piOffset + FROG_R_STAND)
        tongue.setPos(frog.getPos() + RCouple(frogSize.x, yTongue));
    else
        tongue.setPos(frog.getPos() + RCouple(- tongueSize.x, yTongue));
}


void
BatrachiansEngine::animateSplash(RSprite *&splash)
{
    assert(splash != NULL);
    unsigned long ticksLeft = splash->getTimeToLive();
    if (ticksLeft > 0)
    {
        if (ticksLeft % 2 == 0)
            splash->currentPixmapIndex ^= 1;
        splash->decTimeToLive();
        return;
    }

    delete splash;
    splash = NULL;
}


bool
BatrachiansEngine::draw()
{
    // Game time.
    // If no game has been played yet, show day.
    // If no game currently on, but a game has been played, leave night displayed.
    //
    Uint32 msPlayed = (gameOver ? (firstTime ? 0 : GAME_LENGTH_IN_MS) : SDL_GetTicks() - gameStartTime - totalPauseTime);
    Uint32 msToPlay = (msPlayed < GAME_LENGTH_IN_MS ? GAME_LENGTH_IN_MS - msPlayed : 0);


    // Background:
    {
        Uint8 alpha = Uint8(msToPlay > DARK_MS ? 255 : msToPlay * 255 / DARK_MS);
        Uint8 starAlpha;
        if (msToPlay > BEGIN_STARS_MS)
            starAlpha = 0;  // no stars
        else if (msToPlay < FULL_STARS_MS)
            starAlpha = 255;  // brightest stars
        else
            starAlpha = 255 - (msToPlay - FULL_STARS_MS) * 255 / (BEGIN_STARS_MS - FULL_STARS_MS);

        setTextureAlphaMod(starPA.getImage(0), starAlpha);
        setTextureAlphaMod(starPA.getImage(1), starAlpha);

        fillRect(0, 0, SCRNWID, SCRNHT, blackColor);  // background for alpha blending

        putSpriteList(stars);

        setTextureAlphaMod(skyTexture, alpha);
        copyPixmap(skyTexture, Couple(0, 0));
        setTextureAlphaMod(waterTexture, alpha);
        copyPixmap(waterTexture, Couple(0, 390));

        Uint8 lilyPadAlpha;
        if (msToPlay > DARK_MS)
            lilyPadAlpha = 255;  // full color
        else
            lilyPadAlpha = msToPlay * (255 - 63) / DARK_MS + 63;

        setTextureAlphaMod(lilyPadTexture, lilyPadAlpha);
        copyPixmap(lilyPadTexture, lilyPadSurfacePos);

        setTextureAlphaMod(cloudTextures[0], alpha);
        setTextureAlphaMod(cloudTextures[1], alpha);

        copyPixmap(cloudTextures[0], Couple(SCRNWID * 20 / 100, 50));
        copyPixmap(cloudTextures[1], Couple(SCRNWID * 60 / 100, SCRNHT / 4));
    }


    // Score, clock and lighting:
    {
        // Draw rectangles that will surround the two scores.
        const int scorePos = 15, margin = 4;
        const int width = 5 * fontDim.x + 2 * margin;
        const int height = fontDim.y + 2 * margin;
        fillRect(scorePos - margin, scorePos - margin, width, height, userFrog->color);
        fillRect(SCRNWID - scorePos + margin - width, scorePos - margin, width, height, compFrog->color);

        // Write the scores inside those rectangles.
        char s[16];
        snprintf(s, sizeof(s), " %3ld ", userFrog->score);
        writeString(s, Couple(scorePos, scorePos));
        snprintf(s, sizeof(s), " %3ld ", compFrog->score);
        writeStringRightJustified(s, Couple(SCRNWID - scorePos, scorePos));

        {
            // Show clock:
            Uint32 secondsToPlay = (msToPlay + 999) / 1000;  // round up
            Uint32 minutes = secondsToPlay / 60;
            Uint32 seconds = secondsToPlay % 60;
            snprintf(s, sizeof(s), " %u:%02u ", unsigned(minutes), unsigned(seconds));
            writeStringXCentered(s, Couple(SCRNWID / 2, 15));
        }
    }


    // Flies (when stars appear, the files start to flicker):
    for (Iter it = flies.begin(); it != flies.end(); it++)
    {
        if (msToPlay >= FLICKERING_FLIES_MS || rand() % 6 == 0)
            putSprite(*it);
    }


    // Frogs:
    assert(compFrog->frogSprite->currentPixmapIndex < frogPA.getNumImages());
    putSprite(compFrog->frogSprite);
    putSprite(userFrog->frogSprite);

    if (!gameOver)
    {
        if (compFrog->tongueTicksLeft > 0)
        {
            assert(compFrog->tongueSprite->getPos().isNonZero());
            putSprite(compFrog->tongueSprite);
            compFrog->tongueTicksLeft--;
        }

        if (userFrog->tongueTicksLeft > 0)
        {
            assert(userFrog->tongueSprite->getPos().isNonZero());
            putSprite(userFrog->tongueSprite);
            userFrog->tongueTicksLeft--;
        }

        // Splashes:
        if (userFrog->splashSprite != NULL)
            putSprite(userFrog->splashSprite);
        if (compFrog->splashSprite != NULL)
            putSprite(compFrog->splashSprite);


        // Misc.:
        putSprite(crosshairs);
    }


    // Scores:
    putSpriteList(scoreSprites);


    Couple center = Couple(SCRNWID, SCRNHT) / 2;
    if (gameOver)
    {
        if (!firstTime)
            writeStringXCentered(" GAME OVER ", center + Couple(0, 70));

        static const char *lines[] =
        {
            "                                ",
            "      " PACKAGE_FULL_NAME_EN " " VERSION "         ",
            "      By Pierre Sarrazin        ",
            "                                ",
            " Your frog is the red one.      ",
            " You control the crosshairs.    ",
            " Move them with the ARROW keys. ",
            " Jump with the LEFT CTRL key.   ",
            " Stick out the tongue with the  ",
            " LEFT SHIFT key.                ",
            "                                ",
            " Press F11 toggle full screen.  ",
            "                                ",
            " Press SPACE to start a game.   ",
            "                                ",
        };

        size_t numLines = sizeof(lines) / sizeof(lines[0]);
        Couple pos = center - Couple(0, fontDim.y * (numLines - 4));
        for (size_t i = 0; i < numLines; ++i)
        {
            writeStringXCentered(lines[i], pos);
            pos.y += fontDim.y;
        }
    }

    return msToPlay <= 0;
}


void
BatrachiansEngine::detectCollisions()
{
    detectTongueFlyCollisions(*userFrog);
    detectTongueFlyCollisions(*compFrog);
}


size_t
BatrachiansEngine::detectTongueFlyCollisions(Frog &frog)
{
    if (frog.tongueTicksLeft <= 0)  // if tongue not out
        return 0;

    size_t numFliesEaten = 0;
    for (Iter it = flies.begin(); it != flies.end(); it++)
    {
        RSprite &fly = **it;
        if (frog.tongueSprite->collidesWithRSprite(fly))
        {
            numFliesEaten++;

            size_t type = getFlyType(fly);
            assert(type < 4);
            int score = 0;
            switch (type)
            {
                case 0: score = 35; break;
                case 1: score = 15; break;
                case 2: score = 10; break;
                case 3: score =  5; break;
                default: assert(false);
            }

            createScoreSprites(score, fly.getCenterPos(), frog);

            delete *it;
            *it = NULL;

            playSoundEffect(sounds.flyEaten);
        }
    }

    removeNulls(flies);

    return numFliesEaten;
}


void
BatrachiansEngine::moveFlies()
{
    double radius = 4.5;

    if (flies.size() < maxNumFlies && --ticksBeforeNextFly == 0)
    {
        setTicksBeforeNextFly();

        // Create at least one new fly.
        //
        size_t numMissingFlies = maxNumFlies - flies.size();
        size_t numNewFlies = rand() % numMissingFlies + 1;

        for (size_t j = 0; j < numNewFlies; ++j)
        {
            size_t type = (size_t) rand() % 4;
            PixmapArray *pa = NULL;
            switch (type)
            {
                case 0: pa = &fly0PA; break;
                case 1: pa = &fly1PA; break;
                case 2: pa = &fly2PA; break;
                case 3: pa = &fly3PA; break;
                default: assert(false);
            }
            Couple flySize = pa->getImageSize();

            int dx = (rand() % 2 == 0 ? -1 : +1);
            RCouple pos(dx < 0 ? SCRNWID : - flySize.x, rand() % (SCRNHT / 2));
            double angle = (rand() * PI / 2 / RAND_MAX) + (dx < 0 ? 0.75 * PI : -0.25 * PI);
            RCouple speed = getCoupleFromAngle(radius, angle);

            RSprite *fly = new RSprite(*pa, pos, speed, RCouple(),
                                        RCouple(5, 5), flySize - RCouple(5, 5));
            fly->currentPixmapIndex = 2;

            flies.push_back(fly);
        }
    }

    for (Iter it = flies.begin(); it != flies.end(); it++)
    {
        RSprite &fly = **it;
        int type = getFlyType(fly);

        if (rand() % (5 + 15 * type) == 0)  // if time to change direction
        {
            double angle = getAngleFromCouple(fly.getSpeed());
            double max = PI / 180 * (rand() % (3 + type) == 0 ? 360 : 4);
            double change = rand() * max / RAND_MAX - (max / 2);
            fly.setSpeed(getCoupleFromAngle(radius, angle + change));
        }

        fly.addSpeedToPos();

        RCouple &pos = fly.getPos();
        RCouple lrPos = fly.getLowerRightPos();
        RCouple &speed = fly.getSpeed();

        if (pos.x >= SCRNWID || lrPos.x <= 0 || lrPos.y < 0)
        {
            delete *it;
            *it = NULL;
        }
        else if (pos.y > LilyPad::y - 100)
        {
            fly.subSpeedFromPos();
            speed.y = - speed.y;
            fly.addSpeedToPos();
        }
    }

    removeNulls(flies);
}


void
BatrachiansEngine::controlCrosshairs()
{
    const RCouple chPos = crosshairs->getPos();
    const RCouple chLRPos = crosshairs->getLowerRightPos();
    const Couple chSize = crosshairs->getSize();
    RCouple newCHPos = crosshairs->getPos();

    enum { CROSSHAIRS_SPEED = FPS * 3 / 4 };

    if (controller.isLeftRequested())
        newCHPos.x -= CROSSHAIRS_SPEED;
    if (controller.isRightRequested())
        newCHPos.x += CROSSHAIRS_SPEED;
    if (controller.isUpRequested())
        newCHPos.y -= CROSSHAIRS_SPEED;
    if (controller.isDownRequested())
        newCHPos.y += CROSSHAIRS_SPEED;

    if (newCHPos.x < 0)
        newCHPos.x = 0;
    else if (newCHPos.x + chSize.x > SCRNWID)
        newCHPos.x = SCRNWID - chSize.x;

    int yWater = getYWater();
    if (newCHPos.y < 0)
        newCHPos.y = 0;
    else if (newCHPos.y + chSize.y > yWater)
        newCHPos.y = yWater - chSize.y;

    RCouple newDelta = newCHPos - userFrog->frogSprite->getCenterPos();
    double newDist = newDelta.length();
    double maxDist = SCRNHT * 0.75;
    if (newDist > maxDist)
    {
        newCHPos = userFrog->frogSprite->getCenterPos() + newDelta * maxDist / newDist;
        if (newCHPos.y + chSize.y > yWater)
            newCHPos.y = yWater - chSize.y;
    }

    crosshairs->setPos(newCHPos);
}


void
BatrachiansEngine::controlUserFrogJump()
{
    if (controller.isJumpRequested() && userFrog->frogSprite->getSpeed().isZero())
    {
        try
        {
            RCouple speed = computeJumpSpeed(
                    userFrog->frogSprite->getPos(),
                    crosshairs->getCenterPos() - userFrog->frogSprite->getSize() / 2,
                    g);
            userFrog->frogSprite->setSpeed(speed);
            userFrog->frogSprite->setAccel(RCouple(0, g));

            if (speed.x < 0)
                userFrog->frogSprite->currentPixmapIndex = FROG_L_STAND;
            else
                userFrog->frogSprite->currentPixmapIndex = FROG_R_STAND;

            playSoundEffect(sounds.frogJumps);
        }
        catch (invalid_argument &e)
        {
            cerr << "logic_error: " << e.what() << endl;
            abort();
        }
    }
}


void
BatrachiansEngine::controlUserFrogTongue()
{
    if (controller.isTongueRequested()
                && userFrog->tongueTicksLeft == 0
                && !isFrogSwimming(*userFrog->frogSprite))
    {
        RCouple frogSize = userFrog->frogSprite->getSize();
        userFrog->tongueTicksLeft = FPS * 3 / 4;
        setTonguePosition(*userFrog->frogSprite, 0, *userFrog->tongueSprite);
        playSoundEffect(sounds.tongueOut);
    }
}


RSprite *
BatrachiansEngine::findNearestFly(RCouple pos)
{
    RSprite *nearestFly = NULL;
    double shortestDist = HUGE_VAL;
    for (Iter it = flies.begin(); it != flies.end(); it++)
    {
        RSprite *fly = *it;
        double dist = (fly->getPos() - pos).length();
        if (dist < shortestDist)
        {
            shortestDist = dist;
            nearestFly = fly;
        }
    }
    return nearestFly;
}


void
BatrachiansEngine::controlComputerFrogJump()
{
    if (compFrog->frogSprite->getSpeed().isNonZero())
        return;
    if (isFrogSwimming(*compFrog->frogSprite))
        return;
    if (compFrog->ticksBeforeNextJump > 0)
    {
        compFrog->ticksBeforeNextJump--;
        return;
    }

    RSprite *nearestFly = findNearestFly(compFrog->frogSprite->getPos());
    if (nearestFly == NULL)
        return;
    RCouple nfPos = nearestFly->getPos();
    if (nfPos.x < 100 || nearestFly->getLowerRightPos().x > SCRNWID - 100)
        return;
    double dist = (nfPos - compFrog->frogSprite->getPos()).length();
    double maxDist = SCRNHT * 0.70;
    if (dist > maxDist)
        return;

    try
    {
        RCouple target = nearestFly->getCenterPos()
                                        - compFrog->frogSprite->getSize() / 2;

        // Introduce a clumsiness coefficient...
        target += RCouple(rand() % 50 - 25, rand() % 50 - 25);

        // Move the target in the expected direction of the fly:
        target += 10 * nearestFly->getSpeed();

        RCouple speed = computeJumpSpeed(compFrog->frogSprite->getPos(), target, g);
        compFrog->frogSprite->setSpeed(speed);
        compFrog->frogSprite->setAccel(RCouple(0, g));

        if (speed.x < 0)
            compFrog->frogSprite->currentPixmapIndex = 4 + FROG_L_STAND;
        else
            compFrog->frogSprite->currentPixmapIndex = 4 + FROG_R_STAND;

        playSoundEffect(sounds.frogJumps);
    }
    catch (invalid_argument &e)
    {
        cerr << "logic_error: " << e.what() << endl;
        abort();
    }
}


void
BatrachiansEngine::controlComputerFrogTongue()
{
    if (compFrog->tongueTicksLeft == 0
                && compFrog->frogSprite->getSpeed().isNonZero()
                && !isFrogSwimming(*compFrog->frogSprite))
    {
        RSprite *nearestFly = findNearestFly(compFrog->frogSprite->getPos());
        if (nearestFly != NULL)
        {
            double dist = (nearestFly->getPos()
                                        - compFrog->frogSprite->getPos()).length();
            if (dist <= 120)
            {
                compFrog->tongueTicksLeft = FPS * 3 / 4;
                setTonguePosition(*compFrog->frogSprite, 4, *compFrog->tongueSprite);
                playSoundEffect(sounds.tongueOut);
            }
        }
    }
}


/** Calculates the initial speed vector for a jump to a target.

    The parabola of the jump is defined by these parametric equations:

        x = vx * t
        y = vy * t - g * t^2 / 2

    where g is the vertical gravitational acceleration.  Let (a, b) be
    the peak of the jump.  Let T be the time when the peak is reached.
    Then y'(T) = 0, thus vy - g*T = 0, so T = vy / g.  By using this T
    for t in the two first equations, we get

        a = vx * vy / g
        b = vy^2 / g - g * vy^2 / g^2 / 2

    From the last equation, we get vy = sqrt(2 * g * b), and then
    we have vx = a * g / vy.

    THANK YOU PAUL GUERTIN

    @param        start                point whence the jump starts
    @param        target                highest point of the jump
    @param        g                vertical gravitational acceleration
                                (positive number of pixels)
    @returns                        the speed vector to use at the start
                                    of the jump; the acceleration vector
                                RCouple(0, g) should be used to animate
                                the jump
    @throws        invalid_argument        if 'g' is non-positive or if the
                                            target is not above the start
*/
RCouple
BatrachiansEngine::computeJumpSpeed(RCouple start, RCouple target, double g)
{
    if (g <= 0)
        throw invalid_argument("invalid gravitational constant");

    double a = target.x - start.x;
    double b = target.y - start.y;
    if (b >= 0)
        throw invalid_argument("target is below frog");

    double vy = sqrt(2 * g * -b) + 1;
                // +1 as patch to compensate for accumulated round off errors
    double vx = a * g / vy;
    return RCouple(vx, -vy);
}


void
BatrachiansEngine::playSoundEffect(SoundMixer::Chunk &chunk)
{
    if (theSoundMixer != NULL)
    {
        try
        {
            theSoundMixer->playChunk(chunk);
        }
        catch (const SoundMixer::Error &e)
        {
            fprintf(stderr, "playSoundEffect: %s (chunk at %p)\n",
                        e.what().c_str(), &chunk);
        }
    }
}


void
BatrachiansEngine::createScoreSprites(long n, RCouple center, Frog &frog)
{
    if (n < 0)
        n = -n;

    frog.score += n;

    char number[64];
    snprintf(number, sizeof(number), "%ld", n);
    size_t numDigits = strlen(number);

    RCouple digitSize = userDigitPA.getImageSize();
    RCouple totalSize((digitSize.x + 2) * numDigits - 2, digitSize.y);
    RCouple scorePos = center - totalSize / 2;

    const PixmapArray &pa = *frog.digitPixmapArray;

    for (size_t i = 0; i < numDigits; i++)
    {
        int digit = number[i] - '0';
        RSprite *s = new RSprite(pa,
                            scorePos + i * RCouple(digitSize.x + 2, 0),
                            RCouple(0, -1), RCouple(),
                            RCouple(), RCouple());
        s->setTimeToLive(FPS);
        s->currentPixmapIndex = digit;
        scoreSprites.push_back(s);
    }
}


void
BatrachiansEngine::animateTemporarySprites(RSpriteList &sl) const
/*  'slist' must be a list of sprites that die when their "time to live"
    expires.  This method removes sprites from 'slist' when they die.
    Sprites that live are advanced by adding their speed to their position.
*/
{
    for (Iter it = sl.begin(); it != sl.end(); it++)
    {
        RSprite *s = *it;
        assert(s != NULL);
        if (s->getTimeToLive() == 0)
        {
            delete s;
            *it = NULL;  // mark list element for deletion
        }
        else
        {
            s->decTimeToLive();
            s->addSpeedToPos();
        }
    }

    removeNulls(sl);
}


int
BatrachiansEngine::getFlyType(const RSprite &fly) const
{
    int type = 3;
    if (fly.getPixmapArray() == &fly0PA)
        type = 0;
    else if (fly.getPixmapArray() == &fly1PA)
        type = 1;
    else if (fly.getPixmapArray() == &fly2PA)
        type = 2;
    else if (fly.getPixmapArray() == &fly3PA)
        type = 3;
    else
        assert(false);
    return type;
}
