/*
 *                           TERMS AND CONDITIONS
 *                                   FOR
 *                         OPEN SOURCE CODE LICENSE
 *                               Version 1.1
 * 
 * Japan Registry Services Co., Ltd. ("JPRS"), a Japanese corporation
 * having its head office at Chiyoda First Bldg. East 13F 3-8-1 Nishi-Kanda,
 * Chiyoda-ku, Tokyo 101-0065, Japan, grants you the license for open source
 * code specified in EXHIBIT A the "Code" subject to the following Terms and
 * Conditions ("OSCL").
 * 
 * 1. License Grant.
 *   JPRS hereby grants you a worldwide, royalty-free, non-exclusive
 *   license, subject to third party intellectual property claims:
 *   (a) under intellectual property rights (other than patent or
 *       trademark) licensable by JPRS to use, reproduce, modify, display,
 *       perform, sublicense and distribute the Code (or portions thereof)
 *       with or without modifications, and/or as part of a derivative work;
 *       or
 *   (b) under claims of the infringement through the making, using,
 *       offering to sell and/or otherwise disposing the JPRS Revised Code
 *       (or portions thereof);
 *   (c) the licenses granted in this Section 1(a) and (b) are effective on
 *       the date JPRS first distributes the Code to you under the terms of
 *       this OSCL;
 *   (d) Notwithstanding the above stated terms, no patent license is
 *       granted:
 *       1)  for a code that you delete from the Code;
 *       2)  separate from the Code; or
 *       3)  for infringements caused by:
 *            i) modification of the Code; or
 *           ii) combination of the Code with other software or devices.
 * 
 * 2. Consents.
 *   You agree that:
 *   (a) you must include a copy of this OSCL and the notice set forth in
 *       EXHIBIT A with every copy of the Code you distribute;
 *   (b) you must include a copy of this OSCL and the notice set forth in
 *       EXHIBIT A with every copy of binary form of the Code in the
 *       documentation and/or other materials provided with the distribution;
 *   (c) you may not offer or impose any terms on any source code version
 *       that alters or restricts the applicable version of this OSCL or
 *       the recipients' rights hereunder.
 *   (d) If the terms and conditions are set forth in EXHIBIT A, you must
 *       comply with those terms and conditions.
 * 
 * 3. Proprietary Information.
 *   All trademarks, service marks, patents, copyrights, trade secrets, and
 *   other proprietary rights in or related to the Code are and will remain
 *   the exclusive property of JPRS or its licensors, whether or not
 *   specifically recognized or perfected under local law except specified
 *   in this OSCL; provided however you agree and understand that the JPRS
 *   name may not be used to endorse or promote this Code without prior
 *   written approval of JPRS.
 * 
 * 4. WARRANTY DISCLAIMER.
 *   JPRS MAKES NO REPRESENTATIONS AND WARRANTIES REGARDING THE USE OF THE
 *   CODE, NOR DOES JPRS MAKE ANY REPRESENTATIONS THAT THE CODE WILL BECOME
 *   COMMERCIALLY AVAILABLE. JPRS, ITS AFFILIATES, AND ITS SUPPLIERS DO NOT
 *   WARRANT OR REPRESENT THAT THE CODE IS FREE OF ERRORS OR THAT THE CODE
 *   IS SUITABLE FOR TRANSLATION AND/OR LOCALIZATION. THE CODE IS PROVIDED
 *   ON AN "AS IS" BASIS AND JPRS AND ITS SUPPLIERS HAVE NO OBLIGATION TO
 *   CORRECT ERRORS OR TO SUPPORT THE CODE UNDER THIS OSCL FOR ANY REASON.
 *   TO THE FULL EXTENT PERMITTED BY LAW, ALL OBLIGATIONS ARE HEREBY
 *   EXCLUDED WHETHER EXPRESS, STATUTORY OR IMPLIED UNDER LAW, COURSE OF
 *   DEALING, CUSTOM, TRADE USAGE, ORAL OR WRITTEN STATEMENT OR OTHERWISE,
 *   INCLUDING BUT NOT LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
 *   OR FITNESS FOR A PARTICULAR PURPOSE CONCERNING THE CODE.
 * 
 * 5. NO LIABILITY.
 *   UNDER NO CIRCUMSTANCES SHALL JPRS AND/OR ITS AFFILIATES, LICENSORS, OR
 *   REPRESENTATIVES BE LIABLE FOR ANY DAMAGES INCLUDING BUT NOT LIMITED TO
 *   CONSEQUENTIAL, INDIRECT, SPECIAL, PUNITIVE OR INCIDENTAL DAMAGES,
 *   WHETHER FORESEEABLE OR UNFORESEEABLE, BASED ON YOUR CLAIMS, INCLUDING,
 *   BUT NOT LIMITED TO, CLAIMS FOR LOSS OF DATA, GOODWILL, PROFITS, USE OF
 *   MONEY, INTERRUPTION IN USE OR AVAILABILITY OF DATA, STOPPAGE, IMPLIED
 *   WARRANTY, BREACH OF CONTRACT, MISREPRESENTATION, NEGLIGENCE, STRICT
 *   LIABILITY IN TORT, OR OTHERWISE.
 * 
 * 6. Indemnification.
 *   You hereby agree to indemnify, defend, and hold harmless JPRS for any
 *   liability incurred by JRPS due to your terms of warranty, support,
 *   indemnity, or liability offered by you to any third party.
 * 
 * 7. Termination.
 * 7.1 This OSCL shall be automatically terminated in the events that:
 *   (a) You fail to comply with the terms herein and fail to cure such
 *       breach within 30 days of becoming aware of the breach;
 *   (b) You initiate patent or copyright infringement litigation against
 *       any party (including a cross-claim or counterclaim in a lawsuit)
 *       alleging that the Code constitutes a direct or indirect patent or
 *       copyright infringement, in such case, this OSCL to you shall
 *       terminate as of the date such litigation is filed;
 * 7.2 In the event of termination under Sections 7.1(a) or 7.1(b) above,
 *     all end user license agreements (excluding distributors and
 *     resellers) which have been validly granted by You or any distributor
 *     hereunder prior to termination shall survive termination.
 *
 * 
 * 8. General.
 *   This OSCL shall be governed by, and construed and enforced in
 *   accordance with, the laws of Japan. Any litigation or arbitration
 *   between the parties shall be conducted exclusively in Tokyo, Japan
 *   except written consent of JPRS provides other venue.
 * 
 * 
 *                                EXHIBIT A
 * 
 * The original open source code of idnkit-2 is idnkit-1.0 developed and
 * conceived by Japan Network Information Center ("JPNIC"), a Japanese
 * association, Kokusai-Kougyou-Kanda Bldg 6F, 2-3-4 Uchi-Kanda,
 * Chiyoda-ku, Tokyo 101-0047, Japan, and JPRS modifies above original code
 * under following Terms and Conditions set forth by JPNIC.
 * 
 *                                  JPNIC
 * 
 * Copyright (c) 2000-2002 Japan Network Information Center.  All rights reserved.
 * 
 * By using this file, you agree to the terms and conditions set forth bellow.
 * 
 *                       LICENSE TERMS AND CONDITIONS
 * 
 * The following License Terms and Conditions apply, unless a different
 * license is obtained from Japan Network Information Center ("JPNIC"),
 * a Japanese association, Kokusai-Kougyou-Kanda Bldg 6F, 2-3-4 Uchi-Kanda,
 * Chiyoda-ku, Tokyo 101-0047, Japan.
 * 
 * 1. Use, Modification and Redistribution (including distribution of any
 *    modified or derived work) in source and/or binary forms is permitted
 *    under this License Terms and Conditions.
 * 
 * 2. Redistribution of source code must retain the copyright notices as they
 *    appear in each source code file, this License Terms and Conditions.
 * 
 * 3. Redistribution in binary form must reproduce the Copyright Notice,
 *    this License Terms and Conditions, in the documentation and/or other
 *    materials provided with the distribution. For the purposes of binary
 *    distribution the "Copyright Notice" refers to the following language:
 *    "Copyright (c) 2000-2002 Japan Network Information Center.  All rights reserved."
 * 
 * 4. The name of JPNIC may not be used to endorse or promote products
 *    derived from this Software without specific prior written approval of
 *    JPNIC.
 * 
 * 5. Disclaimer/Limitation of Liability: THIS SOFTWARE IS PROVIDED BY JPNIC
 *    "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *    LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 *    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL JPNIC BE LIABLE
 *    FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *    CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *    SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *    BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *    WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 *    OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 *    ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * 
 *                        JPRS Public License Notice
 *                                   For
 *                                idnkit-2.
 * 
 * The contents of this file are subject to the Terms and Conditions for
 * the Open Source Code License (the "OSCL"). You may not use this file
 * except in compliance with above terms and conditions. A copy of the OSCL
 * is available at <http://jprs.co.jp/idn/>.
 * The JPRS Revised Code is idnkit-2.
 * The Initial Developer of the JPRS Revised Code is Japan Network
 * Information Center ("JPNIC"), a Japanese association,
 * Kokusai-Kougyou-Kanda Bldg 6F, 2-3-4 Uchi-Kanda, Chiyoda-ku, Tokyo
 * 101-0047, Japan.
 * "Copyright (c) 2000-2002 Japan Network Information Center.  All rights reserved."
 * "Copyright (c) 2010-2012 Japan Registry Services Co., Ltd.  All rights reserved."
 * Contributor(s): ______________________________________.
 * 
 * If you wish to allow use of your version of this file only under the
 * above License(s) and not to allow others to use your version of this
 * file, please indicate your decision by deleting the relevant provisions
 * above and replacing them with the notice and other provisions required
 * by the above License(s). If you do not delete the relevant provisions,
 * a recipient may use your version of this file under either the above
 * License(s).
 */

#include <config.h>

#include <stddef.h>
#include <stdlib.h>
#include <string.h>

#include <idn/assert.h>
#include <idn/debug.h>
#include <idn/logmacro.h>
#include <idn/result.h>
#include <idn/ctxrule.h>
#include <idn/auto/combiningclass.h>
#include <idn/auto/joiningtype.h>
#include <idn/auto/script.h>

static int	idn__ctxrule_zerowidthnonjoiner(const unsigned long *s,
						size_t idx);
static int	idn__ctxrule_zerowidthjoiner(const unsigned long *s,
					     size_t idx);
static int	idn__ctxrule_middledot(const unsigned long *s, size_t idx);
static int	idn__ctxrule_greeklowernumeralsign(const unsigned long *s,
						   size_t idx);
static int	idn__ctxrule_hebrewpunctuation(const unsigned long *s,
					       size_t idx);
static int	idn__ctxrule_katakanamiddledot(const unsigned long *s,
					       size_t idx);
static int	idn__ctxrule_arabicindicdigits(const unsigned long *s,
					       size_t idx);
static int	idn__ctxrule_extendedarabicindicdigits(const unsigned long *s,
						       size_t idx);
/*
 * Context rule table.
 * If new context rule is defined, add the rule to this table.
 *
 * Note that idn__ctxrule_t ctxrules[] must be sorted in ascending
 * code point order.
 */
typedef struct {
	unsigned long min_codepoint;
	unsigned long max_codepoint;
	int lookup;
	int (*rulesetproc)(const unsigned long *s, size_t idx);
} idn__ctxrule_t;

static const idn__ctxrule_t ctxrules[] = {
	/* MIDDLE DOT */
	{  0x00B7,   0x00B7, 0, idn__ctxrule_middledot},

	/* GREEK LOWER NUMERAL SIGN (KERAIA) */
	{  0x0375,   0x0375, 0, idn__ctxrule_greeklowernumeralsign},

	/* HEBREW PUNCTUATION GERESH */
	{  0x05F3,   0x05F3, 0, idn__ctxrule_hebrewpunctuation},

	/* HEBREW PUNCTUATION GERSHAYIM */
	{  0x05F4,   0x05F4, 0, idn__ctxrule_hebrewpunctuation},

	/* ARABIC-INDIC DIGITS */
	{  0x0660,   0x0669, 0, idn__ctxrule_arabicindicdigits},

	/* EXTENDED ARABIC-INDIC DIGITS */
	{  0x06F0,   0x06F9, 0, idn__ctxrule_extendedarabicindicdigits},

	/* ZERO WIDTH NON-JOINER */
	{  0x200C,   0x200C, 1, idn__ctxrule_zerowidthnonjoiner},

	/* ZERO WIDTH JOINER */
	{  0x200D,   0x200D, 1, idn__ctxrule_zerowidthjoiner},

	/* KATAKANA MIDDLE DOT */
	{  0x30FB,   0x30FB, 0, idn__ctxrule_katakanamiddledot},
};

static const idn__ctxrule_t *
		idn__find_ctxrule(const unsigned long v);

/*
 * Test if a context rule for code point s[idx] exists and the ruleset
 * returns 'true'.
 */
int
idn__ctxrule_check(int lookup, const unsigned long *s, size_t idx) {
	const idn__ctxrule_t *ctxrule;

	ctxrule = idn__find_ctxrule(s[idx]);
	if (ctxrule == NULL)
		return (0);
	if (lookup && !ctxrule->lookup)
		return (0);

	return ((*ctxrule->rulesetproc)(s, idx));
}

/*
 * Test if a context rule for code point s[idx] exists.
 */
int
idn__ctxrule_exist(int lookup, const unsigned long *s, size_t idx) {
	const idn__ctxrule_t *ctxrule;
	
	ctxrule = idn__find_ctxrule(s[idx]);
	if (ctxrule == NULL)
		return (0);

	return (!lookup || ctxrule->lookup);
}

/*
 * Test if a context rule for code point 'v' exists.
 */
static const idn__ctxrule_t *
idn__find_ctxrule(const unsigned long v) {
	int lo, hi, mid;

	lo = 0;
	hi = sizeof(ctxrules) / sizeof(idn__ctxrule_t) - 1;
	while (lo <= hi) {
		mid = (lo + hi) / 2;
		if (ctxrules[mid].max_codepoint < v)
			lo = mid + 1;
		else if (ctxrules[mid].min_codepoint > v)
			hi = mid - 1;
		else
			return (ctxrules + mid);
	}

	return (NULL);
}

/*
 * Inspect "ZERO WIDTH NON-JOINER".
 *
 *  Code point:
 *     U+200C
 *   Lookup:
 *     True
 *   Rule Set:
 *     False;
 *     If Canonical_Combining_Class(Before(cp)) .eq.  Virama Then True;
 *     If RegExpMatch((Joining_Type:{L,D})(Joining_Type:T)*\u200C
 *        (Joining_Type:T)*(Joining_Type:{R,D})) Then True;
 *
 * Note that Unicode Character Database (UCD) assigns 9 to Canonical
 * Combining Class 'Virama'.
 *
 * The context rule above doesn't mention a label containing two or
 * more U+200C, but the rule set has no ambiguity since joining type
 * of U+200C itself is U.
 */
static int
idn__ctxrule_zerowidthnonjoiner(const unsigned long *s, size_t idx) {
	idn__joiningtype_t jt;
	int passed;
	int i;

	if (idx > 0 && idn__sparsemap_getcombiningclass(s[idx - 1]) == 9)
		return (1);

	if (s[idx] == '\0')
		return (0);
	for (passed = 0, i = idx + 1; s[i] != '\0'; i++) {
		jt = idn__sparsemap_getjoiningtype(s[i]);
		if (jt == idn__joiningtype_R || jt == idn__joiningtype_D) {
			passed = 1;
			break;
		} else if (jt != idn__joiningtype_T) {
			break;
		}
	}
	if (!passed)
		return (0);

	for (passed = 0, i = idx - 1; i >= 0; i--) {
		jt = idn__sparsemap_getjoiningtype(s[i]);
		if (jt == idn__joiningtype_L || jt == idn__joiningtype_D) {
			passed = 1;
			break;
		} else if (jt != idn__joiningtype_T) {
			break;
		}
	}
	if (!passed)
		return (0);

	return (1);
}

/*
 * Inspect "ZERO WIDTH JOINER".
 *    Code point:
 *       U+200D
 *    Lookup:
 *       True
 *    Rule Set:
 *       False;
 *       If Canonical_Combining_Class(Before(cp)) .eq.  Virama Then True;
 *
 * Note that Unicode Character Database (UCD) assigns 9 to Canonical
 * Combining Class 'Virama'.
 */
static int
idn__ctxrule_zerowidthjoiner(const unsigned long *s, size_t idx) {
	if (idx == 0)
		return (0);
	return (idn__sparsemap_getcombiningclass(s[idx - 1]) == 9);
}

/*
 * Inspect "MIDDLE DOT".
 *    Code point:
 *       U+00B7
 *    Lookup:
 *       False
 *    Rule Set:
 *       False;
 *       If Before(cp) .eq.  U+006C And
 *          After(cp) .eq.  U+006C Then True;
 */
static int
idn__ctxrule_middledot(const unsigned long *s, size_t idx) {
	if (idx == 0 || s[idx] == '\0')
		return (0);
	return (s[idx - 1] == 0x006C && s[idx + 1] == 0x006C);
}

/*
 * Inspect "GREEK LOWER NUMERAL SIGN (KERAIA)".
 *    Code point:
 *       U+0375
 *    Lookup:
 *       False
 *    Rule Set:
 *       False;
 *       If Script(After(cp)) .eq.  Greek Then True;
 */
static int
idn__ctxrule_greeklowernumeralsign(const unsigned long *s, size_t idx) {
	idn__script_t script;

	if (s[idx] == '\0')
		return (0);
	script = (idn__script_t)idn__sparsemap_getscript(s[idx + 1]);
	return (script == idn__script_greek);
}

/*
 * Inspect "HEBREW PUNCTUATION GERESH".
 * Inspect "HEBREW PUNCTUATION GERSHAYIM".
 *    Code point:
 *       U+05F3, U+05F4
 *    Lookup:
 *       False
 *    Rule Set:
 *       False;
 *       If Script(Before(cp)) .eq.  Hebrew Then True;
 */
static int
idn__ctxrule_hebrewpunctuation(const unsigned long *s, size_t idx) {
	idn__script_t script;

	if (idx == 0)
		return (0);
	script = (idn__script_t)idn__sparsemap_getscript(s[idx - 1]);
	return (script == idn__script_hebrew);
}

/*
 * Inspect "KATAKANA MIDDLE DOT".
 *    Code point:
 *       U+30FB
 *    Lookup:
 *       False
 *    Rule Set:
 *       False;
 *       For All Characters:
 *          If Script(cp) .in. {Hiragana, Katakana, Han} Then True;
 *       End For;
 */
static int
idn__ctxrule_katakanamiddledot(const unsigned long *s, size_t idx) {
	idn__script_t script;
	int i;

	for (i = 0; s[i] != '\0'; i++) {
		script = (idn__script_t)idn__sparsemap_getscript(s[i]);
		if (script == idn__script_hiragana ||
		    script == idn__script_katakana ||
		    script == idn__script_han)
			return (1);
	}

	return (0);
}

/*
 * Inspect "ARABIC-INDIC DIGITS".
 *    Code point:
 *       0660..0669
 *    Lookup:
 *       False
 *    Rule Set:
 *       True;
 *       For All Characters:
 *          If cp .in. 06F0..06F9 Then False;
 *       End For;
 */
static int
idn__ctxrule_arabicindicdigits(const unsigned long *s, size_t idx) {
	int i;

	for (i = 0; s[i] != '\0'; i++) {
		if (s[i] >= 0x06F0 && s[i] <= 0x06F9)
			return (0);
	}

	return (1);
}

/*
 * Inspect "EXTENDED ARABIC-INDIC DIGITS".
 *    Code point:
 *       06F0..06F9
 *    Lookup:
 *       False
 *    Rule Set:
 *       True;
 *       For All Characters:
 *          If cp .in. 0660..0669 Then False;
 *       End For;
 */
static int
idn__ctxrule_extendedarabicindicdigits(const unsigned long *s, size_t idx) {
	int i;

	for (i = 0; s[i] != '\0'; i++) {
		if (s[i] >= 0x0660 && s[i] <= 0x0669)
			return (0);
	}

	return (1);
}
