/*
 *                           TERMS AND CONDITIONS
 *                                   FOR
 *                         OPEN SOURCE CODE LICENSE
 *                               Version 1.1
 * 
 * Japan Registry Services Co., Ltd. ("JPRS"), a Japanese corporation
 * having its head office at Chiyoda First Bldg. East 13F 3-8-1 Nishi-Kanda,
 * Chiyoda-ku, Tokyo 101-0065, Japan, grants you the license for open source
 * code specified in EXHIBIT A the "Code" subject to the following Terms and
 * Conditions ("OSCL").
 * 
 * 1. License Grant.
 *   JPRS hereby grants you a worldwide, royalty-free, non-exclusive
 *   license, subject to third party intellectual property claims:
 *   (a) under intellectual property rights (other than patent or
 *       trademark) licensable by JPRS to use, reproduce, modify, display,
 *       perform, sublicense and distribute the Code (or portions thereof)
 *       with or without modifications, and/or as part of a derivative work;
 *       or
 *   (b) under claims of the infringement through the making, using,
 *       offering to sell and/or otherwise disposing the JPRS Revised Code
 *       (or portions thereof);
 *   (c) the licenses granted in this Section 1(a) and (b) are effective on
 *       the date JPRS first distributes the Code to you under the terms of
 *       this OSCL;
 *   (d) Notwithstanding the above stated terms, no patent license is
 *       granted:
 *       1)  for a code that you delete from the Code;
 *       2)  separate from the Code; or
 *       3)  for infringements caused by:
 *            i) modification of the Code; or
 *           ii) combination of the Code with other software or devices.
 * 
 * 2. Consents.
 *   You agree that:
 *   (a) you must include a copy of this OSCL and the notice set forth in
 *       EXHIBIT A with every copy of the Code you distribute;
 *   (b) you must include a copy of this OSCL and the notice set forth in
 *       EXHIBIT A with every copy of binary form of the Code in the
 *       documentation and/or other materials provided with the distribution;
 *   (c) you may not offer or impose any terms on any source code version
 *       that alters or restricts the applicable version of this OSCL or
 *       the recipients' rights hereunder.
 *   (d) If the terms and conditions are set forth in EXHIBIT A, you must
 *       comply with those terms and conditions.
 * 
 * 3. Proprietary Information.
 *   All trademarks, service marks, patents, copyrights, trade secrets, and
 *   other proprietary rights in or related to the Code are and will remain
 *   the exclusive property of JPRS or its licensors, whether or not
 *   specifically recognized or perfected under local law except specified
 *   in this OSCL; provided however you agree and understand that the JPRS
 *   name may not be used to endorse or promote this Code without prior
 *   written approval of JPRS.
 * 
 * 4. WARRANTY DISCLAIMER.
 *   JPRS MAKES NO REPRESENTATIONS AND WARRANTIES REGARDING THE USE OF THE
 *   CODE, NOR DOES JPRS MAKE ANY REPRESENTATIONS THAT THE CODE WILL BECOME
 *   COMMERCIALLY AVAILABLE. JPRS, ITS AFFILIATES, AND ITS SUPPLIERS DO NOT
 *   WARRANT OR REPRESENT THAT THE CODE IS FREE OF ERRORS OR THAT THE CODE
 *   IS SUITABLE FOR TRANSLATION AND/OR LOCALIZATION. THE CODE IS PROVIDED
 *   ON AN "AS IS" BASIS AND JPRS AND ITS SUPPLIERS HAVE NO OBLIGATION TO
 *   CORRECT ERRORS OR TO SUPPORT THE CODE UNDER THIS OSCL FOR ANY REASON.
 *   TO THE FULL EXTENT PERMITTED BY LAW, ALL OBLIGATIONS ARE HEREBY
 *   EXCLUDED WHETHER EXPRESS, STATUTORY OR IMPLIED UNDER LAW, COURSE OF
 *   DEALING, CUSTOM, TRADE USAGE, ORAL OR WRITTEN STATEMENT OR OTHERWISE,
 *   INCLUDING BUT NOT LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
 *   OR FITNESS FOR A PARTICULAR PURPOSE CONCERNING THE CODE.
 * 
 * 5. NO LIABILITY.
 *   UNDER NO CIRCUMSTANCES SHALL JPRS AND/OR ITS AFFILIATES, LICENSORS, OR
 *   REPRESENTATIVES BE LIABLE FOR ANY DAMAGES INCLUDING BUT NOT LIMITED TO
 *   CONSEQUENTIAL, INDIRECT, SPECIAL, PUNITIVE OR INCIDENTAL DAMAGES,
 *   WHETHER FORESEEABLE OR UNFORESEEABLE, BASED ON YOUR CLAIMS, INCLUDING,
 *   BUT NOT LIMITED TO, CLAIMS FOR LOSS OF DATA, GOODWILL, PROFITS, USE OF
 *   MONEY, INTERRUPTION IN USE OR AVAILABILITY OF DATA, STOPPAGE, IMPLIED
 *   WARRANTY, BREACH OF CONTRACT, MISREPRESENTATION, NEGLIGENCE, STRICT
 *   LIABILITY IN TORT, OR OTHERWISE.
 * 
 * 6. Indemnification.
 *   You hereby agree to indemnify, defend, and hold harmless JPRS for any
 *   liability incurred by JRPS due to your terms of warranty, support,
 *   indemnity, or liability offered by you to any third party.
 * 
 * 7. Termination.
 * 7.1 This OSCL shall be automatically terminated in the events that:
 *   (a) You fail to comply with the terms herein and fail to cure such
 *       breach within 30 days of becoming aware of the breach;
 *   (b) You initiate patent or copyright infringement litigation against
 *       any party (including a cross-claim or counterclaim in a lawsuit)
 *       alleging that the Code constitutes a direct or indirect patent or
 *       copyright infringement, in such case, this OSCL to you shall
 *       terminate as of the date such litigation is filed;
 * 7.2 In the event of termination under Sections 7.1(a) or 7.1(b) above,
 *     all end user license agreements (excluding distributors and
 *     resellers) which have been validly granted by You or any distributor
 *     hereunder prior to termination shall survive termination.
 *
 * 
 * 8. General.
 *   This OSCL shall be governed by, and construed and enforced in
 *   accordance with, the laws of Japan. Any litigation or arbitration
 *   between the parties shall be conducted exclusively in Tokyo, Japan
 *   except written consent of JPRS provides other venue.
 * 
 * 
 *                                EXHIBIT A
 * 
 * The original open source code of idnkit-2 is idnkit-1.0 developed and
 * conceived by Japan Network Information Center ("JPNIC"), a Japanese
 * association, Kokusai-Kougyou-Kanda Bldg 6F, 2-3-4 Uchi-Kanda,
 * Chiyoda-ku, Tokyo 101-0047, Japan, and JPRS modifies above original code
 * under following Terms and Conditions set forth by JPNIC.
 * 
 *                                  JPNIC
 * 
 * Copyright (c) 2000-2002 Japan Network Information Center.  All rights reserved.
 * 
 * By using this file, you agree to the terms and conditions set forth bellow.
 * 
 *                       LICENSE TERMS AND CONDITIONS
 * 
 * The following License Terms and Conditions apply, unless a different
 * license is obtained from Japan Network Information Center ("JPNIC"),
 * a Japanese association, Kokusai-Kougyou-Kanda Bldg 6F, 2-3-4 Uchi-Kanda,
 * Chiyoda-ku, Tokyo 101-0047, Japan.
 * 
 * 1. Use, Modification and Redistribution (including distribution of any
 *    modified or derived work) in source and/or binary forms is permitted
 *    under this License Terms and Conditions.
 * 
 * 2. Redistribution of source code must retain the copyright notices as they
 *    appear in each source code file, this License Terms and Conditions.
 * 
 * 3. Redistribution in binary form must reproduce the Copyright Notice,
 *    this License Terms and Conditions, in the documentation and/or other
 *    materials provided with the distribution. For the purposes of binary
 *    distribution the "Copyright Notice" refers to the following language:
 *    "Copyright (c) 2000-2002 Japan Network Information Center.  All rights reserved."
 * 
 * 4. The name of JPNIC may not be used to endorse or promote products
 *    derived from this Software without specific prior written approval of
 *    JPNIC.
 * 
 * 5. Disclaimer/Limitation of Liability: THIS SOFTWARE IS PROVIDED BY JPNIC
 *    "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *    LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 *    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL JPNIC BE LIABLE
 *    FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *    CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *    SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *    BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *    WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 *    OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 *    ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * 
 *                        JPRS Public License Notice
 *                                   For
 *                                idnkit-2.
 * 
 * The contents of this file are subject to the Terms and Conditions for
 * the Open Source Code License (the "OSCL"). You may not use this file
 * except in compliance with above terms and conditions. A copy of the OSCL
 * is available at <http://jprs.co.jp/idn/>.
 * The JPRS Revised Code is idnkit-2.
 * The Initial Developer of the JPRS Revised Code is Japan Network
 * Information Center ("JPNIC"), a Japanese association,
 * Kokusai-Kougyou-Kanda Bldg 6F, 2-3-4 Uchi-Kanda, Chiyoda-ku, Tokyo
 * 101-0047, Japan.
 * "Copyright (c) 2000-2002 Japan Network Information Center.  All rights reserved."
 * "Copyright (c) 2010-2012 Japan Registry Services Co., Ltd.  All rights reserved."
 * Contributor(s): ______________________________________.
 * 
 * If you wish to allow use of your version of this file only under the
 * above License(s) and not to allow others to use your version of this
 * file, please indicate your decision by deleting the relevant provisions
 * above and replacing them with the notice and other provisions required
 * by the above License(s). If you do not delete the relevant provisions,
 * a recipient may use your version of this file under either the above
 * License(s).
 */

#include <config.h>

#include <stddef.h>
#include <sys/types.h>
#include <ctype.h>
#include <errno.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_PWD_H
#include <pwd.h>
#endif

#ifdef HAVE_SYS_PARAM_H
#include <sys/param.h>
#endif
#ifndef PATH_MAX
#ifdef MAXPATHLEN
#define PATH_MAX MAXPATHLEN
#else
#define PATH_MAX 1023
#endif
#endif

#include <idn/assert.h>
#include <idn/debug.h>
#include <idn/logmacro.h>
#include <idn/result.h>
#include <idn/resconf_internal.h>
#include <idn/delimitermap.h>
#include <idn/foreignset.h>
#include <idn/labellist.h>
#include <idn/lang.h>
#include <idn/langlocalmap.h>
#include <idn/localconverter.h>
#include <idn/localencoding.h>
#include <idn/lowercasemap.h>
#include <idn/tr46map.h>
#include <idn/tr46check.h>
#include <idn/maplist.h>
#include <idn/normalizer.h>
#include <idn/tldlocalmap.h>
#include <idn/util.h>
#include <idn/widthmap.h>

#ifndef SYSCONFDIR
#define SYSCONFDIR		"/etc"
#endif
#define SYSTEM_CONF_FILE	"idn2.conf"
#define USER_CONF_FILE		".idn2rc"

#define MAX_CONF_LINE_LENGTH	255
#define MAX_CONF_LINE_ARGS	127

struct idn_resconf {
	idn__localencoding_t localencoding;
	idn__lang_t lang;
	int localconverter_flags;
	idn__maplist_t maplist;
	idn__delimitermap_t delimitermap;
	idn__langlocalmap_t langlocalmap;
	idn__tldlocalmap_t tldlocalmap;
	idn__foreignset_t localcheck;
	int need_reset;
};

typedef struct {
	int lineno;
	int language_count;
	int map_count;
	int delimiters_count;
} confparser_t;

static int initialized = 0;

static idn_result_t	set_default_maps(idn_resconf_t ctx);
static idn_result_t	set_default_delimiters(idn_resconf_t ctx);
static idn_result_t	open_userconffile(FILE **fpp);
static idn_result_t	open_systemconffile(FILE **fpp);
static idn_result_t	open_conffile(const char *file, FILE **fpp);
static idn_result_t	parse_confline(idn_resconf_t ctx, confparser_t *cp,
				       const char *line);
static int		split_args(char *s, char **av, int max_ac);
static idn_result_t	parse_language(idn_resconf_t ctx, char *args,
				       int lineno);
static idn_result_t	parse_map(idn_resconf_t ctx, char *args,
				  int lineno);
static idn_result_t	parse_delimiters(idn_resconf_t ctx, char *args,
					 int lineno);
static idn_result_t	parse_tld_local(idn_resconf_t ctx, char *args,
					int lineno);
static idn_result_t	parse_language_local(idn_resconf_t ctx, char *args,
					     int lineno);
static idn_result_t	add_map(idn_resconf_t ctx, const char *mapname);
static void		confparser_init(confparser_t *cp);

/*
 * Initialize the module.
 */
idn_result_t
idn_resconf_initialize(void) {
	idn_result_t r = idn_success;

	TRACE(("idn_resconf_initialize()\n"));

	if (initialized)
		goto ret;

	/*
	 * Initialize sub modules.
	 */
	if ((r = idn__localencoding_initialize()) != idn_success)
		goto ret;
	if ((r = idn__lang_initialize()) != idn_success)
		goto ret;

	initialized = 1;
ret:
	TRACE(("idn_resconf_initialize(): %s\n", idn_result_tostring(r)));
	return (r);
}

/*
 * Create a resconf object.
 */
idn_result_t
idn_resconf_create(idn_resconf_t *ctxp) {
	idn_result_t r = idn_success;
	idn_resconf_t ctx = NULL;
	idn__localencoding_t localencoding = NULL;
	idn__lang_t lang = NULL;
	idn__maplist_t maplist = NULL;
	idn__delimitermap_t delimitermap = NULL;
	idn__langlocalmap_t langlocalmap = NULL;
	idn__tldlocalmap_t tldlocalmap = NULL;

	assert(ctxp != NULL);

	TRACE(("idn_resconf_create()\n"));

	if (!initialized) {
		r = idn_failure;
		goto ret;
	}
	ctx = (idn_resconf_t)malloc(sizeof(*ctx));
	if (ctx == NULL) {
		r = idn_nomemory;
		goto ret;
	}

	r = idn__localencoding_create(&localencoding);
	if (r != idn_success)
		goto ret;

	r = idn__lang_create(&lang);
	if (r != idn_success)
		goto ret;

	r = idn__maplist_create(&maplist);
	if (r != idn_success)
		goto ret;

	r = idn__delimitermap_create(&delimitermap);
	if (r != idn_success)
		goto ret;

	r = idn__langlocalmap_create(&langlocalmap);
	if (r != idn_success)
		goto ret;

	r = idn__tldlocalmap_create(&tldlocalmap);
	if (r != idn_success)
		goto ret;

	ctx->localencoding = localencoding;
	ctx->lang = lang;
	ctx->localconverter_flags = 0;
	ctx->maplist = maplist;
	ctx->delimitermap = delimitermap;
	ctx->langlocalmap = langlocalmap;
	ctx->tldlocalmap = tldlocalmap;
	ctx->localcheck = NULL;

	r = set_default_maps(ctx);
	if (r != idn_success)
		goto ret;

	r = set_default_delimiters(ctx);
	if (r != idn_success)
		goto ret;

	ctx->need_reset = 0;
	*ctxp = ctx;

ret:
	TRACE(("idn_resconf_create(): %s\n", idn_result_tostring(r)));
	if (r != idn_success) {
		if (localencoding != NULL)
			idn__localencoding_destroy(localencoding);
		if (lang != NULL)
			idn__lang_destroy(lang);
		if (maplist != NULL)
			idn__maplist_destroy(maplist);
		if (delimitermap != NULL)
			idn__delimitermap_destroy(delimitermap);
		if (langlocalmap != NULL)
			idn__langlocalmap_destroy(langlocalmap);
		if (tldlocalmap != NULL)
			idn__tldlocalmap_destroy(tldlocalmap);
		free(ctx);
	}
	return (r);
}

/*
 * Set default mapping procedures.
 */
static idn_result_t
set_default_maps(idn_resconf_t ctx) {
	idn_result_t r = idn_success;
	static const char *default_map_names[] = {
		IDN_RESCONF_WIDTHMAP_NAME,
		IDN_RESCONF_LOWERCASEMAP_NAME, 
		IDN_RESCONF_NFCMAP_NAME,
		IDN_RESCONF_DELIMITERMAP_NAME,
		IDN_RESCONF_LANGLOCALMAP_NAME,
		IDN_RESCONF_NFCMAP_NAME,
		NULL
	};
	const char **map_name;

	for (map_name = default_map_names; *map_name != NULL; map_name++) {
		r = add_map(ctx, *map_name);
		if (r != idn_success)
			return (r);
	}

	return (idn_success);
}

/*
 * Set default delimiters.
 */
static idn_result_t
set_default_delimiters(idn_resconf_t ctx) {
	idn_result_t r = idn_success;
	static unsigned long default_delimiters[] = {
		0x3002, 0
	};
	unsigned long *v;

	for (v = default_delimiters; *v != 0; v++) {
		r = idn__delimitermap_add(ctx->delimitermap, *v);
		if (r != idn_success)
			return (r);
	}

	return (idn_success);
}

/*
 * Destroy a resconf object.
 */
void
idn_resconf_destroy(idn_resconf_t ctx) {
	assert(ctx != NULL);

	TRACE(("idn_resconf_destroy()\n"));

	idn__localencoding_destroy(ctx->localencoding);
	idn__lang_destroy(ctx->lang);
	idn__maplist_destroy(ctx->maplist);
	idn__delimitermap_destroy(ctx->delimitermap);
	idn__langlocalmap_destroy(ctx->langlocalmap);
	idn__tldlocalmap_destroy(ctx->tldlocalmap);
	if (ctx->localcheck != NULL)
		idn__foreignset_destroy(ctx->localcheck);
	free(ctx);

	TRACE(("idn_resconf_destroy: the object is destroyed\n"));
}

/*
 * Set 'ctx' to default configuration.
 */
idn_result_t
idn_resconf_setdefaults(idn_resconf_t ctx) {
	idn_result_t r = idn_success;
	idn_resconf_t ctx2 = NULL;

	if (!ctx->need_reset)
		return (idn_success);
		
	r = idn_resconf_create(&ctx2);
	if (r != idn_success)
		return (r);

	idn__localencoding_destroy(ctx->localencoding);
	idn__lang_destroy(ctx->lang);
	idn__maplist_destroy(ctx->maplist);
	idn__delimitermap_destroy(ctx->delimitermap);
	idn__langlocalmap_destroy(ctx->langlocalmap);
	idn__tldlocalmap_destroy(ctx->tldlocalmap);
	if (ctx->localcheck != NULL)
		idn__foreignset_destroy(ctx->localcheck);

	ctx->localencoding        = ctx2->localencoding;
	ctx->lang                 = ctx2->lang;
	ctx->localconverter_flags = ctx2->localconverter_flags;
	ctx->maplist              = ctx2->maplist;
	ctx->delimitermap         = ctx2->delimitermap;
	ctx->langlocalmap         = ctx2->langlocalmap;
	ctx->tldlocalmap          = ctx2->tldlocalmap;
	ctx->localcheck           = ctx2->localcheck;
	ctx->need_reset           = 0;
	free(ctx2);

	return (idn_success);
}

/*
 * Set local encoding name.
 */
idn_result_t
idn_resconf_setlocalencoding(idn_resconf_t ctx, const char *name) {
	idn_result_t r = idn_success;

	assert(ctx != NULL);

	TRACE(("idn_resconf_setlocalencoding(name=\"%s\")\n",
	       idn__debug_xstring(name)));

	idn__localencoding_setname(ctx->localencoding, name);
	ctx->need_reset = 1;

	TRACE(("idn_resconf_setlocalencoding(): %s\n",
	       idn_result_tostring(r)));

	return (r);
}

/*
 * Get flags of the local converter.
 */
int
idn_resconf_getlocalconverterflags(idn_resconf_t ctx) {
	return (ctx->localconverter_flags);
}

/*
 * Set flags of the local converter.
 */
void
idn_resconf_setlocalconverterflags(idn_resconf_t ctx, int flags) {
	ctx->localconverter_flags = flags;
	ctx->need_reset = 1;
}

/*
 * Get the current language.
 */
const char *
idn_resconf_getlanguage(idn_resconf_t ctx) {
	assert(ctx != NULL);

	TRACE(("idn_resconf_getlanguage()\n"));
	return (idn__lang_getname(ctx->lang));
}

/*
 * Set language.
 */
idn_result_t
idn_resconf_setlanguage(idn_resconf_t ctx, const char *name) {
	idn_result_t r = idn_success;

	assert(ctx != NULL);

	TRACE(("idn_resconf_setlanguage(name=\"%s\")\n",
	       idn__debug_xstring(name)));

	idn__lang_setname(ctx->lang, name);
	idn__langlocalmap_setlang(ctx->langlocalmap, name);
	ctx->need_reset = 1;

	TRACE(("idn_resconf_setlanguage(): %s\n",
	       idn_result_tostring(r)));

	return (r);
}

/*
 * Get a 'foreignset' object for localcheck.
 */
idn__foreignset_t
idn__resconf_getlocalcheck(idn_resconf_t ctx) {
	return (ctx->localcheck);
}

/*
 * Set a 'foreignset' object for localcheck.
 */
idn_result_t
idn_resconf_setlocalcheckfile(idn_resconf_t ctx, const char *file) {
	idn_result_t r = idn_success;
	idn__foreignset_t localcheck = NULL;

	assert(ctx != NULL);

	TRACE(("idn_resconf_setlocalcheckfile(file=\"%s\")\n", 
	       idn__debug_xstring(file)));

	if (file != NULL) {
		r = idn__foreignset_create(&localcheck);
		if (r != idn_success)
			goto ret;
		r = idn__foreignset_addfromfile(localcheck, file);
		if (r != idn_success)
			goto ret;
	}

	free(ctx->localcheck);
	ctx->localcheck = localcheck;

ret:
	ctx->need_reset = 1;
	TRACE(("idn_resconf_setlocalcheckfile(): %s\n",
	       idn_result_tostring(r)));
	if (r != idn_success)
		free(localcheck);

	return (r);
}

/*
 * Set a 'maplist' object.
 */
idn__maplist_t
idn__resconf_getmaplist(idn_resconf_t ctx) {
	return (ctx->maplist);
}

/*
 * Open an user configuration file.
 */
static idn_result_t
open_userconffile(FILE **fpp) {
	char filepath[PATH_MAX + 1];

#ifdef WIN32
	{
		char dir[PATH_MAX + 1];

		if (!idn__util_win32getuserdirectory(dir, sizeof(dir)))
			return (idn_nofile);
		if (strlen(dir) + strlen(USER_CONF_FILE) + 1 > PATH_MAX)
			return (idn_nofile);
		sprintf(filepath, "%s/%s", dir, USER_CONF_FILE);
	}
#else
	{
		struct passwd *pw;

		pw = getpwuid(getuid());
		if (pw == NULL)
			return (idn_nofile);
		if (strlen(pw->pw_dir) + strlen(USER_CONF_FILE) + 1 > PATH_MAX)
			return (idn_nofile);
		sprintf(filepath, "%s/%s", pw->pw_dir, USER_CONF_FILE);
	}
#endif

	*fpp = fopen(filepath, "r");
	if (*fpp == NULL)
		return (idn_nofile);

	return (idn_success);
}

/*
 * Open a system configuration file.
 */
static idn_result_t
open_systemconffile(FILE **fpp) {
	char filepath[PATH_MAX + 1];

#ifdef WIN32
	{
		char dir[PATH_MAX + 1];

		if (!idn__util_win32getsysconfdir(dir, sizeof(dir)))
			return (idn_nofile);
		if (strlen(dir) + strlen(SYSTEM_CONF_FILE) + 1 > PATH_MAX)
			return (idn_nofile);
		sprintf(filepath, "%s/%s", dir, SYSTEM_CONF_FILE);
	}
#else
	{
		if (strlen(SYSCONFDIR) + strlen(SYSTEM_CONF_FILE) + 1
		    > PATH_MAX)
			return (idn_nofile);
		sprintf(filepath, "%s/%s", SYSCONFDIR, SYSTEM_CONF_FILE);
	}
#endif

	*fpp = fopen(filepath, "r");
	if (*fpp == NULL)
		return (idn_nofile);

	return (idn_success);
}

/*
 * Open a configuration file.
 */
static idn_result_t
open_conffile(const char *file, FILE **fpp) {
	idn_result_t r = idn_success;

	if (file == NULL || *file == '\0') {
		r = open_userconffile(fpp);
		if (r == idn_success) {
			TRACE(("open user configuraiton file\n"));
			return (r);
		}
		if (r != idn_nofile)
			return (r);

		r = open_systemconffile(fpp);
		if (r == idn_success) {
			TRACE(("open user configuraiton file\n"));
			return (r);
		}
		TRACE(("cannot open the default configuraiton file\n"));
		r = idn_nofile;
	} else {
		*fpp = fopen(file, "r");
		if (*fpp != NULL)
			return (idn_success);
		TRACE(("cannot open a configuraiton file: file=\"%s\"\n",
		       idn__debug_xstring(file)));
		r = idn_nofile;
	}

	return (r);
}

/*
 * Load a configuration file to 'ctx'.
 */
idn_result_t
idn_resconf_loadfile(idn_resconf_t ctx, const char *file) {
	idn_result_t r = idn_success;
	confparser_t cp;
	char line[MAX_CONF_LINE_LENGTH + 1];
	FILE *fp = NULL;

	assert(ctx != NULL);

	TRACE(("idn_resconf_loadfile(file=\"%s\")\n",
	       idn__debug_xstring(file)));

	r = idn_resconf_setdefaults(ctx);
	if (r != idn_success)
		goto ret;

	r = open_conffile(file, &fp);
	if (r != idn_success) {
		if ((file == NULL || *file == '\0') && r == idn_nofile)
			r = idn_success;
		goto ret;
	}

	confparser_init(&cp);
	while (fgets(line, sizeof(line), fp) != NULL) {
		char *newline;

		newline = strpbrk(line, "\r\n");
		if (newline != NULL)
			*newline = '\0';
		else if (fgetc(fp) != EOF) {
			ERROR(("libidnkit: too long line, line %d: \"%s\"\n",
			       cp.lineno + 1, idn__debug_xstring(line)));
			r = idn_invalid_syntax;
			goto ret;
		}
		r = parse_confline(ctx, &cp, line);
		if (r != idn_success)
			goto ret;
	}

ret:
	ctx->need_reset = 1;
	TRACE(("idn_resconf_loadfile(): %s\n", idn_result_tostring(r)));
	if (fp != NULL)
		fclose(fp);
	return (r);
}

/*
 * Load a configuration file to 'ctx'.
 */
idn_result_t
idn_resconf_loadstrings(idn_resconf_t ctx, char *strings[]) {
	idn_result_t r = idn_success;
	confparser_t cp;
	char **line;

	assert(ctx != NULL && strings != NULL);

	TRACE(("idn_resconf_loadstrings()\n"));

	r = idn_resconf_setdefaults(ctx);
	if (r != idn_success)
		goto ret;

	confparser_init(&cp);
	for (line = strings; *line != NULL; line++) {
		r = parse_confline(ctx, &cp, *line);
		if (r != idn_success)
			goto ret;
	}
	
ret:
	ctx->need_reset = 1;
	TRACE(("idn_resconf_loadstrings(): %s\n", idn_result_tostring(r)));
	return (r);
}

/*
 * Get the current local encoding name.
 */
const char *
idn_resconf_getlocalencoding(idn_resconf_t ctx) {
	assert(ctx != NULL);

	TRACE(("idn_resconf_getlocalencoding()\n"));
	return (idn__localencoding_getname(ctx->localencoding));
}

/*
 * Parse a configuration file.
 */
static idn_result_t
parse_confline(idn_resconf_t ctx, confparser_t *cp, const char *line) {
	idn_result_t r = idn_success;
	char workline[MAX_CONF_LINE_LENGTH + 1];
	char *argv[3];
	int argc;

	TRACE(("parse_confline(lineno=%d, line=\"%s\")\n", cp->lineno, line));

	cp->lineno++;
	if (*line == '#' || *line == '\0')
		goto ret;

	r = idn__util_strcpy(workline, sizeof(workline), line);
	if (r != idn_success) {
		ERROR(("libidnkit: too long line, line %d: \"%s\"\n",
		       cp->lineno, idn__debug_xstring(line)));
		r = idn_invalid_syntax;
		goto ret;
	}

	argc = split_args(workline, argv, 3);
	if (argc == -1) {
		ERROR(("libidnkit: syntax error, line %d\n", cp->lineno));
		r = idn_invalid_syntax;
		goto ret;
	} else if (argc == 0) {
		goto ret;
	}

	if (strcmp(argv[0], "language") == 0) {
		if (cp->language_count > 0) {
			ERROR(("libidnkit: 'language' redefined, line %d\n",
			       cp->lineno));
			r = idn_invalid_syntax;
			goto ret;
		}
		cp->language_count++;
		r = parse_language(ctx, argv[1], cp->lineno);
	} else if (strcmp(argv[0], "map") == 0) {
		if (cp->map_count > 0) {
			ERROR(("libidnkit: 'map' redefined, line %d\n",
			       cp->lineno));
			r = idn_invalid_syntax;
			goto ret;
		}
		cp->map_count++;
		r = parse_map(ctx, argv[1], cp->lineno);
	} else if (strcmp(argv[0], "delimiters") == 0) {
		if (cp->delimiters_count > 0) {
			ERROR(("libidnkit: 'delimiters' redefined, line %d\n",
			       cp->lineno));
			r = idn_invalid_syntax;
			goto ret;
		}
		cp->delimiters_count++;
		r = parse_delimiters(ctx, argv[1], cp->lineno);
	} else if (strcmp(argv[0], "local-map") == 0 ||
		   strcmp(argv[0], "tld-local") == 0) {
		r = parse_tld_local(ctx, argv[1], cp->lineno);
	} else if (strcmp(argv[0], "language-local") == 0) {
		r = parse_language_local(ctx, argv[1], cp->lineno);
	} else if (strcmp(argv[0], "tld-local") == 0) {
		r = parse_tld_local(ctx, argv[1], cp->lineno);
	} else if (strcmp(argv[0], "idn-encoding") == 0 ||
		   strcmp(argv[0], "nameprep") == 0) {
		WARNING(("libidnkit: obsolete command \"%s\" ignored, "
			 "line %d\n", argv[0], cp->lineno));
	} else {
		ERROR(("libidnkit: unknown command \"%s\", line %d\n",
		       argv[0], cp->lineno));
		r = idn_invalid_syntax;
	}
	if (r != idn_success)
		goto ret;

ret:
	TRACE(("parse_confline(lineno=%d, line=\"%s\"): %s\n",
		cp->lineno - 1, line, idn_result_tostring(r)));
	return (r);
}

/*
 * For parsing a configuration file, split a line into a list of arguments.
 */
static int
split_args(char *s, char **av, int max_ac) {
	int ac;
	int i;

	if (s == NULL) {
		for (i = 0; i < max_ac; i++)
			av[i] = NULL;
		return (0);
	}

	for (ac = 0; ac + 1 < max_ac && *s != '\0'; ac++) {
		if (ac > 0)
			*s++ = '\0';
		while (isspace((unsigned char)*s))
			s++;
		if (*s == '\0')
			break;
		if (*s == '"' || *s == '\'') {
			int qc = *s++;
			av[ac] = s;
			while (*s != qc) {
				if (*s == '\0')
					return (-1);
				s++;
			}
			*s++ = '\0';
		} else {
			av[ac] = s;
			while (*s != '\0' && !isspace((unsigned char)*s))
				s++;
		}
	}

	for (i = ac; i < max_ac; i++)
		av[i] = NULL;

	return (ac);
}

/*
 * Parse a 'language' entry of a configuration file.
 */
static idn_result_t
parse_language(idn_resconf_t ctx, char *args, int lineno) {
	idn_result_t r = idn_success;
	int argc;
	char *argv[MAX_CONF_LINE_ARGS + 1];

	if (args == NULL) {
		ERROR(("libidnkit: missing argument for \"language\", "
		       "line %d\n", lineno));
		return (idn_invalid_syntax);
	}

	argc = split_args(args, argv, MAX_CONF_LINE_ARGS + 1);
	if (argc != 1) {
		ERROR(("libidnkit: wrong number of arguments "
		       "for \"language\", line %d\n", lineno));
		return (idn_invalid_syntax);
	}

	r = idn_resconf_setlanguage(ctx, argv[0]);
	if (r != idn_success) {
		ERROR(("libidnkit: failed to set language, line %d: %s\n",
		       lineno, idn_result_tostring(r)));
		return (r);
	}

	TRACE(("parse_confline(): set language, line %d (lang=\"%s\")\n",
	       lineno, idn__debug_xstring(argv[0])));

	return (idn_success);
}

/*
 * Parse a 'map' entry of a configuration file.
 */
static idn_result_t
parse_map(idn_resconf_t ctx, char *args, int lineno) {
	idn_result_t r = idn_success;
	int argc;
	char *argv[MAX_CONF_LINE_ARGS + 1];
	idn__maplist_t maplist;
	int i;

	/*
	 * Clear the maplist.
	 */
	r = idn__maplist_create(&maplist);
	if (r != idn_success) {
		ERROR(("libidnkit: failed to clear a mapper, line %d: "
		       "%s\n", lineno, idn_result_tostring(r)));
		return (r);
	}
	idn__maplist_destroy(ctx->maplist);
	ctx->maplist = maplist;

	if (args == NULL)
		return (idn_success);
		
	argc = split_args(args, argv, MAX_CONF_LINE_ARGS + 1);
	if (argc < 0 || argc > MAX_CONF_LINE_ARGS) {
		ERROR(("libidnkit: wrong number of arguments "
		       "for \"map\", line %d\n", lineno));
		return (idn_invalid_syntax);
	}

	for (i = 0; i < argc; i++) {
		r = add_map(ctx, argv[i]);
		if (r != idn_success) {
			ERROR(("libidnkit: failed to add a mapper, line %d: "
			       "%s\n", lineno, idn_result_tostring(r)));
			return (r);
		}
		TRACE(("parse_confline(): add map, line %d (mapper=%s)\n",
		       lineno, argv[i]));
	}

	return (idn_success);
}

/*
 * Parse a 'delimiters' entry of a configuration file.
 */
static idn_result_t
parse_delimiters(idn_resconf_t ctx, char *args, int lineno) {
	idn_result_t r = idn_success;
	int argc;
	char *argv[MAX_CONF_LINE_ARGS + 1];
	int i;

	idn__delimitermap_clear(ctx->delimitermap);

	if (args == NULL)
		return (idn_success);
		
	argc = split_args(args, argv, MAX_CONF_LINE_ARGS + 1);
	if (argc < 0 || argc > MAX_CONF_LINE_ARGS) {
		ERROR(("libidnkit: wrong number of arguments for "
		       "\"delimiters\", line %d\n", lineno));
		return (idn_invalid_syntax);
	}

	if (r != idn_success) {
		ERROR(("libidnkit: failed to clear delimitermap, "
		       "line %d: %s\n", lineno, idn_result_tostring(r)));
		return (r);
	}
	
	for (i = 0; i < argc; i++) {
		char *arg = argv[i];
		char *endp;
		unsigned long v;

		if (*arg == 'U' && *(arg + 1) == '+')
			arg += 2;
		v = strtoul(arg, &endp, 16);
		if (!isxdigit(*arg) || endp - arg > 6 || *endp != '\0')
			return (idn_invalid_syntax);

		r = idn__delimitermap_add(ctx->delimitermap, v);
		if (r != idn_success) {
			ERROR(("libidnkit: failed to add a delimiter, "
			       "line %d: %s\n", lineno, 
				idn_result_tostring(r)));
			return (r);
		}
		TRACE(("parse_confline(): add delimiter, line %d "
		       "(codepoint=\\x%lx)\n", lineno, v));
	}

	return (idn_success);
}

/*
 * Parse a 'language-local' entry of a configuration file.
 */
static idn_result_t
parse_language_local(idn_resconf_t ctx, char *args, int lineno) {
	idn_result_t r = idn_success;
	int argc;
	char *argv[3];
	char *lang;
	char *file;

	if (args == NULL) {
		ERROR(("libidnkit: missing argument for \"language-local\", "
		       "line %d\n", lineno));
		return (idn_invalid_syntax);
	}
		
	argc = split_args(args, argv, 3);
	if (argc != 2) {
		ERROR(("libidnkit: wrong number of arguments for "
		       "\"language-local\", line %d\n", lineno));
		return (idn_invalid_syntax);
	}

	lang = argv[0];
	file = argv[1];
	r = idn__langlocalmap_addfromfile(ctx->langlocalmap, lang, file);
	if (r != idn_success) {
		ERROR(("libidnkit: failed to set a language localmap, "
		       "line %d: %s\n", lineno, idn_result_tostring(r)));
		return (r);
	}
	TRACE(("parse_confline(): add language-local, line %d (lang=\"%s\")\n",
	       lineno, idn__debug_xstring(lang)));

	return (r);
}

/*
 * Parse a 'tld-local' entry of a configuration file.
 */
static idn_result_t
parse_tld_local(idn_resconf_t ctx, char *args, int lineno) {
	idn_result_t r = idn_success;
	int argc;
	char *argv[3];
	char *tld;
	char *file;

	if (args == NULL) {
		ERROR(("libidnkit: missing argument for \"language-local\", "
		       "line %d\n", lineno));
		return (idn_invalid_syntax);
	}
		
	argc = split_args(args, argv, 3);
	if (argc != 2) {
		ERROR(("libidnkit: wrong number of arguments for "
		       "\"language-local\", line %d\n", lineno));
		return (idn_invalid_syntax);
	}

	tld = argv[0];
	file = argv[1];
	r = idn__tldlocalmap_addfromfile(ctx->tldlocalmap, tld, file);
	if (r != idn_success) {
		ERROR(("libidnkit: failed to set a TLD localmap, "
		       "line %d: %s\n", lineno, idn_result_tostring(r)));
		return (r);
	}
	TRACE(("parse_confline(): add tld-local, line %d (tld=\"%s\")\n",
	       lineno, idn__debug_xstring(tld)));

	return (r);
}

/*
 * Add a mapping procedure to 'ctx'.
 */
static idn_result_t
add_map(idn_resconf_t ctx, const char *mapname) {
	idn_result_t r = idn_success;

	assert(ctx != NULL && mapname != NULL);

	if (strcmp(mapname, IDN_RESCONF_NFCMAP_NAME) == 0) {
		r = idn__maplist_add(ctx->maplist, mapname, NULL, 
				     idn__normalizer_formc, NULL);
	} else if (strcmp(mapname, IDN_RESCONF_NFKCMAP_NAME) == 0) {
		r = idn__maplist_add(ctx->maplist, mapname, NULL,
				     idn__normalizer_formkc, NULL);
	} else if (strcmp(mapname, IDN_RESCONF_LOWERCASEMAP_NAME) == 0) {
		r = idn__maplist_add(ctx->maplist, mapname, ctx->lang,
				     idn__lowercasemap_map, NULL);
	} else if (strcmp(mapname, IDN_RESCONF_DELIMITERMAP_NAME) == 0) {
		r = idn__maplist_add(ctx->maplist, mapname, ctx->delimitermap,
				     idn__delimitermap_mapproc, NULL);
	} else if (strcmp(mapname, IDN_RESCONF_WIDTHMAP_NAME) == 0) {
		r = idn__maplist_add(ctx->maplist, mapname, NULL,
				     idn__widthmap_map, NULL);
	} else if (strcmp(mapname, IDN_RESCONF_TR46MAP_NAME) == 0) {
		r = idn__maplist_add(ctx->maplist, mapname, NULL,
				     idn__tr46map_map, NULL);
	} else if (strcmp(mapname, IDN_RESCONF_TR46DEVIATIONMAP_NAME) == 0) {
		r = idn__maplist_add(ctx->maplist, mapname, NULL,
				     idn__tr46map_deviationmap, NULL);
	} else if (strcmp(mapname, IDN_RESCONF_TR46CHECK_NAME) == 0) {
		r = idn__maplist_add(ctx->maplist, mapname, NULL,
				     idn__tr46check_map, NULL);
	} else if (strcmp(mapname, IDN_RESCONF_TR46DEVIATIONCHECK_NAME) == 0) {
		r = idn__maplist_add(ctx->maplist, mapname, NULL,
				     idn__tr46check_deviationmap, NULL);
	} else if (strcmp(mapname, IDN_RESCONF_LANGLOCALMAP_NAME) == 0) {
		r = idn__maplist_add(ctx->maplist, mapname, ctx->langlocalmap,
				     idn__langlocalmap_mapproc, NULL);
	} else if (strcmp(mapname, IDN_RESCONF_TLDLOCALMAP_NAME) == 0) {
		r = idn__maplist_add(ctx->maplist, mapname, ctx->tldlocalmap,
				     idn__tldlocalmap_mapproc, NULL);
	} else if (strcmp(mapname, IDN_RESCONF_RESMANIDNA2008MAP_NAME) == 0 ||
		   strcmp(mapname, IDN_RESCONF_RFC5895_NAME) == 0) {
		r = add_map(ctx, IDN_RESCONF_LOWERCASEMAP_NAME);
		if (r != idn_success)
			goto ret;
		r = add_map(ctx, IDN_RESCONF_WIDTHMAP_NAME);
		if (r != idn_success)
			goto ret;
		r = add_map(ctx, IDN_RESCONF_NFCMAP_NAME);
		if (r != idn_success)
			goto ret;
		r = add_map(ctx, IDN_RESCONF_DELIMITERMAP_NAME);
	} else if (strcmp(mapname, IDN_RESCONF_TR46PROCESSING_NAME) == 0) {
		r = add_map(ctx, IDN_RESCONF_TR46MAP_NAME);
		if (r != idn_success)
			goto ret;
		r = add_map(ctx, IDN_RESCONF_NFCMAP_NAME);
		if (r != idn_success)
			goto ret;
		r = add_map(ctx, IDN_RESCONF_TR46CHECK_NAME);
	} else if (strcmp(mapname, IDN_RESCONF_TR46DEVIATIONPROCESSING_NAME)
		   == 0) {
		r = add_map(ctx, IDN_RESCONF_TR46DEVIATIONMAP_NAME);
		if (r != idn_success)
			goto ret;
		r = add_map(ctx, IDN_RESCONF_NFCMAP_NAME);
		if (r != idn_success)
			goto ret;
		r = add_map(ctx, IDN_RESCONF_TR46DEVIATIONCHECK_NAME);
	} else {
		r = idn_invalid_name;
	}

ret:
	return (r);
}

static void
confparser_init(confparser_t *cp) {
	cp->lineno = 0;
	cp->language_count = 0;
	cp->map_count = 0;
	cp->delimiters_count = 0;
}
