// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/FastJets.hh"
#include "Rivet/Projections/ChargedFinalState.hh"
#include "Rivet/Projections/Thrust.hh"
#include "Rivet/Projections/Sphericity.hh"
#include "Rivet/Projections/ParisiTensor.hh"
#include "Rivet/Projections/Hemispheres.hh"
#include "Rivet/Projections/Beam.hh"

namespace Rivet {


  /// ALEPH jet rates and event shapes at LEP 1 and 2
  class ALEPH_2004_I636645 : public Analysis {
  public:

    RIVET_DEFAULT_ANALYSIS_CTOR(ALEPH_2004_I636645);


    void init() {
      // TODO: According to the paper they seem to discard neutral particles
      //       between 1 and 2 GeV. That correction is included in the systematic
      //       uncertainties and overly complicated to program, so we ignore it.
      const FinalState fs;
      declare(fs, "FS");
      FastJets durhamjets(fs, JetAlg::DURHAM, 0.7, JetMuons::ALL, JetInvisibles::ALL);
      declare(durhamjets, "DurhamJets");

      const Thrust thrust(fs);
      declare(thrust, "Thrust");
      declare(Sphericity(fs), "Sphericity");
      declare(ParisiTensor(fs), "Parisi");
      declare(Hemispheres(thrust), "Hemispheres");

      const ChargedFinalState cfs;
      declare(Beam(), "Beams");
      declare(cfs, "CFS");

      // Histos
      book(_p["mult"], 1, 1, 1);
      book(_p["totaljetbroadening"], 53, 1, 1);
      book(_p["widejetbroadening"],  53, 1, 2);
      book(_p["C"],                  53, 1, 3);
      book(_p["rho"],                53, 1, 4);
      book(_p["thrust"],             53, 1, 5);
      size_t ih=0, ik=0;
      for (const string& en : _p["mult"].binning().edges<0>()) {
        double eval = stod(en);
        if (isCompatibleWithSqrtS(eval))  _sqs = en;
        // offset for the charged particle distributions
        book(_s[en], "_sumW_"+en);
        book(_c[en], "_weightedTotalChargedPartNum_"+en);

        // event shapes
        if (en != "196.0"s) {
          book(_h[en+"thrust"],             ih+54, 1, 1);
          book(_h[en+"heavyjetmass"],       ih+62, 1, 1);
          book(_h[en+"totaljetbroadening"], ih+70, 1, 1);
          book(_h[en+"widejetbroadening"],  ih+78, 1, 1);
          book(_h[en+"cparameter"],         ih+86, 1, 1);
          book(_h[en+"thrustmajor"],        ih+94, 1, 1);
          book(_h[en+"thrustminor"],        ih+102, 1, 1);
          book(_h[en+"jetmassdifference"],  ih+110, 1, 1);
          book(_h[en+"aplanarity"],         ih+118, 1, 1);
          book(_h[en+"oblateness"],         ih+133, 1, 1);
          book(_h[en+"sphericity"],         ih+141, 1, 1);
          if (ih)  book(_h[en+"planarity"], ih+125, 1, 1);

          // Durham n->m jet resolutions
          book(_h[en+"y_Durham0"], ih+149, 1, 1);   // y12 d149 ... d156
          book(_h[en+"y_Durham1"], ih+157, 1, 1);   // y23 d157 ... d164
          if (ih < 6) { // there is no y34, y45 and y56 for 200 gev
            book(_h[en+"y_Durham2"], ih+165, 1, 1); // y34 d165 ... d172, but not 171
            book(_h[en+"y_Durham3"], ih+173, 1, 1); // y45 d173 ... d179
            book(_h[en+"y_Durham4"], ih+180, 1, 1); // y56 d180 ... d186
          }
          else if (ih == 7) {
            book(_h[en+"y_Durham2"], 172, 1, 1);
            book(_h[en+"y_Durham3"], 179, 1, 1);
            book(_h[en+"y_Durham4"], 186, 1, 1);
          }

          // Durham n-jet fractions
          book(_h[en+"R_Durham0"], ih+187, 1, 1); // R1 d187 ... d194
          book(_h[en+"R_Durham1"], ih+195, 1, 1); // R2 d195 ... d202
          book(_h[en+"R_Durham2"], ih+203, 1, 1); // R3 d203 ... d210
          book(_h[en+"R_Durham3"], ih+211, 1, 1); // R4 d211 ... d218
          book(_h[en+"R_Durham4"], ih+219, 1, 1); // R5 d219 ... d226
          book(_h[en+"R_Durham5"], ih+227, 1, 1); // R>=6 d227 ... d234

          ++ih;
        }
        if (en != "91.2"s) {
          book(_avg[en]["xp"],    2+ik, 1, 1);
          book(_avg[en]["xi"],   11+ik, 1, 1);
          book(_avg[en]["xe"],   19+ik, 1, 1);
          book(_avg[en]["pTin"], 27+ik, 1, 1);
          if (ik == 7) book(_avg[en]["pTout"], 35, 1, 1);
          book(_avg[en]["rapidityT"], 36+ik, 1, 1);
          book(_avg[en]["rapidityS"], 44+ik, 1, 1);
          ++ik;
        }
      }
      raiseBeamErrorIf(_sqs.empty());
    }


    void analyze(const Event& e) {

      _s[_sqs]->fill();
      const Thrust& thrust = apply<Thrust>(e, "Thrust");
      const Sphericity& sphericity = apply<Sphericity>(e, "Sphericity");
      const Hemispheres& hemi = apply<Hemispheres>(e, "Hemispheres");
      _p["thrust"]->fill(_sqs, 1.0-thrust.thrust());
      _p["totaljetbroadening"]->fill(_sqs, hemi.Bsum());
      _p["widejetbroadening"]->fill(_sqs, hemi.Bmax());
      _p["rho"]->fill(_sqs, hemi.scaledM2high());
      const ParisiTensor& parisi = apply<ParisiTensor>(e, "Parisi");
      _p["C"]->fill(_sqs, parisi.C());

      if (_sqs != "196.0"s) {
        const bool LEP1(_sqs == "91.2"s);
        // event shapes
        const double thr = LEP1 ? thrust.thrust() : 1.0 - thrust.thrust();
        _h[_sqs+"thrust"]->fill(thr);
        _h[_sqs+"thrustmajor"]->fill(thrust.thrustMajor());
        if (LEP1)  _h[_sqs+"thrustminor"]->fill(log(thrust.thrustMinor()));
        else       _h[_sqs+"thrustminor"]->fill(thrust.thrustMinor());
        _h[_sqs+"oblateness"]->fill(thrust.oblateness());

        _h[_sqs+"heavyjetmass"]->fill(hemi.scaledM2high());
        _h[_sqs+"jetmassdifference"]->fill(hemi.scaledM2diff());
        _h[_sqs+"totaljetbroadening"]->fill(hemi.Bsum());
        _h[_sqs+"widejetbroadening"]->fill(hemi.Bmax());
        _h[_sqs+"cparameter"]->fill(parisi.C());

        _h[_sqs+"aplanarity"]->fill(sphericity.aplanarity());
        if (!LEP1) _h[_sqs+"planarity"]->fill(sphericity.planarity());
        _h[_sqs+"sphericity"]->fill(sphericity.sphericity());

        // Jet rates
        const FastJets& durjet = apply<FastJets>(e, "DurhamJets");
        double log10e = log10(exp(1.0));
        if (durjet.clusterSeq()) {
          double logynm1=0.;
          double logyn;
          for (size_t i=0; i<5; ++i) {
            const double yn = durjet.clusterSeq()->exclusive_ymerge_max(i+1);
            if (yn<=0.0) continue;
            logyn = -log(yn);
            const string suff = "_Durham"+toString(i);
            if (_h.find(_sqs+"y"s+suff) != _h.end()) {
              _h[_sqs+"y"s+suff]->fill(logyn);
            }
            if (!LEP1) logyn *= log10e;
            for (const auto& b : _h[_sqs+"R"s+suff]->bins()) {
              const double xMin = b.xMin();
              if (-xMin <= logynm1)  break;
              if (-xMin<logyn)  _h[_sqs+"R"s+suff]->fill(b.xMid(), b.xWidth());
            }
            logynm1 = logyn;
          }
          for (const auto& b : _h[_sqs+"R_Durham5"]->bins()) {
            if (-b.xMin() <= logynm1)  break;
            _h[_sqs+"R_Durham5"]->fill(b.xMid(), b.xWidth());
          }
        }
      }

      const ChargedFinalState& cfs = apply<ChargedFinalState>(e, "CFS");
      const size_t numParticles = cfs.particles().size();
      _c[_sqs]->fill(numParticles);

      // charged particle distributions
      if (_sqs != "91.2"s) {
        const ParticlePair& beams = apply<Beam>(e, "Beams").beams();
        const double meanBeamMom = 0.5*(beams.first.p3().mod() + beams.second.p3().mod());
        for (const Particle& p : cfs.particles()) {
          const double xp = p.p3().mod()/meanBeamMom;
          _avg[_sqs]["xp"]->fill(xp);
          const double logxp = -std::log(xp);
          _avg[_sqs]["xi"]->fill(logxp);
          const double xe = p.E()/meanBeamMom;
          _avg[_sqs]["xe"]->fill(xe);
          const double pTinT  = dot(p.p3(), thrust.thrustMajorAxis());
          const double pToutT = dot(p.p3(), thrust.thrustMinorAxis());
          _avg[_sqs]["pTin"]->fill(fabs(pTinT/GeV));
          if (_sqs == "206.0"s) _avg[_sqs]["pTout"]->fill(fabs(pToutT/GeV));
          const double momT = dot(thrust.thrustAxis(), p.p3());
          const double rapidityT = 0.5 * std::log((p.E() + momT) / (p.E() - momT));
          _avg[_sqs]["rapidityT"]->fill(fabs(rapidityT));
          const double momS = dot(sphericity.sphericityAxis(), p.p3());
          const double rapidityS = 0.5 * std::log((p.E() + momS) / (p.E() - momS));
          _avg[_sqs]["rapidityS"]->fill(fabs(rapidityS));
        }
      }
    }

    void finalize() {
      scale(_h, crossSectionPerEvent());
      scale(_c, crossSectionPerEvent());
      scale(_s, crossSectionPerEvent());
      for (auto& item : _h) {
        size_t pos = item.first.find("Durham");
        if (pos != string::npos) {
          const string en = item.first.substr(0,pos-2);
          scale(item.second, 1.0/dbl(*_s[en]));
        }
        else {
          normalize(item.second);
        }
      }
      for (const auto& item : _c) {
        const double avgNumParts = dbl(*item.second) /dbl(*_s[item.first]);
        if (!avgNumParts)  continue;
        _p["mult"]->fill(item.first, avgNumParts);
        normalize(_avg[item.first], avgNumParts);
      }
    }


  private:

    map<string,BinnedProfilePtr<string>> _p;
    map<string,map<string,Histo1DPtr>> _avg;
    map<string,Histo1DPtr> _h;
    map<string,CounterPtr> _c, _s;
    string _sqs = "";
  };



  RIVET_DECLARE_ALIASED_PLUGIN(ALEPH_2004_I636645, ALEPH_2004_S5765862);

}
