/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */


#ifndef COMMON_H
#define COMMON_H    1

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gtk/gtk.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <rtl-sdr.h>
#include <semaphore.h>
#include <pthread.h>
#include "sound.h"

/* Color map data */
typedef struct
{
  /* Pseudo-colorization maps */
  unsigned char
    gray_from[6],   gray_to[6],
    red_from[6],    red_to[6],
    green_from[6],  green_to[6],
    blue_from[6],   blue_to[6];

  int
    white_thld,  /* White (clouds) threshold */
    num_ranges;  /* Number of grayscale ranges */

} color_map_t;

/* Runtime config data */
typedef struct
{
  uint32_t
    rtlsdr_dev_index,   /* librtlsdr Device index  */
    rtlsdr_center_freq; /* Center Frequency for SDR Tuner */

  int
    rtlsdr_lpf_bw,      /* Bandwidth in Hz of the Butterworth LPF */
    rtlsdr_freq_corr;   /* Correction factor in ppm for synthesizer */

  char
    pcm_dev[32],  /* PCM device name */
    cap_src[32],  /* Capture source  */
    cap_vol[32],  /* Capture volume  */
    mst_vol[32],  /* Master volume control */
    pcm_vol[32];  /* Playback volume control */

  int
    channel,    /* ALSA "channel" for use in mixer setup */
    cap_lev;    /* Recording/Capture level */

  guint
    use_chn,        /* Channel in use (frontleft=0, frontright=1 etc) */
    snd_buf_size,   /* Size of sound card DSP buffer */
    num_chn;        /* Number of signal channels (2=stereo, 1=mono) */

  /* xwxapt working directory and glade file */
  char xwxapt_dir[64], xwxapt_glade[64];

  int
    window_height,  /* Set default widow height, in sec of image decoding */
    default_dur;    /* Default duration in sec of image decoding */

  int
    ifft_scale,    /* Scale factor to keep IFFT data in range */
    ifft_decimate; /* FFT stride (decimation) */

  /* Color map data */
  color_map_t
    noaa_A_map,
    meteor_map;

} rc_data_t;

/* Actions for above */
enum
{
  INIT_BWORTH_LPF = 0,
  RTL_DAGC_ENABLE
};

typedef struct
{
  double
    cutoff, /* Cutoff frequency as fraction of sample rate */
    ripple; /* Passband ripple as a percentage */

  int
    npoles, /* Number of poles, must be even */
    type;   /* Filter type as below */

  /* a and b Coefficients of the filter */
  double *a, *b;

  /* Saved input and output values */
  double *x, *y;

  /* Ring buffer index for above */
  int ring_idx;

  /* Input samples buffer and its length */
  double *samples_buf;
  int samples_buf_len;

} filter_data_t;

/* Filter type for above struct */
enum
{
  FILTER_LOWPASS = 0,
  FILTER_HIGHPASS,
  FILTER_BANDPASS
};

#define FILTER_RIPPLE   5.0
#define FILTER_POLES    6

/* Return values */
#define ERROR       1
#define SUCCESS     0

/* Generel definitions for image processing */
#define MAX_FILE_NAME      255 /* Max length for optional filenames */

/* Size of char arrays (strings) for error messages etc */
#define MESG_SIZE   128

/* Flow control flags */
#define ICON_SYNC_LOCK          0x000001 /* Sync detector locked icon     */
#define ICON_SYNC_UNLOCK        0x000002 /* Sync detector unlocked icon   */
#define ICON_SYNC_SEARCH        0x000004 /* Sync detector searching icon  */
#define ICON_FLAGS_ALL          0x000007 /* All icon flags (for clearing) */
#define ACTION_PROCESS_DSP      0x000008 /* Process images from dsp       */
#define ACTION_PROCESS_FILE     0x000010 /* Process images from file      */
#define ACTION_RECORD_APT       0x000020 /* Record apt signal to file     */
#define ACTION_SETUP_AUDIO      0x000040 /* Run signal level setup func.  */
#define ACTION_SYNC_BUFFER      0x000080 /* Run buffer synchronizing func */
#define ACTION_RATE_ERROR       0x000100 /* Measure dsp sampling rate err */
#define ACTION_FLAGS_ALL        0x0001f8 /* All action flags (clearing)   */
#define ACTION_PENDING          0x000200 /* Some actio is pending, abort  */
#define IMAGE_NORMALIZE         0x000400 /* Histogram normalize wx image  */
#define IMAGE_COLORIZE          0x000800 /* Pseudo Colorize wx image      */
#define IMAGE_ROTATE            0x001000 /* Rotate wx image 180 degrees   */
#define CAPTURE_SETUP           0x002000 /* Sound Capture has been set-up */
#define MIXER_SETUP             0x004000 /* Sound Mixer has been set-up   */
#define START_STOP_TIMER        0x008000 /* Start-Stop timer is set       */
#define RTLSDR_INIT             0x010000 /* RTLSDR device initialized OK  */
#define USE_RTLSDR_RX           0x020000 /* Use RTLSDR to receive images  */
#define ENABLE_DOPPLER_CORR     0x040000 /* Enable Doppler shift correct. */
#define CARRIER_SPECTRUM        0x080000 /* Display RF carrier spaectrum  */
#define TUNER_GAIN_MANL         0x100000 /* Enable Auto Tuner Gain Control */
#define ENABLE_SCOPE            0x200000 /* Enable Signal scope display */

/* Should have been in math.h */
#ifndef M_2PI
  #define M_2PI     6.28318530717958647692
#endif

/*
 * Standard gettext macros.
 */
#ifdef ENABLE_NLS
#  include <libintl.h>
#  undef _
#  define _(String) dgettext (PACKAGE, String)
#  define Q_(String) g_strip_context ((String), gettext (String))
#  ifdef gettext_noop
#    define N_(String) gettext_noop (String)
#  else
#    define N_(String) (String)
#  endif
#else
#  define textdomain(String) (String)
#  define gettext(String) (String)
#  define dgettext(Domain,Message) (Message)
#  define dcgettext(Domain,Message,Type) (Message)
#  define bindtextdomain(Domain,Directory) (Domain)
#  define _(String) (String)
#  define Q_(String) g_strip_context ((String), (String))
#  define N_(String) (String)
#endif

/*---------------------------------------------------------------------*/

/* Function prototypes produced by cproto */
/* aptsignal.c */
int Detect_NOAA_Sync(int search_limit);
int Detect_Meteor_Sync(int search_limit);
int Detect_Sync_Train(int search_limit);
gboolean Init_Reception(void);
gboolean Setup_Signal_Level(gpointer data);
gboolean Record_APT_Signal(gpointer data);
gboolean File_Fill_Buffer(void);
gboolean Read_Device_Buffer(short *buffer, int buff_size);
gboolean Fill_Samples_Buffer(void);
gboolean Synchronize_Buffer(gpointer data);
void Carrier_Amplitude(short sample, int *amplitude);
void Set_Sync_Icon(gchar *icon_name);
/* callback_func.c */
void Error_Dialog(void);
void File_Chooser(void);
gboolean Cancel_Timer(gpointer data);
void Sensitize_Menu_Item(gchar *item_name, gboolean flag);
void Popup_Menu(void);
void Receive_Togglebutton_Toggled(GtkToggleButton *togglebutton);
void Filechooser_Response(gint response_id);
void Start_Stop_OK_Clicked(GtkButton *button);
void Enter_Center_Freq(uint32_t freq);
void Timeout_Setup(GtkButton *button);
void Hours_Entry(GtkEditable *editable);
void Minutes_Entry(GtkEditable *editable);
void Fft_Drawingarea_Size_Alloc(GtkAllocation *allocation);
void BW_Entry_Activate(GtkEntry *entry);
/* callbacks.c */
void on_main_window_destroy(GObject *object, gpointer user_data);
gboolean on_main_window_delete_event(GtkWidget *widget, GdkEvent *event, gpointer user_data);
gboolean on_viewport_button_press_event(GtkWidget *widget, GdkEventButton *event, gpointer user_data);
void on_receive_togglebutton_toggled(GtkToggleButton *togglebutton, gpointer user_data);
void on_normalize_togglebutton_toggled(GtkToggleButton *togglebutton, gpointer user_data);
void on_colorize_togglebutton_toggled(GtkToggleButton *togglebutton, gpointer user_data);
void on_invert_togglebutton_toggled(GtkToggleButton *togglebutton, gpointer user_data);
void on_level_setup_togglebutton_toggled(GtkToggleButton *togglebutton, gpointer user_data);
void on_operation_timer_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_start_stop_timer_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_cancel_timer_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_decode_from_dsp_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_record_to_file_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_decode_from_file_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_stop_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_quit_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_filechooser_response(GtkDialog *dialog, gint response_id, gpointer user_data);
void on_filechooser_destroy(GObject *object, gpointer user_data);
void on_timer_okbutton_clicked(GtkButton *button, gpointer user_data);
gboolean on_error_dialog_delete_event(GtkWidget *widget, GdkEvent *event, gpointer user_data);
void on_error_dialog_destroy(GObject *object, gpointer user_data);
void on_quit_dialog_destroy(GObject *object, gpointer user_data);
void on_timer_dialog_destroy(GObject *object, gpointer user_data);
void on_startstop_timer_destroy(GObject *object, gpointer user_data);
void on_error_ok_button_clicked(GtkButton *button, gpointer user_data);
void on_error_quit_button_clicked(GtkButton *button, gpointer user_data);
void on_hrs_entry_changed(GtkEditable *editable, gpointer user_data);
void on_min_entry_changed(GtkEditable *editable, gpointer user_data);
void on_startstop_cancel_clicked(GtkButton *button, gpointer user_data);
void on_startstop_ok_clicked(GtkButton *button, gpointer user_data);
void on_quit_cancel_button_clicked(GtkButton *button, gpointer user_data);
void on_quit_button_clicked(GtkButton *button, gpointer user_data);
void on_noaa15_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_noaa18_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_noaa19_activate(GtkMenuItem *menuitem, gpointer user_data);
void on_meteor_activate(GtkMenuItem *menuitem, gpointer user_data);
gboolean on_ifft_drawingarea_draw(GtkWidget *widget, cairo_t *cr, gpointer user_data);
gboolean on_demod_drawingarea_draw(GtkWidget *widget, cairo_t *cr, gpointer user_data);
void on_gain_hscale_value_changed(GtkRange *range, gpointer user_data);
void on_auto_checkbutton_toggled(GtkToggleButton *togglebutton, gpointer user_data);
void on_offset_checkbutton_toggled(GtkToggleButton *togglebutton, gpointer user_data);
void on_bw_entry_activate(GtkEntry *entry, gpointer user_data);
void on_freq_entry_activate(GtkEntry *entry, gpointer user_data);
void on_dagc_checkbutton_toggled(GtkToggleButton *togglebutton, gpointer user_data);
void on_subcarrier_radiobutton_toggled(GtkToggleButton *togglebutton, gpointer user_data);
void on_carrier_radiobutton_toggled(GtkToggleButton *togglebutton, gpointer user_data);
/* display.c */
int IFFT_Bin_Value(int sum_i, int sum_q, gboolean reset);
void Display_Waterfall(void);
void Display_Signal(int plot);
void Draw_Signal(cairo_t *cr);
/* filters.c */
gboolean Init_Chebyshev_Filter(filter_data_t *filter_data);
void Enter_Filter_BW(void);
void DSP_Filter(filter_data_t *filter_data);
/* ifft.c */
void Initialize_IFFT(int16_t width);
void IFFT(int16_t *data);
void IFFT_Real(int16_t *data);
void IFFT_Data(short sample);
/* image.c */
void NOAA_Image_Line(unsigned char *image_line, int *max_gray_val);
void Meteor_Image_Line(unsigned char *image_line, int *max_gray_val);
gboolean Dsp_Process_Image(gpointer data);
gboolean File_Process_Image(gpointer data);
void Normalize(unsigned char *image_buffer, int image_size);
void Rotate(unsigned char *image_buffer, int image_size);
void Colorize(unsigned char *image_buffer_a, unsigned char *image_buffer_b, int image_size, unsigned char *colorized_buffer, int map_type);
void Halfsize_Image(unsigned char *buffer, int width, int height, int offset, gboolean color);
void Halfsize_Row(unsigned char *buffer, int width, int line_cnt, int offset);
/* interface.c */
GtkWidget *Builder_Get_Object(GtkBuilder *builder, gchar *name);
GtkWidget *create_main_window(GtkBuilder **builder);
GtkWidget *create_popup_menu(GtkBuilder **builder);
GtkWidget *create_filechooser(GtkBuilder **builder);
GtkWidget *create_timer_dialog(GtkBuilder **builder);
GtkWidget *create_error_dialog(GtkBuilder **builder);
GtkWidget *create_startstop_timer(GtkBuilder **builder);
GtkWidget *create_quit_dialog(GtkBuilder **builder);
/* main.c */
int main(int argc, char *argv[]);
void Initialize_Top_Window(void);
/* rtlsdr.c */
void Rtl_Cb(unsigned char *buf, uint32_t len, void *ctx);
void *RtlSdr_Read_Async(void *pid);
gboolean Rtlsdr_Set_Center_Freq(uint32_t center_freq);
gboolean Get_Tuner_Gain(void);
gboolean Set_Tuner_Gain_Mode(int mode);
gboolean Set_Tuner_Gain(double gain);
gboolean Rtlsdr_Set_Agc_Mode(int mode);
gboolean Rtlsdr_Init(void);
gboolean Read_RTL_Buffer(short *buffer, int buff_size);
void Close_RTL_Device(void);
/* shared.c */
/* sound.c */
gboolean Open_PCM(snd_pcm_t **handle, snd_pcm_stream_t stream, char *mesg, int *error);
gboolean Open_Capture(char *mesg, int *error);
gboolean Open_Mixer(char *mesg, int *error);
gboolean Set_Capture_Level(int level, char *mesg, int *error);
void Close_Capture(void);
void Close_Mixer(void);
gboolean Read_SND_Buffer(short *buffer, int buff_size);
gboolean Xrun_Recovery(snd_pcm_t *handle, int error);
/* utils.c */
gboolean Load_Config(gpointer data);
int Load_Line(char *buff, FILE *pfile, char *messg);
void File_Name(char *file_name);
char *Fname(char *fpath);
void Usage(void);
void Show_Message(char *mesg, char *attr);
gboolean mem_alloc(void **ptr, size_t req);
gboolean mem_realloc(void **ptr, size_t req);
void free_ptr(void **ptr);
gboolean Open_File(FILE **fp, char *fname, const char *mode);
gboolean File_Image(FILE *fp, const char *type, int width, int height, int max_val, unsigned char *buffer);
void Cleanup(void);
int isFlagSet(int flag);
int isFlagClear(int flag);
void SetFlag(int flag);
void ClearFlag(int flag);
void ToggleFlag(int flag);
void Strlcpy(char *dest, const char *src, size_t n);
void Strlcat(char *dest, const char *src, size_t n);

#endif

