package aQute.rendezvous;

import java.io.*;
import java.net.*;
import java.util.*;

class Query {
	long		time;
	InetAddress	address;
	int			port;
	int			identity;

}

public class DNS extends Thread {
	int				port		= 53;
	boolean			keepgoing	= true;
	int				n;
	String			labels[]	= new String[64];

	Query			transit[]	= new Query[64];
	int				rover;
	String			host		= "172.25.25.10";
	InetAddress		forwardAddress;
	DatagramSocket	socket		= null;

	Vector			nss			= new Vector();

	public DNS() {
		for (int i = 0; i < transit.length; i++)
			transit[i] = new Query();
	}

	public DNS(String host, int port) {
		this();
		this.host = host;
		this.port = port;
	}

	public void setHost(String host) {
		this.host = host;
		setForward(host);
	}

	public void run() {
		System.out.println("Starting DNS");
		while (keepgoing)
			try {
				forwardAddress = InetAddress.getByName(host);
				System.out.println("DNS forward " + forwardAddress);
				listen();
			}
			catch (Exception e) {
				e.printStackTrace();
			}
			finally {
				System.out.println("Exiting DNS");
			}
	}

	void listen() throws Exception {
		int n = 5;
		while (keepgoing)
			try {
				socket = new DatagramSocket(port);
				try {
					while (keepgoing) {
						DatagramPacket packet = new DatagramPacket(
								new byte[1600], 1600);
						socket.receive(packet);
						n = 1;
						byte data[] = packet.getData();

						int flags = getUnsignedShort(data, 2);
						DatagramPacket reply;

						if ((flags & 0x80) != 0) // Reply/Query flag
						{
							reply = response(packet);
						}
						else {
							System.out.println("Get a request");
							reply = parse(packet);
							if (reply == null) {
								reply = forward(packet);
							}
						}

						if (reply != null)
							socket.send(reply);
					}
				}
				finally {
					socket.close();
				}
			}
			catch (BindException eee) {
				System.out.println("Cannot bind to dns address");
				try {
					Thread.sleep(1000 * n++);
				}
				catch (Exception ee) {
				}
			}
	}

	public int getPort() {
		return port;
	}

	public String getForward() {
		return forwardAddress.getHostName();
	}

	public void setForward(String forward) {
		try {
			forwardAddress = InetAddress.getByName(forward);
		}
		catch (Exception e) {
		}
	};

	DatagramPacket parse(DatagramPacket packet) throws Exception {
		ByteArrayInputStream bin = new ByteArrayInputStream(packet.getData());
		DataInputStream in = new DataInputStream(bin);

		int identification = in.readUnsignedShort();
		int flags = in.readUnsignedShort();
		int questions = in.readUnsignedShort();
		int answers = in.readUnsignedShort();
		int authorityRR = in.readUnsignedShort();
		int additionalRR = in.readUnsignedShort();

		System.out.println("Identification " + identification);
		System.out.println("Flags          " + Integer.toHexString(flags));
		System.out.println("Questions      " + questions);
		System.out.println("Answers        " + answers);
		System.out.println("Authority RRs  " + authorityRR);
		System.out.println("Additional RR  " + additionalRR);

		Vector compression = new Vector();

		for (int i = 0; i < questions; i++) {
			String query = getName(compression, in);

			int qtype = in.readUnsignedShort();
			int qclass = in.readUnsignedShort();

			if (qclass == 1) {
				byte[] data = null;

				if (qtype == 1)
					data = bind(identification, query);
				else if (qtype == 0x20)
					data = netbios(identification, query);
				else {
					return null;
				}

				if (data != null)
					return new DatagramPacket(data, data.length, packet
							.getAddress(), packet.getPort());
			}
			return null;
		}
		return null;
	}

	byte[] bind(int identification, String query) throws Exception {
		byte[] address = map(query);
		if (address == null)
			return null;

		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		DataOutputStream out = new DataOutputStream(bout);

		out.writeShort(identification);
		out.writeShort(0x8580); // Flags
		out.writeShort(1); // # of questions
		out.writeShort(1); // # of answers
		out.writeShort(0); // # of authorities
		out.writeShort(0); // # of additional

		// Query
		setName(query, out);
		out.writeShort(1);
		out.writeShort(1);

		// Answer
		setName(query, out); // Domain name
		out.writeShort(1); // Type
		out.writeShort(1); // Class
		out.writeInt(10); // TTL
		out.writeShort(address.length); // Length

		out.write(address);
		out.close();

		return bout.toByteArray();
	}

	byte[] netbios(int identification, String original) throws Exception {

		String query = convert(original);
		byte[] address = map(query);
		if (address == null)
			return null;

		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		DataOutputStream out = new DataOutputStream(bout);

		out.writeShort(identification);
		out.writeShort(0x8180); // Flags
		out.writeShort(0); // # of questions
		out.writeShort(1); // # of answers
		out.writeShort(0); // # of authorities
		out.writeShort(0); // # of additional

		// Answer
		setName(original, out); // Domain name
		out.writeShort(0x20); // Type
		out.writeShort(1); // Class
		out.writeInt(10); // TTL
		out.writeShort(address.length + 2); // Length
		out.write(new byte[] {0, 0});
		out.write(address);
		out.close();

		return bout.toByteArray();
	}

	/**
	 * Convert a NETBIOS name into a domain name.
	 */

	String convert(String name) {
		StringBuffer out = new StringBuffer();
		for (int i = 0; i < name.length() / 2; i += 2) {
			char c = (char) (((name.charAt(i * 2) - 'A') << 4) + (name
					.charAt(i * 2 + 1) - 'A'));
			out.append(c);
		}
		return out.toString().trim();
	}

	void setName(String name, DataOutputStream out) throws IOException {
		StringTokenizer st = new StringTokenizer(name, ".");
		while (st.hasMoreTokens()) {
			String s = st.nextToken();
			byte data[] = s.getBytes();
			out.write(data.length);
			out.write(data);
		}
		out.write(0);
	}

	String getName(Vector compression, DataInputStream in) throws IOException {
		String del = "";
		StringBuffer sb = new StringBuffer();

		while (true) {
			String s;

			int l = in.readByte() & 0xFF;
			if (l == 0)
				break;

			if (l >= 192)
				s = (String) compression.elementAt(l - 192);
			else {
				byte buffer[] = new byte[l];
				in.read(buffer, 0, l);
				s = new String(buffer, 0, l);
				compression.addElement(s);
			}
			sb.append(del);
			sb.append(s);
			del = ".";
		}
		return sb.toString();
	}

	DatagramPacket forward(DatagramPacket packet) throws Exception {
		int index = alloc();
		Query query = transit[index];
		query.identity = getUnsignedShort(packet.getData(), 0);
		query.address = packet.getAddress();
		query.port = packet.getPort();

		byte[] data = getCopy(packet);
		setShort(data, 0, index);
		return new DatagramPacket(data, data.length, forwardAddress, 53);
	}

	public synchronized int alloc() {
		long time = transit[0].time;
		int result = 0;

		for (int i = 0; i < 64; i++) {
			Query q = transit[rover];
			if (q.time == 0) {
				result = rover;
				break;
			}

			if (q.time < time) {
				result = rover;
				time = q.time;
			}
			rover = (rover + 1) % transit.length;
		}
		transit[result].time = System.currentTimeMillis();
		return result;
	}

	DatagramPacket response(DatagramPacket packet) throws Exception {
		int index = getUnsignedShort(packet.getData(), 0);
		Query query = transit[index];

		byte data[] = getCopy(packet);

		setShort(data, 0, query.identity);
		return new DatagramPacket(data, data.length, query.address, query.port);
	}

	byte[] getCopy(DatagramPacket packet) {
		byte data[] = new byte[packet.getLength()];
		System.arraycopy(packet.getData(), 0, data, 0, data.length);
		return data;
	}

	public int getUnsignedShort(byte data[], int index) {
		return (data[index] & 0xFF) + (data[index + 1] & 0xFF) * 256;
	}

	public void setShort(byte data[], int index, int value) {
		data[index] = (byte) value;
		data[index + 1] = (byte) (value >> 8);
	}

	public void addNameServer(NS ns) {
		nss.addElement(ns);
	}

	public void removeNameServer(NS ns) {
		nss.removeElement(ns);
	}

	byte[] map(String query) {
		System.out.println("Query " + query);
		synchronized (nss) {
			for (Enumeration e = nss.elements(); e.hasMoreElements();) {
				NS ns = (NS) e.nextElement();
				byte[] address = ns.map(query);
				if (address != null)
					return address;
			}
		}
		return null;
	}

	public static void main(String args[]) {
		int port = 53;

		for (int i = 0; i < args.length; i++) {
			if (args[i].equals("-port"))
				port = Integer.parseInt(args[++i]);
			else
				System.out.println("Invalid option " + args[i]);
		}

		DNS dns = new DNS("tiuri", port);
		dns.addNameServer(new NS() {
			public byte[] map(String name) {
				try {
					System.out.println("map '" + name + "'");
					if (name.equalsIgnoreCase("MIEKE"))
						return InetAddress.getByName("rapschranzer")
								.getAddress();
				}
				catch (Exception e) {
				}
				return null;
			}
		});

		dns.run();

	}

	public void close() {
		keepgoing = false;
		if (socket != null)
			socket.close();
	}

}
