#!/bin/sh
# Description: connect.sh is a script for non-interactive dialing for dwun.
#
# Usage: connect.sh [open|close|run|list] [session|command]"
#
# For example, to get a file from a machine on the internet we can do
#
#	connect.sh run "wget http://foo.invalid/blob.tar.gz"
#
# Or,
#	connect.sh open foo
#	wget http://foo.invalid/blob.tar.gz
#	connect.sh close foo
#
# Here "foo" is the name of the session. Sessions are only needed with
# open and close. They allow us to have as many independent open/close
# operations as we like.
#
# If you don't supply a session name, then the name "default" will be used.
#
# To get a list of all currently open sessions, run
#	connect.sh list
#
# If no arguments are given to connect.sh, we will just run dial.

PIDDIR="$HOME"/.dial

err() { echo "$*" >&2 ; }

DO="$1"

if [ -z "$DO" ]; then
	exec dial
fi

# Backwards compatibility
# connect.sh used to have "foo open" and "foo close"
# If user does "run open" or "run close", then we go with the new behaviour
# anyway. Hope no-one uses the session name "run", "open" or "close"!
if [ -n "$1" ] && [ -n "$2" ] && [ -z "$3" ]; then
	if [ "$2" = "open" ] || [ "$2" = "close" ]; then
		if ! [ "$DO" = "run" ] && ! [ "$DO" = "open" ] && ! [ "$DO" = "close" ]; then
			DO="$2"
			SESSION="$1"
		fi
	fi
fi
#ends

if [ "$DO" = "list" ]; then
	cd "$PIDDIR"
	for i in "`ls`"
		do echo "$i" | sed -n 's/^connect-\(.*\)\.pid$/\1/p'
	done
	exit 0
fi

if [ "$DO" = "run" ]; then
	COMMAND="$2"
	if [ -z "$COMMAND" ]; then
		err "Need to specify a command to run"
		exit 1
	fi
	exec dial -c "$COMMAND; kill \$DIAL_PID"
fi

# Backwards compatibility
if [ -z "$SESSION" ]; then
	SESSION="$2"
fi
# Ends
if [ -z "$SESSION" ]; then
	SESSION=default
fi
PIDFILE="$PIDDIR"/connect-"$SESSION".pid

# uck!
if [ "$DO" = "open" ]; then
	if [ -e "$PIDFILE" ]; then
		err "Session '$SESSION' session is already open or stale lockfile $PIDFILE"
		exit 1
	fi
	if [ ! -d "$PIDDIR" ]; then
		err "Creating session directory $PIDDIR/"
		mkdir "$PIDDIR" || err "failed"; exit 1
	fi
	export PIDFILE SESSION
	trap 'kill "$!" 2>/dev/null; exit 1' SIGINT SIGTERM
/bin/sh << 'END_OF_OPEN' &
	abort() { kill '$DIALPID' 2>/dev/null; exit 1; }
	trap abort SIGINT SIGTERM
	dial -c "kill $$" &
	DIALPID=$!
	echo "$DIALPID" > "$PIDFILE" || ( err "Creating $PIDFILE failed";
					  exit 1 )
	wait $DIALPID
	/bin/rm -f "$PIDFILE"
	exit 1 # dial exited before we got SIGTERM
END_OF_OPEN
	SUB=$!
	trap "/bin/rm -f $PIDFILE; exit 1" SIGINT SIGTERM
	wait $SUB 2>/dev/null
	STATUS=$?
	#if [ $STATUS = 143 ]; then
	#	exit 0 # 128 + 15 (SIGTERM) = 143
	#fi
	#\rm -f "$PIDFILE"
	#exit $STATUS
	exit 0
fi

if [ "$DO" = "close" ]; then
	if [ ! -e "$PIDFILE" ]; then
		err "Session '$SESSION' is not open"
		exit 1
	fi
        kill `cat "$PIDFILE"`
	STATUS=$?
	/bin/rm -f "$PIDFILE"
        exit "$STATUS"
fi

err "Usage: $0 [open|close|run|list] [session|command]"
exit 1
