#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>
#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif
#include <errno.h>
#include "compat.h"
#include "rcfile.h"

#define MAXLINE 1024

static int mksettings(FILE *fp, const char *filename, int *port, char **user, 
		char **pass, char **server, int *keep_alive_secs);
static int getsetting(FILE *fp, char **argument);
static int getsetting_check(char *buf);
static int check_perms(const char *filename);

enum {
	RC_INVALID,
	RC_EOF,
	RC_FATAL,
	RC_COMMENT,
	RC_PORT,
	RC_SERVER,
	RC_KEEPALIVE,
	RC_USER,
	RC_PASS
};

int get_rcfile_settings(const char *filename, int rcfile_must_exist, int *port, 
		char **user, char **pass, char **server, int *keep_alive_secs)
{
	FILE *fp;

	if (!filename)
		return 0; /* no rcfile */
	if (!(fp = fopen(filename, "r"))) {
		if (!rcfile_must_exist)
			return 0;
		fprintf(stderr, "Opening rcfile '%s' failed: %s\n", filename, 
				strerror(errno));
		return 1;
	}
	if (mksettings(fp, filename, port, user, pass, server,
				keep_alive_secs)) {
		fclose(fp);
		return 1;
	}
	fclose(fp);
	return 0;
}
	
int mksettings(FILE *fp, const char *filename, int *port, char **user, 
		char **pass, char **server, int *keep_alive_secs)
{
	int ret;
	char *arg;
	int line = 0;
	int error = 0;
	
	if (RC_FATAL==(ret = getsetting(fp, &arg)))
		return 1;
	while (ret!=RC_EOF) {
		++line;
		if (ret==RC_INVALID) {
			fprintf(stderr, "Invalid keyword in rcfile '%s' line "
					"%d\n",	filename, line);
			error = 1;
		} else if (!arg) {
			switch(ret) {
			case RC_COMMENT: break;
			default: 
				fprintf(stderr, "Keyword requires an argument "
						"in rcfile '%s' line %d\n", 
						filename, line);
				error = 1;
				break;
			}
		} else {
			switch(ret) {
			case RC_PORT:
			        *port = atoi(arg);
				free(arg);
	    			break;
    			case RC_SERVER:
				free(*server);
				*server = arg;
	    			break;
			case RC_USER:
				free(*user);
				*user = arg;
	    			break;
    			case RC_PASS: 
				if (check_perms(filename))
		    			return 1;
				free(*pass);
				*pass = arg;
				break;
			case RC_KEEPALIVE:
				*keep_alive_secs = atoi(arg);
				free(arg);
				break;
			default:
				fprintf(stderr, "Bug in mksettings\n");
				free(arg);
				break;
			}
		}
		if (RC_FATAL==(ret = getsetting(fp, &arg)))
			return 1;
	}
	if (error)
		return 1; /* having printed all other errors */
	return 0;
}

int getsetting(FILE *fp, char **argument)
{
	char *buf;
	int com_len;
	int ret;
	char *pt;
	int c;

	if (!(buf = malloc(MAXLINE)))
		return RC_FATAL;
	if (!fgets(buf, MAXLINE, fp)) {
		free(buf);
		return RC_EOF;
	}
	if (!strchr(buf, '\n')) {
		free(buf);
		while((c=getc(fp))!=EOF && c!='\n');
		return RC_INVALID; /* too long */
	}
	
	if ((com_len = strchr(buf, ' ') - buf) > 0) {
	 	/* argument */
		if (!(*argument = strdup(buf+com_len+1))) {
			free(buf);
                        return RC_FATAL;
		}
		for (pt=*argument; *pt; ++pt);
		*(pt-1) = '\0'; /* bye bye \n */
                /* command */
                *(buf+com_len) = '\0';
        } else {
                /* command without an argument */
                *argument = NULL;
		for (pt=buf; *pt; ++pt);
		*(pt-1) = '\0'; /* bye bye \n */
        }
	if (*buf=='#' || *buf=='\0') {
		free(buf);
		*argument = NULL;
		return RC_COMMENT; /* starts with # or blank line */
	}
	ret = getsetting_check(buf);
	free(buf);
	return ret;
}

int getsetting_check(char *buf)
{
	int i;
	struct list {
		char *descr;
		int tag;
	};

	struct list parse[] = {
		{ "port", 	RC_PORT },
		{ "server",	RC_SERVER },
		{ "user", 	RC_USER },
		{ "pass",	RC_PASS },
		{ "keepalive",	RC_KEEPALIVE },
		{ NULL,	-1}
	};
	for (i=0; parse[i].descr; ++i) {
		if (0==strcasecmp(buf, parse[i].descr))
			return parse[i].tag;
	}
	return RC_INVALID;
}

int check_perms(const char *filename)
{
	struct stat buf;

	if (-1==stat(filename, &buf)) {
		fprintf(stderr, "Getting rcfile '%s' perms failed: %s\n", 
				filename, strerror(errno));
		return 1;
	}
	if ((buf.st_mode & ~S_IWUSR) != (S_IFREG | S_IRUSR)) {
		fprintf(stderr, "Rcfile '%s' must be a regular file with mode "
				"0600 or 0400 when 'pass' is used.\n", 
				filename);
		return 1;
	}
	return 0;
}
