/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.text.numbers;

import java.math.BigDecimal;
import java.math.MathContext;
import java.math.RoundingMode;
import java.util.function.BiFunction;
import org.apache.commons.rng.RestorableUniformRandomProvider;
import org.apache.commons.rng.UniformRandomProvider;
import org.apache.commons.rng.simple.RandomSource;
import org.apache.commons.text.numbers.ParsedDecimal;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.function.Executable;

public class ParsedDecimalTest {
    private static void assertMaxPrecision(double d, int maxPrecision, boolean negative, String digits, int exponent) {
        ParsedDecimal dec = ParsedDecimal.from((double)d);
        dec.maxPrecision(maxPrecision);
        ParsedDecimalTest.assertSimpleDecimal(dec, negative, digits, exponent);
    }

    private static void assertRound(double d, int roundExponent, boolean negative, String digits, int exponent) {
        ParsedDecimal dec = ParsedDecimal.from((double)d);
        dec.round(roundExponent);
        ParsedDecimalTest.assertSimpleDecimal(dec, negative, digits, exponent);
    }

    private static void assertSimpleDecimal(ParsedDecimal parsed, boolean negative, String digits, int exponent) {
        Assertions.assertEquals((Object)negative, (Object)parsed.negative);
        Assertions.assertEquals((Object)digits, (Object)ParsedDecimalTest.digitString(parsed));
        Assertions.assertEquals((int)exponent, (int)parsed.getExponent());
        Assertions.assertEquals((int)digits.length(), (int)parsed.digitCount);
        Assertions.assertEquals((int)exponent, (int)(parsed.getScientificExponent() - digits.length() + 1));
    }

    private static void assertThrowsWithMessage(Executable fn, Class<? extends Throwable> type, String msg) {
        Throwable exc = Assertions.assertThrows(type, (Executable)fn);
        Assertions.assertEquals((Object)msg, (Object)exc.getMessage());
    }

    private static void checkFrom(double d, String digits, int exponent) {
        boolean negative = Math.signum(d) < 0.0;
        ParsedDecimalTest.assertSimpleDecimal(ParsedDecimal.from((double)d), negative, digits, exponent);
        ParsedDecimalTest.assertSimpleDecimal(ParsedDecimal.from((double)(-d)), !negative, digits, exponent);
    }

    private static void checkToEngineeringString(double d, String expected, ParsedDecimal.FormatOptions opts) {
        ParsedDecimalTest.checkToStringMethod(d, expected, ParsedDecimal::toEngineeringString, opts);
        String pos = ParsedDecimal.from((double)d).toEngineeringString(opts);
        Assertions.assertEquals((int)0, (int)(ParsedDecimalTest.parseExponent(pos, opts) % 3));
        String neg = ParsedDecimal.from((double)(-d)).toEngineeringString(opts);
        Assertions.assertEquals((int)0, (int)(ParsedDecimalTest.parseExponent(neg, opts) % 3));
    }

    private static void checkToPlainString(double d, String expected, ParsedDecimal.FormatOptions opts) {
        ParsedDecimalTest.checkToStringMethod(d, expected, ParsedDecimal::toPlainString, opts);
    }

    private static void checkToScientificString(double d, String expected, ParsedDecimal.FormatOptions opts) {
        ParsedDecimalTest.checkToStringMethod(d, expected, ParsedDecimal::toScientificString, opts);
    }

    private static void checkToStringMethod(double d, String expected, BiFunction<ParsedDecimal, ParsedDecimal.FormatOptions, String> fn, ParsedDecimal.FormatOptions opts) {
        ParsedDecimal pos = ParsedDecimal.from((double)d);
        String actual = fn.apply(pos, opts);
        Assertions.assertEquals((Object)expected, (Object)actual);
    }

    private static double createRandomDouble(UniformRandomProvider rng) {
        long mask = -9218868437227405313L;
        long bits = rng.nextLong() & 0x800FFFFFFFFFFFFFL;
        long exp = rng.nextInt(2045) + 1;
        return Double.longBitsToDouble(bits | exp << 52);
    }

    private static String digitString(ParsedDecimal dec) {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < dec.digitCount; ++i) {
            sb.append(dec.digits[i]);
        }
        return sb.toString();
    }

    private static int parseExponent(String str, ParsedDecimal.FormatOptions opts) {
        char[] expSep = opts.getExponentSeparatorChars();
        int expStartIdx = str.indexOf(String.valueOf(expSep));
        if (expStartIdx > -1) {
            int expIdx = expStartIdx + expSep.length;
            boolean neg = false;
            if (str.charAt(expIdx) == opts.getMinusSign()) {
                ++expIdx;
                neg = true;
            }
            String expStr = str.substring(expIdx);
            int val = Integer.parseInt(expStr);
            return neg ? -val : val;
        }
        return 0;
    }

    @Test
    void testFrom() {
        ParsedDecimalTest.checkFrom(0.0, "0", 0);
        ParsedDecimalTest.checkFrom(1.0, "1", 0);
        ParsedDecimalTest.checkFrom(10.0, "1", 1);
        ParsedDecimalTest.checkFrom(100.0, "1", 2);
        ParsedDecimalTest.checkFrom(1000.0, "1", 3);
        ParsedDecimalTest.checkFrom(10000.0, "1", 4);
        ParsedDecimalTest.checkFrom(0.1, "1", -1);
        ParsedDecimalTest.checkFrom(0.01, "1", -2);
        ParsedDecimalTest.checkFrom(0.001, "1", -3);
        ParsedDecimalTest.checkFrom(1.0E-4, "1", -4);
        ParsedDecimalTest.checkFrom(1.0E-5, "1", -5);
        ParsedDecimalTest.checkFrom(1.2, "12", -1);
        ParsedDecimalTest.checkFrom(0.00971, "971", -5);
        ParsedDecimalTest.checkFrom(56300.0, "563", 2);
        ParsedDecimalTest.checkFrom(123.0, "123", 0);
        ParsedDecimalTest.checkFrom(1230.0, "123", 1);
        ParsedDecimalTest.checkFrom(12300.0, "123", 2);
        ParsedDecimalTest.checkFrom(123000.0, "123", 3);
        ParsedDecimalTest.checkFrom(12.3, "123", -1);
        ParsedDecimalTest.checkFrom(1.23, "123", -2);
        ParsedDecimalTest.checkFrom(0.123, "123", -3);
        ParsedDecimalTest.checkFrom(0.0123, "123", -4);
        ParsedDecimalTest.checkFrom(1.987654321E270, "1987654321", 261);
        ParsedDecimalTest.checkFrom(1.987654321E-270, "1987654321", -279);
        ParsedDecimalTest.checkFrom(Math.PI, "3141592653589793", -15);
        ParsedDecimalTest.checkFrom(Math.E, "2718281828459045", -15);
        ParsedDecimalTest.checkFrom(Double.MAX_VALUE, "17976931348623157", 292);
        ParsedDecimalTest.checkFrom(Double.MIN_VALUE, "49", -325);
        ParsedDecimalTest.checkFrom(Double.MIN_NORMAL, "22250738585072014", -324);
    }

    @Test
    void testFromNotFinite() {
        String msg = "Double is not finite";
        ParsedDecimalTest.assertThrowsWithMessage(() -> ParsedDecimal.from((double)Double.NaN), IllegalArgumentException.class, "Double is not finite");
        ParsedDecimalTest.assertThrowsWithMessage(() -> ParsedDecimal.from((double)Double.NEGATIVE_INFINITY), IllegalArgumentException.class, "Double is not finite");
        ParsedDecimalTest.assertThrowsWithMessage(() -> ParsedDecimal.from((double)Double.POSITIVE_INFINITY), IllegalArgumentException.class, "Double is not finite");
    }

    @Test
    void testIsZero() {
        Assertions.assertTrue((boolean)ParsedDecimal.from((double)0.0).isZero());
        Assertions.assertTrue((boolean)ParsedDecimal.from((double)-0.0).isZero());
        Assertions.assertFalse((boolean)ParsedDecimal.from((double)1.0).isZero());
        Assertions.assertFalse((boolean)ParsedDecimal.from((double)-1.0).isZero());
        Assertions.assertFalse((boolean)ParsedDecimal.from((double)Double.MIN_NORMAL).isZero());
        Assertions.assertFalse((boolean)ParsedDecimal.from((double)-2.2250738585072014E-308).isZero());
        Assertions.assertFalse((boolean)ParsedDecimal.from((double)Double.MAX_VALUE).isZero());
        Assertions.assertFalse((boolean)ParsedDecimal.from((double)-4.9E-324).isZero());
    }

    @Test
    void testMaxPrecision() {
        double d = 1.02576552;
        ParsedDecimalTest.assertMaxPrecision(1.02576552, 10, false, "102576552", -8);
        ParsedDecimalTest.assertMaxPrecision(1.02576552, 9, false, "102576552", -8);
        ParsedDecimalTest.assertMaxPrecision(1.02576552, 8, false, "10257655", -7);
        ParsedDecimalTest.assertMaxPrecision(1.02576552, 7, false, "1025766", -6);
        ParsedDecimalTest.assertMaxPrecision(1.02576552, 6, false, "102577", -5);
        ParsedDecimalTest.assertMaxPrecision(1.02576552, 5, false, "10258", -4);
        ParsedDecimalTest.assertMaxPrecision(1.02576552, 4, false, "1026", -3);
        ParsedDecimalTest.assertMaxPrecision(1.02576552, 3, false, "103", -2);
        ParsedDecimalTest.assertMaxPrecision(1.02576552, 2, false, "1", 0);
        ParsedDecimalTest.assertMaxPrecision(1.02576552, 1, false, "1", 0);
        ParsedDecimalTest.assertMaxPrecision(1.02576552, 0, false, "102576552", -8);
    }

    @Test
    void testMaxPrecision_carry() {
        double d = -9.990999E52;
        ParsedDecimalTest.assertMaxPrecision(-9.990999E52, 8, true, "9990999", 46);
        ParsedDecimalTest.assertMaxPrecision(-9.990999E52, 7, true, "9990999", 46);
        ParsedDecimalTest.assertMaxPrecision(-9.990999E52, 6, true, "9991", 49);
        ParsedDecimalTest.assertMaxPrecision(-9.990999E52, 5, true, "9991", 49);
        ParsedDecimalTest.assertMaxPrecision(-9.990999E52, 4, true, "9991", 49);
        ParsedDecimalTest.assertMaxPrecision(-9.990999E52, 3, true, "999", 50);
        ParsedDecimalTest.assertMaxPrecision(-9.990999E52, 2, true, "1", 53);
        ParsedDecimalTest.assertMaxPrecision(-9.990999E52, 1, true, "1", 53);
        ParsedDecimalTest.assertMaxPrecision(-9.990999E52, 0, true, "9990999", 46);
    }

    @Test
    void testMaxPrecision_halfEvenRounding() {
        ParsedDecimalTest.assertMaxPrecision(5.5, 1, false, "6", 0);
        ParsedDecimalTest.assertMaxPrecision(2.5, 1, false, "2", 0);
        ParsedDecimalTest.assertMaxPrecision(1.6, 1, false, "2", 0);
        ParsedDecimalTest.assertMaxPrecision(1.1, 1, false, "1", 0);
        ParsedDecimalTest.assertMaxPrecision(1.0, 1, false, "1", 0);
        ParsedDecimalTest.assertMaxPrecision(-1.0, 1, true, "1", 0);
        ParsedDecimalTest.assertMaxPrecision(-1.1, 1, true, "1", 0);
        ParsedDecimalTest.assertMaxPrecision(-1.6, 1, true, "2", 0);
        ParsedDecimalTest.assertMaxPrecision(-2.5, 1, true, "2", 0);
        ParsedDecimalTest.assertMaxPrecision(-5.5, 1, true, "6", 0);
    }

    @Test
    void testMaxPrecision_random() {
        RestorableUniformRandomProvider rand = RandomSource.XO_RO_SHI_RO_128_PP.create((Object)0L, new Object[0]);
        FormatOptionsImpl opts = new FormatOptionsImpl();
        for (int i = 0; i < 10000; ++i) {
            double d = ParsedDecimalTest.createRandomDouble((UniformRandomProvider)rand);
            int precision = rand.nextInt(20) + 1;
            MathContext ctx = new MathContext(precision, RoundingMode.HALF_EVEN);
            ParsedDecimal dec = ParsedDecimal.from((double)d);
            dec.maxPrecision(precision);
            Assertions.assertEquals((double)new BigDecimal(Double.toString(d), ctx).doubleValue(), (double)Double.parseDouble(dec.toScientificString((ParsedDecimal.FormatOptions)opts)));
        }
    }

    @Test
    void testMaxPrecision_singleDigits() {
        ParsedDecimalTest.assertMaxPrecision(9.0, 1, false, "9", 0);
        ParsedDecimalTest.assertMaxPrecision(1.0, 1, false, "1", 0);
        ParsedDecimalTest.assertMaxPrecision(0.0, 1, false, "0", 0);
        ParsedDecimalTest.assertMaxPrecision(-0.0, 1, true, "0", 0);
        ParsedDecimalTest.assertMaxPrecision(-1.0, 1, true, "1", 0);
        ParsedDecimalTest.assertMaxPrecision(-9.0, 1, true, "9", 0);
    }

    @Test
    void testRound_mixed() {
        double a = 9.94E-10;
        double b = -3.1415;
        double c = 5.55E10;
        ParsedDecimalTest.assertRound(9.94E-10, -13, false, "994", -12);
        ParsedDecimalTest.assertRound(9.94E-10, -12, false, "994", -12);
        ParsedDecimalTest.assertRound(9.94E-10, -11, false, "99", -11);
        ParsedDecimalTest.assertRound(9.94E-10, -10, false, "1", -9);
        ParsedDecimalTest.assertRound(9.94E-10, -9, false, "1", -9);
        ParsedDecimalTest.assertRound(9.94E-10, -8, false, "0", 0);
        ParsedDecimalTest.assertRound(-3.1415, -5, true, "31415", -4);
        ParsedDecimalTest.assertRound(-3.1415, -4, true, "31415", -4);
        ParsedDecimalTest.assertRound(-3.1415, -3, true, "3142", -3);
        ParsedDecimalTest.assertRound(-3.1415, -2, true, "314", -2);
        ParsedDecimalTest.assertRound(-3.1415, -1, true, "31", -1);
        ParsedDecimalTest.assertRound(-3.1415, 0, true, "3", 0);
        ParsedDecimalTest.assertRound(-3.1415, 1, true, "0", 0);
        ParsedDecimalTest.assertRound(-3.1415, 2, true, "0", 0);
        ParsedDecimalTest.assertRound(5.55E10, 7, false, "555", 8);
        ParsedDecimalTest.assertRound(5.55E10, 8, false, "555", 8);
        ParsedDecimalTest.assertRound(5.55E10, 9, false, "56", 9);
        ParsedDecimalTest.assertRound(5.55E10, 10, false, "6", 10);
        ParsedDecimalTest.assertRound(5.55E10, 11, false, "1", 11);
        ParsedDecimalTest.assertRound(5.55E10, 12, false, "0", 0);
    }

    @Test
    void testRound_nine() {
        double a = 9.0E-10;
        double b = -9.0;
        double c = 9.0E10;
        ParsedDecimalTest.assertRound(9.0E-10, -11, false, "9", -10);
        ParsedDecimalTest.assertRound(9.0E-10, -10, false, "9", -10);
        ParsedDecimalTest.assertRound(9.0E-10, -9, false, "1", -9);
        ParsedDecimalTest.assertRound(-9.0, -1, true, "9", 0);
        ParsedDecimalTest.assertRound(-9.0, 0, true, "9", 0);
        ParsedDecimalTest.assertRound(-9.0, 1, true, "1", 1);
        ParsedDecimalTest.assertRound(9.0E10, 9, false, "9", 10);
        ParsedDecimalTest.assertRound(9.0E10, 10, false, "9", 10);
        ParsedDecimalTest.assertRound(9.0E10, 11, false, "1", 11);
    }

    @Test
    void testRound_one() {
        double a = 1.0E-10;
        double b = -1.0;
        double c = 1.0E10;
        ParsedDecimalTest.assertRound(1.0E-10, -11, false, "1", -10);
        ParsedDecimalTest.assertRound(1.0E-10, -10, false, "1", -10);
        ParsedDecimalTest.assertRound(1.0E-10, -9, false, "0", 0);
        ParsedDecimalTest.assertRound(-1.0, -1, true, "1", 0);
        ParsedDecimalTest.assertRound(-1.0, 0, true, "1", 0);
        ParsedDecimalTest.assertRound(-1.0, 1, true, "0", 0);
        ParsedDecimalTest.assertRound(1.0E10, 9, false, "1", 10);
        ParsedDecimalTest.assertRound(1.0E10, 10, false, "1", 10);
        ParsedDecimalTest.assertRound(1.0E10, 11, false, "0", 0);
    }

    @Test
    void testStringMethodAccuracy_random() {
        RestorableUniformRandomProvider rand = RandomSource.XO_RO_SHI_RO_128_PP.create((Object)0L, new Object[0]);
        FormatOptionsImpl stdOpts = new FormatOptionsImpl();
        FormatOptionsImpl altOpts = new FormatOptionsImpl();
        altOpts.setExponentSeparator("e");
        altOpts.setIncludeFractionPlaceholder(false);
        for (int i = 0; i < 10000; ++i) {
            double d = ParsedDecimalTest.createRandomDouble((UniformRandomProvider)rand);
            Assertions.assertEquals((double)d, (double)Double.parseDouble(ParsedDecimal.from((double)d).toScientificString((ParsedDecimal.FormatOptions)stdOpts)));
            Assertions.assertEquals((double)d, (double)Double.parseDouble(ParsedDecimal.from((double)d).toScientificString((ParsedDecimal.FormatOptions)altOpts)));
            Assertions.assertEquals((double)d, (double)Double.parseDouble(ParsedDecimal.from((double)d).toEngineeringString((ParsedDecimal.FormatOptions)stdOpts)));
            Assertions.assertEquals((double)d, (double)Double.parseDouble(ParsedDecimal.from((double)d).toEngineeringString((ParsedDecimal.FormatOptions)altOpts)));
            Assertions.assertEquals((double)d, (double)Double.parseDouble(ParsedDecimal.from((double)d).toPlainString((ParsedDecimal.FormatOptions)stdOpts)));
            Assertions.assertEquals((double)d, (double)Double.parseDouble(ParsedDecimal.from((double)d).toPlainString((ParsedDecimal.FormatOptions)altOpts)));
        }
    }

    @Test
    void testStringMethodAccuracy_sequence() {
        double min = -1000.0;
        double max = 1000.0;
        double delta = 0.1;
        FormatOptionsImpl stdOpts = new FormatOptionsImpl();
        FormatOptionsImpl altOpts = new FormatOptionsImpl();
        altOpts.setExponentSeparator("e");
        altOpts.setIncludeFractionPlaceholder(false);
        Assertions.assertEquals((double)10.0, (double)Double.parseDouble(ParsedDecimal.from((double)10.0).toScientificString((ParsedDecimal.FormatOptions)stdOpts)));
        for (double d = -1000.0; d <= 1000.0; d += 0.1) {
            Assertions.assertEquals((double)d, (double)Double.parseDouble(ParsedDecimal.from((double)d).toScientificString((ParsedDecimal.FormatOptions)stdOpts)));
            Assertions.assertEquals((double)d, (double)Double.parseDouble(ParsedDecimal.from((double)d).toScientificString((ParsedDecimal.FormatOptions)altOpts)));
            Assertions.assertEquals((double)d, (double)Double.parseDouble(ParsedDecimal.from((double)d).toEngineeringString((ParsedDecimal.FormatOptions)stdOpts)));
            Assertions.assertEquals((double)d, (double)Double.parseDouble(ParsedDecimal.from((double)d).toEngineeringString((ParsedDecimal.FormatOptions)altOpts)));
            Assertions.assertEquals((double)d, (double)Double.parseDouble(ParsedDecimal.from((double)d).toPlainString((ParsedDecimal.FormatOptions)stdOpts)));
            Assertions.assertEquals((double)d, (double)Double.parseDouble(ParsedDecimal.from((double)d).toPlainString((ParsedDecimal.FormatOptions)altOpts)));
        }
    }

    @Test
    void testStringMethods_customDigits() {
        FormatOptionsImpl opts = new FormatOptionsImpl();
        opts.setDigitsFromString("abcdefghij");
        Assertions.assertEquals((Object)"b.a", (Object)ParsedDecimal.from((double)1.0).toPlainString((ParsedDecimal.FormatOptions)opts));
        Assertions.assertEquals((Object)"-a.abcd", (Object)ParsedDecimal.from((double)-0.0123).toPlainString((ParsedDecimal.FormatOptions)opts));
        Assertions.assertEquals((Object)"bc.de", (Object)ParsedDecimal.from((double)12.34).toPlainString((ParsedDecimal.FormatOptions)opts));
        Assertions.assertEquals((Object)"baaaa.a", (Object)ParsedDecimal.from((double)10000.0).toPlainString((ParsedDecimal.FormatOptions)opts));
        Assertions.assertEquals((Object)"jihgfedcba.a", (Object)ParsedDecimal.from((double)9.87654321E9).toPlainString((ParsedDecimal.FormatOptions)opts));
        Assertions.assertEquals((Object)"b.a", (Object)ParsedDecimal.from((double)1.0).toScientificString((ParsedDecimal.FormatOptions)opts));
        Assertions.assertEquals((Object)"-b.cdE-c", (Object)ParsedDecimal.from((double)-0.0123).toScientificString((ParsedDecimal.FormatOptions)opts));
        Assertions.assertEquals((Object)"b.cdeEb", (Object)ParsedDecimal.from((double)12.34).toScientificString((ParsedDecimal.FormatOptions)opts));
        Assertions.assertEquals((Object)"b.aEe", (Object)ParsedDecimal.from((double)10000.0).toScientificString((ParsedDecimal.FormatOptions)opts));
        Assertions.assertEquals((Object)"j.ihgfedcbEj", (Object)ParsedDecimal.from((double)9.87654321E9).toScientificString((ParsedDecimal.FormatOptions)opts));
        Assertions.assertEquals((Object)"b.a", (Object)ParsedDecimal.from((double)1.0).toEngineeringString((ParsedDecimal.FormatOptions)opts));
        Assertions.assertEquals((Object)"-bc.dE-d", (Object)ParsedDecimal.from((double)-0.0123).toEngineeringString((ParsedDecimal.FormatOptions)opts));
        Assertions.assertEquals((Object)"bc.de", (Object)ParsedDecimal.from((double)12.34).toEngineeringString((ParsedDecimal.FormatOptions)opts));
        Assertions.assertEquals((Object)"ba.aEd", (Object)ParsedDecimal.from((double)10000.0).toEngineeringString((ParsedDecimal.FormatOptions)opts));
        Assertions.assertEquals((Object)"j.ihgfedcbEj", (Object)ParsedDecimal.from((double)9.87654321E9).toEngineeringString((ParsedDecimal.FormatOptions)opts));
    }

    @Test
    void testToEngineeringString_altFormat() {
        FormatOptionsImpl opts = new FormatOptionsImpl();
        opts.setIncludeFractionPlaceholder(false);
        opts.setSignedZero(false);
        opts.setDecimalSeparator(',');
        opts.setMinusSign('!');
        opts.setExponentSeparator("x10^");
        opts.setAlwaysIncludeExponent(true);
        ParsedDecimalTest.checkToEngineeringString(0.0, "0x10^0", opts);
        ParsedDecimalTest.checkToEngineeringString(-0.0, "0x10^0", opts);
        ParsedDecimalTest.checkToEngineeringString(1.0, "1x10^0", opts);
        ParsedDecimalTest.checkToEngineeringString(1.5, "1,5x10^0", opts);
        ParsedDecimalTest.checkToEngineeringString(10.0, "10x10^0", opts);
        ParsedDecimalTest.checkToEngineeringString(-1.23E-7, "!123x10^!9", opts);
        ParsedDecimalTest.checkToEngineeringString(1.23E7, "12,3x10^6", opts);
        ParsedDecimalTest.checkToEngineeringString(Math.PI, "3,141592653589793x10^0", opts);
        ParsedDecimalTest.checkToEngineeringString(Math.E, "2,718281828459045x10^0", opts);
        ParsedDecimalTest.checkToEngineeringString(-1.7976931348623157E308, "!179,76931348623157x10^306", opts);
        ParsedDecimalTest.checkToEngineeringString(Double.MIN_VALUE, "4,9x10^!324", opts);
        ParsedDecimalTest.checkToEngineeringString(Double.MIN_NORMAL, "22,250738585072014x10^!309", opts);
    }

    @Test
    void testToEngineeringString_defaults() {
        FormatOptionsImpl opts = new FormatOptionsImpl();
        ParsedDecimalTest.checkToEngineeringString(0.0, "0.0", opts);
        ParsedDecimalTest.checkToEngineeringString(-0.0, "-0.0", opts);
        ParsedDecimalTest.checkToEngineeringString(1.0, "1.0", opts);
        ParsedDecimalTest.checkToEngineeringString(1.5, "1.5", opts);
        ParsedDecimalTest.checkToEngineeringString(10.0, "10.0", opts);
        ParsedDecimalTest.checkToEngineeringString(-1.23E-7, "-123.0E-9", opts);
        ParsedDecimalTest.checkToEngineeringString(1.23E7, "12.3E6", opts);
        ParsedDecimalTest.checkToEngineeringString(Math.PI, "3.141592653589793", opts);
        ParsedDecimalTest.checkToEngineeringString(Math.E, "2.718281828459045", opts);
        ParsedDecimalTest.checkToEngineeringString(-1.7976931348623157E308, "-179.76931348623157E306", opts);
        ParsedDecimalTest.checkToEngineeringString(Double.MIN_VALUE, "4.9E-324", opts);
        ParsedDecimalTest.checkToEngineeringString(Double.MIN_NORMAL, "22.250738585072014E-309", opts);
    }

    @Test
    void testToPlainString_altFormat() {
        FormatOptionsImpl opts = new FormatOptionsImpl();
        opts.setIncludeFractionPlaceholder(false);
        opts.setSignedZero(false);
        opts.setDecimalSeparator(',');
        opts.setMinusSign('!');
        opts.setThousandsGroupingSeparator('_');
        opts.setGroupThousands(true);
        ParsedDecimalTest.checkToPlainString(0.0, "0", opts);
        ParsedDecimalTest.checkToPlainString(-0.0, "0", opts);
        ParsedDecimalTest.checkToPlainString(1.0, "1", opts);
        ParsedDecimalTest.checkToPlainString(1.5, "1,5", opts);
        ParsedDecimalTest.checkToPlainString(12.0, "12", opts);
        ParsedDecimalTest.checkToPlainString(123.0, "123", opts);
        ParsedDecimalTest.checkToPlainString(1234.0, "1_234", opts);
        ParsedDecimalTest.checkToPlainString(12345.0, "12_345", opts);
        ParsedDecimalTest.checkToPlainString(123456.0, "123_456", opts);
        ParsedDecimalTest.checkToPlainString(1234567.0, "1_234_567", opts);
        ParsedDecimalTest.checkToPlainString(1.2345678E7, "12_345_678", opts);
        ParsedDecimalTest.checkToPlainString(1.23456789E8, "123_456_789", opts);
        ParsedDecimalTest.checkToPlainString(1.23456789E9, "1_234_567_890", opts);
        ParsedDecimalTest.checkToPlainString(-1.23E-4, "!0,000123", opts);
        ParsedDecimalTest.checkToPlainString(12301.0, "12_301", opts);
        ParsedDecimalTest.checkToPlainString(Math.PI, "3,141592653589793", opts);
        ParsedDecimalTest.checkToPlainString(Math.E, "2,718281828459045", opts);
        ParsedDecimalTest.checkToPlainString(-12345.6789, "!12_345,6789", opts);
        ParsedDecimalTest.checkToPlainString(1.23E12, "1_230_000_000_000", opts);
        ParsedDecimalTest.checkToPlainString(1.23E-12, "0,00000000000123", opts);
    }

    @Test
    void testToPlainString_defaults() {
        FormatOptionsImpl opts = new FormatOptionsImpl();
        ParsedDecimalTest.checkToPlainString(0.0, "0.0", opts);
        ParsedDecimalTest.checkToPlainString(-0.0, "-0.0", opts);
        ParsedDecimalTest.checkToPlainString(1.0, "1.0", opts);
        ParsedDecimalTest.checkToPlainString(1.5, "1.5", opts);
        ParsedDecimalTest.checkToPlainString(12.0, "12.0", opts);
        ParsedDecimalTest.checkToPlainString(123.0, "123.0", opts);
        ParsedDecimalTest.checkToPlainString(1234.0, "1234.0", opts);
        ParsedDecimalTest.checkToPlainString(12345.0, "12345.0", opts);
        ParsedDecimalTest.checkToPlainString(123456.0, "123456.0", opts);
        ParsedDecimalTest.checkToPlainString(1234567.0, "1234567.0", opts);
        ParsedDecimalTest.checkToPlainString(1.2345678E7, "12345678.0", opts);
        ParsedDecimalTest.checkToPlainString(1.23456789E8, "123456789.0", opts);
        ParsedDecimalTest.checkToPlainString(1.23456789E9, "1234567890.0", opts);
        ParsedDecimalTest.checkToPlainString(-1.23E-4, "-0.000123", opts);
        ParsedDecimalTest.checkToPlainString(12301.0, "12301.0", opts);
        ParsedDecimalTest.checkToPlainString(Math.PI, "3.141592653589793", opts);
        ParsedDecimalTest.checkToPlainString(Math.E, "2.718281828459045", opts);
        ParsedDecimalTest.checkToPlainString(-12345.6789, "-12345.6789", opts);
        ParsedDecimalTest.checkToPlainString(1.23E12, "1230000000000.0", opts);
        ParsedDecimalTest.checkToPlainString(1.23E-12, "0.00000000000123", opts);
    }

    @Test
    void testToScientificString_altFormats() {
        FormatOptionsImpl opts = new FormatOptionsImpl();
        opts.setIncludeFractionPlaceholder(false);
        opts.setSignedZero(false);
        opts.setDecimalSeparator(',');
        opts.setMinusSign('!');
        opts.setExponentSeparator("x10^");
        opts.setAlwaysIncludeExponent(true);
        ParsedDecimalTest.checkToScientificString(0.0, "0x10^0", opts);
        ParsedDecimalTest.checkToScientificString(-0.0, "0x10^0", opts);
        ParsedDecimalTest.checkToScientificString(1.0, "1x10^0", opts);
        ParsedDecimalTest.checkToScientificString(1.5, "1,5x10^0", opts);
        ParsedDecimalTest.checkToScientificString(-1.23E-4, "!1,23x10^!4", opts);
        ParsedDecimalTest.checkToScientificString(12301.0, "1,2301x10^4", opts);
        ParsedDecimalTest.checkToScientificString(Math.PI, "3,141592653589793x10^0", opts);
        ParsedDecimalTest.checkToScientificString(Math.E, "2,718281828459045x10^0", opts);
        ParsedDecimalTest.checkToScientificString(-1.7976931348623157E308, "!1,7976931348623157x10^308", opts);
        ParsedDecimalTest.checkToScientificString(Double.MIN_VALUE, "4,9x10^!324", opts);
        ParsedDecimalTest.checkToScientificString(Double.MIN_NORMAL, "2,2250738585072014x10^!308", opts);
    }

    @Test
    void testToScientificString_defaults() {
        FormatOptionsImpl opts = new FormatOptionsImpl();
        ParsedDecimalTest.checkToScientificString(0.0, "0.0", opts);
        ParsedDecimalTest.checkToScientificString(-0.0, "-0.0", opts);
        ParsedDecimalTest.checkToScientificString(1.0, "1.0", opts);
        ParsedDecimalTest.checkToScientificString(1.5, "1.5", opts);
        ParsedDecimalTest.checkToScientificString(-1.23E-4, "-1.23E-4", opts);
        ParsedDecimalTest.checkToScientificString(12301.0, "1.2301E4", opts);
        ParsedDecimalTest.checkToScientificString(Math.PI, "3.141592653589793", opts);
        ParsedDecimalTest.checkToScientificString(Math.E, "2.718281828459045", opts);
        ParsedDecimalTest.checkToScientificString(-1.7976931348623157E308, "-1.7976931348623157E308", opts);
        ParsedDecimalTest.checkToScientificString(Double.MIN_VALUE, "4.9E-324", opts);
        ParsedDecimalTest.checkToScientificString(Double.MIN_NORMAL, "2.2250738585072014E-308", opts);
    }

    private static final class FormatOptionsImpl
    implements ParsedDecimal.FormatOptions {
        private boolean includeFractionPlaceholder = true;
        private boolean signedZero = true;
        private char[] digits = "0123456789".toCharArray();
        private char decimalSeparator = (char)46;
        private char thousandsGroupingSeparator = (char)44;
        private boolean groupThousands;
        private char minusSign = (char)45;
        private String exponentSeparator = "E";
        private boolean alwaysIncludeExponent;

        private FormatOptionsImpl() {
        }

        public char getDecimalSeparator() {
            return this.decimalSeparator;
        }

        public char[] getDigits() {
            return this.digits;
        }

        public char[] getExponentSeparatorChars() {
            return this.exponentSeparator.toCharArray();
        }

        public char getGroupingSeparator() {
            return this.thousandsGroupingSeparator;
        }

        public char getMinusSign() {
            return this.minusSign;
        }

        public boolean isAlwaysIncludeExponent() {
            return this.alwaysIncludeExponent;
        }

        public boolean isGroupThousands() {
            return this.groupThousands;
        }

        public boolean isIncludeFractionPlaceholder() {
            return this.includeFractionPlaceholder;
        }

        public boolean isSignedZero() {
            return this.signedZero;
        }

        public void setAlwaysIncludeExponent(boolean alwaysIncludeExponent) {
            this.alwaysIncludeExponent = alwaysIncludeExponent;
        }

        public void setDecimalSeparator(char decimalSeparator) {
            this.decimalSeparator = decimalSeparator;
        }

        public void setDigitsFromString(String digits) {
            this.digits = digits.toCharArray();
        }

        public void setExponentSeparator(String exponentSeparator) {
            this.exponentSeparator = exponentSeparator;
        }

        public void setGroupThousands(boolean groupThousands) {
            this.groupThousands = groupThousands;
        }

        public void setIncludeFractionPlaceholder(boolean includeFractionPlaceholder) {
            this.includeFractionPlaceholder = includeFractionPlaceholder;
        }

        public void setMinusSign(char minusSign) {
            this.minusSign = minusSign;
        }

        public void setSignedZero(boolean signedZero) {
            this.signedZero = signedZero;
        }

        public void setThousandsGroupingSeparator(char thousandsGroupingSeparator) {
            this.thousandsGroupingSeparator = thousandsGroupingSeparator;
        }
    }
}

