/*
    libfame - Fast Assembly MPEG Encoder Library
    Copyright (C) 2000-2001 Vivien Chappelier

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "fame.h"
#include "fame_malloc.h"
#include "fame_profile.h"
#include "fame_profile_stats.h"
#include "fame_monitor.h"

static void profile_stats_init(fame_profile_t *profile,
			      fame_context_t *context,
			      fame_parameters_t *params,
			      unsigned char *buffer,
			      unsigned int size);
static void profile_stats_enter(fame_profile_t *profile,
				fame_yuv_t *yuv,
				unsigned char *shape);
static int profile_stats_encode(fame_profile_t *profile);
static void profile_stats_leave(fame_profile_t *profile,
				fame_frame_statistics_t *stats);
static int profile_stats_close(fame_profile_t *profile);

FAME_CONSTRUCTOR(fame_profile_stats_t)
{
  FAME_OBJECT(this)->name = "Stats profile";
  FAME_PROFILE(this)->init = profile_stats_init;
  FAME_PROFILE(this)->enter = profile_stats_enter;
  FAME_PROFILE(this)->encode = profile_stats_encode;
  FAME_PROFILE(this)->leave = profile_stats_leave;
  FAME_PROFILE(this)->close = profile_stats_close;
  this->monitor_flags = 0;
  return(this);
}

/*  profile_stats_init                                                        */
/*                                                                           */
/*  Description:                                                             */
/*    Initialize the profile.                                                */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_profile_t *profile: the profile to initialize                     */
/*    fame_parameters_t *params: the parameters for initialization           */
/*    unsigned char *buffer: the buffer to output data                       */
/*    unsigned int size: the size of the output buffer                       */
/*                                                                           */
/*  Return value:                                                            */
/*    None.                                                                  */

static void profile_stats_init(fame_profile_t *profile,
			      fame_context_t *context,
			      fame_parameters_t *params,
			      unsigned char *buffer,
			      unsigned int size)
{
  fame_profile_stats_t *profile_stats = FAME_PROFILE_STATS(profile);
  int i;

  profile_stats->width = params->width;
  profile_stats->height = params->height;
  profile_stats->coding = strdup(params->coding);
  profile_stats->total_frames = params->total_frames;
  profile_stats->frame_number = 0;


  /* Get the components */
  profile_stats->monitor =
    (fame_monitor_t *) fame_get_object(context, "monitor");

  /* Allocate reference frame */
  for(i = 0; i < 2; i++) {
    profile_stats->ref[i] = (fame_yuv_t *) fame_malloc(sizeof(fame_yuv_t));
    profile_stats->ref[i]->w = profile_stats->width;
    profile_stats->ref[i]->h = profile_stats->height;
    profile_stats->ref[i]->p = profile_stats->width+32;

    profile_stats->ref[i]->y = 
      (unsigned char *) fame_malloc((profile_stats->width+32)*
                                    (profile_stats->height+32)*12/8);
    profile_stats->ref[i]->u = 
      profile_stats->ref[i]->y +
      (profile_stats->width+32)*(profile_stats->height+32);
    profile_stats->ref[i]->v = 
      profile_stats->ref[i]->u +
      (profile_stats->width+32)*(profile_stats->height+32)/4;
    /* add offset to beggining of picture (padding) */
    profile_stats->ref[i]->y += 16*(profile_stats->width+32)+16;
    profile_stats->ref[i]->u += 8*(profile_stats->width+32)/2+8;
    profile_stats->ref[i]->v += 8*(profile_stats->width+32)/2+8;
  }
  profile_stats->current = 0;

  /* Allocate reconstructed shape */
  profile_stats->ref_shape = (unsigned char *) fame_malloc(profile_stats->width*profile_stats->height);

  /* Initialize statistics monitoring */
  if(profile_stats->monitor && profile_stats->monitor->init)
    profile_stats->monitor->init(profile_stats->monitor,
				 params->retrieve_cb,
				 (profile_stats->width >> 4),
				 (profile_stats->height >> 4),
				 profile_stats->total_frames,
				 FAME_PROFILE_STATS(profile)->monitor_flags);  
}



static void profile_stats_enter(fame_profile_t *profile,
				fame_yuv_t *yuv,
				unsigned char *shape)
{
  fame_profile_stats_t *profile_stats = FAME_PROFILE_STATS(profile);
  int y, p, h, w;
  unsigned char *s, *d;
  char coding;
  
  /* Rotate reference frame */
  s = yuv->y;
  d = profile_stats->ref[profile_stats->current]->y;
  p = profile_stats->ref[profile_stats->current]->p;
  h = profile_stats->ref[profile_stats->current]->h;
  w = profile_stats->ref[profile_stats->current]->w;
  for(y = 0; y < h; y++) {
    memcpy(d, s, w);
    d += p;
    s += w;
  }

  /* Update stats and choose coding mode */
  coding = profile_stats->coding[profile_stats->frame_number % strlen(profile_stats->coding)];
  
  if (profile_stats->monitor && profile_stats->monitor->current_frame_stats)
    profile_stats->frame_stats = profile_stats->monitor->current_frame_stats;
  else
    profile_stats->frame_stats = NULL;
  if(profile_stats->monitor && profile_stats->monitor->enter)
      profile_stats->monitor->enter(profile_stats->monitor,
				    profile_stats->frame_number,
				    &(profile_stats->ref[1-profile_stats->current]),
                                    yuv,
				    profile_stats->ref_shape,
				    &coding);
  

  /* Increment frame number */
  profile_stats->frame_number++;

  if(profile_stats->monitor && profile_stats->monitor->leave)
    profile_stats->monitor->leave(profile_stats->monitor, 0, 0);

  /* Rotate reference frame */
  profile_stats->current = (profile_stats->current==1)?0:1;
}


static int profile_stats_encode(fame_profile_t *profile)
{
  return 0;
}



/*  profile_stats_leave                                                      */
/*                                                                           */
/*  Description:                                                             */
/*    Encode a single frame.                                                 */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_profile_t * profile: the profile handle returned by fame_open     */
/*    fame_yuv_t * yuv: the input frame in raw YUV format (YV12 planar)      */
/*    unsigned char * mask: the input mask (0 = transparent, 255 = opaque)   */
/*                                                                           */
/*  Return value:                                                            */
/*    int : the number of bytes written to buffer                            */

static void profile_stats_leave(fame_profile_t *profile,
			       fame_frame_statistics_t *stats)
{
  if(stats)
    *stats = *FAME_PROFILE_STATS(profile)->frame_stats;
}

/*  profile_stats_close                                                */
/*                                                                           */
/*  Description:                                                             */
/*    Flush remaining encoded data and cleanup everything.                   */
/*                                                                           */
/*  Arguments:                                                               */
/*    fame_profile_t * profile: the profile handle returned by fame_open     */
/*                                                                           */
/*  Return value:                                                            */
/*    int : the number of bytes written to buffer                            */

static int profile_stats_close(fame_profile_t *profile)
{
  fame_profile_stats_t *profile_stats = FAME_PROFILE_STATS(profile);

   /* Release statistics monitoring */
  if(profile_stats->monitor && profile_stats->monitor->close)
    profile_stats->monitor->close(profile_stats->monitor);

  /* Free reference shape */
  if(profile_stats->ref_shape)
    fame_free(profile_stats->ref_shape);

  /* Free reference frame */
  profile_stats->ref[0]->y -= 16*(profile_stats->width+32)+16;
  profile_stats->ref[1]->y -= 16*(profile_stats->width+32)+16;
  fame_free(profile_stats->ref[0]->y);
  fame_free(profile_stats->ref[0]);
  fame_free(profile_stats->ref[1]->y);
  fame_free(profile_stats->ref[1]);

  /* Return the number of bytes written to buffer */
  return(0);
}







