
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: extractor.c 2583 2007-07-22 13:53:06Z mschwerin $
 *
 */
#include "config.h"

#include <assert.h>
#include <errno.h>
#include <pthread.h>
#include <string.h>
#include <unistd.h>

#include "disc.h"
#include "environment.h"
#include "extractor.h"
#include "filelist.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "mutex.h"
#include "utils.h"

#include "menu_base.h"
#include "menu_extractor.h"
#include "menu_main.h"

#ifdef HAVE_EXTRACTOR

extern oxine_t *oxine;

static filelist_t *torip = NULL;

static pthread_t thread;
static pthread_mutex_t rip_mutex;

static int progress_all = 0;
static int progress_cur = 0;

static bool progress_wait = false;


static char *
mksafe (const char *in)
{
    int i = 0;

    if (!in) {
        return NULL;
    }

    char *string = ho_strdup (in);

    for (; i < strlen (string); i++) {
        if ((string[i] == '/') || (string[i] == '?')) {
            string[i] = '_';
        }
    }

    return string;
}


#ifdef HAVE_EXTRACTOR_CDDA
static char *
get_cdda_directory (const char *artist, const char *album)
{
    char *s_artist = mksafe (artist);
    if (!s_artist || (strlen (s_artist) == 0)) {
        s_artist = ho_strdup (_("Unknown Artist"));
    }

    char *s_album = mksafe (album);
    if (!s_album || (strlen (s_album) == 0)) {
        s_album = ho_strdup (_("Unknown Album"));
    }

    const char *outputdir = config_get_string ("extractor.cdda.outputdir");
    char *directory = ho_strdup_printf ("%s/%s/%s", outputdir,
                                        s_artist, s_album);

    ho_free (s_artist);
    ho_free (s_album);

    return directory;
}


static char *
get_cdda_filename (const char *directory, const char *title, int track_number)
{
    char *s_title = mksafe (title);
    if (!s_title || (strlen (s_title) == 0)) {
        s_title = ho_strdup (_("Unknown Title"));
    }

    char *filename = ho_strdup_printf ("%s/%02d - %s", directory,
                                       track_number, s_title);
    ho_free (s_title);

    return filename;
}


static int
get_cdda_track_number (const char *mrl)
{
    int track_number = 0;
    char *tstr = rindex (mrl, '/');

    if (tstr) {
        track_number = atoi (tstr + 1);
    }

    return track_number;
}


static bool
rip_cdda (fileitem_t * item)
{
    bool success = true;
    char *artist = NULL;
    char *album = NULL;
    char *title = NULL;
    char *year = NULL;
    char *directory = NULL;
    char *filename = NULL;
    char *escaped = NULL;
    char *command = NULL;
    const char *params = NULL;
    int track_number = get_cdda_track_number (item->mrl);
    info (_("Starting to rip '%s'..."), item->mrl);

    /* Get meta data */
    artist = meta_info_get (META_INFO_ARTIST, item->mrl);
    album = meta_info_get (META_INFO_ALBUM, item->mrl);
    title = meta_info_get (META_INFO_TITLE, item->mrl);
    year = meta_info_get (META_INFO_YEAR, item->mrl);

    directory = get_cdda_directory (artist, album);
    filename = get_cdda_filename (directory, title, track_number);
    escaped = filename_escape_for_shell (filename);

    /* Create output directory */
    if (!mkdir_recursive (directory, 0700)) {
        success = false;
        goto out;
    }

    /* Rip data to harddisk */
    params = config_get_string ("extractor.cdda.parameters");
    command = ho_strdup_printf ("%s %s %d %s.wav", BIN_CDPARANOIA,
                                params, track_number, escaped);
    if (!execute_shell (command, 0)) {
        success = false;
        goto out;
    }
    ho_free (command);

    progress_cur += 1;

    int encoder = 0;
    int encoder_use = config_get_number ("extractor.cdda.encoder");
#ifdef BIN_FLAC
    if (encoder_use == encoder) {
        char *ttitle = NULL;
        if (title && (strlen (title) > 0)) {
            ttitle = ho_strdup_printf ("-T \"TITLE=%s\"", title);
        }
        else {
            ttitle = ho_strdup ("");
        }

        char *tartist = NULL;
        if (artist && (strlen (artist) > 0)) {
            tartist = ho_strdup_printf ("-T \"ARTIST=%s\"", artist);
        }
        else {
            tartist = ho_strdup ("");
        }

        char *talbum = NULL;
        if (album && (strlen (album) > 0)) {
            talbum = ho_strdup_printf ("-T \"ALBUM=%s\"", album);
        }
        else {
            talbum = ho_strdup ("");
        }

        char *tyear = NULL;
        if (year && (strlen (year) > 0)) {
            tyear = ho_strdup_printf ("-T \"DATE=%s\"", year);
        }
        else {
            tyear = ho_strdup ("");
        }

        params = config_get_string ("extractor.cdda.parameters.encoder.flac");
        command = ho_strdup_printf ("%s %s %s %s %s %s -T \"TRACKNUMBER=%d\""
                                    " -f -o %s.flac %s.wav", BIN_FLAC,
                                    params, ttitle, tartist, talbum,
                                    tyear, track_number, escaped, escaped);

        ho_free (ttitle);
        ho_free (tartist);
        ho_free (talbum);
        ho_free (tyear);
    }
    encoder++;
#endif /* BIN_FLAC */
#ifdef BIN_OGGENC
    if (encoder_use == encoder) {
        char *ttitle = NULL;
        if (title && (strlen (title) > 0)) {
            ttitle = ho_strdup_printf ("--title \"%s\"", title);
        }
        else {
            ttitle = ho_strdup ("");
        }

        char *tartist = NULL;
        if (artist && (strlen (artist) > 0)) {
            tartist = ho_strdup_printf ("--artist \"%s\"", artist);
        }
        else {
            tartist = ho_strdup ("");
        }

        char *talbum = NULL;
        if (album && (strlen (album) > 0)) {
            talbum = ho_strdup_printf ("--album \"%s\"", album);
        }
        else {
            talbum = ho_strdup ("");
        }

        char *tyear = NULL;
        if (year && (strlen (year) > 0)) {
            tyear = ho_strdup_printf ("--date \"%s\"", year);
        }
        else {
            tyear = ho_strdup ("");
        }

        params = config_get_string ("extractor.cdda.parameters.encoder.ogg");
        command = ho_strdup_printf ("%s %s %s %s %s %s" " --tracknum %d"
                                    " -o %s.ogg %s.wav", BIN_OGGENC,
                                    params, ttitle, tartist, talbum,
                                    tyear, track_number, escaped, escaped);

        ho_free (ttitle);
        ho_free (tartist);
        ho_free (talbum);
        ho_free (tyear);
    }
    encoder++;
#endif /* BIN_OGGENC */
#ifdef BIN_LAME
    if (encoder_use == encoder) {
        char *ttitle = NULL;
        if (title && (strlen (title) > 0)) {
            ttitle = ho_strdup_printf ("--tt \"%s\"", title);
        }
        else {
            ttitle = ho_strdup ("");
        }

        char *tartist = NULL;
        if (artist && (strlen (artist) > 0)) {
            tartist = ho_strdup_printf ("--ta \"%s\"", artist);
        }
        else {
            tartist = ho_strdup ("");
        }

        char *talbum = NULL;
        if (album && (strlen (album) > 0)) {
            talbum = ho_strdup_printf ("--tl \"%s\"", album);
        }
        else {
            talbum = ho_strdup ("");
        }

        char *tyear = NULL;
        if (year && (strlen (year) > 0)) {
            tyear = ho_strdup_printf ("--ty \"%s\"", year);
        }
        else {
            tyear = ho_strdup ("");
        }

        params = config_get_string ("extractor.cdda.parameters.encoder.mp3");
        command = ho_strdup_printf ("%s %s %s %s %s %s" " --tn %d --add-id3v2"
                                    " %s.wav %s.mp3", BIN_LAME, params,
                                    ttitle, tartist, talbum, tyear,
                                    track_number, escaped, escaped);

        ho_free (ttitle);
        ho_free (tartist);
        ho_free (talbum);
        ho_free (tyear);
    }
    encoder++;
#endif /* BIN_LAME */

    /* Run the encoder */
    if (!execute_shell (command, 0)) {
        success = false;
        goto out;
    }
    ho_free (command);

    progress_cur += 1;

    /* Remove the WAV file */
    command = ho_strdup_printf ("rm -f %s.wav", escaped);
    if (!execute_shell (command, 0)) {
        goto out;
    }

    progress_cur += 1;

    info (_("Successfully ripped '%s'..."), item->mrl);
  out:
    ho_free (artist);
    ho_free (album);
    ho_free (title);
    ho_free (year);
    ho_free (directory);
    ho_free (filename);
    ho_free (escaped);
    ho_free (command);

    return success;
}
#endif /* HAVE_EXTRACTOR_CDDA */


#ifdef HAVE_EXTRACTOR_DVD
static char *
get_dvd_filename (const char *title)
{
    char *s_title = mksafe (title);
    if (!s_title || (strlen (s_title) == 0)) {
        s_title = ho_strdup (_("Unknown DVD"));
    }

    const char *outputdir = config_get_string ("extractor.dvd.outputdir");
    char *filename = ho_strdup_printf ("%s/%s", outputdir, s_title);

    ho_free (s_title);

    return filename;
}


static bool
rip_dvd (fileitem_t * item)
{
    bool success = true;
    char *command = NULL;
    char *title = NULL;
    char *filename = NULL;
    char *escaped = NULL;

    info (_("Starting to rip '%s'..."), item->mrl);
    title = meta_info_get (META_INFO_TITLE, item->mrl);
    filename = get_dvd_filename (title);
    escaped = filename_escape_for_shell (filename);

    /* Create output directory */
    const char *dvdcmd = config_get_string ("extractor.dvd.command");
    const char *outputdir = config_get_string ("extractor.dvd.outputdir");
    if (!mkdir_recursive (outputdir, 0700)) {
        success = false;
        goto out;
    }

    /* Rip data to harddisk */
    command = ho_strdup_printf (dvdcmd, item->device, escaped);
    if (!execute_shell (command, 0)) {
        success = false;
        goto out;
    }

    info (_("Successfully ripped '%s'..."), item->mrl);
  out:
    ho_free (title);
    ho_free (filename);
    ho_free (escaped);
    ho_free (command);

    return success;
}
#endif /* HAVE_EXTRACTOR_DVD */


#ifdef HAVE_EXTRACTOR_VCD
static char *
get_vcd_filename (const char *title)
{
    char *s_title = mksafe (title);
    if (!s_title || (strlen (s_title) == 0)) {
        s_title = ho_strdup (_("Unknown Video CD"));
    }

    const char *outputdir = config_get_string ("extractor.vcd.outputdir");
    char *filename = ho_strdup_printf ("%s/%s", outputdir, s_title);
    ho_free (s_title);

    return filename;
}


static bool
rip_vcd (fileitem_t * item)
{
    bool success = true;
    char *command = NULL;
    char *title = NULL;
    char *filename = NULL;
    char *escaped = NULL;

    info (_("Starting to rip '%s'..."), item->mrl);
    title = meta_info_get (META_INFO_TITLE, item->mrl);
    filename = get_vcd_filename (title);
    escaped = filename_escape_for_shell (filename);

    /* Create output directory */
    const char *vcdcmd = config_get_string ("extractor.vcd.command");
    const char *outputdir = config_get_string ("extractor.vcd.outputdir");
    if (!mkdir_recursive (outputdir, 0700)) {
        success = false;
        goto out;
    }

    /* Rip data to harddisk */
    command = ho_strdup_printf (vcdcmd, item->device, escaped);
    if (!execute_shell (command, 0)) {
        success = false;
        goto out;
    }

    info (_("Successfully ripped '%s'..."), item->mrl);
  out:
    ho_free (title);
    ho_free (filename);
    ho_free (escaped);
    ho_free (command);

    return success;
}
#endif /* HAVE_EXTRACTOR_VCD */


static void
extractor_cont_cb (void *p)
{
    progress_wait = false;
    show_menu_extractor (oxine);
}


static void
extractor_stop_cb (void *p)
{
    extractor_stop ();
    show_menu_main (oxine);
}


static void *
extractor_thread (void *user_data)
{
    assert (torip);

    info (_("Successfully started extractor thread."));
    debug ("  extractor thread: 0x%X", (int) pthread_self ());

    progress_all = 0;
    progress_cur = 0;
    progress_wait = false;
    fileitem_t *item = filelist_first (torip);
    while (item) {
        pthread_testcancel ();

        while (progress_wait) {
            sleep (1);
        }

        progress_cur = 0;
        progress_all += 1;

        oxine_event_t ev;
        ev.type = OXINE_EVENT_GUI_REPAINT;
        odk_oxine_event_send (oxine->odk, &ev);

#ifdef HAVE_EXTRACTOR_CDDA
        if (starts_with (item->mrl, "cdda:") && !rip_cdda (item)) {
            int track_number = get_cdda_track_number (item->mrl);
            char *msg = ho_strdup_printf (_("Extracting Audio CD "
                                            "track %d failed!\n"
                                            "Try to extract next track?"),
                                          track_number);
            show_message_dialog (extractor_cont_cb, NULL,
                                 extractor_stop_cb, NULL,
                                 DIALOG_YES_NO, NULL, msg);
            ho_free (msg);

            progress_wait = true;
        }
        else
#endif
#ifdef HAVE_EXTRACTOR_DVD
        if (starts_with (item->mrl, "dvd:") && !rip_dvd (item)) {
            show_message_dialog (show_menu_main, NULL, NULL, NULL,
                                 DIALOG_OK, NULL,
                                 _("Extracting DVD failed!"));
            goto out;
        }
        else
#endif
#ifdef HAVE_EXTRACTOR_VCD
        if (starts_with (item->mrl, "vcd:") && !rip_vcd (item)) {
            show_message_dialog (show_menu_main, NULL, NULL, NULL,
                                 DIALOG_OK, NULL,
                                 _("Extracting Video CD failed!"));
            goto out;
        }
        else
#endif
        {
            error (_("Failed to rip '%s': %s!"), item->mrl,
                   _("Unknown type"));
        }

        ev.type = OXINE_EVENT_GUI_REPAINT;
        odk_oxine_event_send (oxine->odk, &ev);

        item = filelist_next (torip, item);
    }

    item = filelist_first (torip);
    if (item && config_get_bool ("extractor.eject_when_finished")) {
        drive_eject (item->device);
    }

  out:
    filelist_ref_set (&torip, NULL);
    mutex_unlock (&rip_mutex);

    oxine_event_t ev;
    ev.type = OXINE_EVENT_GUI_REPAINT;
    odk_oxine_event_send (oxine->odk, &ev);

    pthread_exit (NULL);
    return NULL;
}


static void
add_recursive (filelist_t * torip, fileitem_t * item)
{
    if ((strncasecmp (item->mrl, "cdda:", 5) == 0)
        || (strncasecmp (item->mrl, "dvd:", 4) == 0)
        || (strncasecmp (item->mrl, "vcd:", 4) == 0)) {

        if (item->type == FILE_TYPE_CDDA_VFOLDER) {
            filelist_expand (item);
            fileitem_t *cur = filelist_first (item->child_list);
            while (cur) {
                add_recursive (torip, cur);
                cur = filelist_next (item->child_list, cur);
            }
        }
        else {
            fileitem_t *new = filelist_add (torip, item->title,
                                            item->mrl, item->type);
            assert (item->device);
            new->device = ho_strdup (item->device);
        }
    }
}


filelist_t *
extractor_get_list (void)
{
    return torip;
}


int
extractor_get_length_all (void)
{
    return filelist_length (torip);
}


int
extractor_get_progress_all (void)
{
    return progress_all;
}


int
extractor_get_length_cur (void)
{
    return 3;
}


int
extractor_get_progress_cur (void)
{
    return progress_cur;
}


bool
extractor_is_running (void)
{
    bool res = false;

    if (pthread_mutex_trylock (&rip_mutex) == EBUSY) {
        res = true;
    }
    else {
        mutex_unlock (&rip_mutex);
    }

    return res;
}


bool
extractor_start (int num, fileitem_t ** items)
{
    if (pthread_mutex_trylock (&rip_mutex) == EBUSY) {
        info (_("The extractor is currently busy!"));
        return false;
    }

    int i = 0;
    filelist_ref_set (&torip, filelist_new (NULL, NULL, NULL,
                                            ALLOW_FILES_ALL));
    for (; i < num; i++) {
        add_recursive (torip, items[i]);
    }

    bool result = false;
    if (pthread_create (&thread, NULL, extractor_thread, NULL) != 0) {
        error (_("Could not create extractor thread: %s!"), strerror (errno));
        filelist_ref_set (&torip, NULL);
        mutex_unlock (&rip_mutex);
    }
    else {
        result = true;
    }

    return result;
}


bool
extractor_stop (void)
{
    if (pthread_mutex_trylock (&rip_mutex) == EBUSY) {
        pthread_cancel (thread);
        filelist_ref_set (&torip, NULL);
    }
    mutex_unlock (&rip_mutex);

    return true;
}


bool
extractor_is_paused (void)
{
    return progress_wait;
}


void
extractor_pause (void)
{
    progress_wait = !progress_wait;
}


void
extractor_init (void)
{
    pthread_mutex_init (&rip_mutex, NULL);

    config_register_bool ("extractor.eject_when_finished", true,
                          _("eject disc when finished"));

#ifdef HAVE_EXTRACTOR_CDDA
    char *cdda_encoders[] = {
#ifdef BIN_FLAC
        "Lossless (FLAC)",
#endif
#ifdef BIN_OGGENC
        "Lossy (OGG)",
#endif
#ifdef BIN_LAME
        "Lossy (MP3)",
#endif
        NULL
    };

    config_register_enum ("extractor.cdda.encoder", 0, cdda_encoders,
                          _("program used for encoding audio"));

#ifdef BIN_FLAC
    config_register_string ("extractor.cdda.parameters.encoder.flac", "",
                            _("extra parameters for flac (FLAC)"));
#endif

#ifdef BIN_OGGENC
    config_register_string ("extractor.cdda.parameters.encoder.ogg", "",
                            _("extra parameters for oggenc (OGG)"));
#endif

#ifdef BIN_LAME
    config_register_string ("extractor.cdda.parameters.encoder.mp3", "",
                            _("extra parameters for lame (MP3)"));
#endif

    config_register_string ("extractor.cdda.parameters", "",
                            _("extra parameters for cdparanoia"));

    config_register_string ("extractor.cdda.outputdir", "/public/Music",
                            _("output directory for audio CDs"));
#endif /* HAVE_EXTRACTOR_CDDA */

#ifdef HAVE_EXTRACTOR_DVD
    char *dvdcmd = ho_strdup_printf ("%s if=%%s of=%%s.iso", BIN_DD);
    config_register_string ("extractor.dvd.command", dvdcmd,
                            _("command used to extract DVDs"));
    ho_free (dvdcmd);
    config_register_string ("extractor.dvd.outputdir", "/public/DVD",
                            _("output directory for DVDs"));
#endif /* HAVE_EXTRACTOR_DVD */

#ifdef HAVE_EXTRACTOR_VCD
    char *vcdcmd = ho_strdup_printf ("%s if=%%s of=%%s.iso", BIN_DD);
    config_register_string ("extractor.vcd.command", vcdcmd,
                            _("command used to extract video CDs"));
    ho_free (vcdcmd);
    config_register_string ("extractor.vcd.outputdir", "/public/VCD",
                            _("output directory for video CDs"));
#endif /* HAVE_EXTRACTOR_VCD */
}


void
extractor_free (void)
{
    if (pthread_mutex_trylock (&rip_mutex) == EBUSY) {
        pthread_cancel (thread);
    }
    filelist_ref_set (&torip, NULL);
    pthread_mutex_destroy (&rip_mutex);
    info (_("Successfully stopped extractor thread."));
}

#endif /* HAVE_EXTRACTOR */
