
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: filelist.h 2601 2007-07-25 08:29:04Z mschwerin $
 *
 */

#ifndef HAVE_FILELIST_H
#define HAVE_FILELIST_H

#include <assert.h>

#include "list.h"
#include "mutex.h"
#include "types.h"

#define FILELIST_TOPLEVEL_MRL           "computer:///"
#define FILELIST_TOPLEVEL_TITLE         "[Computer]"

#ifdef HAVE_VDR
#define FILELIST_VDR_RECORDINGS_MRL     "vdr-recordings://"
#define FILELIST_VDR_TIMERS_MRL         "vdr-timers://"
#endif

#ifdef HAVE_YOUTUBE
#define FILELIST_YOUTUBE_MRL            "http://www.youtube.com"
#endif

#ifdef HAVE_TVLINKS
#define FILELIST_TVLINKS_MRL            "http://www.tv-links.co.uk"
#endif


typedef enum {
    FILE_TYPE_UNKNOWN,
    /// A normal file or MRL.
    FILE_TYPE_REGULAR,
    /// A normal directory.
    FILE_TYPE_DIRECTORY,
    /// A mountpoint.
    FILE_TYPE_MOUNTPOINT,
    /// A playlist file (M3U format).
    FILE_TYPE_PLAYLIST_M3U,
    /// A playlist file (PLS format).
    FILE_TYPE_PLAYLIST_PLS,
    /// A playlist file (oxine's format).
    FILE_TYPE_PLAYLIST_OXP,
    /// A virtual folder containing favorites.
    FILE_TYPE_FAVORITES,
    /// A virtual folder containing mediamarks.
    FILE_TYPE_MEDIAMARKS,
    /// A virtual folder containing audio CD tracks.
    FILE_TYPE_CDDA_VFOLDER,
#ifdef HAVE_VDR
    /// A virtual folder containing VDR recordings.
    FILE_TYPE_VDR_RECORDINGS_VFOLDER,
    /// A VDR recording.
    FILE_TYPE_VDR_RECORDING,
    /// A virtual folder containing VDR timers.
    FILE_TYPE_VDR_TIMERS_VFOLDER,
    /// A VDR timer.
    FILE_TYPE_VDR_TIMER,
#endif
#ifdef HAVE_SHOUTCAST
    /// A virtual folder containing SHOUTcast stations.
    FILE_TYPE_SHOUTCAST_VFOLDER,
    /// A SHOUTcast station.
    FILE_TYPE_SHOUTCAST_STATION,
#endif
#ifdef HAVE_YOUTUBE
    /// A virtual folder containing YouTube videos or subfolders.
    FILE_TYPE_YOUTUBE_VFOLDER,
    /// A YouTube video.
    FILE_TYPE_YOUTUBE_VIDEO,
#endif
#ifdef HAVE_TVLINKS
    /// A virtual folder containing TV-LINKS videos or subfolders.
    FILE_TYPE_TVLINKS_VFOLDER,
    /// A TV-LINKS video.
    FILE_TYPE_TVLINKS_VIDEO,
#endif
} fileitem_type_t;

typedef enum {
    /// Allow only music files to be displayed.
    ALLOW_FILES_MUSIC,
    /// Allow only video files to be displayed.
    ALLOW_FILES_VIDEO,
    /// Allow only image files to be displayed.
    ALLOW_FILES_IMAGE,
    /// Allow only subtitle files to be displayed.
    ALLOW_FILES_SUBTITLE,
    /// Allow all multimedia files (music, video, subtitles) to be displayed.
    ALLOW_FILES_MULTIMEDIA,
    /// Allow any file to be displayed.
    ALLOW_FILES_ALL
} fileitem_allowed_t;

typedef struct filelist_s filelist_t;
typedef struct fileitem_s fileitem_t;


/// This represents one entry of a filelist.
struct fileitem_s {
    /// The title of this item.
    char *title;

    /// The MRL of this item.
    char *mrl;

    /// The MRL of the thumbnail belonging to this fileitem.
    char *thumbnail_mrl;

    /// Sort by this string.
    char *sort_by;

    /// The type of this fileitem.
    fileitem_type_t type;

    /// The list of items below this item.
    filelist_t *child_list;

    /// The parent of this fileitem.
    filelist_t *parent_list;

    /**
     * The number of times this item has already been listened to. This is
     * used in the favorites list.
     */
    int listened_count;

#ifdef HAVE_HAL

    /**
     * The device file corresponding to this fileitem.
     *
     * This contains the full pathname of the device. For a CDROM this would
     * be something like /dev/cdrom.
     *
     * This is used for removable media.
     */
    char *device;

    /**
     * The unique device identifier (UDI) of a volume.
     *
     * This is an identifier, the Unique Device Identifer, that is unique for
     * a device object - that is, no other device object can have the same UDI
     * at the same time. The UDI is computed using bus-specific information
     * and is meant to be unique across device insertions and independent of
     * the physical port or slot the device may be plugged into.
     *
     * This is used for removable media.
     */
    char *volume_udi;

    /**
    * The unique device identifier (UDI) of a drive.
    *
     * This is an identifier, the Unique Device Identifer, that is unique for
     * a device object - that is, no other device object can have the same UDI
     * at the same time. The UDI is computed using bus-specific information
     * and is meant to be unique across device insertions and independent of
     * the physical port or slot the device may be plugged into.
     *
     * This is used for removable media.
     */
    char *drive_udi;
#endif                                                          /* HAVE_HAL */
};


/// This represents a filelist.
struct filelist_s {
    /// The linked list containing the fileitems.
    l_list_t *list;

    /// The title of this list.
    char *title;

    /// The MRL of this list.
    char *mrl;

    /// A thumbnail of this list.
    char *thumbnail_mrl;

    /// This is TRUE if a password has to be provided.
    bool wait_for_password;

    /// This is non-NULL if an error occured while reading this list.
    char *error;

    /// The parent item of this list.
    fileitem_t *parent_item;

    /// The parent list of this list.
    filelist_t *parent_list;

    /// The type of files allowed in this filelist.
    fileitem_allowed_t allowed_filetypes;

    /// A mutex to protect the list.
    pthread_mutex_t mutex;
    pthread_mutexattr_t mutex_attr;

    /// This is used to save the top position of list when displaying it.
    int top_position;
    /// This is used to save the current position in the list when displaying it.
    int cur_position;

    /// The number of references to this filelist.
    int reference_count;
};


/**
 * Initializes the list of allowed fileextensions. This is done by comparing a
 * list of known audio, video and subtitle extensions to the extensions
 * xine-lib knows about. Only entries that appear in both lists are added to
 * the allowed-lists.
 */
void filelist_extensions_init (void);


/// Frees the allowed-extension-lists.
void filelist_extensions_free (void);


/**
 * Sets a callback function for providing a password for a MRL. The callback
 * is used to check the encrypted password <code>pwd</code> for an MRL
 * against a user provided password. If the callback returns TRUE access to
 * the MRL is allowed.
 */
void filelist_set_pwd_callback (bool (*callback)
                                  (const char *mrl, const char *pwd));


/**
 * Creates a new filelist. Only files of the allowed type will be added when
 * expanding a directory.
 *
 * @param parent                The parent of this filelist. This can be NULL.
 * @param title                 The title of this filelist.
 * @param mrl                   The MRL of this filelist.
 * @param allowed_filetypes     This defines what filetypes may be added to
 *                              this filelist.
 */
filelist_t *filelist_new (fileitem_t * parent,
                          const char *title, const char *mrl,
                          fileitem_allowed_t allowed_filetypes);


/**
 * Removes all fileitems from the list.
 *
 * @param filelist              This is a pointer to the filelist structure.
 */
void filelist_clear (filelist_t * filelist);


/**
 * Locks the filelist mutex.
 *
 * @param filelist              This is a pointer to the filelist structure.
 */
#define filelist_lock(filelist)     mutex_lock (&filelist->mutex)


/**
 * Tries to lock the filelist mutex.
 *
 * @param filelist              This is a pointer to the filelist structure.
 */
#define filelist_trylock(filelist)  mutex_trylock (&filelist->mutex)


/**
 * Unlocks the filelist mutex.
 *
 * @param filelist              This is a pointer to the filelist structure.
 */
#define filelist_unlock(filelist)   mutex_unlock (&filelist->mutex)


/**
 * Keeps track of the number of references that point to a filelist. The
 * first parameter is a pointer to a filelist pointer, the second the value
 * that is to be assigned to that pointer. As soon as the reference count
 * reaches 0 the filelist will be freed.
 *
 * @param p                     A pointer to the variable that will point to
 *                              the filelist.
 * @param d                     The filelist that will be saved in the
 *                              variable.
 */
void filelist_ref_set (filelist_t ** p, filelist_t * d);


/**
 * Sorts a filelist.
 *
 * @param filelist              This is a pointer to the filelist structure.
 * @param swap_cb               A callback, that can be used to determin if
 *                              two entries must be swapped. This may be NULL
 *                              in which case the standard behaviour is used.
 */
void filelist_sort (filelist_t * filelist, l_swap_cb_t swap_cb);


/**
 * Adds a new entry to the list.
 *
 * @param filelist              This is a pointer to the filelist structure.
 * @param title                 The title of the new entry.
 * @param mrl                   The MRL of the new entry.
 * @param type                  The type of the new entry.
 */
fileitem_t *filelist_add (filelist_t * filelist, const char *title,
                          const char *mrl, fileitem_type_t type);


/**
 * Removes the fileitem from the list.
 *
 * @param filelist              This is a pointer to the filelist structure.
 * @param fileitem              The fileitem to remove.
 */
void filelist_remove (filelist_t * filelist, fileitem_t * fileitem);


/**
 * Depending on the type of the fileitem it is expanded. Expanding means
 * that e.g. a directory is read and any files in the directory are added to
 * a sublist of this fileitem.
 *
 * @param fileitem              The fileitem to expand.
 * @return                      The sublist (if any).
 */
filelist_t *filelist_expand (fileitem_t * fileitem);


/**
 * Returns the first fileitem in the list.
 *
 * @param filelist              This is a pointer to the filelist structure.
 */
fileitem_t *filelist_first (filelist_t * filelist);


/**
 * Returns the previous fileitem in the list.
 *
 * @param filelist              This is a pointer to the filelist structure.
 * @param fileitem              The next fileitem.
 */
fileitem_t *filelist_prev (filelist_t * filelist, fileitem_t * fileitem);


/**
 * Returns the next fileitem in the list.
 *
 * @param filelist              This is a pointer to the filelist structure.
 * @param fileitem              The previous fileitem.
 */
fileitem_t *filelist_next (filelist_t * filelist, fileitem_t * fileitem);


/**
 * Returns (if its there) the fileitem with the passed MRL.
 *
 * @param filelist              This is a pointer to the filelist structure.
 * @param mrl                   This is the MRL to search for.
 */
fileitem_t *filelist_get_by_mrl (filelist_t * filelist, const char *mrl);


/**
 * Returns the fileitem at the passed position.
 *
 * @param filelist              This is a pointer to the filelist structure.
 * @param pos                   This position of the item to return.
 */
fileitem_t *filelist_get_by_pos (filelist_t * filelist, int pos);


/**
 * Determines, if the list contains an entry with the given MRL.
 *
 * @param filelist              This is a pointer to the filelist structure.
 * @param mrl                   This is the MRL to search for.
 */
bool filelist_contains (filelist_t * filelist, const char *mrl);


/**
 * Returns the number of fileitems in this list.
 *
 * @param filelist              This is a pointer to the filelist structure.
 */
int filelist_length (filelist_t * filelist);


/**
 * Creates a title from the MRL. The returned string must be freed when not
 * used any more.
 *
 * @param mrl                   The MRL to create the title from.
 */
char *create_title (const char *mrl);


/**
 * Determines if the file is allowed depending on the allowed filetypes. The
 * suffix of the file is compared to all suffixes of the allowed type.
 *
 * @param mrl                   The MRL to check.
 * @parm allowed                The filetypes that are allowed.
 */
bool is_file_allowed (const char *mrl, fileitem_allowed_t allowed);


/**
 * Returns the MRL of a thumbnail image of a filelist.
 *
 * @param filelist              This is a pointer to the filelist structure.
 */
const char *filelist_get_thumbnail (filelist_t * filelist);


/**
 * Returns the MRL of a thumbnail image of a fileitem.
 *
 * @param fileitem              This is a pointer to the fileitem structure.
 */
const char *fileitem_get_thumbnail (fileitem_t * fileitem);


#ifdef DEBUG

/**
 * Prints the filelist to the console.
 *
 * @param filelist              This is a pointer to the filelist structure.
 */
void filelist_print (filelist_t * filelist);
#endif


#endif /* HAVE_FILELIST_H */
