
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: utils_gui.c 2548 2007-07-19 20:26:43Z mschwerin $
 *
 */
#include "config.h"

#include <assert.h>
#include <stdarg.h>
#include <stdio.h>

#include "environment.h"
#include "i18n.h"
#include "heap.h"
#include "logger.h"
#include "otk.h"
#include "odk.h"
#include "oxine.h"
#include "scheduler.h"
#include "utils.h"

#include "menu_base.h"
#include "menu_playback.h"

extern oxine_t *oxine;

static int hide_gui_job_id = 0;


static void
hide_gui_job (void *p)
{
    oxine_event_t ev;
    ev.type = OXINE_EVENT_GUI_HIDE;

    odk_oxine_event_send (oxine->odk, &ev);
}


void
schedule_hide_gui_job (int timeout)
{
    cancel_job (hide_gui_job_id);
    hide_gui_job_id = schedule_job (timeout, hide_gui_job, NULL);
}


void
cancel_hide_gui_job (void)
{
    cancel_job (hide_gui_job_id);
    hide_gui_job_id = 0;
}


void
show_user_interface (void *data_p)
{
    cancel_hide_gui_job ();

    otk_draw (oxine->otk);

    oxine->user_interface_is_visible = true;
    odk_set_forward_events_to_xine (oxine->odk, false);
}


void
hide_user_interface (void *data_p)
{
    cancel_hide_gui_job ();

    otk_clear (oxine->otk);

    oxine->user_interface_is_visible = false;
    oxine->playback_controls_are_visible = false;
    oxine->stream_parameter_is_visible = false;

    if (odk_current_is_dvd (oxine->odk)) {
        odk_set_forward_events_to_xine (oxine->odk, true);
    }
#ifdef HAVE_VDR
    else if (odk_current_is_vdr (oxine->odk)) {
        odk_set_forward_events_to_xine (oxine->odk, true);
    }
#endif
    else {
        odk_set_forward_events_to_xine (oxine->odk, false);
    }

    set_current_menu (hide_user_interface, NULL);
}


static void
set_time_string (void *data_p, otk_widget_t * widget)
{
    char time_str[128];

    static char format[128] = { '\0' };
    if (strlen (format) == 0) {
        const char *cformat = config_get_string ("gui.clock_format");
        if (!cformat || (strlen (cformat) == 0)) {
            strncpy (format, _("%a, %d. %b, %H:%M"), 128);
        }
        else {
            strncpy (format, cformat, 128);
        }
    }

    time_t current = time (NULL);
    struct tm *brokentime = localtime (&current);
    strftime (time_str, 128, format, brokentime);
    otk_label_set_text (widget, time_str);
}


static void
show_clock (oxine_t * oxine)
{
    int a = OTK_ALIGN_RIGHT | OTK_ALIGN_VCENTER;
    int x = odk_osd_get_width (oxine->odk) - 20;
    int w = odk_osd_get_width (oxine->odk) - 40;

    otk_widget_t *time = otk_label_new (oxine->otk, x, 40, w, a, "");
    set_time_string (oxine, time);
    otk_label_set_update_cb (time, set_time_string, oxine);
    otk_widget_set_updated (time, true);
}


otk_widget_t *
create_new_window (bool persistent, bool with_clock)
{
    otk_widget_t *w = otk_window_new (oxine->otk, 0, 0,
                                      odk_osd_get_width (oxine->odk),
                                      odk_osd_get_height (oxine->odk));

    if (persistent) {
        otk_window_keep (w, true);
    }
    if (with_clock) {
        show_clock (oxine);
    }

    return w;
}


void
show_menu_background (const char *background)
{
    const char *current = odk_current_get_mrl (oxine->odk);
    char *mrl = NULL;

    if (background) {
        mrl = ho_strdup_printf ("%s/backgrounds/%s",
                                      get_dir_oxine_skin (),
                                      background);
    }

    if (!file_exists (mrl)) {
        ho_free (mrl);
        mrl = ho_strdup_printf ("%s/backgrounds/menu_main.png",
                                get_dir_oxine_skin ());
    }

    if (odk_current_is_playback_mode (oxine->odk)) {
        odk_play_background_stream (oxine->odk, mrl);
        goto out_free;
    }

    if (mrl && current && (strcmp (mrl, current) == 0)) {
        goto out_free;
    }

    if (mrl && odk_play_stream (oxine->odk, "Background",
                                mrl, NULL, 0, ODK_MODE_LOGO)) {
        goto out_free;
    }

    error (_("Could not open or play menu background image!"));
    fatal (_("Please make sure that your version of xine-lib "
             "supports playback of PNG files."));
    abort ();

  out_free:
    ho_free (mrl);
}


#define STREAM_INFO_POS_X (20)
#define STREAM_INFO_POS_Y (20)
#define STREAM_INFO_WIDTH (odk_osd_get_width (oxine->odk) - 40)
#define STREAM_INFO_ALIGN (OTK_ALIGN_LEFT | OTK_ALIGN_TOP)


static int
position_get (void *data_p)
{
    return odk_get_stream_param (oxine->odk, ODK_PARAM_POSITION);
}


static int
volume_get (void *data_p)
{
    return odk_get_stream_param (oxine->odk, ODK_PARAM_AO_VOLUME);
}


void
show_stream_parameter (odk_stream_param_t param)
{
    /* If we're in the playback menu and the parameter that was changed is
     * AUDIO_MUTE, we have to repaint the menu to change the mute/ no-mute
     * icon. */
    if (is_current_menu (show_menu_playback)
        && (param == ODK_PARAM_AO_VOLUME_MUTE)) {
        show_menu_current ();
        return;
    }

    /* If we're in any menu (meaning we're not showing no OSD and not showing
     * playback info), we return. */
    if (!is_current_menu (hide_user_interface)
        && !is_current_menu (show_playback_info)) {
        return;
    }

    hide_user_interface (oxine);

    create_new_window (false, false);

    if ((param == ODK_PARAM_POSITION) || (param == ODK_PARAM_SPEED)) {
        int y = odk_osd_get_height (oxine->odk) - 65;
        int w = odk_osd_get_width (oxine->odk) - 70;
        otk_widget_t *s = otk_slider_new (oxine->otk, 50, y, w, 40, 0, 20,
                                          false, OTK_SLIDER_HORIZONTAL,
                                          NULL, 0, false, false, 0, 5, 0, 100,
                                          position_get, oxine, NULL, NULL);
        otk_widget_set_updated (s, true);
        otk_widget_set_enabled (s, false);

        char *media_pplay_label = NULL;
        if (odk_get_stream_param (oxine->odk, ODK_PARAM_SPEED) ==
            ODK_SPEED_PAUSE) {
            media_pplay_label = "<";
        }
        else {
            media_pplay_label = ">";
        }
        otk_widget_t *b = otk_text_button_new (oxine->otk, 20, y, 40, 40,
                                               media_pplay_label, NULL, NULL);
        otk_widget_set_selectable (b, OTK_SELECTABLE_NONE);
        otk_widget_set_font (b, "cetus", 20);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
    }

    else if ((param == ODK_PARAM_AO_VOLUME)
             || (param == ODK_PARAM_AO_VOLUME_MUTE)) {
        int y = odk_osd_get_height (oxine->odk) - 65;
        int w = odk_osd_get_width (oxine->odk) - 70;
        otk_widget_t *s = otk_slider_new (oxine->otk, 50, y, w, 40, 0, 20,
                                          false, OTK_SLIDER_HORIZONTAL,
                                          NULL, 0, false, false, 0, 5, 0, 100,
                                          volume_get, oxine, NULL, NULL);
        otk_widget_set_updated (s, true);
        otk_widget_set_enabled (s, false);

        odk_osd_vector_type_t type = OSD_VECTOR_VOLUME;
        if (odk_get_stream_param (oxine->odk, ODK_PARAM_AO_VOLUME_MUTE)) {
            type = OSD_VECTOR_VOLUME_MUTE;
        }
        otk_widget_t *b = otk_vector_button_new (oxine->otk, 20, y, 40, 40,
                                                 type, 30, 30, NULL, NULL);
        otk_widget_set_selectable (b, OTK_SELECTABLE_NONE);
    }

    else {
        char msg[256];
        char *param_str = odk_get_stream_param_as_string (oxine->odk, param);

        if (param == ODK_PARAM_SPEED) {
            snprintf (msg, 256, "%s", param_str);
        }

        else {
            char *param_name = odk_get_stream_param_name (oxine->odk, param);
            snprintf (msg, 256, "%s: %s", param_name, param_str);
            ho_free (param_name);
        }

        ho_free (param_str);

        otk_label_new (oxine->otk, STREAM_INFO_POS_X, STREAM_INFO_POS_Y,
                       STREAM_INFO_WIDTH, STREAM_INFO_ALIGN, msg);
    }

    otk_draw (oxine->otk);

    oxine->stream_parameter_is_visible = true;

    schedule_hide_gui_job (6 * 1000);
}


void
repaint_message_dialog (void *data_p)
{
    show_message_dialog (oxine->current_dialog_data.ok_cb,
                         oxine->current_dialog_data.ok_cb_data,
                         oxine->current_dialog_data.cancel_cb,
                         oxine->current_dialog_data.cancel_cb_data,
                         oxine->current_dialog_data.type,
                         oxine->current_dialog_data.background_mrl,
                         oxine->current_dialog_data.msg);
}


void
show_message_dialog (otk_cb_t ok_cb, void *ok_cb_data,
                     otk_cb_t cancel_cb, void *cancel_cb_data,
                     message_dialog_type_t type, const char *background_mrl,
                     const char *format, ...)
{
    char msg[1024];
    va_list args;
    va_start (args, format);
    vsnprintf (msg, 1024, format, args);
    va_end (args);

    ho_free (oxine->current_dialog_data.background_mrl);
    ho_free (oxine->current_dialog_data.msg);

    oxine->current_dialog_data.ok_cb = ok_cb;
    oxine->current_dialog_data.ok_cb_data = ok_cb_data;
    oxine->current_dialog_data.cancel_cb = cancel_cb;
    oxine->current_dialog_data.cancel_cb_data = cancel_cb_data;
    oxine->current_dialog_data.type = type;
    oxine->current_dialog_data.background_mrl =
        background_mrl ? ho_strdup (background_mrl) : NULL;
    oxine->current_dialog_data.msg = ho_strdup (msg);

    create_new_window (false, true);

    /* First we count the lines we have to display. */
    int count = 0;
    char *cline = msg;
    while (cline) {
        cline = index (cline, '\n');
        if (cline) {
            cline++;
        }
        count++;
    }

    /* Display the border. */
    int w = 600;
    int h = count * 30 + 60;
    if (type != DIALOG_PLAIN) {
        h += 70;
    }

    int x = (odk_osd_get_width (oxine->odk) - w) / 2;
    int y = (odk_osd_get_height (oxine->odk) - h) / 2;
    otk_border_new (oxine->otk, x, y, w, h);
    y += 60;

    /* Display the broken up message. */
    cline = msg;
    while (cline) {
        char *nline = index (cline, '\n');
        if (nline) {
            nline[0] = 0;
            nline++;
        }

        otk_label_new (oxine->otk, x + (w / 2), y, w - 40,
                       OTK_ALIGN_CENTER | OTK_ALIGN_BOTTOM, cline);

        cline = nline;
        y += 30;
    }

    otk_widget_t *b;
    if (type == DIALOG_OK) {
        b = otk_text_button_new (oxine->otk, x + 220, y, 160, 40,
                                 _("OK"), ok_cb, ok_cb_data);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        otk_widget_set_focused (b, true);
    }

    else if (type == DIALOG_OK_CANCEL) {
        b = otk_text_button_new (oxine->otk, x + 130, y, 160, 40,
                                 _("OK"), ok_cb, ok_cb_data);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        otk_widget_set_focused (b, true);

        b = otk_text_button_new (oxine->otk, x + 310, y, 160, 40,
                                 _("Cancel"), cancel_cb, cancel_cb_data);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
    }

    else if (type == DIALOG_YES_NO) {
        b = otk_text_button_new (oxine->otk, x + 130, y, 160, 40,
                                 _("Yes"), ok_cb, ok_cb_data);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        otk_widget_set_focused (b, true);

        b = otk_text_button_new (oxine->otk, x + 310, y, 160, 40,
                                 _("No"), cancel_cb, cancel_cb_data);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
    }

    else if (type == DIALOG_RETRY_CANCEL) {
        b = otk_text_button_new (oxine->otk, x + 130, y, 160, 40,
                                 _("Retry"), ok_cb, ok_cb_data);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        otk_widget_set_focused (b, true);

        b = otk_text_button_new (oxine->otk, x + 310, y, 160, 40,
                                 _("Cancel"), cancel_cb, cancel_cb_data);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
    }

    set_current_menu (repaint_message_dialog, NULL);

    show_user_interface (NULL);
    show_menu_background (background_mrl);
}
