/* gkrellm-bluez - A BlueZ monitor plug-in for GKrellM2
 *
 * Copyright (C) 2006 Ludovic Cintrat <lcintrat@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include "gkrellm-bluez.h"

#include <errno.h>
#include <string.h>
#include <unistd.h>

#include <sys/ioctl.h>

#include <bluetooth/bluetooth.h>
#include <bluetooth/hci.h>

#include "gkrellm-bluez-monitor.h"
#include "gkrellm-bluez-linux.h"

#define MAX_BTDEV         8
#define MAX_CONN_PER_DEV 16

static int bluez_socket;

void scan_bluez_devices (void);

void update_bluez_device_stats (void);

void update_conn_list (GkrellmBlueZMonitor *bluezmon);


/*  public functions  */

void
gkrellm_bluez_init ()
{
    bluez_socket = socket(AF_BLUETOOTH, SOCK_RAW, BTPROTO_HCI);
    if (bluez_socket == -1)
    {
        g_error(_("Error creating BlueZ socket: %s"),
		strerror(errno));
    }
}

void
gkrellm_bluez_info_read (void)
{
    if (bluez_socket != -1)
    {
      update_bluez_device_stats();
    }
}

/*  private functions  */

void
gkrellm_bluez_scan_devices (void)
{
    int                  i;
    GkrellmBlueZMonitor *bluezmon;
    
    struct hci_dev_list_req *dev_list;
    struct hci_dev_req      *dev_req;

    char buf[MAX_BTDEV * sizeof(*dev_req) + sizeof(*dev_list)];

    dev_list = (struct hci_dev_list_req*) buf;
    dev_list->dev_num = MAX_BTDEV;

    dev_req = dev_list->dev_req;

    /* get device list */
    if (ioctl( bluez_socket, HCIGETDEVLIST, (void*)dev_list ) == -1)
    {
        g_warning(_("ioctl HCIGETDEVLIST failed: %s"),
                  strerror(errno));
        return;
    }

    /* check for new devices */
    for (i = 0; i < dev_list->dev_num; i++)
    {
        gint16 devid = dev_list->dev_req[i].dev_id;

        if (! gkrellm_bluez_monitor_find (devid))
        {
            bluezmon = gkrellm_bluez_monitor_create (devid);
	    bluezmon->present = TRUE;
	    bluezmon->chart = NULL;
        }
    }
}


void
update_bluez_device_stats (void)
{
    int                  i;
    GkrellmBlueZMonitor *bluezmon;
    GList               *list;
    
    struct hci_dev_list_req *dev_list;
    struct hci_dev_req      *dev_req;

    char buf[MAX_BTDEV * sizeof(*dev_req) + sizeof(*dev_list)];

    dev_list = (struct hci_dev_list_req*) buf;
    dev_list->dev_num = MAX_BTDEV;

    dev_req = dev_list->dev_req;

    /* get device list */
    if (ioctl( bluez_socket, HCIGETDEVLIST, (void*)dev_list ) == -1)
    {
        g_warning(_("ioctl HCIGETDEVLIST failed: %s"),
                  strerror(errno));
        return;
    }

    /* check for present devices */
    for (list = gkrellm_bluez_monitor_list; list; list = g_list_next (list))
    {
        bluezmon = (GkrellmBlueZMonitor*)list->data;

	i = 0;
	bluezmon->present = FALSE;

	while ((i < dev_list->dev_num) && !bluezmon->present)
	{
	    bluezmon->present = (bluezmon->devid == dev_list->dev_req[i].dev_id);
	    i++;
	}
    }

    /* update devices info */
    for (list = gkrellm_bluez_monitor_list; list; list = g_list_next (list))
    {
        bluezmon = list->data;

	bluezmon->updated = FALSE;

	if (bluezmon->present)
	{
	  struct hci_dev_info dev_info;
	  
	  dev_info.dev_id = bluezmon->devid;

	  if (ioctl( bluez_socket, HCIGETDEVINFO, (void*)&dev_info ) == -1)
	  {
	    g_warning(_("ioctl HCIGETDEVINFO (dev=%d) failed: %s"),
                      bluezmon->devid, strerror(errno));
	  }
	  else
	    {
	      bluezmon->rx_bytes_old = bluezmon->rx_bytes;
	      bluezmon->tx_bytes_old = bluezmon->tx_bytes;

	      bluezmon->rx_bytes = dev_info.stat.byte_rx;
	      bluezmon->tx_bytes = dev_info.stat.byte_tx;
	      
	      /* update connections */
	      update_conn_list (bluezmon);

	      bluezmon->updated = TRUE;
	    }
	}
    }
}


void
update_conn_list (GkrellmBlueZMonitor *bluezmon)
{
    struct hci_conn_list_req *conn_list;
    struct hci_conn_info     *conn_info;

    char buf[MAX_CONN_PER_DEV * sizeof(*conn_info) + sizeof(*conn_list)];

    conn_list = (struct hci_conn_list_req*) buf;
    conn_list->conn_num = MAX_CONN_PER_DEV;
    conn_list->dev_id   = bluezmon->devid;

    conn_info = conn_list->conn_info;

    /* get connection list */
    if (ioctl( bluez_socket, HCIGETCONNLIST, (void*)conn_list ) == -1)
    {
        g_warning(_("ioctl HCIGETCONNLIST failed: %s"),
                  strerror(errno));
        return;
    }

    bluezmon->connections = conn_list->conn_num;
}
