// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

package reaper

import (
	"testing"
	"time"
)

func TestIsExceeded(t *testing.T) {
	defer func() {
		now = time.Now
	}()
	now = func() time.Time {
		return mustParse("2006-01-02T15:00:00")
	}
	var tests = []struct {
		unix     int64
		timeout  time.Duration
		buffer   time.Duration
		exceeded bool
	}{
		// timestamp equal to current time, not expired
		{
			unix:     mustParse("2006-01-02T15:00:00").Unix(),
			timeout:  time.Minute * 60,
			buffer:   time.Minute * 5,
			exceeded: false,
		},
		// timestamp is not gt current time - timeout, not expired
		{
			unix:     mustParse("2006-01-02T14:00:00").Unix(),
			timeout:  time.Minute * 60,
			buffer:   0,
			exceeded: false,
		},
		// timestamp is gt current time - timeout, expired
		{
			unix:     mustParse("2006-01-02T13:59:00").Unix(),
			timeout:  time.Minute * 60,
			buffer:   0,
			exceeded: true,
		},
		// timestamp is not gt current time - timeout - buffer, not expired
		{
			unix:     mustParse("2006-01-02T13:59:00").Unix(),
			timeout:  time.Minute * 60,
			buffer:   time.Minute * 5,
			exceeded: false,
		},
		// timestamp is gt current time - timeout - buffer, expired
		{
			unix:     mustParse("2006-01-02T13:04:05").Unix(),
			timeout:  time.Minute * 60,
			buffer:   time.Minute * 5,
			exceeded: true,
		},
	}
	for i, test := range tests {
		got, want := isExceeded(test.unix, test.timeout, test.buffer), test.exceeded
		if got != want {
			t.Errorf("Want exceeded %v, got %v at index %v", want, got, i)
		}
	}
}

func mustParse(s string) time.Time {
	t, err := time.Parse("2006-01-02T15:04:05", s)
	if err != nil {
		panic(err)
	}
	return t
}
