// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

package perm

import (
	"context"
	"database/sql"
	"testing"

	"github.com/drone/drone/core"
	"github.com/drone/drone/store/repos"
	"github.com/drone/drone/store/shared/db/dbtest"
	"github.com/drone/drone/store/shared/encrypt"
	"github.com/drone/drone/store/user"
)

var noContext = context.TODO()

func TestPerms(t *testing.T) {
	conn, err := dbtest.Connect()
	if err != nil {
		t.Error(err)
		return
	}
	defer func() {
		dbtest.Reset(conn)
		dbtest.Disconnect(conn)
	}()

	// no-op encrypter
	enc, _ := encrypt.New("")

	// seeds the database with a dummy user account.
	auser := &core.User{Login: "spaceghost"}
	users := user.New(conn, enc)
	err = users.Create(noContext, auser)
	if err != nil {
		t.Error(err)
	}

	// seeds the database with a dummy repository.
	arepo := &core.Repository{UID: "1", Slug: "octocat/hello-world"}
	repos := repos.New(conn)
	err = repos.Create(noContext, arepo)
	if err != nil {
		t.Error(err)
	}
	if err != nil {
		t.Error(err)
	}

	store := New(conn).(*permStore)
	t.Run("Create", testPermCreate(store, auser, arepo))
	t.Run("Find", testPermFind(store, auser, arepo))
	t.Run("List", testPermList(store, auser, arepo))
	t.Run("Update", testPermUpdate(store, auser, arepo))
	t.Run("Delete", testPermDelete(store, auser, arepo))
}

func testPermCreate(store *permStore, user *core.User, repo *core.Repository) func(t *testing.T) {
	return func(t *testing.T) {
		item := &core.Perm{
			UserID:  user.ID,
			RepoUID: repo.UID,
			Read:    true,
			Write:   true,
			Admin:   false,
		}
		err := store.Create(noContext, item)
		if err != nil {
			t.Error(err)
		}
	}
}

func testPermFind(store *permStore, user *core.User, repo *core.Repository) func(t *testing.T) {
	return func(t *testing.T) {
		item, err := store.Find(noContext, repo.UID, user.ID)
		if err != nil {
			t.Error(err)
		} else {
			t.Run("Fields", testPerm(item))
		}
	}
}

func testPermList(store *permStore, user *core.User, repo *core.Repository) func(t *testing.T) {
	return func(t *testing.T) {
		list, err := store.List(noContext, repo.UID)
		if err != nil {
			t.Error(err)
			return
		}
		if got, want := len(list), 1; got != want {
			t.Errorf("Want collaborator count %d, got %d", want, got)
			return
		}
		if got, want := list[0].Login, user.Login; got != want {
			t.Errorf("Want username %q, got %q", want, got)
		}
		t.Run("Fields", testPerm(
			&core.Perm{
				Read:  list[0].Read,
				Write: list[0].Write,
				Admin: list[0].Admin,
			},
		))
	}
}

func testPermUpdate(store *permStore, user *core.User, repo *core.Repository) func(t *testing.T) {
	return func(t *testing.T) {
		before := &core.Perm{
			UserID:  user.ID,
			RepoUID: repo.UID,
			Read:    true,
			Write:   true,
			Admin:   true,
		}
		err := store.Update(noContext, before)
		if err != nil {
			t.Error(err)
			return
		}
		after, err := store.Find(noContext, before.RepoUID, before.UserID)
		if err != nil {
			t.Error(err)
			return
		}
		if got, want := after.Admin, before.Admin; got != want {
			t.Errorf("Want updated Admin %v, got %v", want, got)
		}
	}
}

func testPermDelete(store *permStore, user *core.User, repo *core.Repository) func(t *testing.T) {
	return func(t *testing.T) {
		err := store.Delete(noContext, &core.Perm{UserID: user.ID, RepoUID: repo.UID})
		if err != nil {
			t.Error(err)
			return
		}
		_, err = store.Find(noContext, "3", user.ID)
		if got, want := sql.ErrNoRows, err; got != want {
			t.Errorf("Want sql.ErrNoRows, got %v", got)
			return
		}
	}
}

func testPerm(item *core.Perm) func(t *testing.T) {
	return func(t *testing.T) {
		if got, want := item.Read, true; got != want {
			t.Errorf("Want Read %v, got %v", want, got)
		}
		if got, want := item.Write, true; got != want {
			t.Errorf("Want Write %v, got %v", want, got)
		}
		if got, want := item.Admin, false; got != want {
			t.Errorf("Want Admin %v, got %v", want, got)
		}
	}
}
