/*
 * uriparser - RFC 3986 URI parsing library
 *
 * Copyright (C) 2025, Sebastian Pipping <sebastian@pipping.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <gtest/gtest.h>

#include <uriparser/Uri.h>

namespace {

static void testIsWellFormedHostRegName(const char * candidate, bool expectedWellFormed) {
	const char * const first = candidate;
	const char * const afterLast = (candidate == NULL) ? NULL : (candidate + strlen(candidate));

	const UriBool actualWellFormed = uriIsWellFormedHostRegNameA(first, afterLast);

	ASSERT_EQ(actualWellFormed, expectedWellFormed);
}

static UriUriA parseWellFormedUri(const char * text) {
	UriUriA uri;
	const int error = uriParseSingleUriA(&uri, text, NULL);
	// NOTE: we cannot use ASSERT_EQ here because of the outer non-void return type
	assert(error == URI_SUCCESS);
	return uri;
}

static void assertUriEqual(const UriUriA * uri, const char * expected) {
	int charsRequired = -1;
	ASSERT_EQ(uriToStringCharsRequiredA(uri, &charsRequired), URI_SUCCESS);
	ASSERT_TRUE(charsRequired >= 0);

	char * const buffer = (char *)malloc(charsRequired + 1);
	ASSERT_TRUE(buffer != NULL);

	ASSERT_EQ(uriToStringA(buffer, uri, charsRequired + 1, NULL), URI_SUCCESS);

	EXPECT_STREQ(buffer, expected);

	free(buffer);
}

}  // namespace

TEST(IsWellFormedHostRegName, Null) {
	testIsWellFormedHostRegName(NULL, false);
}

TEST(IsWellFormedHostRegName, Empty) {
	testIsWellFormedHostRegName("", true);
}

TEST(IsWellFormedHostRegName, AllowedCharacters) {
	// The related grammar subset is this:
	//
	//   reg-name = *( unreserved / pct-encoded / sub-delims )
	//   unreserved  = ALPHA / DIGIT / "-" / "." / "_" / "~"
	//   pct-encoded = "%" HEXDIG HEXDIG
	//   sub-delims  = "!" / "$" / "&" / "'" / "(" / ")"
	//               / "*" / "+" / "," / ";" / "="
	//
	// NOTE: Percent encoding has dedicated tests further down
	testIsWellFormedHostRegName(
			"0123456789"
			"ABCDEF"
			"abcdef"
			"gGhHiIjJkKlLmMnNoOpPqQrRsStTuUvVwWxXyYzZ"
			"-._~"
			"!$&'()*+,;=",
			true);
}

TEST(IsWellFormedHostRegName, ForbiddenCharacters) {
	testIsWellFormedHostRegName(" ", false);
}

TEST(IsWellFormedHostRegName, PercentEncodingWellFormed) {
	testIsWellFormedHostRegName("%" "aa" "%" "AA", true);
}

TEST(IsWellFormedHostRegName, PercentEncodingMalformedCutOff1) {
	testIsWellFormedHostRegName("%", false);
}

TEST(IsWellFormedHostRegName, PercentEncodingMalformedCutOff2) {
	testIsWellFormedHostRegName("%" "a", false);
}

TEST(IsWellFormedHostRegName, PercentEncodingMalformedForbiddenCharacter1) {
	testIsWellFormedHostRegName("%" "ga", false);
}

TEST(IsWellFormedHostRegName, PercentEncodingMalformedForbiddenCharacter2) {
	testIsWellFormedHostRegName("%" "ag", false);
}

TEST(SetHostRegName, NullUriOnly) {
	UriUriA * const uri = NULL;
	const char * const first = "localhost";
	const char * const afterLast = first + strlen(first);
	ASSERT_EQ(uriSetHostRegNameA(uri, first, afterLast), URI_ERROR_NULL);
}

TEST(SetHostRegName, NullFirstOnly) {
	UriUriA uri = {};
	const char * const fragment = "localhost";
	const char * const first = NULL;
	const char * const afterLast = fragment + strlen(fragment);
	ASSERT_EQ(uriSetHostRegNameA(&uri, first, afterLast), URI_ERROR_NULL);
}

TEST(SetHostRegName, NullAfterLastOnly) {
	UriUriA uri = {};
	const char * const first = "localhost";
	const char * const afterLast = NULL;
	ASSERT_EQ(uriSetHostRegNameA(&uri, first, afterLast), URI_ERROR_NULL);
}

TEST(SetHostRegName, NullValueLeavesOwnerAtFalse) {
	UriUriA uri = parseWellFormedUri("scheme://host/");
	EXPECT_EQ(uri.owner, URI_FALSE);  // self-test

	EXPECT_EQ(uriSetHostRegNameA(&uri, NULL, NULL), URI_SUCCESS);

	EXPECT_EQ(uri.owner, URI_FALSE);  // i.e. still false

	uriFreeUriMembersA(&uri);
}

TEST(SetHostRegName, NonNullValueMakesOwner) {
	UriUriA uri = parseWellFormedUri("scheme://old/");
	const char * const first = "new";
	const char * const afterLast = first + strlen(first);
	EXPECT_EQ(uri.owner, URI_FALSE);  // self-test

	EXPECT_EQ(uriSetHostRegNameA(&uri, first, afterLast), URI_SUCCESS);

	EXPECT_EQ(uri.owner, URI_TRUE);  // i.e. now owned

	uriFreeUriMembersA(&uri);
}

TEST(SetHostRegName, NullValueAppliedDotInserted) {
	UriUriA uri = parseWellFormedUri("scheme://host//path1/path2");

	EXPECT_EQ(uriSetHostRegNameA(&uri, NULL, NULL), URI_SUCCESS);

	assertUriEqual(&uri, "scheme:/.//path1/path2");  // i.e. not scheme://path1/path2

	uriFreeUriMembersA(&uri);
}

TEST(SetHostRegName, NullValueAppliedDotNotInserted) {
	UriUriA uri = parseWellFormedUri("//host/./path1/path2");

	EXPECT_EQ(uriSetHostRegNameA(&uri, NULL, NULL), URI_SUCCESS);

	assertUriEqual(&uri, "/./path1/path2");  // i.e. not /././path1/path2

	uriFreeUriMembersA(&uri);
}

TEST(SetHostRegName, NullValueAppliedPriorNull) {
	UriUriA uri = parseWellFormedUri("scheme:/path");

	EXPECT_EQ(uriSetHostRegNameA(&uri, NULL, NULL), URI_SUCCESS);

	assertUriEqual(&uri, "scheme:/path");

	uriFreeUriMembersA(&uri);
}

TEST(SetHostRegName, NullValueAppliedPriorIp4) {
	UriUriA uri = parseWellFormedUri("scheme://1.2.3.4/path");

	EXPECT_EQ(uriSetHostRegNameA(&uri, NULL, NULL), URI_SUCCESS);

	assertUriEqual(&uri, "scheme:/path");

	uriFreeUriMembersA(&uri);
}

TEST(SetHostRegName, NullValueAppliedPriorIp6) {
	UriUriA uri = parseWellFormedUri("scheme://[::1]/path");

	EXPECT_EQ(uriSetHostRegNameA(&uri, NULL, NULL), URI_SUCCESS);

	assertUriEqual(&uri, "scheme:/path");

	uriFreeUriMembersA(&uri);
}

TEST(SetHostRegName, NullValueAppliedPriorIpFuture) {
	UriUriA uri = parseWellFormedUri("scheme://[v7.host]/path");

	EXPECT_EQ(uriSetHostRegNameA(&uri, NULL, NULL), URI_SUCCESS);

	assertUriEqual(&uri, "scheme:/path");

	uriFreeUriMembersA(&uri);
}

TEST(SetHostRegName, NullValueAppliedPriorRegName) {
	UriUriA uri = parseWellFormedUri("scheme://host/path");

	EXPECT_EQ(uriSetHostRegNameA(&uri, NULL, NULL), URI_SUCCESS);

	assertUriEqual(&uri, "scheme:/path");

	uriFreeUriMembersA(&uri);
}

TEST(SetHostRegName, NonNullValueAppliedEmpty) {
	UriUriA uri = parseWellFormedUri("scheme://host/path");
	const char * const empty = "";

	EXPECT_EQ(uriSetHostRegNameA(&uri, empty, empty), URI_SUCCESS);

	assertUriEqual(&uri, "scheme:///path");

	uriFreeUriMembersA(&uri);
}

TEST(SetHostRegName, NonNullValueAppliedNonEmptyPriorNull) {
	UriUriA uri = parseWellFormedUri("scheme:");
	const char * const first = "localhost";
	const char * const afterLast = first + strlen(first);

	EXPECT_EQ(uriSetHostRegNameA(&uri, first, afterLast), URI_SUCCESS);

	assertUriEqual(&uri, "scheme://localhost");

	uriFreeUriMembersA(&uri);
}

TEST(SetHostRegName, NonNullValueAppliedNonEmptyPriorIp4) {
	UriUriA uri = parseWellFormedUri("//1.2.3.4");
	const char * const first = "localhost";
	const char * const afterLast = first + strlen(first);

	EXPECT_EQ(uriSetHostRegNameA(&uri, first, afterLast), URI_SUCCESS);

	assertUriEqual(&uri, "//localhost");

	uriFreeUriMembersA(&uri);
}

TEST(SetHostRegName, NonNullValueAppliedNonEmptyPriorIp6) {
	UriUriA uri = parseWellFormedUri("//[::1]");
	const char * const first = "localhost";
	const char * const afterLast = first + strlen(first);

	EXPECT_EQ(uriSetHostRegNameA(&uri, first, afterLast), URI_SUCCESS);

	assertUriEqual(&uri, "//localhost");

	uriFreeUriMembersA(&uri);
}

TEST(SetHostRegName, NonNullValueAppliedNonEmptyPriorIpFuture) {
	UriUriA uri = parseWellFormedUri("//[v7.host]");
	const char * const first = "localhost";
	const char * const afterLast = first + strlen(first);

	EXPECT_EQ(uriSetHostRegNameA(&uri, first, afterLast), URI_SUCCESS);

	assertUriEqual(&uri, "//localhost");

	uriFreeUriMembersA(&uri);
}

TEST(SetHostRegName, NonNullValueAppliedNonEmptyPriorRegName) {
	UriUriA uri = parseWellFormedUri("//hostname.test");
	const char * const first = "localhost";
	const char * const afterLast = first + strlen(first);

	EXPECT_EQ(uriSetHostRegNameA(&uri, first, afterLast), URI_SUCCESS);

	assertUriEqual(&uri, "//localhost");

	uriFreeUriMembersA(&uri);
}

TEST(SetHostRegName, MalformedValueRejected) {
	UriUriA uri = parseWellFormedUri("scheme://host/");
	const char * const first = "not well-formed";
	const char * const afterLast = first + strlen(first);

	EXPECT_EQ(uriSetHostRegNameA(&uri, first, afterLast), URI_ERROR_SYNTAX);

	uriFreeUriMembersA(&uri);
}

TEST(SetHostRegName, UriWithPortRejected) {
	UriUriA uri = parseWellFormedUri("//host:1234");
	EXPECT_TRUE(uri.portText.first != NULL);  // self-test

	EXPECT_EQ(uriSetHostRegNameA(&uri, NULL, NULL), URI_ERROR_SETHOST_PORT_SET);

	uriFreeUriMembersA(&uri);
}

TEST(SetHostRegName, UriWithUserInfoRejected) {
	UriUriA uri = parseWellFormedUri("//user:password@host");
	EXPECT_TRUE(uri.userInfo.first != NULL);  // self-test

	EXPECT_EQ(uriSetHostRegNameA(&uri, NULL, NULL), URI_ERROR_SETHOST_USERINFO_SET);

	uriFreeUriMembersA(&uri);
}
