/* Copyright JS Foundation and other contributors, http://js.foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * Unit test generator for jerry-math.
 * To be compiled separately from the rest of jerry and to be linked to a trusted libm.
 * Its output should be redirected to test-math.inc.h.
 *
 * Example:
 * gcc gen-test-math.c -o gen-test-math -lm
 * ./gen-test-math >test-math.inc.h
 */

#include <math.h>
#include <stdio.h>

#define GEN_INT_TEST(EXPR) printf("check_int (\"%s\", %s, %d);\n", #EXPR, #EXPR, EXPR);
#define GEN_DBL_TEST(EXPR) printf("check_double (\"%s\", %s, %.20E);\n", #EXPR, #EXPR, EXPR);

int
main (int argc, char **args)
{
  printf ("/* Copyright JS Foundation and other contributors, http://js.foundation\n"
          " *\n"
          " * Licensed under the Apache License, Version 2.0 (the \"License\");\n"
          " * you may not use this file except in compliance with the License.\n"
          " * You may obtain a copy of the License at\n"
          " *\n"
          " *     http://www.apache.org/licenses/LICENSE-2.0\n"
          " *\n"
          " * Unless required by applicable law or agreed to in writing, software\n"
          " * distributed under the License is distributed on an \"AS IS\" BASIS\n"
          " * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\n"
          " * See the License for the specific language governing permissions and\n"
          " * limitations under the License.\n"
          " */\n"
          "\n"
          "/* This file is automatically generated by the gen-test-math.sh script.\n"
          " * Do not edit! */\n"
          "\n");

  /* acos tests */
  GEN_DBL_TEST (acos (0.0));
  GEN_DBL_TEST (acos (-0.0));
  GEN_DBL_TEST (acos (1.0));
  GEN_DBL_TEST (acos (-1.0));
  GEN_DBL_TEST (acos (0.5));
  GEN_DBL_TEST (acos (-0.5));
  GEN_DBL_TEST (acos (INFINITY));
  GEN_DBL_TEST (acos (-INFINITY));
  GEN_DBL_TEST (acos (NAN));
  GEN_DBL_TEST (acos (6.9e-18));
  GEN_DBL_TEST (acos (-6.9e-18));
  GEN_DBL_TEST (acos (7.0e-18));
  GEN_DBL_TEST (acos (-7.0e-18));
  GEN_DBL_TEST (acos (7.4e-9));
  GEN_DBL_TEST (acos (-7.4e-9));
  GEN_DBL_TEST (acos (7.5e-9));
  GEN_DBL_TEST (acos (-7.5e-9));
  GEN_DBL_TEST (acos (0.1));
  GEN_DBL_TEST (acos (-0.1));
  GEN_DBL_TEST (acos (0.4));
  GEN_DBL_TEST (acos (-0.4));
  GEN_DBL_TEST (acos (0.6));
  GEN_DBL_TEST (acos (-0.6));
  GEN_DBL_TEST (acos (0.99));
  GEN_DBL_TEST (acos (-0.99));
  GEN_DBL_TEST (acos (1.1));
  GEN_DBL_TEST (acos (-1.1));
  GEN_DBL_TEST (acos (0.7));

  /* asin tests*/
  GEN_DBL_TEST (asin (0.0));
  GEN_DBL_TEST (asin (-0.0));
  GEN_DBL_TEST (asin (1.0));
  GEN_DBL_TEST (asin (-1.0));
  GEN_DBL_TEST (asin (0.5));
  GEN_DBL_TEST (asin (-0.5));
  GEN_DBL_TEST (asin (0.98));
  GEN_DBL_TEST (asin (-0.98));
  GEN_DBL_TEST (asin (INFINITY));
  GEN_DBL_TEST (asin (-INFINITY));
  GEN_DBL_TEST (asin (NAN));
  GEN_DBL_TEST (asin (6.9e-18));
  GEN_DBL_TEST (asin (-6.9e-18));
  GEN_DBL_TEST (asin (7.0e-18));
  GEN_DBL_TEST (asin (-7.0e-18));
  GEN_DBL_TEST (asin (7.4e-9));
  GEN_DBL_TEST (asin (-7.4e-9));
  GEN_DBL_TEST (asin (7.5e-9));
  GEN_DBL_TEST (asin (-7.5e-9));
  GEN_DBL_TEST (asin (0.1));
  GEN_DBL_TEST (asin (-0.1));
  GEN_DBL_TEST (asin (0.4));
  GEN_DBL_TEST (asin (-0.4));
  GEN_DBL_TEST (asin (0.6));
  GEN_DBL_TEST (asin (-0.6));
  GEN_DBL_TEST (asin (0.97));
  GEN_DBL_TEST (asin (-0.97));
  GEN_DBL_TEST (asin (0.99));
  GEN_DBL_TEST (asin (-0.99));
  GEN_DBL_TEST (asin (1.1));
  GEN_DBL_TEST (asin (-1.1));
  GEN_DBL_TEST (asin (0.7));

  /* atan tests*/
  GEN_DBL_TEST (atan (0.0));
  GEN_DBL_TEST (atan (-0.0));
  GEN_DBL_TEST (atan (7.0 / 16.0));
  GEN_DBL_TEST (atan (-7.0 / 16.0));
  GEN_DBL_TEST (atan (11.0 / 16.0));
  GEN_DBL_TEST (atan (-11.0 / 16.0));
  GEN_DBL_TEST (atan (19.0 / 16.0));
  GEN_DBL_TEST (atan (-19.0 / 16.0));
  GEN_DBL_TEST (atan (39.0 / 16.0));
  GEN_DBL_TEST (atan (-39.0 / 16.0));
  GEN_DBL_TEST (atan (1.0));
  GEN_DBL_TEST (atan (-1.0));
  GEN_DBL_TEST (atan (INFINITY));
  GEN_DBL_TEST (atan (-INFINITY));
  GEN_DBL_TEST (atan (NAN));
  GEN_DBL_TEST (atan (6.9 / 16.0));
  GEN_DBL_TEST (atan (-6.9 / 16.0));
  GEN_DBL_TEST (atan (7.1 / 16.0));
  GEN_DBL_TEST (atan (-7.1 / 16.0));
  GEN_DBL_TEST (atan (10.9 / 16.0));
  GEN_DBL_TEST (atan (-10.9 / 16.0));
  GEN_DBL_TEST (atan (11.1 / 16.0));
  GEN_DBL_TEST (atan (-11.1 / 16.0));
  GEN_DBL_TEST (atan (18.9 / 16.0));
  GEN_DBL_TEST (atan (-18.9 / 16.0));
  GEN_DBL_TEST (atan (19.1 / 16.0));
  GEN_DBL_TEST (atan (-19.1 / 16.0));
  GEN_DBL_TEST (atan (38.9 / 16.0));
  GEN_DBL_TEST (atan (-38.9 / 16.0));
  GEN_DBL_TEST (atan (39.1 / 16.0));
  GEN_DBL_TEST (atan (-39.1 / 16.0));
  GEN_DBL_TEST (atan (0.99));
  GEN_DBL_TEST (atan (-0.99));
  GEN_DBL_TEST (atan (1.1));
  GEN_DBL_TEST (atan (-1.1));
  GEN_DBL_TEST (atan (7.37e+19));
  GEN_DBL_TEST (atan (-7.37e+19));
  GEN_DBL_TEST (atan (7.38e+19));
  GEN_DBL_TEST (atan (-7.38e+19));
  GEN_DBL_TEST (atan (0.7));

  /* atan2 tests*/
  GEN_DBL_TEST (atan2 (NAN, NAN));
  GEN_DBL_TEST (atan2 (0.0, NAN));
  GEN_DBL_TEST (atan2 (-0.0, NAN));
  GEN_DBL_TEST (atan2 (1.0, NAN));
  GEN_DBL_TEST (atan2 (-1.0, NAN));
  GEN_DBL_TEST (atan2 (INFINITY, NAN));
  GEN_DBL_TEST (atan2 (-INFINITY, NAN));
  GEN_DBL_TEST (atan2 (NAN, 0.0));
  GEN_DBL_TEST (atan2 (NAN, -0.0));
  GEN_DBL_TEST (atan2 (NAN, 1.0));
  GEN_DBL_TEST (atan2 (NAN, -1.0));
  GEN_DBL_TEST (atan2 (NAN, INFINITY));
  GEN_DBL_TEST (atan2 (NAN, -INFINITY));
  GEN_DBL_TEST (atan2 (0.0, 0.0));
  GEN_DBL_TEST (atan2 (0.0, -0.0));
  GEN_DBL_TEST (atan2 (-0.0, 0.0));
  GEN_DBL_TEST (atan2 (-0.0, -0.0));
  GEN_DBL_TEST (atan2 (0.0, 1.0));
  GEN_DBL_TEST (atan2 (0.0, -1.0));
  GEN_DBL_TEST (atan2 (0.0, INFINITY));
  GEN_DBL_TEST (atan2 (0.0, -INFINITY));
  GEN_DBL_TEST (atan2 (-0.0, 1.0));
  GEN_DBL_TEST (atan2 (-0.0, -1.0));
  GEN_DBL_TEST (atan2 (-0.0, INFINITY));
  GEN_DBL_TEST (atan2 (-0.0, -INFINITY));
  GEN_DBL_TEST (atan2 (1.0, 0.0));
  GEN_DBL_TEST (atan2 (1.0, -0.0));
  GEN_DBL_TEST (atan2 (INFINITY, 0.0));
  GEN_DBL_TEST (atan2 (INFINITY, -0.0));
  GEN_DBL_TEST (atan2 (-1.0, 0.0));
  GEN_DBL_TEST (atan2 (-1.0, -0.0));
  GEN_DBL_TEST (atan2 (-INFINITY, 0.0));
  GEN_DBL_TEST (atan2 (-INFINITY, -0.0));
  GEN_DBL_TEST (atan2 (1.0, INFINITY));
  GEN_DBL_TEST (atan2 (-1.0, INFINITY));
  GEN_DBL_TEST (atan2 (1.0, -INFINITY));
  GEN_DBL_TEST (atan2 (-1.0, -INFINITY));
  GEN_DBL_TEST (atan2 (INFINITY, INFINITY));
  GEN_DBL_TEST (atan2 (INFINITY, -INFINITY));
  GEN_DBL_TEST (atan2 (-INFINITY, INFINITY));
  GEN_DBL_TEST (atan2 (-INFINITY, -INFINITY));
  GEN_DBL_TEST (atan2 (INFINITY, 1.0));
  GEN_DBL_TEST (atan2 (INFINITY, -1.0));
  GEN_DBL_TEST (atan2 (-INFINITY, 1.0));
  GEN_DBL_TEST (atan2 (-INFINITY, -1.0));
  GEN_DBL_TEST (atan2 (0.7, 1.0));
  GEN_DBL_TEST (atan2 (-0.7, 1.0));
  GEN_DBL_TEST (atan2 (0.7, -1.0));
  GEN_DBL_TEST (atan2 (-0.7, -1.0));
  GEN_DBL_TEST (atan2 (0.4, 0.0003));
  GEN_DBL_TEST (atan2 (1.4, -0.93));

  /* acosh tests */
  GEN_DBL_TEST (acosh (0.0));
  GEN_DBL_TEST (acosh (-0.0));
  GEN_DBL_TEST (acosh (1.0));
  GEN_DBL_TEST (acosh (-1.0));
  GEN_DBL_TEST (acosh (INFINITY));
  GEN_DBL_TEST (acosh (-INFINITY));
  GEN_DBL_TEST (acosh (NAN));
  GEN_DBL_TEST (acosh (7.08e+02));
  GEN_DBL_TEST (acosh (7.10e+02));
  GEN_DBL_TEST (acosh (-7.40e+02));
  GEN_DBL_TEST (acosh (-7.50e+02));
  GEN_DBL_TEST (acosh (0.34));
  GEN_DBL_TEST (acosh (-0.34));
  GEN_DBL_TEST (acosh (0.35));
  GEN_DBL_TEST (acosh (-0.35));
  GEN_DBL_TEST (acosh (1.03));
  GEN_DBL_TEST (acosh (-1.03));
  GEN_DBL_TEST (acosh (1.04));
  GEN_DBL_TEST (acosh (-1.04));
  GEN_DBL_TEST (acosh (3.72e-09));
  GEN_DBL_TEST (acosh (-3.72e-09));
  GEN_DBL_TEST (acosh (3.73e-09));
  GEN_DBL_TEST (acosh (-3.73e-09));
  GEN_DBL_TEST (acosh (2.0));
  GEN_DBL_TEST (acosh (3.0));
  GEN_DBL_TEST (acosh (0.7));
  GEN_DBL_TEST (acosh (38.0));

  /* asinh tests */
  GEN_DBL_TEST (asinh (0.0));
  GEN_DBL_TEST (asinh (-0.0));
  GEN_DBL_TEST (asinh (1.0));
  GEN_DBL_TEST (asinh (-1.0));
  GEN_DBL_TEST (asinh (INFINITY));
  GEN_DBL_TEST (asinh (-INFINITY));
  GEN_DBL_TEST (asinh (NAN));
  GEN_DBL_TEST (asinh (7.08e+02));
  GEN_DBL_TEST (asinh (7.10e+02));
  GEN_DBL_TEST (asinh (-7.40e+02));
  GEN_DBL_TEST (asinh (-7.50e+02));
  GEN_DBL_TEST (asinh (0.34));
  GEN_DBL_TEST (asinh (-0.34));
  GEN_DBL_TEST (asinh (0.35));
  GEN_DBL_TEST (asinh (-0.35));
  GEN_DBL_TEST (asinh (1.03));
  GEN_DBL_TEST (asinh (-1.03));
  GEN_DBL_TEST (asinh (1.04));
  GEN_DBL_TEST (asinh (-1.04));
  GEN_DBL_TEST (asinh (3.72e-09));
  GEN_DBL_TEST (asinh (-3.72e-09));
  GEN_DBL_TEST (asinh (3.73e-09));
  GEN_DBL_TEST (asinh (-3.73e-09));
  GEN_DBL_TEST (asinh (2.0));
  GEN_DBL_TEST (asinh (3.0));
  GEN_DBL_TEST (asinh (0.7));
  GEN_DBL_TEST (asinh (38.0));

  /* atanh tests */
  GEN_DBL_TEST (atanh (0.0));
  GEN_DBL_TEST (atanh (-0.0));
  GEN_DBL_TEST (atanh (1.0));
  GEN_DBL_TEST (atanh (-1.0));
  GEN_DBL_TEST (atanh (INFINITY));
  GEN_DBL_TEST (atanh (-INFINITY));
  GEN_DBL_TEST (atanh (NAN));
  GEN_DBL_TEST (atanh (7.08e+02));
  GEN_DBL_TEST (atanh (7.10e+02));
  GEN_DBL_TEST (atanh (-7.40e+02));
  GEN_DBL_TEST (atanh (-7.50e+02));
  GEN_DBL_TEST (atanh (0.34));
  GEN_DBL_TEST (atanh (-0.34));
  GEN_DBL_TEST (atanh (0.35));
  GEN_DBL_TEST (atanh (-0.35));
  GEN_DBL_TEST (atanh (1.03));
  GEN_DBL_TEST (atanh (-1.03));
  GEN_DBL_TEST (atanh (1.04));
  GEN_DBL_TEST (atanh (-1.04));
  GEN_DBL_TEST (atanh (3.72e-09));
  GEN_DBL_TEST (atanh (-3.72e-09));
  GEN_DBL_TEST (atanh (3.73e-09));
  GEN_DBL_TEST (atanh (-3.73e-09));
  GEN_DBL_TEST (atanh (2.0));
  GEN_DBL_TEST (atanh (3.0));
  GEN_DBL_TEST (atanh (0.7));
  GEN_DBL_TEST (atanh (38.0));

  /* ceil tests */
  GEN_DBL_TEST (ceil (0.0));
  GEN_DBL_TEST (ceil (-0.0));
  GEN_DBL_TEST (ceil (INFINITY));
  GEN_DBL_TEST (ceil (-INFINITY));
  GEN_DBL_TEST (ceil (NAN));
  GEN_DBL_TEST (ceil (3.14));
  GEN_DBL_TEST (ceil (-3.14));
  GEN_DBL_TEST (ceil (3.72e-09));
  GEN_DBL_TEST (ceil (-3.72e-09));
  GEN_DBL_TEST (ceil (7.37e+19));
  GEN_DBL_TEST (ceil (-7.37e+19));

  /* copysign tests */
  /* SKIPPED: not publicly declared in jerry-math
  GEN_DBL_TEST (copysign (0.0, 0.0));
  GEN_DBL_TEST (copysign (0.0, -0.0));
  GEN_DBL_TEST (copysign (-0.0, 0.0));
  GEN_DBL_TEST (copysign (-0.0, -0.0));
  GEN_DBL_TEST (copysign (0.0, 1.0));
  GEN_DBL_TEST (copysign (0.0, -1.0));
  GEN_DBL_TEST (copysign (-0.0, 1.0));
  GEN_DBL_TEST (copysign (-0.0, -1.0));
  GEN_DBL_TEST (copysign (0.0, INFINITY));
  GEN_DBL_TEST (copysign (0.0, -INFINITY));
  GEN_DBL_TEST (copysign (-0.0, INFINITY));
  GEN_DBL_TEST (copysign (-0.0, -INFINITY));
  GEN_DBL_TEST (copysign (0.0, NAN));
  GEN_DBL_TEST (copysign (-0.0, NAN));
  GEN_DBL_TEST (copysign (1.0, 0.0));
  GEN_DBL_TEST (copysign (1.0, -0.0));
  GEN_DBL_TEST (copysign (-1.0, 0.0));
  GEN_DBL_TEST (copysign (-1.0, -0.0));
  GEN_DBL_TEST (copysign (1.0, 1.0));
  GEN_DBL_TEST (copysign (1.0, -1.0));
  GEN_DBL_TEST (copysign (-1.0, 1.0));
  GEN_DBL_TEST (copysign (-1.0, -1.0));
  GEN_DBL_TEST (copysign (1.0, INFINITY));
  GEN_DBL_TEST (copysign (1.0, -INFINITY));
  GEN_DBL_TEST (copysign (-1.0, INFINITY));
  GEN_DBL_TEST (copysign (-1.0, -INFINITY));
  GEN_DBL_TEST (copysign (1.0, NAN));
  GEN_DBL_TEST (copysign (-1.0, NAN));
  GEN_DBL_TEST (copysign (INFINITY, 0.0));
  GEN_DBL_TEST (copysign (INFINITY, -0.0));
  GEN_DBL_TEST (copysign (-INFINITY, 0.0));
  GEN_DBL_TEST (copysign (-INFINITY, -0.0));
  GEN_DBL_TEST (copysign (INFINITY, 1.0));
  GEN_DBL_TEST (copysign (INFINITY, -1.0));
  GEN_DBL_TEST (copysign (-INFINITY, 1.0));
  GEN_DBL_TEST (copysign (-INFINITY, -1.0));
  GEN_DBL_TEST (copysign (INFINITY, INFINITY));
  GEN_DBL_TEST (copysign (INFINITY, -INFINITY));
  GEN_DBL_TEST (copysign (-INFINITY, INFINITY));
  GEN_DBL_TEST (copysign (-INFINITY, -INFINITY));
  GEN_DBL_TEST (copysign (INFINITY, NAN));
  GEN_DBL_TEST (copysign (-INFINITY, NAN));
  GEN_DBL_TEST (copysign (NAN, 0.0));
  GEN_DBL_TEST (copysign (NAN, -0.0));
  GEN_DBL_TEST (copysign (NAN, 1.0));
  GEN_DBL_TEST (copysign (NAN, -1.0));
  GEN_DBL_TEST (copysign (NAN, INFINITY));
  GEN_DBL_TEST (copysign (NAN, -INFINITY));
  GEN_DBL_TEST (copysign (NAN, NAN));
  GEN_DBL_TEST (copysign (3.14, -1.0));
  GEN_DBL_TEST (copysign (-3.14, 1.0));
  GEN_DBL_TEST (copysign (1.0, -3.14));
  GEN_DBL_TEST (copysign (-1.0, 3.14));
  */

  /* exp tests */
  GEN_DBL_TEST (exp (0.0));
  GEN_DBL_TEST (exp (-0.0));
  GEN_DBL_TEST (exp (1.0));
  GEN_DBL_TEST (exp (-1.0));
  GEN_DBL_TEST (exp (INFINITY));
  GEN_DBL_TEST (exp (-INFINITY));
  GEN_DBL_TEST (exp (NAN));
  GEN_DBL_TEST (exp (7.08e+02));
  GEN_DBL_TEST (exp (7.10e+02));
  GEN_DBL_TEST (exp (-7.40e+02));
  GEN_DBL_TEST (exp (-7.50e+02));
  GEN_DBL_TEST (exp (0.34));
  GEN_DBL_TEST (exp (-0.34));
  GEN_DBL_TEST (exp (0.35));
  GEN_DBL_TEST (exp (-0.35));
  GEN_DBL_TEST (exp (1.03));
  GEN_DBL_TEST (exp (-1.03));
  GEN_DBL_TEST (exp (1.04));
  GEN_DBL_TEST (exp (-1.04));
  GEN_DBL_TEST (exp (3.72e-09));
  GEN_DBL_TEST (exp (-3.72e-09));
  GEN_DBL_TEST (exp (3.73e-09));
  GEN_DBL_TEST (exp (-3.73e-09));
  GEN_DBL_TEST (exp (2.0));
  GEN_DBL_TEST (exp (3.0));
  GEN_DBL_TEST (exp (0.7));
  GEN_DBL_TEST (exp (38.0));

  /* expm1 tests */
  GEN_DBL_TEST (expm1 (0.0));
  GEN_DBL_TEST (expm1 (-0.0));
  GEN_DBL_TEST (expm1 (1.0));
  GEN_DBL_TEST (expm1 (-1.0));
  GEN_DBL_TEST (expm1 (INFINITY));
  GEN_DBL_TEST (expm1 (-INFINITY));
  GEN_DBL_TEST (expm1 (NAN));
  GEN_DBL_TEST (expm1 (7.08e+02));
  GEN_DBL_TEST (expm1 (7.095e+02));
  GEN_DBL_TEST (expm1 (7.10e+02));
  GEN_DBL_TEST (expm1 (-7.40e+02));
  GEN_DBL_TEST (expm1 (-7.50e+02));
  GEN_DBL_TEST (expm1 (0.34));
  GEN_DBL_TEST (expm1 (-0.34));
  GEN_DBL_TEST (expm1 (0.35));
  GEN_DBL_TEST (expm1 (-0.35));
  GEN_DBL_TEST (expm1 (1.03));
  GEN_DBL_TEST (expm1 (-1.03));
  GEN_DBL_TEST (expm1 (1.04));
  GEN_DBL_TEST (expm1 (-1.04));
  GEN_DBL_TEST (expm1 (3.72e-09));
  GEN_DBL_TEST (expm1 (-3.72e-09));
  GEN_DBL_TEST (expm1 (3.73e-09));
  GEN_DBL_TEST (expm1 (-3.73e-09));
  GEN_DBL_TEST (expm1 (2.0));
  GEN_DBL_TEST (expm1 (3.0));
  GEN_DBL_TEST (expm1 (0.7));
  GEN_DBL_TEST (expm1 (38.0));

  /* fabs tests */
  GEN_DBL_TEST (fabs (0.0));
  GEN_DBL_TEST (fabs (-0.0));
  GEN_DBL_TEST (fabs (1.0));
  GEN_DBL_TEST (fabs (-1.0));
  GEN_DBL_TEST (fabs (INFINITY));
  GEN_DBL_TEST (fabs (-INFINITY));
  GEN_DBL_TEST (fabs (NAN));
  GEN_DBL_TEST (fabs (3.14));
  GEN_DBL_TEST (fabs (-3.14));
  GEN_DBL_TEST (fabs (0.7));
  GEN_DBL_TEST (fabs (-0.7));
  GEN_DBL_TEST (fabs (3.72e-09));
  GEN_DBL_TEST (fabs (-3.72e-09));
  GEN_DBL_TEST (fabs (7.37e+19));
  GEN_DBL_TEST (fabs (-7.37e+19));

  /* floor tests */
  GEN_DBL_TEST (floor (0.0));
  GEN_DBL_TEST (floor (-0.0));
  GEN_DBL_TEST (floor (INFINITY));
  GEN_DBL_TEST (floor (-INFINITY));
  GEN_DBL_TEST (floor (NAN));
  GEN_DBL_TEST (floor (3.14));
  GEN_DBL_TEST (floor (-3.14));
  GEN_DBL_TEST (floor (3.72e-09));
  GEN_DBL_TEST (floor (-3.72e-09));
  GEN_DBL_TEST (floor (7.37e+19));
  GEN_DBL_TEST (floor (-7.37e+19));

  /* fmod tests */
  GEN_DBL_TEST (fmod (0.0, 0.0));
  GEN_DBL_TEST (fmod (0.0, -0.0));
  GEN_DBL_TEST (fmod (-0.0, 0.0));
  GEN_DBL_TEST (fmod (-0.0, -0.0));
  GEN_DBL_TEST (fmod (0.0, 3.0));
  GEN_DBL_TEST (fmod (0.0, -3.0));
  GEN_DBL_TEST (fmod (-0.0, 3.0));
  GEN_DBL_TEST (fmod (-0.0, -3.0));
  GEN_DBL_TEST (fmod (0.0, INFINITY));
  GEN_DBL_TEST (fmod (0.0, -INFINITY));
  GEN_DBL_TEST (fmod (-0.0, INFINITY));
  GEN_DBL_TEST (fmod (-0.0, -INFINITY));
  GEN_DBL_TEST (fmod (0.0, NAN));
  GEN_DBL_TEST (fmod (-0.0, NAN));
  GEN_DBL_TEST (fmod (3.0, 0.0));
  GEN_DBL_TEST (fmod (3.0, -0.0));
  GEN_DBL_TEST (fmod (-3.0, 0.0));
  GEN_DBL_TEST (fmod (-3.0, -0.0));
  GEN_DBL_TEST (fmod (3.0, 3.0));
  GEN_DBL_TEST (fmod (3.0, -3.0));
  GEN_DBL_TEST (fmod (-3.0, 3.0));
  GEN_DBL_TEST (fmod (-3.0, -3.0));
  GEN_DBL_TEST (fmod (3.0, INFINITY));
  GEN_DBL_TEST (fmod (3.0, -INFINITY));
  GEN_DBL_TEST (fmod (-3.0, INFINITY));
  GEN_DBL_TEST (fmod (-3.0, -INFINITY));
  GEN_DBL_TEST (fmod (3.0, NAN));
  GEN_DBL_TEST (fmod (-3.0, NAN));
  GEN_DBL_TEST (fmod (INFINITY, 0.0));
  GEN_DBL_TEST (fmod (INFINITY, -0.0));
  GEN_DBL_TEST (fmod (-INFINITY, 0.0));
  GEN_DBL_TEST (fmod (-INFINITY, -0.0));
  GEN_DBL_TEST (fmod (INFINITY, 3.0));
  GEN_DBL_TEST (fmod (INFINITY, -3.0));
  GEN_DBL_TEST (fmod (-INFINITY, 3.0));
  GEN_DBL_TEST (fmod (-INFINITY, -3.0));
  GEN_DBL_TEST (fmod (INFINITY, INFINITY));
  GEN_DBL_TEST (fmod (INFINITY, -INFINITY));
  GEN_DBL_TEST (fmod (-INFINITY, INFINITY));
  GEN_DBL_TEST (fmod (-INFINITY, -INFINITY));
  GEN_DBL_TEST (fmod (INFINITY, NAN));
  GEN_DBL_TEST (fmod (-INFINITY, NAN));
  GEN_DBL_TEST (fmod (NAN, 0.0));
  GEN_DBL_TEST (fmod (NAN, -0.0));
  GEN_DBL_TEST (fmod (NAN, 3.0));
  GEN_DBL_TEST (fmod (NAN, -3.0));
  GEN_DBL_TEST (fmod (NAN, INFINITY));
  GEN_DBL_TEST (fmod (NAN, -INFINITY));
  GEN_DBL_TEST (fmod (NAN, NAN));
  GEN_DBL_TEST (fmod (3.0, 1.0));
  GEN_DBL_TEST (fmod (3.0, -1.0));
  GEN_DBL_TEST (fmod (-3.0, 1.0));
  GEN_DBL_TEST (fmod (-3.0, -1.0));
  GEN_DBL_TEST (fmod (6.5, 2.3));
  GEN_DBL_TEST (fmod (6.5, -2.3));
  GEN_DBL_TEST (fmod (-6.5, 2.3));
  GEN_DBL_TEST (fmod (-6.5, -2.3));

  /* isfinite tests */
  GEN_INT_TEST (isfinite (0.0));
  GEN_INT_TEST (isfinite (-0.0));
  GEN_INT_TEST (isfinite (1.0));
  GEN_INT_TEST (isfinite (-1.0));
  GEN_INT_TEST (isfinite (INFINITY));
  GEN_INT_TEST (isfinite (-INFINITY));
  GEN_INT_TEST (isfinite (NAN));
  GEN_INT_TEST (isfinite (3.14));
  GEN_INT_TEST (isfinite (-3.14));
  GEN_INT_TEST (isfinite (0.7));
  GEN_INT_TEST (isfinite (-0.7));
  GEN_INT_TEST (isfinite (3.72e-09));
  GEN_INT_TEST (isfinite (-3.72e-09));
  GEN_INT_TEST (isfinite (7.37e+19));
  GEN_INT_TEST (isfinite (-7.37e+19));

  /* isinf tests */
  GEN_INT_TEST (isinf (0.0));
  GEN_INT_TEST (isinf (-0.0));
  GEN_INT_TEST (isinf (1.0));
  GEN_INT_TEST (isinf (-1.0));
  GEN_INT_TEST (isinf (INFINITY));
  GEN_INT_TEST (isinf (-INFINITY));
  GEN_INT_TEST (isinf (NAN));
  GEN_INT_TEST (isinf (3.14));
  GEN_INT_TEST (isinf (-3.14));
  GEN_INT_TEST (isinf (0.7));
  GEN_INT_TEST (isinf (-0.7));
  GEN_INT_TEST (isinf (3.72e-09));
  GEN_INT_TEST (isinf (-3.72e-09));
  GEN_INT_TEST (isinf (7.37e+19));
  GEN_INT_TEST (isinf (-7.37e+19));

  /* isnan tests */
  GEN_INT_TEST (isnan (0.0));
  GEN_INT_TEST (isnan (-0.0));
  GEN_INT_TEST (isnan (1.0));
  GEN_INT_TEST (isnan (-1.0));
  GEN_INT_TEST (isnan (INFINITY));
  GEN_INT_TEST (isnan (-INFINITY));
  GEN_INT_TEST (isnan (NAN));
  GEN_INT_TEST (isnan (3.14));
  GEN_INT_TEST (isnan (-3.14));
  GEN_INT_TEST (isnan (0.7));
  GEN_INT_TEST (isnan (-0.7));
  GEN_INT_TEST (isnan (3.72e-09));
  GEN_INT_TEST (isnan (-3.72e-09));
  GEN_INT_TEST (isnan (7.37e+19));
  GEN_INT_TEST (isnan (-7.37e+19));

  /* log tests */
  GEN_DBL_TEST (log (0.0));
  GEN_DBL_TEST (log (-0.0));
  GEN_DBL_TEST (log (1.0));
  GEN_DBL_TEST (log (-1.0));
  GEN_DBL_TEST (log (INFINITY));
  GEN_DBL_TEST (log (-INFINITY));
  GEN_DBL_TEST (log (NAN));
  GEN_DBL_TEST (log (M_E));
  GEN_DBL_TEST (log (1.0 / M_E));
  GEN_DBL_TEST (log (2));
  GEN_DBL_TEST (log (10));
  GEN_DBL_TEST (log (0.7));
  GEN_DBL_TEST (log (2.22e-308));
  GEN_DBL_TEST (log (2.23e-308));
  GEN_DBL_TEST (log (0.17));
  GEN_DBL_TEST (log (0.18));
  GEN_DBL_TEST (log (1999.0));
  GEN_DBL_TEST (log (2000.0));
  GEN_DBL_TEST (log (2001.0));

  /* log1p tests */
  GEN_DBL_TEST (log1p (0.0));
  GEN_DBL_TEST (log1p (-0.0));
  GEN_DBL_TEST (log1p (1.0));
  GEN_DBL_TEST (log1p (-1.0));
  GEN_DBL_TEST (log1p (INFINITY));
  GEN_DBL_TEST (log1p (-INFINITY));
  GEN_DBL_TEST (log1p (NAN));
  GEN_DBL_TEST (log1p (M_E));
  GEN_DBL_TEST (log1p (1.0 / M_E));
  GEN_DBL_TEST (log1p (2));
  GEN_DBL_TEST (log1p (10));
  GEN_DBL_TEST (log1p (0.7));
  GEN_DBL_TEST (log1p (2.22e-308));
  GEN_DBL_TEST (log1p (2.23e-308));
  GEN_DBL_TEST (log1p (0.17));
  GEN_DBL_TEST (log1p (0.18));
  GEN_DBL_TEST (log1p (1999.0));
  GEN_DBL_TEST (log1p (2000.0));
  GEN_DBL_TEST (log1p (2001.0));

  /* log2 tests */
  GEN_DBL_TEST (log2 (0.0));
  GEN_DBL_TEST (log2 (-0.0));
  GEN_DBL_TEST (log2 (1.0));
  GEN_DBL_TEST (log2 (-1.0));
  GEN_DBL_TEST (log2 (INFINITY));
  GEN_DBL_TEST (log2 (-INFINITY));
  GEN_DBL_TEST (log2 (NAN));
  GEN_DBL_TEST (log2 (M_E));
  GEN_DBL_TEST (log2 (1.0 / M_E));
  GEN_DBL_TEST (log2 (2));
  GEN_DBL_TEST (log2 (10));
  GEN_DBL_TEST (log2 (0.7));
  GEN_DBL_TEST (log2 (2.22e-308));
  GEN_DBL_TEST (log2 (2.23e-308));
  GEN_DBL_TEST (log2 (0.17));
  GEN_DBL_TEST (log2 (0.18));
  GEN_DBL_TEST (log2 (1999.0));
  GEN_DBL_TEST (log2 (2000.0));
  GEN_DBL_TEST (log2 (2001.0));

  /* log10 tests */
  GEN_DBL_TEST (log10 (0.0));
  GEN_DBL_TEST (log10 (-0.0));
  GEN_DBL_TEST (log10 (1.0));
  GEN_DBL_TEST (log10 (-1.0));
  GEN_DBL_TEST (log10 (INFINITY));
  GEN_DBL_TEST (log10 (-INFINITY));
  GEN_DBL_TEST (log10 (NAN));
  GEN_DBL_TEST (log10 (M_E));
  GEN_DBL_TEST (log10 (1.0 / M_E));
  GEN_DBL_TEST (log10 (2));
  GEN_DBL_TEST (log10 (10));
  GEN_DBL_TEST (log10 (0.7));
  GEN_DBL_TEST (log10 (2.22e-308));
  GEN_DBL_TEST (log10 (2.23e-308));
  GEN_DBL_TEST (log10 (0.17));
  GEN_DBL_TEST (log10 (0.18));
  GEN_DBL_TEST (log10 (1999.0));
  GEN_DBL_TEST (log10 (2000.0));
  GEN_DBL_TEST (log10 (2001.0));

  /* pow tests */
  GEN_DBL_TEST (pow (0.0, 0.0));
  GEN_DBL_TEST (pow (0.0, -0.0));
  GEN_DBL_TEST (pow (-0.0, 0.0));
  GEN_DBL_TEST (pow (-0.0, -0.0));
  GEN_DBL_TEST (pow (0.0, 1.0));
  GEN_DBL_TEST (pow (0.0, -1.0));
  GEN_DBL_TEST (pow (-0.0, 1.0));
  GEN_DBL_TEST (pow (-0.0, -1.0));
  GEN_DBL_TEST (pow (0.0, INFINITY));
  GEN_DBL_TEST (pow (0.0, -INFINITY));
  GEN_DBL_TEST (pow (-0.0, INFINITY));
  GEN_DBL_TEST (pow (-0.0, -INFINITY));
  GEN_DBL_TEST (pow (0.0, NAN));
  GEN_DBL_TEST (pow (-0.0, NAN));
  GEN_DBL_TEST (pow (1.0, 0.0));
  GEN_DBL_TEST (pow (1.0, -0.0));
  GEN_DBL_TEST (pow (-1.0, 0.0));
  GEN_DBL_TEST (pow (-1.0, -0.0));
  GEN_DBL_TEST (pow (1.0, 1.0));
  GEN_DBL_TEST (pow (1.0, -1.0));
  GEN_DBL_TEST (pow (-1.0, 1.0));
  GEN_DBL_TEST (pow (-1.0, -1.0));
  GEN_DBL_TEST (pow (1.0, INFINITY));
  GEN_DBL_TEST (pow (1.0, -INFINITY));
  GEN_DBL_TEST (pow (-1.0, INFINITY));
  GEN_DBL_TEST (pow (-1.0, -INFINITY));
  GEN_DBL_TEST (pow (1.0, NAN));
  GEN_DBL_TEST (pow (-1.0, NAN));
  GEN_DBL_TEST (pow (INFINITY, 0.0));
  GEN_DBL_TEST (pow (INFINITY, -0.0));
  GEN_DBL_TEST (pow (-INFINITY, 0.0));
  GEN_DBL_TEST (pow (-INFINITY, -0.0));
  GEN_DBL_TEST (pow (INFINITY, 1.0));
  GEN_DBL_TEST (pow (INFINITY, -1.0));
  GEN_DBL_TEST (pow (-INFINITY, 1.0));
  GEN_DBL_TEST (pow (-INFINITY, -1.0));
  GEN_DBL_TEST (pow (INFINITY, INFINITY));
  GEN_DBL_TEST (pow (INFINITY, -INFINITY));
  GEN_DBL_TEST (pow (-INFINITY, INFINITY));
  GEN_DBL_TEST (pow (-INFINITY, -INFINITY));
  GEN_DBL_TEST (pow (INFINITY, NAN));
  GEN_DBL_TEST (pow (-INFINITY, NAN));
  GEN_DBL_TEST (pow (NAN, 0.0));
  GEN_DBL_TEST (pow (NAN, -0.0));
  GEN_DBL_TEST (pow (NAN, 1.0));
  GEN_DBL_TEST (pow (NAN, -1.0));
  GEN_DBL_TEST (pow (NAN, INFINITY));
  GEN_DBL_TEST (pow (NAN, -INFINITY));
  GEN_DBL_TEST (pow (NAN, NAN));
  GEN_DBL_TEST (pow (0.9, INFINITY));
  GEN_DBL_TEST (pow (0.9, -INFINITY));
  GEN_DBL_TEST (pow (-0.9, INFINITY));
  GEN_DBL_TEST (pow (-0.9, -INFINITY));
  GEN_DBL_TEST (pow (1.1, INFINITY));
  GEN_DBL_TEST (pow (1.1, -INFINITY));
  GEN_DBL_TEST (pow (-1.1, INFINITY));
  GEN_DBL_TEST (pow (-1.1, -INFINITY));
  GEN_DBL_TEST (pow (0.0, 2.0));
  GEN_DBL_TEST (pow (0.0, -2.0));
  GEN_DBL_TEST (pow (-0.0, 2.0));
  GEN_DBL_TEST (pow (-0.0, -2.0));
  GEN_DBL_TEST (pow (0.0, 3.0));
  GEN_DBL_TEST (pow (0.0, -3.0));
  GEN_DBL_TEST (pow (-0.0, 3.0));
  GEN_DBL_TEST (pow (-0.0, -3.0));
  GEN_DBL_TEST (pow (0.0, 3.14));
  GEN_DBL_TEST (pow (0.0, -3.14));
  GEN_DBL_TEST (pow (-0.0, 3.14));
  GEN_DBL_TEST (pow (-0.0, -3.14));
  GEN_DBL_TEST (pow (1.0, 3.14));
  GEN_DBL_TEST (pow (1.0, -3.14));
  GEN_DBL_TEST (pow (-1.0, 3.14));
  GEN_DBL_TEST (pow (-1.0, -3.14));
  GEN_DBL_TEST (pow (3.14, 0.0));
  GEN_DBL_TEST (pow (3.14, -0.0));
  GEN_DBL_TEST (pow (-3.14, 0.0));
  GEN_DBL_TEST (pow (-3.14, -0.0));
  GEN_DBL_TEST (pow (3.14, 1.0));
  GEN_DBL_TEST (pow (3.14, -1.0));
  GEN_DBL_TEST (pow (-3.14, 1.0));
  GEN_DBL_TEST (pow (-3.14, -1.0));
  GEN_DBL_TEST (pow (3.14, 2.0));
  GEN_DBL_TEST (pow (3.14, -2.0));
  GEN_DBL_TEST (pow (-3.14, 2.0));
  GEN_DBL_TEST (pow (-3.14, -2.0));
  GEN_DBL_TEST (pow (3.14, 3.0));
  GEN_DBL_TEST (pow (3.14, -3.0));
  GEN_DBL_TEST (pow (-3.14, 3.0));
  GEN_DBL_TEST (pow (-3.14, -3.0));
  GEN_DBL_TEST (pow (3.14, 3.14));
  GEN_DBL_TEST (pow (3.14, -3.14));
  GEN_DBL_TEST (pow (-3.14, 3.14));
  GEN_DBL_TEST (pow (-3.14, -3.14));
  GEN_DBL_TEST (pow (INFINITY, 2.0));
  GEN_DBL_TEST (pow (INFINITY, -2.0));
  GEN_DBL_TEST (pow (-INFINITY, 2.0));
  GEN_DBL_TEST (pow (-INFINITY, -2.0));
  GEN_DBL_TEST (pow (INFINITY, 3.0));
  GEN_DBL_TEST (pow (INFINITY, -3.0));
  GEN_DBL_TEST (pow (-INFINITY, 3.0));
  GEN_DBL_TEST (pow (-INFINITY, -3.0));
  GEN_DBL_TEST (pow (INFINITY, 3.14));
  GEN_DBL_TEST (pow (INFINITY, -3.14));
  GEN_DBL_TEST (pow (-INFINITY, 3.14));
  GEN_DBL_TEST (pow (-INFINITY, -3.14));
  GEN_DBL_TEST (pow (0.7, 1.2));

  /* scalbn tests */
  /* SKIPPED: not publicly declared in jerry-math
  GEN_DBL_TEST (scalbn (0.0, 0));
  GEN_DBL_TEST (scalbn (-0.0, 0));
  GEN_DBL_TEST (scalbn (0.0, 1));
  GEN_DBL_TEST (scalbn (0.0, -1));
  GEN_DBL_TEST (scalbn (-0.0, 1));
  GEN_DBL_TEST (scalbn (-0.0, -1));
  GEN_DBL_TEST (scalbn (1.0, 0));
  GEN_DBL_TEST (scalbn (-1.0, 0));
  GEN_DBL_TEST (scalbn (1.0, 1));
  GEN_DBL_TEST (scalbn (1.0, -1));
  GEN_DBL_TEST (scalbn (-1.0, 1));
  GEN_DBL_TEST (scalbn (-1.0, -1));
  GEN_DBL_TEST (scalbn (INFINITY, 0));
  GEN_DBL_TEST (scalbn (-INFINITY, 0));
  GEN_DBL_TEST (scalbn (INFINITY, 1));
  GEN_DBL_TEST (scalbn (INFINITY, -1));
  GEN_DBL_TEST (scalbn (-INFINITY, 1));
  GEN_DBL_TEST (scalbn (-INFINITY, -1));
  GEN_DBL_TEST (scalbn (NAN, 0));
  GEN_DBL_TEST (scalbn (NAN, 1));
  GEN_DBL_TEST (scalbn (NAN, -1));
  GEN_DBL_TEST (scalbn (3.14, -1));
  GEN_DBL_TEST (scalbn (-3.14, 1));
  GEN_DBL_TEST (scalbn (0.7, 4));
  GEN_DBL_TEST (scalbn (0.7, -4));
  GEN_DBL_TEST (scalbn (-0.7, 4));
  GEN_DBL_TEST (scalbn (-0.7, -4));
  GEN_DBL_TEST (scalbn (0.8, 5));
  GEN_DBL_TEST (scalbn (0.8, -5));
  GEN_DBL_TEST (scalbn (-0.8, 5));
  GEN_DBL_TEST (scalbn (-0.8, -5));
  GEN_DBL_TEST (scalbn (5.55e-18, 49999));
  GEN_DBL_TEST (scalbn (5.55e-18, 50000));
  GEN_DBL_TEST (scalbn (5.55e-18, 50001));
  GEN_DBL_TEST (scalbn (1.0, -49999));
  GEN_DBL_TEST (scalbn (1.0, -50000));
  GEN_DBL_TEST (scalbn (1.0, -50001));
  */

  /* sqrt tests */
  GEN_DBL_TEST (sqrt (0.0));
  GEN_DBL_TEST (sqrt (-0.0));
  GEN_DBL_TEST (sqrt (1.0));
  GEN_DBL_TEST (sqrt (-1.0));
  GEN_DBL_TEST (sqrt (INFINITY));
  GEN_DBL_TEST (sqrt (-INFINITY));
  GEN_DBL_TEST (sqrt (NAN));
  GEN_DBL_TEST (sqrt (0.7));
  GEN_DBL_TEST (sqrt (2));
  GEN_DBL_TEST (sqrt (10));
  GEN_DBL_TEST (sqrt (2.22e-308));
  GEN_DBL_TEST (sqrt (2.23e-308));
  GEN_DBL_TEST (sqrt (3.72e-09));
  GEN_DBL_TEST (sqrt (7.37e+19));
  GEN_DBL_TEST (sqrt (2209));
  GEN_DBL_TEST (sqrt (4));
  GEN_DBL_TEST (sqrt (0.25));
  GEN_DBL_TEST (sqrt (6642.25));
  GEN_DBL_TEST (sqrt (15239.9025));

  /* cbrt tests */
  GEN_DBL_TEST (cbrt (0.0));
  GEN_DBL_TEST (cbrt (-0.0));
  GEN_DBL_TEST (cbrt (1.0));
  GEN_DBL_TEST (cbrt (-1.0));
  GEN_DBL_TEST (cbrt (INFINITY));
  GEN_DBL_TEST (cbrt (-INFINITY));
  GEN_DBL_TEST (cbrt (NAN));
  GEN_DBL_TEST (cbrt (0.7));
  GEN_DBL_TEST (cbrt (2));
  GEN_DBL_TEST (cbrt (10));
  GEN_DBL_TEST (cbrt (2.22e-308));
  GEN_DBL_TEST (cbrt (2.23e-308));
  GEN_DBL_TEST (cbrt (3.72e-09));
  GEN_DBL_TEST (cbrt (7.37e+19));
  GEN_DBL_TEST (cbrt (2209));
  GEN_DBL_TEST (cbrt (4));
  GEN_DBL_TEST (cbrt (0.25));
  GEN_DBL_TEST (cbrt (6642.25));
  GEN_DBL_TEST (cbrt (15239.9025));
  GEN_DBL_TEST (cbrt (3));
  GEN_DBL_TEST (cbrt (9));
  GEN_DBL_TEST (cbrt (-17.87));
  GEN_DBL_TEST (cbrt (-8941));

  /* sin tests */
  GEN_DBL_TEST (sin (0.0));
  GEN_DBL_TEST (sin (-0.0));
  GEN_DBL_TEST (sin (1.0));
  GEN_DBL_TEST (sin (-1.0));
  GEN_DBL_TEST (sin (INFINITY));
  GEN_DBL_TEST (sin (-INFINITY));
  GEN_DBL_TEST (sin (NAN));
  GEN_DBL_TEST (sin (M_PI));
  GEN_DBL_TEST (sin (-M_PI));
  GEN_DBL_TEST (sin (2.0 * M_PI));
  GEN_DBL_TEST (sin (-2.0 * M_PI));
  GEN_DBL_TEST (sin (M_PI / 2.0));
  GEN_DBL_TEST (sin (-M_PI / 2.0));
  GEN_DBL_TEST (sin (M_PI / 3.0));
  GEN_DBL_TEST (sin (-M_PI / 3.0));
  GEN_DBL_TEST (sin (M_PI / 4.0));
  GEN_DBL_TEST (sin (-M_PI / 4.0));
  GEN_DBL_TEST (sin (M_PI / 6.0));
  GEN_DBL_TEST (sin (-M_PI / 6.0));
  GEN_DBL_TEST (sin (M_PI * 2.0 / 3.0));
  GEN_DBL_TEST (sin (-M_PI * 2.0 / 3.0));
  GEN_DBL_TEST (sin (M_PI * 5.0 / 6.0));
  GEN_DBL_TEST (sin (-M_PI * 5.0 / 6.0));
  GEN_DBL_TEST (sin (6.9e-18));
  GEN_DBL_TEST (sin (-6.9e-18));
  GEN_DBL_TEST (sin (7.0e-18));
  GEN_DBL_TEST (sin (-7.0e-18));
  GEN_DBL_TEST (sin (7.4e-9));
  GEN_DBL_TEST (sin (-7.4e-9));
  GEN_DBL_TEST (sin (7.5e-9));
  GEN_DBL_TEST (sin (-7.5e-9));
  GEN_DBL_TEST (sin (0.2));
  GEN_DBL_TEST (sin (-0.2));
  GEN_DBL_TEST (sin (0.4));
  GEN_DBL_TEST (sin (-0.4));
  GEN_DBL_TEST (sin (0.7));
  GEN_DBL_TEST (sin (-0.7));
  GEN_DBL_TEST (sin (0.8));
  GEN_DBL_TEST (sin (-0.8));
  GEN_DBL_TEST (sin (3.0));
  GEN_DBL_TEST (sin (-3.0));
  GEN_DBL_TEST (sin (4.0));
  GEN_DBL_TEST (sin (-4.0));
  GEN_DBL_TEST (sin (6.0));
  GEN_DBL_TEST (sin (-6.0));
  GEN_DBL_TEST (sin (7.0));
  GEN_DBL_TEST (sin (-7.0));

  /* cos tests */
  GEN_DBL_TEST (cos (0.0));
  GEN_DBL_TEST (cos (-0.0));
  GEN_DBL_TEST (cos (1.0));
  GEN_DBL_TEST (cos (-1.0));
  GEN_DBL_TEST (cos (INFINITY));
  GEN_DBL_TEST (cos (-INFINITY));
  GEN_DBL_TEST (cos (NAN));
  GEN_DBL_TEST (cos (M_PI));
  GEN_DBL_TEST (cos (-M_PI));
  GEN_DBL_TEST (cos (2.0 * M_PI));
  GEN_DBL_TEST (cos (-2.0 * M_PI));
  GEN_DBL_TEST (cos (M_PI / 2.0));
  GEN_DBL_TEST (cos (-M_PI / 2.0));
  GEN_DBL_TEST (cos (M_PI / 3.0));
  GEN_DBL_TEST (cos (-M_PI / 3.0));
  GEN_DBL_TEST (cos (M_PI / 4.0));
  GEN_DBL_TEST (cos (-M_PI / 4.0));
  GEN_DBL_TEST (cos (M_PI / 6.0));
  GEN_DBL_TEST (cos (-M_PI / 6.0));
  GEN_DBL_TEST (cos (M_PI * 2.0 / 3.0));
  GEN_DBL_TEST (cos (-M_PI * 2.0 / 3.0));
  GEN_DBL_TEST (cos (M_PI * 5.0 / 6.0));
  GEN_DBL_TEST (cos (-M_PI * 5.0 / 6.0));
  GEN_DBL_TEST (cos (6.9e-18));
  GEN_DBL_TEST (cos (-6.9e-18));
  GEN_DBL_TEST (cos (7.0e-18));
  GEN_DBL_TEST (cos (-7.0e-18));
  GEN_DBL_TEST (cos (7.4e-9));
  GEN_DBL_TEST (cos (-7.4e-9));
  GEN_DBL_TEST (cos (7.5e-9));
  GEN_DBL_TEST (cos (-7.5e-9));
  GEN_DBL_TEST (cos (0.2));
  GEN_DBL_TEST (cos (-0.2));
  GEN_DBL_TEST (cos (0.4));
  GEN_DBL_TEST (cos (-0.4));
  GEN_DBL_TEST (cos (0.7));
  GEN_DBL_TEST (cos (-0.7));
  GEN_DBL_TEST (cos (0.8));
  GEN_DBL_TEST (cos (-0.8));
  GEN_DBL_TEST (cos (3.0));
  GEN_DBL_TEST (cos (-3.0));
  GEN_DBL_TEST (cos (4.0));
  GEN_DBL_TEST (cos (-4.0));
  GEN_DBL_TEST (cos (6.0));
  GEN_DBL_TEST (cos (-6.0));
  GEN_DBL_TEST (cos (7.0));
  GEN_DBL_TEST (cos (-7.0));

  /* tan tests */
  GEN_DBL_TEST (tan (0.0));
  GEN_DBL_TEST (tan (-0.0));
  GEN_DBL_TEST (tan (1.0));
  GEN_DBL_TEST (tan (-1.0));
  GEN_DBL_TEST (tan (INFINITY));
  GEN_DBL_TEST (tan (-INFINITY));
  GEN_DBL_TEST (tan (NAN));
  GEN_DBL_TEST (tan (M_PI));
  GEN_DBL_TEST (tan (-M_PI));
  GEN_DBL_TEST (tan (2.0 * M_PI));
  GEN_DBL_TEST (tan (-2.0 * M_PI));
  GEN_DBL_TEST (tan (M_PI / 2.0));
  GEN_DBL_TEST (tan (-M_PI / 2.0));
  GEN_DBL_TEST (tan (M_PI / 3.0));
  GEN_DBL_TEST (tan (-M_PI / 3.0));
  GEN_DBL_TEST (tan (M_PI / 4.0));
  GEN_DBL_TEST (tan (-M_PI / 4.0));
  GEN_DBL_TEST (tan (M_PI / 6.0));
  GEN_DBL_TEST (tan (-M_PI / 6.0));
  GEN_DBL_TEST (tan (M_PI * 2.0 / 3.0));
  GEN_DBL_TEST (tan (-M_PI * 2.0 / 3.0));
  GEN_DBL_TEST (tan (M_PI * 5.0 / 6.0));
  GEN_DBL_TEST (tan (-M_PI * 5.0 / 6.0));
  GEN_DBL_TEST (tan (3.7e-9));
  GEN_DBL_TEST (tan (-3.7e-9));
  GEN_DBL_TEST (tan (3.8e-9));
  GEN_DBL_TEST (tan (-3.8e-9));
  GEN_DBL_TEST (tan (0.6));
  GEN_DBL_TEST (tan (-0.6));
  GEN_DBL_TEST (tan (0.7));
  GEN_DBL_TEST (tan (-0.7));
  GEN_DBL_TEST (tan (3.0));
  GEN_DBL_TEST (tan (-3.0));
  GEN_DBL_TEST (tan (4.0));
  GEN_DBL_TEST (tan (-4.0));
  GEN_DBL_TEST (tan (6.0));
  GEN_DBL_TEST (tan (-6.0));
  GEN_DBL_TEST (tan (7.0));
  GEN_DBL_TEST (tan (-7.0));

    /* cosh tests */
  GEN_DBL_TEST (cosh (0.0));
  GEN_DBL_TEST (cosh (-0.0));
  GEN_DBL_TEST (cosh (1.0));
  GEN_DBL_TEST (cosh (-1.0));
  GEN_DBL_TEST (cosh (INFINITY));
  GEN_DBL_TEST (cosh (-INFINITY));
  GEN_DBL_TEST (cosh (NAN));
  GEN_DBL_TEST (cosh (M_PI));
  GEN_DBL_TEST (cosh (-M_PI));
  GEN_DBL_TEST (cosh (2.0 * M_PI));
  GEN_DBL_TEST (cosh (-2.0 * M_PI));
  GEN_DBL_TEST (cosh (M_PI / 2.0));
  GEN_DBL_TEST (cosh (-M_PI / 2.0));
  GEN_DBL_TEST (cosh (M_PI / 3.0));
  GEN_DBL_TEST (cosh (-M_PI / 3.0));
  GEN_DBL_TEST (cosh (M_PI / 4.0));
  GEN_DBL_TEST (cosh (-M_PI / 4.0));
  GEN_DBL_TEST (cosh (M_PI / 6.0));
  GEN_DBL_TEST (cosh (-M_PI / 6.0));
  GEN_DBL_TEST (cosh (M_PI * 2.0 / 3.0));
  GEN_DBL_TEST (cosh (-M_PI * 2.0 / 3.0));
  GEN_DBL_TEST (cosh (M_PI * 5.0 / 6.0));
  GEN_DBL_TEST (cosh (-M_PI * 5.0 / 6.0));
  GEN_DBL_TEST (cosh (6.9e-18));
  GEN_DBL_TEST (cosh (-6.9e-18));
  GEN_DBL_TEST (cosh (7.0e-18));
  GEN_DBL_TEST (cosh (-7.0e-18));
  GEN_DBL_TEST (cosh (7.4e-9));
  GEN_DBL_TEST (cosh (-7.4e-9));
  GEN_DBL_TEST (cosh (7.5e-9));
  GEN_DBL_TEST (cosh (-7.5e-9));
  GEN_DBL_TEST (cosh (0.2));
  GEN_DBL_TEST (cosh (-0.2));
  GEN_DBL_TEST (cosh (0.4));
  GEN_DBL_TEST (cosh (-0.4));
  GEN_DBL_TEST (cosh (0.7));
  GEN_DBL_TEST (cosh (-0.7));
  GEN_DBL_TEST (cosh (0.8));
  GEN_DBL_TEST (cosh (-0.8));
  GEN_DBL_TEST (cosh (3.0));
  GEN_DBL_TEST (cosh (-3.0));
  GEN_DBL_TEST (cosh (4.0));
  GEN_DBL_TEST (cosh (-4.0));
  GEN_DBL_TEST (cosh (6.0));
  GEN_DBL_TEST (cosh (-6.0));
  GEN_DBL_TEST (cosh (7.0));
  GEN_DBL_TEST (cosh (-7.0));

  /* sinh tests */
  GEN_DBL_TEST (sinh (0.0));
  GEN_DBL_TEST (sinh (-0.0));
  GEN_DBL_TEST (sinh (1.0));
  GEN_DBL_TEST (sinh (-1.0));
  GEN_DBL_TEST (sinh (INFINITY));
  GEN_DBL_TEST (sinh (-INFINITY));
  GEN_DBL_TEST (sinh (NAN));
  GEN_DBL_TEST (sinh (M_PI));
  GEN_DBL_TEST (sinh (-M_PI));
  GEN_DBL_TEST (sinh (2.0 * M_PI));
  GEN_DBL_TEST (sinh (-2.0 * M_PI));
  GEN_DBL_TEST (sinh (M_PI / 2.0));
  GEN_DBL_TEST (sinh (-M_PI / 2.0));
  GEN_DBL_TEST (sinh (M_PI / 3.0));
  GEN_DBL_TEST (sinh (-M_PI / 3.0));
  GEN_DBL_TEST (sinh (M_PI / 4.0));
  GEN_DBL_TEST (sinh (-M_PI / 4.0));
  GEN_DBL_TEST (sinh (M_PI / 6.0));
  GEN_DBL_TEST (sinh (-M_PI / 6.0));
  GEN_DBL_TEST (sinh (M_PI * 2.0 / 3.0));
  GEN_DBL_TEST (sinh (-M_PI * 2.0 / 3.0));
  GEN_DBL_TEST (sinh (M_PI * 5.0 / 6.0));
  GEN_DBL_TEST (sinh (-M_PI * 5.0 / 6.0));
  GEN_DBL_TEST (sinh (6.9e-18));
  GEN_DBL_TEST (sinh (-6.9e-18));
  GEN_DBL_TEST (sinh (7.0e-18));
  GEN_DBL_TEST (sinh (-7.0e-18));
  GEN_DBL_TEST (sinh (7.4e-9));
  GEN_DBL_TEST (sinh (-7.4e-9));
  GEN_DBL_TEST (sinh (7.5e-9));
  GEN_DBL_TEST (sinh (-7.5e-9));
  GEN_DBL_TEST (sinh (0.2));
  GEN_DBL_TEST (sinh (-0.2));
  GEN_DBL_TEST (sinh (0.4));
  GEN_DBL_TEST (sinh (-0.4));
  GEN_DBL_TEST (sinh (0.7));
  GEN_DBL_TEST (sinh (-0.7));
  GEN_DBL_TEST (sinh (0.8));
  GEN_DBL_TEST (sinh (-0.8));
  GEN_DBL_TEST (sinh (3.0));
  GEN_DBL_TEST (sinh (-3.0));
  GEN_DBL_TEST (sinh (4.0));
  GEN_DBL_TEST (sinh (-4.0));
  GEN_DBL_TEST (sinh (6.0));
  GEN_DBL_TEST (sinh (-6.0));
  GEN_DBL_TEST (sinh (7.0));
  GEN_DBL_TEST (sinh (-7.0));

  /* tanh tests */
  GEN_DBL_TEST (tanh (0.0));
  GEN_DBL_TEST (tanh (-0.0));
  GEN_DBL_TEST (tanh (1.0));
  GEN_DBL_TEST (tanh (-1.0));
  GEN_DBL_TEST (tanh (INFINITY));
  GEN_DBL_TEST (tanh (-INFINITY));
  GEN_DBL_TEST (tanh (NAN));
  GEN_DBL_TEST (tanh (M_PI));
  GEN_DBL_TEST (tanh (-M_PI));
  GEN_DBL_TEST (tanh (2.0 * M_PI));
  GEN_DBL_TEST (tanh (-2.0 * M_PI));
  GEN_DBL_TEST (tanh (M_PI / 2.0));
  GEN_DBL_TEST (tanh (-M_PI / 2.0));
  GEN_DBL_TEST (tanh (M_PI / 3.0));
  GEN_DBL_TEST (tanh (-M_PI / 3.0));
  GEN_DBL_TEST (tanh (M_PI / 4.0));
  GEN_DBL_TEST (tanh (-M_PI / 4.0));
  GEN_DBL_TEST (tanh (M_PI / 6.0));
  GEN_DBL_TEST (tanh (-M_PI / 6.0));
  GEN_DBL_TEST (tanh (M_PI * 2.0 / 3.0));
  GEN_DBL_TEST (tanh (-M_PI * 2.0 / 3.0));
  GEN_DBL_TEST (tanh (M_PI * 5.0 / 6.0));
  GEN_DBL_TEST (tanh (-M_PI * 5.0 / 6.0));
  GEN_DBL_TEST (tanh (6.9e-18));
  GEN_DBL_TEST (tanh (-6.9e-18));
  GEN_DBL_TEST (tanh (7.0e-18));
  GEN_DBL_TEST (tanh (-7.0e-18));
  GEN_DBL_TEST (tanh (7.4e-9));
  GEN_DBL_TEST (tanh (-7.4e-9));
  GEN_DBL_TEST (tanh (7.5e-9));
  GEN_DBL_TEST (tanh (-7.5e-9));
  GEN_DBL_TEST (tanh (0.2));
  GEN_DBL_TEST (tanh (-0.2));
  GEN_DBL_TEST (tanh (0.4));
  GEN_DBL_TEST (tanh (-0.4));
  GEN_DBL_TEST (tanh (0.7));
  GEN_DBL_TEST (tanh (-0.7));
  GEN_DBL_TEST (tanh (0.8));
  GEN_DBL_TEST (tanh (-0.8));
  GEN_DBL_TEST (tanh (3.0));
  GEN_DBL_TEST (tanh (-3.0));
  GEN_DBL_TEST (tanh (4.0));
  GEN_DBL_TEST (tanh (-4.0));
  GEN_DBL_TEST (tanh (6.0));
  GEN_DBL_TEST (tanh (-6.0));
  GEN_DBL_TEST (tanh (7.0));
  GEN_DBL_TEST (tanh (-7.0));
} /* main */
