/***************************************************************************
 *                                                                         *
 *                                                                         *
 *                                                                         *
 *   Parallel IQPNNI - Important Quartet Puzzle with NNI                   *
 *                                                                         *
 *   Copyright (C) 2005 by Le Sy Vinh, Bui Quang Minh, Arndt von Haeseler  *
 *   Copyright (C) 2003-2004 by Le Sy Vinh, Arndt von Haeseler             *
 *   {vinh,minh}@cs.uni-duesseldorf.de                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "codonny98.h"
#include <assert.h>
#include "opturtree.h"
#include "model.h"

const double MIN_CLASS_PROPOTION = 0.0001;
const double MAX_CLASS_PROPOTION = 0.9999;

/** constructor
	@param classes number of classes for nonsynonymous/synonymous ratio 
*/
CodonNY98::CodonNY98(int classes, NSSITE_TYPE nstype, double *ratio_val, double *prob_val)
		: CodonYN98() {
	int classNo;
	nsSyClasses = classes;
	ns_type = nstype;
	/*
	eigen_coeff_vec = new DCube20[nsSyClasses];
	eigen_coeff_derv1_vec = new DCube20[nsSyClasses];
	eigen_coeff_derv2_vec = new DCube20[nsSyClasses];
	*/
	double sum = 0.0;
	for( classNo = 0; classNo < nsSyClasses; classNo ++ ) {
		//nsSyRatioType[classNo] = USER_DEFINED;
		nsSyRatioType[classNo] = ESTIMATE;
		nsSyProbType[classNo] = ESTIMATE;
		//nsSyProbType[classNo] = USER_DEFINED;
		if (classNo < nsSyClasses - 1)
			if (prob_val[0] >= 0)
				nsSyProbVec[classNo] = prob_val[classNo];
			else
				nsSyProbVec[classNo] = 1.0 / nsSyClasses;
		nsSyRatioVec[classNo] = ratio_val[classNo]; //+ 0.1;
		if (classNo < nsSyClasses - 1)
			sum += nsSyProbVec[classNo];
	}
	nsSyProbVec[nsSyClasses - 1 ] = 1.0 - sum;
	nsSyProbType[nsSyClasses - 1] = USER_DEFINED;
	if (nsSyClasses == 1) {
		nsSyRatioVec[0] = 0.1;
	}

	switch (nstype) {
	case NEUTRAL:
		nsSyRatioVec[0] = 0.0; nsSyRatioType[0] = USER_DEFINED;
		nsSyRatioVec[1] = 1.0; nsSyRatioType[1] = USER_DEFINED;
		break;
	case NEARLY_NEUTRAL:
		nsSyRatioVec[1] = 1.0; nsSyRatioType[1] = USER_DEFINED;
		break;
	case SELECTION:
		nsSyRatioVec[0] = 0.0; nsSyRatioType[0] = USER_DEFINED;
		nsSyRatioVec[1] = 1.0; nsSyRatioType[1] = USER_DEFINED;
		break;
	case NEARLY_SELECTION:
		nsSyRatioVec[1] = 1.0; nsSyRatioType[1] = USER_DEFINED;
		break;
	default: break;
	}

	cout << "Init p: ";
	for( classNo = 0; classNo < nsSyClasses; classNo ++ ) {
		cout << nsSyProbVec[classNo] << " ";
	}
	cout << endl << "Init w: ";
	for( classNo = 0; classNo < nsSyClasses; classNo ++ ) {
		cout << nsSyRatioVec[classNo] << " ";
	}
	cout << endl;
}

/**
	all things are inited here
*/
void CodonNY98::init () {
	memset(unitRateMat_, 0, sizeof(unitRateMat_));
	for (int classNo = 0; classNo < nsSyClasses; classNo ++)
		reInit(classNo);
	calcTotalNumSubst();
}

/*
		calc total number of substitutions for 1 site class
*/
double CodonNY98::calcTotalSubst() {

	double sum, temp;
	int i, j;
	for (i = 0, sum = 0.0; i < nState_; i++) {
		for (j = 0, temp = 0.0; j < nState_; j++)
			temp += unitRateMat_[i][j] * stateFrqArr_[j];
		sum += temp*stateFrqArr_[i]; /* exp. rate */
	}

	return sum;
}


/**
	Reinitialize
*/
void CodonNY98::reInit () {
	if (isActPam_ == NS_SY_RATIO) {

		//cout << "Ns/Sy = " << nsSyRatioVec[actClassNo];

		reInit(actClassNo);

		/*
		for (int cl = 0; cl < nsSyClasses; cl++) {
			nsSyRatioVec[cl] = nsSyRatioVec[actClassNo];
			reInit(cl);
		}
		*/

	} else if (isActPam_ == TS_TV_RATIO) {
		//cout << "Ts/Tv = " << tsTvRatio_;
		for (int classNo = 0; classNo < nsSyClasses; classNo ++)
			reInit(classNo);
	} { // prob vec
		double sum = 0.0;
		for (int i = 0; i < nsSyClasses-1; i++)
			sum += nsSyProbVec[i];
		nsSyProbVec[nsSyClasses - 1] = 1.0 - sum;
	}
	calcTotalNumSubst();
}

/*
	calc the total number of substitutions for all site classes
*/
void CodonNY98::calcTotalNumSubst() {

	total_num_subst = 0.0;
	for (int cNo = 0; cNo < nsSyClasses; cNo++)
		total_num_subst += total_subst[cNo] * nsSyProbVec[cNo];
	total_num_subst = total_num_subst * SCALE_RATE_CODON;
}

/**
	Reinitialize
*/
void CodonNY98::reInit (int classNo) {
	nsSyRatio = nsSyRatioVec[classNo];
	//cout << "class " << classNo << ": nsSy = " << nsSyRatio << endl;
	calcRateParameters ();
	total_subst[classNo] = calcTotalSubst();
	tranprobmat(nState_);
	for (int i = 0; i < nState_; i++) {
		eigValArr_vec[classNo][i] = eigValArr_[i];
		for (int j = 0; j < nState_; j++)
			for (int k = 0; k < nState_; k++) {
				eigen_coeff_vec[classNo][i][j][k] = eigen_coeff[i][j][k];
				eigen_coeff_derv1_vec[classNo][i][j][k] = eigen_coeff_derv1[i][j][k];
				eigen_coeff_derv2_vec[classNo][i][j][k] = eigen_coeff_derv2[i][j][k];
			}
	}
}

/**
	check of range of parameters
*/

bool CodonNY98::checkRange(double x[]) {

	if (nsSyClasses <= 2)
		return true;

	int i = 0, count = 1, prob_count = 0;
	if (mymodel.getTsTvRatioType() == ESTIMATE && !pamFirstStage()) {
		count++;
	}

	for (i = 0; i < nsSyClasses; i++)
		if (nsSyRatioType[i] == ESTIMATE) {
			count++;
		}

	double sum = 0.0;
	double max_propotion = MAX_CLASS_PROPOTION;

	for (i = 0; i < nsSyClasses-1; i++)
		if (nsSyProbType[i] == ESTIMATE) {
			if (x[count+prob_count] >= MIN_CLASS_PROPOTION + EPS_BOUND)
				sum += x[count + prob_count];
			else
				max_propotion -= x[count+prob_count];
			prob_count++;
		} else
			max_propotion -= nsSyProbVec[i];
	if (sum > max_propotion) {
		int i;
		/*
		cout.precision(6);
		for (i = 0; i < nsSyClasses - 1; i++) {
			cout << x[count + i] << " ";
		}
		cout << " sum = " << sum << endl;
		*/

		// check the propotion sum to less than 1

		if (prob_count <= 2) {
			sum = (sum - max_propotion) / prob_count;
			prob_count = 0;
			for (i = 0; i < nsSyClasses-1; i++)
				if (nsSyProbType[i] == ESTIMATE) {
					x[count+prob_count] -= sum;
					prob_count++;

				}
		} else {

			prob_count = 0;
			for (i = 0; i < nsSyClasses-1; i++)
				if (nsSyProbType[i] == ESTIMATE) {
					x[count+prob_count] *= (max_propotion/sum);
					prob_count++;

				}
			//cout << endl;
		}
		return false;
	}

	return true;
}

/**
	check of range of parameters
*/

bool CodonNY98::checkProportionSum(double x[]) {

	if (nsSyClasses <= 2)
		return true;

	double sum = 0.0;
	int i = 0, count = 1, prob_count = 0;
	if (mymodel.getTsTvRatioType() == ESTIMATE && !pamFirstStage()) {
		count++;
	}

	for (i = 0; i < nsSyClasses; i++)
		if (nsSyRatioType[i] == ESTIMATE) {
			count++;
		}

	for (i = 0; i < nsSyClasses-1; i++)
		if (nsSyProbType[i] == ESTIMATE) {
			sum += x[count + prob_count];
			prob_count++;
		} else
			sum += nsSyProbVec[i];

	if (sum > 1.0) {
		return false;
	}

	return true;
}

/**
	return TRUE if values is at the boundary
*/
bool CodonNY98::paramAtBoundary() {
	int i;

	if (mymodel.getTsTvRatioType() == ESTIMATE && !pamFirstStage()) 
		if (tsTvRatio_ < MIN_TS_TV_RATIO + EPS_BOUND || tsTvRatio_ > MAX_TS_TV_RATIO - EPS_BOUND)
			return true;
	
	for (i = 0; i < nsSyClasses; i++)
		if (nsSyRatioType[i] == ESTIMATE) 
			if (nsSyRatioVec[i] < MIN_NS_SY_RATIO + EPS_BOUND || nsSyRatioVec[i] > MAX_NS_SY_RATIO - EPS_BOUND)
				return true;		

	for (i = 0; i < nsSyClasses; i++)
		//if (nsSyProbType[i] == ESTIMATE) 
			if (nsSyProbVec[i] < MIN_CLASS_PROPOTION + EPS_BOUND || nsSyProbVec[i] > MAX_CLASS_PROPOTION - EPS_BOUND)
				return true;
	return false;
		
}

/**
	return TRUE if values is at the boundary
*/
bool CodonNY98::atBoundary(double x[]) {

	if (nsSyClasses <= 2)
		return false;

	double sum = 0.0;
	int i = 0, count = 1, prob_count = 0;
	if (mymodel.getTsTvRatioType() == ESTIMATE && !pamFirstStage()) {
		count++;
	}

	for (i = 0; i < nsSyClasses; i++)
		if (nsSyRatioType[i] == ESTIMATE) {
			count++;
		}

	for (i = 0; i < nsSyClasses-1; i++)
		if (nsSyProbType[i] == ESTIMATE) {
			sum += x[count + prob_count];
			prob_count++;
		} else
			sum += nsSyProbVec[i];

	if (sum > MAX_CLASS_PROPOTION - EPS_BOUND) {
		return true;
	}

	return false;
}


/**
	return TRUE if at the first stage of the parameter estimation
*/
bool CodonNY98::pamFirstStage() {
	if (!mymodel.initialized_tsTv)
		return false;
	int nstep = 0;
	switch (ns_type) {
	case ONE_RATIO:
		nstep = 1;
		break;
	case NEUTRAL:
		nstep = 1;
		break;
	case NEARLY_NEUTRAL:
		nstep = 1;
		break;
	case SELECTION:
		nstep = 1;
		break;
	case NEARLY_SELECTION:
		nstep = 2;
		break;
	case DISCRETE:
		nstep = nsSyClasses-1;
		if (nstep > 3) nstep = 3;
		break;
	default:
		nstep = 2;
		break;
	}

	return mymodel.nPamStep <= 1;
}


/**
	return the number of dimensions
*/
int CodonNY98::getNDim() {
	int ndim = 0;
	for (int i = 0; i < nsSyClasses; i++) {
		if (nsSyRatioType[i] == ESTIMATE) ndim++;
		if (i < nsSyClasses-1 && nsSyProbType[i] == ESTIMATE) ndim++;
	}

	return (mymodel.getTsTvRatioType() == ESTIMATE && !pamFirstStage()) ? ndim + 1 : ndim;
	//return ndim;
}

/**
	pack the data to the vector x
*/
void CodonNY98::packData( double x[], double lower[], double upper[], bool bound_check[]) {
	int count = 1, i;
	if (mymodel.getTsTvRatioType() == ESTIMATE && !pamFirstStage()) {
		x[count] = tsTvRatio_;
		lower[count] = MIN_TS_TV_RATIO;
		upper[count] = MAX_TS_TV_RATIO;
		bound_check[count] = true;
		count++;
	}
	for (i = 0; i < nsSyClasses; i++)
		if (nsSyRatioType[i] == ESTIMATE) {
			x[count] = nsSyRatioVec[i];
			lower[count] = MIN_NS_SY_RATIO;
			upper[count] = MAX_NS_SY_RATIO;
			bound_check[count] = true;
			count++;
		}

	for (i = 0; i < nsSyClasses-1; i++)
		if (nsSyProbType[i] == ESTIMATE) {
			x[count] = nsSyProbVec[i];
			lower[count] = MIN_CLASS_PROPOTION;
			upper[count] = MAX_CLASS_PROPOTION;
			bound_check[count] = true;
			count++;
		}
}

/**
	unpack data from vector x
*/
void CodonNY98::unpackData(double x[]) {
	int count = 1, i;
	double sum = 0.0;
	if (mymodel.getTsTvRatioType() == ESTIMATE && !pamFirstStage()) {
		tsTvRatio_ = x[count];
		count++;
	}

	for (i = 0; i < nsSyClasses; i++)
		if (nsSyRatioType[i] == ESTIMATE) {
			nsSyRatioVec[i] = x[count];
			count++;
		}

	for (i = 0; i < nsSyClasses-1; i++) {
		if (nsSyProbType[i] == ESTIMATE) {
			nsSyProbVec[i] = x[count];
			count++;
		}

		sum += nsSyProbVec[i];
	}

	nsSyProbVec[nsSyClasses-1] = 1.0 - sum;

}

/**
	check direction of the search and fix it if neccesary
*/
/*
void CodonNY98::checkDirection(double x[], double dir[]) {
	// check the propotion
	
	if (nsSyClasses <= 2)
		return;
 
	double sum = 0.0;
	int i, count = 1, prob_count = 0;
	if (mymodel.getTsTvRatioType() == ESTIMATE && !pamFirstStage())
		count++;
 
	for (i = 0; i < nsSyClasses; i++)
		if (nsSyRatioType[i] == ESTIMATE)
			count++;
 
	for (i = 0; i < nsSyClasses-1; i++)
		if (nsSyProbType[i] == ESTIMATE) {
			sum += x[count + prob_count] + dir[count + prob_count];
			prob_count ++;
		} else
			sum += nsSyProbVec[i];
 
	// check the propotion sum to less than 1
	if (sum > MAX_SUM_PROPOTION) {
		sum = (sum - MAX_SUM_PROPOTION) / prob_count;
		for (i = 0; i < nsSyClasses-1; i++)
			if (nsSyProbType[i] == ESTIMATE) {
				cout << x[count] + dir[count] << " ";
				dir[count] -= sum;
				count++;
			}
		//cout << endl;
	}
	
}
*/

double CodonNY98::targetFunk(double values[]) {


	if (!checkProportionSum(values)) {
		int ndim = getNDim();
		for (int i = 1; i <= ndim; i++)
			cout << values[i] << " ";
		cout << endl;
		return INFINITIVE;
	}


	int count = 1;
	bool changeall = false;
	if (mymodel.getTsTvRatioType() == ESTIMATE && !pamFirstStage()) {
		if (values[count] != tsTvRatio_)
			changeall = true;
		tsTvRatio_ = values[count];
		count++;
	}

	if (!changeall)
		for (int cNo = 0; cNo < nsSyClasses; cNo++) {
			if (nsSyRatioType[cNo] == ESTIMATE) {
				if (nsSyRatioVec[cNo] != values[count]) {
					nsSyRatioVec[cNo] = values[count];
					//cout << "  " << nsSyRatioVec[cNo];
					reInit(cNo);
				}
				count++;
			}
		}

	unpackData(values);
	if (changeall)
		for (int classNo = 0; classNo < nsSyClasses; classNo++) {
			reInit(classNo);
		}

	calcTotalNumSubst();
	opt_urtree.cmpLiNd ();
	double logLi_ = opt_urtree.getLogLi ();
	//cout << "  logli = " << logLi_ << endl;
	//cout << "logli = " << logLi_ << endl;
	return -logLi_;
}


double CodonNY98::cmpNegLogLi (double value) {
	if (isActPam_ == TS_TV_RATIO)
		tsTvRatio_ = value;
	else if (isActPam_ == NS_SY_RATIO) {
		//nsSyRatio = value;
		nsSyRatioVec[actClassNo] = value;
	} else {
		//double change = nsSyProbVec[actClassNo] - value;
		nsSyProbVec[actClassNo] = value;
		//nsSyProbVec[nsSyClasses - 1] += change;
	}

	reInit ();
	if (isActPam_ == TS_TV_RATIO || isActPam_ == NS_SY_RATIO) {
		opt_urtree.cmpLiNd ();
	} else {
		opt_urtree.cmpLiNd ();
	}
	//opt_urtree.opt (MAX_IT_UR_BR_PAM_OPT);
	double logLi_ = opt_urtree.getLogLi ();
	cout.precision(10);
	//cout << "  logli = " << logLi_ << endl;
	//cout << "logli = " << logLi_ << endl;
	return -logLi_;
}


bool CodonNY98::optPam () {
	double fx_, error_;
	double oriTsTvRatio_ = tsTvRatio_;
	double oriNsSyRatio[MAX_NUM_CLASS];
	double oriNsSyProb[MAX_NUM_CLASS];
	int classNo;
	for (classNo = 0; classNo < nsSyClasses; classNo++) {
		oriNsSyRatio[classNo] = nsSyRatioVec[classNo];
		oriNsSyProb[classNo] = nsSyProbVec[classNo];
	}

	//  double logLi_ = opt_urtree.getLogLi ();
	if (isMasterProc())
		std::cout <<"Optimizing Nielsen & Yang 1998 codon-based model parameters ..." << endl;
	Brent::turnOnOptedPam ();
	//double logLi_ = opt_urtree.cmpLogLi ();
	//  std::cout <<"Log likelihood: " << logLi_ << endl;

	//mymodel.setTsTvRatioType(USER_DEFINED);

	if (mymodel.pam_brent == 2 || getNDim() <= 1) {

		if (mymodel.getTsTvRatioType() == ESTIMATE && !pamFirstStage()) {
			isActPam_ = TS_TV_RATIO;
			if (isMasterProc())
				std::cout <<"   Optimizing transition/transversion ratio..." << endl;
			tsTvRatio_ = Brent::optOneDim (MIN_TS_TV_RATIO, tsTvRatio_, MAX_TS_TV_RATIO,
			                               EPS_MODEL_PAM_ERROR, &fx_, &error_);
			//tsTvRatio_ = 100;
			//tsTvRatio_ = 0.72586;
			//tsTvRatio_ = 0.92697;
			//tsTvRatio_ = 3.37068;
			reInit ();
		}



		if (isMasterProc())
			cout <<"   Optimizing nonsynonymous/synonymous ratio..." << endl;
		for (actClassNo = 0; actClassNo < nsSyClasses; actClassNo ++ ) {
			if (nsSyRatioType[actClassNo] == ESTIMATE) {
				isActPam_ = NS_SY_RATIO;
				double min_ratio = MIN_NS_SY_RATIO;
				double max_ratio = 100.0;
				if (actClassNo > 0)
					min_ratio = nsSyRatioVec[actClassNo-1];
				if (actClassNo < nsSyClasses - 1)
					max_ratio = nsSyRatioVec[actClassNo+1];
				nsSyRatioVec[actClassNo] = Brent::optOneDim (MIN_NS_SY_RATIO, nsSyRatioVec[actClassNo],
				                           MAX_NS_SY_RATIO, EPS_MODEL_PAM_ERROR, &fx_, &error_);
				//cout << nsSyRatioVec[actClassNo] << endl;
				reInit ();
			}
		}


		if (isMasterProc())
			cout <<"   Optimizing propotion of classes..." << endl;
		for (actClassNo = 0; actClassNo < nsSyClasses-1; actClassNo ++ )
			if (nsSyProbType[actClassNo] == ESTIMATE ) {
				isActPam_ = NS_SY_RATIO_PROB;
				double max_propotion = MAX_CLASS_PROPOTION;
				for (int i = 0; i < nsSyClasses-1; i++)
					if (i != actClassNo)
						max_propotion -= nsSyProbVec[i];
				oriNsSyProb[actClassNo] = nsSyProbVec[actClassNo];

				nsSyProbVec[actClassNo] = Brent::optOneDim (MIN_CLASS_PROPOTION, nsSyProbVec[actClassNo],
				                          max_propotion, EPS_MODEL_PAM_ERROR, &fx_, &error_);
				reInit ();

				//      logLi_ = opt_urtree.cmpLogLi ();
				//     std::cout <<"Log likelihood: " << logLi_ << endl;
				//------------------------------/------------------------------/
			}

	} else {

		int ndim = getNDim();
		double *variables = new double[ndim+1];
		double *upper_bound = new double[ndim+1];
		double *lower_bound = new double[ndim+1];
		bool *bound_check = new bool[ndim+1];

		if (mymodel.pam_brent == 1 && mymodel.getPamStep() == 1 && getNDim() >= 5) {
			if (isMasterProc())
				cout <<"   Optimizing nonsynonymous/synonymous ratio by Simulated Annealing..." << endl;
			// discrete model M3
			packData(variables, lower_bound, upper_bound, bound_check);
			simulatedAnnealing(ndim, variables, lower_bound, upper_bound);
			unpackData(variables);
			for (classNo = 0; classNo < nsSyClasses; classNo++) {
				reInit(classNo);
			}
			calcTotalNumSubst();
		}

		cout <<"   Optimizing nonsynonymous/synonymous ratio by BFGS..." << endl;
		// by BFGS algorithm
		//int ndim = getNDim();
		packData(variables, lower_bound, upper_bound, bound_check);
		//optMultiDim(variables, ndim, EPS_MODEL_PAM_ERROR);
		optMultiDim(variables, ndim, lower_bound, upper_bound, bound_check, EPS_MODEL_FUNC_ERROR);
		unpackData(variables);
		for (classNo = 0; classNo < nsSyClasses; classNo++) {
			reInit(classNo);
		}
		calcTotalNumSubst();

		delete bound_check;
		delete lower_bound;
		delete upper_bound;
		delete variables;

	}


	opt_urtree.cmpLiNd ();
	//logLi_ = opt_urtree.getLogLi ();

	//std::cout.precision (10);
	//std::cout << "Log likelihood: " << logLi_ << endl;

	if (isMasterProc()) {

		std::cout.precision (5);
		std::cout << "Transition/transversion ratio  =  " << tsTvRatio_/2.0 << endl;
		std::cout << "Nonsynonymous/synonymous ratio of  " << endl;
		for (int classNo = 0; classNo < nsSyClasses; classNo ++) {
			//cout << "class " << classNo+1 << " = " << nsSyRatioVec[classNo] << ", propotion = " <<
			//nsSyProbVec[classNo] << endl;
			cout << "propotion = ";
			printf("%7.5f", nsSyProbVec[classNo]);
			cout  << "  class " << classNo+1 << " = " << nsSyRatioVec[classNo] << endl;
		}
		//cout << endl;
	}

	double sum = 0.0;
	for (int i = 0; i < nsSyClasses; i++)
		sum += nsSyProbVec[i];
	if (fabs(sum-1.0) > ZERO) {
		cout.precision(8);
		cout << "hey: prob sum = " << sum << " is not 1.0!" << endl;
	}

	Brent::turnOffOptedPam ();
	if (fabs (tsTvRatio_ - oriTsTvRatio_) > EPS_MODEL_PAM_ERROR)
		return true;
	for (classNo = 0; classNo < nsSyClasses; classNo ++) {
		if (fabs (nsSyRatioVec[classNo] - oriNsSyRatio[classNo]) > EPS_MODEL_PAM_ERROR )
			return true;
		if (fabs(nsSyProbVec[classNo] - oriNsSyProb[classNo]) > EPS_MODEL_PAM_ERROR)
			return true;
	}
	return false;
}

/**
	get the Ns/Sy Ratio of a class
*/
double CodonNY98::getNsSyRatio(int classNo) {
	return nsSyRatioVec[classNo];
}

CodonNY98::~CodonNY98() {
	/*
	delete eigen_coeff_derv2_vec;
	delete eigen_coeff_derv1_vec;
	delete eigen_coeff_vec;
	*/
}






