/***************************************************************************
 *                                                                         *
 *                  (begin: Feb 20 2003)                                   *
 *                                                                         *
 *   Parallel IQPNNI - Important Quartet Puzzle with NNI                   *
 *                                                                         *
 *   Copyright (C) 2005 by Le Sy Vinh, Bui Quang Minh, Arndt von Haeseler  *
 *   Copyright (C) 2003-2004 by Le Sy Vinh, Arndt von Haeseler             *
 *   {vinh,minh}@cs.uni-duesseldorf.de                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#include <iostream>
#include <fstream>

#include "model.h"
#include "ali.h"
#include "rate.h"
#include "outstream.h"
#include "ptnls.h"
#include "model.h"
#include "interface.h"
#include <string>
#include "rannum.h"


#ifdef _OPENMP
#include <omp.h>
#endif

extern int isContinuous;

int program_mode = PROG_ALL;
bool load_check_point = true;
bool major_step_test = false;
bool nni_test = false;
int omp_threads = 0;


char outfile_suffix_new[] = ".iqtree";
char outfile_suffix_old[] = ".iqpnni";
char *outfile_suffix = outfile_suffix_old;

char bionj_suffix[] = ".bionj";
char treefile_suffix[] = ".treefile";
char boottree_suffix[] = ".boottrees";
char contree_suffix[] = ".contree";
char splits_suffix[] = ".splits";
char treels_suffix[] = ".treels";
char dist_suffix[] = ".dist";
char bootsample_suffix[] = ".bootsample";
char checkpoint_suffix[] = ".checkpoint";
char prediction_suffix[] = ".prediction";
char sitelh_suffix[] = ".sitelh";
char patlh_suffix[] = ".patlh";
char siterate_suffix[] = ".rate";
char nni_lh_suffix[] = ".nni_lh";
char nni_tree_suffix[] = ".nni_tree";

// append the required suffix to str
void getOutFileName(SUFFIX_TYPE suf, string &str) {
	str = alignment.out_prefix;
	str += outfile_suffix;
	switch (suf) {
		case SUF_BIONJ: str += bionj_suffix; break;
		case SUF_TREEFILE: str += treefile_suffix; break;
		case SUF_BOOTTREE: str += boottree_suffix; break;
		case SUF_CONTREE: str += contree_suffix; break;
		case SUF_SPLITS: str += splits_suffix; break;
		case SUF_TREELS: str += treels_suffix; break;
		case SUF_DIST: str += dist_suffix; break;
		case SUF_BOOTSAMPLE: str += bootsample_suffix; break;
		case SUF_CHECKPOINT: str += checkpoint_suffix; break;
		case SUF_PREDICTION: str += prediction_suffix; break;
		case SUF_SITELH: str += sitelh_suffix; break;
		case SUF_PATLH: str += patlh_suffix; break;
		case SUF_SITERATE: str += siterate_suffix; break;
		case SUF_NNI_LH: str += nni_lh_suffix; break;
		case SUF_NNI_TREE: str += nni_tree_suffix; break;
		case SUF_MAIN: break;
	}
}


int Interface::cmpnIter () {

	// testing
	//	return 3;
	//end testing
	//if (nSeq_ < 31)
	//	return 20;

	int n = nSeq_ * 2;

	if (n < 200) 
		n = 200;
	
#ifdef PARALLEL
	if (n < 20 * mpi_size) 
		n = 20 * mpi_size;
#endif

	//if (nSeq_ < 51) 
		//return 100;

	//if (nSeq_ < 201)
		//return nSeq_* 2;

		/*
	if (nSeq_ < 151)
		return 100;

	if (nSeq_ < 201)
		return 200;
*/
	//if (nSeq_ < 1001)
		//return 500;

	return n;
}

double Interface::cmpProbSel (PAM_TYPE sourceType) {
	if (sourceType == SIMULATION)
		return 0.3;

	if (nSeq_ < 51)
		return 0.5;

	if (nSeq_ < 100)
		return 0.3;

	if (nSeq_ < 200)
		return 0.2;

	return 0.1;
}

void Interface::getIntNumber (istream &fpin, const char *msg, double lowerBound, double upperBound, int &value) {
	char sValue_[20];
	char *stopPos_;
	do {
		std::cout << msg;
		fpin >> sValue_;
		value = static_cast<int> (strtod (sValue_, &stopPos_));

		std::cout.precision (12);
		if (value < lowerBound || value > upperBound)
			std::cout <<"The entered number must be between " << lowerBound << " and " << upperBound << ". Please, try again!!!" << endl;
	} while (value < lowerBound || value > upperBound);
}

void Interface::getDoubleNumber (istream &fpin, const char *msg, double lowerBound, double upperBound, double &value) {
	char sValue_[20];
	char *stopPos_;
	do {
		std::cout << msg;
		fpin >> sValue_;
		value = strtod (sValue_, &stopPos_);

		std::cout.precision (12);
		if (value < lowerBound || value > upperBound)
			std::cout <<"The entered number must be between " << lowerBound << " and " << upperBound << ". Please, try again!!!" << endl;
	} while (value < lowerBound || value > upperBound);
}

//=====================================================================================
//=====================================================================================
int Interface::getOutGrpSeq (istream &fpin) {
	std::cout << "Enter name of outgroup sequence: ";
	char tmpOutGrpSeqName_[MAX_SEQ_NAME_LEN];
	fpin >> tmpOutGrpSeqName_;

	Vec<char> userOutGrpSeqName_(MAX_SEQ_NAME_LEN);
	int pos_ = 0;
	while (Utl::isControlChar (tmpOutGrpSeqName_[pos_] ) == 1)
		pos_ ++;
	while (tmpOutGrpSeqName_[pos_] != '\0') {
		userOutGrpSeqName_ += tmpOutGrpSeqName_[pos_];
		pos_ ++;
	}

	int userOutGrpSeqNo_ = alignment.getSeqNo (userOutGrpSeqName_);
	if (userOutGrpSeqNo_ != -1)
		return userOutGrpSeqNo_;
	else {
		std::cout <<"Warning!!! The entered outgroup sequence \"" << tmpOutGrpSeqName_ << "\" does not exist!!!" << endl;
		return -1;
	}
}

void initializeParams(CommonParameters &params) {
	params.model           = HKY85; // must be changed later
	params.tsTvRatio      = 4.0;
	params.pyPuRatio      = 1.0;


	params.tsAG = 1.0 / 6.0;
	params.tsCT = 1.0 / 6.0;
	params.tvAC = 1.0 / 6.0;
	params.tvAT = 1.0 / 6.0;
	params.tvCG = 1.0 / 6.0;
	params.tvGT = 0.1; //1.0 / 6.0;

	params.baseA = 0.25;
	params.baseC = 0.25;
	params.baseG = 0.25;
	params.baseT = 0.25;

	params.nIter = -1; // number of iterations, must be changed later
	params.max_nIter = -1;
	params.cur_nIter = -1;
	params.probDel = -1; // must be changed later
	params.nRep = 4;
	params.outGrpSeqNo = 0;

	params.baseFrqType = ESTIMATE;
	params.tsTvRatioType = ESTIMATE;
	params.pyPuRatioType = ESTIMATE;
	params.genPamType = ESTIMATE;
	params.gammaShapeType = ESTIMATE;
	params.prob_invar_type = USER_DEFINED;
	params.rateType = UNIFORM;
	params.nRate = 1;
	params.gammaShape = 1.0;
	params.prob_invar = 0.0;
	//params.invariable_site = false;

	params.codon_model = UNDEF_MODEL;
	params.nsSy_ratio_type = DISCRETE;
	params.nsSy_classes = 3;
	for (int i = 0; i < MAX_NUM_CLASS; i++) {
		params.nsSy_ratio_val[i] = i + 1;	
		params.nsSy_prob_val[i] = -1;
	}
	
	params.stoppingRule = NO;
	params.pam_brent = 0;
	params.ap_sitespec = true;
	params.gen_bootstrap = 0;
	params.cur_bootstrap = 0;
	params.print_pat_logl = false;
	params.print_site_logl = false;
	params.print_splits = false;
	params.build_consensus = false;
	params.bestLogLi = -INFINITIVE;
	params.progTime = 0;
}

void initInputParams(InputParameters &in_pam) {
	in_pam.out_prefix = NULL;
	in_pam.param_file = NULL;
	in_pam.ali_file[0] = 0;
	in_pam.tree_file[0] = 0;
	in_pam.text_menu = true;
	in_pam.random_seed = -1;
	in_pam.nni_lh_record = false;
}

/*=========================================================================================
  Get the parameter by interactive interface
//=========================================================================================*/
int Interface::getParameter (istream &fpin, CommonParameters &params, InputParameters &in_pam) {

	if (isMasterProc())
		std::cout << endl << endl << endl << endl << endl;

	nSeq_ = alignment.getNSeq ();

	string checkPointFileName_;
	getOutFileName(SUF_CHECKPOINT, checkPointFileName_);


	int isPosContinuous_ = 0;
	if (load_check_point && isExistedFile (checkPointFileName_.c_str()) == 1){
		CommonParameters tmp_param;
		initializeParams(tmp_param);
		isPosContinuous_= getPamCheckPoint (checkPointFileName_.c_str(), tmp_param);
		if (isPosContinuous_ != 0) {
			params = tmp_param;
			if (isMasterProc()) 
				cout << "The program was not done from the last run! " << endl <<
					"Load parameters from the checkpoint file..." << endl;
		}
	}

	isContinuous = isPosContinuous_;
	if (isContinuous == 0) // start from scratch
	{
		PAM_TYPE sourceType_;
		sourceType_ = alignment.detectDataSource ();
		params.dataType = alignment.detectDataType ();
		//baseFrqType = ESTIMATE;

		if (params.dataType != NUCLEOTIDE)
			if (params.model == HKY85 || params.model == TN93 || params.model == GTR ||
			        isCodonModel(params.model))
				params.model = UNDEF_MODEL;

		if (isCodonModel( params.model)) {
			params.dataType = CODON;
			/*
			if (params.baseFrqType == ESTIMATE)
				params.baseFrqType = F3x4;
			*/
		}

		if (params.model == UNDEF_MODEL)
			if (params.dataType == NUCLEOTIDE)
				params.model  = HKY85;
			else
				params.model = WAG;

		if (params.probDel < 0 || params.probDel > 1)
			params.probDel = cmpProbSel (sourceType_);
		if (params.nIter < 0)
			params.nIter = cmpnIter();
		if (params.max_nIter < params.nIter)
			params.max_nIter = params.nIter * 10;

	} //end of if

	Vec<char> outGrpSeqName_;
	alignment.getName (params.outGrpSeqNo, outGrpSeqName_);

	char option_='y';

	if (isMasterProc())
		do {
			std::cout <<     "GENERAL OPTIONS" << endl;
			if (isPosContinuous_ == 1)
				if (isContinuous == 1)
					std::cout <<      " b                 Restart from the last stop? Yes" << endl;
				else
					std::cout <<      " b                 Restart from the last stop? No (start a new search)" << endl;
	
			std::cout << " z                   Non-parametric bootstrap? ";
			if (params.gen_bootstrap) 
				std::cout << "Yes, " << params.gen_bootstrap << " samples" << endl;
			else 
				std::cout << "No" << endl;
			std::cout <<         " o                        Display as outgroup? " << outGrpSeqName_ << endl;
			std::cout <<         " n               Minimum number of iterations? " << params.nIter << endl;
			switch (params.stoppingRule) {
			case YES:
				std::cout <<         " s              Stopping rule (if applicable)? " << "Yes" << endl << endl;
				break;
			case YES_MIN_ITER:
				std::cout <<         " s              Stopping rule (if applicable)? " << "Yes" << endl << endl;
				break;
			case YES_MAX_ITER:
				std::cout <<         " s              Stopping rule (if applicable)? " << "Yes, and at most " << params.max_nIter << " iterations" << endl << endl;
				break;
			default:
				std::cout <<         " s                              Stopping rule? " << "No, stop after " << params.nIter << " iterations" << endl << endl;
			}

			std::cout <<     "IQP OPTIONS" << endl;
			std::cout <<         " p         Probability of deleting a sequence? " << params.probDel << endl;
			std::cout <<         " k                     Number representatives? " << params.nRep << endl << endl;

			std::cout << "SUBSTITUTION PROCESS" << endl;
			if (params.dataType == AMINO_ACID) {
				std::cout <<       " d                Type of sequence input data? " << "Amino acids" << endl;
				switch (params.model) {
				case WAG:
					std::cout <<     " m                      Model of substitution? " << "WAG (Whelan-Goldman 2000)" << endl;
					break;
				case JTT:
					std::cout <<     " m                      Model of substitution? " << "JTT (Jones et at. 1992)" << endl;
					break;
				case VT:
					std::cout <<     " m                      Model of substitution? " << "VT (Mueller-Vingron 2000)" << endl;
					break;
				case MtREV24:
					std::cout <<     " m                      Model of substitution? " << "mtREV24 (Adachi-Hasegawa 1996)" << endl;
					break;
				case rtREV:
					std::cout <<     " m                      Model of substitution? " << "rtREV (Dimmic et al. 2001)" << endl;
					break;
				case Blosum62:
					std::cout <<     " m                      Model of substitution? " << "BLOSUM62 (Henikoff-Henikoff 1992)" << endl;
					break;
				case AA_USER:
					std::cout <<     " m                      Model of substitution? " << "From file '" << params.rate_file << "'"<< endl;
					break;
				default:
					std::cout <<     " m                      Model of substitution? " << "Dayhoff (Dayhoff et al. 1978)" << endl;
				} //end of switch

				if (params.baseFrqType == ESTIMATE)
					std::cout <<   " f                    Amino acide frequencies? " << "Estimate from data" << endl;
				else
					std::cout <<   " f                    Amino acide frequencies? " << "Default" << endl;

			} else if (params.dataType == NUCLEOTIDE) {
				std::cout <<       " d                Type of sequence input data? " << "Nucleotides" << endl;
				switch (params.model) {
				case HKY85:
					std::cout <<     " m                      Model of substitution? " << "HKY85 (Hasegawa et al. 1985)" << endl;
					if (params.tsTvRatioType == ESTIMATE)
						std::cout <<     " t                 Ts/Tv ratio (0.5 for JC69)? " << "Estimate from data" << endl;
					else
						std::cout <<     " t                 Ts/Tv ratio (0.5 for JC69)? " << params.tsTvRatio << endl;
					break;

				case TN93:
					std::cout <<     " m                      Model of substitution? " << "TN93 (Tamura-Nei 1993)" << endl;
					if (params.tsTvRatioType == ESTIMATE)
						std::cout <<     " t                 Ts/Tv ratio (0.5 for JC69)? " << "Estimate from data" << endl;
					else
						std::cout <<     " t                 Ts/Tv ratio (0.5 for JC69)? " << params.tsTvRatio << endl;

					if (params.pyPuRatioType == ESTIMATE)
						std::cout <<     " u                    Pyrimidine/Purine ratio? " << "Estimate from data" << endl;
					else
						std::cout <<     " u                    Pyrimidine/Purine ratio? " << params.pyPuRatio << endl;
					break;

				default:
					std::cout <<     " m                      Model of substitution? " << "General time reversible" << endl;
					if (params.genPamType == ESTIMATE)
						std::cout <<    " g   General time reversible model parameters? " << "Estimate from data" << endl;
					else
						std::cout <<    " g   General time reversible model parameters? " << "User defined" << endl;
				}

				if (params.baseFrqType == ESTIMATE)
					std::cout <<       " f                           Base frequencies? " << "Estimate from data" << endl;
				else
					if (params.baseFrqType == EQUAL)
						std::cout <<   " f                           Base frequencies? " << "Equal" << endl;
					else
						std::cout <<   " f                           Base frequencies? " << "User defined" << endl;
			} else { // CODON Model
				std::cout <<       " d                Type of sequence input data? " << "Protein-coding DNA" << endl;
				switch (params.model) {
				case Codon_GY94:
					std::cout <<     " m          Codon-based model of substitution? " << "Goldman & Yang 1994" << endl;
					break;
				case Codon_YN98:
					std::cout <<     " m          Codon-based model of substitution? " << "Yang & Nielsen 1998" << endl;
					break;
				case Codon_NY98:
					std::cout <<     " m          Codon-based model of substitution? " << "Nielsen & Yang 1998" << endl;
					break;
				case Codon_Pedersen98:
					std::cout <<     " m          Codon-based model of substitution? " << "CpG Depression 1998" << endl;
					break;
				case Codon_GTR:
					std::cout <<     " m          Codon-based model of substitution? " << "GTR" << endl;
					break;
				case Codon_PosRate:
					std::cout <<     " m          Codon-based model of substitution? " << "different position rate" << endl;
					break;
				default: break;
				}

				if (params.model == Codon_GY94 || params.model == Codon_YN98 || params.model == Codon_Pedersen98 || 	params.model == Codon_NY98 || params.model == Codon_PosRate) {
					if (params.tsTvRatioType == ESTIMATE)
						std::cout <<     " t                 Ts/Tv ratio (0.5 for MG94)? " << "Estimate from data" << endl;
					else
						std::cout <<     " t                 Ts/Tv ratio (0.5 for MG94)? " << params.tsTvRatio << endl;
				}

				if (params.model == Codon_NY98) {
					cout << " w             Nonsynonymous/synonymous ratio? ";
					switch (params.nsSy_ratio_type) {
					case ONE_RATIO:
						cout << "One Ratio" << endl;
						break;
					case NEUTRAL:
						cout << "Neutral (w1=0,w2=1)" << endl;
						break;
					case NEARLY_NEUTRAL:
						cout << "Nearly Neutral (w1 free,w2=1)" << endl;
						break;
					case SELECTION:
						cout << "Selection (w1=0,w2=1,w3 free)" << endl;
						break;
					case NEARLY_SELECTION:
						cout << "Pos. Selection (w2=1,w1&w3 free)" << endl;
						break;
					case DISCRETE:
						cout << "Discrete (unconstrained)" << endl;
						cout << " g           Nonsynonymous/synonymous classes? " << params.nsSy_classes << endl;
						break;
					}
				}

				std::cout <<   " f                          Codon frequencies? ";
				switch (params.baseFrqType) {
				case EQUAL:
					std::cout << "Equal" << endl;
					break;
				case F1x4:
					cout << "F1x4" << endl;
					break;
				case F3x4:
					cout << "F3x4" << endl;
					break;
				case ESTIMATE:
					cout << "Codon table" << endl;
					break;
				default: break;
				}
			} //end of model


			if (params.model != Codon_NY98) {
				std::cout <<  endl <<   "RATE HETEROGENEITY" << endl;
				if (params.rateType == UNIFORM)
					std::cout <<        " r                Model of rate heterogeneity? " << "Uniform rate" << endl;
				else
					if (params.rateType == SITE_SPECIFIC)
						std::cout <<     " r                Model of rate heterogeneity? " << "Site-specific substitution rates" << endl;
					else {
						std::cout <<    " r                Model of rate heterogeneity? " << "Gamma distributed rates" << endl;
						cout << " i             Proportion of invariable sites? ";
						//if (params.invariable_site)
							if (params.prob_invar_type == ESTIMATE)
								cout << "Estimate from data" << endl;
							else if (params.prob_invar > 0.0)
								cout << params.prob_invar << endl;
							else
								cout << "No" << endl;
						if (params.gammaShapeType == ESTIMATE)
							std::cout << " a         Gamma distribution parameter alpha? " << "Estimate from data" << endl;
						else

							std::cout << " a         Gamma distribution parameter alpha? " << params.gammaShape << endl;

						std::cout <<  " c            Number of Gamma rate categories? " << params.nRate << endl;
					}
			}

			std::cout << endl << "quit [q]," << " confirm [y]," << " or change [menu] settings: ";


			if (in_pam.text_menu)
				fpin >> option_;
			else
				option_ = 'y';



			if (option_ == 'q')
				break;
			if (option_ == 'y')
				break;

			if (option_ == 'b' && isPosContinuous_ == 1) {
				isContinuous = 1 - isContinuous;
				if (isContinuous == 1)
					getPamCheckPoint (checkPointFileName_.c_str(), params);
			} //end of option 'b'

			int num;

			if (option_ == 's') {
				switch (params.stoppingRule) {
				case NO:
					params.stoppingRule = YES_MIN_ITER;
					break;
				case YES_MIN_ITER:
					params.stoppingRule = YES_MAX_ITER;	
					getIntNumber (fpin, "Enter maximum number of iterations (0 for default number): ", 0, INFINITIVE, num);
					if (num > params.nIter) params.max_nIter = num;
					break;
				case YES_MAX_ITER:
					params.stoppingRule = NO;
					break;
				case YES:
					params.stoppingRule = NO;
					break;
				}
			} //end of option 's'

			if (option_ == 'o') {
				int tmpOutGrpSeqNo_ = getOutGrpSeq (fpin);
				if (tmpOutGrpSeqNo_ >= 0 && tmpOutGrpSeqNo_ < nSeq_)
					params.outGrpSeqNo = tmpOutGrpSeqNo_;
				alignment.getName (params.outGrpSeqNo, outGrpSeqName_);
			}

			if (option_ == 'n')
				getIntNumber (fpin, "Enter minimum number of iterations: ", 0, INFINITIVE, params.nIter);

			if (option_ == 'p') {
				getDoubleNumber (fpin, "Enter probability of deleting a sequence (0.01, 1.0): ", 0.01, 1.0, params.probDel);
			}

			if (option_ == 'k') {
				getIntNumber (fpin,  "Enter the number of representatives: ", 1, nSeq_, params.nRep);
			}


			if (option_ == 'f') {
				if (params.dataType == AMINO_ACID) {
					if (params.baseFrqType == ESTIMATE)
						params.baseFrqType = USER_DEFINED;
					else
						params.baseFrqType = ESTIMATE;
				} else if (params.dataType == NUCLEOTIDE) {
					if (params.baseFrqType == ESTIMATE) {
						params.baseFrqType = EQUAL;
						params.baseA = 0.25;
						params.baseC = 0.25;
						params.baseG = 0.25;
						params.baseT = 0.25;
					} else
						if (params.baseFrqType == EQUAL) {
							do {
								params.baseFrqType = USER_DEFINED;
								getDoubleNumber (fpin, " Base A (0.0, 1.0): ", 0.0, 1.0, params.baseA);
								getDoubleNumber (fpin, " Base C (0.0, 1.0): ", 0.0, 1.0, params.baseC);
								getDoubleNumber (fpin, " Base G (0.0, 1.0): ", 0.0, 1.0, params.baseG);

								if (params.baseT + params.baseC + params.baseG > 1.0)
									std::cout <<"Error!!! the sum of base frequencies is greater than 1.0. Please, enter again..." << endl;
								params.baseT = 1.0 - params.baseA - params.baseC - params.baseG;
							} while (params.baseT < -ZERO);
						} else
							params.baseFrqType = ESTIMATE;
				} else {
					// codon frequencies
					if (params.baseFrqType == EQUAL) {
						params.baseFrqType = F1x4;
					} else if (params.baseFrqType == F1x4) {
						params.baseFrqType = F3x4;
					} else if (params.baseFrqType == F3x4) {
						params.baseFrqType = ESTIMATE;
					} else if (params.baseFrqType == ESTIMATE) {
						params.baseFrqType = EQUAL;
					}
				}//end of else
			}//end of option 'f'


			if (option_ == 't') {
				if (params.tsTvRatioType == USER_DEFINED)
					params.tsTvRatioType = ESTIMATE;
				else {
					params.tsTvRatioType = USER_DEFINED;
					getDoubleNumber (fpin, "Enter ts/tv ratio (sensible number): ", ZERO, INFINITIVE, params.tsTvRatio);
				}
			}

			if (option_ == 'u') {
				if (params.pyPuRatioType == USER_DEFINED)
					params.pyPuRatioType = ESTIMATE;
				else {
					params.pyPuRatioType = USER_DEFINED;
					getDoubleNumber (fpin, "Enter pyridinine/purine ratio (sensible number): ", ZERO, INFINITIVE, params.pyPuRatio);
				}
			}

			if (option_ == 'g') {
				if (isCodonModel(params.model)) {
					if (params.nsSy_ratio_type == DISCRETE) {
						getIntNumber(fpin, "Enter the number of classes (1-8): ", 1, 8, params.nsSy_classes);
						if (params.nsSy_classes == 1)
							params.nsSy_ratio_type = ONE_RATIO;
					}
				} else
					if (params.genPamType == ESTIMATE) {
						params.genPamType = USER_DEFINED;
						getDoubleNumber (fpin, "Enter transversion rate from A to C: ", ZERO, INFINITIVE, params.tvAC);
						getDoubleNumber (fpin, "Enter transition   rate from A to G: ", ZERO, INFINITIVE, params.tsAG);
						getDoubleNumber (fpin, "Enter transversion rate from A to T: ", ZERO, INFINITIVE, params.tvAT);
						getDoubleNumber (fpin, "Enter transversion rate from C to G: ", ZERO, INFINITIVE, params.tvCG);
						getDoubleNumber (fpin, "Enter transition   rate from C to T: ", ZERO, INFINITIVE, params.tsCT);
						getDoubleNumber (fpin, "Enter transversion rate from G to T: ", ZERO, INFINITIVE, params.tvGT);
					} else
						params.genPamType = ESTIMATE;
			}

			if (option_ == 'd')
				if (params.dataType == NUCLEOTIDE) {
					params.dataType = AMINO_ACID;
					params.model = WAG;
					params.baseFrqType = ESTIMATE;
				} else if (params.dataType == AMINO_ACID) {
					params.dataType = CODON;
					params.model = Codon_NY98;
					params.baseFrqType = ESTIMATE;
				} else {
					params.dataType = NUCLEOTIDE;
					params.model = HKY85;
					params.baseFrqType = ESTIMATE;
				}

			if (option_ == 'm') {
				if (params.dataType == AMINO_ACID) {
					switch (params.model) {
					case WAG:
						params.model = JTT;
						break;
					case JTT:
						params.model = VT;
						break;
					case VT:
						params.model = MtREV24;
						break;
					case MtREV24:
						params.model = rtREV;
						break;
					case rtREV:
						params.model = Blosum62;
						break;
					case Blosum62:
						params.model = Dayhoff;
						break;
					case Dayhoff:
						params.model = AA_USER;
						cout << "Enter name of rate file: ";
						fpin >> params.rate_file;
						break;
					default:
						params.model = WAG;
					} // end of swith
				} else if (params.dataType == NUCLEOTIDE) {//DNA
					if (params.model == HKY85)
						params.model = TN93;
					else
						if (params.model == TN93)
							params.model = GTR;
						else
							params.model = HKY85;
				} else { // CODON
					if (params.model == Codon_GY94)
						params.model = Codon_YN98;
					else if (params.model == Codon_YN98)
						params.model = Codon_Pedersen98;
					else if (params.model == Codon_Pedersen98)
						params.model = Codon_GTR;
					else if (params.model == Codon_GTR)
						params.model = Codon_NY98;
					else if (params.model == Codon_NY98)
						params.model = Codon_GY94;
					else if (params.model == Codon_PosRate)
						params.model = Codon_GY94;
				}
			}  //end of option 'm'

			if (option_ == 'r') {
				if (params.rateType == SITE_SPECIFIC) {
					params.nRate = 1;
					//params.invariable_site = 0;
					params.rateType = UNIFORM;
				} else
					if (params.rateType == UNIFORM) {
						params.rateType = GAMMA;
						//params.invariable_site = 0;
						params.nRate = 4;
						params.gammaShape = 1.0;
					} else {
						params.nRate = alignment.getNSite ();
						//params.invariable_site = 0;
						params.rateType = SITE_SPECIFIC;
					}
			}

			if (option_ == 'a') {
				if (params.gammaShapeType == USER_DEFINED)
					params.gammaShapeType = ESTIMATE;
				else {
					params.gammaShapeType = USER_DEFINED;
					getDoubleNumber (fpin, "Gamma distribution parameter alpha (sensible number): ", ZERO, INFINITIVE, params.gammaShape);
				}
			}

			if (option_ == 'c') {
				getIntNumber (fpin, "Number of Gamma rate categories ( 1, 32 ): ", 1,  32, params.nRate);
			}

			if (option_ == 'i')
				//if (params.invariable_site)
					if (params.prob_invar_type == ESTIMATE)  {
						params.prob_invar_type = USER_DEFINED;
						getDoubleNumber (fpin, "Proportion of invariable_site (0 - 0.99): ", 0.0, 0.99, params.prob_invar);
						//if (params.prob_invar == 0.0)
							//params.invariable_site = false;
					} else
						params.prob_invar_type = ESTIMATE;
				/*else {
					params.invariable_site = true;
					params.prob_invar_type = ESTIMATE;
				}*/

			if (option_ == 'w' && isCodonModel(params.model)) {
				switch( params.nsSy_ratio_type ) {
				case ONE_RATIO:
					params.nsSy_ratio_type = NEUTRAL;
					params.nsSy_classes = 2;
					break;
				case NEUTRAL:
					params.nsSy_ratio_type = NEARLY_NEUTRAL;
					params.nsSy_classes = 2;
					break;
				case NEARLY_NEUTRAL:
					params.nsSy_ratio_type = SELECTION;
					params.nsSy_classes = 3;
					break;
				case SELECTION:
					params.nsSy_ratio_type = NEARLY_SELECTION;
					params.nsSy_classes = 3;
					break;
				case NEARLY_SELECTION:
					params.nsSy_ratio_type = DISCRETE;
					params.nsSy_classes = 3;
					break;
				case DISCRETE:
					params.nsSy_ratio_type = ONE_RATIO;
					params.nsSy_classes = 1;
					break;
				}
			}
			if (option_ == 'z') {
				getIntNumber(fpin, "Enter the number of bootstrap replicates (0 for turning bootstrap off): ", 0, 10000, params.gen_bootstrap);
			}

			for (int count_ = 0; count_ < 15; count_ ++)
				std::cout << endl;
		} while (in_pam.text_menu); //finish user defined parameters!!!!!!!!!!!


#ifdef PARALLEL
	MPI_Bcast(&option_, 1, MPI_CHAR, mpi_master_rank, MPI_COMM_WORLD);
#endif


	if (option_ != 'y')
		return 0;

#ifdef PARALLEL
	if (in_pam.text_menu) {
		/* Broadcast parameters */

		MPI_Bcast(&params.nIter, 1, MPI_INT, mpi_master_rank, MPI_COMM_WORLD);
		MPI_Bcast(&params.probDel, 1, MPI_DOUBLE, mpi_master_rank, MPI_COMM_WORLD);
		MPI_Bcast(&params.nRep, 1, MPI_INT, mpi_master_rank, MPI_COMM_WORLD);

		MPI_Bcast(&params.stoppingRule, 1, MPI_INT, mpi_master_rank, MPI_COMM_WORLD);
		MPI_Bcast(&params.dataType, 1, MPI_INT, mpi_master_rank, MPI_COMM_WORLD);
		MPI_Bcast(&params.model, 1, MPI_INT, mpi_master_rank, MPI_COMM_WORLD);
		MPI_Bcast(&params.baseA, 1, MPI_DOUBLE, mpi_master_rank, MPI_COMM_WORLD);
		MPI_Bcast(&params.baseC, 1, MPI_DOUBLE, mpi_master_rank, MPI_COMM_WORLD);
		MPI_Bcast(&params.baseG, 1, MPI_DOUBLE, mpi_master_rank, MPI_COMM_WORLD);
		MPI_Bcast(&params.baseT, 1, MPI_DOUBLE, mpi_master_rank, MPI_COMM_WORLD);
		MPI_Bcast(&params.baseFrqType, 1, MPI_INT, mpi_master_rank, MPI_COMM_WORLD);

		MPI_Bcast(&params.tsTvRatio, 1, MPI_DOUBLE, mpi_master_rank, MPI_COMM_WORLD);

		MPI_Bcast(&params.tsTvRatioType, 1, MPI_INT, mpi_master_rank, MPI_COMM_WORLD);

		MPI_Bcast(&params.pyPuRatio, 1, MPI_DOUBLE, mpi_master_rank, MPI_COMM_WORLD);

		MPI_Bcast(&params.pyPuRatioType, 1, MPI_INT, mpi_master_rank, MPI_COMM_WORLD);

		MPI_Bcast(&params.tsAG, 1, MPI_DOUBLE, mpi_master_rank, MPI_COMM_WORLD);
		MPI_Bcast(&params.tsCT, 1, MPI_DOUBLE, mpi_master_rank, MPI_COMM_WORLD);
		MPI_Bcast(&params.tvAC, 1, MPI_DOUBLE, mpi_master_rank, MPI_COMM_WORLD);
		MPI_Bcast(&params.tvAT, 1, MPI_DOUBLE, mpi_master_rank, MPI_COMM_WORLD);
		MPI_Bcast(&params.tvCG, 1, MPI_DOUBLE, mpi_master_rank, MPI_COMM_WORLD);
		MPI_Bcast(&params.tvGT, 1, MPI_DOUBLE, mpi_master_rank, MPI_COMM_WORLD);

		MPI_Bcast(&params.genPamType, 1, MPI_INT, mpi_master_rank, MPI_COMM_WORLD);

		MPI_Bcast(&params.rateType, 1, MPI_INT, mpi_master_rank, MPI_COMM_WORLD);

		MPI_Bcast(&params.nRate, 1, MPI_INT, mpi_master_rank, MPI_COMM_WORLD);
		MPI_Bcast(&params.gammaShape, 1, MPI_DOUBLE, mpi_master_rank, MPI_COMM_WORLD);
		MPI_Bcast(&params.gammaShapeType, 1, MPI_INT, mpi_master_rank, MPI_COMM_WORLD);

		MPI_Bcast(&params.prob_invar, 1, MPI_DOUBLE, mpi_master_rank, MPI_COMM_WORLD);
		MPI_Bcast(&params.prob_invar_type, 1, MPI_INT, mpi_master_rank, MPI_COMM_WORLD);
		//MPI_Bcast(&params.invariable_site, 1, MPI_CHAR, mpi_master_rank, MPI_COMM_WORLD);
		//MPI_Bcast(&params.codon_model, 1, MPI_INT, mpi_master_rank, MPI_COMM_WORLD);
		MPI_Bcast(&params.nsSy_ratio_type, 1, MPI_CHAR, mpi_master_rank, MPI_COMM_WORLD);
		MPI_Bcast(&params.nsSy_classes, 1, MPI_INT, mpi_master_rank, MPI_COMM_WORLD);

		MPI_Bcast(&params.outGrpSeqNo, 1, MPI_INT, mpi_master_rank, MPI_COMM_WORLD);

		int name_len = params.rate_file.length();
		MPI_Bcast(&name_len, 1, MPI_INT, mpi_master_rank, MPI_COMM_WORLD);

		if (name_len != 0) {
			char *name_str = new char[name_len+1];
			strcpy(name_str, params.rate_file.c_str());
			MPI_Bcast(name_str, name_len+1, MPI_CHAR, mpi_master_rank, MPI_COMM_WORLD);
			params.rate_file = name_str;
			delete name_str;
		}
	}
#endif


	in_pam.random_seed = RanNum::initRandom (in_pam.random_seed);

	if (isMasterProc())
		cout << "Initial seed: " << in_pam.random_seed << endl;
	

	if (isContinuous == 0) {

		if (params.baseFrqType == ESTIMATE) {
			params.baseA = -1.0;
			params.baseC = -1.0;
			params.baseG = -1.0;
			params.baseT = -1.0;
		}
	
		if (params.tsTvRatioType == ESTIMATE)
			params.tsTvRatio = - 1.0;
		if (params.pyPuRatioType == ESTIMATE)
			params.pyPuRatio = -1.0;
		if (params.genPamType == ESTIMATE) {
			params.tsAG = -1.0;
			params.tsCT = -1.0;
			params.tvAC = -1.0;
			params.tvAT = -1.0;
			params.tvCG = -1.0;
			params.tvGT = -1.0;
		}
	
	
		if (params.rateType == GAMMA) {
			if (params.gammaShapeType == ESTIMATE)
				params.gammaShape = -1.0;
		} else {
			params.gammaShape = -1.0;
			if (params.rateType == SITE_SPECIFIC)
				params.nRate = -1;
		}


	}

	// codon model
	if (isCodonModel(params.model)) {
		params.codon_model = params.model;
		params.codonFrqType = params.baseFrqType;
		params.codon_tsTvRatio = params.tsTvRatio;
		params.codon_tsTvRatioType = params.tsTvRatioType;
		// set default parameters for topology search
		params.model = DEFAULT_SUBST_MODEL;
		params.baseFrqType = ESTIMATE;
		if (isContinuous == 0) {
			params.tsTvRatio = -1;
		}
		params.tsTvRatioType = ESTIMATE;
		params.dataType = NUCLEOTIDE;
	}

	return 1;
} // getParameter



//====================================================================================
//ab:cccc:dd: X
int Interface::getIntNum (const char  *aLine) {
	char intNumStr_[20];
	int numPos_ = 0;

	int len_ = strlen (aLine);
	int semiColonPos_ = len_ - 1;
	while (aLine[semiColonPos_] != ':')
		semiColonPos_ --;

	for (int linePos_ = semiColonPos_ + 1; linePos_ < len_; linePos_ ++)
		if (aLine[linePos_] >= '0' && aLine[linePos_] <= '9') {
			intNumStr_[numPos_] = aLine[linePos_];
			numPos_ ++;
		}
	intNumStr_[numPos_]= '\0';
	char *endPos_;
	double doubleVal_ = strtod (intNumStr_, &endPos_);

	return static_cast<int> (doubleVal_ + ZERO);
}

//====================================================================================
//ab:cccc:dd: X
double Interface::getDoubleNum (const char  *aLine) {

	char doubleNumStr_[40];
	int numPos_ = 0;

	int len_ = strlen (aLine);
	int semiColonPos_ = len_ - 1;
	while (aLine[semiColonPos_] != ':')
		semiColonPos_ --;

	for (int linePos_ = semiColonPos_ + 1; linePos_ < len_; linePos_ ++)
		if (aLine[linePos_] == '.' || aLine[linePos_] == '-' ||
		        (aLine[linePos_] >= '0' && aLine[linePos_] <= '9') ) {
			doubleNumStr_[numPos_] = aLine[linePos_];

			numPos_ ++;
		}

	doubleNumStr_[numPos_]= '\0';
	char *endPos_;
	double doubleVal_ = strtod (doubleNumStr_, &endPos_);
	return doubleVal_;
}

bool checkLine(ifstream &checkPointFile_, string &line, const char *str) {
	getline (checkPointFile_, line);
	if (line.compare(0, strlen(str), str) != 0) {
		cout << "Incorrect checkpoint file...ignore it." << endl;
		checkPointFile_.close();
		return false;
	}
	return true;
}

//====================================================================================
//====================================================================================
//get all parameters from the last stopped point
int Interface::getPamCheckPoint (const char *checkPointFileName, CommonParameters &params) {

	ifstream checkPointFile_;
	checkPointFile_.open (checkPointFileName);

	if (!checkPointFile_.is_open())
		return 0;

	string aLine_;

	if (!checkLine(checkPointFile_, aLine_, "Number iterations: ")) return 0;
	params.nIter = getIntNum (aLine_.c_str());

	if (!checkLine(checkPointFile_, aLine_, "Maximum number iterations: ")) return 0;
	params.max_nIter = getIntNum (aLine_.c_str());

	if (!checkLine(checkPointFile_, aLine_, "Current number iterations: ")) return 0;
	params.cur_nIter = getIntNum (aLine_.c_str());

	if (!checkLine(checkPointFile_, aLine_, "Probability of deleting a sequence: ")) return 0;
	params.probDel = getDoubleNum (aLine_.c_str());

	if (!checkLine(checkPointFile_, aLine_, "Number representatives: ")) return 0;
	params.nRep = getIntNum (aLine_.c_str());

	if (!checkLine(checkPointFile_, aLine_, "Stopping rule ")) return 0;
	int stoppingRuleNum_ = getIntNum (aLine_.c_str());
	switch (stoppingRuleNum_) {
	case 0:
		params.stoppingRule = YES;
		break;
	case 1:
		params.stoppingRule = YES_MIN_ITER;
		break;
	case 2:
		params.stoppingRule = YES_MAX_ITER;
		break;
	default:
		params.stoppingRule = NO;
	}

	if (!checkLine(checkPointFile_, aLine_, "Type of data ")) return 0;
	int dataTypeNum_ = getIntNum (aLine_.c_str());

	if (dataTypeNum_ == 0)
		params.dataType = NUCLEOTIDE;
	else
		params.dataType = AMINO_ACID;

	if (!checkLine(checkPointFile_, aLine_, "Substitution model ")) return 0;
	int modelTypeNum_ = getIntNum (aLine_.c_str());
	if (modelTypeNum_ == 0)
		params.model = HKY85;
	if (modelTypeNum_ == 1)
		params.model = TN93;
	if (modelTypeNum_ == 2)
		params.model = GTR;
	if (modelTypeNum_ == 3)
		params.model = WAG;
	if (modelTypeNum_ == 4)
		params.model = JTT;
	if (modelTypeNum_ == 5)
		params.model = VT;
	if (modelTypeNum_ == 6)
		params.model = MtREV24;
	if (modelTypeNum_ == 7)
		params.model = Blosum62;
	if (modelTypeNum_ == 8)
		params.model = Dayhoff;
	if (modelTypeNum_ == 9)
		params.model = rtREV;
	if (modelTypeNum_ == 10)
		params.model = AA_USER;

	if (!checkLine(checkPointFile_, aLine_, "Frequency of Base A: ")) return 0;
	params.baseA = getDoubleNum (aLine_.c_str());

	if (!checkLine(checkPointFile_, aLine_, "Frequency of Base C: ")) return 0;
	params.baseC = getDoubleNum (aLine_.c_str());

	if (!checkLine(checkPointFile_, aLine_, "Frequency of Base G: ")) return 0;
	params.baseG = getDoubleNum (aLine_.c_str());

	if (!checkLine(checkPointFile_, aLine_, "Frequency of Base T: ")) return 0;
	params.baseT = getDoubleNum (aLine_.c_str());

	if (!checkLine(checkPointFile_, aLine_, "Type of parameters ")) return 0;
	int baseFrqTypeNum_ = getIntNum (aLine_.c_str());
	if (baseFrqTypeNum_ == 0)
		params.baseFrqType = ESTIMATE;
	if (baseFrqTypeNum_ == 1)
		params.baseFrqType = USER_DEFINED;
	if (baseFrqTypeNum_ == 2)
		params.baseFrqType = EQUAL;


	//======================================================
	if (!checkLine(checkPointFile_, aLine_, "Transition/transversion ratito: ")) return 0;
	params.tsTvRatio = getDoubleNum (aLine_.c_str());

	if (!checkLine(checkPointFile_, aLine_, "Type of parameters ")) return 0;
	int tsTvRatioTypeNum_ = getIntNum (aLine_.c_str());
	if (tsTvRatioTypeNum_ == 0)
		params.tsTvRatioType = ESTIMATE;
	else
		params.tsTvRatioType = USER_DEFINED;


	if (!checkLine(checkPointFile_, aLine_, "Pyridimine/purine ratito: ")) return 0;
	params.pyPuRatio = getDoubleNum (aLine_.c_str());

	if (!checkLine(checkPointFile_, aLine_, "Type of parameters ")) return 0;
	int pyPuRatioTypeNum_ = getIntNum (aLine_.c_str());
	if (pyPuRatioTypeNum_ == 0)
		params.pyPuRatioType = ESTIMATE;
	else
		params.pyPuRatioType = USER_DEFINED;

	if (!checkLine(checkPointFile_, aLine_, "Transition rate from")) return 0;
	params.tsAG = getDoubleNum (aLine_.c_str());

	if (!checkLine(checkPointFile_, aLine_, "Transition rate from")) return 0;
	params.tsCT = getDoubleNum (aLine_.c_str());

	if (!checkLine(checkPointFile_, aLine_, "Transversion rate from")) return 0;
	params.tvAC = getDoubleNum (aLine_.c_str());

	if (!checkLine(checkPointFile_, aLine_, "Transversion rate from")) return 0;
	params.tvAT = getDoubleNum (aLine_.c_str());

	if (!checkLine(checkPointFile_, aLine_, "Transversion rate from")) return 0;
	params.tvCG = getDoubleNum (aLine_.c_str());

	if (!checkLine(checkPointFile_, aLine_, "Transversion rate from")) return 0;
	params.tvGT = getDoubleNum (aLine_.c_str());

	if (!checkLine(checkPointFile_, aLine_, "Type of parameters ")) return 0;
	int genPamTypeNum_ = getIntNum (aLine_.c_str());
	if (genPamTypeNum_ == 0)
		params.genPamType = ESTIMATE;
	else
		params.genPamType = USER_DEFINED;


	if (!checkLine(checkPointFile_, aLine_, "Type of rate heterogeneity")) return 0;
	int rateTypeNum_ = getIntNum (aLine_.c_str());
	if (rateTypeNum_ == 0)
		params.rateType = UNIFORM;
	if (rateTypeNum_ == 1)
		params.rateType = SITE_SPECIFIC;
	if (rateTypeNum_ == 2)
		params.rateType = GAMMA;

	if (!checkLine(checkPointFile_, aLine_, "Number rates: ")) return 0;
	params.nRate = getIntNum (aLine_.c_str());

	if (!checkLine(checkPointFile_, aLine_, "Gamma distribution parameter alpha: ")) return 0;
	params.gammaShape = getDoubleNum (aLine_.c_str());

	if (!checkLine(checkPointFile_, aLine_, "Type of parameters")) return 0;
	int gammaShapeTypeNum_ = getIntNum (aLine_.c_str());
	if (gammaShapeTypeNum_ == 0)
		params.gammaShapeType = ESTIMATE;
	else
		params.gammaShapeType = USER_DEFINED;



	// invariant site information
	if (!checkLine(checkPointFile_, aLine_, "Invariant type ")) return 0;
	switch (getIntNum(aLine_.c_str())) {
	case 0: // none
		params.prob_invar_type = USER_DEFINED;
		//params.invariable_site = false;
		break;
	case 1: // estimate
		//params.invariable_site = true;
		params.prob_invar_type = ESTIMATE;
		break;
	case 2: // user defined
		//params.invariable_site = true;
		params.prob_invar_type = USER_DEFINED;
		break;
	}

	// get the probability that a site be invariable
	if (!checkLine(checkPointFile_, aLine_, "Proportion of invariable sites: ")) return 0;
	params.prob_invar = getDoubleNum (aLine_.c_str());

	// out group number
	if (!checkLine(checkPointFile_, aLine_, "Out group sequence: ")) return 0;
	params.outGrpSeqNo = getIntNum (aLine_.c_str());

	// running from bootstrap sample
	if (!checkLine(checkPointFile_, aLine_, "Bootstrap sample: ")) return 0;
	params.gen_bootstrap = getIntNum (aLine_.c_str());

	// running from bootstrap sample
	if (!checkLine(checkPointFile_, aLine_, "Current bootstrap sample: ")) return 0;
	params.cur_bootstrap = getIntNum (aLine_.c_str());


	if (!checkLine(checkPointFile_, aLine_, "Build consensus: ")) return 0;
	params.build_consensus = getIntNum (aLine_.c_str());

	if (!checkLine(checkPointFile_, aLine_, "Current best log-likelihood: ")) return 0;
	params.bestLogLi = getDoubleNum(aLine_.c_str());

	if (!checkLine(checkPointFile_, aLine_, "Elapsed time: ")) return 0;
	params.progTime = getDoubleNum(aLine_.c_str());
	
	// get finish flag
	if (!checkLine(checkPointFile_, aLine_, "Finished: ")) return 0;
	int isContinuous_ = (1 - getIntNum (aLine_.c_str()));
	return isContinuous_;
}

/**
	write the parameter information, in case of restarting
*/
void writePamCheckPoint (CommonParameters &params, 
	const char  *checkPointFileName, int isFinished) {

	ofstream checkPointFile_;
	checkPointFile_.open (checkPointFileName);

	checkPointFile_ << "Number iterations: " << params.nIter << endl;
	checkPointFile_ << "Maximum number iterations: " << params.max_nIter << endl;
	checkPointFile_ << "Current number iterations: " << params.cur_nIter << endl;

	checkPointFile_ << "Probability of deleting a sequence: " << params.probDel << endl;
	checkPointFile_ << "Number representatives: " << params.nRep - 1 << endl;
	checkPointFile_ << "Stopping rule (0: YES, 1: YES_MIN_ITER, 2: YES_MAX_ITER, 3: NO): " << params.stoppingRule << endl;

	checkPointFile_ << "Type of data (0:NUCLEOTIDE, 1:AMINO_ACID): " << mymodel.getDataType () << endl;


	checkPointFile_ << "Substitution model (0:HKY85, 1: TN93, 2:GTR, 3:WAG, 4:JTT, 5:VT, 6:MtREV24, 7:Blosum62, 8:Dayhoff, 9:rtREV, 10: User-defined): " << mymodel.getType () << endl;
	DVec20 baseFrqArr_;
	mymodel.getStateFrq (baseFrqArr_);
	checkPointFile_ << "Frequency of Base A: " << baseFrqArr_[BASE_A] << endl;
	checkPointFile_ << "Frequency of Base C: " << baseFrqArr_[BASE_C] << endl;
	checkPointFile_ << "Frequency of Base G: " << baseFrqArr_[BASE_G] << endl;
	checkPointFile_ << "Frequency of Base T: " << baseFrqArr_[BASE_T] << endl;
	checkPointFile_ << "Type of parameters (0:ESTIMATE,  1:USER_DEFINED, 2: EQUAL): " << mymodel.getBaseFrqType () << endl;


	checkPointFile_ << "Transition/transversion ratito: " << mymodel.getTsTvRatio () / 2.0 << endl;
	checkPointFile_ << "Type of parameters (0:ESTIMATE,  1:USER_DEFINED): " << mymodel.getTsTvRatioType () << endl;

	checkPointFile_ << "Pyridimine/purine ratito: " << mymodel.getPyPuRatio () << endl;
	checkPointFile_ << "Type of parameters (0:ESTIMATE,  1:USER_DEFINED): " << mymodel.getPyPuRatioType () << endl;

	double tsAG_, tsCT_, tvAC_, tvAT_, tvCG_, tvGT_;
	mymodel.getGenPam (tsAG_, tsCT_, tvAC_, tvAT_, tvCG_, tvGT_);

	checkPointFile_ << "Transition rate from A to G: " << tsAG_ << endl;
	checkPointFile_ << "Transition rate from C to T: " << tsCT_ << endl;

	checkPointFile_ << "Transversion rate from A to C: " << tvAC_ << endl;
	checkPointFile_ << "Transversion rate from A to T: " << tvAT_ << endl;
	checkPointFile_ << "Transversion rate from C to G: " << tvCG_ << endl;
	checkPointFile_ << "Transversion rate from G to T: " << tvGT_ << endl;
	checkPointFile_ << "Type of parameters (0:ESTIMATE,  1:USER_DEFINED): " << mymodel.getPyPuRatioType () << endl;

	RATE_TYPE rateType_ = myrate.getType ();
	checkPointFile_ << "Type of rate heterogeneity (0:UNIFORM, 1:SITE_SPECIFIC, 2:GAMMA): " << rateType_ << endl;
	checkPointFile_ << "Number rates: " << myrate.getNRate () /*- myrate.use_invar_site*/ << endl;

	checkPointFile_ << "Gamma distribution parameter alpha: " << myrate.getGammaShape () << endl ;
	checkPointFile_ << "Type of parameters (0:ESTIMATE,  1:USER_DEFINED): " << myrate.getGammaShapeType () << endl;

	
	int invar_site ;
	if (myrate.prob_invar_site == 0.0)
		invar_site = 0;
	else if (myrate.getInvarSiteType() == ESTIMATE)
		invar_site = 1;
	else 
		invar_site = 2;
	checkPointFile_ << "Invariant type (0: NONE, 1:ESTIMATE, 2: USER_DEFINED): " << invar_site << endl;
	checkPointFile_ << "Proportion of invariable sites: " << myrate.prob_invar_site << endl;
	

	checkPointFile_ << "Out group sequence: " << params.outGrpSeqNo << endl;
	checkPointFile_ << "Bootstrap sample: " << params.gen_bootstrap << endl;
	checkPointFile_ << "Current bootstrap sample: " << params.cur_bootstrap << endl;
	checkPointFile_ << "Build consensus: " << params.build_consensus << endl;
	checkPointFile_.precision(10);
	checkPointFile_ << "Current best log-likelihood: " << params.bestLogLi << endl;
	checkPointFile_ << "Elapsed time: " << params.progTime << endl;

	checkPointFile_ << "Finished: " << isFinished << endl;

	checkPointFile_.close ();
}


//====================================================================================
//====================================================================================
int Interface::isExistedFile (const char *fileName) {
	ifstream existedFile_;
	existedFile_.open (fileName);


	int isExistedFile_ = 1;
	if (existedFile_ == 0)
		isExistedFile_ = 0;

	existedFile_.close ();

	return isExistedFile_;
}

int Interface::getCommandLine( int argc, char *argv[], CommonParameters &params, InputParameters &in_pam) {
	/* parse the arguments */
	int count = 1;
	while (count < argc) {

		if ( strcmp(argv[count], "-h") == 0 || strcmp(argv[count], "-?") == 0 ) {
			// print the usage dialog
			if (isMasterProc())
				usage();
			Finalize(1);

		} else if (strcmp(argv[count], "-n") == 0)  {
			// set the iteration count
			params.nIter = atoi(argv[count+1]);
			if (params.nIter < 0) {
				if (isMasterProc())
					cout << "Please enter a non-negative iteration count" << endl;
				Finalize(1);
			}
			count++;

		} else if (strcmp(argv[count], "-w") == 0)  {
			// set the rate type
			if (strcmp(argv[count+1], "uniform") == 0) {
				params.nRate = 1;
				params.rateType = UNIFORM;
			} else if (strcmp(argv[count+1], "gamma") == 0) {
				params.rateType = GAMMA;
				//params.invariable_site = false;
				params.prob_invar = 0.0;
				params.prob_invar_type = USER_DEFINED;
				params.nRate = 4;
				params.gammaShape = 1.0;

			} else if (strcmp(argv[count+1], "igamma") == 0) {
				params.rateType = GAMMA;
				//params.invariable_site = true;
				params.prob_invar = 0.0;
				params.prob_invar_type = ESTIMATE;
				params.nRate = 4;
				params.gammaShape = 1.0;

			} else if (strcmp(argv[count+1], "sitespec") == 0) {
				params.nRate = alignment.getNSite ();
				params.rateType = SITE_SPECIFIC;
			} else {
				if (isMasterProc())
					cout << "Invalid parameter " << argv[count+1] << endl;
				Finalize(1);
			}
			count++;

		} else if (strcmp(argv[count], "-s") == 0)  {
			// set the stopping rule
			if (strcmp(argv[count+1], "on") == 0) {
				params.stoppingRule = YES_MIN_ITER;
			} else if (strcmp(argv[count+1], "off") == 0) {
				params.stoppingRule = NO;
			} else if (strcmp(argv[count+1], "max") == 0) {
				params.stoppingRule = YES_MAX_ITER;
			// set the maximum iteration count
				if (argc > count+2) params.max_nIter = atoi(argv[count+2]);
				if (params.max_nIter < 0) {
					if (isMasterProc())
						cout << "Please enter a non-negative iteration count" << endl;
					Finalize(1);
				}
				count++;
			} else {
				if (isMasterProc())
					cout << "Invalid parameter " << argv[count+1] << endl;
				Finalize(1);
			}
			count++;

		} else if (strcmp(argv[count], "-ni") == 0)  {
			// don't use the interactive interface
			in_pam.text_menu = false;

		} else if (strcmp(argv[count], "-p") == 0)  {
			// set the probability of deleting sequences
			params.probDel = atof(argv[count+1]);
			if (params.probDel < 0 || params.probDel > 1) {
				if (isMasterProc())
					cout << "You must specify probability between 0 and 1" << endl;
				Finalize(1);
			}
			count++;

		} else if (strcmp(argv[count], "-k") == 0)  {
			// set the number of representatives
			params.nRep = atoi(argv[count+1]);
			if (params.nRep < 1) {
				if (isMasterProc())
					cout << "Invalid number of representatives" << endl;
				Finalize(1);
			}
			count++;

		} else if (strcmp(argv[count], "-m") == 0)  {
			// set the model type
			count++;
			if (strcmp(argv[count], "HKY85") == 0 || strcmp(argv[count], "HKY") == 0) {
				params.model = HKY85;
			} else if (strcmp(argv[count], "JC69") == 0 || strcmp(argv[count], "JC") == 0) {
				params.model = HKY85;
				params.tsTvRatio = 0.5;
				params.tsTvRatioType = USER_DEFINED;
				params.baseFrqType = EQUAL;
			} else if (strcmp(argv[count], "K2P") == 0) {
				params.model = HKY85;
				params.baseFrqType = EQUAL;
			} else if (strcmp(argv[count], "F81") == 0) {
				params.model = HKY85;
				params.tsTvRatio = 0.5;
				params.tsTvRatioType = USER_DEFINED;
			} else if (strcmp(argv[count], "TN93") == 0 || strcmp(argv[count], "TN") == 0) {
				params.model = TN93;
			} else if (strcmp(argv[count], "GTR") == 0) {
				params.model = GTR;
			} else if (strcmp(argv[count], "GY94") == 0) {
				params.model = Codon_GY94;
			} else if (strcmp(argv[count], "YN98") == 0) {
				params.model = Codon_YN98;
			} else if (strcmp(argv[count], "NY98") == 0) {
				params.model = Codon_NY98;
				params.nsSy_ratio_type = DISCRETE;
				params.nsSy_classes = 3;
			} else if (strcmp(argv[count], "CP98") == 0) {
				params.model = Codon_Pedersen98;
			} else if (strcmp(argv[count], "CGTR") == 0) {
				params.model = Codon_GTR;
			} else if (strcmp(argv[count], "CPR") == 0) {
				params.model = Codon_PosRate;

			} else if (strcmp(argv[count], "WAG") == 0) {
				params.model = WAG;
			} else if (strcmp(argv[count], "Dayhoff") == 0) {
				params.model = Dayhoff;
			} else if (strcmp(argv[count], "JTT") == 0) {
				params.model = JTT;
			} else if (strcmp(argv[count], "VT") == 0) {
				params.model = VT;
			} else if (strcmp(argv[count], "mtREV") == 0) {
				params.model = MtREV24;
			} else if (strcmp(argv[count], "rtREV") == 0) {
				params.model = rtREV;
			} else if (strcmp(argv[count], "Blosum") == 0) {
				params.model = Blosum62;
			} else {
				params.model = AA_USER;
				params.rate_file = argv[count];
				/*
				if (isMasterProc())
					cout << "Error: undefined model type" << endl;
				Finalize(1);
				*/
			}

		} else if (strcmp(argv[count], "-cdm") == 0)  {
			// set program mode to only compute distance matrix
			program_mode = PROG_COMPUTE_MATRIX;

		} else if (strcmp(argv[count], "-ep") == 0)  {
			// set program mode to only estimating parameters
			program_mode = PROG_PARM_ESTIMATE;
		} else if (strcmp(argv[count], "-ip") == 0)  {
			params.ap_sitespec = false;
		} else if (strcmp(argv[count], "-nni") == 0)  {
			// set program mode to only estimating parameters
			nni_test = true;
		} else if (strcmp(argv[count], "-opt") == 0)  {
			// set the testing of the major optimization step
			major_step_test = true;
		} else if (strcmp(argv[count], "-sfc") == 0)  {
			// don't load the check point file
			load_check_point = false;
		} else if (strcmp(argv[count], "-u") == 0)  {
			strcpy(in_pam.tree_file, argv[count+1]);
			count++;
		} else if (strcmp(argv[count], "-wp") == 0)  {
			count++;
			params.model = Codon_NY98;
			params.nsSy_ratio_type = DISCRETE;
			params.nsSy_classes = atoi(argv[count]);
			for (int i = 0; i < params.nsSy_classes; i++) {
				count++;
				params.nsSy_ratio_val[i] = atof(argv[count]);
			}
		} else if (strcmp(argv[count], "-pp") == 0)  {
			count++;
			params.model = Codon_NY98;
			params.nsSy_ratio_type = DISCRETE;
			params.nsSy_classes = atoi(argv[count]);
			double sum = 0;
			for (int i = 0; i < params.nsSy_classes - 1; i++) {
				count++;
				params.nsSy_prob_val[i] = atof(argv[count]);
				if (params.nsSy_prob_val[i] < 0 || params.nsSy_prob_val[i] > 1) {
					if (isMasterProc())
						cout << "Probability must be between 0 and 1.0" << endl;
					Finalize(1);
				}
				sum += params.nsSy_prob_val[i];

			}
			if (sum > 1) {
				if (isMasterProc())
					cout << "Probability sum exceeds 1.0" << endl;
				Finalize(1);
			}

		} else if (strcmp(argv[count], "-ww") == 0)  {
			// set the Nielsen Yang codon based model typedef
			count++;
			params.model = Codon_NY98;
			params.nsSy_ratio_type = DISCRETE;
			params.nsSy_classes = 3;

			if (strcmp(argv[count], "1ratio") == 0) {
				params.nsSy_ratio_type = ONE_RATIO;
				params.nsSy_classes = 1;
			} else if (strcmp(argv[count], "neutral") == 0) {
				params.nsSy_ratio_type = NEUTRAL;
				params.nsSy_classes = 2;
			} else if (strcmp(argv[count], "nneutral") == 0) {
				params.nsSy_ratio_type = NEARLY_NEUTRAL;
				params.nsSy_classes = 2;
			} else if (strcmp(argv[count], "sel") == 0) {
				params.nsSy_ratio_type = SELECTION;
				params.nsSy_classes = 3;
			} else if (strcmp(argv[count], "nsel") == 0) {
				params.nsSy_ratio_type = NEARLY_SELECTION;
				params.nsSy_classes = 3;
			} else if (strcmp(argv[count], "discrete") == 0) {
				params.nsSy_ratio_type = DISCRETE;
				params.nsSy_classes = 3;
			} else {
				cout << "NY98 model must be either 1ratio, neutral, nneutral, sel, nsel or discrete" << endl;
				Finalize(1);
			}
		} else if (strcmp(argv[count], "-f") == 0)  {
			// set state frequencies
			count++;
			if (strcmp(argv[count], "equal")  == 0)
				params.baseFrqType = EQUAL;
			else if (strcmp(argv[count], "F61")  == 0)
				params.baseFrqType = ESTIMATE;
			else if (strcmp(argv[count], "F3x4")  == 0)
				params.baseFrqType = F3x4;
		} else if (strcmp(argv[count], "-brent") == 0)  {
			params.pam_brent = 2;
		} else if (strcmp(argv[count], "-sa") == 0)  {
			params.pam_brent = 1;
		} else if (strcmp(argv[count], "-seed") == 0)  {
			count++;
			in_pam.random_seed = atoi(argv[count]);
		} else if (strcmp(argv[count], "-pam") == 0)  {
			count++;
			MAX_IT_ALL_PAM_OPT = atoi(argv[count]);
		} else if (strcmp(argv[count], "-c") == 0)  {
			// number of gamma rate categories
			count++;
			params.nRate = atoi(argv[count]);
			if (params.nRate < 1 || params.nRate > MAX_NUM_RATE) {
				if (isMasterProc())
					cout << "Number of gamma rate categories must be between 2 and " << MAX_NUM_RATE << endl;
				Finalize(1);
			}
		} else if (strcmp(argv[count], "-bs") == 0)  {
			params.gen_bootstrap = 1;
		} else if (strcmp(argv[count], "-b") == 0)  {
			count++;
			params.gen_bootstrap = atoi(argv[count]);
		} else if (strcmp(argv[count], "-prefix") == 0)  {
			count++;
			in_pam.out_prefix = argv[count];
		} else if (strcmp(argv[count], "-param") == 0)  {
			count++;
			in_pam.param_file = argv[count];
		} else if (strcmp(argv[count], "-pl") == 0)  {
			params.print_pat_logl = true;
		} else if (strcmp(argv[count], "-wpl") == 0)  {
			params.print_pat_logl = true;
		} else if (strcmp(argv[count], "-wsl") == 0)  {
			params.print_site_logl = true;
		} else if (strcmp(argv[count], "-wsf") == 0)  {
			params.print_splits = true;
		} else if (strcmp(argv[count], "-con") == 0)  {
			params.build_consensus = true;
		} else if (strcmp(argv[count], "-nni_lh") == 0)  {
			in_pam.nni_lh_record = true;
		/*} else if (strcmp(argv[count], "-oldout") == 0)  {
			outfile_suffix = outfile_suffix_old;*/
		} else { // the paramter is a file name
			strcpy (in_pam.ali_file, argv[count]);
			if (isExistedFile (in_pam.ali_file) == 0) {
				if (isMasterProc())
					cout << "File " << in_pam.ali_file << " not found " << endl;
				Finalize(1);
			}
		}
		count++;
	}
	if (in_pam.out_prefix == NULL)
		in_pam.out_prefix = in_pam.ali_file;
	return 0;
}

/*====================================================================================
 
Interactive interface to get the user paramemters
 
====================================================================================*/
int Interface::getUserPam ( int argc, char *argv[], CommonParameters &params, InputParameters &in_pam) {

	if (isMasterProc()) {
		std::cout << "WELCOME TO IQPNNI " << PRO_VERSION << " built " << __DATE__ << " (";
#ifdef PARALLEL
#	ifdef _OPENMP // hybrid MPI and OpenMP
		#pragma omp parallel default(shared)
		{
			 omp_threads = omp_get_num_threads();
		}
		cout << "hybrid MPI/OpenMP with " << mpi_size << " processes, " << omp_threads << " threads)" << endl << endl;
#	else // pure MPI
		cout << "MPI with " << mpi_size << " processes)" << endl << endl;
#	endif
#else // not MPI
#	ifdef _OPENMP // pure OpenMP
		#pragma omp parallel default(shared)
			 omp_threads = omp_get_num_threads();
		cout << "OpenMP with " << omp_threads << " threads)" << endl << endl;
#	else
		cout << "sequential version)" << endl << endl;
#	endif
#endif

	}

	params.model = UNDEF_MODEL;
	//invariable_site = false;

	getCommandLine(argc, argv, params, in_pam);

	istream *fpin;
	
	if (in_pam.param_file != NULL) {
		if ( isExistedFile(in_pam.param_file) == 0)
			Utl::announceError("Cannot open parameter input file!");
		fpin = new ifstream(in_pam.param_file);
	} else {
		fpin = &(std::cin);
	}
	
	// check whether input file exist, otherwise prompt for different file name
	if (in_pam.text_menu)
		if (isExistedFile (in_pam.ali_file) == 0) {
			if (isMasterProc()) { // master process
				std::cout << "Please enter a file name for the sequence data: ";
				(*fpin) >> in_pam.ali_file;
				int count = 0;
				do {
					if (isExistedFile (in_pam.ali_file) == 0)
						std::cout << "file " << in_pam.ali_file << " not found, please enter alternative name or type q to quit: ";
					else
						break;
					(*fpin) >> in_pam.ali_file;
					if (strcmp(in_pam.ali_file, "q") == 0)
						return 0;
					count++;
				} while (count < 3); // only prompt for file name 3 times
			} // masterProc

			if ( isExistedFile(in_pam.ali_file) == 0 ) {
				Finalize(1);
			}

#ifdef PARALLEL
			// broadcast the file name to all slaves
			int ali_len = strlen(in_pam.ali_file)+1;
			MPI_Bcast(&ali_len, 1, MPI_INT, mpi_master_rank, MPI_COMM_WORLD);
			MPI_Bcast(in_pam.ali_file, ali_len, MPI_CHAR, mpi_master_rank, MPI_COMM_WORLD);
			//cout << mpi_myrank << ": " << ali_len << " " << aliFileName << endl;
#endif

		}

	if (isMasterProc())
		std::cout <<"Reading data..." << endl;
	int aliStatus_ = alignment.readInput (in_pam.ali_file);
	alignment.out_prefix = in_pam.out_prefix;
	//alignment.sortSeq ();

	if (aliStatus_ == 0) {
		if (isMasterProc())
			std::cout << "The alignment file " << in_pam.ali_file << " is not correct!!!";
		return EXIT_SUCCESS;
	}

	alignment.checkAli ();
	nSeq_ = alignment.getNSeq ();

	if (nSeq_ < 3) {
		if (isMasterProc())
			std::cout <<"The alignment file MUST consist of at least 3 sequences !!!";
		return 0;
	}

	int status_ = getParameter    (*fpin, params, in_pam);
	
	if (alignment.getNSeq() <= 3)
		params.nIter = 0;

	
	if (in_pam.param_file != NULL) {
		delete fpin;
	}

	return status_;
}

/*****************************************************************
Print the usage dialog  
****************************************************************/
void Interface::usage() {
	cout << "Syntax: iqpnni [OPTIONS] [Filename]" << endl;

	cout << endl << "GENERAL OPTIONS:" << endl;
	cout << "  -h, -?               print this help dialog" << endl;
	cout << "  -n <min_iterations>  make the main loop to at least min_iterations" << endl;
	cout << "  -s <stopping_rule>   either on, off or max <max_iterations>; defaut is off" << endl;
	cout << "  -u <user_tree>       read the starting tree from user_tree file" << endl;
	cout << "  -b <#num_samples>    do non-parametric bootstrap with #num_samples" << endl;
	cout << "  -bs                  same as '-b 1'" << endl;
	cout << "  -prefix <prefix_out> set prefix of output files, default is aln name" << endl;
	cout << "  -sfc                 start from scratch, don't load the check point file" << endl;
	cout << "  -ni                  don't prompt for user option" << endl;

	cout << endl << "IQP OPTIONS:" << endl;
	cout << "  -p <probability>     set the probability of deleting a sequence" << endl;
	cout << "  -k <representatives> set the number of representatives" << endl;

	cout << endl << "MODEL OPTIONS:" << endl;
	cout << "  -m <model>           set the model type for: " << endl;
	cout << "          Nucleotides: JC69, K2P, F81, HKY85, TN93, GTR" << endl;
	cout << "          Amino acids: WAG, Dayhoff, JTT, VT, mtREV, rtREV, Blosum" << endl;
	cout << "          Protein-coding DNA: GY94, YN98, NY98, CP98, CGTR, CPR" << endl;
	cout << "          Otherwise: Name of file containing user protein model" << endl;
	cout << "  -w <rate_type>       either uniform, gamma, igamma or sitespec" << endl;
	cout << "  -c <num_rate>        number of rate categories, for gamma and igamma only" << endl;

	cout << endl << "OTHER OPTIONS:" << endl;
	cout << "  -param <pam_file>    use <pam_file> for parameter input (instead of stdin)" << endl;
	cout << "  -seed <number>       set random number generator seed to <number>" << endl;
	cout << "  -wsl                 write site log-likelihood to .sitelh (PHYLIP-like)" << endl;
	cout << "  -wpl                 write pattern log-likelihood to .patlh" << endl;
	cout << "  -wsf                 write splits from bootstrap trees to .splits file" << endl;
	cout << "  -con                 turn on writing .treels, off by default" << endl;
	//cout << "  -oldout              use .iqpnni suffix for backward compatibility" << endl;
	cout << endl;
	
	//cout << "ADVANCED OPTIONS:" << endl;
	//cout << "  -cdm                 stop after computing distance matrix" << endl;
	//cout << "  -ep                  stop after estimating paramters" << endl;
	//cout << "  -nni                 stop after NNI" << endl;
	//cout << "  -opt                 test optimization step by turning off random generator" << endl;
	Finalize(1);
}

//=================================================================
//=================================================================
void Interface::writeSpecificRate (const char *rateFileName) {
	ofstream rateFile_;
	rateFile_.open (rateFileName);
	Vec<double> *ptnRateArr_;
	ptnRateArr_ = myrate.getPtnRate ();

	int nSite_ = alignment.getNSite ();
	//  double totalRate_ = 0.0;
	for (int siteNo_ = 0; siteNo_ < nSite_; siteNo_ ++) {
		int ptnNo_ = alignment.getPtn (siteNo_);
		double siteRate_ = (*ptnRateArr_)[ptnNo_];
		rateFile_ << siteNo_ + 1 << "\t";
		rateFile_.width (10);
		rateFile_.precision (4);
		rateFile_.setf (ios::left);
		//    totalRate_ += siteRate_;

		if (!atBound(MIN_PTN_RATE, siteRate_, MAX_PTN_RATE))
			rateFile_  << siteRate_ << endl;
		else 
			rateFile_  << "UNKNOWN" << endl;

	}
	//  std::cout <<"total rates: " << totalRate_ << endl;
	rateFile_.close ();
}

void checkDistance(ostream &out) {
	Mat<double> *dist = alignment.getGenDis();
	int i, j;
	int ntaxa = alignment.getNSeq();
	bool found = false;
	Vec<char> name;
	for (i = 0; i < ntaxa-1; i++) {
		bool first = true;
		for (j = i+1; j < ntaxa; j++) 
			if ((*dist)[i][j] <= 1e-6) {
				if (!found) {
					out << endl << "The following sets of sequences (each set in a line) are identical according to" << endl;
					out << "the pairwise genetic distances." << endl;
					out << "To speed up the program, one should only keep one sequence from each set" << endl;
					out << "and exclude the rest from the analysis." << endl << endl;
				}
				if (first) {
					alignment.getName(i, name);
					out << name; 
				}
				found = true;
				first = false;
				alignment.getName(j, name);
				out << ", " << name;
			}
		if (!first) out << endl;
	}
	if (found) out << endl;
}



//=====================================================================================================
//=====================================================================================================
//=====================================================================================================
void Interface::writeOut (CommonParameters params, InputParameters in_pam, UrTree &tree,
	               Vec<char> &conTree, const char *startedDate)
/*const char *aliFile,
                          UrTree &tree,
                          int nIter, double probSel, int nRep,
                          STOPPING_TYPE stoppingRule,
                          Vec<char> &conTree,
                          const char *startedDate, double progTime, int random_seed)*/ {

	//getOutFileName(SUF_, );

	std::ofstream outFile_;
	string outFileName_;
	getOutFileName(SUF_MAIN, outFileName_);

	string topoFileName_;
	getOutFileName(SUF_TREEFILE, topoFileName_);
	string rateFileName_;
	getOutFileName(SUF_SITERATE, rateFileName_);

	outFile_.open (outFileName_.c_str());



	outFile_ << "THESE RESULTS ARE FROM IQPNNI " << PRO_VERSION << " built " << __DATE__ << "!!!!" << endl;
	outFile_ << "Le Sy Vinh and Arndt von Haeseler, IQPNNI: Moving Fast Through Tree Space" << endl;
	outFile_ << "and Stopping in Time, Mol. Biol. Evol., 21(8):1565-1571. 2004" << endl << endl;
	outFile_ << "Bui Quang Minh, Le Sy Vinh, Arndt von Haeseler, and Heiko A. Schmidt, "<< endl;
	outFile_ << "pIQPNNI - parallel reconstruction of large maximum likelihood phylogenies," << endl;
	outFile_ << "Bioinformatics, 21:3794-3796. 2005" << endl << endl << endl;

	outFile_ << "Some parts of the code were taken from TREE-PUZZLE package:" << endl;
	outFile_ << "Heiko A. Schmidt, Korbinian Strimmer, Martin Vingron, and Arndt von Haeseler," << endl;
	outFile_ << "TREE-PUZZLE: Maximum likelihood phylogenetic analysis" << endl;
	outFile_ << "using quartets and parallel computing, Bioinformatics, 18:502-504, 2002" << endl << endl;

	outFile_ << "The source codes to construct the BIONJ tree were taken from BIONJ software:" << endl;
	outFile_ << "Oliver Gascuel, BIONJ: An Improved Version of the NJ Algorithm" << endl;
	outFile_ << "Based on a Simple Model of Sequence Data, Mol. Bio. Evol., 14:685-695, 1997" << endl << endl;

	outFile_ <<"================================================================================="<< endl;
	outFile_ <<"                              INPUT DATA INFORMATION                                                                         " << endl;
	outFile_ <<"================================================================================="<< endl << endl;
	outFile_ <<"General information:" << endl;
	outFile_ << "Random number generator seed: " << in_pam.random_seed << endl;
	outFile_ << "Alignment file name: " << in_pam.ali_file << endl;
	outFile_ << "Bootstrap analysis: ";
	if (params.gen_bootstrap)
		outFile_ << "Yes, " << params.gen_bootstrap << " replicates" << endl;
	else
		outFile_ << "No" << endl;
	int nSeq_ = alignment.getNSeq ();
	int seqLen_ = alignment.getSeq (0).getSize ();
	outFile_ << "Number of sequences: "  << nSeq_ << endl;
	outFile_ << "Number of sites: " << seqLen_ << endl;
	outFile_ << "Number of patterns: " << ptnlist.getNPtn () << endl;
	int outGrpNdNo_ = tree.getOutGrpNd ();
	Vec<char> outGrpSeqName_;
	alignment.getName (outGrpNdNo_, outGrpSeqName_);
	outFile_ << "Display as outgroup: ";
	OutStream::write (outGrpSeqName_, outFile_) << endl;
	outFile_ << "Number of iterations: " << params.nIter << endl;
	switch (params.stoppingRule) {
	case 0:
		outFile_ <<"Stopping rule: Yes" << endl << endl;
		break;
	case 1:
		outFile_ <<"Stopping rule: Yes, but at least " << params.nIter << " iterations" << endl << endl;
		break;
	case 2:
		outFile_ <<"Stopping rule: Yes, but at most " << params.nIter << " iterations" << endl << endl;
		break;
	default:
		outFile_ <<"Stopping rule: No" << endl << endl;
	} //end of switch

	outFile_ <<"IQP information:" << endl;
	outFile_ << "Probability of deleting one sequence: " << params.probDel << endl;
	outFile_ << "Number of representatives: " << params.nRep << endl << endl;

	outFile_ <<"Substitution process: " << endl;
	MODEL model_ = mymodel.getModelType ();

	if (mymodel.getDataType () == AMINO_ACID) {
		outFile_ << "Type of sequence input data: Amino acids" << endl;
		outFile_ << "Model of substitution: ";
		switch (model_) {
		case JTT:
			outFile_ << "JTT (Jones et al., 1992)" << endl;
			break;
		case Dayhoff:
			outFile_ << "Dayhoff (Dayhoff et al., 1978)" << endl;
			break;
		case VT:
			outFile_ << "VT (Mueller and Vingron, 2000)" << endl;
			break;

		case MtREV24:
			outFile_ << "mtREV24 (Adachi, J. and Hasegawa, M., 1996)" << endl;
			break;
		case rtREV:
			outFile_ << "rtREV (Dimmic et al., 2001)" << endl;
			break;
		case Blosum62:
			outFile_ << "Blosum62 (S. Henikoff and J. G. Henikoff., 1992)" << endl;
			break;
		case AA_USER:
			outFile_ << "User-defined" << endl;
			break;

		default:
			outFile_ << "WAG (S. Whelan and N. Goldman., 2000)" << endl;
		} //end of switch

		if (mymodel.getBaseFrqType () == ESTIMATE)
			outFile_ << endl << "Amino acid Frequencies (Estimate from the data set):" << endl;
		else
			outFile_ << endl << "Amino acid Frequencies (From the model):" << endl;

		DVec20 stateFrqArr_;
		mymodel.getStateFrq (stateFrqArr_);
		outFile_ << "Amino acid A: " << stateFrqArr_[STATE_A] << endl; //1
		outFile_ << "Amino acid C: " << stateFrqArr_[STATE_C] << endl; //2
		outFile_ << "Amino acid D: " << stateFrqArr_[STATE_D] << endl; //3
		outFile_ << "Amino acid E: " << stateFrqArr_[STATE_E] << endl; //4
		outFile_ << "Amino acid F: " << stateFrqArr_[STATE_F] << endl; //5

		outFile_ << "Amino acid G: " << stateFrqArr_[STATE_G] << endl; //6
		outFile_ << "Amino acid H: " << stateFrqArr_[STATE_H] << endl; //7
		outFile_ << "Amino acid I: " << stateFrqArr_[STATE_I] << endl; //8
		outFile_ << "Amino acid K: " << stateFrqArr_[STATE_K] << endl; //9
		outFile_ << "Amino acid L: " << stateFrqArr_[STATE_L] << endl; //10

		outFile_ << "Amino acid M: " << stateFrqArr_[STATE_M] << endl; //11
		outFile_ << "Amino acid N: " << stateFrqArr_[STATE_N] << endl; //12
		outFile_ << "Amino acid P: " << stateFrqArr_[STATE_P] << endl; //13
		outFile_ << "Amino acid Q: " << stateFrqArr_[STATE_Q] << endl; //14
		outFile_ << "Amino acid R: " << stateFrqArr_[STATE_R] << endl; //15

		outFile_ << "Amino acid S: " << stateFrqArr_[STATE_S] << endl; //16
		outFile_ << "Amino acid T: " << stateFrqArr_[STATE_T] << endl; //17
		outFile_ << "Amino acid V: " << stateFrqArr_[STATE_V] << endl; //18
		outFile_ << "Amino acid W: " << stateFrqArr_[STATE_W] << endl; //19
		outFile_ << "Amino acid Y: " << stateFrqArr_[STATE_Y] << endl << endl; //20

	} else if ( mymodel.getDataType() == NUCLEOTIDE ) {
		outFile_ << "Type of sequence input data: Nucleotides" << endl;
		if (model_ == HKY85) {
			outFile_ <<"Model of substitution: HKY85 (Hasegawa et al. 1985)" << endl;

			double tsTvRatio_ = mymodel.model_->getTsTvRatio ();
			outFile_ << "Transition/transversion ratio (0.5 for JC69): ";

			outFile_.precision (5);
			outFile_ << tsTvRatio_/2.0 << endl;
			if (mymodel.getTsTvRatioType () == ESTIMATE)
				if (tsTvRatio_ < MIN_TS_TV_RATIO + 0.1 || tsTvRatio_ > MAX_TS_TV_RATIO - 0.1)
					outFile_ <<"Warning!!! Estimated transition/transversion ratio = " << tsTvRatio_/2.0 << " is close to the boundary of admissible values" << endl;
		}  //end of HKY85


		if (model_ == TN93) {
			outFile_ <<"Model of substitution:  TN93 (Tamura-Nei 1993)" << endl;

			double tsTvRatio_ = mymodel.model_->getTsTvRatio ();
			outFile_ << "Transition/transversion ratio (0.5 for JC69): ";
			outFile_.precision (5);
			outFile_ << tsTvRatio_/2.0 << endl;
			if (mymodel.getTsTvRatioType () == ESTIMATE)
				if (tsTvRatio_ < MIN_TS_TV_RATIO + 0.1 || tsTvRatio_ > MAX_TS_TV_RATIO - 0.1)
					outFile_ <<"Warning!!! Estimated transition/transversion ratio = " << tsTvRatio_/2.0 << " is close to the boundary of admissible values" << endl;

			double pyPuRatio_ = mymodel.model_->getPyPuRatio ();
			outFile_ << "Pyrimidine/purine ratio: " << pyPuRatio_ << endl;

			if (mymodel.getPyPuRatioType () == ESTIMATE)
				if (pyPuRatio_ < MIN_PY_PU_RATIO + 0.1 || pyPuRatio_ > MAX_PY_PU_RATIO - 0.1)
					outFile_ <<"Warning!!! Estimated pyrimidine/purine ratio = " << tsTvRatio_/2.0 << " is close to the boundary of admissible values" << endl;
		} //end of YN93

		if (model_ == GTR) {// model_ == GTR
			outFile_ <<"Model of substitution: General time reversible"  <<  endl;
			double tsAG_, tsCT_, tvAC_, tvAT_, tvCG_, tvGT_;
			mymodel.model_->getGenPam (tsAG_, tsCT_, tvAC_, tvAT_, tvCG_, tvGT_);

			outFile_.precision (5);
			outFile_ <<"    1. Transversion rate from A to C: " << tvAC_ << endl;
			outFile_ <<"    2. Transition   rate from A to G: " << tsAG_ << endl;
			outFile_ <<"    3. Transversion rate from A to T: " << tvAT_ << endl;
			outFile_ <<"    4. Transversion rate from C to G: " << tvCG_ << endl;
			outFile_ <<"    5. Transition   rate from C to T: " << tsCT_ << endl;
			outFile_ <<"    6. Transversion rate from G to T: " << tvGT_ << endl;
		}

		double baseFrqA = mymodel.getStateFrq (BASE_A);
		double baseFrqC = mymodel.getStateFrq (BASE_C);
		double baseFrqG = mymodel.getStateFrq (BASE_G);
		double baseFrqT = mymodel.getStateFrq (BASE_T);


		if (mymodel.getBaseFrqType () == ESTIMATE)
			outFile_ << "Nucleotide Frequencies (estimate from the data set):" << endl;
		else
			outFile_ << "Nucleotide Frequencies (user defined):" << endl;
		outFile_.precision (4);
		outFile_ << "    1. Base A: " << baseFrqA << endl;
		outFile_ << "    2. Base C: " << baseFrqC << endl;
		outFile_ << "    3. Base G: " << baseFrqG << endl;
		outFile_ << "    4. Base T: " << baseFrqT << endl;
	} else { // codon
		outFile_ << "Type of sequence input data: Protein-coding DNA" << endl;

		outFile_ << "Codon-based model of substitution: ";
		double tsTvRatio_;
		double v_parm;
		double nsSyRatio;
		double CpG_Ratio;
		int nClass, cNo;
		double prob;
		switch (model_) {
		case Codon_GY94:
			outFile_ << "Goldman & Yang 1994 " << endl;
			tsTvRatio_ = mymodel.model_->getTsTvRatio();
			v_parm = ((CodonModel*) mymodel.model_)->getVParm();
			outFile_.precision (5);
			outFile_ << "Transition/transversion ratio: " << tsTvRatio_/2.0 << endl;
			outFile_ << "V Parameter                  : " << v_parm << endl;
			break;
		case Codon_YN98:
			outFile_ << "Yang & Nielsen 1998 " << endl;
			tsTvRatio_ = mymodel.model_->getTsTvRatio();
			nsSyRatio = ((CodonYN98*) mymodel.model_)->getNsSyRatio();
			outFile_.precision (5);
			outFile_ << "Transition/transversion ratio : " << tsTvRatio_/2.0 << endl;
			outFile_ << "Nonsynonymous/Synonymous ratio: " << nsSyRatio << endl;
			break;
		case Codon_NY98:
			outFile_ << "Nielsen & Yang 1998 " << endl;
			tsTvRatio_ = mymodel.model_->getTsTvRatio();
			outFile_.precision (5);
			outFile_ << "Transition/transversion ratio : " << tsTvRatio_/2.0 << endl;
			outFile_ << "Nonsynonymous/Synonymous ratio of " << endl;
			nClass = myrate.getNRate();
			for (cNo = 0; cNo < nClass; cNo++) {
				nsSyRatio = ((CodonNY98*) mymodel.model_)->getNsSyRatio(cNo);
				outFile_ << "class " << cNo+1 << ": omega (w) = " << nsSyRatio;
				prob = mymodel.getClassProb(cNo);
				outFile_ << " with propotion (p) = " << prob << endl;
			}
			break;
		case Codon_Pedersen98:
			outFile_ << "CpG depression (Pedersen et al. 1998) " << endl;
			tsTvRatio_ = mymodel.model_->getTsTvRatio();
			nsSyRatio = ((CodonYN98*) mymodel.model_)->getNsSyRatio();
			CpG_Ratio = ((CodonPedersen98*) mymodel.model_)->getCpG_Ratio();
			outFile_.precision (5);
			outFile_ << "Transition/transversion ratio : " << tsTvRatio_/2.0 << endl;
			outFile_ << "Nonsynonymous/Synonymous ratio: " << nsSyRatio << endl;
			outFile_ << "CpG depression ratio          : " << CpG_Ratio << endl;
			break;
		case Codon_GTR:
			outFile_ << "GTR " << endl;
			nsSyRatio = ((CodonYN98*) mymodel.model_)->getNsSyRatio();
			double tsAG_, tsCT_, tvAC_, tvAT_, tvCG_, tvGT_;
			mymodel.model_->getGenPam (tsAG_, tsCT_, tvAC_, tvAT_, tvCG_, tvGT_);
			outFile_.precision (5);
			outFile_ <<"    1. Transversion rate from A to C: " << tvAC_ << endl;
			outFile_ <<"    2. Transition   rate from A to G: " << tsAG_ << endl;
			outFile_ <<"    3. Transversion rate from A to T: " << tvAT_ << endl;
			outFile_ <<"    4. Transversion rate from C to G: " << tvCG_ << endl;
			outFile_ <<"    5. Transition   rate from C to T: " << tsCT_ << endl;
			outFile_ <<"    6. Transversion rate from G to T: " << tvGT_ << endl;
			outFile_ << "Nonsynonymous/Synonymous ratio: " << nsSyRatio << endl;
			break;
		default: break;

		} //end of switch

		outFile_ << "Codon frequencies: ";
		for (int cdon = 0; cdon < NUM_CODON; cdon ++) {
			if (cdon % 4 == 0)
				outFile_ << endl;
			outFile_.precision(8);
			outFile_ << mymodel.stateFrqArr_[cdon] << " ";
		}
		outFile_ << endl;
	}//end of writing model infomration


	outFile_ << endl << "Rate Heterogeneity: ";

	RATE_TYPE rateType_;
	rateType_ = myrate.getType ();
	if (rateType_ == UNIFORM)
		outFile_ << "Uniform substitution rate" << endl;
	else
		if (rateType_ == SITE_SPECIFIC) {
			outFile_ << "Site-specific substitution rates (see " << rateFileName_ << ")" << endl;
			writeSpecificRate (rateFileName_.c_str());
		} else {
			int nRate_ = myrate.getNRate ();
			double gammaShape_ = myrate.getGammaShape ();

			outFile_ << "Gamma distributed rates" << endl << "   (see " << rateFileName_ << " for empirical Bayesian estimates)" << endl;
			outFile_ << "Gamma distribution parameter alpha: ";

			outFile_.precision (5);
			outFile_ << gammaShape_ << endl;
			outFile_ << "Number of Gamma rate categories:  " << nRate_ /*- myrate.use_invar_site*/ << endl;
			outFile_ << "Proportion of invariable sites: " << myrate.getInvarRatio() << endl;
			outFile_ << endl << "Rates and their respective probabilities: " << endl;
			outFile_ << "Category   Rate   Probability" << endl;
			if (myrate.prob_invar_site > 0.0) {
				outFile_ << "  0   ";
				outFile_.width(10);
				outFile_.precision(4);
				outFile_ << right << 0 << "   " << myrate.prob_invar_site << endl;
			}
			for (int rateNo = 0; rateNo < nRate_; rateNo++) {
				outFile_ << "  " << rateNo+1 << "   ";
				outFile_.width(10);
				outFile_.precision(4);
				outFile_ << right << myrate.getRate(rateNo) << "   " << myrate.getProb() << endl;
			}
		}

	checkDistance(outFile_);

	outFile_ <<"================================================================================="<< endl;
	outFile_ <<"                              RESULTS INFORMATION                                                                         " << endl;
	outFile_ <<"================================================================================="<< endl << endl;


	outFile_.precision (10);
	outFile_ << "Unconstrained Log-likelihood: " << ptnlist.computeUnconstrainedLogL() << endl << endl;
	outFile_ << "Log likelihood: " << tree.getLogLi () << endl;

	outFile_ << "The tree with highest likelihood: " << endl;
	tree.writeNewickForm (outFile_);

	outFile_ << endl << "The illustrated picture from the reconstructed tree with highest likelihood:" << endl;
	outFile_ << "Note: it is an unrooted tree!!!" << endl;

	Mat<char> image_;
	tree.createImage (BR_LEN, outGrpNdNo_, image_);
	OutStream::write (image_, outFile_);
	OutStream::write (tree.getBr (), outFile_);

	if (conTree.getSize() != 0) {
		outFile_ << endl << "The majority consensus tree from all intermediate trees:" << endl;
		OutStream::write (conTree, outFile_) << endl << endl;
	}

	if (mymodel.getDataType() == CODON && (mymodel.getModelType() == Codon_NY98)) {
		if (((CodonNY98*)mymodel.model_)->paramAtBoundary()) {
			outFile_ << endl 
				<< "WARNING: The following positive selection may be instably inferred due to" << endl 
				<< "         numerical problem. It is recommended to rerun the program!" << endl << endl;
			cout << endl << separator_str << endl <<
			"WARNING: Estimated parameters are at boundary! You should run program again" << endl <<
			separator_str << endl;
		}
		opt_urtree.liBrArr_[opt_urtree.outGrpBrNo_].empiricalBayesAnalysis(outFile_);
	}


	if (myrate.getType() == GAMMA) {
		cout << "Estimating site rates by empirical Bayesian..." << endl;
		//outFile_ << "Site-specific rates by empirical Bayesian estimate: " << endl;
		opt_urtree.liBrArr_[opt_urtree.outGrpBrNo_].empiricalBayesGammaRate(outFile_);
		writeSpecificRate(rateFileName_.c_str());
	}


	outFile_ << endl << "Time information:" << endl;
	outFile_ << "Date and time: " << startedDate;
	int hour_ = static_cast<int> (params.progTime /3600.0);
	int min_ =  static_cast<int> ( (params.progTime - hour_ * 3600)/60.0);
	int sec_ =  static_cast<int> ( params.progTime - hour_ * 3600 - min_ * 60);

	outFile_ << "Runtime: " << hour_ << "h:" << min_ << "m:" << sec_ << "s" << endl;
	outFile_.close ();

	tree.writeTop(topoFileName_.c_str(), 0);

	if (!params.gen_bootstrap) {
		std::cout << "The results were written to following files: " << endl;
		std::cout << "   1. " << outFileName_ << endl;
		std::cout << "   2. " <<  topoFileName_ << endl;
		if (myrate.getType () == SITE_SPECIFIC || myrate.getType() == GAMMA)
			std::cout << "   3. " << rateFileName_ << endl;
	}

}
