/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
/**
 * Moves an item one index in an array to another.
 * @param array Array in which to move the item.
 * @param fromIndex Starting index of the item.
 * @param toIndex Index to which the item should be moved.
 */
export function moveItemInArray(array, fromIndex, toIndex) {
    const from = clamp(fromIndex, array.length - 1);
    const to = clamp(toIndex, array.length - 1);
    if (from === to) {
        return;
    }
    const target = array[from];
    const delta = to < from ? -1 : 1;
    for (let i = from; i !== to; i += delta) {
        array[i] = array[i + delta];
    }
    array[to] = target;
}
/**
 * Moves an item from one array to another.
 * @param currentArray Array from which to transfer the item.
 * @param targetArray Array into which to put the item.
 * @param currentIndex Index of the item in its current array.
 * @param targetIndex Index at which to insert the item.
 */
export function transferArrayItem(currentArray, targetArray, currentIndex, targetIndex) {
    const from = clamp(currentIndex, currentArray.length - 1);
    const to = clamp(targetIndex, targetArray.length);
    if (currentArray.length) {
        targetArray.splice(to, 0, currentArray.splice(from, 1)[0]);
    }
}
/**
 * Copies an item from one array to another, leaving it in its
 * original position in current array.
 * @param currentArray Array from which to copy the item.
 * @param targetArray Array into which is copy the item.
 * @param currentIndex Index of the item in its current array.
 * @param targetIndex Index at which to insert the item.
 *
 */
export function copyArrayItem(currentArray, targetArray, currentIndex, targetIndex) {
    const to = clamp(targetIndex, targetArray.length);
    if (currentArray.length) {
        targetArray.splice(to, 0, currentArray[currentIndex]);
    }
}
/** Clamps a number between zero and a maximum. */
function clamp(value, max) {
    return Math.max(0, Math.min(max, value));
}
//# sourceMappingURL=data:application/json;base64,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