// CD_Ctrl.cc for BBCD - a CD player for X11 / BlackBox
// Copyright (c) 2002 Bertrand Duret <bertrand.duret at libertysurf.fr>
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#include "CD_Ctrl.hh"
//#include <linux/cdrom.h>
#include <cdaudio.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

using namespace std;

CD_Controler::CD_Controler(const std::string & deviceName_)
  : deviceName(deviceName_),
    cd_fd(-1)
{
#ifdef DEBUG
  std::cerr<<"*** New CD_Controler built... \n";
  if(getCdFd() == -1) {
    std::cerr<<"*** Device not open !"<<std::endl;
  } else {
    std::cerr<<"*** Device open\n";
  }
  if(! isActive()) closeCdFd();
#endif
}



CD_Controler::~CD_Controler() {
  closeCdFd();
}



bool CD_Controler::isDiscPresent() {
  struct disc_info di;

  cd_stat(getCdFd(), &di);
  if(! di.disc_present) closeCdFd();

  return di.disc_present;
}



CD_Controler::AudioStatus CD_Controler::getAudioStatus() {
  struct disc_info di;
  AudioStatus retStatus(NoStatus);

  cd_stat(getCdFd(), &di);
  if(di.disc_present) {
    switch(di.disc_mode) {
    case CDAUDIO_INVALID:
      retStatus = Invalid;
      break;
    case CDAUDIO_PLAYING:
      retStatus = Play;
      break;
    case CDAUDIO_PAUSED:
      retStatus = Paused;
      break;
    case CDAUDIO_COMPLETED:
      retStatus = Completed;
      break;
    case CDAUDIO_ERROR:
      retStatus = Error;
      break;
    case CDAUDIO_NOSTATUS:
      retStatus = NoStatus;
      break;
    default:
#ifdef DEBUG
      std::cerr<<"*** Unexpected value for disc_mode from cd_stat in "
        "CD_Controler::getAudioStatus !"<<endl;
#endif
      break;
    }
  }

  if((retStatus != Play) && (retStatus != Paused)) closeCdFd();
  return retStatus;
}



unsigned int CD_Controler::play() {
  struct disc_info di;

  cd_stat(getCdFd(), &di);
  if(di.disc_present) {
    if(di.disc_mode == CDAUDIO_PLAYING) {
      return di.disc_current_track;
    }
    if(di.disc_mode == CDAUDIO_PAUSED) {
      cd_resume(getCdFd());
      return di.disc_current_track;
    }
    if(di.disc_mode == CDAUDIO_COMPLETED) {
      cd_play(getCdFd(), di.disc_first_track);
      return di.disc_first_track;
    }
  }

  closeCdFd();
  return 0;
}



unsigned int CD_Controler::playNext() {
  struct disc_info di;

  cd_stat(getCdFd(), &di);
  if((di.disc_present) && (di.disc_mode == CDAUDIO_PLAYING)) {
    unsigned int current_track(di.disc_current_track);
    while((current_track++ < di.disc_total_tracks) &&
      (di.disc_track[current_track].track_type != CDAUDIO_TRACK_AUDIO));
    if(current_track <= di.disc_total_tracks) {
      cd_play(getCdFd(), current_track);
      return current_track;
    } else {
      return di.disc_current_track;
    }
  }

  closeCdFd();
  return 0;
}



unsigned int CD_Controler::playPrevious() {
  struct disc_info di;

  cd_stat(getCdFd(), &di);
  if((di.disc_present) && (di.disc_mode == CDAUDIO_PLAYING)) {
    unsigned int current_track(di.disc_current_track);
    while((current_track-- > 1) &&
      (di.disc_track[current_track].track_type != CDAUDIO_TRACK_AUDIO));
    if(current_track >= 1) {
      cd_play(getCdFd(), current_track);
      return current_track;
    } else {
      return di.disc_current_track;
    }
  }

  closeCdFd();
  return 0;
}



void CD_Controler::pause() {
  struct disc_info di;

  cd_stat(getCdFd(), &di);
  if(di.disc_present) {
    if(di.disc_mode == CDAUDIO_PLAYING) {
      cd_pause(getCdFd());
      return;
    } else if(di.disc_mode == CDAUDIO_PAUSED) {
      cd_resume(getCdFd());
      return;
    }
  }

  closeCdFd();
}



void CD_Controler::stop() {
  cd_stop(getCdFd());
  closeCdFd();
}



int CD_Controler::getCdFd() {
  if(cd_fd == -1) {
#ifdef DEBUG
    std::cerr<<"*** Opening CDROM device..."<<std::endl;
#endif
    cd_fd = cd_init_device(const_cast<char *>(deviceName.c_str()));
  }
  return cd_fd;
}



void CD_Controler::closeCdFd() {
  if(cd_fd != -1) {
#ifdef DEBUG
    std::cerr<<"*** Closing CDROM device..."<<std::endl;
#endif
    close(cd_fd);
    cd_fd = -1;
  }
}



