/*
 * ReplaceFilter.java
 *
 * Brazil project web application toolkit,
 * export version: 2.1 
 * Copyright (c) 1999-2004 Sun Microsystems, Inc.
 *
 * Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License Version 
 * 1.0 (the "License"). You may not use this file except in compliance with 
 * the License. A copy of the License is included as the file "license.terms",
 * and also available at http://www.sun.com/
 * 
 * The Original Code is from:
 *    Brazil project web application toolkit release 2.1.
 * The Initial Developer of the Original Code is: suhler.
 * Portions created by suhler are Copyright (C) Sun Microsystems, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s): suhler.
 *
 * Version:  2.2
 * Created by suhler on 99/09/01
 * Last modified by suhler on 04/11/30 15:19:39
 */

package sunlabs.brazil.filter;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import sunlabs.brazil.handler.ResourceHandler;
import sunlabs.brazil.server.Request;
import sunlabs.brazil.server.Server;
import sunlabs.brazil.util.http.MimeHeaders;

/**
 * Filter to replace current content with a static form, or template.
 * This should be called
 * the TemplateFilter, but that name's already taken.  The content is
 * replaced by the template lock-stock-and-barrel.  Typically, an upstream 
 * filter has extracted the relevent parts of the content, and a down-stream
 * filter will combine it with the template.
 * The filename to use for the template
 * is computed at each request, so it may be modified dynamically.
 *
 * The following server properties are used:
 * <dl class=props>
 * <dt>type	<dd>    Text subtype of content to filter.  Defaults to "html"
 * <dt>debug	<dd>    If set, the template is re-read each time.  Otherwise
 *			a cached copy is used.
 * <dt>fileName	<dd>    Name of the file to use as the form or template.
 *			The file is searched for as a <i>Resource</i> if not
 *			found in the filesystem.
 * <dt>root	<dd>	The document root used to find the template file.
 *			If not found, "root" with no prefix is used instead.
 * </dl>
 *
 * @author		Stephen Uhler
 * @version		2.2
 */

public class ReplaceFilter implements Filter {
    static final String FILE = "fileName";
    String type;	// the text sub-type to replace (defaults to html)
    String prefix;	// the properties prefix
    byte[] data;	// cache of the last template
    String path;	// cache of last file name;
    boolean debug;	// don't cache file

    public boolean
    init(Server server, String prefix) {
	type = server.props.getProperty(prefix + "type", "html");
	debug = server.props.getProperty(prefix + "debug") != null;
	this.prefix = prefix;
	path = null;
	return true;
    }

    /**
     * This is the request object before the content was fetched
     */

    public boolean respond(Request request) {
	return false;
    }

    /**
     * Only replace text documents
     */

    public boolean
    shouldFilter(Request request, MimeHeaders headers) {
	String type = headers.get("content-type");
	return (type != null && type.startsWith("text/"));
    }

    /**
     * Grab the template file name, Read in the file, and
     * deliver it as content.
     */

    public byte[]
    filter(Request request, MimeHeaders headers, byte[] content) {
	if (!(headers.get("content-type")).startsWith("text/" + type)) {
	    return content;
	}
	String name = request.props.getProperty(prefix + "fileName");
	if (name == null) {
	    request.log(Server.LOG_WARNING,"No fileName property found for " +
		prefix);
	    return content;
	}

	if (!debug && path != null && name.equals(path)) {
	    request.log(Server.LOG_DIAGNOSTIC, "using cached content for: "
		    + name);
	    return data;
	}

	try {
	    data =  ResourceHandler.getResourceBytes(request.props,
		prefix, name);
	    return data;
	} catch (IOException e) {
	    request.log(Server.LOG_WARNING, prefix + " can't get " + name +
		    ": " + e);
	    return content;
	}
    }
}
