/* $Id: op_log_pcap.c,v 1.10 2004/03/16 04:18:20 andrewbaker Exp $ */
/*
** Copyright (C) 2001-2002 Andrew R. Baker <andrewb@snort.org>
**
** This program is distributed under the terms of version 1.0 of the 
** Q Public License.  See LICENSE.QPL for further details.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
*/


/*  I N C L U D E S  *****************************************************/
#include <string.h>
#include <errno.h>
#include <stdlib.h>

#include "plugbase.h"
#include "op_plugbase.h"
#include "mstring.h"
#include "util.h"
#include "barnyard.h"
#include "time.h"
#include "input-plugins/dp_log.h"
#include "barnyard.h"

/*  D A T A   S T R U C T U R E S  **************************************/
typedef struct _OpLogPcap_Data 
{
    char *filename;
    char *filepath;
    FILE *file;
} OpLogPcap_Data;

/*  P R O T O T Y P E S  ************************************************/
/* Output plug-in APIs */
static int OpLogPcap_Setup(OutputPlugin *outputPlugin, char *args);
static int OpLogPcap_Exit(OutputPlugin *);
static int OpLogPcap_Start(OutputPlugin *, void *);
static int OpLogPcap_Stop(OutputPlugin *);
static int OpLogPcap(void *, void *);
static int OpLogPcap_LogConfig(OutputPlugin *);

/* Internal functions */
static OpLogPcap_Data *OpLogPcap_ParseArgs(char *);
static FILE *OpenPcapFile(char *filename);

/* Global Variables */
/* make this a global so that i do not use stack space */
static char filepath[FILEPATH_BUFSIZE];

/* init routine makes this processor available for dataprocessor directives */
void OpLogPcap_Init()
{
    OutputPlugin *outputPlugin;
    
    outputPlugin = RegisterOutputPlugin("log_pcap", "log");
    
    outputPlugin->setupFunc = OpLogPcap_Setup;
    outputPlugin->exitFunc = OpLogPcap_Exit;
    outputPlugin->startFunc = OpLogPcap_Start;
    outputPlugin->stopFunc = OpLogPcap_Stop;
    outputPlugin->outputFunc = OpLogPcap;
    outputPlugin->logConfigFunc = OpLogPcap_LogConfig;
}

static int OpLogPcap_LogConfig(OutputPlugin *outputPlugin)
{
    OpLogPcap_Data *data = NULL;
    
    if(!outputPlugin || !outputPlugin->data)
        return -1;

    data = (OpLogPcap_Data *)outputPlugin->data;

    LogMessage("OpLogPcap configured\n");
    LogMessage("  Filename: %s\n", data->filename);

    return 0;
}

/* link the output processor functions to an output function node */
int OpLogPcap_Setup(OutputPlugin *outputPlugin, char *args)
{
    /* setup the run time context for this output plugin */
    outputPlugin->data = OpLogPcap_ParseArgs(args);

    return 0;
}

/* this gets called on exit */
int OpLogPcap_Exit(OutputPlugin *outputPlugin)
{

    OpLogPcap_Data *data = (OpLogPcap_Data *)outputPlugin->data;
    if(data != NULL && data->filename != NULL)
        free(data->filename);

    return 0;
}

typedef struct _PcapFileHeader
{
    u_int32_t magic;
    u_int16_t version_major;
    u_int16_t version_minor;
    u_int32_t timezone;
    u_int32_t sigfigs;
    u_int32_t snaplen;
    u_int32_t linktype;
} PcapFileHeader;


int OpLogPcap_Start(OutputPlugin *outputPlugin, void *spool_header)
{
    PcapFileHeader pcap_header;
    UnifiedLogFileHeader *file_header = (UnifiedLogFileHeader *)spool_header;
    OpLogPcap_Data *data = (OpLogPcap_Data *)outputPlugin->data;

    if(data == NULL)
        FatalError("ERROR: Unable to find context for LogPcap startup!\n");
    
    if(pv.verbose >= 2)
        OpLogPcap_LogConfig(outputPlugin);
    
    data->filepath = ProcessFileOption(data->filename);
    
    data->file = OpenPcapFile(data->filepath);
                
    /* Write the header here */
    pcap_header.magic = 0xa1b2c3d4;
    pcap_header.version_major = 2;
    pcap_header.version_minor = 4;
    pcap_header.timezone = file_header->timezone;
    pcap_header.sigfigs = file_header->sigfigs;
    pcap_header.snaplen = file_header->snaplen;
    pcap_header.linktype = file_header->linktype;
    fwrite(&pcap_header, sizeof(pcap_header), 1, data->file);
    fflush(data->file);
#ifdef DEBUG
    LogMessage("LogPcap output plugin started\n");
#endif
    return 0;
}


int OpLogPcap_Stop(OutputPlugin *outputPlugin)
{
    OpLogPcap_Data *data = (OpLogPcap_Data *)outputPlugin->data;
    fclose(data->file);
    data->file = NULL;
    if(data->filepath)
        free(data->filepath);
    data->filepath = NULL;
    return 0;
}

/* 
 * this is the primary output function for the plugin, this is what gets called
 * for every record read 
 */
int OpLogPcap(void *context, void *data)
{
    UnifiedLogRecord *record = (UnifiedLogRecord *)data; 
    OpLogPcap_Data *op_data = (OpLogPcap_Data *)context;

    fwrite(&record->log.pkth, sizeof(record->log.pkth), 1, op_data->file);
    fwrite(record->pkt + 2, sizeof(u_int8_t), record->log.pkth.caplen, 
            op_data->file);
    fflush(op_data->file);
    return 0;
}


/* initialize the output processor for this particular instantiation */
OpLogPcap_Data *OpLogPcap_ParseArgs(char *args)
{
    char **toks;
    int num_toks;
    OpLogPcap_Data *data;

    data = (OpLogPcap_Data *)SafeAlloc(sizeof(OpLogPcap_Data));

    if(args == NULL)
    {
        data->filename = strdup("barnyard.pcap");

    }
    else
    {
        toks = mSplit(args, " ", 2, &num_toks, 0);
        data->filename = strdup(toks[0]);
        FreeToks(toks, num_toks);
    }

    return data;
}


/* 
 * this function gets called at start time, you should open any output files
 * or establish DB connections, etc, here
 */

FILE *OpenPcapFile(char *filename)
{
    time_t curr_time;
    struct tm *loc_time;
    FILE *rval;
    int value;
    char timebuf[20];

    
    curr_time = time(NULL);
    loc_time = localtime(&curr_time);
    strftime(timebuf, 20, "%Y-%m-%d@%H-%M-%S", loc_time);
    timebuf[19] = '\0';
                   
    if(filename[0] == '/')
        value = snprintf(filepath, FILEPATH_BUFSIZE, "%s.%s", 
               filename, timebuf);
    else
        value = snprintf(filepath, FILEPATH_BUFSIZE, "%s/%s.%s",
                pv.log_dir, filename, timebuf);

    if(value == -1)
        FatalError("ERROR: pcap output filepath is too long, aborting\n");

#ifdef DEBUG
    LogMessage("Opening: %s\n", filepath);
#endif
       
    if((rval = fopen(filepath, "w")) == NULL)
        FatalError("OpenPcapFile(%s) failed: %s\n", filepath, strerror(errno));
    
    return rval;
}
