/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.time;

import java.time.DateTimeException;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.temporal.ChronoField;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import org.elasticsearch.common.time.CharSubSequence;
import org.elasticsearch.common.time.DateTime;
import org.elasticsearch.common.time.DecimalSeparator;
import org.elasticsearch.common.time.ParseResult;
import org.elasticsearch.common.time.TimezonePresence;
import org.elasticsearch.core.Nullable;

class Iso8601Parser {
    private static final Set<ChronoField> VALID_SPECIFIED_FIELDS = EnumSet.of(ChronoField.YEAR, new ChronoField[]{ChronoField.MONTH_OF_YEAR, ChronoField.DAY_OF_MONTH, ChronoField.HOUR_OF_DAY, ChronoField.MINUTE_OF_HOUR, ChronoField.SECOND_OF_MINUTE, ChronoField.NANO_OF_SECOND});
    private static final Set<ChronoField> VALID_DEFAULT_FIELDS = EnumSet.of(ChronoField.MONTH_OF_YEAR, new ChronoField[]{ChronoField.DAY_OF_MONTH, ChronoField.HOUR_OF_DAY, ChronoField.MINUTE_OF_HOUR, ChronoField.SECOND_OF_MINUTE, ChronoField.NANO_OF_SECOND});
    private final Set<ChronoField> mandatoryFields;
    private final boolean optionalTime;
    @Nullable
    private final ChronoField maxAllowedField;
    private final DecimalSeparator decimalSeparator;
    private final TimezonePresence timezonePresence;
    private final Map<ChronoField, Integer> defaults;
    private static final int[] NANO_MULTIPLICANDS = new int[]{1, 10, 100, 1000, 10000, 100000, 1000000, 10000000, 100000000};
    private final ThreadLocal<ZoneOffset> lastOffset = ThreadLocal.withInitial(() -> ZoneOffset.UTC);
    private static final char ZERO = '0';
    private static final char NINE = '9';

    Iso8601Parser(Set<ChronoField> mandatoryFields, boolean optionalTime, @Nullable ChronoField maxAllowedField, DecimalSeparator decimalSeparator, TimezonePresence timezonePresence, Map<ChronoField, Integer> defaults) {
        Iso8601Parser.checkChronoFields(mandatoryFields, VALID_SPECIFIED_FIELDS);
        if (maxAllowedField != null && !VALID_SPECIFIED_FIELDS.contains(maxAllowedField)) {
            throw new IllegalArgumentException("Invalid chrono field specified " + String.valueOf(maxAllowedField));
        }
        Iso8601Parser.checkChronoFields(defaults.keySet(), VALID_DEFAULT_FIELDS);
        this.mandatoryFields = EnumSet.of(ChronoField.YEAR);
        this.mandatoryFields.addAll(mandatoryFields);
        this.optionalTime = optionalTime;
        this.maxAllowedField = maxAllowedField;
        this.decimalSeparator = Objects.requireNonNull(decimalSeparator);
        this.timezonePresence = Objects.requireNonNull(timezonePresence);
        this.defaults = defaults.isEmpty() ? Map.of() : new EnumMap<ChronoField, Integer>(defaults);
    }

    private static void checkChronoFields(Set<ChronoField> fields, Set<ChronoField> validFields) {
        if (fields.isEmpty()) {
            return;
        }
        fields = EnumSet.copyOf(fields);
        fields.removeAll(validFields);
        if (!fields.isEmpty()) {
            throw new IllegalArgumentException("Invalid chrono fields specified " + String.valueOf(fields));
        }
    }

    boolean optionalTime() {
        return this.optionalTime;
    }

    Set<ChronoField> mandatoryFields() {
        return this.mandatoryFields;
    }

    ChronoField maxAllowedField() {
        return this.maxAllowedField;
    }

    DecimalSeparator decimalSeparator() {
        return this.decimalSeparator;
    }

    TimezonePresence timezonePresence() {
        return this.timezonePresence;
    }

    private boolean isOptional(ChronoField field) {
        return !this.mandatoryFields.contains(field);
    }

    private Integer defaultZero(ChronoField field) {
        return this.defaults.getOrDefault(field, 0);
    }

    ParseResult tryParse(CharSequence str, @Nullable ZoneId defaultTimezone) {
        if (str.charAt(0) == '-') {
            ParseResult result = this.parse(new CharSubSequence(str, 1, str.length()), defaultTimezone);
            if (result.errorIndex() >= 0) {
                return ParseResult.error(result.errorIndex() + 1);
            }
            DateTime dt = (DateTime)result.result();
            return new ParseResult(new DateTime(-dt.years(), dt.months(), dt.days(), dt.hours(), dt.minutes(), dt.seconds(), dt.nanos(), dt.zoneId(), dt.offset()));
        }
        return this.parse(str, defaultTimezone);
    }

    private ParseResult parse(CharSequence str, @Nullable ZoneId defaultTimezone) {
        char c;
        int pos;
        int len = str.length();
        Integer years = Iso8601Parser.parseInt(str, 0, 4);
        if (years == null) {
            return ParseResult.error(0);
        }
        if (len == 4) {
            return this.isOptional(ChronoField.MONTH_OF_YEAR) ? new ParseResult(Iso8601Parser.withZoneOffset(years, this.defaults.get(ChronoField.MONTH_OF_YEAR), this.defaults.get(ChronoField.DAY_OF_MONTH), this.defaults.get(ChronoField.HOUR_OF_DAY), this.defaults.get(ChronoField.MINUTE_OF_HOUR), this.defaults.get(ChronoField.SECOND_OF_MINUTE), this.defaults.get(ChronoField.NANO_OF_SECOND), defaultTimezone)) : ParseResult.error(4);
        }
        if (str.charAt(4) != '-' || this.maxAllowedField == ChronoField.YEAR) {
            return ParseResult.error(4);
        }
        Integer months = Iso8601Parser.parseInt(str, 5, 7);
        if (months == null || months > 12) {
            return ParseResult.error(5);
        }
        if (len == 7) {
            return this.isOptional(ChronoField.DAY_OF_MONTH) ? new ParseResult(Iso8601Parser.withZoneOffset(years, months, this.defaults.get(ChronoField.DAY_OF_MONTH), this.defaults.get(ChronoField.HOUR_OF_DAY), this.defaults.get(ChronoField.MINUTE_OF_HOUR), this.defaults.get(ChronoField.SECOND_OF_MINUTE), this.defaults.get(ChronoField.NANO_OF_SECOND), defaultTimezone)) : ParseResult.error(7);
        }
        if (str.charAt(7) != '-' || this.maxAllowedField == ChronoField.MONTH_OF_YEAR) {
            return ParseResult.error(7);
        }
        Integer days = Iso8601Parser.parseInt(str, 8, 10);
        if (days == null || days > 31) {
            return ParseResult.error(8);
        }
        if (len == 10) {
            return this.optionalTime || this.isOptional(ChronoField.HOUR_OF_DAY) ? new ParseResult(Iso8601Parser.withZoneOffset(years, months, days, this.defaults.get(ChronoField.HOUR_OF_DAY), this.defaults.get(ChronoField.MINUTE_OF_HOUR), this.defaults.get(ChronoField.SECOND_OF_MINUTE), this.defaults.get(ChronoField.NANO_OF_SECOND), defaultTimezone)) : ParseResult.error(10);
        }
        if (str.charAt(10) != 'T' || this.maxAllowedField == ChronoField.DAY_OF_MONTH) {
            return ParseResult.error(10);
        }
        if (len == 11) {
            return this.isOptional(ChronoField.HOUR_OF_DAY) ? new ParseResult(Iso8601Parser.withZoneOffset(years, months, days, this.defaults.get(ChronoField.HOUR_OF_DAY), this.defaults.get(ChronoField.MINUTE_OF_HOUR), this.defaults.get(ChronoField.SECOND_OF_MINUTE), this.defaults.get(ChronoField.NANO_OF_SECOND), defaultTimezone)) : ParseResult.error(11);
        }
        Integer hours = Iso8601Parser.parseInt(str, 11, 13);
        if (hours == null || hours > 23) {
            return ParseResult.error(11);
        }
        if (len == 13) {
            return this.isOptional(ChronoField.MINUTE_OF_HOUR) && this.timezonePresence != TimezonePresence.MANDATORY ? new ParseResult(Iso8601Parser.withZoneOffset(years, months, days, hours, this.defaultZero(ChronoField.MINUTE_OF_HOUR), this.defaultZero(ChronoField.SECOND_OF_MINUTE), this.defaultZero(ChronoField.NANO_OF_SECOND), defaultTimezone)) : ParseResult.error(13);
        }
        if (Iso8601Parser.isZoneId(str, 13)) {
            ZoneId timezone = this.parseZoneId(str, 13);
            return timezone != null && this.isOptional(ChronoField.MINUTE_OF_HOUR) ? new ParseResult(Iso8601Parser.withZoneOffset(years, months, days, hours, this.defaultZero(ChronoField.MINUTE_OF_HOUR), this.defaultZero(ChronoField.SECOND_OF_MINUTE), this.defaultZero(ChronoField.NANO_OF_SECOND), timezone)) : ParseResult.error(13);
        }
        if (str.charAt(13) != ':' || this.maxAllowedField == ChronoField.HOUR_OF_DAY) {
            return ParseResult.error(13);
        }
        Integer minutes = Iso8601Parser.parseInt(str, 14, 16);
        if (minutes == null || minutes > 59) {
            return ParseResult.error(14);
        }
        if (len == 16) {
            return this.isOptional(ChronoField.SECOND_OF_MINUTE) && this.timezonePresence != TimezonePresence.MANDATORY ? new ParseResult(Iso8601Parser.withZoneOffset(years, months, days, hours, minutes, this.defaultZero(ChronoField.SECOND_OF_MINUTE), this.defaultZero(ChronoField.NANO_OF_SECOND), defaultTimezone)) : ParseResult.error(16);
        }
        if (Iso8601Parser.isZoneId(str, 16)) {
            ZoneId timezone = this.parseZoneId(str, 16);
            return timezone != null && this.isOptional(ChronoField.SECOND_OF_MINUTE) ? new ParseResult(Iso8601Parser.withZoneOffset(years, months, days, hours, minutes, this.defaultZero(ChronoField.SECOND_OF_MINUTE), this.defaultZero(ChronoField.NANO_OF_SECOND), timezone)) : ParseResult.error(16);
        }
        if (str.charAt(16) != ':' || this.maxAllowedField == ChronoField.MINUTE_OF_HOUR) {
            return ParseResult.error(16);
        }
        Integer seconds = Iso8601Parser.parseInt(str, 17, 19);
        if (seconds == null || seconds > 59) {
            return ParseResult.error(17);
        }
        if (len == 19) {
            return this.isOptional(ChronoField.NANO_OF_SECOND) && this.timezonePresence != TimezonePresence.MANDATORY ? new ParseResult(Iso8601Parser.withZoneOffset(years, months, days, hours, minutes, seconds, this.defaultZero(ChronoField.NANO_OF_SECOND), defaultTimezone)) : ParseResult.error(19);
        }
        if (Iso8601Parser.isZoneId(str, 19)) {
            ZoneId timezone = this.parseZoneId(str, 19);
            return timezone != null ? new ParseResult(Iso8601Parser.withZoneOffset(years, months, days, hours, minutes, seconds, this.defaultZero(ChronoField.NANO_OF_SECOND), timezone)) : ParseResult.error(19);
        }
        if (!this.checkDecimalSeparator(str.charAt(19)) || this.maxAllowedField == ChronoField.SECOND_OF_MINUTE) {
            return ParseResult.error(19);
        }
        int nanos = 0;
        for (pos = 20; pos < len && pos < 29 && (c = str.charAt(pos)) >= '0' && c <= '9'; ++pos) {
            nanos = nanos * 10 + (c - 48);
        }
        if (pos == 20) {
            return ParseResult.error(20);
        }
        nanos *= NANO_MULTIPLICANDS[29 - pos];
        if (len == pos) {
            return this.timezonePresence != TimezonePresence.MANDATORY ? new ParseResult(Iso8601Parser.withZoneOffset(years, months, days, hours, minutes, seconds, nanos, defaultTimezone)) : ParseResult.error(pos);
        }
        if (Iso8601Parser.isZoneId(str, pos)) {
            ZoneId timezone = this.parseZoneId(str, pos);
            return timezone != null ? new ParseResult(Iso8601Parser.withZoneOffset(years, months, days, hours, minutes, seconds, nanos, timezone)) : ParseResult.error(pos);
        }
        return ParseResult.error(pos);
    }

    private boolean checkDecimalSeparator(char separator) {
        boolean isDot = separator == '.';
        boolean isComma = separator == ',';
        return switch (this.decimalSeparator) {
            default -> throw new MatchException(null, null);
            case DecimalSeparator.DOT -> isDot;
            case DecimalSeparator.COMMA -> isComma;
            case DecimalSeparator.BOTH -> isDot || isComma;
        };
    }

    private static boolean isZoneId(CharSequence str, int pos) {
        char c = str.charAt(pos);
        return c == '+' || c == '-' || c >= 'A' && c <= 'Z' || c >= 'a' && c <= 'z';
    }

    private ZoneId parseZoneId(CharSequence str, int pos) {
        Integer seconds;
        Integer minutes;
        boolean positive;
        if (this.timezonePresence == TimezonePresence.FORBIDDEN) {
            return null;
        }
        int len = str.length();
        char first = str.charAt(pos);
        if (first == 'Z' && len == pos + 1) {
            return ZoneOffset.UTC;
        }
        switch (first) {
            case '+': {
                positive = true;
                break;
            }
            case '-': {
                positive = false;
                break;
            }
            default: {
                try {
                    return ZoneId.of(str.subSequence(pos, str.length()).toString());
                }
                catch (DateTimeException e) {
                    return null;
                }
            }
        }
        Integer hours = Iso8601Parser.parseInt(str, ++pos, pos += 2);
        if (hours == null || hours > 23) {
            return null;
        }
        if (len == pos) {
            return this.ofHoursMinutesSeconds(hours, 0, 0, positive);
        }
        boolean hasColon = false;
        if (str.charAt(pos) == ':') {
            hasColon = true;
        }
        if ((minutes = Iso8601Parser.parseInt(str, ++pos, pos += 2)) == null || minutes > 59) {
            return null;
        }
        if (len == pos) {
            return this.ofHoursMinutesSeconds(hours, minutes, 0, positive);
        }
        if (str.charAt(pos) == ':' != hasColon) {
            return null;
        }
        if (hasColon) {
            // empty if block
        }
        if ((seconds = Iso8601Parser.parseInt(str, ++pos, pos += 2)) == null || seconds > 59) {
            return null;
        }
        if (len == pos) {
            return this.ofHoursMinutesSeconds(hours, minutes, seconds, positive);
        }
        return null;
    }

    private ZoneOffset ofHoursMinutesSeconds(int hours, int minutes, int seconds, boolean positive) {
        ZoneOffset lastOffset;
        int totalSeconds = hours * 3600 + minutes * 60 + seconds;
        if (!positive) {
            totalSeconds = -totalSeconds;
        }
        if (totalSeconds == (lastOffset = this.lastOffset.get()).getTotalSeconds()) {
            return lastOffset;
        }
        try {
            ZoneOffset offset = ZoneOffset.ofTotalSeconds(totalSeconds);
            this.lastOffset.set(lastOffset);
            return offset;
        }
        catch (DateTimeException e) {
            return null;
        }
    }

    private static DateTime withZoneOffset(int years, Integer months, Integer days, Integer hours, Integer minutes, Integer seconds, Integer nanos, ZoneId zoneId) {
        if (zoneId instanceof ZoneOffset) {
            ZoneOffset zo = (ZoneOffset)zoneId;
            return new DateTime(years, months, days, hours, minutes, seconds, nanos, zoneId, zo);
        }
        return new DateTime(years, months, days, hours, minutes, seconds, nanos, zoneId, null);
    }

    private static Integer parseInt(CharSequence str, int startInclusive, int endExclusive) {
        if (str.length() < endExclusive) {
            return null;
        }
        int result = 0;
        for (int i = startInclusive; i < endExclusive; ++i) {
            char c = str.charAt(i);
            if (c < '0' || c > '9') {
                return null;
            }
            result = result * 10 + (c - 48);
        }
        return result;
    }
}

