#!/usr/bin/perl -w
use strict;
use IO::File;

my @testfiles = glob("*.t");
my $nfailed = 0;

sub parse_testfile ( $ )
{
    my ($filename) = @_;
    my $file = IO::File->new($filename) or die "$0: could not open file: $filename: $!";

    my @tests;
    my $state = 'command'; # command or output
    my $test;

    while (<$file>) {
	unless ($state eq 'output') {
		next if /^\s*#/;
		next if /^\s*$/;
	}

	$test = {COMMANDS => [],EXITCODE => undef,OUTPUT => []} unless defined($test);

	if ($state eq 'command') {
	    if (/^=>\s*(\d+)/) {
		$test->{EXITCODE} = $1;
		$state = 'output';
		next;
	    }

	    chomp;
	    push @{$test->{COMMANDS}},$_;
	}
	elsif ($state eq 'output') {
	    if (/^--\s*$/) {
		push @tests,$test;
		undef $test;
		$state = 'command';
		next;
	    }

	    push @{$test->{OUTPUT}},$_;
	}
    }
    return @tests;
}

sub perform_test ( $ )
{
    my ($test) = @_;

    my $exitcode;
    for my $cmd (@{$test->{COMMANDS}}) {
	if ($cmd =~ /(.*)\bbinfind\b(.*)/) {
	    $cmd = $1.'../src/binfind'.$2.' >OUTPUT 2>ERROR';
	}
#print "$cmd\n";
	$exitcode = system($cmd);
	if ($exitcode & 0xff != 0) {
	    print "could not execute command: $cmd: $!\n";
	    return 0;
	}
    }
    $exitcode >>= 8;

    if ($exitcode != $test->{EXITCODE}) {
	print "wrong exitcode $exitcode, expected ".$test->{EXITCODE}."\n";
	return 0;
    }

    my $output = IO::File->new("OUTPUT") or die "$0: could not open file: OUTPUT: $!";
    my @out = @{$test->{OUTPUT}};
    while (<$output>) {
	my $line = shift @out;
	if (defined $line) {
	    if ($line ne $_) {
	        print "wrong output:\n$_\nexpected:\n$line\n";
		return 0;
	    }
	}
	else {
	    print "extra output:\n$_";
	    return 0;
	}
    }
    if (@out) {
	print "missing output:\n";
	print $_ for @out;
	return 0;
    }
    undef $output;
    return 1;
}

sub testfile ( $ )
{
    my ($filename) = @_;

    my @tests = parse_testfile($filename);
    my $n = @tests;

    my $i = 1;
    for my $t (@tests) {
	print "$filename, $i/$n...";
        if (perform_test($t)) {
	    print "ok\n";
	}
	else {
	    $nfailed++;
	}
	$i++;
    }
}

for my $t (@testfiles) {
    testfile($t);
}
exit $nfailed;

