# Getting Started

First, you need a grammar. Tahiti supports various schema language, but
let's assume you have "foo.rng" file as your grammar. Then the first
thing you should do is to run the schema compiler.

```java
    java -jar tahiti.jar foo.rng
```

Schema compiler will generate Java source codes to the current
directory. If you want it to place those files to another directory,
then you can use the `-d` option like this:

```java
    java Compiler foo.rng -d ../../src
```

Although the compiler takes every effort to extract a meaningful mapping
from the schema, often it falls short of your expectation. If that is
the case, you can control the way the compiler maps schemas to Java
classes by annotating schema files.

The annotation depends on the schema language you use. Consult the
documentation for your choice of the schema language.

* see [RELAX NG annotation](RELAX%20NG%20Bind.md)

## Using the generated classes

This section describes how you can use the genrated Java classes to do
what you want.

You should know two important operations involving the generated
classes. One is *marshalling* and the other is *unmarshalling*

## Unmarshalling

Unmarshalling is the process that reads an XML document and creates the
corresponding Java objects. The easiest way to do this is to do as
follows:

```java
    import org.example.test.Grammar;

    class Main {
      public static void main() {
        // specify URL
        MyClass o1 = Grammar.unmarshall("file://abc/def.xml");
        // specify org.xml.sax.InputSource
        MyClass o2 = Grammar.unmarshall(new InputSource(new FileInputStream(...)));
      }
```

The above method will read the document from the specified location and
returns the result.

`org.example.test.Grammar` is the name of the compiled grammar, which is
generated by the schema compiler. You should replace it with appropriate
class name.

There are severl other overloaded version of this method, which accepts
various type of inputs like `org.xml.sax.InputSource` or
`java.io.InputStream`.

Should any error happens during the process of unmarshalling, an
exception is thrown. See javadoc for details.

## Marshalling

Marshalling is the process that writes XML representation of the Java
object model. The easiest way to do this is to use the `MarshallerAPI`
class.

```java
    import org.example.test.Grammar;
    import com.sun.tahiti.runtime.sm.MarshallerAPI;

    class Main {
      public static void main() {
        // create an object model.
        MyClass o1 = new MyClass();

        // then do something with it.
        ...

        // marshall it to DOM,
        Document dom = MarshallerAPI.marshallToDOM(o1);
        // marshall it to a File,
        MarshallerAPI.marshall( o1, "test.xml" );
        // or marshall it to an OutputStream
        MarshallerAPI.marshall( o1, System.out );
      }
```

This class lets you produce XML in many ways. There are handful of
overloaded versions of the marshall method, each support different type
of output.

Unfortunately, the implementation of the marshaller is rather poor at
this moment. Sometimes, the compiler fails to produce a marshaller for a
grammar. If it fails, please let us know so that we invent a better
algorithm. Also, there is a workaround. If the compiler fails to produce
a marshaller, then you can derive that class and implement it by hand.

## Creating objects programatically

Unmarshalling is not the only way to construct an object model. Instead,
you can programatically create it from scratch, much like you can create
a DOM tree from scratch by using the createXXX methods of a
`org.w3c.dom.Document` object.
