#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <gpgme.h>

#include "utils.h"
#include "array.h"
#include "iarray.h"

#define MAX_N_YEAR	256
#define SECONDS_PER_YEAR	(86400.0 * 365.25)

char *get_domain(char *email)
{
	char *at = strchr(email, '@');
	if (!at)
		return mystrdup(email);

	return mystrdup(at + 1);
}

char *get_tld(char *email)
{
	char *dot = strrchr(email, '.');
	if (!dot)
		return mystrdup(email);

	return mystrdup(dot + 1);
}

char *escape(char *in)
{
	int len = strlen(in);
	char *out = (char *)mymalloc(len*6 + 1, "unescape buffer");
	int loop, index=0;

	for(loop=0; loop<len;loop++)
	{
		if (in[loop] == '<')
		{
			sprintf(&out[index], "&lt;");
			index += 4;
		}
		else if (in[loop] == '>')
		{
			sprintf(&out[index], "&gt;");
			index += 4;
		}
		else if (in[loop] == '&')
		{
			sprintf(&out[index], "&amp;");
			index += 5;
		}
		else
		{
			out[index++] = in[loop];
		}
	}
	out[index] = 0x00;

	return out;
}

int main(int argc, char *argv[])
{
        gpgme_ctx_t     ctx;
        gpgme_error_t   err;
	time_t		start = time(NULL);
	int		show_n = 10, loop;
        char 		*weekday[7+1] = { "", "Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday" };
        char 		*month[12+1]  = { "", "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };
	char		*validity[6] = { "Unknown", "Undefined", "Never", "Marginal", "Full", "Ultimate" };

	fprintf(stderr, "GPGstats v" VERSION ", (C) 2005 by folkert@vanheusden.com\n\n");

	(void)gpgme_check_version(NULL);
//      err = gpgme_check_engine();
        err = gpgme_new(&ctx);

	err = gpgme_set_keylist_mode(ctx, GPGME_KEYLIST_MODE_LOCAL | GPGME_KEYLIST_MODE_SIGS);

	err = gpgme_op_keylist_start (ctx, NULL, 0);

	unsigned int n_mainkeys = 0;
	unsigned int n_mainkey_revoked = 0, n_mainkey_expired = 0, n_mainkey_disabled = 0, n_mainkey_invalid = 0;
	unsigned int n_mainkey_can_crypt = 0, n_mainkey_can_sign = 0, n_mainkey_can_certify = 0;
	unsigned int n_subkey = 0;
	unsigned int n_subkey_revoked = 0, n_subkey_expired = 0, n_subkey_disabled = 0, n_subkey_invalid = 0;
	unsigned int n_subkey_can_crypt = 0, n_subkey_can_sign = 0, n_subkey_can_certify = 0, n_subkey_can_authenticate = 0;
	unsigned int n_subkey_invalid_creation_ts = 0, n_subkey_creation_ts_not_available = 0, n_subkey_does_not_expire = 0;
	unsigned int n_uids = 0;
	unsigned int n_uids_revoked = 0, n_uids_invalid = 0;
	unsigned int n_uids_with_comment = 0, n_uids_without_name = 0, n_uids_without_email = 0;
	unsigned int n_sigs = 0;
	unsigned int n_sigs_revoked = 0, n_sigs_expired = 0, n_sigs_invalid = 0, n_sigs_exportable = 0;
	unsigned int n_sigs_invalid_creation_ts = 0, n_sigs_creation_ts_not_available = 0, n_sigs_does_not_expire = 0, n_signs_expired = 0;
	unsigned int n_sigs_with_comment = 0, n_sigs_without_name = 0, n_sigs_without_email = 0;
	time_t ts_oldest_sigs = 0, ts_newest_sigs = 0;
	unsigned long long int avg_sigs_expiration = 0, avg_sigs_expiration_after_now = 0;

	array top_sigs(1);
	array top_uids(1);

	unsigned long long int avg_subkey_expiration = 0;
	unsigned long long int avg_subkey_expiration_after_now = 0;

	intarray n_pubkey_algos(1);
	intarray n_key_length(1);

	intarray sig_per_year(1);
	intarray sig_per_month(1);
	intarray sig_per_day(1);
	intarray key_per_year(1);
	intarray key_per_month(1);
	intarray key_per_day(1);

	array first_name(1);
	array last_name(1);
	array domains(1);
	array tld(1);

	array sig_first_name(1);
	array sig_last_name(1);
	array sig_domains(1);
	array sig_tld(1);

	time_t ts_oldest_subkey = 0, ts_newest_subkey = 0;

	for(;;)
	{
		unsigned int cur_n_uids = 0, cur_n_subkeys = 0;
		unsigned int cur_n_subkey_revoked = 0, cur_n_subkey_expired = 0, cur_n_subkey_disabled = 0, cur_n_subkey_invalid = 0;
		unsigned int cur_n_subkey_can_crypt = 0, cur_n_subkey_can_sign = 0, cur_n_subkey_can_certify = 0, cur_n_subkey_can_authenticate = 0;
		unsigned int cur_n_uids_revoked = 0, cur_n_uids_invalid = 0;
		gpgme_key_t r_key;
		char *current_person_name  = NULL;
		char *current_person_email = NULL;

		err = gpgme_op_keylist_next(ctx, &r_key);
		if (err) break;

		if (!r_key)
			break;

		n_mainkeys++;
		fprintf(stderr, "%d\r", n_mainkeys);

		n_mainkeys++;

		/* True if key is revoked.  */
		if (r_key -> revoked)
			n_mainkey_revoked++;

		/* True if key is expired.  */
		if (r_key -> expired)
			n_mainkey_expired++;

		/* True if key is disabled.  */
		if (r_key -> disabled)
			n_mainkey_disabled++;

		/* True if key is invalid.  */
		if (r_key -> invalid)
			n_mainkey_invalid++;

		/* True if key can be used for encryption.  */
		if (r_key -> can_encrypt)
			n_mainkey_can_crypt++;

		/* True if key can be used for signing.  */
		if (r_key -> can_sign)
			n_mainkey_can_sign++;

		/* True if key can be used for certification.  */
		if (r_key -> can_certify)
			n_mainkey_can_certify++;

		if (r_key -> protocol == GPGME_PROTOCOL_CMS)
		{
			printf("%s / %s / %s\n", r_key -> issuer_serial, r_key -> issuer_name, r_key -> chain_id);
		}

		gpgme_subkey_t subkeys = r_key -> subkeys;
		while(subkeys)
		{
			n_subkey++;
			cur_n_subkeys++;

			/* True if key is revoked.  */
			if (subkeys -> revoked)
			{
				n_subkey_revoked++;
				cur_n_subkey_revoked++;
			}

			/* True if key is expired.  */
			if (subkeys -> expired)
			{
				n_subkey_expired++;
				cur_n_subkey_expired++;
			}

			/* True if key is disabled.  */
			if (subkeys -> disabled)
			{
				n_subkey_disabled++;
				cur_n_subkey_disabled++;
			}

			/* True if key is invalid.  */
			if (subkeys -> invalid)
			{
				n_subkey_invalid++;
				cur_n_subkey_invalid++;
			}

			/* True if key can be used for encryption.  */
			if (subkeys -> can_encrypt)
			{
				n_subkey_can_crypt++;
				cur_n_subkey_can_crypt++;
			}

			/* True if key can be used for signing.  */
			if (subkeys -> can_sign)
			{
				n_subkey_can_sign++;
				cur_n_subkey_can_sign++;
			}

			/* True if key can be used for certification.  */
			if (subkeys -> can_certify)
			{
				n_subkey_can_certify++;
				cur_n_subkey_can_certify++;
			}

			/* True if subkey can be used for authentication.  */
			if (subkeys -> can_authenticate)
			{
				n_subkey_can_authenticate++;
				cur_n_subkey_can_authenticate++;
			}

			/* Public key algorithms from libgcrypt.  */
			n_pubkey_algos.addvalue(subkeys -> pubkey_algo, 1);

			/* Length of the subkey.  */
			n_key_length.addvalue(subkeys -> length, 1);

			/* The creation timestamp, -1 if invalid, 0 if not available.  */
			if (subkeys -> timestamp == -1)
				n_subkey_invalid_creation_ts++;
			else if (subkeys -> timestamp == 0)
				n_subkey_creation_ts_not_available++;
			else
			{
				if (ts_oldest_subkey == 0 || ts_oldest_subkey < subkeys -> timestamp)
					ts_oldest_subkey = subkeys -> timestamp;
				if (ts_newest_subkey > subkeys -> timestamp)
					ts_newest_subkey = subkeys -> timestamp;
			}

			/* determine subkeys per year */
			if (subkeys -> timestamp > 0)
			{
				struct tm *sts = localtime(&subkeys -> timestamp);

				key_per_year.addvalue(sts -> tm_year + 1900, 1);
				key_per_month.addvalue(sts -> tm_mon + 1, 1);
				key_per_day.addvalue(sts -> tm_wday + 1, 1);
			}

			/* The expiration timestamp, 0 if the subkey does not expire.  */
			if (subkeys -> expires == 0)
				n_subkey_does_not_expire++;
			else
			{
				avg_subkey_expiration += subkeys -> expires - subkeys -> timestamp;

				if (subkeys -> expires < start)
					n_subkey_expired++;
				else
					avg_subkey_expiration_after_now += subkeys -> expires - start;
			}

			subkeys = subkeys -> next;
		}

		gpgme_user_id_t uids = r_key -> uids;
		while(uids)
		{
			int cur_n_sigs = 0;

			cur_n_uids++;
			n_uids++;

			/* True if the user ID is revoked.  */
			if (uids -> revoked)
			{
				n_uids_revoked++;
				cur_n_uids_revoked++;
			}

			/* True if the user ID is invalid.  */
			if (uids -> invalid)
			{
				n_uids_invalid++;
				cur_n_uids_invalid++;
			}

			/* FIXME: iets met 'gpgme_validity_t' doen? */

			/* The email part of the user ID.  */
			if (!current_person_email)
				current_person_email = mystrdup(uids -> email);

			/* The name part of the user ID.  */
			if (!current_person_name)
				current_person_name = mystrdup(uids -> name);

			if (uids -> comment && strlen(uids -> comment) > 0)
				n_uids_with_comment++;

			if (uids -> name == NULL || strlen(uids -> name) == 0)
				n_uids_without_name++;

			if (uids -> email == NULL || strlen(uids -> email) == 0)
				n_uids_without_email++;

			/* count first- and last- names */
			char *copy_name = mystrdup(uids -> name);
			char *dummy = strchr(copy_name, ' ');
			if (dummy)
			{
				*dummy = 0x00;
				first_name.addstring(copy_name, 1, 1);
				while( *(++dummy) == ' ');
				last_name.addstring(dummy, 1, 1);
			}
			free(copy_name);

			/* count domains & tlds */
			char *email_copy = mystrdup(uids -> email);
			dummy = strchr(email_copy, '@');
			if (dummy)
			{
				dummy++;
				/* domain */
				domains.addstring(dummy, 1, 1);
				/* tld */
				dummy = strrchr(dummy, '.');
				if (dummy)
					tld.addstring(dummy, 1, 1);
			}
			free(email_copy);

			/* process signatures */
			/* The signatures of the user ID.  */
			gpgme_key_sig_t sigs = uids -> signatures;
			while(sigs)
			{
				n_sigs++;
				cur_n_sigs++;

				if (sigs -> revoked)
					n_sigs_revoked++;

				if (sigs -> expired)
					n_sigs_expired++;

				if (sigs -> invalid)
					n_sigs_invalid++;

				if (sigs -> exportable)
					n_sigs_exportable++;

				if (sigs -> comment && strlen(sigs -> comment) > 0)
					n_sigs_with_comment++;

				if (sigs -> name == NULL || strlen(sigs -> name) == 0)
					n_sigs_without_name++;

				if (sigs -> email == NULL || strlen(sigs -> email) == 0)
					n_sigs_without_email++;

				/* The creation timestamp, -1 if invalid, 0 if not available.  */
				if (sigs -> timestamp == -1)
					n_sigs_invalid_creation_ts++;
				else if (sigs -> timestamp == 0)
					n_sigs_creation_ts_not_available++;
				else
				{
					if ((ts_oldest_sigs == 0 || ts_oldest_sigs > sigs -> timestamp) && sigs -> timestamp != 0)
						ts_oldest_sigs = sigs -> timestamp;
					if (ts_newest_sigs < sigs -> timestamp)
						ts_newest_sigs = sigs -> timestamp;
				}

				/* determine sigs per year/month/day */
				if (sigs -> timestamp > 0)
				{
					struct tm *sts = localtime(&sigs -> timestamp);

					sig_per_year.addvalue(sts -> tm_year + 1900, 1);
					sig_per_month.addvalue(sts -> tm_mon + 1, 1);
					sig_per_day.addvalue(sts -> tm_wday + 1, 1);
				}

				/* The expiration timestamp, 0 if the sigs does not expire.  */
				if (sigs -> expires == 0)
					n_sigs_does_not_expire++;
				else
				{
					avg_sigs_expiration += sigs -> expires - sigs -> timestamp;

					if (sigs -> expires < start)
						n_signs_expired++;
					else
						avg_sigs_expiration_after_now += sigs -> expires - start;
				}

				sigs = sigs -> next;
			}

			/* uid with the most signatures? */
			char *top_sigs_name = (char *)mymalloc(strlen(uids -> name) + strlen(uids -> email) + 4, "max n sigs");
			sprintf(top_sigs_name, "%s <%s>", uids -> name, uids -> email);
			int dummy_index = top_sigs.addstring(top_sigs_name, 0, 1);
			if (cur_n_sigs > top_sigs.getcounter(dummy_index, 0))
				top_sigs.setcounter(dummy_index, 0, cur_n_sigs);
			free(top_sigs_name);

			uids = uids -> next;
		}

		char *top_uids_name = (char *)mymalloc(strlen(current_person_name) + strlen(current_person_email) + 4, "max n uids");
		sprintf(top_uids_name, "%s <%s>", current_person_name, current_person_email);
		int dummy_index = top_uids.addstring(top_uids_name, 0, 1);
		if (cur_n_uids > top_uids.getcounter(dummy_index, 0))
			top_uids.setcounter(dummy_index, 0, cur_n_uids);
		free(top_uids_name);

		free(current_person_email);
		free(current_person_name);

		gpgme_key_release(r_key);
	}

	err = gpgme_op_keylist_end(ctx);

	printf("<HTML>\n");
	printf("<HEAD><TITLE>Statistics for keyring generated at %s</TITLE></HEAD>\n", ctime(&start));
	printf("<BODY>\n");
	printf("<H1>Statistics for keyring generated at %s</H1>\n", ctime(&start));

	printf("<H2>Global</H2>\n");
	printf("<TABLE>\n");
	printf("<TR><TD>Number of mainkeys in ring:</TD><TD>%d</TD></TR>\n", n_mainkeys);
	printf("<TR><TD>Number of revoked mainkeys:</TD><TD>%d</TD></TR>\n", n_mainkey_revoked);
	printf("<TR><TD>Number of expired mainkeys:</TD><TD>%d</TD></TR>\n", n_mainkey_expired);
	printf("<TR><TD>Number of disabled mainkeys:</TD><TD>%d</TD></TR>\n", n_mainkey_disabled);
	printf("<TR><TD>Number of invalid mainkeys:</TD><TD>%d</TD></TR>\n", n_mainkey_invalid);
	printf("<TR><TD>Number of mainkeys that can crypt:</TD><TD>%d</TD></TR>\n", n_mainkey_can_crypt);
	printf("<TR><TD>Number of mainkeys that can sign:</TD><TD>%d</TD></TR>\n", n_mainkey_can_sign);
	printf("<TR><TD>Number of mainkeys that can certify:</TD><TD>%d</TD></TR>\n", n_mainkey_can_certify);
	printf("</TABLE>\n");
	printf("<TABLE>\n");
	printf("<TR><TD>Number of subkeys:</TD><TD>%d</TD></TR>\n", n_subkey);
	printf("<TR><TD>Number of revoked subkeys:</TD><TD>%d</TD></TR>\n", n_subkey_revoked);
	printf("<TR><TD>Number of expired subkeys:</TD><TD>%d</TD></TR>\n", n_subkey_expired);
	printf("<TR><TD>Number of disabled subkeys:</TD><TD>%d</TD></TR>\n", n_subkey_disabled);
	printf("<TR><TD>Number of invalid subkeys:</TD><TD>%d</TD></TR>\n", n_subkey_invalid);
	printf("<TR><TD>Number of subkeys that can crypt:</TD><TD>%d</TD></TR>\n", n_subkey_can_crypt);
	printf("<TR><TD>Number of subkeys that can sign:</TD><TD>%d</TD></TR>\n", n_subkey_can_sign);
	printf("<TR><TD>Number of subkeys that can certify:</TD><TD>%d</TD></TR>\n", n_subkey_can_certify);
	printf("<TR><TD>Number of subkeys that can authenticate:</TD><TD>%d</TD></TR>\n", n_subkey_can_authenticate);
	printf("<TR><TD>Number of subkeys with invalid creation timestamp:</TD><TD>%d</TD></TR>\n", n_subkey_invalid_creation_ts);
	printf("<TR><TD>Number of subkeys without creation timestamp:</TD><TD>%d</TD></TR>\n", n_subkey_creation_ts_not_available);
	printf("<TR><TD>Number of subkeys that do not expire:</TD><TD>%d</TD></TR>\n", n_subkey_does_not_expire);
	printf("<TR><TD>Number of subkeys that are expired:</TD><TD>%d</TD></TR>\n", n_subkey_expired);
	long int dummy = n_subkey - n_subkey_expired;
	if (dummy) printf("<TR><TD>Average subkey expiration for not already expired subkeys:</TD><TD>%d</TD></TR>\n", avg_subkey_expiration_after_now / dummy);
	printf("</TABLE>\n");
	printf("<TABLE>\n");
	printf("<TR><TD>Number of UIDs:</TD><TD>%d</TD></TR>\n", n_uids);
	printf("<TR><TD>Number of revoked UIDs:</TD><TD>%d</TD></TR>\n", n_uids_revoked);
	printf("<TR><TD>Number of invalid UIDs:</TD><TD>%d</TD></TR>\n", n_uids_invalid);
	printf("<TR><TD>Number of UIDs with a comment:</TD><TD>%d</TD></TR>\n", n_uids_with_comment);
	printf("<TR><TD>Number of UIDs without a name:</TD><TD>%d</TD></TR>\n", n_uids_without_name);
	printf("<TR><TD>Number of UIDs without an e-mail address:</TD><TD>%d</TD></TR>\n", n_uids_without_email);
	printf("</TABLE>\n");
	printf("<TABLE>\n");
	printf("<TR><TD>Number of signatures:</TD><TD>%d</TD></TR>\n", n_sigs);
	printf("<TR><TD>Number of revoked signatures:</TD><TD>%d</TD></TR>\n", n_sigs_revoked);
	printf("<TR><TD>Number of expired signatures:</TD><TD>%d</TD></TR>\n", n_sigs_expired);
	printf("<TR><TD>Number of invalid signatures:</TD><TD>%d</TD></TR>\n", n_sigs_invalid);
	printf("<TR><TD>Number of exportable signatures:</TD><TD>%d</TD></TR>\n", n_sigs_exportable);
	printf("<TR><TD>Number of signatures with invalid creation timestamp:</TD><TD>%d</TD></TR>\n", n_sigs_invalid_creation_ts);
	printf("<TR><TD>Number of signatures without creation timestamp:</TD><TD>%d</TD></TR>\n", n_sigs_creation_ts_not_available);
	printf("<TR><TD>Number of signatures that do not expire:</TD><TD>%d</TD></TR>\n", n_sigs_does_not_expire);
	printf("<TR><TD>Number of signatures that are expired:</TD><TD>%d</TD></TR>\n", n_sigs_expired);
	dummy = n_sigs - n_sigs_expired;
	if (dummy) printf("<TR><TD>Average signature expiration for not already expired sigs:</TD><TD>%d</TD></TR>\n", avg_sigs_expiration_after_now / dummy);
	printf("<TR><TD>Number of signatures with a comment:</TD><TD>%d</TD></TR>\n", n_sigs_with_comment);
	printf("<TR><TD>Number of signatures without a name:</TD><TD>%d</TD></TR>\n", n_sigs_without_name);
	printf("<TR><TD>Number of signatures without an e-mail address:</TD><TD>%d</TD></TR>\n", n_sigs_without_email);
	printf("<TR><TD>Oldest signature:</TD><TD>%s</TD></TR>\n", ctime(&ts_oldest_sigs));
	printf("<TR><TD>Newest signature:</TD><TD>%s</TD></TR>\n", ctime(&ts_newest_sigs));
	printf("</TABLE>\n");

	printf("<H2>Top lists</H2>\n");
	top_sigs.sort(0);
	printf("<H3>Key with a UID with the most signatures</H3>\n");
	printf("<TABLE>\n");
	printf("<TR><TD><B>Entry</B></TD><TD><B>N sigs</B></TD><TD><B>Who</B></TD></TR>\n");
	for(loop=0; loop<min(top_sigs.getn(), show_n); loop++)
	{
		printf("<TR><TD>%d</TD><TD>%d</TD><TD>%s</TD></TR>\n", loop, top_sigs.getcounter(loop, 0), escape(top_sigs.getstring(loop)));
	}
	printf("</TABLE>\n");

	top_uids.sort(0);
	printf("<H3>Key with the most UIDs</H3>\n");
	printf("<TABLE>\n");
	printf("<TR><TD><B>Entry</B></TD><TD><B>N uids</B></TD><TD><B>Who</B></TD></TR>\n");
	for(loop=0; loop<min(top_uids.getn(), show_n); loop++)
	{
		printf("<TR><TD>%d</TD><TD>%d</TD><TD>%s</TD></TR>\n", loop, top_uids.getcounter(loop, 0), escape(top_uids.getstring(loop)));
	}
	printf("</TABLE>\n");

	first_name.sort(0);
	printf("<H3>Most common first-name</H3>\n");
	printf("<TABLE>\n");
	printf("<TR><TD><B>Entry</B></TD><TD><B>Occurences</B></TD><TD><B>Who</B></TD></TR>\n");
	for(loop=0; loop<min(first_name.getn(), show_n); loop++)
	{
		printf("<TR><TD>%d</TD><TD>%d</TD><TD>%s</TD></TR>\n", loop, first_name.getcounter(loop, 0), escape(first_name.getstring(loop)));
	}
	printf("</TABLE>\n");

	last_name.sort(0);
	printf("<H3>Most common last-name</H3>\n");
	printf("<TABLE>\n");
	printf("<TR><TD><B>Entry</B></TD><TD><B>Occurences</B></TD><TD><B>Who</B></TD></TR>\n");
	for(loop=0; loop<min(last_name.getn(), show_n); loop++)
	{
		printf("<TR><TD>%d</TD><TD>%d</TD><TD>%s</TD></TR>\n", loop, last_name.getcounter(loop, 0), escape(last_name.getstring(loop)));
	}
	printf("</TABLE>\n");

	tld.sort(0);
	printf("<H3>Most common TLD</H3>\n");
	printf("<TABLE>\n");
	printf("<TR><TD><B>Entry</B></TD><TD><B>Occurences</B></TD><TD><B>What TLD</B></TD></TR>\n");
	for(loop=0; loop<min(tld.getn(), show_n); loop++)
	{
		printf("<TR><TD>%d</TD><TD>%d</TD><TD>%s</TD></TR>\n", loop, tld.getcounter(loop, 0), escape(tld.getstring(loop)));
	}
	printf("</TABLE>\n");

	domains.sort(0);
	printf("<H3>Most common domains</H3>\n");
	printf("<TABLE>\n");
	printf("<TR><TD><B>Entry</B></TD><TD><B>Occurences</B></TD><TD><B>What domain</B></TD></TR>\n");
	for(loop=0; loop<min(domains.getn(), show_n); loop++)
	{
		printf("<TR><TD>%d</TD><TD>%d</TD><TD>%s</TD></TR>\n", loop, domains.getcounter(loop, 0), escape(domains.getstring(loop)));
	}
	printf("</TABLE>\n");

	n_pubkey_algos.sort();
	printf("<H3>Most used public key algorithm</H3>\n");
	printf("<TABLE>\n");
	printf("<TR><TD><B>Entry</B></TD><TD><B>Occurences</B></TD><TD><B>Algorithm</B></TD></TR>\n");
	for(loop=0; loop<min(n_pubkey_algos.getn(), show_n); loop++)
	{
		printf("<TR><TD>%d</TD><TD>%d</TD><TD>%s</TD></TR>\n", loop, n_pubkey_algos.getcounter(loop), escape((char *)gpgme_pubkey_algo_name((gpgme_pubkey_algo_t)n_pubkey_algos.getvalue(loop))));
	}
	printf("</TABLE>\n");

	n_key_length.sort();
	printf("<H3>Most used key size</H3>\n");
	printf("<TABLE>\n");
	printf("<TR><TD><B>Entry</B></TD><TD><B>Occurences</B></TD><TD><B>Key size</B></TD></TR>\n");
	for(loop=0; loop<min(n_key_length.getn(), show_n); loop++)
	{
		printf("<TR><TD>%d</TD><TD>%d</TD><TD>%d</TD></TR>\n", loop, n_key_length.getcounter(loop), n_key_length.getvalue(loop));
	}
	printf("</TABLE>\n");

	key_per_year.sort();
	printf("<H3>Most new keys per year</H3>\n");
	printf("<TABLE>\n");
	printf("<TR><TD><B>Entry</B></TD><TD><B>Occurences</B></TD><TD><B>Year</B></TD></TR>\n");
	for(loop=0; loop<min(key_per_year.getn(), show_n); loop++)
	{
		printf("<TR><TD>%d</TD><TD>%d</TD><TD>%d</TD></TR>\n", loop, key_per_year.getcounter(loop), key_per_year.getvalue(loop));
	}
	printf("</TABLE>\n");

	key_per_month.sort();
	printf("<H3>Most new keys per month</H3>\n");
	printf("<TABLE>\n");
	printf("<TR><TD><B>Occurences</B></TD><TD><B>Month</B></TD></TR>\n");
	for(loop=0; loop<min(key_per_month.getn(), show_n); loop++)
	{
		printf("<TR><TD>%d</TD><TD>%s</TD></TR>\n", key_per_month.getcounter(loop), month[key_per_month.getvalue(loop)]);
	}
	printf("</TABLE>\n");

	key_per_day.sort();
	printf("<H3>Most new keys per weekday</H3>\n");
	printf("<TABLE>\n");
	printf("<TR><TD><B>Occurences</B></TD><TD><B>Weekday</B></TD></TR>\n");
	for(loop=0; loop<min(key_per_day.getn(), show_n); loop++)
	{
		printf("<TR><TD>%d</TD><TD>%s</TD></TR>\n", key_per_day.getcounter(loop), weekday[key_per_day.getvalue(loop)]);
	}
	printf("</TABLE>\n");

        sig_per_year.sort();
        printf("<H3>Most new signatures per year</H3>\n");
        printf("<TABLE>\n");
        printf("<TR><TD><B>Entry</B></TD><TD><B>Occurences</B></TD><TD><B>Year</B></TD></TR>\n");
        for(loop=0; loop<min(sig_per_year.getn(), show_n); loop++)
        {
                printf("<TR><TD>%d</TD><TD>%d</TD><TD>%d</TD></TR>\n", loop, sig_per_year.getcounter(loop), sig_per_year.getvalue(loop));
        }
	printf("</TABLE>\n");

        sig_per_month.sort();
        printf("<H3>Most new signatures per month</H3>\n");
        printf("<TABLE>\n");
        printf("<TR><TD><B>Occurences</B></TD><TD><B>Month</B></TD></TR>\n");
        for(loop=0; loop<min(sig_per_month.getn(), show_n); loop++)
        {
                printf("<TR><TD>%d</TD><TD>%s</TD></TR>\n", sig_per_month.getcounter(loop), month[sig_per_month.getvalue(loop)]);
        }
	printf("</TABLE>\n");

        sig_per_day.sort();
        printf("<H3>Most new signatures per weekday</H3>\n");
        printf("<TABLE>\n");
        printf("<TR><TD><B>Occurences</B></TD><TD><B>Weekday</B></TD></TR>\n");
        for(loop=0; loop<min(sig_per_day.getn(), show_n); loop++)
        {
                printf("<TR><TD>%d</TD><TD>%s</TD></TR>\n", sig_per_day.getcounter(loop), weekday[sig_per_day.getvalue(loop)]);
        }
	printf("</TABLE>\n");

	printf("</BODY></HTML>\n");

	return 0;
}
