/*
 * scdtools - Tools for Scdaemon and OpenPGP smartcards
 * Copyright (C) 2014,2015 Damien Goutte-Gattat
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <ctype.h>
#include <time.h>
#include <getopt.h>
#include <err.h>
#include <limits.h>

#include <gcrypt.h>

#include "gpg-util.h"
#include "otpauth.h"

#define UNSET_PARAM         UINT_MAX
#define MAX_HASH_SIZE       64  /* Size of a SHA-512 hash. */

static void
usage(int status)
{
    puts("Usage: scdtotp [options]\n\
Generate a time-based one-time password (TOTP) from a seed\n\
found in a OpenPGP smartcard.\n\
");

    puts("Options:\n\
  -h, --help            Display this help message.\n\
  -v, --version         Display the version message.\n\
");

    puts("\
  -t, --time SECONDS    Generate OTP for the specified time\n\
                        (in seconds) instead of current time.\n\
  -w, --window N        Generate OTP for N time windows around\n\
                        the current time.\n\
");

    printf("\
  -n, --private-do N    Read key from private data object N\n\
                        (default is %d).\n\
\n", DEFAULT_PRIVATE_DO);

    puts("The following options will override the parameters\n\
read from the smartcard:\n\
  -p, --period N        Use a period of N seconds.\n\
  -d, --digits N        Output N digits (must be 6, 7, or 8).\n\
  -m, --mac-algo ALG    Use the specified HMAC algorithm (must\n\
                        be 'sha1', 'sha256', or 'sha512').\n\
");

    printf("Report bugs to <%s>.\n", PACKAGE_BUGREPORT);

    exit(status);
}

static void
info(void)
{
    printf("\
scdtotp (scdtools %s)\n\
Copyright (C) 2016 Damien Goutte-Gattat\n\
\n\
This program is released under the GNU General Public License.\n\
See the COPYING file or <http://www.gnu.org/licenses/gpl.html>.\n\
", VERSION);

    exit(EXIT_SUCCESS);
}

/*
 * Convert a time value (in seconds) into a moving factor for the
 * TOTP algorithm as per RFC 6238.
 *
 * @param time   Elapsed seconds since Unix epoch.
 * @param step   Time step parameter.
 * @param buffer Byte buffer to store the formatted factor. Must be
 *               at least 8 bytes long.
 * @param len    Size of the output buffer.
 *
 * @return The number of bytes written to the buffer (always 8).
 */
static int
time2factor(time_t time, int step, unsigned char *buffer, size_t len)
{
    unsigned n_steps;
    int i, j;

    (void)len;

    n_steps = time / step;

    for ( i = 0, j = 7; i < 8; i++, j-- )
        buffer[i] = (n_steps & (uint64_t)(0xFF << (j * 8))) >> (j * 8);

    return 8;
}

/*
 * Generate a time-based one-time password as per RFC 6238.
 *
 * @param mac    Libgcrypt object for the HMAC algorithm to use.
 * @param time   Time in seconds.
 * @param step   Time step parameter.
 * @param digits Number of digits of the OTP to generate
 *               (must be 6, 7, or 8).
 * @param otp    Pointer to an unsigned integer to store the
 *               generated OTP value.
 *
 * @return 0 if OTP generation was successful, or a gpg_error_t
 *         error code.
 */
static gpg_error_t
generate_totp(gcry_mac_hd_t mac,
              time_t        time,
              unsigned      step,
              int           digits,
              unsigned     *otp)
{
    gpg_error_t e;
    unsigned char buffer[MAX_HASH_SIZE];
    size_t len;
    int offset;

    static int modulo[] = { 0, 0, 0, 0, 0, 0, 1000000, 10000000, 100000000 };

    (void)time2factor(time, step, buffer, sizeof(buffer));
    if ( ! (e = gcry_mac_write(mac, buffer, 8)) ) {
        len = sizeof(buffer);
        if ( ! (e = gcry_mac_read(mac, buffer, &len)) ) {
            offset = buffer[len - 1] & 0xF;
            *otp = ((buffer[offset]     & 0x7F) << 24) |
                   ((buffer[offset + 1] & 0xFF) << 16) |
                   ((buffer[offset + 2] & 0xFF) <<  8) |
                    (buffer[offset + 3] & 0xFF);
            *otp = *otp % modulo[digits];

            e = gcry_error(GPG_ERR_NO_ERROR);
        }
    }

    return e;
}

/*
 * Callback for the below function.
 */
static gpg_error_t
get_otp_params_cb(void *arg, const char *line)
{
    otp_t **otp = arg;

    if ( ! strncmp("PRIVATE-DO-", line, 11) ) {

        if ( (*otp = otp_parse_uri(line + 13)) )
            return 0;
        else
            return gcry_error(GPG_ERR_BAD_URI);
    }

    return gcry_error(GPG_ERR_NO_DATA);
}

/*
 * Retrieve OTP parameters from an inserted OpenPGP smartcard.
 *
 * @param privatedo Number of the private Data Object to read from.
 * @param otp       Double pointer to a otp_t structure
 *                  representing the OTP parameters. It will be
 *                  automatically allocated and should be freed
 *                  by the caller.
 *
 * @return 0 if successful, or a gpg_error_t error code.
 */
static gpg_error_t
get_otp_params(unsigned privatedo, otp_t **otp)
{
    assuan_context_t ctx;
    gpg_error_t e;
    char command[32];

    if ( privatedo < 1 || privatedo > 4 )
        return gcry_error(GPG_ERR_INV_ARG);

    if ( ! (e = connect_to_scdaemon(&ctx)) ) {

        snprintf(command, sizeof(command), "GETATTR PRIVATE-DO-%d", privatedo);

        e = assuan_transact(ctx, command, NULL, NULL, NULL, NULL,
                get_otp_params_cb, otp);

        assuan_release(ctx);
    }

    return e;
}

static gpg_error_t
get_serial_cb(void *arg, const char *line)
{
    char *serial = (char *)arg;

    if ( ! strncmp("SERIALNO ", line, 9) ) {
        strcpy(serial, line + 9);
        return GPG_ERR_NO_ERROR;
    }
    else
        return GPG_ERR_NO_DATA;
}

static gpg_error_t
verify_pin(int admin)
{
    assuan_context_t ctx;
    gpg_error_t e;
    char command[64], serial[64];

    if ( ! (e = connect_to_agent(&ctx, 1)) ) {

        if ( ! (e = assuan_transact(ctx, "SCD GETATTR SERIALNO",
                        NULL, NULL, NULL, NULL, get_serial_cb, serial)) ) {

            snprintf(command, sizeof(command), "SCD CHECKPIN %s%s",
                    serial, admin ? "[CHV3]" : "");
            e = assuan_transact(ctx, command, NULL, NULL, NULL, NULL,
                    NULL, NULL);
        }

        assuan_release(ctx);
    }

    return e;
}


static void
print_otp(unsigned otp, unsigned digits)
{
    char fmt[] = "%06d\n";

    fmt[2] = digits + '0';
    printf(fmt, otp);
}

static unsigned long
get_uinteger_or_die(const char *arg)
{
    unsigned long val;
    char *endptr;

    errno = 0;
    val = strtoul(arg, &endptr, 10);
    if ( errno != 0 || endptr == arg )
        errx(EXIT_FAILURE, "Invalid argument, unsigned integer expected: %s", arg);

    return val;
}

int
main(int argc, char **argv)
{
    int c, n;
    unsigned algo, period, digits, privatedo, value, window;
    time_t secs;
    gcry_error_t e;
    otp_t *otp;
    gcry_mac_hd_t mac;

    struct option options[] = {
        { "help",       0, NULL, 'h' },
        { "version",    0, NULL, 'v' },
        { "time",       1, NULL, 't' },
        { "window",     1, NULL, 'w' },
        { "period",     1, NULL, 'p' },
        { "digits",     1, NULL, 'd' },
        { "mac-algo",   1, NULL, 'm' },
        { "private-do", 1, NULL, 'n' },
        { NULL,         0, NULL, 0 }
    };

    setprogname(argv[0]);
    secs = time(NULL);
    algo = period = digits = UNSET_PARAM;
    privatedo = DEFAULT_PRIVATE_DO;
    value = window = 0;

    while ( (c = getopt_long(argc, argv, "hvt:w:p:d:m:n:",
                             options, NULL)) != -1 ) {
        switch ( c ) {
        case 'h':
            usage(EXIT_SUCCESS);
            break;

        case '?':
            usage(EXIT_FAILURE);
            break;

        case 'v':
            info();
            break;

        case 't':
            secs = get_uinteger_or_die(optarg);
            break;

        case 'p':
            period = get_uinteger_or_die(optarg);
            break;

        case 'd':
            digits = get_uinteger_or_die(optarg);
            if ( digits < 6 || digits > 8 )
                errx(EXIT_FAILURE, "digits must be either 6, 7, or 8");
            break;

        case 'm':
            if ( ! strcmp("sha1", optarg) )
                algo = GCRY_MAC_HMAC_SHA1;
            else if ( ! strcmp("sha256", optarg) )
                algo = GCRY_MAC_HMAC_SHA256;
            else if ( ! strcmp("sha512", optarg) )
                algo = GCRY_MAC_HMAC_SHA512;
            else
                errx(EXIT_FAILURE, "unsupported HMAC algorithm: %s", optarg);
            break;

        case 'n':
            privatedo = get_uinteger_or_die(optarg);
            if ( privatedo < 1 || privatedo > 4 )
                errx(EXIT_FAILURE, "DO number must be 1, 2, 3, or 4");
            break;

        case 'w':
            window = get_uinteger_or_die(optarg);
            break;
        }
    }

    if ( ! gcry_check_version(GCRYPT_VERSION) )
        errx(EXIT_FAILURE, "libgcrypt version mismatch");

    gcry_control(GCRYCTL_DISABLE_SECMEM, 0);
    gcry_control(GCRYCTL_INITIALIZATION_FINISHED, 0);

    if ( privatedo > 2 && (e = verify_pin(privatedo == 4)) )
        errx(EXIT_FAILURE, "Cannot get OTP info from token: %s", gcry_strerror(e));

    if ( (e = get_otp_params(privatedo, &otp)) )
        errx(EXIT_FAILURE, "Cannot get OTP info from token: %s", gcry_strerror(e));

    if ( algo == UNSET_PARAM ) {
        if ( otp->algo == OTP_ALGO_SHA1 )
            algo = GCRY_MAC_HMAC_SHA1;
        else if ( otp->algo == OTP_ALGO_SHA256 )
            algo = GCRY_MAC_HMAC_SHA256;
        else if ( otp->algo == OTP_ALGO_SHA512 )
            algo = GCRY_MAC_HMAC_SHA512;
    }

    if ( period == UNSET_PARAM )
        period = otp->period;

    if ( digits == UNSET_PARAM )
        digits = otp->digits;

    if ( (e = gcry_mac_open(&mac, algo, 0, NULL)) ||
            (e = gcry_mac_setkey(mac, otp->secret, otp->length)) )
        errx(EXIT_FAILURE, "Cannot initialize HMAC: %s", gcry_strerror(e));

    for ( n = -window; n < (signed) window + 1; n++ ) {
        if ( n > (signed) -window )
            gcry_mac_reset(mac);

        if ( (e = generate_totp(mac, secs + (n * (signed)period),
                        period, digits, &value)) )
            errx(EXIT_FAILURE, "Cannot generate OTP: %s", gcry_strerror(e));

        print_otp(value, digits);
    }

    gcry_mac_close(mac);

    return EXIT_SUCCESS;
}
