/*  amplitude.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include <iostream>
#include <fstream>
#include <string>
#include <vector>
#include "diagram.h"
#include "amplitude.h"
#include "integralfamily.h"
#include "ginac/ginac.h"
#include "functions.h"
#include "sector.h"
#include "propagator.h"
#include "files.h"
#include "color.h"
#include "polarization.h"
#include "int.h"
#include "ginacutils.h"
#include "feynmanrules.h"
#include "equation.h"
#include "globalsymbols.h"

using namespace std;
using namespace GiNaC;

namespace Reduze {

REGISTER_FUNCTION(Lorentz, dummy())

//
//
//

// public constructor
/*OpenFermionChain::OpenFermionChain() :
	left_(symbol("undefined")), right_(symbol("undefined")), value_(0) {
}*/

OpenFermionChain::OpenFermionChain(const GiNaC::ex& left,
		const GiNaC::exvector& line, const GiNaC::ex& right) {
	*this = OpenFermionChain(left, ncmul(line), right);

	if (!is_a<indexed> (left))
		ABORT("The expression " << left << " is not an indexed" );
	if (!is_a<indexed> (right))
		ABORT("The expression " << right << " is not an indexed" );
	VERIFY(left.nops() > 1);
	VERIFY(right.nops() > 1);
}

// private constructor
OpenFermionChain::OpenFermionChain(const GiNaC::ex& left,
		const GiNaC::ex& value, const GiNaC::ex& right) :
	left_(left), right_(right), value_(value) {
}

OpenFermionChain OpenFermionChain::conjugate() const {
	return OpenFermionChain(right_.conjugate(), value_.conjugate(),
			left_.conjugate());
}

OpenFermionChain OpenFermionChain::substitute(const GiNaC::exmap& m,
		unsigned int opt) const {
	return OpenFermionChain(left_.subs(m, opt), value_.subs(m, opt),
			right_.subs(m, opt));
}

ostream& operator<<(ostream& os, const OpenFermionChain& line) {
	os << line.left_ << " * " << line.value_ << " * " << line.right_;
	return os;
}

// static public member
std::vector<std::vector<OpenFermionChain> > OpenFermionChain::sort_open_fermion_chains(
		const std::vector<OpenFermionChain>& ofcs1,
		const std::vector<OpenFermionChain>& ofcs2) {

	std::list<OpenFermionChain> ofchains;
	ofchains.insert(ofchains.end(), ofcs1.begin(), ofcs1.end());
	ofchains.insert(ofchains.end(), ofcs2.begin(), ofcs2.end());

	std::vector<std::vector<OpenFermionChain> > result;
	std::vector<OpenFermionChain> sorted;

	while (!ofchains.empty()) {
		if (sorted.empty()) {
			sorted.push_back(ofchains.front()); // first chain
			ofchains.pop_front();
		}
		GiNaC::ex plrzt_left = sorted.back().right().op(0).conjugate();
		bool found_next = false;
		std::list<OpenFermionChain>::iterator next;
		for (next = ofchains.begin(); next != ofchains.end(); ++next) {
			if (next->left().op(0).is_equal(plrzt_left)) {
				sorted.push_back(*next); // following chain
				found_next = true;
				ofchains.erase(next);
				break;
			}
		}

		if (!found_next || ofchains.empty()) { // sorted is not empty anyway
			if (!(sorted.front().left().op(0).is_equal(
					sorted.back().right().op(0).conjugate()))) {
				throw std::runtime_error(
						"Failed to find pairs of conjugated and not conjugated spinors/ghosts!");
			}
			result.push_back(std::vector<OpenFermionChain>());
			result.back().swap(sorted);
		}
	}
	return result;
}

//
//
//

// public constructor
ClosedFermionChain::ClosedFermionChain() {
}
ClosedFermionChain::ClosedFermionChain(const GiNaC::exvector& line) :
	value_(ncmul(line)) {
}

// private constructor
ClosedFermionChain::ClosedFermionChain(const GiNaC::ex& value) :
	value_(value) {
}

ClosedFermionChain ClosedFermionChain::conjugate() const {
	return ClosedFermionChain(value_.conjugate());
}

ClosedFermionChain ClosedFermionChain::substitute(const GiNaC::exmap& m,
		unsigned int opt) const {
	return ClosedFermionChain(value_.subs(m, opt));
}

ostream& operator <<(ostream& os, const ClosedFermionChain& line) {
	os << line.value_;
	return os;
}

// static public member
std::vector<ClosedFermionChain> ClosedFermionChain::close_open_fermion_chains(
		const std::vector<OpenFermionChain>& ofcs1,
		const std::vector<OpenFermionChain>& ofcs2, const FeynmanRules* fr) {
	vector<ClosedFermionChain> result;
	vector<vector<OpenFermionChain> > ofchains;
	try {
		ofchains = OpenFermionChain::sort_open_fermion_chains(ofcs1, ofcs2);
	} catch (std::exception& e) {
		throw std::runtime_error(
				"Failed to close open fermion chains: " + to_string(e.what()));
	}
	vector<vector<OpenFermionChain> >::iterator ofc;
	vector<OpenFermionChain>::iterator current, next;
	for (ofc = ofchains.begin(); ofc != ofchains.end(); ++ofc) {
		exvector tmp;
		for (current = ofc->begin(); current != ofc->end(); ++current) {
			next = (++current != ofc->end()) ? current : ofc->begin();
			--current;
			ex ps = fr->get_polarization_sum(current->right(), next->left());
			tmp.push_back(current->value());
			tmp.push_back(ps);
		}
		result.push_back(ClosedFermionChain(tmp));
	}
	return result;
}

//
//
//

Amplitude::Amplitude() :
		amp_(0) {
}

void Amplitude::insert(const OpenFermionChain& e) {
	open_fermion_chains_.push_back(e);
}
void Amplitude::insert(const ClosedFermionChain& e) {
	closed_fermion_chains_.push_back(e);
}
void Amplitude::set_external_bosons(const GiNaC::exvector& e) {
	external_bosons_ = e;
}
void Amplitude::set_amp(const GiNaC::ex& e) {
	amp_ = e;
}

void Amplitude::set_sector(const Sector& sector) {
	sector_.clear();
	sector_.push_front(sector);
}

const Sector& Amplitude::sector() const {
	if (sector_.empty())
		throw runtime_error("No sector available");
	return sector_.front();
}

const std::string& Amplitude::name() const {
	return name_;
}

void Amplitude::set_name(const string& name) {
	name_ = name;
}

void Amplitude::set_local_indices(const map<string, GiNaC::symbol>& indices) {
	local_indices_ = indices;
}

Amplitude Amplitude::conjugate() const {
	Amplitude amp_conjugated(*this);
	vector<OpenFermionChain>::iterator o;
	o = amp_conjugated.open_fermion_chains_.begin();
	for (; o != amp_conjugated.open_fermion_chains_.end(); ++o)
		*o = o->conjugate();
	vector<ClosedFermionChain>::iterator c;
	c = amp_conjugated.closed_fermion_chains_.begin();
	for (; c != amp_conjugated.closed_fermion_chains_.end(); ++c)
		*c = c->conjugate();

	amp_conjugated.amp_ = amp_conjugated.amp_.conjugate();
	amp_conjugated.external_bosons_ = conjugate_in_exvector(
			amp_conjugated.external_bosons_);
	return amp_conjugated;
}

void Amplitude::make_new_index_names() {
	map<string, symbol> old;
	old.swap(local_indices_);
	exmap m;
	for (map<string, symbol>::iterator it = old.begin(); it != old.end(); ++it) {
		symbol s;
		m[it->second] = s;
		local_indices_.insert(make_pair(s.get_name(), s));
	}

	unsigned int options = subs_options::no_pattern;
	amp_ = amp_.subs(m, options);
	external_bosons_ = subs_in_exvector(external_bosons_, m, options);
	vector<OpenFermionChain>::iterator o;
	for (o = open_fermion_chains_.begin(); o != open_fermion_chains_.end(); ++o)
		*o = o->substitute(m, options);
	vector<ClosedFermionChain>::iterator c;
	for (c = closed_fermion_chains_.begin(); c != closed_fermion_chains_.end(); ++c)
		*c = c->substitute(m, options);
}

ostream& operator <<(ostream& os, const Amplitude& amp) {
	os << "Amplitude " << amp.name_ << endl;
	os << "Open fermion lines:" << endl;
	for (unsigned i = 0; i < amp.open_fermion_chains_.size(); ++i) {
		os << "Line: " << i << endl;
		os << amp.open_fermion_chains_[i] << endl;
	}
	os << "Closed fermion lines:" << endl;
	for (unsigned i = 0; i < amp.closed_fermion_chains_.size(); ++i) {
		os << "Line: " << i << endl;
		os << amp.closed_fermion_chains_[i] << endl;
	}
	os << "External:" << endl;
	os << amp.external_bosons_ << endl;
	os << "Remaining Term:" << endl;
	os << amp.amp_ << endl;

	return os;
}

//
//
//

namespace {

// replaces the scalar product SP(p,p) by invariants
GiNaC::ex substitute_scalar_products(const GiNaC::ex& e) {
	exmap invar_rules = Files::instance()->kinematics()->rules_sp_to_invariants();
	return e.subs(invar_rules, subs_options::no_pattern);
}

// replaces the scalar product SP(p,q) by propagators
GiNaC::ex substitute_scalar_products_2(const GiNaC::ex& e,
		const IntegralFamily* ic) {
	exmap invar_rules = ic->kinematics()->rules_sp_to_invariants();
	exmap prop_rules = ic->rules_sp_to_prop();
	lst loop_momenta = ic->loop_momenta();
	ex res = e;
	exset found;
	exmap rules;
	res.find(ScalarProduct(wild(1), wild(2)), found);
	for (exset::const_iterator it = found.begin(); it != found.end(); ++it) {
		ex tmp = ex_to<ScalarProduct> (*it).expand();
		tmp = tmp.subs(invar_rules, subs_options::no_pattern);
		tmp = tmp.subs(prop_rules, subs_options::no_pattern);
		rules[*it] = tmp;
	}
	LOGX("   SP: " << rules);
	res = res.subs(rules, subs_options::no_pattern);
	found.clear();
	rules.clear();
	res.find(Propagator(wild(2), wild(1)), found);
	for (exset::const_iterator it = found.begin(); it != found.end(); ++it) {
		if (freeof(*it, loop_momenta)) {
			ex tmp = pow(ex_to<Propagator> (*it).inverse_ex(), -1);
			tmp = tmp.expand().subs(invar_rules, subs_options::no_pattern);
			rules[*it] = tmp;
		}
	}

	res = res.subs(rules, subs_options::no_pattern);

	found.clear();
	res.find(ScalarProduct(wild(1), wild(2)), found);
	if (!found.empty())
		ABORT("Unmatched ScalarProducts: " << found);
	found.clear();
	res.find(Propagator(wild(1), wild(2)), found);
	for (exset::const_iterator it = found.begin(); it != found.end(); ++it) {
		if (freeof(*it, loop_momenta))
			ABORT("Unmatched Propagators without loop momenta: " << found);
	}
	return res.expand();
}

void factor_commutative_from_mul_or_ncmul(const GiNaC::ex& e,
		GiNaC::ex& non_com, GiNaC::ex& com);
void factor_commutative_from_add(const GiNaC::ex&e, GiNaC::ex& non_com,
		GiNaC::ex& com);

// factors a product into a non-commutative and commutative part
void factor_commutative_from_mul_or_ncmul(const GiNaC::ex& e,
		GiNaC::ex& non_com, GiNaC::ex& com) {
	non_com = com = 1;
	VERIFY(is_a<mul>(e) || is_a<ncmul>(e));
	for (size_t i = 0; i < e.nops(); ++i) {
		if (is_a<mul> (e.op(i)) || is_a<ncmul> (e.op(i))) {
			ex non_com2(1), com2(1);
			factor_commutative_from_mul_or_ncmul(e.op(i), non_com2, com2);
			non_com = non_com * non_com2;
			com *= com2;
		} else if (is_a<add> (e.op(i))) {
			ex non_com2(1), com2(1);
			factor_commutative_from_add(e.op(i), non_com2, com2);
			non_com = non_com * non_com2;
			com *= com2;
		} else if (e.op(i).return_type() == return_types::commutative) {
			com *= e.op(i);
		} else {
			non_com = non_com * e.op(i);
		}
	}
}

// factors a sum with common factors into a non-commutative and commutative part
void factor_commutative_from_add(const GiNaC::ex& e, GiNaC::ex& non_com,
		GiNaC::ex& com) {
	non_com = com = 1;
	VERIFY(is_a<add>(e));
	ex tmp = collect_common_factors(e);
	if (is_a<ncmul> (tmp) || is_a<mul> (tmp))
		factor_commutative_from_mul_or_ncmul(tmp, non_com, com);
	else if (tmp.return_type() == return_types::commutative)
		com = tmp;
	else
		non_com = tmp;
}

// expression MUST BE A MUL
void product_to_deltas_and_exvector(const GiNaC::ex& e,
		GiNaC::exvector& deltas, GiNaC::exvector& v) {
	VERIFY(is_a<mul>(e));
	deltas.clear();
	v.clear();
	size_t num = e.nops();
	deltas.reserve(2 * num);
	v.reserve(2 * num);
	for (size_t i = 0; i < num; ++i) {
		ex f = e.op(i);
		if (is_a<power> (f) && f.op(1) == 2) {
			if (is_a<indexed> (f.op(0)) && (is_a<tensdelta> (f.op(0).op(0))
					|| is_a<minkmetric> (f.op(0).op(0)))) {
				deltas.push_back(f.op(0));
				v.push_back(f.op(0));
			} else {
				v.push_back(f.op(0));
				v.push_back(f.op(0));
			}
		} else {
			if (is_a<indexed> (f) && (is_a<tensdelta> (f.op(0)) || is_a<
					minkmetric> (f.op(0))))
				deltas.push_back(f);
			else
				v.push_back(f);
		}
	}
}

// contracts only if the expression is a mul
GiNaC::ex contract_delta_tensors(const GiNaC::ex& e) {
	if (!is_a<mul> (e))
		return e;

	exvector v, deltas;
	product_to_deltas_and_exvector(e, deltas, v);
	bool done_something = false;
	exvector::iterator it1, it2;
	for (it1 = deltas.begin(); it1 != deltas.end(); ++it1) {
		VERIFY( is_a<indexed>(*it1) && (is_a<tensdelta>(it1->op(0)) || is_a<minkmetric>(it1->op(0))) );
		for (it2 = v.begin(); it2 != v.end(); ++it2) {
			if (!is_a<indexed> (*it2))
				continue;
			bool contracted = false;
			size_t num = it2->nops();
			for (size_t i = 1; i < num; ++i) {
				if (is_dummy_pair(it1->op(1), it2->op(i))) {
					it2->let_op(i) = it1->op(2);
					*it1 = 1;
					contracted = true;
					break;
				}
				if (is_dummy_pair(it1->op(2), it2->op(i))) {
					it2->let_op(i) = it1->op(1);
					*it1 = 1;
					contracted = true;
					break;
				}
			}
			if (contracted) {
				done_something = true;
				break;
			}
		} // it2
	} // it1

	ex res(1);
	for (it1 = deltas.begin(); it1 != deltas.end(); it1++)
		res *= *it1;
	for (it1 = v.begin(); it1 != v.end(); it1++)
		res *= *it1;

	if (done_something)
		return contract_delta_tensors(res);
	else
		return res;
}

// decomposes the product:  e = lorentz * color * rest
void factor_indexed(const GiNaC::ex& e, GiNaC::ex& lorentz, GiNaC::ex& color,
		GiNaC::ex& rest) {
	lorentz = color = rest = 1;

	if (is_a<numeric> (e) || is_a<symbol> (e)) {
		rest = e;
		return;
	}

	if (is_a<power> (e)) {
		if (is_a<indexed> (e.op(0)))
			if (is_a<varidx> (e.op(0).op(1)))
				lorentz = e;
			else
				color = e;
		else
			rest = e;
		return;
	}

	if (!is_a<mul> (e))
		ABORT("Not a mul: \n" << e);

	ex result(1);
	for (unsigned i = 0; i < e.nops(); ++i) {
		if (is_a<power> (e.op(i)))
			if (is_a<indexed> (e.op(i).op(0)))
				if (is_a<varidx> (e.op(i).op(0).op(1)))
					lorentz *= e.op(i);
				else
					color *= e.op(i);
			else
				rest *= e.op(i);
		else if (is_a<indexed> (e.op(i)) && is_a<varidx> (e.op(i).op(1)))
			lorentz *= e.op(i);
		else if (is_a<indexed> (e.op(i)) && !is_a<varidx> (e.op(i).op(1)))
			color *= e.op(i);
		else
			rest *= e.op(i);
	}

	color = contract_delta_tensors(color);
	lorentz = contract_delta_tensors(lorentz);
}

/*
GiNaC::ex to_Lorentz_lst(const GiNaC::ex& ea) {
	ex e = ea.eval(); // important: removes f^aac etc \todo understand why
	lst l;
	if (is_a<mul> (e)) {
		for (size_t i = 0; i < e.nops(); ++i)
			l.append(e.op(i));
	} else {
		l.append(e);
	}
	return Lorentz(l);
}
*/

GiNaC::ex to_Color_lst(const GiNaC::ex& ea) {
	ex e = ea.eval(); // important: removes f^aac etc \todo understand why
	lst l;
	if (is_a<mul> (e)) {
		for (size_t i = 0; i < e.nops(); ++i)
			l.append(e.op(i));
	} else {
		l.append(e);
	}
	return Color(l);
}

GiNaC::ex bracket_color(const GiNaC::ex& e) {
	ex e_expanded = e.expand();
	ex result(0);
	if (is_a<add> (e_expanded)) {
		for (size_t i = 0; i < e_expanded.nops(); ++i) {
			ex lor(1), col(1), rest(1);
			ex tmp = e_expanded.op(i);
			factor_indexed(tmp, lor, col, rest);
			result += rest * lor * to_Color_lst(col);
		}
	} else {
		ex lor(1), col(1), rest(1);
		factor_indexed(e_expanded, lor, col, rest);
		result += rest * lor * to_Color_lst(col);
	}
	return result;
}

/*
GiNaC::ex bracket_lorentz(const GiNaC::ex& e, bool is_expanded) {
	//Timer timer;
	ex e_expanded = is_expanded ? e : e.expand();
	//LOGX("    - Bracket lorentz structures: Time to expand: " << timer.get_wall_time() << "s");

	//timer.restart();
	ex result(0);
	if (is_a<add> (e_expanded)) {
		for (size_t i = 0; i < e_expanded.nops(); ++i) {
			ex lor(1), col(1), rest(1);
			ex tmp = e_expanded.op(i);
			factor_indexed(tmp, lor, col, rest);
			result += rest * to_Lorentz_lst(lor) * col;
		}
	} else {
		ex lor(1), col(1), rest(1);
		factor_indexed(e_expanded, lor, col, rest);
		result += rest * to_Lorentz_lst(lor) * col;
	}
	//LOGX("    - Bracket lorentz structures: rest: " << timer.get_wall_time() << "s");
	return result;
}
*/

struct calc_lorentz_new : public map_function {
	ex operator()(const ex &e) {
		if (is_a<add>(e)) {
			return e.map(*this);
		} else if (is_a<mul>(e) && e.nops() >= 2) {
			const_iterator t;
			size_t nadd = 0;
			for (t = e.begin() ; t != e.end() && nadd < 2; ++t)
				if (is_a<add>(*t))
					++nadd;
			if (nadd < 2)
				return substitute_scalar_products(e.expand().simplify_indexed());
			ex pre = 1;
			for (t = e.begin() ; t != e.end() && !is_a<add>(*t); ++t)
				pre *= *t;
			const_iterator i = t;
			VERIFY(i != e.end());
			for (++t ; t != e.end() && !is_a<add>(*t); ++t)
				pre *= *t;
			const_iterator j = t;
			VERIFY(j != e.end());
			ex rest = 1;
			for (++t ; t != e.end() ; ++t)
				if (!is_a<add>(*t))
					pre *= *t;
				else
					rest *= *t;
			pre = substitute_scalar_products(pre.expand().simplify_indexed());
			rest = (*this)(rest);
        	ex res = 0;
        	for (const_iterator a = i->begin() ; a != i->end() ; ++a)
        		for (const_iterator b = j->begin() ; b != j->end() ; ++b)
        			res += (*this)(pre*(*a)*(*b));
        	return (*this)(res*rest);
        } else if (is_a<mul>(e) && e.nops() == 2 &&
        		(is_a<add>(e.op(0)) || is_a<add>(e.op(1)))) {
#ifdef NEW_GINAC
            const GiNaC::ex& a = is_a<add>(e.op(0)) ? e.op(0) : lst({e.op(0)});
            const GiNaC::ex& b = is_a<add>(e.op(1)) ? e.op(1) : lst({e.op(1)});
#else
            const GiNaC::ex& a = is_a<add>(e.op(0)) ? e.op(0) : lst(e.op(0));
            const GiNaC::ex& b = is_a<add>(e.op(1)) ? e.op(1) : lst(e.op(1));
#endif
        	ex res = 0;
        	for (const_iterator i = a.begin() ; i != a.end() ; ++i)
        		for (const_iterator j = b.begin() ; j != b.end() ; ++j)
        			res += (*this)((*i)*(*j));
        	return res;
        } else {
        	return substitute_scalar_products(e.expand().simplify_indexed());
        }
	}
};
}

AmplitudeProd::AmplitudeProd(const Amplitude& amp1, const Amplitude& amp2a,
		const FeynmanRules* fr) :
	fr_(fr) {
	Amplitude amp2(amp2a);

	// set the name of the product
	name_ = amp1.name() + "_X_" + amp2.name();

	// make sure no internal dummy index appears in both amplitudes
	amp2.make_new_index_names();

	LOGX("First amplitude:");
	LOGX(amp1);
	LOGX("Second amplitude (already conjugated):");
	LOGX(amp2);

	// insert closed spinor chains, take overall commutative factor
	ex col(1);
	col *= insert(amp1.closed_fermion_chains());
	col *= insert(amp2.closed_fermion_chains());

	try {
		// close and insert the open fermion chains, sum over all spins (completeness relation)
		col *= insert(
				ClosedFermionChain::close_open_fermion_chains(
						amp1.open_fermion_chains(), amp2.open_fermion_chains(),
						fr_));
	} catch (std::exception& e) {
		ABORT("Failed to multiply amplitudes: " << e.what());
	}

	// polarization sum of external bosons
	const exvector& v = amp1.external_bosons();
	const exvector& v2 = amp2.external_bosons();
	//v.insert(v.end(), v2.begin(), v2.end());
	ex external = Reduze::polarization_sum(v, v2, fr_);

	// the amplitude without the spinor traces
	amp_ = amp1.amp() * amp2.amp() * external * col;

	// define the sector of the product (if both amps define a sector)
	try {
		set_sector(Sector(amp1.sector(), amp2.sector()));
		loop_momenta_ = sector().integralfamily()->loop_momenta();
	} catch (std::exception& e) {
		size_t nloop = amp1.loop_momenta().nops() + amp2.loop_momenta().nops();
		const Kinematics* kin = Files::instance()->kinematics();
		string pre = kin->loop_momentum_prefix();
		for (size_t i = 1; i <= nloop; ++i)
			loop_momenta_.append(GiNaC::symbol(pre + to_string(i)));
	}

	// replace the local loop momenta with those of the integral family
	lst loop_momenta1 = amp1.loop_momenta();
	lst loop_momenta2 = amp2.loop_momenta();
	exmap old2new_mom;
	ASSERT(loop_momenta_.nops() == loop_momenta1.nops() + loop_momenta2.nops());
	lst::const_iterator it, lit = loop_momenta_.begin();
	for (it = loop_momenta1.begin(); it != loop_momenta1.end(); ++it, ++lit)
		old2new_mom[*it] = *lit;
	for (it = loop_momenta2.begin(); it != loop_momenta2.end(); ++it, ++lit)
		old2new_mom[*it] = *lit;
	ASSERT(loop_momenta_.nops() == old2new_mom.size());

	for (exvector::iterator it = spinor_traces_.begin(); it
			!= spinor_traces_.end(); ++it)
		*it = it->subs(old2new_mom, subs_options::no_pattern);
	amp_ = amp_.subs(old2new_mom, subs_options::no_pattern);
}

void AmplitudeProd::set_sector(const Sector& sector) {
	sector_.clear();
	sector_.push_front(sector);
}

const Sector& AmplitudeProd::sector() const {
	if (sector_.empty())
		throw std::runtime_error("No sector available");
	return sector_.front();
}

EquationInfo AmplitudeProd::calculate(LinearCombination& lincomb) const {
	LOG("Calculation of Amplitude " << name_ << " requested");
	Timer timer;

	EquationInfo info;
	info.amplitude_name_ = name();

	ex result_nodirac = 1;
	ex result_prefac = 1;
	LOG(" * Extracting scalar prefactor");
	if (is_a<mul> (amp_)) {
		for (GiNaC::const_iterator t = amp_.begin(); t != amp_.end(); ++t) {
			if (t->get_free_indices().empty() && !has_object<indexed> (*t))
				result_prefac *= *t;
			else
				result_nodirac *= *t;
		}
	} else {
		LOG("   no product found");
		result_nodirac = amp_;
	}
	LOGX("   found scalar prefactor: " << result_prefac);

	LOG(" * Calculating the Color structure");
	timer.restart();
	int max_color_order;
	LOGX("color: " << result_nodirac);
	result_nodirac = calc_color(result_nodirac, max_color_order); // expanded
	LOG(" * Color structure calculated! Time: " << timer.get_wall_time() << "s");
	info.max_color_order_ = max_color_order;
	LOG("   Color order: Nc^" << max_color_order);

	if (sector_.empty()) {
		LOG("No sector found ! Premature exit.");
		info.valid_equation_ = false;
		return info;
	}

	LOG(" * Calculating the Dirac traces");
	timer.restart();
	// idea: we could leave out e.g. the propagators here and
	//       only multiply them at the very end
	ex result_nopre = calc_dirac(spinor_traces_, result_nodirac);
	LOG(" * Dirac traces calculated! Time: " << timer.get_wall_time() << "s");

	LOG(" * Contracting remaining Lorentz indices");
	timer.restart();
	// note: expansion needed for Lorentz contractions easily leads to heavy
	// RAM and run time issues when done naively, be careful here
	//cout << "*******" << endl << result_nopre << endl << "*******" << endl;
//	result_nopre = result_nopre.simplify_indexed().expand().simplify_indexed();
//	result_nopre = calc_lorentz(result_nopre.simplify_indexed(), false);
	result_nopre = calc_lorentz_new()(result_nopre);
	//cout << "*******" << endl << result_nopre << endl << "*******" << endl;
	ex result = result_nopre * result_prefac;
	LOG(" * Lorentz indices contracted! Time: " << timer.get_wall_time() << "s");

	LOG(" * Replacing scalarproducts by invariants and propagators");
	timer.restart();
	// idea: calculate everything up to identification of integral family in last step:
	//       we could thus calculate r, s, t even if no family known
	const IntegralFamily* ic = sector().integralfamily();
	result = substitute_scalar_products_2(result, ic);
	LOG(" * Matching integrals");
	propagator_products_to_integrals(result, lincomb, ic);
	//  let a later job handle this since it can be time consuming:
	//	LOG(" * Applying shift reductions");
	//	lincomb.apply_shifts(true);
	LOG(" * Scalar products replaced, integrals matched! Time: " << timer.get_wall_time() << "s");

	LOG(" * Collecting common factors");
	timer.restart();
	LinearCombination tmp = lincomb;
	lincomb.clear();
	for (LinearCombination::const_iterator t = tmp.begin(); t != tmp.end(); ++t) {
		ex e = t->second;
		//e = e.subs(Color(wild(1)) == wild(1));
		e = collect_common_factors(e);
		lincomb.insert(t->first, e);
	}
	LOG(" * Common factors collected ! Time: " << timer.get_wall_time() << "s");

	exmap clean_up;
	clean_up[Ident(wild(0))] = wild(0);
	lincomb.apply(clean_up);

	return info;
}

// private

GiNaC::ex AmplitudeProd::insert(const std::vector<ClosedFermionChain>& lines) {
	ex result(1);
	vector<ClosedFermionChain>::const_iterator it;
	for (it = lines.begin(); it != lines.end(); ++it) {
		const ex e = it->value();
		VERIFY(!has_object<Spinor> (e));
		ex trace(1), coeff(1);
		if (is_a<add> (e))
			factor_commutative_from_add(e, trace, coeff);
		else if (is_a<mul> (e) || is_a<ncmul> (e))
			factor_commutative_from_mul_or_ncmul(e, trace, coeff);
		else if (e.return_type() == return_types::commutative)
			coeff = e;
		else
			trace = e;
		VERIFY((e - coeff * trace).expand().is_zero());
		result *= coeff;
		if (!trace.is_equal(1)) {
			spinor_traces_.push_back(trace);
			VERIFY(trace.return_type() != return_types::commutative);
		}
	}
	return result;
}

// note: dirac_trace() is very expensive for large number of gamma matrices:
//       e.g. tr(12 gammas) takes 2 s on a 3GHz CPU
//            tr(14 gammas) takes >> 1 min. on a 3GHz CPU
//       => important to perform contractions with metric tensors before
//          evaluating the trace (also avoids intermediate blow up in number
//          of terms, tr(12 gammas) gives already 10395 terms)
GiNaC::ex AmplitudeProd::calc_dirac(const GiNaC::exvector& traces,
		const GiNaC::ex& prefactor) const {
	ex pre = prefactor.expand().simplify_indexed();
	pre = substitute_scalar_products(pre);

	LOG("   performing " << traces.size() << " Dirac trace(s)");
	if (traces.empty())
		return pre;
	ex result(1);
	exvector::const_iterator it;
	// multiply prefactor with longest fermion chain
	// (for which contraction with metric tensors might be most important)
	exvector::const_iterator longest = traces.begin();
	for (it = traces.begin(); it != traces.end(); ++it) {
		if (it->nops() > longest->nops())
			longest = it;
	}
	for (it = traces.begin(); it != traces.end(); ++it) {
		ex expanded = it->expand();
		if (!is_a<add> (expanded)) // handle special case of only one term
#ifdef NEW_GINAC
            expanded = lst({expanded});
#else
            expanded = lst(expanded);
#endif
		ex curr = 0;
		LOGN("   " << expanded.nops() << " sub trace(s): ");
		for (const_iterator i = expanded.begin(); i != expanded.end(); ++i) {
			LOGN(".");
			Timer timer_local;
			ex term = *i;
			if (it == longest)
				term = (term * pre).expand().simplify_indexed();
			VERIFY(has_object<clifford> (term) || term.is_zero());
			ex term_tr = dirac_trace(term);
			term_tr = term_tr.expand().simplify_indexed();
			term_tr = substitute_scalar_products(term_tr);
			curr += term_tr;
			LOGX(" time: " << timer_local.get_wall_time() << " s");
		}
		result *= curr;
		LOG("");
	}

	return result;
}

GiNaC::ex AmplitudeProd::calc_color(const GiNaC::ex& col, int& Nc_degree) const {
	ex result = bracket_color(col);
	exset found;
	result.find(Color(wild()), found);
	LOGX(" * Color structures: \n" << found << "\nnumber " << found.size());
	ex Na = Files::instance()->globalsymbols()->Na();
	ex Nc = Files::instance()->globalsymbols()->Nc();
	exmap Na2Nc;
	Na2Nc[Na] = pow(Nc, 2) - 1;
	Na2Nc[Ident(wild(1))] = wild(1);

	ex resultLeadingNc = result;
	for (exset::const_iterator it = found.begin(); it != found.end(); ++it) {
		VERIFY(is_a<GiNaC::function>(*it) && ex_to<GiNaC::function>(*it).get_name() == "Color");
		VERIFY(is_a<lst>(it->op(0)));
		ex res(1);
		for (size_t i = 0; i < it->op(0).nops(); ++i)
			res *= it->op(0).op(i);
		exvector v = res.get_free_indices();
		// ex old_res = res;
		if (!v.empty()) {
			//ABORT("Free indices detected: " << v << " in " << res);
			WARNING("Free indices detected: " << v << " in " << res);
		}
		res = res.simplify_indexed();
		res = res.subs(Na2Nc, subs_options::no_pattern).expand();
		int degree = res.degree(Nc);
		// factor result
		res = factor_rational(res);
		lst num_rest = extract_numerical_factor(res);
		res = num_rest[0] * Color(num_rest[1]);
		result = result.subs(*it == res, subs_options::algebraic);
		resultLeadingNc = resultLeadingNc.subs(*it == pow(Nc, degree),
				subs_options::algebraic);
		LOGX("   color result (degree Nc^" << degree << "): " << *it << " == " << res);
	}
	Nc_degree = resultLeadingNc.degree(Nc);

	return result;
}

// DEPRECATED
/*
GiNaC::ex AmplitudeProd::calc_lorentz(const GiNaC::ex& lor, bool is_expanded) const {
	//const scalar_products sp =
	//		Files::instance()->kinematics()->momenta_to_invariants();
	// using this for simplify_indexed(sp) doesn't work (probably because
	// scalar products are replaced by our ScalarProduct() structs too early)
	Timer timer;
	ex result = bracket_lorentz(lor, is_expanded);
	LOGX("    - Time to bracket lorentz structures: " << timer.get_wall_time() << "s");
	timer.restart();
	exset found;
	result.find(Lorentz(wild()), found);
	LOGX("    - Lorentz structures: " << found.size());
	LOGX("    - Time to find lorentz structures: " << timer.get_wall_time() << "s");

	timer.restart();
	exmap m;
	for (exset::const_iterator it = found.begin(); it != found.end(); ++it) {
		VERIFY(is_a<function>(*it) && ex_to<function>(*it).get_name() == "Lorentz");
		VERIFY(is_a<lst>(it->op(0)));

		ex res(1);
		for (size_t i = 0; i < it->op(0).nops(); ++i)
			res *= it->op(0).op(i);
		exvector v = res.get_free_indices();
		if (!v.empty())
			ABORT("Free indices detected: " << v << " in " << res);
		res = res.simplify_indexed();
		m[*it] = res;
	}

	LOGX("    - Map created: " << timer.get_wall_time() << "s");
	timer.restart();
	result = result.subs(m, subs_options::no_pattern);
	LOGX("    - Substituted: " << timer.get_wall_time() << "s");

	found.clear();
	result.find(Lorentz(wild()), found);

	if (!found.empty())
		ABORT("Still lorentz structures in the result: " << found);

	return result;
}
*/
ostream& operator <<(ostream& os, const AmplitudeProd& amp) {
	os << "Multiplied amplitudes " << amp.name() << endl;
	exvector::const_iterator it2;
	for (unsigned i = 0; i < amp.spinor_traces_.size(); ++i) {
		os << "Chain: " << i << endl;
		os << amp.spinor_traces_[i] << endl;
	}
	os << "Rest" << endl;
	os << amp.amp_;
	return os;
}

}
// namespace Reduze
