/*  feynmanrules.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "ginac/ginac.h"
#include "feynmanrules.h"
#include "functions.h"
#include "ginacutils.h"
#include "files.h"
#include "globalsymbols.h"
#include "color.h"
#include "polarization.h"
//#include "kinematics.h"

using namespace std;
using namespace GiNaC;

namespace Reduze {

// register type
namespace {
YAMLProxy<FeynmanRules> dummy;
}

// color

static ex delta_a_eval(const ex& a1, const ex& a2) {
	ex Na = Files::instance()->globalsymbols()->Na();
	ex Nc = Files::instance()->globalsymbols()->Nc();
	return delta_tensor(idx2(a1, Na, Nc), idx2(a2, Na, Nc));
}
static ex delta_f_eval(const ex& i1, const ex& i2) {
	ex Nc = Files::instance()->globalsymbols()->Nc();
	return delta_tensor(idx(i1, Nc), idx(i2, Nc));
}

static ex ColT_eval(const ex& a, const ex& i1, const ex& i2) {
	ex Na = Files::instance()->globalsymbols()->Na();
	ex Nc = Files::instance()->globalsymbols()->Nc();
	return (sunt(idx2(a, Na, Nc), idx(i1, Nc), idx(i2, Nc)));
}
static ex ColF3_eval(const ex& a1, const ex& a2, const ex& a3) {
	ex Na = Files::instance()->globalsymbols()->Na();
	ex Nc = Files::instance()->globalsymbols()->Nc();
	return sunf(idx2(a1, Na, Nc), idx2(a2, Na, Nc), idx2(a3, Na, Nc));
}
static ex ColF4_eval(const ex& a1, const ex& a2, const ex& a3, const ex& a4) {
	ex Na = Files::instance()->globalsymbols()->Na();
	ex Nc = Files::instance()->globalsymbols()->Nc();
	return sunf(idx2(a1, Na, Nc), idx2(a2, Na, Nc), idx2(a3, Na, Nc),
			idx2(a4, Na, Nc));
}

REGISTER_FUNCTION(delta_a, eval_func(delta_a_eval))
REGISTER_FUNCTION(delta_f, eval_func(delta_f_eval))
REGISTER_FUNCTION(ColT, eval_func(ColT_eval))
REGISTER_FUNCTION(ColF3, eval_func(ColF3_eval))
REGISTER_FUNCTION(ColF4, eval_func(ColF4_eval))

// external legs

static ex U_eval(const ex& momentum, const ex& mass, const ex& i) {
	VERIFY(is_a<symbol>(i) || i.is_zero());
	ex Nc = Files::instance()->globalsymbols()->Nc();
	return u_wrapper(four_vector_tensor(momentum), mass, idx(i, Nc));
}

static ex Ubar_eval(const ex& momentum, const ex& mass, const ex& i) {
	VERIFY(is_a<symbol>(i) || i.is_zero());
	ex Nc = Files::instance()->globalsymbols()->Nc();
	return ubar_wrapper(four_vector_tensor(momentum), mass, idx(i, Nc));
}

static ex V_eval(const ex& momentum, const ex& mass, const ex& i) {
	VERIFY(is_a<symbol>(i) || i.is_zero());
	ex Nc = Files::instance()->globalsymbols()->Nc();
	return v_wrapper(four_vector_tensor(momentum), mass, idx(i, Nc));
}

static ex Vbar_eval(const ex& momentum, const ex& mass, const ex& i) {
	VERIFY(is_a<symbol>(i) || i.is_zero());
	ex Nc = Files::instance()->globalsymbols()->Nc();
	return vbar_wrapper(four_vector_tensor(momentum), mass, idx(i, Nc));
}
static ex Pol_gluon_eval(const ex& momentum, const ex& mass, const ex& mu,
		const ex& a, const ex& ortho) {
	ex Na = Files::instance()->globalsymbols()->Na();
	ex Nc = Files::instance()->globalsymbols()->Nc();
	ex Dd = Files::instance()->globalsymbols()->d();
	// mu  = covariant
	return pol_gluon_wrapper(four_vector_tensor(momentum), mass,
			varidx(mu, Dd, true), idx2(a, Na, Nc), four_vector_tensor(ortho));
}
static ex PolC_gluon_eval(const ex& momentum, const ex& mass, const ex& mu,
		const ex& a, const ex& ortho) {
	ex Na = Files::instance()->globalsymbols()->Na();
	ex Nc = Files::instance()->globalsymbols()->Nc();
	ex Dd = Files::instance()->globalsymbols()->d();
	// mu  = covariant
	return polc_gluon_wrapper(four_vector_tensor(momentum), mass,
			varidx(mu, Dd, true), idx2(a, Na, Nc), four_vector_tensor(ortho));
}
static ex Pol_ghost_eval(const ex& momentum, const ex& mass, const ex& a) {
	ex Na = Files::instance()->globalsymbols()->Na();
	ex Nc = Files::instance()->globalsymbols()->Nc();
	return pol_ghost_wrapper(four_vector_tensor(momentum), mass,
			idx2(a, Na, Nc));
}
static ex PolC_ghost_eval(const ex& momentum, const ex& mass, const ex& a) {
	ex Na = Files::instance()->globalsymbols()->Na();
	ex Nc = Files::instance()->globalsymbols()->Nc();
	return polc_ghost_wrapper(four_vector_tensor(momentum), mass,
			idx2(a, Na, Nc));
}
static ex Pol_photon_eval(const ex& momentum, const ex& mass, const ex& mu) {
	ex Dd = Files::instance()->globalsymbols()->d();
	// mu  = covariant
	return pol_photon_wrapper(four_vector_tensor(momentum), mass,
			varidx(mu, Dd, true));
}
static ex PolC_photon_eval(const ex& momentum, const ex& mass, const ex& mu) {
	ex Dd = Files::instance()->globalsymbols()->d();
	// mu  = covariant
	return polc_photon_wrapper(four_vector_tensor(momentum), mass,
			varidx(mu, Dd, true));
}

REGISTER_FUNCTION(U, eval_func(U_eval))
REGISTER_FUNCTION(Ubar, eval_func(Ubar_eval))
REGISTER_FUNCTION(V, eval_func(V_eval))
REGISTER_FUNCTION(Vbar, eval_func(Vbar_eval))

REGISTER_FUNCTION(Pol_gluon, eval_func(Pol_gluon_eval))
REGISTER_FUNCTION(PolC_gluon, eval_func(PolC_gluon_eval))
REGISTER_FUNCTION(Pol_ghost, eval_func(Pol_ghost_eval))
REGISTER_FUNCTION(PolC_ghost, eval_func(PolC_ghost_eval))
REGISTER_FUNCTION(Pol_photon, eval_func(Pol_photon_eval))
REGISTER_FUNCTION(PolC_photon, eval_func(PolC_photon_eval))

// minkowski metric

#define TOGGLE(x)  x=ex_to<varidx>(x.toggle_variance())

static GiNaC::ex metric_l_eval(const GiNaC::ex& mu1, const GiNaC::ex& mu2) {
	ex Dd = Files::instance()->globalsymbols()->d();
	varidx i1(mu1, Dd), i2(mu2, Dd);

	// handle special case of two equal indices

	if (mu1.is_equal(mu2))
		TOGGLE(i2);

	return lorentz_g(i1, i2);
}

static GiNaC::ex metric_l_diff_eval(const GiNaC::ex& mu1, const GiNaC::ex& mu2,
		const GiNaC::ex& mu3, const GiNaC::ex& mu4) {
	ex Dd = Files::instance()->globalsymbols()->d();
	varidx i1(mu1, Dd), i2(mu2, Dd), i3(mu3, Dd), i4(mu4, Dd);

	// handle special cases of two equal indices

	if (mu1.is_equal(mu2)) {
		TOGGLE(i2);
		if (mu3.is_equal(mu4))
			TOGGLE(i4);
	} else if (mu1.is_equal(mu3)) {
		TOGGLE(i3);
		if (mu2.is_equal(mu4))
			TOGGLE(i4);
	} else if (mu1.is_equal(mu4)) {
		TOGGLE(i4);
		if (mu2.is_equal(mu3))
			TOGGLE(i3);
	} else if (mu2.is_equal(mu3)) {
		TOGGLE(i3);
		if (mu1.is_equal(mu4))
			TOGGLE(i4);
	} else if (mu2.is_equal(mu4)) {
		TOGGLE(i4);
		if (mu1.is_equal(mu3))
			TOGGLE(i3);
	} else if (mu3.is_equal(mu4)) {
		TOGGLE(i4);
		if (mu1.is_equal(mu2))
			TOGGLE(i2);
	}

	if (true) {
		return metric_l_diff_wrapper(i1, i2, i3, i4);
	} else {
		return lorentz_g(i1, i2) * lorentz_g(i3, i4) //
		- lorentz_g(i1, i4) * lorentz_g(i2, i3);
	}
}

REGISTER_FUNCTION(metric_l, eval_func(metric_l_eval))
REGISTER_FUNCTION(metric_l_diff, eval_func(metric_l_diff_eval))

// indexed four vector

static GiNaC::ex Mom_eval(const GiNaC::ex& p, const GiNaC::ex& mu) {
	ex Dd = Files::instance()->globalsymbols()->d();
	return four_vector_tensor_wrapper(p, varidx(mu, Dd));
}

REGISTER_FUNCTION(Mom, eval_func(Mom_eval))

// gamma matrices

static ex GaOne_eval(const ex& e) {
	return dirac_ONE();
}
static GiNaC::ex Ga_eval(const GiNaC::ex& mu) {
	ex Dd = Files::instance()->globalsymbols()->d();
	return dirac_gamma(varidx(mu, Dd));
}
static GiNaC::ex GaS_eval(const GiNaC::ex& momentum) {
	ex Dd = Files::instance()->globalsymbols()->d();
	return dirac_slash(four_vector_tensor(momentum), Dd);
}

// dirac
REGISTER_FUNCTION(GaOne, eval_func(GaOne_eval))
REGISTER_FUNCTION(GaS, eval_func(GaS_eval))
REGISTER_FUNCTION(Ga, eval_func(Ga_eval))

//
//
//

FeynmanRules::FeynmanRules() :
		momentum_prefix_("mom"), mass_prefix_("mass"), polarization_orthogonal_vector_prefix_(
				"ortho"), max_vertex_degree_(4) {
	index_prefixes_[IndexFlags::color_a] = "a";
	index_prefixes_[IndexFlags::color_f] = "i";
	index_prefixes_[IndexFlags::lorentz] = "mu";
}

FeynmanRules::~FeynmanRules() {
}

// \todo verify YAML specification by hand
void FeynmanRules::read(const YAML::Node& node) {
	using namespace YAML;
	verify_yaml_spec(node);

	// the symbols "coupling_constants", "su_n_dimensions" should be defined
	// in the file global.yaml in the new Reduze version 2.1
	// if they are defined in the feynman rules file they will be read in by
	// GlobalSymbols but will get lost when a FeynmanRules object is printed.
	// However, we don't print it.

	if (node.FindValue("coupling_constants")) {
		WARNING(
				"FeynmanRules interface changed: define the coupling constants in the file global.yaml !");
	}
	if (node.FindValue("su_n_dimensions")) {
		WARNING(
				"FeynmanRules interface changed: define the su(N) dimensions in the file global.yaml !");
	}

	if (node.FindValue("momentum_prefix"))
		node["momentum_prefix"] >> momentum_prefix_;
	if (node.FindValue("mass_prefix"))
		node["mass_prefix"] >> mass_prefix_;
	if (node.FindValue("polarization_orthogonal_vector_prefix"))
		node["polarization_orthogonal_vector_prefix"]
				>> polarization_orthogonal_vector_prefix_;
	string s;
	const Node& prefix_node = node["index_prefixes"];
	if (prefix_node.FindValue("lorentz") != 0) {
		prefix_node["lorentz"] >> s;
		index_prefixes_.insert(
				make_pair(static_cast<int>(IndexFlags::lorentz), s));
	}
	if (prefix_node.FindValue("color_a") != 0) {
		prefix_node["color_a"] >> s;
		index_prefixes_.insert(
				make_pair(static_cast<int>(IndexFlags::color_a), s));
	}
	if (prefix_node.FindValue("color_f") != 0) {
		prefix_node["color_f"] >> s;
		index_prefixes_.insert(
				make_pair(static_cast<int>(IndexFlags::color_f), s));
	}
	if (node.FindValue("max_vertex_degree"))
		node["max_vertex_degree"] >> max_vertex_degree_;

	init_symbols();

	if (node.FindValue("external_legs")) {
		const Node& n_e = node["external_legs"];
		for (Iterator i = n_e.begin(); i != n_e.end(); ++i) {
			string id, rule;
			i.first() >> id;
			i.second() >> rule;
			insert_rule(id, FeynmanRulesType::external, rule);
		}
	}
	const Node& n_p = node["propagators"];
	for (Iterator i = n_p.begin(); i != n_p.end(); ++i) {
		string id, rule;
		i.first() >> id;
		i.second() >> rule;
		insert_rule(id, FeynmanRulesType::propagator, rule);
	}
	const Node& n_n = node["vertices"];
	for (Iterator i = n_n.begin(); i != n_n.end(); ++i) {
		string id, rule;
		i.first() >> id;
		i.second() >> rule;
		insert_rule(id, FeynmanRulesType::vertex, rule);
	}

	if (node.FindValue("polarization_sums")) {
		const Node& n_pol = node["polarization_sums"];
		for (Iterator i = n_pol.begin(); i != n_pol.end(); ++i) {
			string key, pol, polc, val;
			(*i)[0] >> pol;
			(*i)[1] >> polc;
			(*i)[2] >> val;
			insert_polarization_sum(pol, polc, val);
		}
	}
}

void FeynmanRules::print(YAML::Emitter& os) const {
	using namespace YAML;
	os << BeginMap;

	// print FeynmanRules member

	os << Key << "momentum_prefix" << Value << momentum_prefix_;
	os << Key << "mass_prefix" << Value << mass_prefix_;
	os << Key << "polarization_orthogonal_vector_prefix" << Value
			<< polarization_orthogonal_vector_prefix_;

	os << Key << "index_prefixes" << Value;
	os << BeginMap;
	map<int, string>::const_iterator i;
	if ((i = index_prefixes_.find(IndexFlags::lorentz))
			!= index_prefixes_.end())
		os << Key << "lorentz" << Value << i->second;
	if ((i = index_prefixes_.find(IndexFlags::color_a))
			!= index_prefixes_.end())
		os << Key << "color_a" << Value << i->second;
	if ((i = index_prefixes_.find(IndexFlags::color_f))
			!= index_prefixes_.end())
		os << Key << "color_f" << Value << i->second;
	os << EndMap;

	os << Key << "max_vertex_degree" << Value << max_vertex_degree_;

	map<string, string>::const_iterator m;
	os << Key << "external_legs" << Value;
	os << BeginMap;
	for (m = unparsed_rules_.begin(); m != unparsed_rules_.end(); ++m)
		if (rules_type_.at(m->first) == FeynmanRulesType::external)
			os << Key << m->first << Value << DoubleQuoted << m->second;
	os << EndMap;

	os << Key << "propagators" << Value;
	os << BeginMap;
	for (m = unparsed_rules_.begin(); m != unparsed_rules_.end(); ++m)
		if (rules_type_.at(m->first) == FeynmanRulesType::propagator)
			os << Key << m->first << Value << DoubleQuoted << m->second;
	os << EndMap;

	os << Key << "vertices" << Value;
	os << BeginMap;
	for (m = unparsed_rules_.begin(); m != unparsed_rules_.end(); ++m)
		if (rules_type_.at(m->first) == FeynmanRulesType::vertex)
			os << Key << m->first << Value << DoubleQuoted << m->second;
	os << EndMap;

	// \todo print polarization sums

	os << EndMap;
}

symbol FeynmanRules::mom(int i) const {
	return local_symbol(momentum_prefix_ + to_string(i));
}

symbol FeynmanRules::mass(int i) const {
	return local_symbol(mass_prefix_ + to_string(i));
}

symbol FeynmanRules::polarization_orthogonal_vector(int i) const {
	return local_symbol(polarization_orthogonal_vector_prefix_ + to_string(i));
}

symbol FeynmanRules::index(const std::string& s, int i) const {
	return local_symbol(s + to_string(i));
}

symbol FeynmanRules::local_symbol(const string& s) const {
	map<string, symbol>::const_iterator it = symbols_by_name_.find(s);
	if (it == symbols_by_name_.end())
		ABORT("Symbol " << s << " not found");
	return it->second;
}

void FeynmanRules::init_symbols() {
	map<int, string>::const_iterator it;
	set<string> prefixes;
	for (it = index_prefixes_.begin(); it != index_prefixes_.end(); ++it) {
		string pref = it->second;
		assert_valid_prefix_name(pref);
		prefixes.insert(pref);
		ps_symbols_.append(symbol(pref));
		for (int i = 1; i <= max_vertex_degree_; ++i) {
			string s_name(pref + to_string(i));
			symbol s(s_name);
			symbols_.append(s);
			symbols_by_name_.insert(make_pair(s_name, s));
			if (i < 3)
				ps_symbols_.append(symbol(s_name)); // new name for polarizations
		}
	}
	if (max_vertex_degree_ < 3)
		throw runtime_error("max_vertex_degree must be greater than 2");

	assert_valid_prefix_name(momentum_prefix_);
	assert_valid_prefix_name(mass_prefix_);
	assert_valid_prefix_name(polarization_orthogonal_vector_prefix_);
	prefixes.insert(momentum_prefix_);
	prefixes.insert(mass_prefix_);
	prefixes.insert(polarization_orthogonal_vector_prefix_);
	if (prefixes.size() != index_prefixes_.size() + 3)
		throw runtime_error("multiple declaration of prefixes.");

	for (int i = 1; i <= max_vertex_degree_; ++i) {
		string mom_str = momentum_prefix_ + to_string(i);
		symbol mom = symbol(mom_str);
		symbols_.append(mom);
		symbols_by_name_.insert(make_pair(mom_str, mom));

		string mass_str = mass_prefix_ + to_string(i);
		symbol mass = symbol(mass_str);
		symbols_.append(mass);
		symbols_by_name_.insert(make_pair(mass_str, mass));

		string ortho_str = polarization_orthogonal_vector_prefix_
				+ to_string(i);
		symbol ortho = symbol(ortho_str);
		symbols_.append(ortho);
		symbols_by_name_.insert(make_pair(ortho_str, ortho));

		if (i == 1) {
			ps_symbols_.append(symbol(momentum_prefix_));
			ps_symbols_.append(symbol(mass_prefix_));
			ps_symbols_.append(symbol(polarization_orthogonal_vector_prefix_));
		}
	}
}

void FeynmanRules::assert_allowed_external_function(const GiNaC::ex& e) const {
	static string allowed(
			"U, Ubar, V, Vbar, Pol_gluon, PolC_gluon, Pol_ghost, PolC_ghost, Pol_photon, PolC_photon");

	if (is_a<indexed>(e) && e.nops() > 0 && is_a<Polarization>(e.op(0)))
		return;

	ABORT(
			"Invalid Feynman rule for external leg: " << e << "\nrule must be one of the functions: " << allowed);
}

void FeynmanRules::insert_rule(const string& id, int type, const string& rule) {
	const GlobalSymbols* fg = Files::instance()->globalsymbols();
	lst all_symbols = add_lst(symbols_, fg->all());

	ex rule_ex;
	try {
		ex tmp(rule, all_symbols);
		rule_ex.swap(tmp);
	} catch (std::exception& e) {
		throw runtime_error(
				"Failed to parse Feynman rule " + id + "\n" + rule + "\n"
						+ e.what());
	}

	if (type == FeynmanRulesType::external)
		assert_allowed_external_function(rule_ex);

	if (type == FeynmanRulesType::propagator)
		rule_ex = toggle_non_numeric_varidx(rule_ex);

	// \todo check for equal varidx's in different indexed objects
	// and toggle variance of one of them

	insert_rule(id, type, rule_ex);
	unparsed_rules_.insert(make_pair(id, rule));
}

void FeynmanRules::insert_rule(const string& id, int type,
		const GiNaC::ex& rule) {
	if (rules_.find(id) != rules_.end())
		ABORT("Feynman rule for " << id << " already inserted");
	int num = num_by_id_.size() + 1;
	num_by_id_[id] = num;
	rules_[id] = rule;
	rules_type_[id] = type;
	LOGX(
			"Generic Feynman rule inserted for identifier " << id << " (" << num << "): " << rule);
}

const GiNaC::ex& FeynmanRules::rule(const std::string& id) const {
	map<string, ex>::const_iterator it = rules_.find(id);
	if (it == rules_.end())
		ABORT("No Feynman rule defined for " << id);
	return it->second;
}

int FeynmanRules::num_by_id(const std::string& id) const {
	map<string, int>::const_iterator it = num_by_id_.find(id);
	if (it == num_by_id_.end())
		ABORT("No identification number defined for " << id);
	return it->second;
}

GiNaC::ex FeynmanRules::parsed_ps::mass() const {
	ASSERT(is_a<indexed>(pol_) && pol_.nops() > 0);
	ASSERT(is_a<indexed>(polc_) && polc_.nops() > 0);
	ASSERT(is_a<Polarization>(pol_.op(0)));
	ASSERT(is_a<Polarization>(polc_.op(0)));
	Polarization p1 = ex_to<Polarization>(pol_.op(0));
	Polarization p2 = ex_to<Polarization>(polc_.op(0));
	ASSERT(p1.mass().is_equal(p2.mass()));
	return p1.mass();
}

GiNaC::ex FeynmanRules::parsed_ps::raw_momentum() const {
	ASSERT(is_a<indexed>(pol_) && pol_.nops() > 0);
	ASSERT(is_a<indexed>(polc_) && polc_.nops() > 0);
	ASSERT(is_a<Polarization>(pol_.op(0)));
	ASSERT(is_a<Polarization>(polc_.op(0)));
	Polarization p1 = ex_to<Polarization>(pol_.op(0));
	Polarization p2 = ex_to<Polarization>(polc_.op(0));
	ASSERT(p1.momentum().is_equal(p2.momentum()));
	return p1.momentum_raw();
}

GiNaC::ex FeynmanRules::parsed_ps::raw_ortho() const {
	ASSERT(is_a<indexed>(pol_) && pol_.nops() > 0);
	ASSERT(is_a<indexed>(polc_) && polc_.nops() > 0);
	ASSERT(is_a<Polarization>(pol_.op(0)));
	ASSERT(is_a<Polarization>(polc_.op(0)));
	Polarization p1 = ex_to<Polarization>(pol_.op(0));
	Polarization p2 = ex_to<Polarization>(polc_.op(0));

	ASSERT(p1.orthogonal_vector().is_equal(p2.orthogonal_vector()));
	return p1.orthogonal_vector_raw();
}

// \todo replace verifies with ABORT or throw exception
void FeynmanRules::insert_polarization_sum(const std::string& pol,
		const std::string& polc, const std::string& val) {

	LOGX("insert polarization sum " << pol << " * " << polc << " == " << val);

	const GlobalSymbols* fg = Files::instance()->globalsymbols();
	lst all_symbols = add_lst(ps_symbols_, fg->all());

	ex pp(pol, all_symbols);
	ex ppc(polc, all_symbols);
	ex pv(val, all_symbols);
	// toggle varidx
	pv = toggle_non_numeric_varidx(pv);
	parsed_ps polarization(pp, ppc, pv);

	// check indexed

	VERIFY(is_a<indexed>(pp) && pp.nops() > 0);
	VERIFY(is_a<indexed>(ppc) && ppc.nops() > 0);
	ex e1 = pp.op(0);
	ex e2 = ppc.op(0);

	// check polarization

	VERIFY(is_a<Polarization>(e1));
	VERIFY(is_a<Polarization>(e2));
	Polarization polbase = ex_to<Polarization>(e1);
	Polarization polbase_c = ex_to<Polarization>(e2);
	VERIFY(!polbase.is_conjugated() && polbase_c.is_conjugated());
	VERIFY(polbase.mass().is_equal(polbase_c.mass()));
	VERIFY(polbase.momentum().is_equal(polbase_c.momentum()));
	VERIFY(!polbase.mass().is_equal(polbase.momentum()));

	// polarization type

	string key;

	if (is_a<GluonPolarization>(e1)) {
		VERIFY(is_a<GluonPolarization> (e2));
		GluonPolarization x = ex_to<GluonPolarization>(e1);
		GluonPolarization xcon = ex_to<GluonPolarization>(e2);
		VERIFY(x.orthogonal_vector().is_equal(xcon.orthogonal_vector()));
		key = x.unique_pol_sum_name(xcon);
	} else if (is_a<GhostPolarization>(e1)) {
		VERIFY(is_a<GhostPolarization> (e2));
		key = ex_to<GhostPolarization>(e1).unique_pol_sum_name(
				ex_to<GhostPolarization>(e2));
	} else if (is_a<PhotonPolarization>(e1)) {
		VERIFY(is_a<PhotonPolarization> (e2));
		key = ex_to<PhotonPolarization>(e1).unique_pol_sum_name(
				ex_to<PhotonPolarization>(e2));
	} else if (is_a<Spinor>(e1)) {
		VERIFY(is_a<Spinor>(e2));
		Spinor x = ex_to<Spinor>(e1);
		Spinor xbar = ex_to<Spinor>(e2);
		VERIFY((x.is_u() && xbar.is_ubar()) || (x.is_v() && xbar.is_vbar()));
		key = x.unique_pol_sum_name(xbar);
	} else {
		ABORT("Unknown polarizations:  " << e1 << " and " << e2);
	}

	insert_ps(key, polarization);

	unparsed_ps ups;
	ups.pol_ = pol;
	ups.polc_ = polc;
	ups.val_ = val;
	unparsed_polarization_sums_.insert(make_pair(key, ups));
}

GiNaC::ex FeynmanRules::get_polarization_sum(const GiNaC::ex& a,
		const GiNaC::ex& b) const {
	if (a.nops() == 0 || b.nops() == 0)
		ABORT("Can't perform polarization sum in " << a << " * " << b);

	ex e1 = a.op(0);
	ex e2 = b.op(0);

	// check polarization

	VERIFY(is_a<Polarization>(e1));
	VERIFY(is_a<Polarization>(e2));
	Polarization polbase = ex_to<Polarization>(e1);
	Polarization polbase_c = ex_to<Polarization>(e2);
	if (polbase.is_conjugated() && !polbase_c.is_conjugated())
		return get_polarization_sum(b, a);
	VERIFY(!polbase.is_conjugated() && polbase_c.is_conjugated());
	VERIFY(polbase.mass().is_equal(polbase_c.mass()));
	VERIFY(polbase.momentum().is_equal(polbase_c.momentum()));
	VERIFY(polbase.orthogonal_vector().is_equal(polbase_c.orthogonal_vector()));

	ex mass = polbase.mass();
	ex momentum_raw = polbase.momentum_raw();
	ex orthogonal_vector_raw = polbase.orthogonal_vector_raw();

	string key;

	if (is_a<GluonPolarization>(e1)) {
		VERIFY(is_a<GluonPolarization>(e2));
		key = ex_to<GluonPolarization>(e1).unique_pol_sum_name(
				ex_to<GluonPolarization>(e2));
		if (orthogonal_vector_raw.is_zero())
			LOGX("using Feynman gluon polarization sum for " << a << "*" << b);
		else
			LOGX("using physical gluon polarization sum for " << a << "*" << b);
	} else if (is_a<GhostPolarization>(e1)) {
		VERIFY(is_a<GhostPolarization>(e2));
		key = ex_to<GhostPolarization>(e1).unique_pol_sum_name(
				ex_to<GhostPolarization>(e2));
	} else if (is_a<PhotonPolarization>(e1)) {
		VERIFY(is_a<PhotonPolarization>(e2));
		key = ex_to<PhotonPolarization>(e1).unique_pol_sum_name(
				ex_to<PhotonPolarization>(e2));
	} else if (is_a<Spinor>(e1)) {
		VERIFY(is_a<Spinor>(e2));
		Spinor x = ex_to<Spinor>(e1);
		Spinor xbar = ex_to<Spinor>(e2);
		VERIFY((x.is_u() && xbar.is_ubar()) || (x.is_v() && xbar.is_vbar()));
		key = x.unique_pol_sum_name(xbar);
	} else {
		ABORT("No polarization sum for " << e1 << " * " << e2 << " found");
	}

	parsed_ps polarization = ps(key);
	exmap replace;
	ex ortho = polarization.raw_ortho();
	if (!ortho.is_zero()) {
		VERIFY(!orthogonal_vector_raw.is_zero());
		replace[ortho] = orthogonal_vector_raw;
	}
	replace[polarization.mass()] = mass;
	replace[polarization.raw_momentum()] = momentum_raw;

	ex result = polarization.val_;
	result = result.subs(replace, subs_options::no_pattern);
	VERIFY(polarization.pol_.nops() == a.nops());
	VERIFY(polarization.polc_.nops() == b.nops());
	for (unsigned i = 1; i < polarization.pol_.nops(); ++i) {
		result = result.subs(polarization.pol_.op(i) == a.op(i),
				subs_options::really_subs_idx);
	}
	for (unsigned i = 1; i < polarization.polc_.nops(); ++i) {
		result = result.subs(polarization.polc_.op(i) == b.op(i),
				subs_options::really_subs_idx);
	}

	return result;
}

void FeynmanRules::insert_ps(const std::string& type,
		const FeynmanRules::parsed_ps& ps) {
	map<string, parsed_ps>::const_iterator p = polarization_sums_.find(type);
	if (p != polarization_sums_.end()) {
		throw runtime_error(
				"Polarization sum for " + type + " already inserted.");
	}
	polarization_sums_[type] = ps;
}
const FeynmanRules::parsed_ps& FeynmanRules::ps(const std::string& type) const {
	map<string, parsed_ps>::const_iterator p = polarization_sums_.find(type);
	if (p == polarization_sums_.end())
		throw runtime_error(
				"Polarization sum of type '" + type + "' not found.");
	return p->second;
}

} // namespace Reduze

