/*  filedata.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "filedata.h"

#include <stdexcept>
#include "functions.h"
#include "int.h"
#include "equation.h"
#include "files.h"
#include "globalsymbols.h"
#include "ginacutils.h"
#include "kinematics.h"
#include "integralfamily.h"
#include "feynmanrules.h"

using namespace std;

namespace Reduze {

/// \todo denote filename in exceptions
/// \todo rethrow exceptions to include filename
/// \todo throw in outfile in case of failures

OutFileData::OutFileData(const std::string& filename,
		const std::string& format_str) :
	filename_(filename), tmp_filename_(filename + ".tmp"), empty_(true) {
	if (format_str == "mma")
		format_ = MMA;
	else if (format_str == "maple")
		format_ = MAPLE;
	else if (format_str == "form")
		format_ = FORM;
	else if (format_str == "reduze")
		format_ = Reduze;
	else
		throw std::runtime_error("Unknown output format " + format_str);
	LOGXX("opening " << tmp_filename_);
	raw_ostream_.open(tmp_filename_.c_str());
	if (!raw_ostream_.good())
		throw std::runtime_error("can't open file " + tmp_filename_);

	if (format_ == MMA) {
		raw_ostream_ << "{\n";
	} else if (format_ == MAPLE) {
		string listname = get_filename_without_directory(filename);
		listname = get_filename_without_suffix(listname);
		listname = to_safe_variable_name(listname);
		raw_ostream_ << listname << " := [\n\n";
	}
}

OutFileData::~OutFileData() {
	close();
}

void OutFileData::close() {
	if (raw_ostream_.is_open()) {
		if (!empty_)
			raw_ostream_ << "\n";
		if (format_ == MMA) {
			raw_ostream_ << "}\n";
		} else if (format_ == MAPLE) {
			raw_ostream_ << "];\n";
		}
		raw_ostream_.close();
	}
}

void OutFileData::finalize() {
	this->close();
	Reduze::rename(tmp_filename_, filename_);
}

void OutFileData::write_comment(const std::string& s) {
	/// \todo: implement me
}

std::string OutFileData::default_filename_suffix(const std::string& format) {
	if (format == "mma")
		return ".m";
	else if (format == "maple")
		return ".mpl";
	else if (format == "form")
		return ".inc";
	else if (format == "reduze")
		return ".reduze";
	else
		throw std::runtime_error("unknown output format " + format);
}

size_t InFileData::file_size() {
	size_t pos = raw_istream_.tellg();
	raw_istream_.seekg(0);
	size_t begin = raw_istream_.tellg();
	ASSERT(begin == 0);
	raw_istream_.seekg(0, ios::end);
	size_t end = raw_istream_.tellg();
	raw_istream_.seekg(pos);
	return (end - begin);
}

InFileData::InFileData(const std::string& filename) : filename_(filename) {
	raw_istream_.open(filename.c_str());
	if (!raw_istream_.good())
		throw runtime_error("can't open file " + filename);
}

InFileData& InFileData::content_line(std::string& l) {
	l.clear();
	while (l.empty() && getline(raw_istream_, l)) {
		size_t to = std::min(l.find("#"), l.find("//"));
		l = chop_whitespaces(l.substr(0, to));
	}
	// abort if badbit was set or the failbit not at eof
	if (raw_istream_.bad() || (!raw_istream_.eof() && raw_istream_.fail()))
		ERROR("I/O error reading from file '" + filename() + "'");
	return *this;
}

// OutFileINTs

OutFileINTs& OutFileINTs::operator<<(const INT& i) {
	this->OutFileData::operator<<(i);
	return *this;
}

OutFileINTs& OutFileINTs::operator<<(const std::set<INT>& is) {
	for (set<INT>::const_iterator i = is.begin(); i != is.end(); ++i)
		this->OutFileData::operator<<(*i);
	return *this;
}

// InFileINTs
namespace {
// removes simple Mathematica (* comment *), brackets and lead/trail whitespaces
string remove_mma_decoration(const string& sin) {
	string s(sin);
	while (true) {
		size_t to = s.find("*)");
		size_t from = s.find("(*");
		to = (to == string::npos ? 0 : to);
		from = (from == string::npos || from > to ? 0 : from);
		if (from == 0 && to == 0)
			break;
		else
			s.replace(from, to + 2 - from, to + 2 - from, ' ');
	}
	return chop_whitespaces(replace_chars(s, "{}", ' '));
}
}

INT InFileINTs::get_next() {
	ASSERT(!fail());
	string l;
	line_.swap(l);
	set_content_line();
	try {
		try {
			return INT(l);
		} catch (exception&) {
			return INT::from_mma(l);
		}
	} catch (exception& e) {
    	ERROR("Failed reading integral from file '" + filename() + "':\n" + e.what());
	}
	return INT(""); // never reached, suppress compiler warning
}

void InFileINTs::get_all(set<INT>& ints) {
	while (*this) // = while (!fail())
		ints.insert(get_next());
}

void InFileINTs::get_all(list<INT>& ints) {
	while (*this) // = while (!fail())
		ints.push_back(get_next());
}

void InFileINTs::set_content_line() {
	line_.clear();
	while (line_.empty() && !fail()) {
		content_line(line_);
		line_ = remove_mma_decoration(replace_chars(line_, "{}", ' '));
	}
}

// OutFileLinearCombinations

OutFileLinearCombinations& OutFileLinearCombinations::operator<<(
		const LinearCombination& lc) {
	this->OutFileData::operator<<(lc);
	return *this;
}

// OutFileLinearCombinationsGeneric

OutFileLinearCombinationsGeneric& OutFileLinearCombinationsGeneric::operator<<(
		const LinearCombinationGeneric& lc) {
	this->OutFileData::operator<<(lc);
	return *this;
}

// InFileLinearCombinationsGeneric

InFileLinearCombinationsGeneric::InFileLinearCombinationsGeneric(
		const std::string& filename) :
	InFileData(filename) {
	Files* files = Files::instance();
	symbols_ = files->kinematics()->kinematic_invariants();
	GiNaC::lst s = files->globalsymbols()->all();
	symbols_ = add_lst(symbols_, s);
}

InFileLinearCombinationsGeneric& InFileLinearCombinationsGeneric::get(
		LinearCombinationGeneric& lc) {
	string lcstr, l;
	while (l != ";" && !fail()) {
		content_line(l);
		lcstr += (l.empty() ? "" : l + '\n');
	}
	if (!lcstr.empty()) {
		istringstream is(lcstr);
		lc = LinearCombinationGeneric(is, symbols_);
	}
	return *this;
}

// InFileLinearCombinations

bool InFileLinearCombinations::check_filetype(const string& filename) {
	InFileLinearCombinations in(filename);
	try {
		LinCombHLight lc;
		in.get_noabort(lc, true); // throw if not a lin comb, silent failure for empty file
	} catch (exception&) {
		return false;
	}
	return true;
}

template<class TSet>
void InFileLinearCombinations::find_INT(const std::string& filename, TSet& ints) {
	InFileLinearCombinations in(filename);
	LinCombHLight lc;
	bool discard_coeff = true;
	while (in.get(lc, discard_coeff))
		lc.find_INT(ints);
}

InFileLinearCombinations::InFileLinearCombinations(const std::string& filename) :
	InFileData(filename) {
	Files* files = Files::instance();
	symbols_ = files->kinematics()->kinematic_invariants();
	GiNaC::lst s = files->globalsymbols()->all();
	symbols_ = add_lst(symbols_, s);
}

InFileLinearCombinations& InFileLinearCombinations::get(LinearCombination& lc) {
	try {
		string lcstr, l;
		while (l != ";" && !fail()) {
			content_line(l);
			lcstr += (l.empty() ? "" : l + '\n');
		}
		if (!lcstr.empty()) {
			istringstream is(lcstr);
			lc = LinearCombination(is, symbols_);
		}
	} catch (std::exception& e) {
		ERROR("Failed reading linear combination from file '" + filename()
				+ "':\n" + e.what());
	}
	return *this;
}

InFileLinearCombinations& InFileLinearCombinations::get_noabort(
		LinCombHLight& lc, bool discard_coeff) {
	string lcstr, l;
	while (l != ";" && !fail()) {
		content_line(l);
		lcstr += (l.empty() ? "" : l + '\n');
	}
	if (!lcstr.empty()) {
		istringstream is(lcstr);
		lc = LinCombHLight(is, discard_coeff);
	}
	return *this;
}

InFileLinearCombinations& InFileLinearCombinations::get(LinCombHLight& lc,
		bool discard_coeff) {
    try {
		get_noabort(lc, discard_coeff);
    } catch (std::exception& e) {
    	ERROR("Failed reading linear combination from file '" + filename()
    			+ "':\n" + e.what());
    }
	return *this;
}

// OutFileEquations

OutFileEquations& OutFileEquations::operator<<(const Identity& eq) {
	this->OutFileData::operator<<(eq);
	return *this;
}

OutFileEquations& OutFileEquations::operator<<(const EquationHLight& eq) {
	if (format() == Reduze)
		write_reduze(eq);
	else
		this->OutFileData::operator<<(Identity(eq));
	return *this;
}

// InFileEquations

bool InFileEquations::check_filetype(const string& filename) {
	InFileEquations in(filename);
	try {
		EquationHLight id;
		in.get_noabort(id); // throw if not an equation, silent failure for empty file
	} catch (exception&) {
		return false;
	}
	return true;
}

InFileEquations::InFileEquations(const std::string& filename) :
	InFileData(filename) {
	Files* files = Files::instance();
	symbols_ = files->kinematics()->kinematic_invariants_and_dimension();
	GiNaC::lst s = files->globalsymbols()->coupling_constants();
	symbols_ = add_lst(symbols_, s);
	symbols_.append(files->globalsymbols()->Nc());
}

InFileEquations& InFileEquations::get(Identity& eq) {
	try {
		string lcstr, l;
		while (l != ";" && !fail()) {
			content_line(l);
			lcstr += (l.empty() ? "" : l + '\n');
		}
		if (!lcstr.empty()) {
			istringstream is(lcstr);
			LinearCombination lc(is, symbols_);
			if (!lc.name().empty())
				throw runtime_error("found name but expected unnamed equation");
			lc.swap_terms(eq);
		}
	} catch (std::exception& e) {
		ERROR("Failed reading equation from file '" + filename() + "':\n"
				+ e.what());
	}
	return *this;
}

InFileEquations& InFileEquations::get_noabort(EquationHLight& eq) {
	string lcstr, l;
	while (l != ";" && !fail()) {
		content_line(l);
		lcstr += (l.empty() ? "" : l + '\n');
	}
	if (!lcstr.empty()) {
		istringstream is(lcstr);
		LinCombHLight lc(is);
		if (!lc.name().empty())
			throw runtime_error("found name but expected unnamed equation");
		lc.swap_terms(eq);
	}
	return *this;
}

InFileEquations& InFileEquations::get(EquationHLight& eq) {
	try {
       get_noabort(eq);
	} catch (std::exception& e) {
		ERROR("Failed reading equation from file '" + filename() + "':\n"
				+ e.what());
	}
	return *this;
}

}

// explicit instantiation
template void
Reduze::InFileLinearCombinations::find_INT<std::set<Reduze::INT> >(
		const std::string& filename, std::set<Reduze::INT>& ints);
template void
Reduze::InFileLinearCombinations::find_INT<std::multiset<Reduze::INT> >(
		const std::string& filename, std::multiset<Reduze::INT>& ints);
#ifdef USE_HASH_TABLE
template void
Reduze::InFileLinearCombinations::find_INT<tr1::unordered_set<Reduze::INT> >(
		const std::string& filename, tr1::unordered_set<Reduze::INT>& ints);
#endif /* USE_HASH_TABLE */

