/*  files.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef FILES_H_
#define FILES_H_

#include <string>
#include <map>
#include <set>
#include <list>
#include "yamlconfigurable.h"

namespace Reduze {

class Kinematics;
class IntegralFamily;
class SectorMappings;
class SectorReduzer;
class Job;
class JobQueue;
class Sector;
class OutFileData;
class FeynmanRules;
class GlobalSymbols;
class GlobalOptions;
class Paths;
class OrderedCrossings;

/// class to store the keywords used in YAML files
class FileKeys {
public:
	virtual ~FileKeys() {
	}
	static FileKeys* instance() {
		static FileKeys* keys = 0;
		if (keys == 0) {
			keys = new FileKeys();
		}
		return keys;
	}

	const std::string& kinematics() const {
		return kinematics_;
	}
	const std::string& crossings() const {
		return crossings_;
	}
	const std::string& integralfamilies() const {
		return integralfamilies_;
	}
	const std::string& sectormappings() const {
		return sectormappings_;
	}

private:
	FileKeys() {
		kinematics_ /*      */= "kinematics";
		crossings_ /*       */= "crossings";
		integralfamilies_ /**/= "integralfamilies";
		sectormappings_ /*  */= "sectormappings";
	}

	std::string kinematics_;
	std::string crossings_;
	std::string integralfamilies_;
	std::string sectormappings_;
};

/// Manager for storing objects to files and cached reading from files
class Files {
public:
	virtual ~Files();
	static Files* instance();

	void set_project_directory(const std::string& dir_name);
	const std::string& project_directory() const;

	// To these directories jobs may freely write, all other files
	// should be accessed via the get_filename*() names. Output should
	// always be written to a temporary file first, then renamed to the
	// final filename.
	std::string get_tmp_directory() const;
	std::string get_log_base_directory() const;
	std::string get_log_run_directory() const;
	std::string get_reductions_directory() const;
	std::string get_graph_directory() const;
	std::string get_graph_directory(const IntegralFamily* ic) const;
	std::string get_config_directory() const;
	std::string sectormappings_directory() const;
	std::string crossings_directory() const;

	Paths* paths();
	GlobalSymbols* globalsymbols();
	GlobalOptions* globaloptions();
	FeynmanRules* feynmanrules();
	OrderedCrossings* crossings(const std::string& kin = std::string());
	/// get the kinematics with name 'name', default kinematics (first defined in kinematics.yaml) can also be accessed by an empty string
	Kinematics* kinematics(const std::string& name = std::string()); const
	std::map<std::string, Kinematics*>& all_kinematics();
	IntegralFamily* integralfamily(const std::string& name);
	/// access to the uncrossed integral families (they might have wrong id if crossings are not loaded yet)
	IntegralFamily* uncrossed_integralfamily(const std::string& name);
	/// returns product IntegralFamily
	const IntegralFamily* integralfamily(const IntegralFamily* ic1,
			const IntegralFamily* ic2);
	std::list<IntegralFamily*> integralfamilies();
	std::list<IntegralFamily*> uncrossed_integralfamilies();
	SectorMappings* sectormappings(const std::string& name);

	std::string get_filename_global_yaml() const;
	std::string get_filename_globalsymbols() const;
	std::string get_filename_paths() const;
	std::string get_filename_globaloptions() const;
	std::string get_filename_kinematics() const;
	std::string get_filename_crossings() const;
	std::string get_filename_integralfamilies() const;
	std::string get_filename_feynmanrules() const;
	std::string get_filename_sectormappings(const std::string& name);
	std::list<std::string> get_filenames_all_sectormappings();
	std::string get_filename_sectorreduction(const Sector& sector) const;
	std::string get_filename_sectorexternal(const Sector& sector) const;
	std::string get_filename_sectorexternal_shared() const;
	std::string get_filename_sectorreduction(const Sector& sector,
			const std::string& dir) const;
	std::string get_filename_generic_identities(const std::string& family,
			const std::string& type) const;

	std::string get_safe_log_filename_for_job(const Job* const job) const;

	//void read(JobQueue& j, const std::string& filename);

	void save_crossings(const std::map<std::string, OrderedCrossings>&) const;
	void save(const SectorMappings& m);

	bool has_empty_sectormappings_file(const std::string& intfam);
	bool sectormappings_already_setup(const std::string& intfam);
	void remove_empty_sectormappings_files();
	/// clears cached sector mappings
	void clear_sectormappings();
	/// clears cached sector mappings and crossed sector mappings of family 'name'
	void clear_sectormappings(const std::string& name);
	/// verifies if the two files contain the same sector mappings for the input family
	/** symmetry shifts are not tested if they contain exactly the same shifts **/
	static void verify_equivalent_sector_mapping_files(
			const std::string& intfam_name, const std::string& secmap_file1,
			const std::string& secmap_file2);
	/// returns the name of the integral family if the file is a sector mapping file
	/** on failure an empty string is returned */
	static std::string family_in_sector_mapping_file(
			const std::string& filename);

	bool crossings_loaded() const {
		return crossings_loaded_;
	}

private:
	Files();
	void load_kinematics();
	void load_integralfamilies();
	void load_uncrossed_integralfamilies();
	void setup_crossed_integralfamilies(IntegralFamily*);
	void load_crossings();
	bool kinematics_loaded_, uncrossed_intfam_loaded_, intfam_loaded_,
			crossings_loaded_;
	std::string project_directory_;
	std::string default_kinematics_;
	Paths* paths_;
	GlobalSymbols* globalsymbols_;
	GlobalOptions* globaloptions_;
	FeynmanRules* feynmanrules_;
	std::map<std::string, OrderedCrossings*> crossings_;
	std::map<std::string, Kinematics*> kinematics_;
	std::map<std::string, IntegralFamily*> integralfamilies_;
	std::list<IntegralFamily*> ordered_integralfamilies_; // by id
	std::map<std::string, SectorMappings*> sectormappings_;

	std::map<std::string, IntegralFamily*> uncrossed_integralfamilies_;
	std::list<IntegralFamily*> ordered_uncrossed_integralfamilies_; // by id
};

/// returns directory part of filename with "/" appended if not already present
std::string get_directory_of_filename(const std::string& filename);
/// returns filename with its directory stripped
std::string get_filename_without_directory(const std::string& filename);
/// returns filename with the given suffix stripped
std::string get_filename_without_suffix(const std::string& filename,
		const std::string& suffix);
/// returns filename with the given suffix stripped
std::string get_filename_without_suffix(const std::string& filename);
/// returns canonical form of filename, prepending "./" for relative paths
std::string get_canonical_filename(const std::string& filename);
/// returns a short version of the filename
std::string short_filename(const std::string& str);
/// returns the string with potentially dangerous special characters substituted
std::string safe_filename(const std::string& str);
/// concatenates directory and filename, inserting '/' in between if necessary
std::string cat_directory_and_filename(const std::string& dir,
		const std::string& filename);

/// returns a directory name with suffix .tmp/
std::string get_temporary_dir_name(const std::string& directory);
/// create a directory
void make_directory(const std::string& dirname);
/// tries to open the file and returns if this was successful
bool is_readable_file(const std::string& filename);
bool are_readable_files(const std::list<std::string>& filenames);
bool is_readable_directory(const std::string& path);

// operations of files
void rename(const std::string& fn1, const std::string& fn2);
void remove(const std::string& fn);
void remove(const std::list<std::string>& filenames);
void remove_empty_dir(const std::string& dir);
void remove_empty_dir(const std::list<std::string>& dir);

/// get the files in the directors
std::list<std::string> get_files_from_directory(const std::string& dirname);
/// removes all files in given directory and the directory itself
/** does not touch symbolic links, does not touch subdirectories **/
bool remove_directory_with_files(const std::string& dirname);
/// copy the content of file 'from' to file 'to'
void copy_file(const std::string& from, const std::string& to);

/// writes some settings like integral family definitions to output stream
void write_header(OutFileData& of);

} // namespace Reduze

#endif /* FILES_H_ */
