/*  functions.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include <sstream>
#include <fstream>
#include <ctime>
#include <sys/time.h> // gettimeofday
#include <stdexcept>
#include "../yaml/include/yaml.h"

#include "integralfamily.h"
#include "int.h"
#include "functions.h"
#include "streamutils.h"
#include "files.h"
#include "kinematics.h"

#ifdef HAVE_MPI
#include <mpi.h>
#endif

using namespace std;

namespace Reduze {

// GENERAL FUNCTIONS

list<string> split_string(const string& full, size_t maxlen) {
	list<string> res;
	string part;
	size_t last, pos;
	last = 0;
	bool newline = false;
	do {
		pos = full.find_first_of(" \n", last);
		size_t len = (pos == string::npos ? full.length() + 1 : pos) - last;
		if (part.empty()) {
			part = full.substr(last, len);
		} else if (part.length() + len <= maxlen && !newline) {
			part += " " + full.substr(last, len);
		} else {
			res.push_back(part);
			part = full.substr(last, len);
		}
		newline = (pos != string::npos && full[pos] == '\n');
		last = pos + 1;
	} while (pos != string::npos);
	res.push_back(part);
	return res;
}

string chop_whitespaces(const std::string& s) {
	size_t from = s.find_first_not_of(" \t\n");
	size_t to = s.find_last_not_of(" \t\n");
	if (from == string::npos || to == string::npos)
		return "";
	return s.substr(from, to - from + 1);
}

string replace_chars(const string& s, const string& tokens, char replacewith) {
	string res(s);
	size_t pos = 0;
	while ((pos = res.find_first_of(tokens, pos)) != string::npos)
		res.replace(pos, 1, 1, replacewith);
	return res;
}

int power_int(int a, int b) {
	VERIFY(b >= 0);
	int res = 1;
	for (int i = 0; i < b; ++i)
		res *= a;
	return res;
}

unsigned long power_ulong(int a, int b) {
	if (a == 0 && b == 0)
		return 1;
	if (a == 0)
		return 0;
	if (b == 0)
		return 1;
	VERIFY(a > 0 && b > 0);
	unsigned long res = 1;
	for (int i = 0; i < b; ++i)
		res *= a;
	return res;
}

unsigned long factorial_ulong(int n) {
	VERIFY(n >= 0);
	unsigned long res = 1;
	for (int i = n; i > 1; --i)
		res *= i;
	return res;
}

unsigned long binom_int(int n, int k) {
	if (k > n || k < 0)
		return 0;
	if (k == n)
		return 1;
	unsigned long res = 1;
	for (int i = 0; i < k; ++i) {
		res *= n - i;
		VERIFY(res % (i + 1) == 0);
		res /= i + 1;
	}
	return res;
}

// {n, k} = 1/k! Sum_j=0^k (-1)^(j) (k, j) (k-j)^n
unsigned long stirling2_ulong(int n, int k) {
	VERIFY(n >= 0 && k >= 0);
	if (n == 0 && k == 0)
		return 1;
	if (n == 0 || k == 0)
		return 0;
	unsigned long plus = 0, minus = 0, res, divide;
	for (int j = 0; j <= k; ++j) {
		if (j % 2 == 0) {
			plus += binom_int(k, j) * power_ulong(k - j, n);
		} else {
			minus += binom_int(k, j) * power_ulong(k - j, n);
		}
	}
	VERIFY(plus >= minus);
	res = plus - minus;
	divide = factorial_ulong(k);
	VERIFY(res % divide == 0);
	return res / divide;
}


// recurrence relation for number of partitions of n elements
// into k subsets, each with minimum size r
// associated Striling numbers of the second kind

// S_r(n+1,k)=k*S_r(n,k)+binomial(n,r-1)*S_r(n-r+1,k-1)

unsigned long associated_stirling_2(int n, int k, int r) {
	VERIFY(r >= 0 && k >= 0 && n >= 0);
	if (k > n || r > n)
		return 0;
	if ((n == 0 && k == 0) || (k == 1 && r != 0))
		return 1;
	if (n == 0 || k == 0 || r * k > n)
		return 0;

	unsigned long res1 = 0;
	if (k != 0)
		res1 += k * associated_stirling_2(n - 1, k, r);
	unsigned long res2 = 0;

	ASSERT(n != 0);
	if (r != 0)
		res2 += binom_int(n - 1, r - 1)
				* associated_stirling_2(n - r, k - 1, r);

	return res1 + res2;
}


/*
 uintx NumberOfSeed(int t, int r, int s) {
 uintx res = binom_int(r - 1, r - t) * binom_int(
 IntegralFamily::auxtop1()->get_NoP() - 1 - t + s, s);
 return res;
 }

 uintx NumberOfSeed(int t, int r1, int r2, int s1, int s2) {
 uintx res = 0;
 for (int i = r1; i <= r2; ++i) {
 for (int j = s1; j <= s2; ++j) {
 res += NumberOfSeed(t, i, j);
 }
 }
 return res;
 }

 uintx NumberOfIBP(int t, int r, int s) {
 uintx res = NumberOfSeed(t, r, s);
 res *= IntegralFamily::auxtop1()->get_NoLM() * (IntegralFamily::auxtop1()->get_NoEM()
 + IntegralFamily::auxtop1()->get_NoLM());
 return res;
 }

 uintx NumberOfIBP(int t, int r1, int r2, int s1, int s2) {
 uintx res = NumberOfSeed(t, r1, r2, s1, s2);
 res *= IntegralFamily::auxtop1()->get_NoLM() * (IntegralFamily::auxtop1()->get_NoEM()
 + IntegralFamily::auxtop1()->get_NoLM());
 return res;
 }

 uintx NumberOfIBP(int t, int r1, int r2, int s1, int s2, uintx nr_of_seed) {
 uintx res = nr_of_seed;
 res *= IntegralFamily::auxtop1()->get_NoLM() * (IntegralFamily::auxtop1()->get_NoEM()
 + IntegralFamily::auxtop1()->get_NoLM());
 return res;
 }

 uintx NumberOfLI(int t, int r, int s) {
 uintx res = NumberOfSeed(t, r, s);
 res *= binom_int(IntegralFamily::auxtop1()->get_NoEM(), 2);
 return res;
 }

 uintx NumberOfLI(int t, int r1, int r2, int s1, int s2) {
 uintx res = NumberOfSeed(t, r1, r2, s1, s2);
 res *= binom_int(IntegralFamily::auxtop1()->get_NoEM(), 2);
 return res;
 }

 uintx NumberOfLI(int t, int r1, int r2, int s1, int s2, uintx nr_of_seed) {
 uintx res = nr_of_seed;
 res *= binom_int(IntegralFamily::auxtop1()->get_NoEM(), 2);
 return res;
 }

 uintx NumberOfEquation(int t, int r, int s) {
 uintx res = 0;
 if (Constants::UseIBP)
 res += NumberOfIBP(t, r, s);
 if (Constants::UseLI)
 res += NumberOfLI(t, r, s);
 return res;
 }

 uintx NumberOfEquation(int t, int r1, int r2, int s1, int s2) {
 uintx res = 0;
 if (Constants::UseIBP)
 res += NumberOfIBP(t, r1, r2, s1, s2);
 if (Constants::UseLI)
 res += NumberOfLI(t, r1, r2, s1, s2);
 return res;
 }

 uintx NumberOfEquation(int t, int r1, int r2, int s1, int s2, uintx nr_of_seed) {
 uintx res = 0;
 if (Constants::UseIBP)
 res += NumberOfIBP(t, r1, r2, s1, s2, nr_of_seed);
 if (Constants::UseLI)
 res += NumberOfLI(t, r1, r2, s1, s2, nr_of_seed);
 return res;
 }
 */
int number_of_digits(unsigned long n) {
	int num_digits = 0;
	while (n > 0) {
		++num_digits;
		n /= 10;
	}
	return num_digits;
}

std::string to_safe_variable_name(const std::string& str) {
	string allowed =
			"abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";
	string res;
	for (string::const_iterator c = str.begin(); c != str.end(); ++c)
		if (allowed.find(*c) != string::npos)
			res.push_back(*c);
	return res;
}

string time_string() {
	time_t rawtime;
	struct tm * timeinfo;

	time(&rawtime);
	timeinfo = localtime(&rawtime);
	return asctime(timeinfo);
}

string convert_seconds(time_t difftime) {
	time_t rest, value = difftime;
	ostringstream str;

	rest = value % 60; // 60 sec = 1 min
	if (rest == value) {
		str << "00:00:" << setfill('0') << setw(2) << rest;
		return str.str();
	}
	time_t sec = rest;
	value = (value - rest) / 60;

	rest = value % 60; // 60 min = 1 hour
	if (rest == value) {
		str << "00:" << setfill('0') << setw(2) << rest << ":";
		str << setfill('0') << setw(2) << sec;
		return str.str();
	}

	str << setfill('0') << setw(2) << (value - rest) / 60 << ":";
	str << setfill('0') << setw(2) << rest << ":";
	str << setfill('0') << setw(2) << sec;

	return str.str();
}

// RunIdentification

RunIdentification* RunIdentification::instance() {
	static RunIdentification* i = 0;
	if (i == 0)
		i = new RunIdentification();
	return i;
}

RunIdentification::RunIdentification() {
	// gettimeofday() is deprecated but in this way we avoid linking -lrt
	// for its successor clock_gettime()
	//timespec t;
	//clock_gettime(CLOCK_REALTIME, &t); (with tv_nsec instead of tv_usec)
	timeval t;
	gettimeofday(&t, 0);
	// (typical minimal accuracy of gettimeofday() is 0.01s)
	long long n = t.tv_sec * 100 + t.tv_usec / 10000;
	set_number(n);
}

void RunIdentification::set_number(long long n) {
	number_ = n;
	timeval t;
	t.tv_sec = n / 100;
	t.tv_usec = (n % 100) * 10000;
	stringstream ss, ssl;
	ss << setfill('0') << setw(8) << hex << static_cast<int>(number_);
	string_ = ss.str();
	tm* l = localtime(&t.tv_sec);
	/*
	 ssl << setfill('0') << setw(4) << (1900 + l->tm_year) //
	 << '_' << setw(2) << (1 + l->tm_mon) //
	 << '_' << setw(2) << l->tm_mday //
	 << '_' << setw(2) << l->tm_hour //
	 << '_' << setw(2) << l->tm_min //
	 << '_' << setw(2) << l->tm_sec //
	 << '_' << setw(2) << (t.tv_usec / 10000);
	 */
	ssl << setfill('0') << setw(2) << ((1900 + l->tm_year) % 100) //
			<< setw(2) << (1 + l->tm_mon) << setw(2) << l->tm_mday //
			<< '_' << setw(2) << l->tm_hour << setw(2) << l->tm_min //
			<< '_' << setw(2) << l->tm_sec << setw(2) << (t.tv_usec / 10000);
	long_string_ = ssl.str();
}

long long RunIdentification::number() const {
	return number_;
}

const string& RunIdentification::string() const {
	return string_;
}

const string& RunIdentification::long_string() const {
	return long_string_;
}

void add_to_each_element(std::vector<int>& v, int i) {
	for (vector<int>::iterator it = v.begin(); it != v.end(); ++it)
		*it = *it + i;
}
void add_to_each_element(std::vector<std::vector<int> >& v, int i) {
	for (vector<vector<int> >::iterator it = v.begin(); it != v.end(); ++it)
		add_to_each_element(*it, i);
}

std::string welcome() {
	stringstream os;
	os << "________     _________" << endl;
	os << "___  __ \\__________  /___  ___________" << endl;
	os << "__  /_/ /  _ \\  __  /_  / / /__  /  _ \\" << endl;
	os << "_  _, _//  __/ /_/ / / /_/ /__  /_  __/" << endl;
	os << "/_/ |_| \\___/\\__,_/  \\__,_/ _____\\___/ " << endl;
	os << endl;
	os << "Package:     \t" << PACKAGE_NAME << endl;
	os << "Version:     \t" << PACKAGE_VERSION << endl;
	os << "Authors:     \t";
	os << "Andreas von Manteuffel, Cedric Studerus" << endl << endl;
	return os.str();
}

// logging facilities

LogStream rlog1(1);
LogStream rlog2(2);
LogStream rlog3(3);

LogStream::LogStream(int details) :
	std::ostream(new NullBuffer), file(0) /*, message_counter(0), details(details)*/ {
	nullbuffer = static_cast<NullBuffer*> (rdbuf());
}

LogStream::~LogStream() {
	delete nullbuffer;
	delete file;
}

/*
 void LogStream::print_date() {
 *this << "Date: " << time_string();
 }
 */

void LogStream::set_file(const std::string& filename) {
	delete file;
	//filename = fn;
	file = new std::ofstream(filename.c_str(), ios_base::app);
	if (!*file) {
		std::cerr << "ERROR: could not open logging file " << filename
				<< std::endl;
		r_abort(2);
	}
	rdbuf(file->rdbuf());
}

void LogStream::set_stdout() {
	//filename = "";
	delete file;
	file = 0;
	rdbuf(std::cout.rdbuf());
}

void LogStream::set_stderr() {
	//filename = "";
	delete file;
	file = 0;
	rdbuf(std::cerr.rdbuf());
}

void LogStream::set_null() {
	//filename = "";
	delete file;
	file = 0;
	rdbuf(nullbuffer);
}

/*
 std::string LogStream::get_filename() const {
 return filename;
 }
 */

#ifdef HAVE_MPI
void LogStream::print_prefix() {
	/*
	 if (details >= 2)
	 *this << MPI::COMM_WORLD.Get_rank() << ":";
	 if (details >= 3)
	 *this << ++message_counter << ":";
	 */
}
#else
void LogStream::print_prefix() {
	/*
	 if (details >= 3)
	 *this << ++message_counter << ":";
	 */
}
#endif

const char* error_asciiart = ""
	"+-------+\n"
	"| ERROR |\n"
	"+-------+\n";

#ifdef HAVE_MPI
void r_abort(int i) {
	//exit(i);
	MPI::COMM_WORLD.Abort(i);
}
#else
void r_abort(int i) {
	exit(i);
}
#endif

// timers

double Timer::get_absolute_wall_time() const {
#ifdef HAVE_MPI
	return MPI::Wtime();
#else
	// for now, we use deprecated gettimeofday() to avoid linking -lrt
	timeval t;
	gettimeofday(&t, 0);
	return t.tv_sec + t.tv_usec / 1.0e6;
#endif
}

double Timer::get_absolute_cpu_time() const {
	std::clock_t t = std::clock();
	// should pick the right cast chain for various clock_t implementations:
	return static_cast<double> ((t + 0.0) / CLOCKS_PER_SEC);
}

double Timer::restart() {
	double t = get_wall_time();
	wall_start_ = wall_stop_ = get_absolute_wall_time();
	cpu_start_ = cpu_stop_ = get_absolute_cpu_time();
	return t;
}

void Timer::unpause() {
	wall_start_ = wall_stop_ = get_absolute_wall_time() - get_wall_time();
	cpu_start_ = cpu_stop_ = get_absolute_cpu_time() - get_cpu_time();
}

void Timer::pause() {
	wall_stop_ = get_absolute_wall_time();
	cpu_stop_ = get_absolute_cpu_time();
}

double Timer::get_wall_time() const {
	if (wall_stop_ == wall_start_)
		return get_absolute_wall_time() - wall_start_;
	else
		return wall_stop_ - wall_start_;
}

double Timer::get_cpu_time() const {
	if (cpu_stop_ == cpu_start_)
		return get_absolute_cpu_time() - cpu_start_;
	else
		return cpu_stop_ - cpu_start_;
}

string Timer::get_wall_time_nice_string() const {
	return Timer::print_nice_string(get_wall_time());
}

std::string Timer::print_nice_string(double d) {
	ostringstream s;
	s << "(time: " << fixed << setprecision(1) << d
			<< resetiosflags(ios::fixed) << " s)";
	return s.str();
}

// progress bar

ProgressBar::ProgressBar() :
	width_(50), dot_("*"), printed_dots_(0), counter_(1), max_(1), offset_(0),
			first_(true) {
}

ProgressBar::ProgressBar(long offset, const string& titel, long max) :
	width_(50), dot_("*"), printed_dots_(0), titel_(titel), counter_(1), max_(
			max), offset_(offset), first_(true) {
}

void ProgressBar::start() {
	counter_ = 1;
	printed_dots_ = 0;
	first_ = true;
	for (long i = 0; i < offset_; ++i)
		LOGN(" ");
	LOGN(titel_ << "\n" << flush);
	for (long i = 0; i < offset_; ++i)
		LOGN(" ");
	for (long i = 0; i < width_; ++i)
		LOGN(".");
	LOGN("\r" << flush);
}

void ProgressBar::end(const std::string& message) const {
	LOGN("\n" << flush);
	if (!message.empty()) {
		for (long i = 0; i < offset_; ++i)
			LOGN(" ");
		LOG(message);
	}
}

// \todo: move to global functions
static long round_to_long(double x) {
	return long(x > 0.0 ? x + 0.5 : x - 0.5);
}

void ProgressBar::print() {
	print(counter_++, max_);
}

void ProgressBar::print(long count) {
	print(count, max_);
}

void ProgressBar::print(long count, long max) {
	if (max <= 0)
		return;
	if (first_) {
		for (long i = 0; i < offset_; ++i)
			LOGN(" ");
		first_ = false;
	}
	if (count > max)
		return;
	long should_be_printed =
			round_to_long(double(width_ * count) / double(max));
	for (; printed_dots_ < should_be_printed; ++printed_dots_)
		LOGN(dot_ << flush);
}

void ProgressBar::set_max(long max) {
	max_ = max;
}
void ProgressBar::set_title(const std::string& title) {
	titel_ = title;
}

}
// namespace Reduze
