/*  job_computedimrecurrence.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "functions.h"
#include "int.h"
#include "filedata.h"
#include "equation.h"
#include "ginacutils.h"
#include "identitygenerator.h"
#include "propagator.h"
#include "integralfamily.h"
#include "kinematics.h"
#include "files.h"
#include "job_computedimrecurrence.h"

using namespace std;

namespace Reduze {

namespace {
JobProxy<ComputeDimRecurrence> dummy;
}

bool ComputeDimRecurrence::find_dependencies(const set<string>& outothers,//
		list<string>& in, list<string>& out, list<Job*>& auxjobs) {
	in.push_back(input_file_);
	out.push_back(output_file_);
	return true;
}

void ComputeDimRecurrence::init() {
}

std::string ComputeDimRecurrence::get_description() const {
	return "compute dimensional recurrence relations";
}

// A dimension raise identity according to 1007.2256 (with mu=-1 for Minkowski
// propagators), where d^d k / Pi^(d/2) per loop is the integration measure.
// see also 1212.2685
void ComputeDimRecurrence::run_serial() {
	using namespace GiNaC;
	LOG("Reading integrals from file \"" << input_file_ << '\"');
	InFileINTs in(input_file_);
	set<INT> ints;
	in.get_all(ints);
	in.close();
	LOG("Writing dimensional shift identities to file \""<<output_file_<<'\"');
	OutFileEquations out(output_file_);
	for (set<INT>::const_iterator ii = ints.begin(); ii != ints.end(); ++ii) {
		// note: this is a quick implementation, it is very easy to make it faster
		LOG("  processing " << *ii);
		const IntegralFamily* f = ii->integralfamily();
		size_t l = f->loop_momenta().nops();
		OPSUM opsumdimraise;
		matrix mm(l, l);
		const vector<Propagator>& props = f->propagators();
		lst Asymbol = create_symbols("A", props.size());
		vector<symbol> ll(l);
		for (size_t i = 0; i < l; ++i) {
			ASSERT(is_a<symbol>(f->loop_momenta().op(i)));
			ll[i] = ex_to<symbol>(f->loop_momenta().op(i));
		}
		for (size_t i = 0; i < l; ++i) {
			for (size_t j = 0; j < l; ++j) {
				mm(i, j) = 0;
				for (size_t k = 0; k < props.size(); k++) {
					ex sk = props[k].scalarproduct();
					// we use 2^deltaij d/dsij D_k = d/dli d/dlj D_k
					ex dk = (sk.op(0) * sk.op(1)).diff(ll[i]).diff(ll[j]);
					mm(i, j) += dk * Asymbol.op(k);
				}
			}
		}
		exmap toprops;
		for (size_t k = 0; k < props.size(); k++) {
			ex ak = f->propagator_exponents()[k];
			toprops[Asymbol.op(k)] = ak * props[k];
		}
		// overall minus sign since we cmopute -RHS to be inserted in eq later
		ex mrhs = -pow(numeric(-1,2), l) * mm.determinant().subs(toprops);
		OPSUM opsum = ex_to_OPSUM(mrhs, props);
		LinearCombinationGeneric gen = INTGeneric(f).shift(opsum);
		LinearCombination lc = gen.get_linear_combination(*ii);
		Identity id;
		id.swap_terms(lc);
		string fn = ii->integralfamily()->name();
		string fdm2n = Kinematics::dimension_shifted_label(fn, -2);
		IntegralFamily* fdm2 = Files::instance()->integralfamily(fdm2n);
		id.insert(INT(fdm2, ii->v()), 1);
		out << id;
	}
	out.finalize();
}

}
