/*  job_export.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef JOB_EXPORT_H_
#define JOB_EXPORT_H_

#include "job.h"
#include <ginac/ginac.h>

namespace Reduze {

class OutFileData;

/// Job to export reductions
class Export: public Job {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("export");
		s.set_short_description(
				"Exports to FORM, Mathematica or Maple format.");
		s.set_long_description("" //
						"Job to export reductions, linear combinations or integrals"
						" to a format readable by third party software,"
						" e.g. FORM or Mathematica."
						" Substitutions and a series expansion may be performed"
						" optionally. If a symbol is set to one with the option"
						" \"symbol_to_replace_by_one\" in the kinematics file the"
						" dependence on that symbol is restored in the output.");
		s.add_option("input_file", true, "string", "" //
						"Name of input file containing linear combinations or equations.");
		s.add_option("output_format", true, "string", "" //
						"Output format: one of 'mma', 'form' or 'reduze'.");
		s.add_option("output_file", false, "string", "" //
						"Name of output file. If left empty, a default"
						" extension is appended to the input file name.");
		s.add_option("preferred_masters_file", false, "string", "" //
						"Name of a file containing integrals which should be preferred"
						" in the result wrt. other integrals (changes the default"
						" ordering of integrals).");
		s.add_option("new_parameters", false, "list of strings", "" //
						"New symbols to be introduced in the exported file, e.g."
						" [ eps ].");
		s.add_option("reparametrization_rules", false, "" //
						"sequence of 2-element sequences", "" //
						"Substitutions to be performed, e.g. [ [\"d\", \"4-2*eps\"] ]");
		s.add_option("series_expansion", false, "" //
						"sequence of zero or 3 elements", "" //
						"If non-empty, a series expansion is performed. For"
						" [ PARAMETER, POINT, ORDER ], an expansion in the symbol"
						" PARAMETER around the point POINT up to and including"
						" order ORDER is performed, e.g.: [ eps, 0, 4 ].");
		s.add_option("toggle_metric_convention", false, "boolean", "" //
						"Whether to use Euclidean instead of default Minkowkian"
						" metric convention. If true, all integrals are multiplied"
						" with a factor (-1)^(r+s).");
		s.add_options(Job::yaml_spec());
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}
	Export() :
			undefined("*"), toggle_metric_convention_(false) {
		series_parameter_ = undefined;
		add_auto_options();
	}
	virtual ~Export() {
	}
	virtual void run_serial();
	virtual bool find_dependencies(const std::set<std::string>&, //
			std::list<std::string>&, std::list<std::string>&, std::list<Job*>&);
	virtual std::string get_description() const;

protected:
	virtual void add_auto_options() {
		add_auto_io("input_file", input_filename_);
		add_auto_io("output_file", output_filename_);
		add_auto_io("output_format", output_format_);
		add_auto_io("preferred_masters_file", preferred_masters_filename_);
	}
	virtual void init();
	virtual void print_manual_options(YAML::Emitter& os) const;
	virtual void read_manual_options(const YAML::Node&);

private:
	void write_export_header(OutFileData& of);
	void export_INTs();
	void export_equations();
	void export_linearcombinations();
	std::string input_filename_, output_filename_, preferred_masters_filename_;
	std::string output_format_;
	GiNaC::lst new_parameters_;
	GiNaC::exmap reparametrization_rules_;
	GiNaC::symbol series_parameter_;
	GiNaC::ex series_point_;
	int series_order_;
	GiNaC::symbol undefined;
	bool toggle_metric_convention_;
};

}

#endif /* JOB_EXPORT_H_ */
