/*  job_printsectorinfo.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "job_printsectorinfo.h"
#include "functions.h"
#include "files.h"
#include "filedata.h"
#include "kinematics.h"
#include "sector.h"
#include "integralfamily.h"
#include "sectormappings.h"
#include "int.h"
#include "graph.h"
#include "ginacutils.h"
#include "identitygenerator.h"
#include "globalsymbols.h"
//#include <ginac/ginac.h>

using namespace std;

namespace Reduze {

// register job type at JobFactory
namespace {
JobProxy<PrintSectorInfo> dummy;
}

set<Sector> find_sectors(SectorSelection& sel, const string& integralsfile,
		map<Sector, int> multiplicities) {
	set<Sector> res = sel.find_sectors();
	if (!integralsfile.empty()) {
		InFileINTs ints(integralsfile);
		set<INT> intset;
		ints.get_all(intset);
		ints.close();
		for (set<INT>::const_iterator i = intset.begin(); i != intset.end(); ++i) {
			res.insert(i->get_sector());
			map<Sector, int>::iterator m = multiplicities.find(i->get_sector());
			if (m == multiplicities.end())
				m->second = 0;
			++(m->second);
		}
	}
	return res;
}

void print_config_info_mma(const string& filename) {
	LOG("Exporting config to file \"" << filename.c_str() << "\"");
	ofstream os(filename.c_str());
	os << "{\n";
	Kinematics* kin = Files::instance()->kinematics();
	os << "(* kinematics *)\n";
	os << "IncomingMomenta -> " << kin->incoming_momenta() << ",\n";
	os << "OutgoingMomenta -> " << kin->outgoing_momenta() << ",\n";
	os << "MomentumConservation -> ";
	print_as_mma(kin->rule_momentum_conservation(), os, false);
	os << ",\n";
	os << "KinematicInvariants -> " << kin->kinematic_invariants() << ",\n";
	os << "ScalarProductRules -> ";
	print_as_mma(kin->rules_sp_to_invariants(), os, true);
	os << ",\n";
	os << "(* integral families *)\n";
	list<IntegralFamily*> fams = Files::instance()->integralfamilies();
	list<IntegralFamily*>::const_iterator f;
	for (f = fams.begin(); f != fams.end(); ++f) {
		os << "LoopMomenta[\"" << (*f)->name() << "\"] -> "
				<< (*f)->loop_momenta() << ",\n";
		os << "Propagators[\"" << (*f)->name() << "\"] -> {";
		const vector<Propagator>& props = (*f)->propagators();
		for (size_t i = 0; i < props.size(); ++i) {
			os << (i == 0 ? "\n  " : ",\n  ");
			props[i].eval().print(print_mma(os));
			os << " (*" << (i + 1) << "*)";
		}
		os << "\n},\n";
		os << "PermutationSymmetries[\"" << (*f)->name() << "\"] -> {";
		const list<Permutation>& sl = (*f)->permutation_symmetries();
		list<Permutation>::const_iterator s;
		for (s = sl.begin(); s != sl.end(); ++s) {
			os << (s == sl.begin() ? "\n  " : ",\n  ");
			s->print_mma_to_onebased(os);
		}
		os << "\n},\n";
	}
	os << "ReduzeVersion -> \"" << PACKAGE_VERSION << "\"\n";
	os << "}\n";
	os.close();
	LOG("done");
}

void print_sector_mappings_mma(const string& filename) {
	using namespace GiNaC;
	LOG("Exporting sector mappings to file \"" << filename.c_str() << "\"");
	ofstream os(filename.c_str());
	os << "{\n";
	os << "(* sector mappings *)\n";
	list<IntegralFamily*> fams = Files::instance()->integralfamilies();
	list<IntegralFamily*>::const_iterator f;
	for (f = fams.begin(); f != fams.end(); ++f) {
		SectorMappings* m = Files::instance()->sectormappings((*f)->name());
		os << "ZeroSectors[\"" << (*f)->name() << "\"] -> {";
		int n = 0;
		const set<int>& zs = m->zero_sectors();
		set<int>::const_iterator s;
		for (s = zs.begin(); s != zs.end(); ++s) {
			if (s != zs.begin())  os << ", ";
			if ((n++) % 10 == 0)  os << "\n ";
		    Sector(*f, *s).print_mma(os);
		}
		os << "\n},\n";
		os << "SectorsWithoutGraph[\"" << (*f)->name() << "\"] -> {";
		const set<int>& ngs = m->sectors_without_graph();
		for (s = ngs.begin(); s != ngs.end(); ++s) {
			if (s != ngs.begin())  os << ", ";
			if ((n++) % 10 == 0)  os << "\n ";
		    os << " {\"" << (*f)->name() << "\"," << *s << "}";
		}
		os << "\n},\n";
		os << "SectorRelations[\"" << (*f)->name() << "\"] -> {\n";
		const map<int, GiNaC::exmap>& srshifts = m->sector_relation_shifts();
		const map<int, Sector>& relations = m->sector_relations();
		map<int, GiNaC::exmap>::const_iterator r;
		for (r = srshifts.begin(); r != srshifts.end(); ++r) {
			if (r != srshifts.begin())  os << ",\n";
			int from_equiv = r->first;
			const GiNaC::exmap& shift = r->second;
			map<int, Sector>::const_iterator equiv = relations.find(from_equiv);
			VERIFY(equiv != relations.end());
			const Sector& to_equiv = equiv->second;
			os << " ";
			Sector(*f, from_equiv).print_mma(os);
			os << " -> {";
			to_equiv.print_mma(os);
			os << ", ";
			for (exmap::const_iterator sr = shift.begin(); sr != shift.end(); ++sr) {
				if (sr != shift.begin())  os << ", ";
				os << sr->first << "->(" << sr->second << ")";
			}
			os << "}";
		}
		os << "\n},\n";
	}
	os << "ReduzeVersion -> \"" << PACKAGE_VERSION << "\"\n";
	os << "}\n";
	os.close();
	LOG("done");
}

#if 0 // OLD VERSION 1
// version 1
void print_finred_generic_equations() {
	using namespace GiNaC;
	const std::string prefix = "eqtempl";
	list<IntegralFamily*> fams = Files::instance()->integralfamilies();
	list<IntegralFamily*>::const_iterator f;
	for (f = fams.begin(); f != fams.end(); ++f) {
		if ((*f)->is_crossed())  continue;
		string filename = prefix + "_" + (*f)->name();
		LOG("Exporting generic equations to file \"" << filename << "\"");
		GenericIdentityGenerator gen(*f);
		LinearCombinationGenericList l, ladd;
		l    = gen.generic_equations("ibp");
		ladd = gen.generic_equations("lorentz");
		l.splice(ladd);
		ofstream os(filename.c_str());
		os << "finred eqtempl 1\n---\n";
		os << (*f)->id() << " " << (*f)->name();
		os << " " << (*f)->num_propagators() << "\n";
		os << l.size() << "\n\n";
		exmap m;
		const exvector& exponents = (*f)->propagator_exponents();
		for (size_t j = 0; j < (*f)->num_propagators(); j++)
			m[exponents[j]] = 0;
		ex d = (*f)->kinematics()->dimension();
		m[d] = 0;
		const symbol* one = (*f)->kinematics()->symbol_to_replace_by_one();
		if (one)  m[*one] = 1;
		LinearCombinationGenericList::const_iterator e;
		for (e = l.begin(); e != l.end(); ++e) {
			os << "\n" << e->size() << "\n";
			LinearCombinationGeneric::const_iterator t;
			// normalize to eliminate denominators
			ex norm = 1;
			for (t = e->begin(); t != e->end(); ++t) {
				for (size_t j = 0; j < (*f)->num_propagators(); j++) {
					ex cj = t->second.coeff(exponents[j], 1).subs(m);
					if (cj.is_zero())  continue;
					norm = lcm(norm, denom(cj));
				}
			}
			// process equation
			for (t = e->begin(); t != e->end(); ++t) {
				INT s(t->first, m);
				size_t n = 0;
				for (size_t i = 0; i < s.size(); ++i)  if (s.v_i(i) != 0) ++n;
				os << "  " << n << " ";
				for (int i = 0; i < (int)s.size(); ++i)
					if (s.v_i(i) == -1)  os << -(i+1) << " ";
					else if (s.v_i(i) == 1)  os << (i+1) << " ";
					else if (s.v_i(i) != 0)
						ERROR("can't handle exponent structure: " << t->first);
				os << "\n";
				ex c = t->second;
				list<pair<int, ex> > symfac;
				ex cd = c.coeff(d, 1).subs(m);
				int cdint = to_int(cd);
				if (cdint != 0)  symfac.push_back(make_pair(0, cdint));
				for (size_t j = 0; j < (*f)->num_propagators(); j++) {
					ex cj = c.coeff(exponents[j], 1).subs(m);
					if (cj.is_zero())  continue;
					symfac.push_back(make_pair(j+1, cj));
				}
				ex check = 0;
				list<pair<int, ex> >::const_iterator p;
				for (p = symfac.begin(); p != symfac.end(); ++p)
					check += p->second*(p->first==0 ? d : exponents[p->first-1]);
				if ((one && check != c.subs(*one == 1)) ||
					(!one && check != c) )
					ERROR("can't handle coefficient structure: " << t->second);
				os << "  " << symfac.size() << " ";
				for (p = symfac.begin(); p != symfac.end(); ++p)
					os << p->first << " ";
				os << "\n    ";
				// note: assume absence of scales here !
				for (p = symfac.begin(); p != symfac.end(); ++p) {
					os << to_int(norm * p->second) << " ";
				}
				os << "\n";
			}
		}
		os << "---\n";
		os.close();
		LOG("done");
	}
}
#endif
#if 0
// version 2
void print_finred_generic_equations() {
	using namespace GiNaC;
	const std::string prefix = "eqtempl";
	list<IntegralFamily*> fams = Files::instance()->integralfamilies();
	list<IntegralFamily*>::const_iterator f;
	for (f = fams.begin(); f != fams.end(); ++f) {
		if ((*f)->is_crossed())  continue;
		string filename = prefix + "_" + (*f)->name();
		LOG("Exporting generic equations to file \"" << filename << "\"");
		GenericIdentityGenerator gen(*f);
		LinearCombinationGenericList l, ladd;
		l    = gen.generic_equations("ibp");
		ladd = gen.generic_equations("lorentz");
		l.splice(ladd);
		ofstream os(filename.c_str());
		os << "finred eqtempl 2\n---\n";
		os << (*f)->name();
		os << " " << (*f)->num_propagators() << "\n";
		const lst& varsraw = (*f)->kinematics()->kinematic_invariants_and_dimension();
		const symbol* one = (*f)->kinematics()->symbol_to_replace_by_one();
		vector<ex> vars;
		for (lst::const_iterator p = varsraw.begin(); p != varsraw.end(); ++p)
			if (!one || !p->is_equal(*one))  vars.push_back(*p);
		os << vars.size() << " ";
		for (size_t i = 0; i < vars.size(); ++i)  os << " " << vars[i];
		os << "\n";
		os << l.size() << "\n\n";
		exmap m;
		const exvector& exponents = (*f)->propagator_exponents();
		for (size_t j = 0; j < (*f)->num_propagators(); j++)
			m[exponents[j]] = 0;
		if (one)  m[*one] = 1;
		LinearCombinationGenericList::const_iterator e;
		for (e = l.begin(); e != l.end(); ++e) {
			os << "\n" << e->size() << "\n";
			LinearCombinationGeneric::const_iterator t;
			// normalize to eliminate denominators
			ex norm = 1;
			for (t = e->begin(); t != e->end(); ++t) {
				for (size_t j = 0; j < (*f)->num_propagators(); j++) {
					ex cj = normal(t->second.coeff(exponents[j], 1).subs(m));
					if (cj.is_zero())  continue;
					norm = lcm(norm, denom(cj));
				}
			}
			// process equation
			for (t = e->begin(); t != e->end(); ++t) {
				INT s(t->first, m);
				size_t n = 0;
				for (size_t i = 0; i < s.size(); ++i)  if (s.v_i(i) != 0) ++n;
				os << "  " << n << " ";
				for (int i = 0; i < (int)s.size(); ++i)
					if (s.v_i(i) == -1)  os << -(i+1) << " ";
					else if (s.v_i(i) == 1)  os << (i+1) << " ";
					else if (s.v_i(i) != 0)
						ERROR("can't handle exponent structure: " << t->first);
				os << "\n";
				ex c = t->second.expand();
				list<pair<int, ex> > symfac;
				ex c0 = c.subs(m);
				if (!c0.is_zero())  symfac.push_back(make_pair(0, c0));
				for (size_t j = 0; j < (*f)->num_propagators(); j++) {
					ex cj = c.coeff(exponents[j], 1).subs(m);
					if (cj.is_zero())  continue;
					symfac.push_back(make_pair(j+1, cj));
				}
				ex check = 0;
				list<pair<int, ex> >::const_iterator p;
				for (p = symfac.begin(); p != symfac.end(); ++p)
					if (p->first == 0)
						check += p->second;
					else
						check += exponents.at(p->first-1) * p->second;
                ex diff = normal((one ? c.subs(*one == 1) : c) - check);
				if (!diff.is_zero()) {
					cout << "reconstruction had delta: " << diff << endl;
					ERROR("can't handle coefficient structure: " << t->second);
				}
				os << "  " << symfac.size() << " ";
				for (p = symfac.begin(); p != symfac.end(); ++p)
					os << p->first << " ";
				for (p = symfac.begin(); p != symfac.end(); ++p) {
					vector<ex> coeff;
					vector<int> degs;
					coeff.push_back((norm * p->second).expand());
					for (size_t v = 0; v < vars.size(); ++v) {
						vector<ex> ncoeff;
						for (size_t j = 0; j < coeff.size(); ++j) {
							size_t deg = coeff[j].degree(vars[v]);
							degs.push_back(deg);
							for (int o = deg; o >= 0; --o)
								ncoeff.push_back(coeff[j].coeff(vars[v], o));
						}
						coeff.swap(ncoeff);
					}
					os << "\n    ";
					for (size_t i = 0; i < degs.size(); ++i)
						os << " " << degs[i];
					os << "  ";
					for (size_t i = 0; i < coeff.size(); ++i)
						os << " " << to_int(coeff[i]);
				}
				os << "\n";
			}
		}
		os << "---\n";
		os.close();
		LOG("done");
	}
}
#endif
#if 1
// version 3
void print_finred_generic_equations() {
    using namespace GiNaC;
    const std::string prefix = "eqtempl";
    list<IntegralFamily*> fams = Files::instance()->integralfamilies();
    list<IntegralFamily*>::const_iterator f;
    for (f = fams.begin(); f != fams.end(); ++f) {
        if ((*f)->is_crossed())  continue;
        string filename = prefix + "_" + (*f)->name();
        LOG("Exporting generic equations to file \"" << filename << "\"");
        GenericIdentityGenerator gen(*f);
        LinearCombinationGenericList l, ladd;
        l    = gen.generic_equations("ibp");
        ladd = gen.generic_equations("lorentz");
        l.splice(ladd);
        ofstream os(filename.c_str());
        os << "finred eqtempl 3\n---\n";
        os << (*f)->name();
        os << " " << (*f)->num_propagators() << "\n";
        const lst& varsraw = (*f)->kinematics()->kinematic_invariants_and_dimension();
        const symbol* one = (*f)->kinematics()->symbol_to_replace_by_one();
        vector<ex> vars;
        for (lst::const_iterator p = varsraw.begin(); p != varsraw.end(); ++p)
            if (!one || !p->is_equal(*one))  vars.push_back(*p);
        os << vars.size() << " ";
        for (size_t i = 0; i < vars.size(); ++i)  os << " " << vars[i];
        os << "\n";
        os << l.size() << "\n\n";
        exmap m;
        const exvector& exponents = (*f)->propagator_exponents();
        for (size_t j = 0; j < (*f)->num_propagators(); j++)
            m[exponents[j]] = 0;
        if (one)  m[*one] = 1;
        LinearCombinationGenericList::const_iterator e;
        for (e = l.begin(); e != l.end(); ++e) {
            os << "\n" << e->size() << "\n";
            LinearCombinationGeneric::const_iterator t;
            // normalize to eliminate denominators
            ex norm = 1;
            for (t = e->begin(); t != e->end(); ++t) {
                for (size_t j = 0; j < (*f)->num_propagators(); j++) {
                    ex cj = normal(t->second.coeff(exponents[j], 1).subs(m));
                    if (cj.is_zero())  continue;
                    norm = lcm(norm, denom(cj));
                }
            }
            // process equation
            for (t = e->begin(); t != e->end(); ++t) {
                INT s(t->first, m);
                size_t n = 0;
                for (size_t i = 0; i < s.size(); ++i)  if (s.v_i(i) != 0) ++n;
                os << " " << n << "\n";
                for (int i = 0; i < (int)s.size(); ++i) {
                   if (s.v_i(i) != 0)
                     os << " " << (i + 1) << " " << (int)s.v_i(i) << "\n";
                }
                ex c = t->second.expand();
                list<pair<int, ex> > symfac;
                ex c0 = c.subs(m);
                if (!c0.is_zero())  symfac.push_back(make_pair(0, c0));
                for (size_t j = 0; j < (*f)->num_propagators(); j++) {
                    ex cj = c.coeff(exponents[j], 1).subs(m);
                    if (cj.is_zero())  continue;
                    symfac.push_back(make_pair(j+1, cj));
                }
                ex check = 0;
                list<pair<int, ex> >::const_iterator p;
                for (p = symfac.begin(); p != symfac.end(); ++p)
                    if (p->first == 0)
                        check += p->second;
                    else
                        check += exponents.at(p->first-1) * p->second;
                ex diff = normal((one ? c.subs(*one == 1) : c) - check);
                if (!diff.is_zero()) {
                    cout << "reconstruction had delta: " << diff << endl;
                    ERROR("can't handle coefficient structure: " << t->second);
                }
                os << "  " << symfac.size() << " ";
                for (p = symfac.begin(); p != symfac.end(); ++p) {
                    vector<ex> coeff;
                    vector<int> degs;
                    coeff.push_back((norm * p->second).expand());
                    for (size_t v = 0; v < vars.size(); ++v) {
                        vector<ex> ncoeff;
                        for (size_t j = 0; j < coeff.size(); ++j) {
                            size_t deg = coeff[j].degree(vars[v]);
                            degs.push_back(deg);
                            for (int o = deg; o >= 0; --o)
                                ncoeff.push_back(coeff[j].coeff(vars[v], o));
                        }
                        coeff.swap(ncoeff);
                    }
                    os << "\n  ";
                    os << p->first << " ";
                    for (size_t i = 0; i < degs.size(); ++i)
                        os << "  " << degs[i];
                    os << "  ";
                    for (size_t i = 0; i < coeff.size(); ++i)
                        os << " " << to_int(coeff[i]);
                }
                os << "\n";
            }
        }
        os << "---\n";
        os.close();
        LOG("done");
    }
}
#endif

void print_finred_zero_sectors() {
	using namespace GiNaC;
	const std::string prefix = "zerosectors";
	list<IntegralFamily*> fams = Files::instance()->integralfamilies();
	list<IntegralFamily*>::const_iterator f;
	for (f = fams.begin(); f != fams.end(); ++f) {
		if ((*f)->is_crossed())  continue;
		string filename = prefix + "_" + (*f)->name();
		const SectorMappings* m = Files::instance()->sectormappings((*f)->name());
		LOG("Exporting zero sectors to file \"" << filename << "\"");
		ofstream os(filename.c_str());
		//os << "finred sectors 1\n---\n";
		//os << (*f)->id() << " " << (*f)->name();
		//os << " " << (*f)->num_propagators() << "\n";
		//os << m->zero_sectors().size() << "\n\n";
		set<int>::const_iterator i;
		for (i = m->zero_sectors().begin(); i != m->zero_sectors().end(); ++i)
			os << *i << "\n";
		//os << "---\n";
		os.close();
		LOG("done");
	}
}

void print_finred_permutations() {
	using namespace GiNaC;
	const std::string prefix = "permutations";
	list<IntegralFamily*> fams = Files::instance()->integralfamilies();
	list<IntegralFamily*>::const_iterator f;
	for (f = fams.begin(); f != fams.end(); ++f) {
		if ((*f)->is_crossed())  continue;
		string filename = prefix + "_" + (*f)->name();
		LOG("Exporting permutations to file \"" << filename << "\"");
		ofstream os(filename.c_str());
		const list<Permutation>& perms = (*f)->permutation_symmetries();
		os << perms.size() << "\n";
		list<Permutation>::const_iterator p;
		for (p = perms.begin(); p != perms.end(); ++p) {
			const vector<vector<int> >& pv = p->vector_rep();
			vector<int> pe((*f)->num_propagators());
			for (size_t i = 0; i < pe.size(); ++i)
				pe[i] = i;
			vector<vector<int> >::const_iterator cyc;
			for (cyc = pv.begin(); cyc != pv.end(); ++cyc) {
				if (cyc->empty())
					continue;
				vector<int>::const_iterator c;
				vector<int>::const_iterator cp = --cyc->end();
				for (c = cyc->begin(); c != cyc->end(); ++c) {
					pe[*cp] = *c;
					if ((++cp) == cyc->end())
						cp = cyc->begin();
				}
			}
			for (size_t i = 0; i < pe.size(); ++i)
				os << pe[i] << " ";
			os << "\n";
		}
		os.close();
		LOG("done");
	}
}

void print_finred_shifts() {
	using namespace GiNaC;
	const std::string prefix = "sectorrelations";
	list<IntegralFamily*> fams = Files::instance()->integralfamilies();
	list<IntegralFamily*>::const_iterator f;
	for (f = fams.begin(); f != fams.end(); ++f) {
		if ((*f)->is_crossed())  continue;
		string filename = prefix + "_" + (*f)->name();
		LOG("Exporting sector relations to file \"" << filename << "\"");
		ofstream os(filename.c_str());
		exmap scale2one;
		const symbol* one = (*f)->kinematics()->symbol_to_replace_by_one();
		if (one)  scale2one[*one] = 1;
		const SectorMappings* m = Files::instance()->sectormappings((*f)->name());
		const map<int, Sector>& rels = m->sector_relations();
		const map<int, GiNaC::exmap>& shifts = m->sector_relation_shifts();
		os << rels.size() << '\n';
		map<int, Sector>::const_iterator r = rels.begin();
		map<int, GiNaC::exmap>::const_iterator s = shifts.begin();
		for (; r != rels.end() && s != shifts.end(); ++r, ++s) {
			//cout << "  " << Sector(*f, r->first) << " -> " << r->second << endl;
			VERIFY(r->first == s->first);
			const IntegralFamily* tf = r->second.integralfamily();
			VERIFY((*f)->kinematics() == tf->kinematics());
			VERIFY((*f)->propagators().size() == tf->propagators().size());
			const exmap& sp2prop = tf->rules_sp_to_prop();
			const exmap& sp2inv = tf->kinematics()->rules_sp_to_invariants();
			Sector rmin = Crossing::to_minimal_crossing(r->second);
			os << r->first << "  " << rmin.integralfamily()->name() << " " << rmin.id() << "  ";
			for (size_t i = 0; i < tf->propagators().size(); ++i) {
				ex sp = (*f)->propagators()[i].inverse_ex().subs(s->second).expand();
				ex props = sp.subs(sp2prop).subs(sp2inv);
				OPSUM l = ex_to_OPSUM(props, tf->propagators());
				os << l.size() << " ";
				for (OPSUM::const_iterator o = l.begin(); o != l.end(); ++o) {
					const OPPROD& p = o->first;
					VERIFY(p.size() == 1);
					os << -p.begin()->n() << " ";
					ex cn = o->second.subs(scale2one);
					VERIFY(is_a<numeric>(cn) && ex_to<numeric>(cn).is_rational());
					os << cn << " ";
				}
				os << " ";
			}
			os << '\n';
		}
		os.close();
		LOG("done");
	}
}

void print_finred_symmetries() {
	using namespace GiNaC;
	const std::string prefix = "sectorsymmetries";
	list<IntegralFamily*> fams = Files::instance()->integralfamilies();
	list<IntegralFamily*>::const_iterator f;
	for (f = fams.begin(); f != fams.end(); ++f) {
		if ((*f)->is_crossed())  continue;
		string filename = prefix + "_" + (*f)->name();
		LOG("Exporting sector symmetries to file \"" << filename << "\"");
		ofstream os(filename.c_str());
		exmap scale2one;
		const symbol* one = (*f)->kinematics()->symbol_to_replace_by_one();
		if (one)  scale2one[*one] = 1;
		const SectorMappings* m = Files::instance()->sectormappings((*f)->name());
		const exmap& sp2prop = (*f)->rules_sp_to_prop();
		const exmap& sp2inv = (*f)->kinematics()->rules_sp_to_invariants();
		const map<int, list<exmap> >& ss = m->sector_symmetries();
		os << ss.size() << '\n';
		map<int, list<exmap> >::const_iterator p;
		for (p = ss.begin(); p != ss.end(); ++p) {
			os << p->first << "  " << p->second.size() << "\n";
			list<exmap>::const_iterator s;
			for (s = p->second.begin(); s != p->second.end(); ++s) {
				os << "  ";
				for (size_t i = 0; i < (*f)->propagators().size(); ++i) {
					ex sp = (*f)->propagators()[i].inverse_ex().subs(*s).expand();
					ex props = sp.subs(sp2prop).subs(sp2inv);
					OPSUM l = ex_to_OPSUM(props, (*f)->propagators());
					os << l.size() << " ";
					for (OPSUM::const_iterator o = l.begin(); o != l.end(); ++o) {
						const OPPROD& p = o->first;
						VERIFY(p.size() == 1);
						os << -p.begin()->n() << " ";
						ex cn = o->second.subs(scale2one);
						VERIFY(is_a<numeric>(cn) && ex_to<numeric>(cn).is_rational());
						os << cn << " ";
					}
					os << " ";
				}
				os << '\n';
			}
		}
		os.close();
		LOG("done");
	}
}

void PrintSectorInfo::run_serial() {
	if (!config_info_mma_file_.empty())
		print_config_info_mma(config_info_mma_file_);
	if (!sector_mappings_mma_file_.empty())
		print_sector_mappings_mma(sector_mappings_mma_file_);
	if (generate_finred_config_) {
		//not implemented in presence of scales
		//print_finred_symmetries();
		//print_finred_shifts();
		print_finred_permutations();
		print_finred_generic_equations();
		print_finred_zero_sectors();
	}

	map<Sector, int> multiplicities; // TODO: insert num into DOT graphic
	const set<Sector> sectors = find_sectors(sector_selection_, integral_file_,
			multiplicities);
	LOG("\nAnalyzing " << sectors.size() << " sector(s)\n");

	LOG("Determining compact recursive sector selection");
	SectorSelection sectree =
			SectorSelection::find_compact_recursive_selection(sectors, true);
	YAML::Emitter ye;
	ye << YAML::BeginSeq << YAML::BeginSeq << YAML::BeginSeq;
	ye << YAML::BeginMap << YAML::Key << "sector_selection" << YAML::Value
			<< sectree << YAML::EndMap;
	ye << YAML::EndSeq << YAML::EndSeq << YAML::EndSeq;
	LOG("\nThis is an optimized selection of sectors to be used for reductions:");
	LOG(ye.c_str() << "\n\n");

	LOG("Details for all sectors:\n");
	set<Sector>::const_iterator s;
	size_t w_sec(17), w_sub_dir(7), w_sub_all(6), w_prop(40);
	for (s = sectors.begin(); s != sectors.end(); ++s) {
		w_sec = max(w_sec, to_string(*s).size() + 2);
		w_prop = max(w_prop,
				to_string(INT(s->integralfamily(), s->id())).size() + 1);
		w_sub_dir = max(w_sub_dir, to_string(s->t()).size() + 1);
		w_sub_all = max(w_sub_all, to_string(s->id()).size() + 2);
	}

	// print table
	if (true) {
		LOG(setw(w_sec) << left << "sector"
				<< setw(w_sub_dir + w_sub_all) << "subsectors"
				<< setw(w_prop) << "corner integral");
		LOG(setw(w_sec) << left << "(fam:t:id)"
				<< setw(w_sub_dir) << "direct"
				<< setw(w_sub_all) << "all"
				<< setw(w_prop) << "fam t id  r s  propagator_exponents");
		LOG(setw(w_sec + w_sub_dir + w_sub_all + w_prop)
				<< setfill('.') << "" << setfill(' '));
		for (s = sectors.begin(); s != sectors.end(); ++s)
			LOG(setw(w_sec) << to_string(*s)
					<< setw(w_sub_dir) << SectorSelection::find_prerequisite_sectors(*s, false).size()
					<< setw(w_sub_all) << SectorSelection::find_prerequisite_sectors(*s, true).size()
					<< setw(w_prop) << to_string(INT(*s)));
		LOG(setw(w_sec + w_sub_dir + w_sub_all + w_prop)
				<< setfill('.') << "" << setfill(' '));
	}

	// subsectors
	if (true) {
		for (s = sectors.begin(); s != sectors.end(); ++s) {
			LOG("\nSector " << *s);
			stringstream ssd, ssa;
			set<Sector> a =
					SectorSelection::find_prerequisite_sectors(*s, true);
			copy(a.begin(), a.end(), ostream_iterator<Sector> (ssa, " "));
			LOG("  all prerequisite sectors (subsectors): " << ssa.str());
		}
	}

	if (!(feynman_diagram_dot_file_.empty() && feynman_diagram_mma_file_.empty())) {
		string dotfiletmp = feynman_diagram_dot_file_ + ".tmp";
		string psfilename = feynman_diagram_dot_file_ + ".ps";
		string mmafiletmp = feynman_diagram_mma_file_ + ".tmp";
		LOG("\nFind Feynman diagrams (unphysical and known zero sectors excluded)");
		ProgressBar pbar(0, "find diagrams", sectors.size());
		pbar.start();
		ofstream of, ofmma;
		if (!feynman_diagram_dot_file_.empty())
			of.open(dotfiletmp.c_str());
		if (!feynman_diagram_mma_file_.empty()) {
			ofmma.open(mmafiletmp.c_str());
			ofmma << "{\n";
		}
		int num_dias(0);
		set<Sector> unphysical, zeros;
		// bottom up!
		for (s = sectors.begin(); s != sectors.end(); ++s) {
			pbar.print();
			if (true) {
				set<Sector> subsecs =
						SectorSelection::find_prerequisite_sectors(*s, false);
				set<Sector>::const_iterator sub, found = unphysical.end();
				for (sub = subsecs.begin(); sub != subsecs.end(); ++sub) {
					if ((found = unphysical.find(*sub)) != unphysical.end())
						break;
				}
				if (found != unphysical.end()) {
					LOGX("No graph for sector "
							<< s->get_safe_string_for_filename()
							<< "\n  has unphysical subsector "
							<< found->get_safe_string_for_filename());
					unphysical.insert(*s);
					continue;
				}
			}
			SectorGraph g;
			bool u_poly_non_zero;
			if (!s->find_graph(g, u_poly_non_zero, construct_minimal_graphs_)) {
				LOGX("No graph for sector "
						<< s->get_safe_string_for_filename());
				if (u_poly_non_zero) {
					unphysical.insert(*s);
					LOGX("  is not trivially zero.");
				} else {
					LOGX("  is trivially zero.");
					zeros.insert(*s);
				}
				continue;
			}
			++num_dias;
			if (minimize_graphs_by_twists_)
				g.minimize_by_twists();
			if (!feynman_diagram_dot_file_.empty()) {
				g.print_dot(of);
				of << "\n";
			}
			if (!feynman_diagram_mma_file_.empty()) {
				if (s != sectors.begin())
					ofmma << ",\n";
				ofmma << "{\"" << s->integralfamily()->name() << "\", "
						<< s->t() << ", " << s->id() << ", ";
				g.print_mma(ofmma);
				ofmma << "}";
			}
			LOGX("Found graph for sector " << s->get_safe_string_for_filename());
		}
		pbar.end();
		LOG("Number of printed Feynamn diagrams: " << num_dias);
		LOG("Number of omitted Feynamn diagrams: (U-poly not zero, U-poly zero) (" << unphysical.size() << ", " << zeros.size() << ")");
        if (!feynman_diagram_dot_file_.empty()) {
			of.close();
			rename(dotfiletmp, feynman_diagram_dot_file_);
			LOG("Generated file \"" << feynman_diagram_dot_file_ << "\".");
			LOG("To produce a postscript file run the following command:");
			LOG("neato -Tps " << feynman_diagram_dot_file_ << " -o " << psfilename);
        }
        if (!feynman_diagram_mma_file_.empty()) {
			ofmma.close();
			rename(mmafiletmp, feynman_diagram_mma_file_);
			LOG("Generated file \"" << feynman_diagram_mma_file_ << "\".");
        }
	}

	// \todo possible to print only physical sectors
	if (!sector_tree_dot_file_.empty()) {
		string dotfilename = sector_tree_dot_file_;
		string psfilename = dotfilename + ".ps";
		LOG("\nGenerating relationship graph file: " << dotfilename);
		SectorMappings::print_sectortree_dot(sectors, dotfilename);
		LOG("To produce a postscript file run the following command:");
		LOG("dot -Tps " << dotfilename << " -o " << psfilename);
		LOG("");
	}
}

bool PrintSectorInfo::find_dependencies(const set<string>& outothers,//
		list<string>& in, list<string>& out, list<Job*>& auxjobs) {
	if (!sector_tree_dot_file_.empty())
		out.push_back(sector_tree_dot_file_);
	if (!feynman_diagram_dot_file_.empty())
		out.push_back(feynman_diagram_dot_file_);
	if (!feynman_diagram_mma_file_.empty())
		out.push_back(feynman_diagram_mma_file_);
	return true;
}

void PrintSectorInfo::print_manual_options(YAML::Emitter& os) const {
	os << YAML::Key << "sector_selection" << YAML::Value << sector_selection_;
}

void PrintSectorInfo::read_manual_options(const YAML::Node& node) {
	if (node.FindValue("sector_selection"))
		node["sector_selection"] >> sector_selection_;
}

void PrintSectorInfo::init() {
	if (!sector_tree_dot_file_.empty() && sector_tree_dot_file_
			== feynman_diagram_dot_file_)
		ABORT("sector_tree_dot_file and feynman_diagram_dot_file are identical: "
				<< sector_tree_dot_file_);
}

std::string PrintSectorInfo::get_description() const {
	return string("print sector info");
}

}
