/*  job_reducesectors.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef JOB_REDUCESECTORS_H_
#define JOB_REDUCESECTORS_H_

#include "job.h"
#include <map>
#include "rspoint.h"
#include "reduzer.h"
#include "sector.h"

namespace Reduze {

/// Job to reduces a collection of sectors
class ReduceSectors: public Job {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("reduce_sectors");
		s.set_short_description(
				"Reduces integrals from a selection of sectors.");
		s.set_long_description(""//
					"Main job to reduce integrals from a selection of sectors"
					" to master integrals. The job sets up auxiliary jobs for"
					" the reduction of single sectors which are then launched"
					" in a distributed way if MPI is set up. In the reduction"
					" procedure first the lowest (simplest) sectors are reduced"
					" and then these results are used as substitutions in the"
					" reduction of higher sectors. Different types of identities"
					" can be chosen to set up the systems of equations: most"
					" common are IBP, LI and symmetry identities. Sector relations"
					" defined in the sector mappings files are automatically"
					" applied to the generated identities in order to shift"
					" integrals to lower sectors."
					" Since these shift relations can"
					" introduce new integrals from sub-sectors the replacement"
					" is done in a recursive way and the reduction of a"
					" sector therefore depends on the reduction of its whole"
					" sub-sector tree. The results of a reduction are saved"
					" in the standard directory 'reduction' with default file"
					" names.");

		// if description or defaults change, please check them also in RunReduction
		s.add_option("sector_selection", true, "sector_selection", ""//
					"A selection of sectors which should be reduced. The"
					" selected sectors are automatically replaced by their"
					" minimal equivalents by global permutation symmetries"
					" and sector relations as defined in the sector"
					" mappings files.");
		s.add_option("alternative_input_directory", false, "string", ""//
					"Directory with additional identities for some sectors"
					" which are included when the reduction of these sectors is"
					" requested. In order to be recognized, these files must"
					" follow the file name conventions for reduction results"
					" in the 'reductions' directory."
					" Pure substitutions for sub-sectors are not"
					" loaded from this directory."
					" This option allows to extend existing reduction results "
					" with additional identities in a resumable way:"
					" just move the old"
					" 'reductions' directory to a new directory and"
					" specify the new directory as alternative_input_directory"
					" and set conditional to true in the new run.");
		s.add_option("shared_equation_files", false, "sequence of strings",
					"File names for equations to be included for all sectors."
				    " The identities are treated just like IBP identities for"
				    " the respective sector.");
		s.add_option("shared_substitution_files", false, "sequence of strings",
					"File names for substitutions to be used for all sectors.");
		s.add_option("preferred_masters_file", false, "string",
					"Input file with preferred master integrals.");
		s.add_option("identities", true, "identity_generic_selection", ""//
					"Type of identities (like IBPs, LIs, sector_symmetries) to"
					" be generated. There is no need to use the sector_relations"
					" identities explicitly since they are used anyway to"
					" transform the integrals into minimal equivalents.");
		s.add_option("solutions", false, "reduction_generic_options", ""//
					"Options and generic selections of integrals");
		s.add_option("reduzer_options", false, "reduzer_options", ""//
					"Options for the reduction algorithm.");
		s.add_option("num_seeds_for_identities_auxjobs", false, "integer >= 1",
				"" //
					"Number of seeds per auxiliary job to generate the identities");
#ifdef HAVE_MPI
		s.add_option("use_full_parallelization", false, "boolean", ""//
				"Whether to reduce identities for each sector in a distributed"
				" way employing worker processes."
				" Each distributed reduction job requires one MPI process"
				" for the manager of the job plus at least one MPI process"
				" for a worker (in addition to the \"shared\" MPI process"
				" for the JobCenter).");
		s.add_option("min_workers", false, "integer >= 1", ""//
				" Only relevant if use_full_parallelization is true."
				" Minimum number of worker processes in the distributed"
				" reduction of a single sector.");
		s.add_option("max_workers", false, "integer >= 1", ""//
				" Only relevant if use_full_parallelization is true."
				" Maximum number of worker processes in the distributed"
				" reduction of a single sector.");
#endif
		s.add_option("perform_reductions", false, "boolean", ""//
					"Whether to perform the actual reductions at all. Set"
					" this to false to just generate the identities.");
		s.add_option("delete_temporary_files", false, "boolean", ""//
					"Whether temporary files with seed integrals and generated"
					" identities should be removed after a reduction.");
		s.add_option("run_distribute_external", false, "boolean", ""//
					"Whether to automatically clean up reductions/external*"
				    " files after the primary reduction runs. Usually it is "
				    " a good idea to stick with the default value true.");
		s.add_options(Job::yaml_spec());
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}

	ReduceSectors() ://
				num_seeds_for_identities_auxjobs_(1000)
#ifdef HAVE_MPI
				, use_full_parallelization_(true), min_workers_(1), max_workers_(
						1000)
#endif
				, perform_reductions_(true), delete_temporary_files_(true),
				run_distribute_external_(false) {
		add_auto_options();
	}
	virtual ~ReduceSectors() {
	}

	virtual std::string get_description() const;
	virtual void run_serial();
	virtual bool find_dependencies(const std::set<std::string>&,//
			std::list<std::string>&, std::list<std::string>&, std::list<Job*>&);

protected:
	virtual void add_auto_options() {
		add_auto_io("sector_selection", sector_selection_);
		add_auto_io("alternative_input_directory", alternative_input_directory_);
		add_auto_io("shared_equation_files", shared_equation_files_);
		add_auto_io("shared_substitution_files", shared_substitution_files_);
		add_auto_io("preferred_masters_file", preferred_masters_file_);
		add_auto_io("identities", identities_);
		add_auto_io("solutions", reduction_generic_options_);
		add_auto_io("reduzer_options", reduzer_options_);
		add_auto_io("num_seeds_for_identities_auxjobs",
				num_seeds_for_identities_auxjobs_);
#ifdef HAVE_MPI
		add_auto_io("use_full_parallelization", use_full_parallelization_);
		add_auto_io("min_workers", min_workers_);
		add_auto_io("max_workers", max_workers_);
#endif
		add_auto_io("perform_reductions", perform_reductions_);
		add_auto_io("delete_temporary_files", delete_temporary_files_);
		add_auto_io("run_distribute_external", run_distribute_external_);
	}
	virtual void init();
	/// temporary directory where database and identities/seeds are stored
	std::string get_tmp_dir(const Sector& sector) const;
	/// this directory is inside "get_tmp_directory()"
	std::string get_tmp_dir_seeds(const Sector& sector) const;
	/// this directory is inside "get_tmp_directory()"
	std::string get_tmp_dir_identities(const Sector& sector) const;

private:
	/// sectors to be reduced
	SectorSelection sector_selection_;

	/// alternative directory for input of previous reduction for this sector
	std::string alternative_input_directory_;
	/// extra identities
	std::list<std::string> shared_equation_files_, shared_substitution_files_;
	/// preferred master integrals
	std::string preferred_masters_file_;
	/// identities to be generated: type of identity -> finite (r,s) range
	IdentityGenericSelection identities_;
	///
	ReductionGenericOptions reduction_generic_options_;
	/// details for the reduction algorithm
	ReduzerOptions reduzer_options_;
	/// option for generating the identities
	int num_seeds_for_identities_auxjobs_;
#ifdef HAVE_MPI
	/// distributed reductions switch
	bool use_full_parallelization_;
	int min_workers_, max_workers_;
#endif
	bool perform_reductions_, delete_temporary_files_, run_distribute_external_;
};

}

#endif /* JOB_REDUCESECTORS_H_ */
