/*  main.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include <ginac/ginac.h>
#include <signal.h>

#include "functions.h"
#include <unistd.h>
#include <time.h>
#include "files.h"
#include "jobqueue.h"
#include "kinematics.h"
#include "ginacutils.h"
#include "globalsymbols.h"
#include "int.h"

#ifdef HAVE_DATABASE
#include <dbstl_common.h> // global exit function
#endif

#ifdef HAVE_MPI
#include "jobcenter.h"
#endif
#ifdef HAVE_FERMAT
#include "fermat.h"
#endif

using namespace std;
using namespace GiNaC;

using namespace Reduze;

namespace Reduze {

#ifdef HAVE_MPI
namespace {
	enum Cmd {
		CmdExit, CmdJobCenterCustomer
	};
}
#endif

std::string version() {
	stringstream os;
	os << PACKAGE_NAME << " " << PACKAGE_VERSION;
#ifdef HAVE_MPI
	os << " (MPI build)" << endl;
#else
	os << " (non-MPI build)" << endl;
#endif
	return os.str();
}

std::string usage() {
	stringstream os;
	string emptyspace = "                           ";
	os << "\nUsage:\n";
	os << "  reduze FILE              "
			<< "Runs the jobs specified in the job file FILE.\n" << emptyspace
			<< "The project directory is set to the working directory.\n";
	os << "  reduze -v                " << "Prints the version of Reduze.\n";
	os << "  reduze -h                " << "Prints this help.\n";
	os << "  reduze -h jobs           " << "Lists available jobs.\n";
	os << "  reduze -h topics         " << "Lists help topics.\n";
	os << "  reduze -h TOPIC          "
			<< "Prints extended help for a topic, e.g. for a job name.\n";
#ifdef HAVE_MPI
	os << "This is an MPI build which supports parallel modes.\n";
#else
	os << "This is a non-MPI build which supports serial mode only.\n";
#endif
	os << endl;
	return os.str();
}

// initialization of Fermat
#ifdef HAVE_FERMAT
namespace {
	// Fermat must be initialized very early, delay output of errors somewhat
	std::string fermat_error;
}

void init_fermat() {
	try {
		srand(time(0));
		const lst& symbols = Files::instance()->kinematics()->kinematic_invariants_and_dimension();
		const string& exec = Files::instance()->paths()->fermat();
		if (exec.empty()) {
			fermat_error = "Fermat turned off since no executable defined";
			return;
		}
		bool use_laurent_poly = false;
		Fermat::instance()->init(symbols, exec, use_laurent_poly);
	} catch (missing_file_error& e) { // silently ignore if global.yaml does not exist
		fermat_error = e.what();
	} catch (missing_key_error& e) { // silently ignore if no paths defined in global.yaml
		fermat_error = e.what();
	}
}
#endif

void init_globaloptions() {
	INT::set_ordering(Files::instance()->globaloptions()->integral_ordering);
}

void run_root(int argc, char* argv[]) {
#ifdef HAVE_MPI
	MPI::Intracomm comm(MPI::COMM_WORLD);
	int cmd_exit = CmdExit;
	int cmd_jobs = CmdJobCenterCustomer;
#endif
	ASSERT(argc >= 1);
	vector<string> opts(argv + 1, argv + argc);

	if (opts.size() == 1 && opts[0].size() > 0 && opts[0][0] != '-') {
		std::cout << welcome();
		string jobfile = opts[0];
		if (!is_readable_file(jobfile))
			ERROR("Cannot read job file: " << jobfile);
		Timer timer;
		string tstr = time_string();
		LOG("Run on " << tstr.substr(0, tstr.length() - 1) << " with ID " //
				<< RunIdentification::instance()->string());
		LOG("Using job file " << jobfile);
		string dir = ""; // get_directory_of_filename(jobfile);
		LOG("Setting project directory to " //
				<< (dir == "" ? "current working directory" : dir));
		Files* files = Files::instance();
		files->set_project_directory(dir);
		LOG("Log files are written to " << files->get_log_run_directory());
		init_globaloptions();
#ifdef HAVE_FERMAT
		if(!Fermat::instance()->is_initialized() || !fermat_error.empty()) {
			if (!fermat_error.empty()) {
				LOG(fermat_error);
			}
			LOG("Failed to load Fermat, using GiNaCs normal() method instead");
		} else {
			LOG("Fermat successfully initialized");
		}
#else
		LOG("This build does not support Fermat");
#endif
		string logfn = files->get_log_run_directory() + "j00_"
				+ get_filename_without_directory(jobfile) + ".log";
		rlog2.set_file(logfn);
		rlog2 << "\nDate: " << time_string() << "\n";
		LOG("Setting up job queue from " << jobfile);
		JobQueue jobqueue;
		try {
			jobqueue.read(jobfile);
			//Files::instance()->read(jobqueue, jobfile);
		} catch (exception& e) {
			ERROR(e.what());
		}
		jobqueue.resolve_dependencies(); // create aux jobs if necessary
#ifdef HAVE_MPI
		int n = comm.Get_size();
		if (n > 1) {
			LOG("Processing jobs in parallel mode (" << n << " processes)");
			long long runid = RunIdentification::instance()->number();
			comm.Bcast(&runid, 1, MPI::LONG_LONG, comm.Get_rank());
			comm.Bcast(&cmd_jobs, 1, MPI::INT, comm.Get_rank());
			JobCenter jobcenter(&comm, &jobqueue);
			jobcenter.run();
		} else {
			LOG("Processing jobs in serial mode (only one MPI process)");
			jobqueue.run();
		}
#else
		LOG("Processing jobs in serial mode (non-MPI build)");
		jobqueue.run();
#endif
		LOG("Total run time: " << fixed << setprecision(0) //
				<< timer.get_wall_time() << resetiosflags(ios::fixed) << " s");
		return;
	} else if (opts.size() < 1) {
		std::cout << usage();
		//ABORT("Too few arguments, type 'reduze -h' for usage information.");
	} else if (opts.size() == 1
			&& (opts[0] == "-v" || opts[0] == "--version")) {
		std::cout << version();
	} else if (opts.size() == 1 && (opts[0] == "-h" || opts[0] == "--help")) {
		std::cout << usage();
	} else if (opts.size() == 2 && (opts[0] == "-h" || opts[0] == "--help")) {
		if (opts[1] == "jobs")
			JobQueue::print_jobs(std::cout);
		else if (opts[1] == "topics")
			YAMLFactory::instance().print_help_topics(std::cout);
		else
			YAMLFactory::instance().print_help(opts[1], std::cout);
	} else {
		//ABORT("Unknown arguments, type 'reduze -h' for usage information.");
		cout << "Unknown arguments, type 'reduze -h' for usage information."
				<< endl;
	}

#ifdef HAVE_MPI
	comm.Bcast(&cmd_exit, 1, MPI::INT, comm.Get_rank());
#endif
}

#ifdef HAVE_MPI
void run_leave(int argc, char* argv[], int jobcenter_rank) {
	MPI::Intracomm comm(MPI::COMM_WORLD);
	init_globaloptions();
	long long runid;
	int cmd;
	comm.Bcast(&runid, 1, MPI::LONG_LONG, jobcenter_rank);
	RunIdentification::instance()->set_number(runid);
	LOGX("received run id " << RunIdentification::instance()->string());
	comm.Bcast(&cmd, 1, MPI::INT, jobcenter_rank);
	if (cmd == CmdJobCenterCustomer) {
		LOGX("switch to jobcentercustomer");
		JobCenterCustomer customer(&MPI::COMM_WORLD, jobcenter_rank);
		customer.run();
	} else if (cmd == CmdExit) {
		return;
	} else {
		ABORT("Received unknown command");
	}
}
#endif // HAVE_MPI
} // namespace Reduze

void process_sigterm(int /*param*/) {
	ERROR("Caught SIGTERM");
}

void cleanup() {
#ifdef HAVE_DATABASE
	dbstl::dbstl_exit();
#endif
#ifdef HAVE_FERMAT
	Fermat::instance()->close();
#endif
}

int main(int argc, char* argv[]) {
	rlog1.set_stdout();

	// handle SIGTERM
	void (*prev_fn)(int);
	prev_fn = signal(SIGTERM, process_sigterm);
	if (prev_fn == SIG_IGN) {
		cerr << "Can't set signal handler" << endl;
		signal(SIGTERM, SIG_IGN);
	}

	// global initializations
	init_ginacutils();
	if(argc > 1) {
		string s(argv[1]);
		if(!s.empty() && s[0] != '-') {
#ifdef HAVE_FERMAT
			init_fermat();
#endif
		}
	}
	try {
#ifdef HAVE_MPI
		MPI::Init(argc, argv);
#ifdef DEBUG
		std::stringstream ss;
		ss << "reduze.log.debug." << MPI::COMM_WORLD.Get_rank();
		rlog3.set_file(ss.str());
#endif
		int jobcenter_rank = 0;
		if (MPI::COMM_WORLD.Get_rank() == jobcenter_rank) {
			//LOG("running root");
			run_root(argc, argv);
		} else {
			rlog1.set_null();
			LOG("running leave");
			run_leave(argc, argv, jobcenter_rank);
		}
		MPI::Finalize();
#else  // HAVE_MPI
#ifdef DEBUG
		std::stringstream ss;
		ss << "reduze.log.debug";
		rlog3.set_file(ss.str());
#endif
		rlog1.set_stdout();
		//LOG("running root (non MPI)");
		run_root(argc, argv);
#endif // HAVE_MPI
	} catch (KnownError& e) {
		cleanup();
		r_abort(-1);
	}
	//catch (...) {
	//	cleanup();
	//	throw; // this hides the trace of the exception
	//}
	cleanup();
	return 0;
}
