/*  reduzer.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "reduzer.h"

#include "integralfamily.h"
#include <iostream>
#include "int.h"
#include "files.h"
#include "filedata.h"

using namespace std;
#ifdef HAVE_DATABASE
using namespace dbstl;
#endif

namespace Reduze {

REGISTER_FUNCTION(Coeff, dummy())

// register type
namespace {
YAMLProxy<ReduzerOptions> dummy;
}

//-----------------------------------------------------------------------------
// ReduzerOptions
//-----------------------------------------------------------------------------

void ReduzerOptions::init() {
	if (!back_substitute_helpers && (num_equations_per_subjob > 1
			|| num_blocks_per_subjob > 1)) {
		// this restriction is due to the fact that the current implementation
		// can't correctly switch off back substitutions only for part of the
		// equations in a subjob
		WARNING("forcing num_equations_per_subjob = 1 because of back_substitute_helpers = false");
		WARNING("forcing num_blocks_per_subjob = 1 because of back_substitute_helpers = false");
		num_equations_per_subjob = 1;
		num_blocks_per_subjob = 1;
	}
	if (num_equations_per_subjob < 1)
		ERROR("Value of option \"num_equations_per_subjob\" must be 1 or higher.");
}

//-----------------------------------------------------------------------------
// ReduzerBase
//-----------------------------------------------------------------------------

template<StorageType storage_type>
ReduzerBase<storage_type>::ReduzerBase(const ReduzerOptions& opts) :
			options_(opts),//
			integ(0), coeff(0), subst(0), blocks(0), criticals(0), helpers(0),
			external(0), substinteg(0),//
			maxint_processed(0), done(Unknown), findnextbegin(0) {
}

template<StorageType storage_type>
ReduzerBase<storage_type>::~ReduzerBase() {
}

template<StorageType storage_type>
void ReduzerBase<storage_type>::clear() {
	done = Unknown;
	findnextbegin = 0;
}

template<StorageType storage_type>
void ReduzerBase<storage_type>::load(const list<string>& eq_files,//
		const list<string>& subst_files,//
		const set<INT>& all_integrals,//
		const unord_set_INT& identity, //
		const set<INT>& frgn,//
		const set<INT>& help,//
		const set<INT>& crit,//
		const set<INT>& disc) {

	if (!integ)
		ABORT("Reduzer not initialized");

	begin_transaction();

	LOGX("Building INT translations");
	unord_map_INT integ_abbr;
	set<INT>::const_iterator it = all_integrals.begin();
	for (INTIndex integ_ind = 1; it != all_integrals.end(); ++integ_ind, ++it) {
#ifdef USE_HASH_TABLE
		integ_abbr.insert(make_pair(*it, integ_ind));
#else
		integ_abbr.insert(integ_abbr.end(), make_pair(*it, integ_ind));
#endif
		integ->insert(integ->end(), make_pair(integ_ind, to_string(*it)));
	}

	LOGX("Insert integrals for algorithm control:");
	set<INT>::const_iterator iit;
	for (iit = crit.begin(); iit != crit.end(); ++iit)
		criticals->insert(integ_abbr[*iit]);
	for (iit = help.begin(); iit != help.end(); ++iit)
		helpers->insert(integ_abbr[*iit]);
	for (iit = frgn.begin(); iit != frgn.end(); ++iit)
		external->insert(integ_abbr[*iit]);
	for (iit = disc.begin(); iit != disc.end(); ++iit)
		substinteg->insert(integ_abbr[*iit]);

	LOGX("Insert substitutions");
	list<string>::const_iterator fit;
	for (fit = subst_files.begin(); fit != subst_files.end(); ++fit)
		insert_substitutions(*fit, integ_abbr, identity);

	LOGX("Insert equations to be reduced");
	for (fit = eq_files.begin(); fit != eq_files.end(); ++fit)
		insert_equations(*fit, integ_abbr);

	commit_transaction();

	LOGX("Starting with " << blocks->size() << " reducible integrals");
	LOGX("Setup done");
}

template<StorageType storage_type>
bool ReduzerBase<storage_type>::completed() {
	if (!integ)
		ABORT("Reduzer not initialized");
	switch (done) {
	case True:
		return true;
	case False:
		return false;
	case Unknown:
		begin_transaction(); /// \todo: remove this in safe way (logically all readonly)
		find_next_block(true);
		commit_transaction();
	}
	return completed();
}

template<StorageType storage_type>
EquationXLight ReduzerBase<storage_type>::abbreviate(const EquationHLight& eq,
		const unord_map_INT& integ_abbr) {
	CoeffIndex coeff_ind = (coeff->empty() ? 0 : coeff->rbegin()->first); // highest index so far
	EquationXLight leq;
	for (EquationHLight::const_iterator tit = eq.begin(); tit != eq.end(); ++tit) {
		++coeff_ind; // new highest index (start labeling at 1)
		if (tit->second.empty())
			ABORT("empty coefficient");
		(*coeff)[coeff_ind] = tit->second;
		unord_map_INT::const_iterator ia = integ_abbr.find(tit->first);
		if (ia == integ_abbr.end())
			ABORT("no abbreviation available for integral " << tit->first);
		leq.insert_end(ia->second, coeff_ind);
	}
	return leq;
}

template<StorageType storage_type>
EquationXLight ReduzerBase<storage_type>::abbreviate(const EquationLight& eq) {
	CoeffIndex coeff_ind = (coeff->empty() ? 0 : coeff->rbegin()->first); // highest index so far
	EquationXLight leq;
	for (EquationLight::const_iterator tit = eq.begin(); tit != eq.end(); ++tit) {
		++coeff_ind; // new highest index (start labeling at 1)
		if (tit->second.empty())
			ABORT("empty coefficient");
		(*coeff)[coeff_ind] = tit->second;
		leq.insert_end(tit->first, coeff_ind);
	}
	return leq;
}

template<StorageType storage_type>
EquationXLightList ReduzerBase<storage_type>::abbreviate(
		const EquationLightList& eqs) {
	EquationXLightList l;
	EquationLightList::const_iterator it;
	for (it = eqs.begin(); it != eqs.end(); ++it)
		l.push_back(abbreviate(*it));
	return l;
}

template<StorageType storage_type>
EquationLight ReduzerBase<storage_type>::unabbreviate_coefficients(
		const EquationXLight& leq) const {
	EquationLight eq;
	for (EquationXLight::const_iterator it = leq.begin(); it != leq.end(); ++it) {
		if (options_.mask_substitutions && //
				substinteg->find(it->first) != substinteg->end()) {
			string c = string("Coeff(") + to_string(it->second) + ")";
			eq.insert_end(it->first, c);
		} else {
			typename MapTraits<storage_type, CoeffIndex, string>::type::const_iterator c =
					coeff->find(it->second);
			if (c == coeff->end())
				ABORT("Can't resolve coefficient abbreviation " << it->second);
			eq.insert_end(it->first, c->second);
		}
	}
	return eq;
}

template<StorageType storage_type>
EquationLightList ReduzerBase<storage_type>::unabbreviate_coefficients(
		const EquationXLightList& eqs) const {
	EquationLightList l;
	EquationXLightList::const_iterator it;
	for (it = eqs.begin(); it != eqs.end(); ++it)
		l.push_back(unabbreviate_coefficients(*it));
	return l;
}

template<StorageType storage_type>
EquationHLight ReduzerBase<storage_type>::unabbreviate(
		const EquationXLight& leq) const {
	EquationHLight eq;
	for (EquationXLight::const_iterator it = leq.begin(); it != leq.end(); ++it) {
		typename MapTraits<storage_type, INTIndex, string>::type::const_iterator
				iti = integ->find(it->first);
		typename MapTraits<storage_type, CoeffIndex, string>::type::const_iterator
				itc = coeff->find(it->second);
		if (iti == integ->end())
			ABORT("Can't resolve integral abbreviation " << it->first);
		if (itc == coeff->end())
			ABORT("Can't resolve coefficient abbreviation " << it->second);
		eq.insert_end(INT(iti->second), itc->second);
	}
	return eq;
}

template<StorageType storage_type>
EquationHLightList ReduzerBase<storage_type>::unabbreviate(
		const EquationXLightList& eqs) const {
	EquationHLightList l;
	EquationXLightList::const_iterator it;
	for (it = eqs.begin(); it != eqs.end(); ++it)
		l.push_back(unabbreviate(*it));
	return l;
}

// note: when masking substitution coefficients, we just _never_ delete them
//       from the database (will be needed until the end in many cases)
template<StorageType storage_type>
void ReduzerBase<storage_type>::erase_coefficient_abbreviations(
		const EquationXLight& eq) {
	EquationXLight::const_iterator it;
	for (it = eq.begin(); it != eq.end(); ++it) {
		// don't erase masked coefficients of substitution integrals, might be needed later on
		if (options_.mask_substitutions && substinteg->find(it->first)
				!= substinteg->end())
			continue;
		coeff->erase(it->second);
	}
}

template<StorageType storage_type>
void ReduzerBase<storage_type>::erase_coefficient_abbreviations(
		const EquationXLightList& eqs) {
	EquationXLightList::const_iterator it;
	for (it = eqs.begin(); it != eqs.end(); ++it)
		erase_coefficient_abbreviations(*it);
}

template<StorageType storage_type>
void ReduzerBase<storage_type>::insert_substitutions(const string& filename,
		const unord_map_INT& integ_abbr, const unord_set_INT& integrals) {
	LOG("  Inserting needed substitutions from " << filename << " to database");
	Timer timer;
	InFileEquations in(filename);
	size_t s = in.file_size();
	ProgressBar pbar(2, "scanning " + to_string(s) + " bytes", s);
	pbar.start();
	EquationHLight eq;
	typename MapTraits<storage_type, integral_type, eq_type>::type::iterator
			it = subst->begin();
	while (in.get(eq)) {
		pbar.print(in.tellg());
		if (integrals.find(eq.max_INT()) == integrals.end())
			continue; // load only substitutions actually relevant for identities
		EquationXLight leq = abbreviate(eq, integ_abbr);
		if (!leq.empty()) {
			it = subst->insert(it, make_pair(leq.max_INT(), eq_type()));
			if (it->second.empty() || leq < it->second) {
				it->second = leq; // choose "most reduced" equation
			}
		}
	}
	pbar.end();
	in.close();
	findnextbegin = 0;
	LOG("  file scanned " << timer.get_wall_time_nice_string());
}

template<StorageType storage_type>
void ReduzerBase<storage_type>::insert_equations(const string& filename,
		const unord_map_INT& integ_abbr) {
	LOG("  Inserting equations " << filename << " to database");
	Timer timer;
	InFileEquations in(filename);
	size_t s = in.file_size();
	ProgressBar pbar(2, "inserting " + to_string(s) + " bytes to database", s);
	pbar.start();
	EquationHLight eq;
	while (in.get(eq)) {
		pbar.print(in.tellg());
		EquationXLightList l;
		l.push_back(abbreviate(eq, integ_abbr));
		insert_equations(l);
	}
	pbar.end();
	in.close();
	LOG("  file loaded " << timer.get_wall_time_nice_string());
}

template<StorageType storage_type>
double ReduzerBase<storage_type>::progress() {
	begin_transaction();
	double prog = 1.;
	if (!blocks->empty()) {
		double num = 0.;
		if (maxint_processed >= blocks->begin()->first)
			num = maxint_processed - blocks->begin()->first + 1;
		double den = blocks->rbegin()->first - blocks->begin()->first + 1;
		prog = num / den;
	}
	commit_transaction();
	return prog;
}

template<StorageType storage_type>
void ReduzerBase<storage_type>::insert(SectorJobLightResult& res,
		ProgressBar* pbar) {
	// activate this for ReduzeMonitor:
	//print_status();
	VERIFY(integ); // Reduzer initialized
	begin_transaction();
	if (!res.maxints.empty() && *res.maxints.rbegin() > maxint_processed) {
		VERIFY(!blocks->empty()); // there should never be a Job if blocks empty
		maxint_processed = *res.maxints.rbegin();
		integral_type offset = blocks->begin()->first;
		integral_type done = (maxint_processed - offset + 1);
		integral_type todo = (blocks->rbegin()->first - offset + 1);
		LOGX ("Processed integral " << done << "/" << todo);
		if (pbar)
			pbar->print(done, todo);
	}
	EquationXLightList eqs;
	insert_equations(eqs = abbreviate(res.upper_eqs));
	insert_equations(eqs = abbreviate(res.lower_eqs));

	// clear equations in process, delete unneeded blocks
	list<integral_type>::iterator it;
	for (it = res.maxints.begin(); it != res.maxints.end(); ++it) {
		typename Blocks::iterator bit = blocks->lower_bound(*it);
		if (bit == blocks->end() || blocks->key_comp()(*it, bit->first))
			ABORT("block for leading integral does not exist");

		// direct delete via iterator doesn't work, need bit->second = ...
		Block b = bit->second;
		eqlist_type del;
		del.splice(b.eqs_inprocess);
		if (bit->second.eqs.empty())
			blocks->erase(bit);
		else
			bit->second = b;
		erase_coefficient_abbreviations(del);
	}
	commit_transaction();
}

template<StorageType storage_type>
void ReduzerBase<storage_type>::insert_equations(eqlist_type& eqs) {
	done = Unknown;
	LOGXX("Inserting " << eqs.size() << " equations");
	list<eqlist_type> eqs_maxint;
	eqs.split(eqs_maxint);
	list<eqlist_type>::iterator eit;
	for (eit = eqs_maxint.begin(); eit != eqs_maxint.end(); ++eit) {
		if (eit->empty() || eit->begin()->empty())
			continue;
		const integral_type maxint = eit->begin()->max_INT();
		//if (substinteg->find(maxint) != substinteg->end())
		//	continue; // discard reduction
		Block b = (*blocks)[maxint];
		eqlist_type & beqs = b.eqs;
		// smallest to front
		if (!beqs.empty() && *eit->begin() < *beqs.begin()) { // new to begin
			eit->splice(beqs);
			beqs.swap(*eit);
		} else { // new to end
			beqs.splice(*eit);
		}
		(*blocks)[maxint] = b;
		if (!criticals->empty()) {
			criticals->erase(maxint);
			if (criticals->empty())
				LOG("Last critical integral reduced");
		}
		if (maxint < findnextbegin)
			findnextbegin = maxint;
	}
}

template<class T>
bool has_masked_expression(const T&);

template<>
bool has_masked_expression<GiNaC::ex> (const GiNaC::ex& c) {
	return c.has(Coeff(GiNaC::wild()));
}

template<>
bool has_masked_expression<string> (const string& c) {
	return c.find("Coeff") != c.npos;
}

template<StorageType storage_type>
bool ReduzerBase<storage_type>::have_subst(const eqlist_type& eqs) const {
	set<integral_type, greater<integral_type> > iset, imaxset;
//	eqs.find_pure_subleading_INT(iset);
	eqs.find_INT(iset);
	eqs.find_max_INT(imaxset);
	set<integral_type>::const_iterator iit;
	for (iit = iset.begin(); iit != iset.end(); ++iit) {
		if (criticals->empty() && helpers->find(*iit) != helpers->end())
			continue;
		if (subst->find(*iit) != subst->end())
			return true;
		typename Blocks::const_iterator bit;
		if (imaxset.find(*iit) == imaxset.end() && //
				(bit = blocks->find(*iit)) != blocks->end()) {
			if (!bit->second.eqs.empty())
				return true;
			if (options_.back_substitute_inprocess
					&& !bit->second.eqs_inprocess.empty())
				return true;
		}
	}
	return false;
}

template<StorageType storage_type>
void ReduzerBase<storage_type>::find_subst(const eqlist_type& eqs,
		eqlist_type& substeqs, bool recursive) const {
	LOGXX("Searching substitutions");
	set<integral_type, greater<integral_type> > iset, imaxset;
	eqs.find_INT(iset);
	eqs.find_max_INT(imaxset);
	set<integral_type>::const_iterator iit;
	for (iit = iset.begin(); iit != iset.end(); ++iit) {
		if (criticals->empty() && helpers->find(*iit) != helpers->end())
			continue;
		const eq_type* s = 0;
		typename MapTraits<storage_type, integral_type, eq_type>::type::const_iterator
				substit;
		if ((substit = subst->find(*iit)) != subst->end())
			s = &substit->second;
		typename Blocks::const_iterator bit;
		if (imaxset.find(*iit) == imaxset.end() && //
				(bit = blocks->find(*iit)) != blocks->end()) {
			const eqlist_type* l;
			if (options_.back_substitute_inprocess) {
				l = &bit->second.eqs_inprocess;
				if (!l->empty() && (s == 0 || *l->begin() < *s))
					s = &*l->begin();
			}
			l = &bit->second.eqs;
			if (!l->empty() && (s == 0 || *l->begin() < *s))
				s = &*l->begin();
		}
		if (s != 0) {
			substeqs.push_back(*s);
			if (recursive)
				s->find_subleading_INT(iset); // will not invalidate iit
		}
	}
}

template<StorageType storage_type>
typename ReduzerBase<storage_type>::Blocks::iterator //
ReduzerBase<storage_type>::find_next_block(bool back_subst) {
	LOGXX("Searching for next block (" << back_subst << ")");
	done = False;
	typename Blocks::iterator it;
	for (it = blocks->lower_bound(findnextbegin); it != blocks->end(); ++it) {
		if (it->second.in_process()) {
			continue;
		} else if (criticals->empty() // don't touch helpers anymore
				&& helpers->find(it->first) != helpers->end()) {
			continue;
		} else if (!it->second.reduced()) {
			findnextbegin = it->first;
			return it;
		} else if (back_subst //
				&& (options_.back_substitute_helpers || //
						helpers->find(it->first) == helpers->end())//
				&& have_subst(it->second.eqs)) {
			findnextbegin = it->first;
			return it;
		}
	}
	if (!back_subst) {
		// no forward elimination found, try back substitution
		return find_next_block(true);
	} else {
		LOGXX("Found no block");
		bool anyinprocess = false;
		for (it = blocks->begin(); it != blocks->end(); ++it)
			if (it->second.in_process()) {
				anyinprocess = true;
				break;
			}
		done = (anyinprocess ? False : True);
		return blocks->end();
	}
}

template<StorageType storage_type>
void ReduzerBase<storage_type>::print_status() const {
	//	static ifstream ctrl("/home/andreas/ggtt/deleteme/control.dev");
	//	string dummy;
	//	getline(ctrl, dummy);

	int i = 0;
	int inproc = 0;
	std::stringstream ss;
	typename Blocks::const_iterator it;
	for (it = blocks->begin(); it != blocks->end(); ++it, ++i) {
		ss << it->second.eqs.size() << "/";
		ss << (it->second.reduced() ? "u" : "w");
		ss << (it->second.in_process() ? "*" : ".");
		if (it->second.in_process())
			++inproc;
		if ((i + 1) % 15 == 0)
			ss << endl;
		else
			ss << "\t";
	}
	LOGXX("Blocks: " << blocks->size() << endl << ss.str());
	LOGXX("Blocks in process: " << inproc);
}

template<StorageType storage_type>
void ReduzerBase<storage_type>::write_abbreviations(const string& filename) const {
	LOGX("Exporting abbreviations to " << filename);
	ofstream of(filename.c_str());
	if (!of)
		ABORT("Can't write to file " << filename);
	typename MapTraits<storage_type, INTIndex, string>::type::const_iterator i;
	of << "integrals:" << endl;
	for (i = integ->begin(); i != integ->end(); ++i)
		of << i->first << " => " << i->second << endl;
	of << "coefficients:" << endl;
	typename MapTraits<storage_type, CoeffIndex, string>::type::const_iterator c;
	for (c = coeff->begin(); c != coeff->end(); ++c)
		of << c->first << " => " << c->second << endl;
	of.close();
}

template<StorageType storage_type>
void ReduzerBase<storage_type>::write_substitutions(const string& filename) const {
	LOGX("Exporting substitutions to " << filename);
	OutFileEquations of(filename.c_str());
	typename MapTraits<storage_type, integral_type, eq_type>::type::const_iterator
			it;
	for (it = subst->begin(); it != subst->end(); ++it) {
		EquationHLight e = unabbreviate(it->second);
		of << e;
	}
	of.finalize();
}

template<StorageType storage_type>
void ReduzerBase<storage_type>::write_result(const string& filename,
		const string& filenameexternal) {
	LOG("Writing results to " << filename);
	begin_transaction(); // logically not needed (this method should be const)
	size_t num_blocks = 0;
	size_t int_min = 0;
	size_t int_max = 0;
	if (!blocks->empty()) {
		int_min = blocks->begin()->first;
		int_max = blocks->rbegin()->first;
		num_blocks = int_max - int_min + 1;
	}
	commit_transaction();
	ProgressBar pbar(2, "write result", num_blocks);
	pbar.start();
	Timer t_unabbr, t_write, t_tot;
	t_tot.unpause();
	OutFileEquations of(filename.c_str());
	OutFileEquations* ofexternal = 0;
	size_t pos_begin = of.tellp();
	typename Blocks::const_iterator it;
	for (it = blocks->begin(); it != blocks->end(); ++it) {
		if (options_.delete_helpers_in_result//
				&& helpers->find(it->first) != helpers->end())
			continue;
		EquationHLightList l1, l2;
		t_unabbr.unpause();
		l1 = unabbreviate(it->second.eqs);
		l2 = unabbreviate(it->second.eqs_inprocess);
		t_unabbr.pause();
		EquationHLightList l;
		l.splice(l1);
		l.splice(l2);
		l.sort();
		t_write.unpause();
		for (EquationHLightList::const_iterator e = l.begin(); e != l.end(); ++e) {
			if (external->find(it->first) != external->end()) {
				if (!ofexternal && !filenameexternal.empty())
					ofexternal = new OutFileEquations(filenameexternal.c_str());
				if (ofexternal)
					(*ofexternal) << *e;
			} else {
				of << *e;
			}
		}
		t_write.pause();
		pbar.print(it->first - int_min + 1);
	}
	size_t filesize = (size_t) (of.tellp()) - pos_begin;
	of.finalize();
	pbar.end();
	LOGX("  file size: " << filesize << " Bytes");
	LOGX("  time unabbreviate: " << fixed << setprecision(1)
			<< t_unabbr.get_wall_time() << resetiosflags(ios::fixed) << " s");
	LOGX("  time I/O:        " << fixed << setprecision(1)
			<< t_write.get_wall_time() << resetiosflags(ios::fixed) << " s");
	LOGX("  time write tot   " << fixed << setprecision(1)
			<< t_tot.get_wall_time() << resetiosflags(ios::fixed) << " s"
			<< " (" << (size_t)(filesize/t_tot.get_wall_time()) << " Bytes/s)");
	if (ofexternal) {
		ofexternal->finalize();
		LOG("\nSubsector sector identities written to " << filenameexternal);
		delete ofexternal;
	}
	LOGX("");
}

template<StorageType storage_type>
void ReduzerBase<storage_type>::run() {
	LOGX("Performing computation in serial mode");
	Timer timer;
	ProgressBar pbar(2, "reduction");
	pbar.start();
	while (SectorJobLight* job_light = this->create_next_job()) {
		timer.restart();
		SectorJob job_full(*job_light);
		delete job_light;
		SectorJobResult res_full = job_full.run();
		SectorJobLightResult res_light(res_full);
		this->insert(res_light, &pbar);
		LOGX("Reduced block [" << scientific << setprecision(0) <<
				timer.get_wall_time() << resetiosflags(ios::scientific) << " s]");
	}
	pbar.end();
	ASSERT(completed());
	LOGX("Reduction done !");
}

template<StorageType storage_type>
SectorJobLight* ReduzerBase<storage_type>::create_next_job() {
	LOGXX("Creating next job");
	if (!integ)
		ABORT("Reduzer not initialized");
	begin_transaction();
	bool back_early = options_.back_substitute_early;
	bool back_allow = true;
	typename Blocks::iterator it = find_next_block(back_early);
	if (it == blocks->end()) {
		abort_transaction();
		return 0; // nothing to do
	}
	SectorJobLight* job = new SectorJobLight();
	job->forward_eliminate = true;
	job->only_substitute = options_.only_substitute;
	EquationXLightList eqs;
	unsigned num_blocks = 0;
	while (true) {
		++num_blocks;
		Block b = it->second;
		b.eqs_inprocess.splice(b.eqs); // move
		EquationXLightList neweqs(b.eqs_inprocess); // copy
		eqs.splice(neweqs);
		job->maxints.push_back(it->first);
		it->second = b;
		if (!options_.back_substitute_helpers && //
				helpers->find(it->first) != helpers->end())
			back_allow = false;
		it = find_next_block(back_early);
		if (it == blocks->end() || //
				(eqs.size() + it->second.eqs.size()
						> options_.num_equations_per_subjob && num_blocks
						>= options_.num_blocks_per_subjob))
			break;
	};
	EquationXLightList substeqs;
	if (back_allow)
		find_subst(eqs, substeqs, options_.back_substitute_recursively);
	job->upper_eqs = unabbreviate_coefficients(eqs);
	job->subst_eqs = unabbreviate_coefficients(substeqs);
	commit_transaction();
	return job;
}

//-----------------------------------------------------------------------------
// SectorJob
//-----------------------------------------------------------------------------

SecJob<EquationList>::SecJob() :
	forward_eliminate(true), only_substitute(false) {
}

SecJob<EquationList>::SecJob(const SectorJobLight& l) :
	maxints(l.maxints), upper_eqs(l.upper_eqs), subst_eqs(l.subst_eqs),
			forward_eliminate(l.forward_eliminate), only_substitute(
					l.only_substitute) {
}

SectorJobResult SectorJob::run() {
	SectorJobResult result;
	result.maxints.swap(maxints);
	result.upper_eqs.splice(upper_eqs);
	result.upper_eqs.replace(subst_eqs, false);
	if (!only_substitute)
		result.upper_eqs.solve(); // better to do after all substitutions inserted
	if (!only_substitute && forward_eliminate)
		result.upper_eqs.forward_eliminate_leading(result.lower_eqs);
	return result;
}

//-----------------------------------------------------------------------------
// SecJobResult
//-----------------------------------------------------------------------------

SecJobResult<EquationLightList>::SecJobResult() {
}

SecJobResult<EquationLightList>::SecJobResult(const SectorJobResult& full) :
	maxints(full.maxints), upper_eqs(full.upper_eqs), lower_eqs(full.lower_eqs) {
}

//-----------------------------------------------------------------------------
// Reduzer
//-----------------------------------------------------------------------------

void ReduzerMem::setup() {
	clear();
	integ = new map<INTIndex, string> ;
	coeff = new map<CoeffIndex, string> ;
	subst = new map<INTIndex, EquationXLight> ;
	blocks = new map<INTIndex, Block> ;
#ifdef USE_HASH_TABLE
	criticals = new tr1::unordered_set<INTIndex>;
	helpers = new tr1::unordered_set<INTIndex>;
	external = new tr1::unordered_set<INTIndex>;
	substinteg = new tr1::unordered_set<INTIndex>;
#else
	criticals = new set<INTIndex> ;
	helpers = new set<INTIndex> ;
	external = new set<INTIndex> ;
	substinteg = new set<INTIndex> ;
#endif
	LOGX("Allocated maps for data during reduction.");
}

void ReduzerMem::clear() {
	ReduzerBase<MEMORY>::clear();
	delete integ;
	delete coeff;
	delete subst;
	delete blocks;
	delete criticals;
	delete helpers;
	delete external;
	delete substinteg;
	integ = 0;
	coeff = 0;
	subst = 0;
	blocks = 0;
	criticals = 0;
	helpers = 0;
	external = 0;
	substinteg = 0;
}

//-----------------------------------------------------------------------------
// ReduzerDb
//-----------------------------------------------------------------------------

#ifdef HAVE_DATABASE

// a comparison function for keys in database maps
// implementation just for signed and unsigned integer types so far
template<class T>
struct CompareDB {
	int operator()(DB* dp, const DBT* a, const DBT* b) const {
		T ai = *(T*) a->data;
		T bi = *(T*) b->data;
	    // (returning (ai-bi) wouldn't be clean for unsigned cases)
		if (ai < bi)
			return -1;
		else if (ai > bi)
			return 1;
		else
			return 0;
	}
};

extern "C" {

#if DB_VERSION_MAJOR >= 6
	int compare_db_INTIndex(DB* dp, const DBT* a, const DBT* b, size_t* locp) {
		locp = NULL;
		return CompareDB<INTIndex>()(dp, a, b);
	}
	int compare_db_CoeffIndex(DB* dp, const DBT* a, const DBT* b,
			size_t* locp) {
		locp = NULL;
		return CompareDB<CoeffIndex>()(dp, a, b);
	}
#else
	int compare_db_INTIndex(DB* dp, const DBT* a, const DBT* b) {
		return CompareDB<INTIndex>()(dp, a, b);
	}
	int compare_db_CoeffIndex(DB* dp, const DBT* a, const DBT* b) {
		return CompareDB<CoeffIndex>()(dp, a, b);
	}
#endif

    void copy_EquationXLight(void* dest, const EquationXLight& eq) {
		char* d = (char*) dest;
		EquationXLight::const_iterator eit;
		for (eit = eq.begin(); eit != eq.end(); ++eit) {
			*(INTIndex*) d = eit->first;
			d += sizeof(INTIndex);
			*(CoeffIndex*) d = eit->second;
			d += sizeof(CoeffIndex);
		}
		*(INTIndex*) d = 0;
	}

	u_int32_t size_EquationXLight(const EquationXLight& eq) {
		return eq.size() * (sizeof(INTIndex) + sizeof(CoeffIndex)) + sizeof(INTIndex);
	}

	void restore_EquationXLight(EquationXLight& eq, const void* srcdata) {
		const char* s = (const char*) srcdata;
		INTIndex integ;
		CoeffIndex coeff;
		while (*(INTIndex*) s != 0) {
			integ = *(INTIndex*) s;
			s += sizeof(INTIndex);
			coeff = *(CoeffIndex*) s;
			s += sizeof(CoeffIndex);
			eq.insert(integ, coeff);
		}
	}

	void copy_Block(void* dest, const ReduzerBase<DATABASE>::Block& b) {
		char* d = (char*) dest;
		EquationXLightList::const_iterator lit;
		for (lit = b.eqs.begin(); lit != b.eqs.end(); ++lit) {
			copy_EquationXLight((void*) d, *lit);
			d += size_EquationXLight(*lit);
		}
		*(INTIndex*) d = 0;
		d += sizeof(INTIndex);
		for (lit = b.eqs_inprocess.begin(); lit != b.eqs_inprocess.end(); ++lit) {
			copy_EquationXLight((void*) d, *lit);
			d += size_EquationXLight(*lit);
		}
		*(INTIndex*) d = 0;
		d += sizeof(INTIndex);
	}

	u_int32_t size_Block(const ReduzerBase<DATABASE>::Block& b) {
		u_int32_t size = 0;
		EquationXLightList::const_iterator lit;
		for (lit = b.eqs.begin(); lit != b.eqs.end(); ++lit)
			size += size_EquationXLight(*lit);
		size += sizeof(INTIndex);
		for (lit = b.eqs_inprocess.begin(); lit != b.eqs_inprocess.end(); ++lit)
			size += size_EquationXLight(*lit);
		size += sizeof(INTIndex);
		return size;
	}

	void restore_Block(ReduzerBase<DATABASE>::Block& b, const void* srcdata) {
		const char* s = (const char*) srcdata;
		while (*(INTIndex*) s != 0) {
			EquationXLight eq;
			restore_EquationXLight(eq, (const void*) s);
			b.eqs.push_back(eq);
			s += size_EquationXLight(eq);
		}
		s += sizeof(INTIndex);
		while (*(INTIndex*) s != 0) {
			EquationXLight eq;
			restore_EquationXLight(eq, (const void*) s);
			b.eqs_inprocess.push_back(eq);
			s += size_EquationXLight(eq);
		}
		s += sizeof(INTIndex);
	}

} // extern "C"

ReduzerDb::ReduzerDb(const ReduzerOptions& opts) :
ReduzerBase<DATABASE> (opts),//
db_integ(0), db_coeff(0), db_subst(0), db_blocks(0), db_crit(0),
db_help(0), db_external(0), db_substinteg(0), env(0) {
	clear();
}

void ReduzerDb::setup(const string& db_home, const string& db_filename) {
	clear();
	LOGX("Building database " << db_filename << " from scratch");
	init_env(db_home);

	begin_transaction();
	init_db(db_filename);
	commit_transaction();
}

void ReduzerDb::close_db() {
	delete integ;
	delete coeff;
	delete subst;
	delete blocks;
	delete criticals;
	delete helpers;
	delete external;
	delete substinteg;
	integ = 0;
	coeff = 0;
	subst = 0;
	blocks = 0;
	criticals = 0;
	helpers = 0;
	external = 0;
	substinteg = 0;
	if (db_integ) /* all databases are deleted together */{
		dbstl::close_db(db_integ);
		dbstl::close_db(db_coeff);
		dbstl::close_db(db_subst);
		dbstl::close_db(db_blocks);
		dbstl::close_db(db_crit);
		dbstl::close_db(db_help);
		dbstl::close_db(db_external);
		dbstl::close_db(db_substinteg);
	}
	delete db_integ;
	delete db_coeff;
	delete db_subst;
	delete db_blocks;
	delete db_crit;
	delete db_help;
	delete db_external;
	delete db_substinteg;
	db_integ = 0;
	db_coeff = 0;
	db_subst = 0;
	db_blocks = 0;
	db_crit = 0;
	db_help = 0;
	db_external = 0;
	db_substinteg = 0;
}

void ReduzerDb::close_env() {
	if (env != 0) {
		close_db_env(env);
		delete env;
		env = 0;
	}
}

void ReduzerDb::dbrename(const std::string& old_name,
		const std::string& new_name) {
	if (env == 0) {
		ABORT("Database environment not initialized");
	}
	begin_transaction();
	close_db();
	commit_transaction();
	int i = env->dbrename(0, old_name.c_str(), 0, new_name.c_str(), 0);
	if (i != 0) {
		ABORT("failed to rename database file: " << old_name << " to " + new_name);
	}
	begin_transaction();
	open_db(new_name);
	commit_transaction();
}

void ReduzerDb::clear() {
	ReduzerBase<DATABASE>::clear();
	begin_transaction();
	close_db();
	commit_transaction();
	close_env();
}

void ReduzerDb::sync() {
	if (env != 0) {
		db_integ->sync(0);
		db_coeff->sync(0);
		db_subst->sync(0);
		db_blocks->sync(0);
		db_crit->sync(0);
		db_help->sync(0);
		db_external->sync(0);
	}
}

void ReduzerDb::begin_transaction() {
	if (options_.use_transactions) {
		begin_txn(0, env);
	}
}

void ReduzerDb::commit_transaction() {
	if (options_.use_transactions) {
		commit_txn(env);
	}
}

void ReduzerDb::abort_transaction() {
	if (options_.use_transactions) {
		abort_txn(env);
	}
}

void ReduzerDb::remove_files() {
	if (!env) {
		return;
	}
	LOGX("Removing database directory " << dbhome);
	clear();
	remove_directory_with_files(dbhome);
}

void ReduzerDb::init_env(const string& home) {
	LOGX("Registering database environment in " << home);
	if (!options_.use_transactions) {
		LOG("Database transactions are turned off (no recovery if aborted)");
	}
	dbhome = home;
	env = new DbEnv(DB_CXX_NO_EXCEPTIONS);
	u_int32_t oflags = DB_CREATE | DB_INIT_MPOOL;
	if (options_.use_transactions) {
		oflags |= DB_INIT_LOG | DB_RECOVER | DB_INIT_TXN;
	}
	if (options_.cache_size >= 0) {
		env->set_cachesize(0, options_.cache_size, 0);
	}
	env->open(home.c_str(), oflags, 0);
	if (options_.use_transactions) {
		env->set_flags(DB_TXN_NOSYNC, 1);
		env->set_flags(DB_AUTO_COMMIT, 1);
		env->log_set_config(DB_LOG_AUTO_REMOVE, 1);
	}
	register_db_env(env);
}

void ReduzerDb::init_db(const string& filename) {
	// We don't use dbstl::open_db() etc since we found no way to specify
	// custom key comparison in this way. Instead we need to manually
	// new Db on the heap with DB_CXX_NO_EXCEPTIONS and register with dbstl.
	// Since we share a file across several databases, we need to setup
	// an environment explicitely.
	u_int32_t dbflags = DB_CREATE;
	if (options_.use_transactions) {
		dbflags |= DB_READ_UNCOMMITTED /* | DB_AUTO_COMMIT */;
	}
	DBTYPE type = DB_BTREE;
	//DBTYPE type = DB_HASH;
	// \todo activate hash
	DbstlElemTraits<EquationXLight>* etraits =
	DbstlElemTraits<EquationXLight>::instance();
	etraits->set_copy_function(&copy_EquationXLight);
	etraits->set_size_function(&size_EquationXLight);
	etraits->set_restore_function(&restore_EquationXLight);
	DbstlElemTraits<Block>* btraits = DbstlElemTraits<Block>::instance();
	btraits->set_copy_function(&copy_Block);
	btraits->set_size_function(&size_Block);
	btraits->set_restore_function(&restore_Block);
	db_integ = new Db(env, DB_CXX_NO_EXCEPTIONS);
	db_coeff = new Db(env, DB_CXX_NO_EXCEPTIONS);
	db_subst = new Db(env, DB_CXX_NO_EXCEPTIONS);
	db_blocks = new Db(env, DB_CXX_NO_EXCEPTIONS);
	db_crit = new Db(env, DB_CXX_NO_EXCEPTIONS);
	db_help = new Db(env, DB_CXX_NO_EXCEPTIONS);
	db_external = new Db(env, DB_CXX_NO_EXCEPTIONS);
	db_substinteg = new Db(env, DB_CXX_NO_EXCEPTIONS);
	/// \todo: don't assume integral and coefficient indices to be int
	if (type == DB_BTREE) {
		db_integ->set_bt_compare(compare_db_INTIndex);
		db_coeff->set_bt_compare(compare_db_CoeffIndex);
		db_subst->set_bt_compare(compare_db_INTIndex);
		db_blocks->set_bt_compare(compare_db_INTIndex);
		db_crit->set_bt_compare(compare_db_INTIndex);
		db_help->set_bt_compare(compare_db_INTIndex);
		db_external->set_bt_compare(compare_db_INTIndex);
		db_substinteg->set_bt_compare(compare_db_INTIndex);
		if (options_.compress_coefficients)
		db_coeff->set_bt_compress(0, 0);
	}
	db_integ->open(0, filename.c_str(), "integrals", type, dbflags, 0);
	db_coeff->open(0, filename.c_str(), "coefficients", type, dbflags, 0);
	db_subst->open(0, filename.c_str(), "substitutions", type, dbflags, 0);
	db_blocks->open(0, filename.c_str(), "blocks", type, dbflags, 0);
	db_crit->open(0, filename.c_str(), "criticals", type, dbflags, 0);
	db_help->open(0, filename.c_str(), "helpers", type, dbflags, 0);
	db_external->open(0, filename.c_str(), "external", type, dbflags, 0);
	db_substinteg->open(0, filename.c_str(), "substinteg", type, dbflags, 0);
	register_db(db_integ);
	register_db(db_coeff);
	register_db(db_subst);
	register_db(db_blocks);
	register_db(db_crit);
	register_db(db_help);
	register_db(db_external);
	register_db(db_substinteg);
	integ = new db_map<INTIndex, string> (db_integ, env);
	coeff = new db_map<CoeffIndex, string> (db_coeff, env);
	subst = new db_map<INTIndex, EquationXLight> (db_subst, env);
	blocks = new db_map<INTIndex, Block> (db_blocks, env);
	criticals = new db_set<INTIndex, ElementHolder<INTIndex> > (db_crit, env);
	helpers = new db_set<INTIndex, ElementHolder<INTIndex> > (db_help, env);
	external = new db_set<INTIndex, ElementHolder<INTIndex> > (db_external, env);
	substinteg = new db_set<INTIndex, ElementHolder<INTIndex> > (db_substinteg, env);
}

void ReduzerDb::open_env(const string& db_home) {
	LOGX("Opening existing database environment");
	init_env(db_home);
}

void ReduzerDb::open_db(const string& db_filename) {
	LOGX("Opening existing database");
	init_db(db_filename);
	LOGX("Resetting blocks in process");
	for (Blocks::iterator it = blocks->begin(); it != blocks->end(); ++it) {
		if (it->second.in_process()) {
			Block b = it->second;
			b.eqs.splice(b.eqs_inprocess);
			it->second = b;
		}
	}
}

void ReduzerDb::open(const string& db_home, const string& db_filename) {
	LOGX("Opening existing database " << db_filename);
	clear();
	open_env(db_home);
	begin_transaction();
	open_db(db_filename);
	commit_transaction();
	LOGX("Database opened");
}
#endif // HAVE_DATABASE

// explicit instantiations
template class ReduzerBase<MEMORY> ;
#ifdef HAVE_DATABASE
template class ReduzerBase<DATABASE>;
#endif // HAVE_DATABASE

} // namespace Reduze
