/*  undirectedgraph.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "undirectedgraph.h"
#include "edge.h"
#include "functions.h"
#include "yamlutils.h"

using namespace std;
using namespace GiNaC;

namespace Reduze {

// UndirectedGraph
UndirectedGraph::UndirectedGraph() :
		num_edge_colors_(0) {
}

UndirectedGraph::~UndirectedGraph() {
}

std::map<int, int> combine_colors(const std::map<int, int>& map1, int num_c1,
		const std::map<int, int>& map2, int num_c2) {
	ASSERT(map1.size() == map2.size());
	set<int> cols1 = get_map_values(map1), cols2 = get_map_values(map2);
	ASSERT(cols1.empty() || (*cols1.begin() >= 0 && *cols1.rbegin() < num_c1));
	ASSERT(cols2.empty() || (*cols2.begin() >= 0 && *cols2.rbegin() < num_c2));
	map<int, int>::const_iterator m1 = map1.begin(), m2 = map2.begin();
	map<int, int> combined;
	for (; m1 != map1.end() && m2 != map2.end(); ++m1, ++m2) {
		ASSERT(m1->first == m2->first);
		combined[m1->first] = m1->second * cols2.size() + m2->second;
	}
	return combined;
}

UndirectedGraph::UndirectedGraph(const std::set<int>& nodes,
		const std::map<int, Edge>& edges,
		const std::list<std::map<int, int> >& edge_coloring) :
		original_nodes_(vector<int>(nodes.begin(), nodes.end())), //
		num_edge_colors_(1) {

	// convert colors to consecutive integers >= 0
	vector<map<int, int> > base_0_edge_coloring;
	base_0_edge_coloring.reserve(edge_coloring.size());
	list<map<int, int> >::const_iterator m;
	for (m = edge_coloring.begin(); m != edge_coloring.end(); ++m) {
		ASSERT(m->size() == edges.size());
		map<int, int> e_to_col = map_values_to_base_0(*m);
		base_0_edge_coloring.push_back(e_to_col);
	}

	// combine the different colorings
	map<int, int> edge_coloring_combined;
	map<int, Edge>::const_iterator e;
	for (e = edges.begin(); e != edges.end(); ++e)
		edge_coloring_combined[e->first] = 0;
	num_edge_colors_ = 1;
	for (unsigned i = 0; i < base_0_edge_coloring.size(); ++i) {
		int num_c2 = get_map_values(base_0_edge_coloring[i]).size();
		edge_coloring_combined = combine_colors(edge_coloring_combined,
				num_edge_colors_, base_0_edge_coloring[i], num_c2);
		num_edge_colors_ *= num_c2;
	}

	// construct the adjacency matrix
	try {
		// renumber the nodes
		map<int, int> node_to_number;
		for (unsigned int i = 0; i < original_nodes_.size(); ++i)
			node_to_number[original_nodes_[i]] = i;

		adjm_ = vector<vector<colored_multi_edge> >(nodes.size(),
				vector<colored_multi_edge>(nodes.size(),
						colored_multi_edge(num_edge_colors_)));
		for (e = edges.begin(); e != edges.end(); ++e) {
			int i = node_to_number.at(e->second.from);
			int j = node_to_number.at(e->second.to);
			int col = edge_coloring_combined.at(e->first);
			adjm_[i][j].increment_at(col);
			if (i != j)
				adjm_[j][i].increment_at(col);
		}
	} catch (exception& exc) {
		throw runtime_error(
				"Error constructing adjacency matrix: "
						+ to_string(exc.what()));
	}
}

// public member functions

namespace {
// convert the node coloring, make it non-empty
std::vector<int> convert_node_color(const std::map<int, int>& node_coloring,
		int length) {
	vector<int> nc(length, 0);
	map<int, int>::const_iterator m;
	int lauf = 0;
	for (m = node_coloring.begin(); m != node_coloring.end(); ++m)
		nc[lauf++] = m->second;
	return nc;
}
}

std::pair<std::vector<std::map<int, colored_multi_edge> >, std::vector<int> > UndirectedGraph::find_canonical_label(
		const std::map<int, int>& node_coloring) const {

	ASSERT(node_coloring.empty() || node_coloring.size() == adjm_.size());

	// convert the node coloring, make it non-empty
	vector<int> nc = convert_node_color(node_coloring, adjm_.size());

	// find the label
	pair<vector<map<int, colored_multi_edge> >, vector<int> > lp;
	lp = find_canonical_label(nc);
	vector<map<int, colored_multi_edge> >& label = lp.first;
	vector<int>& relabel = lp.second;

	// convert the relabel vector i -> v_i to original nodes i -> O(v_i)
	for (unsigned int i = 0; i < adjm_.size(); ++i)
		relabel[i] = original_nodes_[relabel[i]];

	return make_pair(label, relabel);
}

// helper functions

namespace {
// append an element to a list without copying
template<class Element>
void move_to_back(std::list<Element>& List, Element& e) {
	List.push_back(Element());
	List.back().swap(e);
}

// prepend an element to a list without copying
template<class Element>
void move_to_front(std::list<Element>& List, Element& e) {
	List.push_front(Element());
	List.front().swap(e);
}

// move last element from a list without copying
template<class Element>
void move_from_back(std::list<Element>& List, Element& e) {
	ASSERT(!List.empty());
	List.back().swap(e);
	List.pop_back();
}

// move first element from a list without copying
template<class Element>
void move_from_front(std::list<Element>& List, Element& e) {
	ASSERT(!List.empty());
	List.front().swap(e);
	List.pop_front();
}

void find_node_permutation(std::vector<int>& perm, const std::vector<int>& from,
		const std::vector<int>& to) {
	VERIFY(from.size() == to.size());
	vector<int> res(from.size(), 0);
	for (size_t n = 0; n < from.size(); ++n)
		res[from[n]] = to[n];
	res.swap(perm);
}

// flatten discrete partition into a vector
std::vector<int> flatten_singletons(const std::list<std::set<int> >& part) {
	vector<int> res;
	res.reserve(part.size());
	list<set<int> >::const_iterator l;
	for (l = part.begin(); l != part.end(); ++l) {
		ASSERT(l->size() == 1);
		res.push_back(*l->begin());
	}
	return res;
}

#if 0
void print_partition(std::ostream& stream, const std::list<set<int> >& part) {
	stream << "(";
	for (list<set<int> >::const_iterator l = part.begin(); l != part.end();
			++l) {
		stream << "{";
		for (set<int>::const_iterator s = l->begin(); s != l->end(); ++s)
			stream << *s << " ";
		stream << "}";
	}
	stream << ")" << endl;
}

void print_label(std::ostream& stream,
		const std::vector<multiset<int> >& label) {
	multiset<int>::const_iterator n;
	stream << "canonical label:\n";
	for (unsigned int i = 0; i < label.size(); ++i) {
		stream << i << ": ";
		for (n = label[i].begin(); n != label[i].end(); ++n) {
			stream << *n << " ";
		}
		stream << endl;
	}
}
#endif
    
} // namespace

void UndirectedGraph::find_node_symmetry_group(
		std::list<std::map<int, int> >& symmetries,
		const std::map<int, int>& node_coloring,
		const std::set<int>& no_free_permutation) const {

	ASSERT(node_coloring.empty() || node_coloring.size() == adjm_.size());

	// convert the node coloring, make it non-empty
	vector<int> nc = convert_node_color(node_coloring, adjm_.size());

	// convert the no_free_permutation
	set<int> nfp;
	for (unsigned int i = 0; i < adjm_.size(); ++i)
		if (no_free_permutation.find(original_nodes_[i])
				!= no_free_permutation.end())
			nfp.insert(i);

	// find the symmetries
	list<vector<int> > permutations_vec;
	find_node_symmetry_group(permutations_vec, nc, nfp);

	// convert the permutations i -> p(i) to original nodes O(i) -> O(p(i))
	const vector<int>& orig = original_nodes_;
	list<map<int, int> > permutations_map;
	while (!permutations_vec.empty()) {
		vector<int> pvec;
		move_from_back(permutations_vec, pvec);
		map<int, int> pmap;
		map<int, int>::iterator pos = pmap.begin();
		for (unsigned int i = 0; i < adjm_.size(); ++i)
			pos = pmap.insert(pos, pair<int, int>(orig[i], orig[pvec[i]]));
		move_to_front(permutations_map, pmap);
	}
	permutations_map.swap(symmetries);
}

// private member functions

UndirectedGraph::NodeSignature UndirectedGraph::node_signature(int node,
		const std::set<int>& W) const {
	NodeSignature sig(num_edge_colors_);
	for (set<int>::const_iterator v = W.begin(); v != W.end(); ++v) {
		colored_multi_edge add = adjm_[*v][node];
		if (add.is_zero())
			continue;
		sig.degree_ += add;
		if (*v == node)
			sig.loops_ += add, sig.degree_ += add;
		else
			sig.multiplicities_.insert(add);
	}
	return sig;
}

std::pair<std::vector<std::map<int, colored_multi_edge> >, std::vector<int> > UndirectedGraph::find_canonical_label(
		const std::vector<int>& node_coloring) const {

	// all allowed discrete partitions
	list<vector<int> > allowed;
	set<int> dummy_nfp;
	set_allowed_node_permutations(allowed, node_coloring, dummy_nfp);
	if (allowed.empty())
		return make_pair(vector<map<int, colored_multi_edge> >(), vector<int>());

	// find minimal adjacency list
	list<vector<int> >::const_iterator t, tmin;
	for (t = allowed.begin(); t != allowed.end(); ++t) {
		ASSERT(t->size() == adjm_.size());
		if (t == allowed.begin() || compare_adjacency_list(*t, *tmin) < 0)
			tmin = t;
	}

	vector<map<int, colored_multi_edge> > label;
	set_adjacency_list(label, *tmin);

	return make_pair(label, *tmin);
}

void UndirectedGraph::find_node_symmetry_group(
		std::list<std::vector<int> >& symmetries,
		const std::vector<int>& node_coloring,
		const std::set<int>& no_free_permutation) const {

	symmetries.clear();

	// all allowed discrete partitions
	list<vector<int> > allowed;
	set_allowed_node_permutations(allowed, node_coloring, no_free_permutation);
	if (allowed.empty())
		return;

	list<vector<int> >::const_iterator t, tmin = allowed.begin();
	for (t = allowed.begin(); t != allowed.end(); ++t) {
		ASSERT(t->size() == adjm_.size());
		int c = compare_adjacency_list(*t, *tmin);
		if (c <= 0) {
			if (c < 0) {
				tmin = t;
				symmetries.clear();
			}
			vector<int> perm; // permutation *t -> rref
			find_node_permutation(perm, *t, *tmin);
			move_to_back(symmetries, perm);
		}
	}
	VERIFY(!symmetries.empty());
	VERIFY(allowed.size() % symmetries.size() == 0);
}

namespace {
list<set<int> >::const_iterator get_next_cell(const list<set<int> >& partition,
		const set<int>& not_preferred) {

	const set<int>& npf = not_preferred;
	list<set<int> >::const_iterator c1, c2;
	for (c1 = partition.begin(); c1 != partition.end(); ++c1)
		if (c1->size() > 1) {
			for (c2 = c1; c2 != partition.end(); ++c2)
				if (c2->size() > 1 && npf.find(*c2->begin()) == npf.end())
					break;
			c1 = ((c2 != partition.end()) ? c2 : c1);
			break;
		}
	return c1;
}

void verify_nfp_cells(const list<set<int> >& partition, const set<int>& nfp) {
	list<set<int> >::const_iterator c;
	for (c = partition.begin(); c != partition.end(); ++c) {
		VERIFY(!c->empty());
		set<int>::const_iterator n;
		if (nfp.find(*c->begin()) != nfp.end()) {
			for (n = c->begin(); n != c->end(); ++n)
				VERIFY(nfp.find(*n) != nfp.end());
		} else {
			for (n = c->begin(); n != c->end(); ++n)
				VERIFY(nfp.find(*n) == nfp.end());
		}
	}
}

vector<int> get_nfp_cell_sizes(const list<set<int> >& partition,
		const set<int>& nfp) {
	verify_nfp_cells(partition, nfp);
	vector<int> result;
	list<set<int> >::const_iterator c;
	for (c = partition.begin(); c != partition.end(); ++c) {
		ASSERT(!c->empty());
		if (nfp.find(*c->begin()) != nfp.end())
			result.push_back(c->size());
	}
	return result;
}
}

void UndirectedGraph::set_allowed_node_permutations(
		std::list<std::vector<int> >& allowed_perms,
		const std::vector<int>& ncs, const std::set<int>& nfp) const {
	allowed_perms.clear();

	// the initial partition of the nodes
	list<set<int> > partition;
	set_colored_node_partition(partition, ncs, nfp);

	// list of partitions to be refined, used as a stack
	list<list<set<int> > > partition_list;
	list<pair<vector<int>, bool> > nfp_cell_sizes_changed;

	// get the coarsest partition which is finer than partition
	list<set<int> > root_tmp = partition; // copy
	if (refine_partition(root_tmp, partition))
		allowed_perms.push_back(flatten_singletons(root_tmp));
	else {
		move_to_back(partition_list, root_tmp);
		vector<int> cc = get_nfp_cell_sizes(partition_list.back(), nfp);
		nfp_cell_sizes_changed.push_back(make_pair(cc, false));
	}

	// move nodes before their cell and refine to get all allowed partitions
	while (!partition_list.empty()) {
		list<set<int> > partition;
		move_from_back(partition_list, partition);
		pair<vector<int>, bool> nfp_cs_c = nfp_cell_sizes_changed.back();
		nfp_cell_sizes_changed.pop_back();

		// find the first non-singleton cell of the partition
		// first the ones not containing nodes from no_free_permutation
		list<set<int> >::const_iterator cell;
		cell = get_next_cell(partition, nfp);
		VERIFY(cell != partition.end());
		set<int>::const_iterator node = cell->begin();

		// check whether the cell has nfp-nodes (eg. external nodes) which we don't want to permute alone
		const bool prune = !nfp_cs_c.second && nfp.find(*node) != nfp.end();

		// create child partition for every node in the non-singleton cell
		for (node = cell->begin(); node != cell->end(); ++node) {
			// construct the singleton cell
			list<set<int> > singleton(1, set<int>());
			singleton.back().insert(*node);

			// construct a child partition
			list<set<int> > child;
			list<set<int> >::const_iterator cp;
			for (cp = partition.begin(); cp != partition.end(); ++cp) {
				if (cp == cell) {
					child.push_back(singleton.back());
					child.push_back(*cp);
					child.back().erase(*node);
				} else {
					child.push_back(*cp);
				}
			}
			// refine the child partition, if it is discrete it is an allowed permutation
			if (refine_partition(child, singleton)) {
				allowed_perms.push_back(flatten_singletons(child));
			} else {
				move_to_back(partition_list, child);
				vector<int> cc = get_nfp_cell_sizes(partition_list.back(), nfp);
				bool b = (nfp_cs_c.second || (!prune && cc != nfp_cs_c.first));
				nfp_cell_sizes_changed.push_back(make_pair(cc, b));
			}
			if (prune)
				break;
		} // for
	} // while
}

std::list<std::set<int> >::iterator UndirectedGraph::set_ordered_partition(
		std::list<std::set<int> >& partition, const std::set<int>& nodes,
		const std::set<int>& test_set) const {

	partition.clear();
	list<set<int> >::iterator itmax = partition.end();

	// split the nodes in the cell into new cells according
	// to the NodeSignature wrt. the test cell
	map<NodeSignature, set<int> > signatures;
	set<int>::const_iterator n;
	for (n = nodes.begin(); n != nodes.end(); ++n)
		signatures[node_signature(*n, test_set)].insert(*n);

	unsigned int max_size = 0;
	map<NodeSignature, set<int> >::iterator sig;
	for (sig = signatures.begin(); sig != signatures.end(); ++sig) {
		move_to_back(partition, sig->second);
		if (max_size < partition.back().size()) {
			max_size = partition.back().size();
			itmax = --partition.end();
		}
	}
	return itmax;
}

//  REFINEMENT PROCEDURE
//
//  Data: π is the input colouring and α is a sequence of some cells of π
//    Result: the final value of π is the output colouring
//    while α is not empty and π is not discrete do
//       Remove some element W from α.
//       for each cell X of π do
//          Let X1 , . . . , Xk be the fragments of X distinguished according
//             to the number of edges from each vertex to W .
//          Replace X by X1 , . . . , Xk in π.
//          if X ∈ α then
//              Replace X by X1 , . . . , Xk in α.
//          else
//              Add all but one of the largest of X1 , . . . , Xk to α.
//          end
//       end
//    end
//
//
//
//  @article{DBLP:journals/corr/abs-1301-1493,
//    author    = {Brendan D. McKay and
//                 Adolfo Piperno},
//    title     = {Practical graph isomorphism, {II}},
//    journal   = {CoRR},
//    volume    = {abs/1301.1493},
//    year      = {2013},
//    url       = {http://arxiv.org/abs/1301.1493},
//    timestamp = {Fri, 01 Feb 2013 13:30:47 +0100},
//    biburl    = {http://dblp.uni-trier.de/rec/bib/journals/corr/abs-1301-1493},
//    bibsource = {dblp computer science bibliography, http://dblp.org}
//  }

bool UndirectedGraph::refine_partition(std::list<std::set<int> >& partition,
		const std::list<std::set<int> >& sequence) const {

	if (partition.size() == adjm_.size())
		return true; // already discrete partition

	list<set<int> > test_partition(sequence.begin(), sequence.end()); // copy
	while (!test_partition.empty() && partition.size() != adjm_.size()) {
		set<int> test_cell; // W
		move_from_front(test_partition, test_cell);

		// loop over the cells of the partition
		list<set<int> >::iterator cell;
		for (cell = partition.begin(); cell != partition.end(); ++cell) {
			ASSERT(!cell->empty());

			list<set<int> > new_cells;
			list<set<int> >::iterator itmax, pos, l;
			itmax = set_ordered_partition(new_cells, *cell, test_cell);
			if (new_cells.size() == 1)
				continue;

			// update the test partition
			pos = find(test_partition.begin(), test_partition.end(), *cell);
			if (pos != test_partition.end()) { // replace the cell by the new cells
				pos = test_partition.erase(pos);
				test_partition.insert(pos, new_cells.begin(), new_cells.end());
			} else { // push new cells up to one max cell
				for (l = new_cells.begin(); l != new_cells.end(); ++l)
					if (l != itmax)
						test_partition.push_back(*l);
			}

			// replace the cell of the partition by the new cells
			cell = partition.erase(cell); // cell points to the cell after the erased object
			partition.splice(cell, new_cells); // insert new cells before cell
			// cell points to the object after the erased and inserted ones
			--cell; // gets again incremented in the for loop
		}
	}
	return partition.size() == adjm_.size();
}

void UndirectedGraph::set_colored_node_partition(
		std::list<std::set<int> >& partition,
		const std::vector<int>& node_colors,
		const std::set<int>& no_free_permute) const {

	ASSERT(node_colors.size() == adjm_.size());
	partition.clear();

	// update the node_coloring: nodes from no_free_perm must go to separate cells
	int cmax = 0, cmin = 0;
	if (!node_colors.empty()) {
		cmax = *std::max_element(node_colors.begin(), node_colors.end());
		cmin = *std::min_element(node_colors.begin(), node_colors.end());
	}
	vector<int> node_color_update = node_colors;
	for (unsigned int c = 0; c < adjm_.size(); ++c)
		if (no_free_permute.find(c) != no_free_permute.end())
			node_color_update[c] += cmax - cmin + 1;

	// partition nodes
	map<int, set<int> > nodes_by_color;
	for (unsigned int i = 0; i < adjm_.size(); ++i)
		nodes_by_color[node_color_update[i]].insert(i);
	map<int, set<int> >::iterator m;
	for (m = nodes_by_color.begin(); m != nodes_by_color.end(); ++m)
		move_to_back(partition, m->second);
}

// compare

int UndirectedGraph::compare_adjacency_matrix(const std::vector<int>& v1,
		const std::vector<int>& v2) const {
	ASSERT(v1.size() == adjm_.size() && v2.size() == adjm_.size());
	for (unsigned int i = 0; i < adjm_.size(); ++i)
		for (unsigned int j = i; j < adjm_.size(); ++j)
			if (adjm_[v1[i]][v1[j]] != adjm_[v2[i]][v2[j]])
				return ((adjm_[v1[i]][v1[j]] < adjm_[v2[i]][v2[j]]) ? -1 : 1);
	return 0;
}

int UndirectedGraph::compare_adjacency_list(const std::vector<int>& v1,
		const std::vector<int>& v2) const {
	ASSERT(v1.size() == adjm_.size() && v2.size() == adjm_.size());
	for (unsigned int i = 0; i < adjm_.size(); ++i) {
		map<int, colored_multi_edge> tmp1, tmp2;
		for (unsigned int j = i; j < adjm_.size(); ++j) {
			if (!adjm_[v1[i]][v1[j]].is_zero())
				tmp1.insert(make_pair(j, adjm_[v1[i]][v1[j]]));
			if (!adjm_[v2[i]][v2[j]].is_zero())
				tmp2.insert(make_pair(j, adjm_[v2[i]][v2[j]]));
		}
		if (tmp1 != tmp2)
			return ((tmp1 < tmp2) ? -1 : 1);
	}
	return 0;
}

// adjacency lists

void UndirectedGraph::set_adjacency_list(
		std::vector<std::map<int, colored_multi_edge> >& adjacency_list,
		const std::vector<int>& v) const {
	ASSERT(v.size() == adjm_.size());
	vector<map<int, colored_multi_edge> > tmp(adjm_.size());
	for (unsigned int i = 0; i < adjm_.size(); ++i)
		for (unsigned int j = i; j < adjm_.size(); ++j)
			if (!adjm_[v[i]][v[j]].is_zero())
				tmp[i].insert(make_pair(j, adjm_[v[i]][v[j]]));
	tmp.swap(adjacency_list);
}

// NodeSignature

UndirectedGraph::NodeSignature::NodeSignature(int num_colors) :
		degree_(num_colors), loops_(num_colors) {
}

bool UndirectedGraph::NodeSignature::operator<(
		const NodeSignature& other) const {
	if (degree_ != other.degree_)
		return degree_ < other.degree_;
	if (loops_ != other.loops_)
		return loops_ < other.loops_;
	if (multiplicities_.size() != other.multiplicities_.size())
		return multiplicities_.size() < other.multiplicities_.size();
	return multiplicities_ < other.multiplicities_;
}

bool UndirectedGraph::NodeSignature::operator==(
		const NodeSignature& other) const {
	return degree_ == other.degree_ && loops_ == other.loops_
			&& multiplicities_ == other.multiplicities_;
}

bool UndirectedGraph::NodeSignature::operator!=(
		const NodeSignature& other) const {
	return !(*this == other);
}

} // namespace Reduze

