/*
 *  Copyright (C) 2005-2007 Renier Roth
 *  www.renier.de
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package de.renier.vdr.channel.editor.actions;

import java.awt.event.ActionEvent;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;

import javax.swing.AbstractAction;
import javax.swing.ImageIcon;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.filechooser.FileFilter;
import javax.swing.tree.DefaultMutableTreeNode;

import de.renier.vdr.channel.editor.ChannelEditor;
import de.renier.vdr.channel.editor.Messages;
import de.renier.vdr.channel.editor.util.Utils;

/**
 * OpenAction
 * 
 * @author <a href="mailto:editor@renier.de">Renier Roth</a>
 */
public class OpenAction extends AbstractAction {

  private static final long serialVersionUID = 4716127180073832326L;
  private File lastDirectory = null;
  private File openFile = null;

  public OpenAction() {
    super(Messages.getString("OpenAction.0"), new ImageIcon(OpenAction.class.getResource("/org/javalobby/icons/20x20/Open.gif"))); //$NON-NLS-1$ //$NON-NLS-2$
  }

  public OpenAction(File file) {
    super(file.getAbsolutePath());
    this.openFile = file;
  }

  /*
   * (non-Javadoc)
   * 
   * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
   */
  public void actionPerformed(ActionEvent e) {
    // check if data changed
    if (ChannelEditor.application.isModified()) {
      // dialogbox to warn unsaved data
      int result = JOptionPane.showConfirmDialog(ChannelEditor.application,
                                                 Messages.getString("OpenAction.2"), //$NON-NLS-1$
                                                 Messages.getString("OpenAction.3"), JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE); //$NON-NLS-1$
      if (result == JOptionPane.NO_OPTION) { return; }
    }

    // check if we have a direkt file open
    File channelFile = null;
    if (this.openFile != null) {
      channelFile = this.openFile;
    } else {
      // open file Chooser
      final JFileChooser fc = new JFileChooser();
      if (lastDirectory != null) {
        fc.setCurrentDirectory(lastDirectory);
      }
      FileFilter fFilter = new FileFilter() {

        public boolean accept(File f) {
          if (f.isDirectory()) { return true; }
          if (f.getName().endsWith(".conf")) { return true; } //$NON-NLS-1$
          return false;
        }

        public String getDescription() {
          return Messages.getString("OpenAction.5"); //$NON-NLS-1$
        }
      };
      fc.setFileFilter(fFilter);
      int ret = fc.showOpenDialog(ChannelEditor.application);
      if (ret == JFileChooser.APPROVE_OPTION) {
        channelFile = fc.getSelectedFile();
      }
    }
    // open the file
    if (channelFile != null) {
      // save last dir
      lastDirectory = channelFile.getParentFile();
      try {
        DefaultMutableTreeNode rootNode = Utils.buildChannelTree(new FileReader(channelFile), channelFile.getName());
        ChannelEditor.application.getChannelListingPanel().setDefaultTreeModel(rootNode);
        // set last opened file list
        Utils.setLastOpenedFile(channelFile);
        // refresh menu
        ChannelEditor.application.refreshLastOpenedFiles();
      } catch (FileNotFoundException fnfe) {
        JOptionPane.showMessageDialog(ChannelEditor.application, Messages.getString("OpenAction.6") //$NON-NLS-1$
            + channelFile.getAbsolutePath() + Messages.getString("OpenAction.7") + fnfe.getMessage()); //$NON-NLS-1$
        channelFile = null;
      }
    }
    ChannelEditor.application.setChannelFile(channelFile);
  }

}
