// @HEADER
// ************************************************************************
//
//                           Intrepid2 Package
//                 Copyright (2007) Sandia Corporation
//
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// 1. Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the Corporation nor the names of the
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY SANDIA CORPORATION "AS IS" AND ANY
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SANDIA CORPORATION OR THE
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// Questions? Contact Kyungjoo Kim  (kyukim@sandia.gov), or
//                    Mauro Perego  (mperego@sandia.gov)
//
// ************************************************************************
// @HEADER

/** \file   Intrepid2_CubatureDirectTetDefaultDef.hpp
    \brief  Definition file for the Intrepid2::CubatureDirectTetSymmetric class.
    \author Created by M. Perego from an initial implementation by John Burkardt 
            (https://people.sc.fsu.edu/~jburkardt/py_src/tetrahedron_jaskowiec_rule/tetrahedron_jaskowiec_rule.html)
*/

namespace Intrepid2 {

  template <typename DT, typename PT, typename WT>
  CubatureDirectTetSymmetric<DT,PT,WT>::
  CubatureDirectTetSymmetric(const ordinal_type degree)
    : CubatureDirect<DT>(degree, 3) {

    INTREPID2_TEST_FOR_EXCEPTION( degree < 0 ||
                                  degree > static_cast<ordinal_type>(Parameters::MaxCubatureDegreeTet), std::out_of_range,
                                  ">>> ERROR (CubatureDirectTetSymmetric): No cubature rule implemented for the desired polynomial degree.");

    typedef Kokkos::DynRankView<PT,Kokkos::LayoutRight,Kokkos::HostSpace> pointViewHostType;
    typedef Kokkos::DynRankView<WT,Kokkos::LayoutRight,Kokkos::HostSpace> weightViewHostType;

    this->cubatureData_.numPoints_ = cubatureDataStatic_[this->degree_].numPoints_;
    const Kokkos::pair<ordinal_type,ordinal_type> pointRange(0, this->cubatureData_.numPoints_);
    {
      // src
      const pointViewHostType points_host(const_cast<PT*>( &(cubatureDataStatic_[this->degree_].points_[0][0]) ),
                                          pointRange.second,
                                          Parameters::MaxDimension);

      auto points = Kokkos::create_mirror_view(typename DT::memory_space(), points_host);

      Kokkos::deep_copy(points,points_host);

      // dst
      this->cubatureData_.points_ = Kokkos::DynRankView<PT,DT>("CubatureDirectTetSymmetric::cubatureData_::points_",
                                                                pointRange.second,
                                                                Parameters::MaxDimension);
      // copy
      Kokkos::deep_copy(this->cubatureData_.points_ , points );
    }
    {
      // src
      const weightViewHostType weights(const_cast<PT*>( &(cubatureDataStatic_[this->degree_].weights_[0]) ),
                                       pointRange.second);

      // dst
      this->cubatureData_.weights_ = Kokkos::DynRankView<WT,DT>("CubatureDirectTetSymmetric::cubatureData_::weights_",
                                                                 pointRange.second);
      // copy
      Kokkos::deep_copy(Kokkos::subdynrankview(this->cubatureData_.weights_, Kokkos::ALL()) , Kokkos::subdynrankview(weights, Kokkos::ALL()));
    }
  }

  //-------------------------------------------------------------------------------------//
  //                          Definition of cubature templates                           //
  //-------------------------------------------------------------------------------------//
  
  /*
    Cubature templates for lines are defined the reference Tetrahedron:
    
    Tetrahedron    -> (0,0,0), (1,0,0), (0,1,0), (0,0,1)
  */
  
  /*
    This static const member contains templates for default tetrahedron rules.
  */
  
  template<typename DT, typename PT, typename WT>
  const typename CubatureDirect<DT,PT,WT>::CubatureDataStatic
  CubatureDirectTetSymmetric<DT,PT,WT>::
  cubatureDataStatic_[cubatureDataStaticSize] = {
    // Cubature templates for the reference tet {(0,0,0), (1,0,0), (0,1,0), (0,0,1)}
    //
    {
      1,
      {{1./4., 1./4., 1./4.}},
      {1./6.}
    },
    {
      1,
      {{1./4., 1./4., 1./4.}},
      {1./6.}
    },
    {
      4,
      {{0.1381966011250105151795413165634361882280, 0.1381966011250105151795413165634361882280, 0.1381966011250105151795413165634361882280},
       {0.5854101966249684544613760503096914353161, 0.1381966011250105151795413165634361882280, 0.1381966011250105151795413165634361882280},
       {0.1381966011250105151795413165634361882280, 0.5854101966249684544613760503096914353161, 0.1381966011250105151795413165634361882280},
       {0.1381966011250105151795413165634361882280, 0.1381966011250105151795413165634361882280, 0.5854101966249684544613760503096914353161}},
      {1./24.,
       1./24.,
       1./24.,
       1./24.}
    },
    {
      8,
      {{0.3286811466653490, 0.3286811466653490, 0.3286811466653490},
      {0.0139565600039530, 0.3286811466653490, 0.3286811466653490},
      {0.3286811466653490, 0.0139565600039530, 0.3286811466653490},
      {0.3286811466653490, 0.3286811466653490, 0.0139565600039530},
      {0.1119207275092916, 0.1119207275092916, 0.1119207275092916},
      {0.6642378174721254, 0.1119207275092916, 0.1119207275092916},
      {0.1119207275092916, 0.6642378174721254, 0.1119207275092916},
      {0.1119207275092916, 0.1119207275092916, 0.6642378174721254}},
      {0.0212485519262511,
      0.0212485519262511,
      0.0212485519262511,
      0.0212485519262511,
      0.0204181147404156,
      0.0204181147404156,
      0.0204181147404156,
      0.0204181147404156}
    },
    {
      14,
      {{0.3095821653179519, 0.3095821653179519, 0.3095821653179519},
      {0.0712535040461443, 0.3095821653179519, 0.3095821653179519},
      {0.3095821653179519, 0.0712535040461443, 0.3095821653179519},
      {0.3095821653179519, 0.3095821653179519, 0.0712535040461443},
      {0.0834427723039776, 0.0834427723039776, 0.0834427723039776},
      {0.7496716830880673, 0.0834427723039776, 0.0834427723039776},
      {0.0834427723039776, 0.7496716830880673, 0.0834427723039776},
      {0.0834427723039776, 0.0834427723039776, 0.7496716830880673},
      {0.0772209540700587, 0.4227790459299413, 0.4227790459299413},
      {0.4227790459299413, 0.0772209540700587, 0.4227790459299413},
      {0.0772209540700587, 0.0772209540700587, 0.4227790459299413},
      {0.4227790459299413, 0.4227790459299413, 0.0772209540700587},
      {0.0772209540700587, 0.4227790459299413, 0.0772209540700587},
      {0.4227790459299413, 0.0772209540700587, 0.0772209540700587}},
      {0.0109015272724087,
      0.0109015272724087,
      0.0109015272724087,
      0.0109015272724087,
      0.0098925440385008,
      0.0098925440385008,
      0.0098925440385008,
      0.0098925440385008,
      0.0139150635705048,
      0.0139150635705048,
      0.0139150635705048,
      0.0139150635705048,
      0.0139150635705048,
      0.0139150635705048}
    },
    {
      14,
      {{0.3108859192633006, 0.3108859192633006, 0.3108859192633006},
      {0.0673422422100982, 0.3108859192633006, 0.3108859192633006},
      {0.3108859192633006, 0.0673422422100982, 0.3108859192633006},
      {0.3108859192633006, 0.3108859192633006, 0.0673422422100982},
      {0.0927352503108912, 0.0927352503108912, 0.0927352503108912},
      {0.7217942490673264, 0.0927352503108912, 0.0927352503108912},
      {0.0927352503108912, 0.7217942490673264, 0.0927352503108912},
      {0.0927352503108912, 0.0927352503108912, 0.7217942490673264},
      {0.0455037041256496, 0.4544962958743504, 0.4544962958743504},
      {0.4544962958743504, 0.0455037041256496, 0.4544962958743504},
      {0.0455037041256496, 0.0455037041256496, 0.4544962958743504},
      {0.4544962958743504, 0.4544962958743504, 0.0455037041256496},
      {0.0455037041256496, 0.4544962958743504, 0.0455037041256496},
      {0.4544962958743504, 0.0455037041256496, 0.0455037041256496}},
      {0.0187813209530026,
      0.0187813209530026,
      0.0187813209530026,
      0.0187813209530026,
      0.0122488405193937,
      0.0122488405193937,
      0.0122488405193937,
      0.0122488405193937,
      0.0070910034628469,
      0.0070910034628469,
      0.0070910034628469,
      0.0070910034628469,
      0.0070910034628469,
      0.0070910034628469}
    },
    {
      24,
      {{0.0406739585346114, 0.0406739585346114, 0.0406739585346114},
      {0.8779781243961660, 0.0406739585346114, 0.0406739585346114},
      {0.0406739585346114, 0.8779781243961660, 0.0406739585346114},
      {0.0406739585346114, 0.0406739585346114, 0.8779781243961660},
      {0.3223378901422755, 0.3223378901422755, 0.3223378901422755},
      {0.0329863295731735, 0.3223378901422755, 0.3223378901422755},
      {0.3223378901422755, 0.0329863295731735, 0.3223378901422755},
      {0.3223378901422755, 0.3223378901422755, 0.0329863295731735},
      {0.2146028712591520, 0.2146028712591520, 0.2146028712591520},
      {0.3561913862225439, 0.2146028712591520, 0.2146028712591520},
      {0.2146028712591520, 0.3561913862225439, 0.2146028712591520},
      {0.2146028712591520, 0.2146028712591520, 0.3561913862225439},
      {0.6030056647916492, 0.0636610018750175, 0.0636610018750175},
      {0.0636610018750175, 0.6030056647916492, 0.0636610018750175},
      {0.0636610018750175, 0.0636610018750175, 0.6030056647916492},
      {0.2696723314583158, 0.0636610018750175, 0.0636610018750175},
      {0.2696723314583158, 0.6030056647916492, 0.0636610018750175},
      {0.2696723314583158, 0.0636610018750175, 0.6030056647916492},
      {0.0636610018750175, 0.2696723314583158, 0.0636610018750175},
      {0.6030056647916492, 0.2696723314583158, 0.0636610018750175},
      {0.0636610018750175, 0.2696723314583158, 0.6030056647916492},
      {0.0636610018750175, 0.0636610018750175, 0.2696723314583158},
      {0.6030056647916492, 0.0636610018750175, 0.2696723314583158},
      {0.0636610018750175, 0.6030056647916492, 0.2696723314583158}},
      {0.0016795351758868,
      0.0016795351758868,
      0.0016795351758868,
      0.0016795351758868,
      0.0092261969239425,
      0.0092261969239425,
      0.0092261969239425,
      0.0092261969239425,
      0.0066537917096946,
      0.0066537917096946,
      0.0066537917096946,
      0.0066537917096946,
      0.0080357142857143,
      0.0080357142857143,
      0.0080357142857143,
      0.0080357142857143,
      0.0080357142857143,
      0.0080357142857143,
      0.0080357142857143,
      0.0080357142857143,
      0.0080357142857143,
      0.0080357142857143,
      0.0080357142857143,
      0.0080357142857143}
    },
    {
      35,
      {{0.2500000000000000, 0.2500000000000000, 0.2500000000000000},
      {0.3157011497782028, 0.3157011497782028, 0.3157011497782028},
      {0.0528965506653916, 0.3157011497782028, 0.3157011497782028},
      {0.3157011497782028, 0.0528965506653916, 0.3157011497782028},
      {0.3157011497782028, 0.3157011497782028, 0.0528965506653916},
      {0.0504898225983964, 0.4495101774016036, 0.4495101774016036},
      {0.4495101774016036, 0.0504898225983964, 0.4495101774016036},
      {0.0504898225983964, 0.0504898225983964, 0.4495101774016036},
      {0.4495101774016036, 0.4495101774016036, 0.0504898225983964},
      {0.0504898225983964, 0.4495101774016036, 0.0504898225983964},
      {0.4495101774016036, 0.0504898225983964, 0.0504898225983964},
      {0.8108302410985485, 0.0212654725414832, 0.0212654725414832},
      {0.0212654725414832, 0.8108302410985485, 0.0212654725414832},
      {0.0212654725414832, 0.0212654725414832, 0.8108302410985485},
      {0.1466388138184850, 0.0212654725414832, 0.0212654725414832},
      {0.1466388138184850, 0.8108302410985485, 0.0212654725414832},
      {0.1466388138184850, 0.0212654725414832, 0.8108302410985485},
      {0.0212654725414832, 0.1466388138184850, 0.0212654725414832},
      {0.8108302410985485, 0.1466388138184850, 0.0212654725414832},
      {0.0212654725414832, 0.1466388138184850, 0.8108302410985485},
      {0.0212654725414832, 0.0212654725414832, 0.1466388138184850},
      {0.8108302410985485, 0.0212654725414832, 0.1466388138184850},
      {0.0212654725414832, 0.8108302410985485, 0.1466388138184850},
      {0.5751716375870001, 0.1888338310260010, 0.1888338310260010},
      {0.1888338310260010, 0.5751716375870001, 0.1888338310260010},
      {0.1888338310260010, 0.1888338310260010, 0.5751716375870001},
      {0.0471607003609979, 0.1888338310260010, 0.1888338310260010},
      {0.0471607003609979, 0.5751716375870001, 0.1888338310260010},
      {0.0471607003609979, 0.1888338310260010, 0.5751716375870001},
      {0.1888338310260010, 0.0471607003609979, 0.1888338310260010},
      {0.5751716375870001, 0.0471607003609979, 0.1888338310260010},
      {0.1888338310260010, 0.0471607003609979, 0.5751716375870001},
      {0.1888338310260010, 0.1888338310260010, 0.0471607003609979},
      {0.5751716375870001, 0.1888338310260010, 0.0471607003609979},
      {0.1888338310260010, 0.5751716375870001, 0.0471607003609979}},
      {0.0159142149106885,
      0.0070549302016612,
      0.0070549302016612,
      0.0070549302016612,
      0.0070549302016612,
      0.0053161546388096,
      0.0053161546388096,
      0.0053161546388096,
      0.0053161546388096,
      0.0053161546388096,
      0.0053161546388096,
      0.0013517951383172,
      0.0013517951383172,
      0.0013517951383172,
      0.0013517951383172,
      0.0013517951383172,
      0.0013517951383172,
      0.0013517951383172,
      0.0013517951383172,
      0.0013517951383172,
      0.0013517951383172,
      0.0013517951383172,
      0.0013517951383172,
      0.0062011884547224,
      0.0062011884547224,
      0.0062011884547224,
      0.0062011884547224,
      0.0062011884547224,
      0.0062011884547224,
      0.0062011884547224,
      0.0062011884547224,
      0.0062011884547224,
      0.0062011884547224,
      0.0062011884547224,
      0.0062011884547224}
    },
    {
      46,
      {{0.1868127582707210, 0.1868127582707210, 0.1868127582707210},
      {0.4395617251878371, 0.1868127582707210, 0.1868127582707210},
      {0.1868127582707210, 0.4395617251878371, 0.1868127582707210},
      {0.1868127582707210, 0.1868127582707210, 0.4395617251878371},
      {0.1144624067612305, 0.1144624067612305, 0.1144624067612305},
      {0.6566127797163085, 0.1144624067612305, 0.1144624067612305},
      {0.1144624067612305, 0.6566127797163085, 0.1144624067612305},
      {0.1144624067612305, 0.1144624067612305, 0.6566127797163085},
      {0.3138065922724016, 0.3138065922724016, 0.3138065922724016},
      {0.0585802231827951, 0.3138065922724016, 0.3138065922724016},
      {0.3138065922724016, 0.0585802231827951, 0.3138065922724016},
      {0.3138065922724016, 0.3138065922724016, 0.0585802231827951},
      {0.0445773794488462, 0.0445773794488462, 0.0445773794488462},
      {0.8662678616534613, 0.0445773794488462, 0.0445773794488462},
      {0.0445773794488462, 0.8662678616534613, 0.0445773794488462},
      {0.0445773794488462, 0.0445773794488462, 0.8662678616534613},
      {0.4345165094561576, 0.0654834905438424, 0.0654834905438424},
      {0.0654834905438424, 0.4345165094561576, 0.0654834905438424},
      {0.4345165094561576, 0.4345165094561576, 0.0654834905438424},
      {0.0654834905438424, 0.0654834905438424, 0.4345165094561576},
      {0.4345165094561576, 0.0654834905438424, 0.4345165094561576},
      {0.0654834905438424, 0.4345165094561576, 0.4345165094561576},
      {0.0050733207504216, 0.2038931746621103, 0.2038931746621103},
      {0.2038931746621103, 0.0050733207504216, 0.2038931746621103},
      {0.2038931746621103, 0.2038931746621103, 0.0050733207504216},
      {0.5871403299253578, 0.2038931746621103, 0.2038931746621103},
      {0.5871403299253578, 0.0050733207504216, 0.2038931746621103},
      {0.5871403299253578, 0.2038931746621103, 0.0050733207504216},
      {0.2038931746621103, 0.5871403299253578, 0.2038931746621103},
      {0.0050733207504216, 0.5871403299253578, 0.2038931746621103},
      {0.2038931746621103, 0.5871403299253578, 0.0050733207504216},
      {0.2038931746621103, 0.2038931746621103, 0.5871403299253578},
      {0.0050733207504216, 0.2038931746621103, 0.5871403299253578},
      {0.2038931746621103, 0.0050733207504216, 0.5871403299253578},
      {0.7146769263933049, 0.0212477796695717, 0.0212477796695717},
      {0.0212477796695717, 0.7146769263933049, 0.0212477796695717},
      {0.0212477796695717, 0.0212477796695717, 0.7146769263933049},
      {0.2428275142675518, 0.0212477796695717, 0.0212477796695717},
      {0.2428275142675518, 0.7146769263933049, 0.0212477796695717},
      {0.2428275142675518, 0.0212477796695717, 0.7146769263933049},
      {0.0212477796695717, 0.2428275142675518, 0.0212477796695717},
      {0.7146769263933049, 0.2428275142675518, 0.0212477796695717},
      {0.0212477796695717, 0.2428275142675518, 0.7146769263933049},
      {0.0212477796695717, 0.0212477796695717, 0.2428275142675518},
      {0.7146769263933049, 0.0212477796695717, 0.2428275142675518},
      {0.0212477796695717, 0.7146769263933049, 0.2428275142675518}},
      {0.0079688313390511,
      0.0079688313390511,
      0.0079688313390511,
      0.0079688313390511,
      0.0049546423477990,
      0.0049546423477990,
      0.0049546423477990,
      0.0049546423477990,
      0.0072545540364959,
      0.0072545540364959,
      0.0072545540364959,
      0.0072545540364959,
      0.0014387893368025,
      0.0014387893368025,
      0.0014387893368025,
      0.0014387893368025,
      0.0061484235449777,
      0.0061484235449777,
      0.0061484235449777,
      0.0061484235449777,
      0.0061484235449777,
      0.0061484235449777,
      0.0024161711184766,
      0.0024161711184766,
      0.0024161711184766,
      0.0024161711184766,
      0.0024161711184766,
      0.0024161711184766,
      0.0024161711184766,
      0.0024161711184766,
      0.0024161711184766,
      0.0024161711184766,
      0.0024161711184766,
      0.0024161711184766,
      0.0011929003112073,
      0.0011929003112073,
      0.0011929003112073,
      0.0011929003112073,
      0.0011929003112073,
      0.0011929003112073,
      0.0011929003112073,
      0.0011929003112073,
      0.0011929003112073,
      0.0011929003112073,
      0.0011929003112073,
      0.0011929003112073}
    },
    {
      59,
      {{0.2500000000000000, 0.2500000000000000, 0.2500000000000000},
      {0.1679066052367428, 0.1679066052367428, 0.1679066052367428},
      {0.4962801842897716, 0.1679066052367428, 0.1679066052367428},
      {0.1679066052367428, 0.4962801842897716, 0.1679066052367428},
      {0.1679066052367428, 0.1679066052367428, 0.4962801842897716},
      {0.0914362705140763, 0.0914362705140763, 0.0914362705140763},
      {0.7256911884577713, 0.0914362705140763, 0.0914362705140763},
      {0.0914362705140763, 0.7256911884577713, 0.0914362705140763},
      {0.0914362705140763, 0.0914362705140763, 0.7256911884577713},
      {0.3218556648176533, 0.3218556648176533, 0.3218556648176533},
      {0.0344330055470402, 0.3218556648176533, 0.3218556648176533},
      {0.3218556648176533, 0.0344330055470402, 0.3218556648176533},
      {0.3218556648176533, 0.3218556648176533, 0.0344330055470402},
      {0.0418376959003656, 0.0418376959003656, 0.0418376959003656},
      {0.8744869122989032, 0.0418376959003656, 0.0418376959003656},
      {0.0418376959003656, 0.8744869122989032, 0.0418376959003656},
      {0.0418376959003656, 0.0418376959003656, 0.8744869122989032},
      {0.3932705614251537, 0.1067294385748464, 0.1067294385748464},
      {0.1067294385748464, 0.3932705614251537, 0.1067294385748464},
      {0.3932705614251537, 0.3932705614251537, 0.1067294385748464},
      {0.1067294385748464, 0.1067294385748464, 0.3932705614251537},
      {0.3932705614251537, 0.1067294385748464, 0.3932705614251537},
      {0.1067294385748464, 0.3932705614251537, 0.3932705614251537},
      {0.7183930939814244, 0.0339571681830889, 0.0339571681830889},
      {0.0339571681830889, 0.7183930939814244, 0.0339571681830889},
      {0.0339571681830889, 0.0339571681830889, 0.7183930939814244},
      {0.2136925696523978, 0.0339571681830889, 0.0339571681830889},
      {0.2136925696523978, 0.7183930939814244, 0.0339571681830889},
      {0.2136925696523978, 0.0339571681830889, 0.7183930939814244},
      {0.0339571681830889, 0.2136925696523978, 0.0339571681830889},
      {0.7183930939814244, 0.2136925696523978, 0.0339571681830889},
      {0.0339571681830889, 0.2136925696523978, 0.7183930939814244},
      {0.0339571681830889, 0.0339571681830889, 0.2136925696523978},
      {0.7183930939814244, 0.0339571681830889, 0.2136925696523978},
      {0.0339571681830889, 0.7183930939814244, 0.2136925696523978},
      {0.0786836344766879, 0.4606581810547776, 0.4606581810547776},
      {0.4606581810547776, 0.0786836344766879, 0.4606581810547776},
      {0.4606581810547776, 0.4606581810547776, 0.0786836344766879},
      {0.0000000034137568, 0.4606581810547776, 0.4606581810547776},
      {0.0000000034137568, 0.0786836344766879, 0.4606581810547776},
      {0.0000000034137568, 0.4606581810547776, 0.0786836344766879},
      {0.4606581810547776, 0.0000000034137568, 0.4606581810547776},
      {0.0786836344766879, 0.0000000034137568, 0.4606581810547776},
      {0.4606581810547776, 0.0000000034137568, 0.0786836344766879},
      {0.4606581810547776, 0.4606581810547776, 0.0000000034137568},
      {0.0786836344766879, 0.4606581810547776, 0.0000000034137568},
      {0.4606581810547776, 0.0786836344766879, 0.0000000034137568},
      {0.5972107227618499, 0.1843879435000152, 0.1843879435000152},
      {0.1843879435000152, 0.5972107227618499, 0.1843879435000152},
      {0.1843879435000152, 0.1843879435000152, 0.5972107227618499},
      {0.0340133902381197, 0.1843879435000152, 0.1843879435000152},
      {0.0340133902381197, 0.5972107227618499, 0.1843879435000152},
      {0.0340133902381197, 0.1843879435000152, 0.5972107227618499},
      {0.1843879435000152, 0.0340133902381197, 0.1843879435000152},
      {0.5972107227618499, 0.0340133902381197, 0.1843879435000152},
      {0.1843879435000152, 0.0340133902381197, 0.5972107227618499},
      {0.1843879435000152, 0.1843879435000152, 0.0340133902381197},
      {0.5972107227618499, 0.1843879435000152, 0.0340133902381197},
      {0.1843879435000152, 0.5972107227618499, 0.0340133902381197}},
      {0.0094777707089253,
      0.0042823794482539,
      0.0042823794482539,
      0.0042823794482539,
      0.0042823794482539,
      0.0003831538921714,
      0.0003831538921714,
      0.0003831538921714,
      0.0003831538921714,
      0.0050760064445985,
      0.0050760064445985,
      0.0050760064445985,
      0.0050760064445985,
      0.0011872870567065,
      0.0011872870567065,
      0.0011872870567065,
      0.0011872870567065,
      0.0061406092468240,
      0.0061406092468240,
      0.0061406092468240,
      0.0061406092468240,
      0.0061406092468240,
      0.0061406092468240,
      0.0017203427970350,
      0.0017203427970350,
      0.0017203427970350,
      0.0017203427970350,
      0.0017203427970350,
      0.0017203427970350,
      0.0017203427970350,
      0.0017203427970350,
      0.0017203427970350,
      0.0017203427970350,
      0.0017203427970350,
      0.0017203427970350,
      0.0012724811923301,
      0.0012724811923301,
      0.0012724811923301,
      0.0012724811923301,
      0.0012724811923301,
      0.0012724811923301,
      0.0012724811923301,
      0.0012724811923301,
      0.0012724811923301,
      0.0012724811923301,
      0.0012724811923301,
      0.0012724811923301,
      0.0033930037697913,
      0.0033930037697913,
      0.0033930037697913,
      0.0033930037697913,
      0.0033930037697913,
      0.0033930037697913,
      0.0033930037697913,
      0.0033930037697913,
      0.0033930037697913,
      0.0033930037697913,
      0.0033930037697913,
      0.0033930037697913}
    },
    {
      81,
      {{0.2500000000000000, 0.2500000000000000, 0.2500000000000000},
      {0.0725517574174378, 0.0725517574174378, 0.0725517574174378},
      {0.7823447277476866, 0.0725517574174378, 0.0725517574174378},
      {0.0725517574174378, 0.7823447277476866, 0.0725517574174378},
      {0.0725517574174378, 0.0725517574174378, 0.7823447277476866},
      {0.3078959478669229, 0.3078959478669229, 0.3078959478669229},
      {0.0763121563992313, 0.3078959478669229, 0.3078959478669229},
      {0.3078959478669229, 0.0763121563992313, 0.3078959478669229},
      {0.3078959478669229, 0.3078959478669229, 0.0763121563992313},
      {0.4762808603927791, 0.0237191396072209, 0.0237191396072209},
      {0.0237191396072209, 0.4762808603927791, 0.0237191396072209},
      {0.4762808603927791, 0.4762808603927791, 0.0237191396072209},
      {0.0237191396072209, 0.0237191396072209, 0.4762808603927791},
      {0.4762808603927791, 0.0237191396072209, 0.4762808603927791},
      {0.0237191396072209, 0.4762808603927791, 0.4762808603927791},
      {0.0965797730691073, 0.4034202269308927, 0.4034202269308927},
      {0.4034202269308927, 0.0965797730691073, 0.4034202269308927},
      {0.0965797730691073, 0.0965797730691073, 0.4034202269308927},
      {0.4034202269308927, 0.4034202269308927, 0.0965797730691073},
      {0.0965797730691073, 0.4034202269308927, 0.0965797730691073},
      {0.4034202269308927, 0.0965797730691073, 0.0965797730691073},
      {0.0738541810184862, 0.2017386068476799, 0.2017386068476799},
      {0.2017386068476799, 0.0738541810184862, 0.2017386068476799},
      {0.2017386068476799, 0.2017386068476799, 0.0738541810184862},
      {0.5226686052861540, 0.2017386068476799, 0.2017386068476799},
      {0.5226686052861540, 0.0738541810184862, 0.2017386068476799},
      {0.5226686052861540, 0.2017386068476799, 0.0738541810184862},
      {0.2017386068476799, 0.5226686052861540, 0.2017386068476799},
      {0.0738541810184862, 0.5226686052861540, 0.2017386068476799},
      {0.2017386068476799, 0.5226686052861540, 0.0738541810184862},
      {0.2017386068476799, 0.2017386068476799, 0.5226686052861540},
      {0.0738541810184862, 0.2017386068476799, 0.5226686052861540},
      {0.2017386068476799, 0.0738541810184862, 0.5226686052861540},
      {0.6875048558830396, 0.1548397007438618, 0.1548397007438618},
      {0.1548397007438618, 0.6875048558830396, 0.1548397007438618},
      {0.1548397007438618, 0.1548397007438618, 0.6875048558830396},
      {0.0028157426292368, 0.1548397007438618, 0.1548397007438618},
      {0.0028157426292368, 0.6875048558830396, 0.1548397007438618},
      {0.0028157426292368, 0.1548397007438618, 0.6875048558830396},
      {0.1548397007438618, 0.0028157426292368, 0.1548397007438618},
      {0.6875048558830396, 0.0028157426292368, 0.1548397007438618},
      {0.1548397007438618, 0.0028157426292368, 0.6875048558830396},
      {0.1548397007438618, 0.1548397007438618, 0.0028157426292368},
      {0.6875048558830396, 0.1548397007438618, 0.0028157426292368},
      {0.1548397007438618, 0.6875048558830396, 0.0028157426292368},
      {0.1741437164939321, 0.4125991504993639, 0.4125991504993639},
      {0.4125991504993639, 0.1741437164939321, 0.4125991504993639},
      {0.4125991504993639, 0.4125991504993639, 0.1741437164939321},
      {0.0006579825073401, 0.4125991504993639, 0.4125991504993639},
      {0.0006579825073401, 0.1741437164939321, 0.4125991504993639},
      {0.0006579825073401, 0.4125991504993639, 0.1741437164939321},
      {0.4125991504993639, 0.0006579825073401, 0.4125991504993639},
      {0.1741437164939321, 0.0006579825073401, 0.4125991504993639},
      {0.4125991504993639, 0.0006579825073401, 0.1741437164939321},
      {0.4125991504993639, 0.4125991504993639, 0.0006579825073401},
      {0.1741437164939321, 0.4125991504993639, 0.0006579825073401},
      {0.4125991504993639, 0.1741437164939321, 0.0006579825073401},
      {0.2657575065627199, 0.0388907275578065, 0.0388907275578065},
      {0.0388907275578065, 0.2657575065627199, 0.0388907275578065},
      {0.0388907275578065, 0.0388907275578065, 0.2657575065627199},
      {0.6564610383216671, 0.0388907275578065, 0.0388907275578065},
      {0.6564610383216671, 0.2657575065627199, 0.0388907275578065},
      {0.6564610383216671, 0.0388907275578065, 0.2657575065627199},
      {0.0388907275578065, 0.6564610383216671, 0.0388907275578065},
      {0.2657575065627199, 0.6564610383216671, 0.0388907275578065},
      {0.0388907275578065, 0.6564610383216671, 0.2657575065627199},
      {0.0388907275578065, 0.0388907275578065, 0.6564610383216671},
      {0.2657575065627199, 0.0388907275578065, 0.6564610383216671},
      {0.0388907275578065, 0.2657575065627199, 0.6564610383216671},
      {0.0932413490352584, 0.0056048479510213, 0.0056048479510213},
      {0.0056048479510213, 0.0932413490352584, 0.0056048479510213},
      {0.0056048479510213, 0.0056048479510213, 0.0932413490352584},
      {0.8955489550626989, 0.0056048479510213, 0.0056048479510213},
      {0.8955489550626989, 0.0932413490352584, 0.0056048479510213},
      {0.8955489550626989, 0.0056048479510213, 0.0932413490352584},
      {0.0056048479510213, 0.8955489550626989, 0.0056048479510213},
      {0.0932413490352584, 0.8955489550626989, 0.0056048479510213},
      {0.0056048479510213, 0.8955489550626989, 0.0932413490352584},
      {0.0056048479510213, 0.0056048479510213, 0.8955489550626989},
      {0.0932413490352584, 0.0056048479510213, 0.8955489550626989},
      {0.0056048479510213, 0.0932413490352584, 0.8955489550626989}},
      {0.0086084820426829,
      0.0022446621119434,
      0.0022446621119434,
      0.0022446621119434,
      0.0022446621119434,
      0.0044111532082816,
      0.0044111532082816,
      0.0044111532082816,
      0.0044111532082816,
      0.0008511746380239,
      0.0008511746380239,
      0.0008511746380239,
      0.0008511746380239,
      0.0008511746380239,
      0.0008511746380239,
      0.0039416660263743,
      0.0039416660263743,
      0.0039416660263743,
      0.0039416660263743,
      0.0039416660263743,
      0.0039416660263743,
      0.0042195611084947,
      0.0042195611084947,
      0.0042195611084947,
      0.0042195611084947,
      0.0042195611084947,
      0.0042195611084947,
      0.0042195611084947,
      0.0042195611084947,
      0.0042195611084947,
      0.0042195611084947,
      0.0042195611084947,
      0.0042195611084947,
      0.0009027105679080,
      0.0009027105679080,
      0.0009027105679080,
      0.0009027105679080,
      0.0009027105679080,
      0.0009027105679080,
      0.0009027105679080,
      0.0009027105679080,
      0.0009027105679080,
      0.0009027105679080,
      0.0009027105679080,
      0.0009027105679080,
      0.0013589431686757,
      0.0013589431686757,
      0.0013589431686757,
      0.0013589431686757,
      0.0013589431686757,
      0.0013589431686757,
      0.0013589431686757,
      0.0013589431686757,
      0.0013589431686757,
      0.0013589431686757,
      0.0013589431686757,
      0.0013589431686757,
      0.0018543286266021,
      0.0018543286266021,
      0.0018543286266021,
      0.0018543286266021,
      0.0018543286266021,
      0.0018543286266021,
      0.0018543286266021,
      0.0018543286266021,
      0.0018543286266021,
      0.0018543286266021,
      0.0018543286266021,
      0.0018543286266021,
      0.0002209464747107,
      0.0002209464747107,
      0.0002209464747107,
      0.0002209464747107,
      0.0002209464747107,
      0.0002209464747107,
      0.0002209464747107,
      0.0002209464747107,
      0.0002209464747107,
      0.0002209464747107,
      0.0002209464747107,
      0.0002209464747107}
    },
    {
      110,
      {{0.2993130923993425, 0.2993130923993425, 0.2993130923993425},
      {0.1020607228019725, 0.2993130923993425, 0.2993130923993425},
      {0.2993130923993425, 0.1020607228019725, 0.2993130923993425},
      {0.2993130923993425, 0.2993130923993425, 0.1020607228019725},
      {0.0326328439651818, 0.0326328439651818, 0.0326328439651818},
      {0.9021014681044546, 0.0326328439651818, 0.0326328439651818},
      {0.0326328439651818, 0.9021014681044546, 0.0326328439651818},
      {0.0326328439651818, 0.0326328439651818, 0.9021014681044546},
      {0.3257955845153108, 0.1742044154846892, 0.1742044154846892},
      {0.1742044154846892, 0.3257955845153108, 0.1742044154846892},
      {0.3257955845153108, 0.3257955845153108, 0.1742044154846892},
      {0.1742044154846892, 0.1742044154846892, 0.3257955845153108},
      {0.3257955845153108, 0.1742044154846892, 0.3257955845153108},
      {0.1742044154846892, 0.3257955845153108, 0.3257955845153108},
      {0.0987846241416251, 0.4012153758583749, 0.4012153758583749},
      {0.4012153758583749, 0.0987846241416251, 0.4012153758583749},
      {0.0987846241416251, 0.0987846241416251, 0.4012153758583749},
      {0.4012153758583749, 0.4012153758583749, 0.0987846241416251},
      {0.0987846241416251, 0.4012153758583749, 0.0987846241416251},
      {0.4012153758583749, 0.0987846241416251, 0.0987846241416251},
      {0.0057019051227004, 0.4942980948772996, 0.4942980948772996},
      {0.4942980948772996, 0.0057019051227004, 0.4942980948772996},
      {0.0057019051227004, 0.0057019051227004, 0.4942980948772996},
      {0.4942980948772996, 0.4942980948772996, 0.0057019051227004},
      {0.0057019051227004, 0.4942980948772996, 0.0057019051227004},
      {0.4942980948772996, 0.0057019051227004, 0.0057019051227004},
      {0.7273303463463835, 0.1231206549747224, 0.1231206549747224},
      {0.1231206549747224, 0.7273303463463835, 0.1231206549747224},
      {0.1231206549747224, 0.1231206549747224, 0.7273303463463835},
      {0.0264283437041718, 0.1231206549747224, 0.1231206549747224},
      {0.0264283437041718, 0.7273303463463835, 0.1231206549747224},
      {0.0264283437041718, 0.1231206549747224, 0.7273303463463835},
      {0.1231206549747224, 0.0264283437041718, 0.1231206549747224},
      {0.7273303463463835, 0.0264283437041718, 0.1231206549747224},
      {0.1231206549747224, 0.0264283437041718, 0.7273303463463835},
      {0.1231206549747224, 0.1231206549747224, 0.0264283437041718},
      {0.7273303463463835, 0.1231206549747224, 0.0264283437041718},
      {0.1231206549747224, 0.7273303463463835, 0.0264283437041718},
      {0.1702280734892642, 0.0130038704855902, 0.0130038704855902},
      {0.0130038704855902, 0.1702280734892642, 0.0130038704855902},
      {0.0130038704855902, 0.0130038704855902, 0.1702280734892642},
      {0.8037641855395553, 0.0130038704855902, 0.0130038704855902},
      {0.8037641855395553, 0.1702280734892642, 0.0130038704855902},
      {0.8037641855395553, 0.0130038704855902, 0.1702280734892642},
      {0.0130038704855902, 0.8037641855395553, 0.0130038704855902},
      {0.1702280734892642, 0.8037641855395553, 0.0130038704855902},
      {0.0130038704855902, 0.8037641855395553, 0.1702280734892642},
      {0.0130038704855902, 0.0130038704855902, 0.8037641855395553},
      {0.1702280734892642, 0.0130038704855902, 0.8037641855395553},
      {0.0130038704855902, 0.1702280734892642, 0.8037641855395553},
      {0.6182598097891158, 0.0429354011719257, 0.0429354011719257},
      {0.0429354011719257, 0.6182598097891158, 0.0429354011719257},
      {0.0429354011719257, 0.0429354011719257, 0.6182598097891158},
      {0.2958693878670327, 0.0429354011719257, 0.0429354011719257},
      {0.2958693878670327, 0.6182598097891158, 0.0429354011719257},
      {0.2958693878670327, 0.0429354011719257, 0.6182598097891158},
      {0.0429354011719257, 0.2958693878670327, 0.0429354011719257},
      {0.6182598097891158, 0.2958693878670327, 0.0429354011719257},
      {0.0429354011719257, 0.2958693878670327, 0.6182598097891158},
      {0.0429354011719257, 0.0429354011719257, 0.2958693878670327},
      {0.6182598097891158, 0.0429354011719257, 0.2958693878670327},
      {0.0429354011719257, 0.6182598097891158, 0.2958693878670327},
      {0.5399001168557016, 0.1844240320470778, 0.1844240320470778},
      {0.1844240320470778, 0.5399001168557016, 0.1844240320470778},
      {0.1844240320470778, 0.1844240320470778, 0.5399001168557016},
      {0.0912518190501429, 0.1844240320470778, 0.1844240320470778},
      {0.0912518190501429, 0.5399001168557016, 0.1844240320470778},
      {0.0912518190501429, 0.1844240320470778, 0.5399001168557016},
      {0.1844240320470778, 0.0912518190501429, 0.1844240320470778},
      {0.5399001168557016, 0.0912518190501429, 0.1844240320470778},
      {0.1844240320470778, 0.0912518190501429, 0.5399001168557016},
      {0.1844240320470778, 0.1844240320470778, 0.0912518190501429},
      {0.5399001168557016, 0.1844240320470778, 0.0912518190501429},
      {0.1844240320470778, 0.5399001168557016, 0.0912518190501429},
      {0.4554353196219595, 0.2591694074642826, 0.2591694074642826},
      {0.2591694074642826, 0.4554353196219595, 0.2591694074642826},
      {0.2591694074642826, 0.2591694074642826, 0.4554353196219595},
      {0.0262258654494752, 0.2591694074642826, 0.2591694074642826},
      {0.0262258654494752, 0.4554353196219595, 0.2591694074642826},
      {0.0262258654494752, 0.2591694074642826, 0.4554353196219595},
      {0.2591694074642826, 0.0262258654494752, 0.2591694074642826},
      {0.4554353196219595, 0.0262258654494752, 0.2591694074642826},
      {0.2591694074642826, 0.0262258654494752, 0.4554353196219595},
      {0.2591694074642826, 0.2591694074642826, 0.0262258654494752},
      {0.4554353196219595, 0.2591694074642826, 0.0262258654494752},
      {0.2591694074642826, 0.4554353196219595, 0.0262258654494752},
      {0.0109775723659603, 0.5351420009314595, 0.1124285605067612},
      {0.0109775723659603, 0.1124285605067612, 0.5351420009314595},
      {0.5351420009314595, 0.0109775723659603, 0.1124285605067612},
      {0.1124285605067612, 0.0109775723659603, 0.5351420009314595},
      {0.5351420009314595, 0.1124285605067612, 0.0109775723659603},
      {0.1124285605067612, 0.5351420009314595, 0.0109775723659603},
      {0.3414518661958190, 0.5351420009314595, 0.1124285605067612},
      {0.3414518661958190, 0.1124285605067612, 0.5351420009314595},
      {0.3414518661958190, 0.0109775723659603, 0.1124285605067612},
      {0.3414518661958190, 0.0109775723659603, 0.5351420009314595},
      {0.3414518661958190, 0.1124285605067612, 0.0109775723659603},
      {0.3414518661958190, 0.5351420009314595, 0.0109775723659603},
      {0.5351420009314595, 0.3414518661958190, 0.1124285605067612},
      {0.1124285605067612, 0.3414518661958190, 0.5351420009314595},
      {0.0109775723659603, 0.3414518661958190, 0.1124285605067612},
      {0.0109775723659603, 0.3414518661958190, 0.5351420009314595},
      {0.1124285605067612, 0.3414518661958190, 0.0109775723659603},
      {0.5351420009314595, 0.3414518661958190, 0.0109775723659603},
      {0.5351420009314595, 0.1124285605067612, 0.3414518661958190},
      {0.1124285605067612, 0.5351420009314595, 0.3414518661958190},
      {0.0109775723659603, 0.1124285605067612, 0.3414518661958190},
      {0.0109775723659603, 0.5351420009314595, 0.3414518661958190},
      {0.1124285605067612, 0.0109775723659603, 0.3414518661958190},
      {0.5351420009314595, 0.0109775723659603, 0.3414518661958190}},
      {0.0024482239310218,
      0.0024482239310218,
      0.0024482239310218,
      0.0024482239310218,
      0.0005634782551036,
      0.0005634782551036,
      0.0005634782551036,
      0.0005634782551036,
      0.0027944614462043,
      0.0027944614462043,
      0.0027944614462043,
      0.0027944614462043,
      0.0027944614462043,
      0.0027944614462043,
      0.0032206067246264,
      0.0032206067246264,
      0.0032206067246264,
      0.0032206067246264,
      0.0032206067246264,
      0.0032206067246264,
      0.0002755422683960,
      0.0002755422683960,
      0.0002755422683960,
      0.0002755422683960,
      0.0002755422683960,
      0.0002755422683960,
      0.0016333285411721,
      0.0016333285411721,
      0.0016333285411721,
      0.0016333285411721,
      0.0016333285411721,
      0.0016333285411721,
      0.0016333285411721,
      0.0016333285411721,
      0.0016333285411721,
      0.0016333285411721,
      0.0016333285411721,
      0.0016333285411721,
      0.0003393797485508,
      0.0003393797485508,
      0.0003393797485508,
      0.0003393797485508,
      0.0003393797485508,
      0.0003393797485508,
      0.0003393797485508,
      0.0003393797485508,
      0.0003393797485508,
      0.0003393797485508,
      0.0003393797485508,
      0.0003393797485508,
      0.0013679356254861,
      0.0013679356254861,
      0.0013679356254861,
      0.0013679356254861,
      0.0013679356254861,
      0.0013679356254861,
      0.0013679356254861,
      0.0013679356254861,
      0.0013679356254861,
      0.0013679356254861,
      0.0013679356254861,
      0.0013679356254861,
      0.0025174978944439,
      0.0025174978944439,
      0.0025174978944439,
      0.0025174978944439,
      0.0025174978944439,
      0.0025174978944439,
      0.0025174978944439,
      0.0025174978944439,
      0.0025174978944439,
      0.0025174978944439,
      0.0025174978944439,
      0.0025174978944439,
      0.0021237321110177,
      0.0021237321110177,
      0.0021237321110177,
      0.0021237321110177,
      0.0021237321110177,
      0.0021237321110177,
      0.0021237321110177,
      0.0021237321110177,
      0.0021237321110177,
      0.0021237321110177,
      0.0021237321110177,
      0.0021237321110177,
      0.0008789045099482,
      0.0008789045099482,
      0.0008789045099482,
      0.0008789045099482,
      0.0008789045099482,
      0.0008789045099482,
      0.0008789045099482,
      0.0008789045099482,
      0.0008789045099482,
      0.0008789045099482,
      0.0008789045099482,
      0.0008789045099482,
      0.0008789045099482,
      0.0008789045099482,
      0.0008789045099482,
      0.0008789045099482,
      0.0008789045099482,
      0.0008789045099482,
      0.0008789045099482,
      0.0008789045099482,
      0.0008789045099482,
      0.0008789045099482,
      0.0008789045099482,
      0.0008789045099482}
    },
    {
      168,
      {{0.2135679944533018, 0.2135679944533018, 0.2135679944533018},
      {0.3592960166400946, 0.2135679944533018, 0.2135679944533018},
      {0.2135679944533018, 0.3592960166400946, 0.2135679944533018},
      {0.2135679944533018, 0.2135679944533018, 0.3592960166400946},
      {0.0808046995114734, 0.0808046995114734, 0.0808046995114734},
      {0.7575859014655797, 0.0808046995114734, 0.0808046995114734},
      {0.0808046995114734, 0.7575859014655797, 0.0808046995114734},
      {0.0808046995114734, 0.0808046995114734, 0.7575859014655797},
      {0.1460894685275485, 0.1460894685275485, 0.1460894685275485},
      {0.5617315944173544, 0.1460894685275485, 0.1460894685275485},
      {0.1460894685275485, 0.5617315944173544, 0.1460894685275485},
      {0.1460894685275485, 0.1460894685275485, 0.5617315944173544},
      {0.0640653770377989, 0.4359346229622011, 0.4359346229622011},
      {0.4359346229622011, 0.0640653770377989, 0.4359346229622011},
      {0.0640653770377989, 0.0640653770377989, 0.4359346229622011},
      {0.4359346229622011, 0.4359346229622011, 0.0640653770377989},
      {0.0640653770377989, 0.4359346229622011, 0.0640653770377989},
      {0.4359346229622011, 0.0640653770377989, 0.0640653770377989},
      {0.1276183049246017, 0.3723816950753983, 0.3723816950753983},
      {0.3723816950753983, 0.1276183049246017, 0.3723816950753983},
      {0.1276183049246017, 0.1276183049246017, 0.3723816950753983},
      {0.3723816950753983, 0.3723816950753983, 0.1276183049246017},
      {0.1276183049246017, 0.3723816950753983, 0.1276183049246017},
      {0.3723816950753983, 0.1276183049246017, 0.1276183049246017},
      {0.6940351772721454, 0.0148147260674486, 0.0148147260674486},
      {0.0148147260674486, 0.6940351772721454, 0.0148147260674486},
      {0.0148147260674486, 0.0148147260674486, 0.6940351772721454},
      {0.2763353705929573, 0.0148147260674486, 0.0148147260674486},
      {0.2763353705929573, 0.6940351772721454, 0.0148147260674486},
      {0.2763353705929573, 0.0148147260674486, 0.6940351772721454},
      {0.0148147260674486, 0.2763353705929573, 0.0148147260674486},
      {0.6940351772721454, 0.2763353705929573, 0.0148147260674486},
      {0.0148147260674486, 0.2763353705929573, 0.6940351772721454},
      {0.0148147260674486, 0.0148147260674486, 0.2763353705929573},
      {0.6940351772721454, 0.0148147260674486, 0.2763353705929573},
      {0.0148147260674486, 0.6940351772721454, 0.2763353705929573},
      {0.0005787943875725, 0.0440679196756298, 0.0440679196756298},
      {0.0440679196756298, 0.0005787943875725, 0.0440679196756298},
      {0.0440679196756298, 0.0440679196756298, 0.0005787943875725},
      {0.9112853662611680, 0.0440679196756298, 0.0440679196756298},
      {0.9112853662611680, 0.0005787943875725, 0.0440679196756298},
      {0.9112853662611680, 0.0440679196756298, 0.0005787943875725},
      {0.0440679196756298, 0.9112853662611680, 0.0440679196756298},
      {0.0005787943875725, 0.9112853662611680, 0.0440679196756298},
      {0.0440679196756298, 0.9112853662611680, 0.0005787943875725},
      {0.0440679196756298, 0.0440679196756298, 0.9112853662611680},
      {0.0005787943875725, 0.0440679196756298, 0.9112853662611680},
      {0.0440679196756298, 0.0005787943875725, 0.9112853662611680},
      {0.7927838364729656, 0.0290048145551582, 0.0290048145551582},
      {0.0290048145551582, 0.7927838364729656, 0.0290048145551582},
      {0.0290048145551582, 0.0290048145551582, 0.7927838364729656},
      {0.1492065344167181, 0.0290048145551582, 0.0290048145551582},
      {0.1492065344167181, 0.7927838364729656, 0.0290048145551582},
      {0.1492065344167181, 0.0290048145551582, 0.7927838364729656},
      {0.0290048145551582, 0.1492065344167181, 0.0290048145551582},
      {0.7927838364729656, 0.1492065344167181, 0.0290048145551582},
      {0.0290048145551582, 0.1492065344167181, 0.7927838364729656},
      {0.0290048145551582, 0.0290048145551582, 0.1492065344167181},
      {0.7927838364729656, 0.0290048145551582, 0.1492065344167181},
      {0.0290048145551582, 0.7927838364729656, 0.1492065344167181},
      {0.7217318354185758, 0.1384125788015036, 0.1384125788015036},
      {0.1384125788015036, 0.7217318354185758, 0.1384125788015036},
      {0.1384125788015036, 0.1384125788015036, 0.7217318354185758},
      {0.0014430069784170, 0.1384125788015036, 0.1384125788015036},
      {0.0014430069784170, 0.7217318354185758, 0.1384125788015036},
      {0.0014430069784170, 0.1384125788015036, 0.7217318354185758},
      {0.1384125788015036, 0.0014430069784170, 0.1384125788015036},
      {0.7217318354185758, 0.0014430069784170, 0.1384125788015036},
      {0.1384125788015036, 0.0014430069784170, 0.7217318354185758},
      {0.1384125788015036, 0.1384125788015036, 0.0014430069784170},
      {0.7217318354185758, 0.1384125788015036, 0.0014430069784170},
      {0.1384125788015036, 0.7217318354185758, 0.0014430069784170},
      {0.2002685156661767, 0.0115518352710014, 0.3422240474937236},
      {0.2002685156661767, 0.3422240474937236, 0.0115518352710014},
      {0.0115518352710014, 0.2002685156661767, 0.3422240474937236},
      {0.3422240474937236, 0.2002685156661767, 0.0115518352710014},
      {0.0115518352710014, 0.3422240474937236, 0.2002685156661767},
      {0.3422240474937236, 0.0115518352710014, 0.2002685156661767},
      {0.4459556015690982, 0.0115518352710014, 0.3422240474937236},
      {0.4459556015690982, 0.3422240474937236, 0.0115518352710014},
      {0.4459556015690982, 0.2002685156661767, 0.3422240474937236},
      {0.4459556015690982, 0.2002685156661767, 0.0115518352710014},
      {0.4459556015690982, 0.3422240474937236, 0.2002685156661767},
      {0.4459556015690982, 0.0115518352710014, 0.2002685156661767},
      {0.0115518352710014, 0.4459556015690982, 0.3422240474937236},
      {0.3422240474937236, 0.4459556015690982, 0.0115518352710014},
      {0.2002685156661767, 0.4459556015690982, 0.3422240474937236},
      {0.2002685156661767, 0.4459556015690982, 0.0115518352710014},
      {0.3422240474937236, 0.4459556015690982, 0.2002685156661767},
      {0.0115518352710014, 0.4459556015690982, 0.2002685156661767},
      {0.0115518352710014, 0.3422240474937236, 0.4459556015690982},
      {0.3422240474937236, 0.0115518352710014, 0.4459556015690982},
      {0.2002685156661767, 0.3422240474937236, 0.4459556015690982},
      {0.2002685156661767, 0.0115518352710014, 0.4459556015690982},
      {0.3422240474937236, 0.2002685156661767, 0.4459556015690982},
      {0.0115518352710014, 0.2002685156661767, 0.4459556015690982},
      {0.2545920450251540, 0.0729586319508263, 0.2495160363595661},
      {0.2545920450251540, 0.2495160363595661, 0.0729586319508263},
      {0.0729586319508263, 0.2545920450251540, 0.2495160363595661},
      {0.2495160363595661, 0.2545920450251540, 0.0729586319508263},
      {0.0729586319508263, 0.2495160363595661, 0.2545920450251540},
      {0.2495160363595661, 0.0729586319508263, 0.2545920450251540},
      {0.4229332866644536, 0.0729586319508263, 0.2495160363595661},
      {0.4229332866644536, 0.2495160363595661, 0.0729586319508263},
      {0.4229332866644536, 0.2545920450251540, 0.2495160363595661},
      {0.4229332866644536, 0.2545920450251540, 0.0729586319508263},
      {0.4229332866644536, 0.2495160363595661, 0.2545920450251540},
      {0.4229332866644536, 0.0729586319508263, 0.2545920450251540},
      {0.0729586319508263, 0.4229332866644536, 0.2495160363595661},
      {0.2495160363595661, 0.4229332866644536, 0.0729586319508263},
      {0.2545920450251540, 0.4229332866644536, 0.2495160363595661},
      {0.2545920450251540, 0.4229332866644536, 0.0729586319508263},
      {0.2495160363595661, 0.4229332866644536, 0.2545920450251540},
      {0.0729586319508263, 0.4229332866644536, 0.2545920450251540},
      {0.0729586319508263, 0.2495160363595661, 0.4229332866644536},
      {0.2495160363595661, 0.0729586319508263, 0.4229332866644536},
      {0.2545920450251540, 0.2495160363595661, 0.4229332866644536},
      {0.2545920450251540, 0.0729586319508263, 0.4229332866644536},
      {0.2495160363595661, 0.2545920450251540, 0.4229332866644536},
      {0.0729586319508263, 0.2545920450251540, 0.4229332866644536},
      {0.3937338659984053, 0.0500181076151871, 0.0023982547005396},
      {0.3937338659984053, 0.0023982547005396, 0.0500181076151871},
      {0.0500181076151871, 0.3937338659984053, 0.0023982547005396},
      {0.0023982547005396, 0.3937338659984053, 0.0500181076151871},
      {0.0500181076151871, 0.0023982547005396, 0.3937338659984053},
      {0.0023982547005396, 0.0500181076151871, 0.3937338659984053},
      {0.5538497716858680, 0.0500181076151871, 0.0023982547005396},
      {0.5538497716858680, 0.0023982547005396, 0.0500181076151871},
      {0.5538497716858680, 0.3937338659984053, 0.0023982547005396},
      {0.5538497716858680, 0.3937338659984053, 0.0500181076151871},
      {0.5538497716858680, 0.0023982547005396, 0.3937338659984053},
      {0.5538497716858680, 0.0500181076151871, 0.3937338659984053},
      {0.0500181076151871, 0.5538497716858680, 0.0023982547005396},
      {0.0023982547005396, 0.5538497716858680, 0.0500181076151871},
      {0.3937338659984053, 0.5538497716858680, 0.0023982547005396},
      {0.3937338659984053, 0.5538497716858680, 0.0500181076151871},
      {0.0023982547005396, 0.5538497716858680, 0.3937338659984053},
      {0.0500181076151871, 0.5538497716858680, 0.3937338659984053},
      {0.0500181076151871, 0.0023982547005396, 0.5538497716858680},
      {0.0023982547005396, 0.0500181076151871, 0.5538497716858680},
      {0.3937338659984053, 0.0023982547005396, 0.5538497716858680},
      {0.3937338659984053, 0.0500181076151871, 0.5538497716858680},
      {0.0023982547005396, 0.3937338659984053, 0.5538497716858680},
      {0.0500181076151871, 0.3937338659984053, 0.5538497716858680},
      {0.0360873566665786, 0.2518367824271116, 0.1135272318519077},
      {0.0360873566665786, 0.1135272318519077, 0.2518367824271116},
      {0.2518367824271116, 0.0360873566665786, 0.1135272318519077},
      {0.1135272318519077, 0.0360873566665786, 0.2518367824271116},
      {0.2518367824271116, 0.1135272318519077, 0.0360873566665786},
      {0.1135272318519077, 0.2518367824271116, 0.0360873566665786},
      {0.5985486290544021, 0.2518367824271116, 0.1135272318519077},
      {0.5985486290544021, 0.1135272318519077, 0.2518367824271116},
      {0.5985486290544021, 0.0360873566665786, 0.1135272318519077},
      {0.5985486290544021, 0.0360873566665786, 0.2518367824271116},
      {0.5985486290544021, 0.1135272318519077, 0.0360873566665786},
      {0.5985486290544021, 0.2518367824271116, 0.0360873566665786},
      {0.2518367824271116, 0.5985486290544021, 0.1135272318519077},
      {0.1135272318519077, 0.5985486290544021, 0.2518367824271116},
      {0.0360873566665786, 0.5985486290544021, 0.1135272318519077},
      {0.0360873566665786, 0.5985486290544021, 0.2518367824271116},
      {0.1135272318519077, 0.5985486290544021, 0.0360873566665786},
      {0.2518367824271116, 0.5985486290544021, 0.0360873566665786},
      {0.2518367824271116, 0.1135272318519077, 0.5985486290544021},
      {0.1135272318519077, 0.2518367824271116, 0.5985486290544021},
      {0.0360873566665786, 0.1135272318519077, 0.5985486290544021},
      {0.0360873566665786, 0.2518367824271116, 0.5985486290544021},
      {0.1135272318519077, 0.0360873566665786, 0.5985486290544021},
      {0.2518367824271116, 0.0360873566665786, 0.5985486290544021}},
      {0.0029358192558388,
      0.0029358192558388,
      0.0029358192558388,
      0.0029358192558388,
      0.0013918726555721,
      0.0013918726555721,
      0.0013918726555721,
      0.0013918726555721,
      0.0029626198677824,
      0.0029626198677824,
      0.0029626198677824,
      0.0029626198677824,
      0.0016946707329287,
      0.0016946707329287,
      0.0016946707329287,
      0.0016946707329287,
      0.0016946707329287,
      0.0016946707329287,
      0.0026234000039923,
      0.0026234000039923,
      0.0026234000039923,
      0.0026234000039923,
      0.0026234000039923,
      0.0026234000039923,
      0.0002518801909774,
      0.0002518801909774,
      0.0002518801909774,
      0.0002518801909774,
      0.0002518801909774,
      0.0002518801909774,
      0.0002518801909774,
      0.0002518801909774,
      0.0002518801909774,
      0.0002518801909774,
      0.0002518801909774,
      0.0002518801909774,
      0.0001454428314581,
      0.0001454428314581,
      0.0001454428314581,
      0.0001454428314581,
      0.0001454428314581,
      0.0001454428314581,
      0.0001454428314581,
      0.0001454428314581,
      0.0001454428314581,
      0.0001454428314581,
      0.0001454428314581,
      0.0001454428314581,
      0.0006188137769096,
      0.0006188137769096,
      0.0006188137769096,
      0.0006188137769096,
      0.0006188137769096,
      0.0006188137769096,
      0.0006188137769096,
      0.0006188137769096,
      0.0006188137769096,
      0.0006188137769096,
      0.0006188137769096,
      0.0006188137769096,
      0.0003930523759546,
      0.0003930523759546,
      0.0003930523759546,
      0.0003930523759546,
      0.0003930523759546,
      0.0003930523759546,
      0.0003930523759546,
      0.0003930523759546,
      0.0003930523759546,
      0.0003930523759546,
      0.0003930523759546,
      0.0003930523759546,
      0.0007484745060453,
      0.0007484745060453,
      0.0007484745060453,
      0.0007484745060453,
      0.0007484745060453,
      0.0007484745060453,
      0.0007484745060453,
      0.0007484745060453,
      0.0007484745060453,
      0.0007484745060453,
      0.0007484745060453,
      0.0007484745060453,
      0.0007484745060453,
      0.0007484745060453,
      0.0007484745060453,
      0.0007484745060453,
      0.0007484745060453,
      0.0007484745060453,
      0.0007484745060453,
      0.0007484745060453,
      0.0007484745060453,
      0.0007484745060453,
      0.0007484745060453,
      0.0007484745060453,
      0.0012582693414833,
      0.0012582693414833,
      0.0012582693414833,
      0.0012582693414833,
      0.0012582693414833,
      0.0012582693414833,
      0.0012582693414833,
      0.0012582693414833,
      0.0012582693414833,
      0.0012582693414833,
      0.0012582693414833,
      0.0012582693414833,
      0.0012582693414833,
      0.0012582693414833,
      0.0012582693414833,
      0.0012582693414833,
      0.0012582693414833,
      0.0012582693414833,
      0.0012582693414833,
      0.0012582693414833,
      0.0012582693414833,
      0.0012582693414833,
      0.0012582693414833,
      0.0012582693414833,
      0.0003084190981779,
      0.0003084190981779,
      0.0003084190981779,
      0.0003084190981779,
      0.0003084190981779,
      0.0003084190981779,
      0.0003084190981779,
      0.0003084190981779,
      0.0003084190981779,
      0.0003084190981779,
      0.0003084190981779,
      0.0003084190981779,
      0.0003084190981779,
      0.0003084190981779,
      0.0003084190981779,
      0.0003084190981779,
      0.0003084190981779,
      0.0003084190981779,
      0.0003084190981779,
      0.0003084190981779,
      0.0003084190981779,
      0.0003084190981779,
      0.0003084190981779,
      0.0003084190981779,
      0.0016301172636590,
      0.0016301172636590,
      0.0016301172636590,
      0.0016301172636590,
      0.0016301172636590,
      0.0016301172636590,
      0.0016301172636590,
      0.0016301172636590,
      0.0016301172636590,
      0.0016301172636590,
      0.0016301172636590,
      0.0016301172636590,
      0.0016301172636590,
      0.0016301172636590,
      0.0016301172636590,
      0.0016301172636590,
      0.0016301172636590,
      0.0016301172636590,
      0.0016301172636590,
      0.0016301172636590,
      0.0016301172636590,
      0.0016301172636590,
      0.0016301172636590,
      0.0016301172636590}
    },
    {
      172,
      {{0.0202381678618097, 0.0202381678618097, 0.0202381678618097},
      {0.9392854964145708, 0.0202381678618097, 0.0202381678618097},
      {0.0202381678618097, 0.9392854964145708, 0.0202381678618097},
      {0.0202381678618097, 0.0202381678618097, 0.9392854964145708},
      {0.2890147352435263, 0.2890147352435263, 0.2890147352435263},
      {0.1329557942694209, 0.2890147352435263, 0.2890147352435263},
      {0.2890147352435263, 0.1329557942694209, 0.2890147352435263},
      {0.2890147352435263, 0.2890147352435263, 0.1329557942694209},
      {0.0940287000821271, 0.0940287000821271, 0.0940287000821271},
      {0.7179138997536187, 0.0940287000821271, 0.0940287000821271},
      {0.0940287000821271, 0.7179138997536187, 0.0940287000821271},
      {0.0940287000821271, 0.0940287000821271, 0.7179138997536187},
      {0.1976498544437255, 0.1976498544437255, 0.1976498544437255},
      {0.4070504366688234, 0.1976498544437255, 0.1976498544437255},
      {0.1976498544437255, 0.4070504366688234, 0.1976498544437255},
      {0.1976498544437255, 0.1976498544437255, 0.4070504366688234},
      {0.4595381032398164, 0.0404618967601836, 0.0404618967601836},
      {0.0404618967601836, 0.4595381032398164, 0.0404618967601836},
      {0.4595381032398164, 0.4595381032398164, 0.0404618967601836},
      {0.0404618967601836, 0.0404618967601836, 0.4595381032398164},
      {0.4595381032398164, 0.0404618967601836, 0.4595381032398164},
      {0.0404618967601836, 0.4595381032398164, 0.4595381032398164},
      {0.0000001951480306, 0.4999998048519694, 0.4999998048519694},
      {0.4999998048519694, 0.0000001951480306, 0.4999998048519694},
      {0.0000001951480306, 0.0000001951480306, 0.4999998048519694},
      {0.4999998048519694, 0.4999998048519694, 0.0000001951480306},
      {0.0000001951480306, 0.4999998048519694, 0.0000001951480306},
      {0.4999998048519694, 0.0000001951480306, 0.0000001951480306},
      {0.6118971302042935, 0.0675292649528024, 0.0675292649528024},
      {0.0675292649528024, 0.6118971302042935, 0.0675292649528024},
      {0.0675292649528024, 0.0675292649528024, 0.6118971302042935},
      {0.2530443398901017, 0.0675292649528024, 0.0675292649528024},
      {0.2530443398901017, 0.6118971302042935, 0.0675292649528024},
      {0.2530443398901017, 0.0675292649528024, 0.6118971302042935},
      {0.0675292649528024, 0.2530443398901017, 0.0675292649528024},
      {0.6118971302042935, 0.2530443398901017, 0.0675292649528024},
      {0.0675292649528024, 0.2530443398901017, 0.6118971302042935},
      {0.0675292649528024, 0.0675292649528024, 0.2530443398901017},
      {0.6118971302042935, 0.0675292649528024, 0.2530443398901017},
      {0.0675292649528024, 0.6118971302042935, 0.2530443398901017},
      {0.4868308338798159, 0.1360105145132029, 0.1360105145132029},
      {0.1360105145132029, 0.4868308338798159, 0.1360105145132029},
      {0.1360105145132029, 0.1360105145132029, 0.4868308338798159},
      {0.2411481370937782, 0.1360105145132029, 0.1360105145132029},
      {0.2411481370937782, 0.4868308338798159, 0.1360105145132029},
      {0.2411481370937782, 0.1360105145132029, 0.4868308338798159},
      {0.1360105145132029, 0.2411481370937782, 0.1360105145132029},
      {0.4868308338798159, 0.2411481370937782, 0.1360105145132029},
      {0.1360105145132029, 0.2411481370937782, 0.4868308338798159},
      {0.1360105145132029, 0.1360105145132029, 0.2411481370937782},
      {0.4868308338798159, 0.1360105145132029, 0.2411481370937782},
      {0.1360105145132029, 0.4868308338798159, 0.2411481370937782},
      {0.4273786080349000, 0.2780783459563702, 0.2780783459563702},
      {0.2780783459563702, 0.4273786080349000, 0.2780783459563702},
      {0.2780783459563702, 0.2780783459563702, 0.4273786080349000},
      {0.0164647000523597, 0.2780783459563702, 0.2780783459563702},
      {0.0164647000523597, 0.4273786080349000, 0.2780783459563702},
      {0.0164647000523597, 0.2780783459563702, 0.4273786080349000},
      {0.2780783459563702, 0.0164647000523597, 0.2780783459563702},
      {0.4273786080349000, 0.0164647000523597, 0.2780783459563702},
      {0.2780783459563702, 0.0164647000523597, 0.4273786080349000},
      {0.2780783459563702, 0.2780783459563702, 0.0164647000523597},
      {0.4273786080349000, 0.2780783459563702, 0.0164647000523597},
      {0.2780783459563702, 0.4273786080349000, 0.0164647000523597},
      {0.5667528680967985, 0.1967550197192861, 0.1967550197192861},
      {0.1967550197192861, 0.5667528680967985, 0.1967550197192861},
      {0.1967550197192861, 0.1967550197192861, 0.5667528680967985},
      {0.0397370924646294, 0.1967550197192861, 0.1967550197192861},
      {0.0397370924646294, 0.5667528680967985, 0.1967550197192861},
      {0.0397370924646294, 0.1967550197192861, 0.5667528680967985},
      {0.1967550197192861, 0.0397370924646294, 0.1967550197192861},
      {0.5667528680967985, 0.0397370924646294, 0.1967550197192861},
      {0.1967550197192861, 0.0397370924646294, 0.5667528680967985},
      {0.1967550197192861, 0.1967550197192861, 0.0397370924646294},
      {0.5667528680967985, 0.1967550197192861, 0.0397370924646294},
      {0.1967550197192861, 0.5667528680967985, 0.0397370924646294},
      {0.0620197072419754, 0.3862498629203497, 0.3862498629203497},
      {0.3862498629203497, 0.0620197072419754, 0.3862498629203497},
      {0.3862498629203497, 0.3862498629203497, 0.0620197072419754},
      {0.1654805669173252, 0.3862498629203497, 0.3862498629203497},
      {0.1654805669173252, 0.0620197072419754, 0.3862498629203497},
      {0.1654805669173252, 0.3862498629203497, 0.0620197072419754},
      {0.3862498629203497, 0.1654805669173252, 0.3862498629203497},
      {0.0620197072419754, 0.1654805669173252, 0.3862498629203497},
      {0.3862498629203497, 0.1654805669173252, 0.0620197072419754},
      {0.3862498629203497, 0.3862498629203497, 0.1654805669173252},
      {0.0620197072419754, 0.3862498629203497, 0.1654805669173252},
      {0.3862498629203497, 0.0620197072419754, 0.1654805669173252},
      {0.1086607820833199, 0.0235355190546528, 0.0235355190546528},
      {0.0235355190546528, 0.1086607820833199, 0.0235355190546528},
      {0.0235355190546528, 0.0235355190546528, 0.1086607820833199},
      {0.8442681798073746, 0.0235355190546528, 0.0235355190546528},
      {0.8442681798073746, 0.1086607820833199, 0.0235355190546528},
      {0.8442681798073746, 0.0235355190546528, 0.1086607820833199},
      {0.0235355190546528, 0.8442681798073746, 0.0235355190546528},
      {0.1086607820833199, 0.8442681798073746, 0.0235355190546528},
      {0.0235355190546528, 0.8442681798073746, 0.1086607820833199},
      {0.0235355190546528, 0.0235355190546528, 0.8442681798073746},
      {0.1086607820833199, 0.0235355190546528, 0.8442681798073746},
      {0.0235355190546528, 0.1086607820833199, 0.8442681798073746},
      {0.5446964977019162, 0.3362108930747719, 0.0035000598698798},
      {0.5446964977019162, 0.0035000598698798, 0.3362108930747719},
      {0.3362108930747719, 0.5446964977019162, 0.0035000598698798},
      {0.0035000598698798, 0.5446964977019162, 0.3362108930747719},
      {0.3362108930747719, 0.0035000598698798, 0.5446964977019162},
      {0.0035000598698798, 0.3362108930747719, 0.5446964977019162},
      {0.1155925493534321, 0.3362108930747719, 0.0035000598698798},
      {0.1155925493534321, 0.0035000598698798, 0.3362108930747719},
      {0.1155925493534321, 0.5446964977019162, 0.0035000598698798},
      {0.1155925493534321, 0.5446964977019162, 0.3362108930747719},
      {0.1155925493534321, 0.0035000598698798, 0.5446964977019162},
      {0.1155925493534321, 0.3362108930747719, 0.5446964977019162},
      {0.3362108930747719, 0.1155925493534321, 0.0035000598698798},
      {0.0035000598698798, 0.1155925493534321, 0.3362108930747719},
      {0.5446964977019162, 0.1155925493534321, 0.0035000598698798},
      {0.5446964977019162, 0.1155925493534321, 0.3362108930747719},
      {0.0035000598698798, 0.1155925493534321, 0.5446964977019162},
      {0.3362108930747719, 0.1155925493534321, 0.5446964977019162},
      {0.3362108930747719, 0.0035000598698798, 0.1155925493534321},
      {0.0035000598698798, 0.3362108930747719, 0.1155925493534321},
      {0.5446964977019162, 0.0035000598698798, 0.1155925493534321},
      {0.5446964977019162, 0.3362108930747719, 0.1155925493534321},
      {0.0035000598698798, 0.5446964977019162, 0.1155925493534321},
      {0.3362108930747719, 0.5446964977019162, 0.1155925493534321},
      {0.0288616049407555, 0.6859816301235152, 0.0105734346867289},
      {0.0288616049407555, 0.0105734346867289, 0.6859816301235152},
      {0.6859816301235152, 0.0288616049407555, 0.0105734346867289},
      {0.0105734346867289, 0.0288616049407555, 0.6859816301235152},
      {0.6859816301235152, 0.0105734346867289, 0.0288616049407555},
      {0.0105734346867289, 0.6859816301235152, 0.0288616049407555},
      {0.2745833302490004, 0.6859816301235152, 0.0105734346867289},
      {0.2745833302490004, 0.0105734346867289, 0.6859816301235152},
      {0.2745833302490004, 0.0288616049407555, 0.0105734346867289},
      {0.2745833302490004, 0.0288616049407555, 0.6859816301235152},
      {0.2745833302490004, 0.0105734346867289, 0.0288616049407555},
      {0.2745833302490004, 0.6859816301235152, 0.0288616049407555},
      {0.6859816301235152, 0.2745833302490004, 0.0105734346867289},
      {0.0105734346867289, 0.2745833302490004, 0.6859816301235152},
      {0.0288616049407555, 0.2745833302490004, 0.0105734346867289},
      {0.0288616049407555, 0.2745833302490004, 0.6859816301235152},
      {0.0105734346867289, 0.2745833302490004, 0.0288616049407555},
      {0.6859816301235152, 0.2745833302490004, 0.0288616049407555},
      {0.6859816301235152, 0.0105734346867289, 0.2745833302490004},
      {0.0105734346867289, 0.6859816301235152, 0.2745833302490004},
      {0.0288616049407555, 0.0105734346867289, 0.2745833302490004},
      {0.0288616049407555, 0.6859816301235152, 0.2745833302490004},
      {0.0105734346867289, 0.0288616049407555, 0.2745833302490004},
      {0.6859816301235152, 0.0288616049407555, 0.2745833302490004},
      {0.1545866468273804, 0.7356772467818598, 0.0987023928216723},
      {0.1545866468273804, 0.0987023928216723, 0.7356772467818598},
      {0.7356772467818598, 0.1545866468273804, 0.0987023928216723},
      {0.0987023928216723, 0.1545866468273804, 0.7356772467818598},
      {0.7356772467818598, 0.0987023928216723, 0.1545866468273804},
      {0.0987023928216723, 0.7356772467818598, 0.1545866468273804},
      {0.0110337135690874, 0.7356772467818598, 0.0987023928216723},
      {0.0110337135690874, 0.0987023928216723, 0.7356772467818598},
      {0.0110337135690874, 0.1545866468273804, 0.0987023928216723},
      {0.0110337135690874, 0.1545866468273804, 0.7356772467818598},
      {0.0110337135690874, 0.0987023928216723, 0.1545866468273804},
      {0.0110337135690874, 0.7356772467818598, 0.1545866468273804},
      {0.7356772467818598, 0.0110337135690874, 0.0987023928216723},
      {0.0987023928216723, 0.0110337135690874, 0.7356772467818598},
      {0.1545866468273804, 0.0110337135690874, 0.0987023928216723},
      {0.1545866468273804, 0.0110337135690874, 0.7356772467818598},
      {0.0987023928216723, 0.0110337135690874, 0.1545866468273804},
      {0.7356772467818598, 0.0110337135690874, 0.1545866468273804},
      {0.7356772467818598, 0.0987023928216723, 0.0110337135690874},
      {0.0987023928216723, 0.7356772467818598, 0.0110337135690874},
      {0.1545866468273804, 0.0987023928216723, 0.0110337135690874},
      {0.1545866468273804, 0.7356772467818598, 0.0110337135690874},
      {0.0987023928216723, 0.1545866468273804, 0.0110337135690874},
      {0.7356772467818598, 0.1545866468273804, 0.0110337135690874}},
      {0.0001434402964857,
      0.0001434402964857,
      0.0001434402964857,
      0.0001434402964857,
      0.0037678878029565,
      0.0037678878029565,
      0.0037678878029565,
      0.0037678878029565,
      0.0016133395544732,
      0.0016133395544732,
      0.0016133395544732,
      0.0016133395544732,
      0.0011951567209288,
      0.0011951567209288,
      0.0011951567209288,
      0.0011951567209288,
      0.0012974784965467,
      0.0012974784965467,
      0.0012974784965467,
      0.0012974784965467,
      0.0012974784965467,
      0.0012974784965467,
      0.0000853378580286,
      0.0000853378580286,
      0.0000853378580286,
      0.0000853378580286,
      0.0000853378580286,
      0.0000853378580286,
      0.0014800453619038,
      0.0014800453619038,
      0.0014800453619038,
      0.0014800453619038,
      0.0014800453619038,
      0.0014800453619038,
      0.0014800453619038,
      0.0014800453619038,
      0.0014800453619038,
      0.0014800453619038,
      0.0014800453619038,
      0.0014800453619038,
      0.0017702412984156,
      0.0017702412984156,
      0.0017702412984156,
      0.0017702412984156,
      0.0017702412984156,
      0.0017702412984156,
      0.0017702412984156,
      0.0017702412984156,
      0.0017702412984156,
      0.0017702412984156,
      0.0017702412984156,
      0.0017702412984156,
      0.0009788482219993,
      0.0009788482219993,
      0.0009788482219993,
      0.0009788482219993,
      0.0009788482219993,
      0.0009788482219993,
      0.0009788482219993,
      0.0009788482219993,
      0.0009788482219993,
      0.0009788482219993,
      0.0009788482219993,
      0.0009788482219993,
      0.0015392558995927,
      0.0015392558995927,
      0.0015392558995927,
      0.0015392558995927,
      0.0015392558995927,
      0.0015392558995927,
      0.0015392558995927,
      0.0015392558995927,
      0.0015392558995927,
      0.0015392558995927,
      0.0015392558995927,
      0.0015392558995927,
      0.0024278870131942,
      0.0024278870131942,
      0.0024278870131942,
      0.0024278870131942,
      0.0024278870131942,
      0.0024278870131942,
      0.0024278870131942,
      0.0024278870131942,
      0.0024278870131942,
      0.0024278870131942,
      0.0024278870131942,
      0.0024278870131942,
      0.0004431233103622,
      0.0004431233103622,
      0.0004431233103622,
      0.0004431233103622,
      0.0004431233103622,
      0.0004431233103622,
      0.0004431233103622,
      0.0004431233103622,
      0.0004431233103622,
      0.0004431233103622,
      0.0004431233103622,
      0.0004431233103622,
      0.0005369754625003,
      0.0005369754625003,
      0.0005369754625003,
      0.0005369754625003,
      0.0005369754625003,
      0.0005369754625003,
      0.0005369754625003,
      0.0005369754625003,
      0.0005369754625003,
      0.0005369754625003,
      0.0005369754625003,
      0.0005369754625003,
      0.0005369754625003,
      0.0005369754625003,
      0.0005369754625003,
      0.0005369754625003,
      0.0005369754625003,
      0.0005369754625003,
      0.0005369754625003,
      0.0005369754625003,
      0.0005369754625003,
      0.0005369754625003,
      0.0005369754625003,
      0.0005369754625003,
      0.0002440344568507,
      0.0002440344568507,
      0.0002440344568507,
      0.0002440344568507,
      0.0002440344568507,
      0.0002440344568507,
      0.0002440344568507,
      0.0002440344568507,
      0.0002440344568507,
      0.0002440344568507,
      0.0002440344568507,
      0.0002440344568507,
      0.0002440344568507,
      0.0002440344568507,
      0.0002440344568507,
      0.0002440344568507,
      0.0002440344568507,
      0.0002440344568507,
      0.0002440344568507,
      0.0002440344568507,
      0.0002440344568507,
      0.0002440344568507,
      0.0002440344568507,
      0.0002440344568507,
      0.0003780591545750,
      0.0003780591545750,
      0.0003780591545750,
      0.0003780591545750,
      0.0003780591545750,
      0.0003780591545750,
      0.0003780591545750,
      0.0003780591545750,
      0.0003780591545750,
      0.0003780591545750,
      0.0003780591545750,
      0.0003780591545750,
      0.0003780591545750,
      0.0003780591545750,
      0.0003780591545750,
      0.0003780591545750,
      0.0003780591545750,
      0.0003780591545750,
      0.0003780591545750,
      0.0003780591545750,
      0.0003780591545750,
      0.0003780591545750,
      0.0003780591545750,
      0.0003780591545750}
    },
    {
      204,
      {{0.3298151517846193, 0.3298151517846193, 0.3298151517846193},
      {0.0105545446461421, 0.3298151517846193, 0.3298151517846193},
      {0.3298151517846193, 0.0105545446461421, 0.3298151517846193},
      {0.3298151517846193, 0.3298151517846193, 0.0105545446461421},
      {0.0575382826897592, 0.0575382826897592, 0.0575382826897592},
      {0.8273851519307224, 0.0575382826897592, 0.0575382826897592},
      {0.0575382826897592, 0.8273851519307224, 0.0575382826897592},
      {0.0575382826897592, 0.0575382826897592, 0.8273851519307224},
      {0.0058561686137835, 0.0058561686137835, 0.0058561686137835},
      {0.9824314941586495, 0.0058561686137835, 0.0058561686137835},
      {0.0058561686137835, 0.9824314941586495, 0.0058561686137835},
      {0.0058561686137835, 0.0058561686137835, 0.9824314941586495},
      {0.1605554758479567, 0.1605554758479567, 0.1605554758479567},
      {0.5183335724561301, 0.1605554758479567, 0.1605554758479567},
      {0.1605554758479567, 0.5183335724561301, 0.1605554758479567},
      {0.1605554758479567, 0.1605554758479567, 0.5183335724561301},
      {0.0987396460740491, 0.0987396460740491, 0.0987396460740491},
      {0.7037810617778527, 0.0987396460740491, 0.0987396460740491},
      {0.0987396460740491, 0.7037810617778527, 0.0987396460740491},
      {0.0987396460740491, 0.0987396460740491, 0.7037810617778527},
      {0.2080531961597265, 0.2080531961597265, 0.2080531961597265},
      {0.3758404115208205, 0.2080531961597265, 0.2080531961597265},
      {0.2080531961597265, 0.3758404115208205, 0.2080531961597265},
      {0.2080531961597265, 0.2080531961597265, 0.3758404115208205},
      {0.0097520288122235, 0.4902479711877765, 0.4902479711877765},
      {0.4902479711877765, 0.0097520288122235, 0.4902479711877765},
      {0.0097520288122235, 0.0097520288122235, 0.4902479711877765},
      {0.4902479711877765, 0.4902479711877765, 0.0097520288122235},
      {0.0097520288122235, 0.4902479711877765, 0.0097520288122235},
      {0.4902479711877765, 0.0097520288122235, 0.0097520288122235},
      {0.3970605998844674, 0.1029394001155326, 0.1029394001155326},
      {0.1029394001155326, 0.3970605998844674, 0.1029394001155326},
      {0.3970605998844674, 0.3970605998844674, 0.1029394001155326},
      {0.1029394001155326, 0.1029394001155326, 0.3970605998844674},
      {0.3970605998844674, 0.1029394001155326, 0.3970605998844674},
      {0.1029394001155326, 0.3970605998844674, 0.3970605998844674},
      {0.4620865365387848, 0.0379134634612151, 0.0379134634612151},
      {0.0379134634612151, 0.4620865365387848, 0.0379134634612151},
      {0.4620865365387848, 0.4620865365387848, 0.0379134634612151},
      {0.0379134634612151, 0.0379134634612151, 0.4620865365387848},
      {0.4620865365387848, 0.0379134634612151, 0.4620865365387848},
      {0.0379134634612151, 0.4620865365387848, 0.4620865365387848},
      {0.1814264238161396, 0.3185735761838604, 0.3185735761838604},
      {0.3185735761838604, 0.1814264238161396, 0.3185735761838604},
      {0.1814264238161396, 0.1814264238161396, 0.3185735761838604},
      {0.3185735761838604, 0.3185735761838604, 0.1814264238161396},
      {0.1814264238161396, 0.3185735761838604, 0.1814264238161396},
      {0.3185735761838604, 0.1814264238161396, 0.1814264238161396},
      {0.4189387788376669, 0.2499092188634998, 0.2499092188634998},
      {0.2499092188634998, 0.4189387788376669, 0.2499092188634998},
      {0.2499092188634998, 0.2499092188634998, 0.4189387788376669},
      {0.0812427834353334, 0.2499092188634998, 0.2499092188634998},
      {0.0812427834353334, 0.4189387788376669, 0.2499092188634998},
      {0.0812427834353334, 0.2499092188634998, 0.4189387788376669},
      {0.2499092188634998, 0.0812427834353334, 0.2499092188634998},
      {0.4189387788376669, 0.0812427834353334, 0.2499092188634998},
      {0.2499092188634998, 0.0812427834353334, 0.4189387788376669},
      {0.2499092188634998, 0.2499092188634998, 0.0812427834353334},
      {0.4189387788376669, 0.2499092188634998, 0.0812427834353334},
      {0.2499092188634998, 0.4189387788376669, 0.0812427834353334},
      {0.5628766802497838, 0.2132633780618757, 0.2132633780618757},
      {0.2132633780618757, 0.5628766802497838, 0.2132633780618757},
      {0.2132633780618757, 0.2132633780618757, 0.5628766802497838},
      {0.0105965636264648, 0.2132633780618757, 0.2132633780618757},
      {0.0105965636264648, 0.5628766802497838, 0.2132633780618757},
      {0.0105965636264648, 0.2132633780618757, 0.5628766802497838},
      {0.2132633780618757, 0.0105965636264648, 0.2132633780618757},
      {0.5628766802497838, 0.0105965636264648, 0.2132633780618757},
      {0.2132633780618757, 0.0105965636264648, 0.5628766802497838},
      {0.2132633780618757, 0.2132633780618757, 0.0105965636264648},
      {0.5628766802497838, 0.2132633780618757, 0.0105965636264648},
      {0.2132633780618757, 0.5628766802497838, 0.0105965636264648},
      {0.8954689727140162, 0.0491812849401591, 0.0491812849401591},
      {0.0491812849401591, 0.8954689727140162, 0.0491812849401591},
      {0.0491812849401591, 0.0491812849401591, 0.8954689727140162},
      {0.0061684574056657, 0.0491812849401591, 0.0491812849401591},
      {0.0061684574056657, 0.8954689727140162, 0.0491812849401591},
      {0.0061684574056657, 0.0491812849401591, 0.8954689727140162},
      {0.0491812849401591, 0.0061684574056657, 0.0491812849401591},
      {0.8954689727140162, 0.0061684574056657, 0.0491812849401591},
      {0.0491812849401591, 0.0061684574056657, 0.8954689727140162},
      {0.0491812849401591, 0.0491812849401591, 0.0061684574056657},
      {0.8954689727140162, 0.0491812849401591, 0.0061684574056657},
      {0.0491812849401591, 0.8954689727140162, 0.0061684574056657},
      {0.0206635789296735, 0.3928626179700601, 0.3928626179700601},
      {0.3928626179700601, 0.0206635789296735, 0.3928626179700601},
      {0.3928626179700601, 0.3928626179700601, 0.0206635789296735},
      {0.1936111851302063, 0.3928626179700601, 0.3928626179700601},
      {0.1936111851302063, 0.0206635789296735, 0.3928626179700601},
      {0.1936111851302063, 0.3928626179700601, 0.0206635789296735},
      {0.3928626179700601, 0.1936111851302063, 0.3928626179700601},
      {0.0206635789296735, 0.1936111851302063, 0.3928626179700601},
      {0.3928626179700601, 0.1936111851302063, 0.0206635789296735},
      {0.3928626179700601, 0.3928626179700601, 0.1936111851302063},
      {0.0206635789296735, 0.3928626179700601, 0.1936111851302063},
      {0.3928626179700601, 0.0206635789296735, 0.1936111851302063},
      {0.1269022024074958, 0.0126176055325707, 0.0126176055325707},
      {0.0126176055325707, 0.1269022024074958, 0.0126176055325707},
      {0.0126176055325707, 0.0126176055325707, 0.1269022024074958},
      {0.8478625865273628, 0.0126176055325707, 0.0126176055325707},
      {0.8478625865273628, 0.1269022024074958, 0.0126176055325707},
      {0.8478625865273628, 0.0126176055325707, 0.1269022024074958},
      {0.0126176055325707, 0.8478625865273628, 0.0126176055325707},
      {0.1269022024074958, 0.8478625865273628, 0.0126176055325707},
      {0.0126176055325707, 0.8478625865273628, 0.1269022024074958},
      {0.0126176055325707, 0.0126176055325707, 0.8478625865273628},
      {0.1269022024074958, 0.0126176055325707, 0.8478625865273628},
      {0.0126176055325707, 0.1269022024074958, 0.8478625865273628},
      {0.0621400031176215, 0.1216187843486990, 0.5767406174550948},
      {0.0621400031176215, 0.5767406174550948, 0.1216187843486990},
      {0.1216187843486990, 0.0621400031176215, 0.5767406174550948},
      {0.5767406174550948, 0.0621400031176215, 0.1216187843486990},
      {0.1216187843486990, 0.5767406174550948, 0.0621400031176215},
      {0.5767406174550948, 0.1216187843486990, 0.0621400031176215},
      {0.2395005950785847, 0.1216187843486990, 0.5767406174550948},
      {0.2395005950785847, 0.5767406174550948, 0.1216187843486990},
      {0.2395005950785847, 0.0621400031176215, 0.5767406174550948},
      {0.2395005950785847, 0.0621400031176215, 0.1216187843486990},
      {0.2395005950785847, 0.5767406174550948, 0.0621400031176215},
      {0.2395005950785847, 0.1216187843486990, 0.0621400031176215},
      {0.1216187843486990, 0.2395005950785847, 0.5767406174550948},
      {0.5767406174550948, 0.2395005950785847, 0.1216187843486990},
      {0.0621400031176215, 0.2395005950785847, 0.5767406174550948},
      {0.0621400031176215, 0.2395005950785847, 0.1216187843486990},
      {0.5767406174550948, 0.2395005950785847, 0.0621400031176215},
      {0.1216187843486990, 0.2395005950785847, 0.0621400031176215},
      {0.1216187843486990, 0.5767406174550948, 0.2395005950785847},
      {0.5767406174550948, 0.1216187843486990, 0.2395005950785847},
      {0.0621400031176215, 0.5767406174550948, 0.2395005950785847},
      {0.0621400031176215, 0.1216187843486990, 0.2395005950785847},
      {0.5767406174550948, 0.0621400031176215, 0.2395005950785847},
      {0.1216187843486990, 0.0621400031176215, 0.2395005950785847},
      {0.5517941321184247, 0.3489022960470246, 0.0155221192232079},
      {0.5517941321184247, 0.0155221192232079, 0.3489022960470246},
      {0.3489022960470246, 0.5517941321184247, 0.0155221192232079},
      {0.0155221192232079, 0.5517941321184247, 0.3489022960470246},
      {0.3489022960470246, 0.0155221192232079, 0.5517941321184247},
      {0.0155221192232079, 0.3489022960470246, 0.5517941321184247},
      {0.0837814526113429, 0.3489022960470246, 0.0155221192232079},
      {0.0837814526113429, 0.0155221192232079, 0.3489022960470246},
      {0.0837814526113429, 0.5517941321184247, 0.0155221192232079},
      {0.0837814526113429, 0.5517941321184247, 0.3489022960470246},
      {0.0837814526113429, 0.0155221192232079, 0.5517941321184247},
      {0.0837814526113429, 0.3489022960470246, 0.5517941321184247},
      {0.3489022960470246, 0.0837814526113429, 0.0155221192232079},
      {0.0155221192232079, 0.0837814526113429, 0.3489022960470246},
      {0.5517941321184247, 0.0837814526113429, 0.0155221192232079},
      {0.5517941321184247, 0.0837814526113429, 0.3489022960470246},
      {0.0155221192232079, 0.0837814526113429, 0.5517941321184247},
      {0.3489022960470246, 0.0837814526113429, 0.5517941321184247},
      {0.3489022960470246, 0.0155221192232079, 0.0837814526113429},
      {0.0155221192232079, 0.3489022960470246, 0.0837814526113429},
      {0.5517941321184247, 0.0155221192232079, 0.0837814526113429},
      {0.5517941321184247, 0.3489022960470246, 0.0837814526113429},
      {0.0155221192232079, 0.5517941321184247, 0.0837814526113429},
      {0.3489022960470246, 0.5517941321184247, 0.0837814526113429},
      {0.0178416406476546, 0.6822383201682893, 0.0154792973131725},
      {0.0178416406476546, 0.0154792973131725, 0.6822383201682893},
      {0.6822383201682893, 0.0178416406476546, 0.0154792973131725},
      {0.0154792973131725, 0.0178416406476546, 0.6822383201682893},
      {0.6822383201682893, 0.0154792973131725, 0.0178416406476546},
      {0.0154792973131725, 0.6822383201682893, 0.0178416406476546},
      {0.2844407418708836, 0.6822383201682893, 0.0154792973131725},
      {0.2844407418708836, 0.0154792973131725, 0.6822383201682893},
      {0.2844407418708836, 0.0178416406476546, 0.0154792973131725},
      {0.2844407418708836, 0.0178416406476546, 0.6822383201682893},
      {0.2844407418708836, 0.0154792973131725, 0.0178416406476546},
      {0.2844407418708836, 0.6822383201682893, 0.0178416406476546},
      {0.6822383201682893, 0.2844407418708836, 0.0154792973131725},
      {0.0154792973131725, 0.2844407418708836, 0.6822383201682893},
      {0.0178416406476546, 0.2844407418708836, 0.0154792973131725},
      {0.0178416406476546, 0.2844407418708836, 0.6822383201682893},
      {0.0154792973131725, 0.2844407418708836, 0.0178416406476546},
      {0.6822383201682893, 0.2844407418708836, 0.0178416406476546},
      {0.6822383201682893, 0.0154792973131725, 0.2844407418708836},
      {0.0154792973131725, 0.6822383201682893, 0.2844407418708836},
      {0.0178416406476546, 0.0154792973131725, 0.2844407418708836},
      {0.0178416406476546, 0.6822383201682893, 0.2844407418708836},
      {0.0154792973131725, 0.0178416406476546, 0.2844407418708836},
      {0.6822383201682893, 0.0178416406476546, 0.2844407418708836},
      {0.1695395507001622, 0.7339982986585810, 0.0177766027701428},
      {0.1695395507001622, 0.0177766027701428, 0.7339982986585810},
      {0.7339982986585810, 0.1695395507001622, 0.0177766027701428},
      {0.0177766027701428, 0.1695395507001622, 0.7339982986585810},
      {0.7339982986585810, 0.0177766027701428, 0.1695395507001622},
      {0.0177766027701428, 0.7339982986585810, 0.1695395507001622},
      {0.0786855478711139, 0.7339982986585810, 0.0177766027701428},
      {0.0786855478711139, 0.0177766027701428, 0.7339982986585810},
      {0.0786855478711139, 0.1695395507001622, 0.0177766027701428},
      {0.0786855478711139, 0.1695395507001622, 0.7339982986585810},
      {0.0786855478711139, 0.0177766027701428, 0.1695395507001622},
      {0.0786855478711139, 0.7339982986585810, 0.1695395507001622},
      {0.7339982986585810, 0.0786855478711139, 0.0177766027701428},
      {0.0177766027701428, 0.0786855478711139, 0.7339982986585810},
      {0.1695395507001622, 0.0786855478711139, 0.0177766027701428},
      {0.1695395507001622, 0.0786855478711139, 0.7339982986585810},
      {0.0177766027701428, 0.0786855478711139, 0.1695395507001622},
      {0.7339982986585810, 0.0786855478711139, 0.1695395507001622},
      {0.7339982986585810, 0.0177766027701428, 0.0786855478711139},
      {0.0177766027701428, 0.7339982986585810, 0.0786855478711139},
      {0.1695395507001622, 0.0177766027701428, 0.0786855478711139},
      {0.1695395507001622, 0.7339982986585810, 0.0786855478711139},
      {0.0177766027701428, 0.1695395507001622, 0.0786855478711139},
      {0.7339982986585810, 0.1695395507001622, 0.0786855478711139}},
      {0.0005742388417206,
      0.0005742388417206,
      0.0005742388417206,
      0.0005742388417206,
      0.0004368758786451,
      0.0004368758786451,
      0.0004368758786451,
      0.0004368758786451,
      0.0000184563873649,
      0.0000184563873649,
      0.0000184563873649,
      0.0000184563873649,
      0.0016714308954123,
      0.0016714308954123,
      0.0016714308954123,
      0.0016714308954123,
      0.0011317347429682,
      0.0011317347429682,
      0.0011317347429682,
      0.0011317347429682,
      0.0011863580569832,
      0.0011863580569832,
      0.0011863580569832,
      0.0011863580569832,
      0.0001766753824273,
      0.0001766753824273,
      0.0001766753824273,
      0.0001766753824273,
      0.0001766753824273,
      0.0001766753824273,
      0.0022783051658301,
      0.0022783051658301,
      0.0022783051658301,
      0.0022783051658301,
      0.0022783051658301,
      0.0022783051658301,
      0.0004484568491127,
      0.0004484568491127,
      0.0004484568491127,
      0.0004484568491127,
      0.0004484568491127,
      0.0004484568491127,
      0.0016941002043071,
      0.0016941002043071,
      0.0016941002043071,
      0.0016941002043071,
      0.0016941002043071,
      0.0016941002043071,
      0.0022573999396212,
      0.0022573999396212,
      0.0022573999396212,
      0.0022573999396212,
      0.0022573999396212,
      0.0022573999396212,
      0.0022573999396212,
      0.0022573999396212,
      0.0022573999396212,
      0.0022573999396212,
      0.0022573999396212,
      0.0022573999396212,
      0.0007705566746116,
      0.0007705566746116,
      0.0007705566746116,
      0.0007705566746116,
      0.0007705566746116,
      0.0007705566746116,
      0.0007705566746116,
      0.0007705566746116,
      0.0007705566746116,
      0.0007705566746116,
      0.0007705566746116,
      0.0007705566746116,
      0.0001388583075412,
      0.0001388583075412,
      0.0001388583075412,
      0.0001388583075412,
      0.0001388583075412,
      0.0001388583075412,
      0.0001388583075412,
      0.0001388583075412,
      0.0001388583075412,
      0.0001388583075412,
      0.0001388583075412,
      0.0001388583075412,
      0.0012067161268095,
      0.0012067161268095,
      0.0012067161268095,
      0.0012067161268095,
      0.0012067161268095,
      0.0012067161268095,
      0.0012067161268095,
      0.0012067161268095,
      0.0012067161268095,
      0.0012067161268095,
      0.0012067161268095,
      0.0012067161268095,
      0.0001458634634996,
      0.0001458634634996,
      0.0001458634634996,
      0.0001458634634996,
      0.0001458634634996,
      0.0001458634634996,
      0.0001458634634996,
      0.0001458634634996,
      0.0001458634634996,
      0.0001458634634996,
      0.0001458634634996,
      0.0001458634634996,
      0.0011954227530164,
      0.0011954227530164,
      0.0011954227530164,
      0.0011954227530164,
      0.0011954227530164,
      0.0011954227530164,
      0.0011954227530164,
      0.0011954227530164,
      0.0011954227530164,
      0.0011954227530164,
      0.0011954227530164,
      0.0011954227530164,
      0.0011954227530164,
      0.0011954227530164,
      0.0011954227530164,
      0.0011954227530164,
      0.0011954227530164,
      0.0011954227530164,
      0.0011954227530164,
      0.0011954227530164,
      0.0011954227530164,
      0.0011954227530164,
      0.0011954227530164,
      0.0011954227530164,
      0.0007144760571318,
      0.0007144760571318,
      0.0007144760571318,
      0.0007144760571318,
      0.0007144760571318,
      0.0007144760571318,
      0.0007144760571318,
      0.0007144760571318,
      0.0007144760571318,
      0.0007144760571318,
      0.0007144760571318,
      0.0007144760571318,
      0.0007144760571318,
      0.0007144760571318,
      0.0007144760571318,
      0.0007144760571318,
      0.0007144760571318,
      0.0007144760571318,
      0.0007144760571318,
      0.0007144760571318,
      0.0007144760571318,
      0.0007144760571318,
      0.0007144760571318,
      0.0007144760571318,
      0.0001626254606918,
      0.0001626254606918,
      0.0001626254606918,
      0.0001626254606918,
      0.0001626254606918,
      0.0001626254606918,
      0.0001626254606918,
      0.0001626254606918,
      0.0001626254606918,
      0.0001626254606918,
      0.0001626254606918,
      0.0001626254606918,
      0.0001626254606918,
      0.0001626254606918,
      0.0001626254606918,
      0.0001626254606918,
      0.0001626254606918,
      0.0001626254606918,
      0.0001626254606918,
      0.0001626254606918,
      0.0001626254606918,
      0.0001626254606918,
      0.0001626254606918,
      0.0001626254606918,
      0.0006263227166278,
      0.0006263227166278,
      0.0006263227166278,
      0.0006263227166278,
      0.0006263227166278,
      0.0006263227166278,
      0.0006263227166278,
      0.0006263227166278,
      0.0006263227166278,
      0.0006263227166278,
      0.0006263227166278,
      0.0006263227166278,
      0.0006263227166278,
      0.0006263227166278,
      0.0006263227166278,
      0.0006263227166278,
      0.0006263227166278,
      0.0006263227166278,
      0.0006263227166278,
      0.0006263227166278,
      0.0006263227166278,
      0.0006263227166278,
      0.0006263227166278,
      0.0006263227166278}
    },
    {
      264,
      {{0.0169264215854722, 0.0169264215854722, 0.0169264215854722},
      {0.9492207352435834, 0.0169264215854722, 0.0169264215854722},
      {0.0169264215854722, 0.9492207352435834, 0.0169264215854722},
      {0.0169264215854722, 0.0169264215854722, 0.9492207352435834},
      {0.2836779254595722, 0.2836779254595722, 0.2836779254595722},
      {0.1489662236212833, 0.2836779254595722, 0.2836779254595722},
      {0.2836779254595722, 0.1489662236212833, 0.2836779254595722},
      {0.2836779254595722, 0.2836779254595722, 0.1489662236212833},
      {0.1821984139975859, 0.1821984139975859, 0.1821984139975859},
      {0.4534047580072424, 0.1821984139975859, 0.1821984139975859},
      {0.1821984139975859, 0.4534047580072424, 0.1821984139975859},
      {0.1821984139975859, 0.1821984139975859, 0.4534047580072424},
      {0.4854919948478154, 0.0145080051521846, 0.0145080051521846},
      {0.0145080051521846, 0.4854919948478154, 0.0145080051521846},
      {0.4854919948478154, 0.4854919948478154, 0.0145080051521846},
      {0.0145080051521846, 0.0145080051521846, 0.4854919948478154},
      {0.4854919948478154, 0.0145080051521846, 0.4854919948478154},
      {0.0145080051521846, 0.4854919948478154, 0.4854919948478154},
      {0.3572558562341436, 0.1427441437658564, 0.1427441437658564},
      {0.1427441437658564, 0.3572558562341436, 0.1427441437658564},
      {0.3572558562341436, 0.3572558562341436, 0.1427441437658564},
      {0.1427441437658564, 0.1427441437658564, 0.3572558562341436},
      {0.3572558562341436, 0.1427441437658564, 0.3572558562341436},
      {0.1427441437658564, 0.3572558562341436, 0.3572558562341436},
      {0.7766357493733004, 0.0732172725619535, 0.0732172725619535},
      {0.0732172725619535, 0.7766357493733004, 0.0732172725619535},
      {0.0732172725619535, 0.0732172725619535, 0.7766357493733004},
      {0.0769297055027926, 0.0732172725619535, 0.0732172725619535},
      {0.0769297055027926, 0.7766357493733004, 0.0732172725619535},
      {0.0769297055027926, 0.0732172725619535, 0.7766357493733004},
      {0.0732172725619535, 0.0769297055027926, 0.0732172725619535},
      {0.7766357493733004, 0.0769297055027926, 0.0732172725619535},
      {0.0732172725619535, 0.0769297055027926, 0.7766357493733004},
      {0.0732172725619535, 0.0732172725619535, 0.0769297055027926},
      {0.7766357493733004, 0.0732172725619535, 0.0769297055027926},
      {0.0732172725619535, 0.7766357493733004, 0.0769297055027926},
      {0.2225606554924344, 0.0105655841048970, 0.0105655841048970},
      {0.0105655841048970, 0.2225606554924344, 0.0105655841048970},
      {0.0105655841048970, 0.0105655841048970, 0.2225606554924344},
      {0.7563081762977716, 0.0105655841048970, 0.0105655841048970},
      {0.7563081762977716, 0.2225606554924344, 0.0105655841048970},
      {0.7563081762977716, 0.0105655841048970, 0.2225606554924344},
      {0.0105655841048970, 0.7563081762977716, 0.0105655841048970},
      {0.2225606554924344, 0.7563081762977716, 0.0105655841048970},
      {0.0105655841048970, 0.7563081762977716, 0.2225606554924344},
      {0.0105655841048970, 0.0105655841048970, 0.7563081762977716},
      {0.2225606554924344, 0.0105655841048970, 0.7563081762977716},
      {0.0105655841048970, 0.2225606554924344, 0.7563081762977716},
      {0.0557502224056597, 0.2637972626688146, 0.2637972626688146},
      {0.2637972626688146, 0.0557502224056597, 0.2637972626688146},
      {0.2637972626688146, 0.2637972626688146, 0.0557502224056597},
      {0.4166552522567110, 0.2637972626688146, 0.2637972626688146},
      {0.4166552522567110, 0.0557502224056597, 0.2637972626688146},
      {0.4166552522567110, 0.2637972626688146, 0.0557502224056597},
      {0.2637972626688146, 0.4166552522567110, 0.2637972626688146},
      {0.0557502224056597, 0.4166552522567110, 0.2637972626688146},
      {0.2637972626688146, 0.4166552522567110, 0.0557502224056597},
      {0.2637972626688146, 0.2637972626688146, 0.4166552522567110},
      {0.0557502224056597, 0.2637972626688146, 0.4166552522567110},
      {0.2637972626688146, 0.0557502224056597, 0.4166552522567110},
      {0.1112909267055329, 0.4354902702993817, 0.4354902702993817},
      {0.4354902702993817, 0.1112909267055329, 0.4354902702993817},
      {0.4354902702993817, 0.4354902702993817, 0.1112909267055329},
      {0.0177285326957037, 0.4354902702993817, 0.4354902702993817},
      {0.0177285326957037, 0.1112909267055329, 0.4354902702993817},
      {0.0177285326957037, 0.4354902702993817, 0.1112909267055329},
      {0.4354902702993817, 0.0177285326957037, 0.4354902702993817},
      {0.1112909267055329, 0.0177285326957037, 0.4354902702993817},
      {0.4354902702993817, 0.0177285326957037, 0.1112909267055329},
      {0.4354902702993817, 0.4354902702993817, 0.0177285326957037},
      {0.1112909267055329, 0.4354902702993817, 0.0177285326957037},
      {0.4354902702993817, 0.1112909267055329, 0.0177285326957037},
      {0.5448865322975603, 0.0534725936418519, 0.0534725936418519},
      {0.0534725936418519, 0.5448865322975603, 0.0534725936418519},
      {0.0534725936418519, 0.0534725936418519, 0.5448865322975603},
      {0.3481682804187358, 0.0534725936418519, 0.0534725936418519},
      {0.3481682804187358, 0.5448865322975603, 0.0534725936418519},
      {0.3481682804187358, 0.0534725936418519, 0.5448865322975603},
      {0.0534725936418519, 0.3481682804187358, 0.0534725936418519},
      {0.5448865322975603, 0.3481682804187358, 0.0534725936418519},
      {0.0534725936418519, 0.3481682804187358, 0.5448865322975603},
      {0.0534725936418519, 0.0534725936418519, 0.3481682804187358},
      {0.5448865322975603, 0.0534725936418519, 0.3481682804187358},
      {0.0534725936418519, 0.5448865322975603, 0.3481682804187358},
      {0.7694703779405406, 0.1105308939580100, 0.1105308939580100},
      {0.1105308939580100, 0.7694703779405406, 0.1105308939580100},
      {0.1105308939580100, 0.1105308939580100, 0.7694703779405406},
      {0.0094678341434394, 0.1105308939580100, 0.1105308939580100},
      {0.0094678341434394, 0.7694703779405406, 0.1105308939580100},
      {0.0094678341434394, 0.1105308939580100, 0.7694703779405406},
      {0.1105308939580100, 0.0094678341434394, 0.1105308939580100},
      {0.7694703779405406, 0.0094678341434394, 0.1105308939580100},
      {0.1105308939580100, 0.0094678341434394, 0.7694703779405406},
      {0.1105308939580100, 0.1105308939580100, 0.0094678341434394},
      {0.7694703779405406, 0.1105308939580100, 0.0094678341434394},
      {0.1105308939580100, 0.7694703779405406, 0.0094678341434394},
      {0.6030936440770986, 0.0410467051499108, 0.3535307263575092},
      {0.6030936440770986, 0.3535307263575092, 0.0410467051499108},
      {0.0410467051499108, 0.6030936440770986, 0.3535307263575092},
      {0.3535307263575092, 0.6030936440770986, 0.0410467051499108},
      {0.0410467051499108, 0.3535307263575092, 0.6030936440770986},
      {0.3535307263575092, 0.0410467051499108, 0.6030936440770986},
      {0.0023289244154814, 0.0410467051499108, 0.3535307263575092},
      {0.0023289244154814, 0.3535307263575092, 0.0410467051499108},
      {0.0023289244154814, 0.6030936440770986, 0.3535307263575092},
      {0.0023289244154814, 0.6030936440770986, 0.0410467051499108},
      {0.0023289244154814, 0.3535307263575092, 0.6030936440770986},
      {0.0023289244154814, 0.0410467051499108, 0.6030936440770986},
      {0.0410467051499108, 0.0023289244154814, 0.3535307263575092},
      {0.3535307263575092, 0.0023289244154814, 0.0410467051499108},
      {0.6030936440770986, 0.0023289244154814, 0.3535307263575092},
      {0.6030936440770986, 0.0023289244154814, 0.0410467051499108},
      {0.3535307263575092, 0.0023289244154814, 0.6030936440770986},
      {0.0410467051499108, 0.0023289244154814, 0.6030936440770986},
      {0.0410467051499108, 0.3535307263575092, 0.0023289244154814},
      {0.3535307263575092, 0.0410467051499108, 0.0023289244154814},
      {0.6030936440770986, 0.3535307263575092, 0.0023289244154814},
      {0.6030936440770986, 0.0410467051499108, 0.0023289244154814},
      {0.3535307263575092, 0.6030936440770986, 0.0023289244154814},
      {0.0410467051499108, 0.6030936440770986, 0.0023289244154814},
      {0.0293931933291708, 0.0920166732769526, 0.0113255282696951},
      {0.0293931933291708, 0.0113255282696951, 0.0920166732769526},
      {0.0920166732769526, 0.0293931933291708, 0.0113255282696951},
      {0.0113255282696951, 0.0293931933291708, 0.0920166732769526},
      {0.0920166732769526, 0.0113255282696951, 0.0293931933291708},
      {0.0113255282696951, 0.0920166732769526, 0.0293931933291708},
      {0.8672646051241815, 0.0920166732769526, 0.0113255282696951},
      {0.8672646051241815, 0.0113255282696951, 0.0920166732769526},
      {0.8672646051241815, 0.0293931933291708, 0.0113255282696951},
      {0.8672646051241815, 0.0293931933291708, 0.0920166732769526},
      {0.8672646051241815, 0.0113255282696951, 0.0293931933291708},
      {0.8672646051241815, 0.0920166732769526, 0.0293931933291708},
      {0.0920166732769526, 0.8672646051241815, 0.0113255282696951},
      {0.0113255282696951, 0.8672646051241815, 0.0920166732769526},
      {0.0293931933291708, 0.8672646051241815, 0.0113255282696951},
      {0.0293931933291708, 0.8672646051241815, 0.0920166732769526},
      {0.0113255282696951, 0.8672646051241815, 0.0293931933291708},
      {0.0920166732769526, 0.8672646051241815, 0.0293931933291708},
      {0.0920166732769526, 0.0113255282696951, 0.8672646051241815},
      {0.0113255282696951, 0.0920166732769526, 0.8672646051241815},
      {0.0293931933291708, 0.0113255282696951, 0.8672646051241815},
      {0.0293931933291708, 0.0920166732769526, 0.8672646051241815},
      {0.0113255282696951, 0.0293931933291708, 0.8672646051241815},
      {0.0920166732769526, 0.0293931933291708, 0.8672646051241815},
      {0.7077814873155971, 0.0682262786800005, 0.2019857964457858},
      {0.7077814873155971, 0.2019857964457858, 0.0682262786800005},
      {0.0682262786800005, 0.7077814873155971, 0.2019857964457858},
      {0.2019857964457858, 0.7077814873155971, 0.0682262786800005},
      {0.0682262786800005, 0.2019857964457858, 0.7077814873155971},
      {0.2019857964457858, 0.0682262786800005, 0.7077814873155971},
      {0.0220064375586166, 0.0682262786800005, 0.2019857964457858},
      {0.0220064375586166, 0.2019857964457858, 0.0682262786800005},
      {0.0220064375586166, 0.7077814873155971, 0.2019857964457858},
      {0.0220064375586166, 0.7077814873155971, 0.0682262786800005},
      {0.0220064375586166, 0.2019857964457858, 0.7077814873155971},
      {0.0220064375586166, 0.0682262786800005, 0.7077814873155971},
      {0.0682262786800005, 0.0220064375586166, 0.2019857964457858},
      {0.2019857964457858, 0.0220064375586166, 0.0682262786800005},
      {0.7077814873155971, 0.0220064375586166, 0.2019857964457858},
      {0.7077814873155971, 0.0220064375586166, 0.0682262786800005},
      {0.2019857964457858, 0.0220064375586166, 0.7077814873155971},
      {0.0682262786800005, 0.0220064375586166, 0.7077814873155971},
      {0.0682262786800005, 0.2019857964457858, 0.0220064375586166},
      {0.2019857964457858, 0.0682262786800005, 0.0220064375586166},
      {0.7077814873155971, 0.2019857964457858, 0.0220064375586166},
      {0.7077814873155971, 0.0682262786800005, 0.0220064375586166},
      {0.2019857964457858, 0.7077814873155971, 0.0220064375586166},
      {0.0682262786800005, 0.7077814873155971, 0.0220064375586166},
      {0.1295809597674534, 0.1890015556386176, 0.6133421548059226},
      {0.1295809597674534, 0.6133421548059226, 0.1890015556386176},
      {0.1890015556386176, 0.1295809597674534, 0.6133421548059226},
      {0.6133421548059226, 0.1295809597674534, 0.1890015556386176},
      {0.1890015556386176, 0.6133421548059226, 0.1295809597674534},
      {0.6133421548059226, 0.1890015556386176, 0.1295809597674534},
      {0.0680753297880064, 0.1890015556386176, 0.6133421548059226},
      {0.0680753297880064, 0.6133421548059226, 0.1890015556386176},
      {0.0680753297880064, 0.1295809597674534, 0.6133421548059226},
      {0.0680753297880064, 0.1295809597674534, 0.1890015556386176},
      {0.0680753297880064, 0.6133421548059226, 0.1295809597674534},
      {0.0680753297880064, 0.1890015556386176, 0.1295809597674534},
      {0.1890015556386176, 0.0680753297880064, 0.6133421548059226},
      {0.6133421548059226, 0.0680753297880064, 0.1890015556386176},
      {0.1295809597674534, 0.0680753297880064, 0.6133421548059226},
      {0.1295809597674534, 0.0680753297880064, 0.1890015556386176},
      {0.6133421548059226, 0.0680753297880064, 0.1295809597674534},
      {0.1890015556386176, 0.0680753297880064, 0.1295809597674534},
      {0.1890015556386176, 0.6133421548059226, 0.0680753297880064},
      {0.6133421548059226, 0.1890015556386176, 0.0680753297880064},
      {0.1295809597674534, 0.6133421548059226, 0.0680753297880064},
      {0.1295809597674534, 0.1890015556386176, 0.0680753297880064},
      {0.6133421548059226, 0.1295809597674534, 0.0680753297880064},
      {0.1890015556386176, 0.1295809597674534, 0.0680753297880064},
      {0.4695333003662676, 0.3054799663906012, 0.1524718900094398},
      {0.4695333003662676, 0.1524718900094398, 0.3054799663906012},
      {0.3054799663906012, 0.4695333003662676, 0.1524718900094398},
      {0.1524718900094398, 0.4695333003662676, 0.3054799663906012},
      {0.3054799663906012, 0.1524718900094398, 0.4695333003662676},
      {0.1524718900094398, 0.3054799663906012, 0.4695333003662676},
      {0.0725148432336915, 0.3054799663906012, 0.1524718900094398},
      {0.0725148432336915, 0.1524718900094398, 0.3054799663906012},
      {0.0725148432336915, 0.4695333003662676, 0.1524718900094398},
      {0.0725148432336915, 0.4695333003662676, 0.3054799663906012},
      {0.0725148432336915, 0.1524718900094398, 0.4695333003662676},
      {0.0725148432336915, 0.3054799663906012, 0.4695333003662676},
      {0.3054799663906012, 0.0725148432336915, 0.1524718900094398},
      {0.1524718900094398, 0.0725148432336915, 0.3054799663906012},
      {0.4695333003662676, 0.0725148432336915, 0.1524718900094398},
      {0.4695333003662676, 0.0725148432336915, 0.3054799663906012},
      {0.1524718900094398, 0.0725148432336915, 0.4695333003662676},
      {0.3054799663906012, 0.0725148432336915, 0.4695333003662676},
      {0.3054799663906012, 0.1524718900094398, 0.0725148432336915},
      {0.1524718900094398, 0.3054799663906012, 0.0725148432336915},
      {0.4695333003662676, 0.1524718900094398, 0.0725148432336915},
      {0.4695333003662676, 0.3054799663906012, 0.0725148432336915},
      {0.1524718900094398, 0.4695333003662676, 0.0725148432336915},
      {0.3054799663906012, 0.4695333003662676, 0.0725148432336915},
      {0.5778346261482753, 0.1509145500825105, 0.2601230834564079},
      {0.5778346261482753, 0.2601230834564079, 0.1509145500825105},
      {0.1509145500825105, 0.5778346261482753, 0.2601230834564079},
      {0.2601230834564079, 0.5778346261482753, 0.1509145500825105},
      {0.1509145500825105, 0.2601230834564079, 0.5778346261482753},
      {0.2601230834564079, 0.1509145500825105, 0.5778346261482753},
      {0.0111277403128063, 0.1509145500825105, 0.2601230834564079},
      {0.0111277403128063, 0.2601230834564079, 0.1509145500825105},
      {0.0111277403128063, 0.5778346261482753, 0.2601230834564079},
      {0.0111277403128063, 0.5778346261482753, 0.1509145500825105},
      {0.0111277403128063, 0.2601230834564079, 0.5778346261482753},
      {0.0111277403128063, 0.1509145500825105, 0.5778346261482753},
      {0.1509145500825105, 0.0111277403128063, 0.2601230834564079},
      {0.2601230834564079, 0.0111277403128063, 0.1509145500825105},
      {0.5778346261482753, 0.0111277403128063, 0.2601230834564079},
      {0.5778346261482753, 0.0111277403128063, 0.1509145500825105},
      {0.2601230834564079, 0.0111277403128063, 0.5778346261482753},
      {0.1509145500825105, 0.0111277403128063, 0.5778346261482753},
      {0.1509145500825105, 0.2601230834564079, 0.0111277403128063},
      {0.2601230834564079, 0.1509145500825105, 0.0111277403128063},
      {0.5778346261482753, 0.2601230834564079, 0.0111277403128063},
      {0.5778346261482753, 0.1509145500825105, 0.0111277403128063},
      {0.2601230834564079, 0.5778346261482753, 0.0111277403128063},
      {0.1509145500825105, 0.5778346261482753, 0.0111277403128063},
      {0.0108219475368229, 0.3368450658547645, 0.2347260337441091},
      {0.0108219475368229, 0.2347260337441091, 0.3368450658547645},
      {0.3368450658547645, 0.0108219475368229, 0.2347260337441091},
      {0.2347260337441091, 0.0108219475368229, 0.3368450658547645},
      {0.3368450658547645, 0.2347260337441091, 0.0108219475368229},
      {0.2347260337441091, 0.3368450658547645, 0.0108219475368229},
      {0.4176069528643034, 0.3368450658547645, 0.2347260337441091},
      {0.4176069528643034, 0.2347260337441091, 0.3368450658547645},
      {0.4176069528643034, 0.0108219475368229, 0.2347260337441091},
      {0.4176069528643034, 0.0108219475368229, 0.3368450658547645},
      {0.4176069528643034, 0.2347260337441091, 0.0108219475368229},
      {0.4176069528643034, 0.3368450658547645, 0.0108219475368229},
      {0.3368450658547645, 0.4176069528643034, 0.2347260337441091},
      {0.2347260337441091, 0.4176069528643034, 0.3368450658547645},
      {0.0108219475368229, 0.4176069528643034, 0.2347260337441091},
      {0.0108219475368229, 0.4176069528643034, 0.3368450658547645},
      {0.2347260337441091, 0.4176069528643034, 0.0108219475368229},
      {0.3368450658547645, 0.4176069528643034, 0.0108219475368229},
      {0.3368450658547645, 0.2347260337441091, 0.4176069528643034},
      {0.2347260337441091, 0.3368450658547645, 0.4176069528643034},
      {0.0108219475368229, 0.2347260337441091, 0.4176069528643034},
      {0.0108219475368229, 0.3368450658547645, 0.4176069528643034},
      {0.2347260337441091, 0.0108219475368229, 0.4176069528643034},
      {0.3368450658547645, 0.0108219475368229, 0.4176069528643034}},
      {0.0000852004961909,
      0.0000852004961909,
      0.0000852004961909,
      0.0000852004961909,
      0.0027012343200378,
      0.0027012343200378,
      0.0027012343200378,
      0.0027012343200378,
      0.0025876310034022,
      0.0025876310034022,
      0.0025876310034022,
      0.0025876310034022,
      0.0002031683766626,
      0.0002031683766626,
      0.0002031683766626,
      0.0002031683766626,
      0.0002031683766626,
      0.0002031683766626,
      0.0014121438088543,
      0.0014121438088543,
      0.0014121438088543,
      0.0014121438088543,
      0.0014121438088543,
      0.0014121438088543,
      0.0002995890509010,
      0.0002995890509010,
      0.0002995890509010,
      0.0002995890509010,
      0.0002995890509010,
      0.0002995890509010,
      0.0002995890509010,
      0.0002995890509010,
      0.0002995890509010,
      0.0002995890509010,
      0.0002995890509010,
      0.0002995890509010,
      0.0001366335870127,
      0.0001366335870127,
      0.0001366335870127,
      0.0001366335870127,
      0.0001366335870127,
      0.0001366335870127,
      0.0001366335870127,
      0.0001366335870127,
      0.0001366335870127,
      0.0001366335870127,
      0.0001366335870127,
      0.0001366335870127,
      0.0012802958787455,
      0.0012802958787455,
      0.0012802958787455,
      0.0012802958787455,
      0.0012802958787455,
      0.0012802958787455,
      0.0012802958787455,
      0.0012802958787455,
      0.0012802958787455,
      0.0012802958787455,
      0.0012802958787455,
      0.0012802958787455,
      0.0007466417309625,
      0.0007466417309625,
      0.0007466417309625,
      0.0007466417309625,
      0.0007466417309625,
      0.0007466417309625,
      0.0007466417309625,
      0.0007466417309625,
      0.0007466417309625,
      0.0007466417309625,
      0.0007466417309625,
      0.0007466417309625,
      0.0009358831013974,
      0.0009358831013974,
      0.0009358831013974,
      0.0009358831013974,
      0.0009358831013974,
      0.0009358831013974,
      0.0009358831013974,
      0.0009358831013974,
      0.0009358831013974,
      0.0009358831013974,
      0.0009358831013974,
      0.0009358831013974,
      0.0002949506142995,
      0.0002949506142995,
      0.0002949506142995,
      0.0002949506142995,
      0.0002949506142995,
      0.0002949506142995,
      0.0002949506142995,
      0.0002949506142995,
      0.0002949506142995,
      0.0002949506142995,
      0.0002949506142995,
      0.0002949506142995,
      0.0001885230548645,
      0.0001885230548645,
      0.0001885230548645,
      0.0001885230548645,
      0.0001885230548645,
      0.0001885230548645,
      0.0001885230548645,
      0.0001885230548645,
      0.0001885230548645,
      0.0001885230548645,
      0.0001885230548645,
      0.0001885230548645,
      0.0001885230548645,
      0.0001885230548645,
      0.0001885230548645,
      0.0001885230548645,
      0.0001885230548645,
      0.0001885230548645,
      0.0001885230548645,
      0.0001885230548645,
      0.0001885230548645,
      0.0001885230548645,
      0.0001885230548645,
      0.0001885230548645,
      0.0001400832234596,
      0.0001400832234596,
      0.0001400832234596,
      0.0001400832234596,
      0.0001400832234596,
      0.0001400832234596,
      0.0001400832234596,
      0.0001400832234596,
      0.0001400832234596,
      0.0001400832234596,
      0.0001400832234596,
      0.0001400832234596,
      0.0001400832234596,
      0.0001400832234596,
      0.0001400832234596,
      0.0001400832234596,
      0.0001400832234596,
      0.0001400832234596,
      0.0001400832234596,
      0.0001400832234596,
      0.0001400832234596,
      0.0001400832234596,
      0.0001400832234596,
      0.0001400832234596,
      0.0005019947679004,
      0.0005019947679004,
      0.0005019947679004,
      0.0005019947679004,
      0.0005019947679004,
      0.0005019947679004,
      0.0005019947679004,
      0.0005019947679004,
      0.0005019947679004,
      0.0005019947679004,
      0.0005019947679004,
      0.0005019947679004,
      0.0005019947679004,
      0.0005019947679004,
      0.0005019947679004,
      0.0005019947679004,
      0.0005019947679004,
      0.0005019947679004,
      0.0005019947679004,
      0.0005019947679004,
      0.0005019947679004,
      0.0005019947679004,
      0.0005019947679004,
      0.0005019947679004,
      0.0008474048879204,
      0.0008474048879204,
      0.0008474048879204,
      0.0008474048879204,
      0.0008474048879204,
      0.0008474048879204,
      0.0008474048879204,
      0.0008474048879204,
      0.0008474048879204,
      0.0008474048879204,
      0.0008474048879204,
      0.0008474048879204,
      0.0008474048879204,
      0.0008474048879204,
      0.0008474048879204,
      0.0008474048879204,
      0.0008474048879204,
      0.0008474048879204,
      0.0008474048879204,
      0.0008474048879204,
      0.0008474048879204,
      0.0008474048879204,
      0.0008474048879204,
      0.0008474048879204,
      0.0011898972880007,
      0.0011898972880007,
      0.0011898972880007,
      0.0011898972880007,
      0.0011898972880007,
      0.0011898972880007,
      0.0011898972880007,
      0.0011898972880007,
      0.0011898972880007,
      0.0011898972880007,
      0.0011898972880007,
      0.0011898972880007,
      0.0011898972880007,
      0.0011898972880007,
      0.0011898972880007,
      0.0011898972880007,
      0.0011898972880007,
      0.0011898972880007,
      0.0011898972880007,
      0.0011898972880007,
      0.0011898972880007,
      0.0011898972880007,
      0.0011898972880007,
      0.0011898972880007,
      0.0005630917613715,
      0.0005630917613715,
      0.0005630917613715,
      0.0005630917613715,
      0.0005630917613715,
      0.0005630917613715,
      0.0005630917613715,
      0.0005630917613715,
      0.0005630917613715,
      0.0005630917613715,
      0.0005630917613715,
      0.0005630917613715,
      0.0005630917613715,
      0.0005630917613715,
      0.0005630917613715,
      0.0005630917613715,
      0.0005630917613715,
      0.0005630917613715,
      0.0005630917613715,
      0.0005630917613715,
      0.0005630917613715,
      0.0005630917613715,
      0.0005630917613715,
      0.0005630917613715,
      0.0003669467962836,
      0.0003669467962836,
      0.0003669467962836,
      0.0003669467962836,
      0.0003669467962836,
      0.0003669467962836,
      0.0003669467962836,
      0.0003669467962836,
      0.0003669467962836,
      0.0003669467962836,
      0.0003669467962836,
      0.0003669467962836,
      0.0003669467962836,
      0.0003669467962836,
      0.0003669467962836,
      0.0003669467962836,
      0.0003669467962836,
      0.0003669467962836,
      0.0003669467962836,
      0.0003669467962836,
      0.0003669467962836,
      0.0003669467962836,
      0.0003669467962836,
      0.0003669467962836}
    },
    {
      304,
      {{0.3296714738440606, 0.3296714738440606, 0.3296714738440606},
      {0.0109855784678181, 0.3296714738440606, 0.3296714738440606},
      {0.3296714738440606, 0.0109855784678181, 0.3296714738440606},
      {0.3296714738440606, 0.3296714738440606, 0.0109855784678181},
      {0.1120421044173788, 0.1120421044173788, 0.1120421044173788},
      {0.6638736867478636, 0.1120421044173788, 0.1120421044173788},
      {0.1120421044173788, 0.6638736867478636, 0.1120421044173788},
      {0.1120421044173788, 0.1120421044173788, 0.6638736867478636},
      {0.2804460259110929, 0.2804460259110929, 0.2804460259110929},
      {0.1586619222667212, 0.2804460259110929, 0.2804460259110929},
      {0.2804460259110929, 0.1586619222667212, 0.2804460259110929},
      {0.2804460259110929, 0.2804460259110929, 0.1586619222667212},
      {0.0394216444407617, 0.0394216444407617, 0.0394216444407617},
      {0.8817350666777151, 0.0394216444407617, 0.0394216444407617},
      {0.0394216444407617, 0.8817350666777151, 0.0394216444407617},
      {0.0394216444407617, 0.0394216444407617, 0.8817350666777151},
      {0.4250825814352324, 0.0749174185647676, 0.0749174185647676},
      {0.0749174185647676, 0.4250825814352324, 0.0749174185647676},
      {0.4250825814352324, 0.4250825814352324, 0.0749174185647676},
      {0.0749174185647676, 0.0749174185647676, 0.4250825814352324},
      {0.4250825814352324, 0.0749174185647676, 0.4250825814352324},
      {0.0749174185647676, 0.4250825814352324, 0.4250825814352324},
      {0.1643068970443654, 0.3356931029556346, 0.3356931029556346},
      {0.3356931029556346, 0.1643068970443654, 0.3356931029556346},
      {0.1643068970443654, 0.1643068970443654, 0.3356931029556346},
      {0.3356931029556346, 0.3356931029556346, 0.1643068970443654},
      {0.1643068970443654, 0.3356931029556346, 0.1643068970443654},
      {0.3356931029556346, 0.1643068970443654, 0.1643068970443654},
      {0.7646870675801803, 0.0490489875955668, 0.0490489875955668},
      {0.0490489875955668, 0.7646870675801803, 0.0490489875955668},
      {0.0490489875955668, 0.0490489875955668, 0.7646870675801803},
      {0.1372149572286861, 0.0490489875955668, 0.0490489875955668},
      {0.1372149572286861, 0.7646870675801803, 0.0490489875955668},
      {0.1372149572286861, 0.0490489875955668, 0.7646870675801803},
      {0.0490489875955668, 0.1372149572286861, 0.0490489875955668},
      {0.7646870675801803, 0.1372149572286861, 0.0490489875955668},
      {0.0490489875955668, 0.1372149572286861, 0.7646870675801803},
      {0.0490489875955668, 0.0490489875955668, 0.1372149572286861},
      {0.7646870675801803, 0.0490489875955668, 0.1372149572286861},
      {0.0490489875955668, 0.7646870675801803, 0.1372149572286861},
      {0.2328268045894251, 0.0141260956830925, 0.0141260956830925},
      {0.0141260956830925, 0.2328268045894251, 0.0141260956830925},
      {0.0141260956830925, 0.0141260956830925, 0.2328268045894251},
      {0.7389210040443899, 0.0141260956830925, 0.0141260956830925},
      {0.7389210040443899, 0.2328268045894251, 0.0141260956830925},
      {0.7389210040443899, 0.0141260956830925, 0.2328268045894251},
      {0.0141260956830925, 0.7389210040443899, 0.0141260956830925},
      {0.2328268045894251, 0.7389210040443899, 0.0141260956830925},
      {0.0141260956830925, 0.7389210040443899, 0.2328268045894251},
      {0.0141260956830925, 0.0141260956830925, 0.7389210040443899},
      {0.2328268045894251, 0.0141260956830925, 0.7389210040443899},
      {0.0141260956830925, 0.2328268045894251, 0.7389210040443899},
      {0.2832417683077947, 0.0623965205815433, 0.0623965205815433},
      {0.0623965205815433, 0.2832417683077947, 0.0623965205815433},
      {0.0623965205815433, 0.0623965205815433, 0.2832417683077947},
      {0.5919651905291188, 0.0623965205815433, 0.0623965205815433},
      {0.5919651905291188, 0.2832417683077947, 0.0623965205815433},
      {0.5919651905291188, 0.0623965205815433, 0.2832417683077947},
      {0.0623965205815433, 0.5919651905291188, 0.0623965205815433},
      {0.2832417683077947, 0.5919651905291188, 0.0623965205815433},
      {0.0623965205815433, 0.5919651905291188, 0.2832417683077947},
      {0.0623965205815433, 0.0623965205815433, 0.5919651905291188},
      {0.2832417683077947, 0.0623965205815433, 0.5919651905291188},
      {0.0623965205815433, 0.2832417683077947, 0.5919651905291188},
      {0.0128318740561182, 0.1890959275696560, 0.1890959275696560},
      {0.1890959275696560, 0.0128318740561182, 0.1890959275696560},
      {0.1890959275696560, 0.1890959275696560, 0.0128318740561182},
      {0.6089762708045698, 0.1890959275696560, 0.1890959275696560},
      {0.6089762708045698, 0.0128318740561182, 0.1890959275696560},
      {0.6089762708045698, 0.1890959275696560, 0.0128318740561182},
      {0.1890959275696560, 0.6089762708045698, 0.1890959275696560},
      {0.0128318740561182, 0.6089762708045698, 0.1890959275696560},
      {0.1890959275696560, 0.6089762708045698, 0.0128318740561182},
      {0.1890959275696560, 0.1890959275696560, 0.6089762708045698},
      {0.0128318740561182, 0.1890959275696560, 0.6089762708045698},
      {0.1890959275696560, 0.0128318740561182, 0.6089762708045698},
      {0.3872709603194903, 0.2750176001295444, 0.2750176001295444},
      {0.2750176001295444, 0.3872709603194903, 0.2750176001295444},
      {0.2750176001295444, 0.2750176001295444, 0.3872709603194903},
      {0.0626938394214209, 0.2750176001295444, 0.2750176001295444},
      {0.0626938394214209, 0.3872709603194903, 0.2750176001295444},
      {0.0626938394214209, 0.2750176001295444, 0.3872709603194903},
      {0.2750176001295444, 0.0626938394214209, 0.2750176001295444},
      {0.3872709603194903, 0.0626938394214209, 0.2750176001295444},
      {0.2750176001295444, 0.0626938394214209, 0.3872709603194903},
      {0.2750176001295444, 0.2750176001295444, 0.0626938394214209},
      {0.3872709603194903, 0.2750176001295444, 0.0626938394214209},
      {0.2750176001295444, 0.3872709603194903, 0.0626938394214209},
      {0.0372380593552354, 0.0059448982525699, 0.0059448982525699},
      {0.0059448982525699, 0.0372380593552354, 0.0059448982525699},
      {0.0059448982525699, 0.0059448982525699, 0.0372380593552354},
      {0.9508721441396247, 0.0059448982525699, 0.0059448982525699},
      {0.9508721441396247, 0.0372380593552354, 0.0059448982525699},
      {0.9508721441396247, 0.0059448982525699, 0.0372380593552354},
      {0.0059448982525699, 0.9508721441396247, 0.0059448982525699},
      {0.0372380593552354, 0.9508721441396247, 0.0059448982525699},
      {0.0059448982525699, 0.9508721441396247, 0.0372380593552354},
      {0.0059448982525699, 0.0059448982525699, 0.9508721441396247},
      {0.0372380593552354, 0.0059448982525699, 0.9508721441396247},
      {0.0059448982525699, 0.0372380593552354, 0.9508721441396247},
      {0.7482941078308859, 0.1183058071099944, 0.1183058071099944},
      {0.1183058071099944, 0.7482941078308859, 0.1183058071099944},
      {0.1183058071099944, 0.1183058071099944, 0.7482941078308859},
      {0.0150942779491252, 0.1183058071099944, 0.1183058071099944},
      {0.0150942779491252, 0.7482941078308859, 0.1183058071099944},
      {0.0150942779491252, 0.1183058071099944, 0.7482941078308859},
      {0.1183058071099944, 0.0150942779491252, 0.1183058071099944},
      {0.7482941078308859, 0.0150942779491252, 0.1183058071099944},
      {0.1183058071099944, 0.0150942779491252, 0.7482941078308859},
      {0.1183058071099944, 0.1183058071099944, 0.0150942779491252},
      {0.7482941078308859, 0.1183058071099944, 0.0150942779491252},
      {0.1183058071099944, 0.7482941078308859, 0.0150942779491252},
      {0.5146357887888395, 0.0801184612787250, 0.0144436385136434},
      {0.5146357887888395, 0.0144436385136434, 0.0801184612787250},
      {0.0801184612787250, 0.5146357887888395, 0.0144436385136434},
      {0.0144436385136434, 0.5146357887888395, 0.0801184612787250},
      {0.0801184612787250, 0.0144436385136434, 0.5146357887888395},
      {0.0144436385136434, 0.0801184612787250, 0.5146357887888395},
      {0.3908021114187921, 0.0801184612787250, 0.0144436385136434},
      {0.3908021114187921, 0.0144436385136434, 0.0801184612787250},
      {0.3908021114187921, 0.5146357887888395, 0.0144436385136434},
      {0.3908021114187921, 0.5146357887888395, 0.0801184612787250},
      {0.3908021114187921, 0.0144436385136434, 0.5146357887888395},
      {0.3908021114187921, 0.0801184612787250, 0.5146357887888395},
      {0.0801184612787250, 0.3908021114187921, 0.0144436385136434},
      {0.0144436385136434, 0.3908021114187921, 0.0801184612787250},
      {0.5146357887888395, 0.3908021114187921, 0.0144436385136434},
      {0.5146357887888395, 0.3908021114187921, 0.0801184612787250},
      {0.0144436385136434, 0.3908021114187921, 0.5146357887888395},
      {0.0801184612787250, 0.3908021114187921, 0.5146357887888395},
      {0.0801184612787250, 0.0144436385136434, 0.3908021114187921},
      {0.0144436385136434, 0.0801184612787250, 0.3908021114187921},
      {0.5146357887888395, 0.0144436385136434, 0.3908021114187921},
      {0.5146357887888395, 0.0801184612787250, 0.3908021114187921},
      {0.0144436385136434, 0.5146357887888395, 0.3908021114187921},
      {0.0801184612787250, 0.5146357887888395, 0.3908021114187921},
      {0.1645739468379099, 0.3102585498627273, 0.4552165700697292},
      {0.1645739468379099, 0.4552165700697292, 0.3102585498627273},
      {0.3102585498627273, 0.1645739468379099, 0.4552165700697292},
      {0.4552165700697292, 0.1645739468379099, 0.3102585498627273},
      {0.3102585498627273, 0.4552165700697292, 0.1645739468379099},
      {0.4552165700697292, 0.3102585498627273, 0.1645739468379099},
      {0.0699509332296337, 0.3102585498627273, 0.4552165700697292},
      {0.0699509332296337, 0.4552165700697292, 0.3102585498627273},
      {0.0699509332296337, 0.1645739468379099, 0.4552165700697292},
      {0.0699509332296337, 0.1645739468379099, 0.3102585498627273},
      {0.0699509332296337, 0.4552165700697292, 0.1645739468379099},
      {0.0699509332296337, 0.3102585498627273, 0.1645739468379099},
      {0.3102585498627273, 0.0699509332296337, 0.4552165700697292},
      {0.4552165700697292, 0.0699509332296337, 0.3102585498627273},
      {0.1645739468379099, 0.0699509332296337, 0.4552165700697292},
      {0.1645739468379099, 0.0699509332296337, 0.3102585498627273},
      {0.4552165700697292, 0.0699509332296337, 0.1645739468379099},
      {0.3102585498627273, 0.0699509332296337, 0.1645739468379099},
      {0.3102585498627273, 0.4552165700697292, 0.0699509332296337},
      {0.4552165700697292, 0.3102585498627273, 0.0699509332296337},
      {0.1645739468379099, 0.4552165700697292, 0.0699509332296337},
      {0.1645739468379099, 0.3102585498627273, 0.0699509332296337},
      {0.4552165700697292, 0.1645739468379099, 0.0699509332296337},
      {0.3102585498627273, 0.1645739468379099, 0.0699509332296337},
      {0.0343586795014570, 0.1085240801928985, 0.0014015410850694},
      {0.0343586795014570, 0.0014015410850694, 0.1085240801928985},
      {0.1085240801928985, 0.0343586795014570, 0.0014015410850694},
      {0.0014015410850694, 0.0343586795014570, 0.1085240801928985},
      {0.1085240801928985, 0.0014015410850694, 0.0343586795014570},
      {0.0014015410850694, 0.1085240801928985, 0.0343586795014570},
      {0.8557156992205752, 0.1085240801928985, 0.0014015410850694},
      {0.8557156992205752, 0.0014015410850694, 0.1085240801928985},
      {0.8557156992205752, 0.0343586795014570, 0.0014015410850694},
      {0.8557156992205752, 0.0343586795014570, 0.1085240801928985},
      {0.8557156992205752, 0.0014015410850694, 0.0343586795014570},
      {0.8557156992205752, 0.1085240801928985, 0.0343586795014570},
      {0.1085240801928985, 0.8557156992205752, 0.0014015410850694},
      {0.0014015410850694, 0.8557156992205752, 0.1085240801928985},
      {0.0343586795014570, 0.8557156992205752, 0.0014015410850694},
      {0.0343586795014570, 0.8557156992205752, 0.1085240801928985},
      {0.0014015410850694, 0.8557156992205752, 0.0343586795014570},
      {0.1085240801928985, 0.8557156992205752, 0.0343586795014570},
      {0.1085240801928985, 0.0014015410850694, 0.8557156992205752},
      {0.0014015410850694, 0.1085240801928985, 0.8557156992205752},
      {0.0343586795014570, 0.0014015410850694, 0.8557156992205752},
      {0.0343586795014570, 0.1085240801928985, 0.8557156992205752},
      {0.0014015410850694, 0.0343586795014570, 0.8557156992205752},
      {0.1085240801928985, 0.0343586795014570, 0.8557156992205752},
      {0.6625317544850510, 0.2483824987814955, 0.0781025122458046},
      {0.6625317544850510, 0.0781025122458046, 0.2483824987814955},
      {0.2483824987814955, 0.6625317544850510, 0.0781025122458046},
      {0.0781025122458046, 0.6625317544850510, 0.2483824987814955},
      {0.2483824987814955, 0.0781025122458046, 0.6625317544850510},
      {0.0781025122458046, 0.2483824987814955, 0.6625317544850510},
      {0.0109832344876490, 0.2483824987814955, 0.0781025122458046},
      {0.0109832344876490, 0.0781025122458046, 0.2483824987814955},
      {0.0109832344876490, 0.6625317544850510, 0.0781025122458046},
      {0.0109832344876490, 0.6625317544850510, 0.2483824987814955},
      {0.0109832344876490, 0.0781025122458046, 0.6625317544850510},
      {0.0109832344876490, 0.2483824987814955, 0.6625317544850510},
      {0.2483824987814955, 0.0109832344876490, 0.0781025122458046},
      {0.0781025122458046, 0.0109832344876490, 0.2483824987814955},
      {0.6625317544850510, 0.0109832344876490, 0.0781025122458046},
      {0.6625317544850510, 0.0109832344876490, 0.2483824987814955},
      {0.0781025122458046, 0.0109832344876490, 0.6625317544850510},
      {0.2483824987814955, 0.0109832344876490, 0.6625317544850510},
      {0.2483824987814955, 0.0781025122458046, 0.0109832344876490},
      {0.0781025122458046, 0.2483824987814955, 0.0109832344876490},
      {0.6625317544850510, 0.0781025122458046, 0.0109832344876490},
      {0.6625317544850510, 0.2483824987814955, 0.0109832344876490},
      {0.0781025122458046, 0.6625317544850510, 0.0109832344876490},
      {0.2483824987814955, 0.6625317544850510, 0.0109832344876490},
      {0.0122689867800652, 0.3960091211067035, 0.5729400176172562},
      {0.0122689867800652, 0.5729400176172562, 0.3960091211067035},
      {0.3960091211067035, 0.0122689867800652, 0.5729400176172562},
      {0.5729400176172562, 0.0122689867800652, 0.3960091211067035},
      {0.3960091211067035, 0.5729400176172562, 0.0122689867800652},
      {0.5729400176172562, 0.3960091211067035, 0.0122689867800652},
      {0.0187818744959751, 0.3960091211067035, 0.5729400176172562},
      {0.0187818744959751, 0.5729400176172562, 0.3960091211067035},
      {0.0187818744959751, 0.0122689867800652, 0.5729400176172562},
      {0.0187818744959751, 0.0122689867800652, 0.3960091211067035},
      {0.0187818744959751, 0.5729400176172562, 0.0122689867800652},
      {0.0187818744959751, 0.3960091211067035, 0.0122689867800652},
      {0.3960091211067035, 0.0187818744959751, 0.5729400176172562},
      {0.5729400176172562, 0.0187818744959751, 0.3960091211067035},
      {0.0122689867800652, 0.0187818744959751, 0.5729400176172562},
      {0.0122689867800652, 0.0187818744959751, 0.3960091211067035},
      {0.5729400176172562, 0.0187818744959751, 0.0122689867800652},
      {0.3960091211067035, 0.0187818744959751, 0.0122689867800652},
      {0.3960091211067035, 0.5729400176172562, 0.0187818744959751},
      {0.5729400176172562, 0.3960091211067035, 0.0187818744959751},
      {0.0122689867800652, 0.5729400176172562, 0.0187818744959751},
      {0.0122689867800652, 0.3960091211067035, 0.0187818744959751},
      {0.5729400176172562, 0.0122689867800652, 0.0187818744959751},
      {0.3960091211067035, 0.0122689867800652, 0.0187818744959751},
      {0.2054604991324105, 0.0636751619713731, 0.5946193880076269},
      {0.2054604991324105, 0.5946193880076269, 0.0636751619713731},
      {0.0636751619713731, 0.2054604991324105, 0.5946193880076269},
      {0.5946193880076269, 0.2054604991324105, 0.0636751619713731},
      {0.0636751619713731, 0.5946193880076269, 0.2054604991324105},
      {0.5946193880076269, 0.0636751619713731, 0.2054604991324105},
      {0.1362449508885895, 0.0636751619713731, 0.5946193880076269},
      {0.1362449508885895, 0.5946193880076269, 0.0636751619713731},
      {0.1362449508885895, 0.2054604991324105, 0.5946193880076269},
      {0.1362449508885895, 0.2054604991324105, 0.0636751619713731},
      {0.1362449508885895, 0.5946193880076269, 0.2054604991324105},
      {0.1362449508885895, 0.0636751619713731, 0.2054604991324105},
      {0.0636751619713731, 0.1362449508885895, 0.5946193880076269},
      {0.5946193880076269, 0.1362449508885895, 0.0636751619713731},
      {0.2054604991324105, 0.1362449508885895, 0.5946193880076269},
      {0.2054604991324105, 0.1362449508885895, 0.0636751619713731},
      {0.5946193880076269, 0.1362449508885895, 0.2054604991324105},
      {0.0636751619713731, 0.1362449508885895, 0.2054604991324105},
      {0.0636751619713731, 0.5946193880076269, 0.1362449508885895},
      {0.5946193880076269, 0.0636751619713731, 0.1362449508885895},
      {0.2054604991324105, 0.5946193880076269, 0.1362449508885895},
      {0.2054604991324105, 0.0636751619713731, 0.1362449508885895},
      {0.5946193880076269, 0.2054604991324105, 0.1362449508885895},
      {0.0636751619713731, 0.2054604991324105, 0.1362449508885895},
      {0.4610678860796995, 0.1757650466139104, 0.2244134963451648},
      {0.4610678860796995, 0.2244134963451648, 0.1757650466139104},
      {0.1757650466139104, 0.4610678860796995, 0.2244134963451648},
      {0.2244134963451648, 0.4610678860796995, 0.1757650466139104},
      {0.1757650466139104, 0.2244134963451648, 0.4610678860796995},
      {0.2244134963451648, 0.1757650466139104, 0.4610678860796995},
      {0.1387535709612253, 0.1757650466139104, 0.2244134963451648},
      {0.1387535709612253, 0.2244134963451648, 0.1757650466139104},
      {0.1387535709612253, 0.4610678860796995, 0.2244134963451648},
      {0.1387535709612253, 0.4610678860796995, 0.1757650466139104},
      {0.1387535709612253, 0.2244134963451648, 0.4610678860796995},
      {0.1387535709612253, 0.1757650466139104, 0.4610678860796995},
      {0.1757650466139104, 0.1387535709612253, 0.2244134963451648},
      {0.2244134963451648, 0.1387535709612253, 0.1757650466139104},
      {0.4610678860796995, 0.1387535709612253, 0.2244134963451648},
      {0.4610678860796995, 0.1387535709612253, 0.1757650466139104},
      {0.2244134963451648, 0.1387535709612253, 0.4610678860796995},
      {0.1757650466139104, 0.1387535709612253, 0.4610678860796995},
      {0.1757650466139104, 0.2244134963451648, 0.1387535709612253},
      {0.2244134963451648, 0.1757650466139104, 0.1387535709612253},
      {0.4610678860796995, 0.2244134963451648, 0.1387535709612253},
      {0.4610678860796995, 0.1757650466139104, 0.1387535709612253},
      {0.2244134963451648, 0.4610678860796995, 0.1387535709612253},
      {0.1757650466139104, 0.4610678860796995, 0.1387535709612253},
      {0.0134478861029963, 0.4779942532006705, 0.1864180300424337},
      {0.0134478861029963, 0.1864180300424337, 0.4779942532006705},
      {0.4779942532006705, 0.0134478861029963, 0.1864180300424337},
      {0.1864180300424337, 0.0134478861029963, 0.4779942532006705},
      {0.4779942532006705, 0.1864180300424337, 0.0134478861029963},
      {0.1864180300424337, 0.4779942532006705, 0.0134478861029963},
      {0.3221398306538996, 0.4779942532006705, 0.1864180300424337},
      {0.3221398306538996, 0.1864180300424337, 0.4779942532006705},
      {0.3221398306538996, 0.0134478861029963, 0.1864180300424337},
      {0.3221398306538996, 0.0134478861029963, 0.4779942532006705},
      {0.3221398306538996, 0.1864180300424337, 0.0134478861029963},
      {0.3221398306538996, 0.4779942532006705, 0.0134478861029963},
      {0.4779942532006705, 0.3221398306538996, 0.1864180300424337},
      {0.1864180300424337, 0.3221398306538996, 0.4779942532006705},
      {0.0134478861029963, 0.3221398306538996, 0.1864180300424337},
      {0.0134478861029963, 0.3221398306538996, 0.4779942532006705},
      {0.1864180300424337, 0.3221398306538996, 0.0134478861029963},
      {0.4779942532006705, 0.3221398306538996, 0.0134478861029963},
      {0.4779942532006705, 0.1864180300424337, 0.3221398306538996},
      {0.1864180300424337, 0.4779942532006705, 0.3221398306538996},
      {0.0134478861029963, 0.1864180300424337, 0.3221398306538996},
      {0.0134478861029963, 0.4779942532006705, 0.3221398306538996},
      {0.1864180300424337, 0.0134478861029963, 0.3221398306538996},
      {0.4779942532006705, 0.0134478861029963, 0.3221398306538996}},
      {0.0006724798228731,
      0.0006724798228731,
      0.0006724798228731,
      0.0006724798228731,
      0.0008772380013413,
      0.0008772380013413,
      0.0008772380013413,
      0.0008772380013413,
      0.0017977318447629,
      0.0017977318447629,
      0.0017977318447629,
      0.0017977318447629,
      0.0003153900048385,
      0.0003153900048385,
      0.0003153900048385,
      0.0003153900048385,
      0.0010024185005001,
      0.0010024185005001,
      0.0010024185005001,
      0.0010024185005001,
      0.0010024185005001,
      0.0010024185005001,
      0.0014486985388122,
      0.0014486985388122,
      0.0014486985388122,
      0.0014486985388122,
      0.0014486985388122,
      0.0014486985388122,
      0.0005846038809642,
      0.0005846038809642,
      0.0005846038809642,
      0.0005846038809642,
      0.0005846038809642,
      0.0005846038809642,
      0.0005846038809642,
      0.0005846038809642,
      0.0005846038809642,
      0.0005846038809642,
      0.0005846038809642,
      0.0005846038809642,
      0.0001981103788865,
      0.0001981103788865,
      0.0001981103788865,
      0.0001981103788865,
      0.0001981103788865,
      0.0001981103788865,
      0.0001981103788865,
      0.0001981103788865,
      0.0001981103788865,
      0.0001981103788865,
      0.0001981103788865,
      0.0001981103788865,
      0.0007242042190778,
      0.0007242042190778,
      0.0007242042190778,
      0.0007242042190778,
      0.0007242042190778,
      0.0007242042190778,
      0.0007242042190778,
      0.0007242042190778,
      0.0007242042190778,
      0.0007242042190778,
      0.0007242042190778,
      0.0007242042190778,
      0.0004434477617005,
      0.0004434477617005,
      0.0004434477617005,
      0.0004434477617005,
      0.0004434477617005,
      0.0004434477617005,
      0.0004434477617005,
      0.0004434477617005,
      0.0004434477617005,
      0.0004434477617005,
      0.0004434477617005,
      0.0004434477617005,
      0.0009255290503540,
      0.0009255290503540,
      0.0009255290503540,
      0.0009255290503540,
      0.0009255290503540,
      0.0009255290503540,
      0.0009255290503540,
      0.0009255290503540,
      0.0009255290503540,
      0.0009255290503540,
      0.0009255290503540,
      0.0009255290503540,
      0.0000255238567566,
      0.0000255238567566,
      0.0000255238567566,
      0.0000255238567566,
      0.0000255238567566,
      0.0000255238567566,
      0.0000255238567566,
      0.0000255238567566,
      0.0000255238567566,
      0.0000255238567566,
      0.0000255238567566,
      0.0000255238567566,
      0.0003718673561193,
      0.0003718673561193,
      0.0003718673561193,
      0.0003718673561193,
      0.0003718673561193,
      0.0003718673561193,
      0.0003718673561193,
      0.0003718673561193,
      0.0003718673561193,
      0.0003718673561193,
      0.0003718673561193,
      0.0003718673561193,
      0.0004120628593750,
      0.0004120628593750,
      0.0004120628593750,
      0.0004120628593750,
      0.0004120628593750,
      0.0004120628593750,
      0.0004120628593750,
      0.0004120628593750,
      0.0004120628593750,
      0.0004120628593750,
      0.0004120628593750,
      0.0004120628593750,
      0.0004120628593750,
      0.0004120628593750,
      0.0004120628593750,
      0.0004120628593750,
      0.0004120628593750,
      0.0004120628593750,
      0.0004120628593750,
      0.0004120628593750,
      0.0004120628593750,
      0.0004120628593750,
      0.0004120628593750,
      0.0004120628593750,
      0.0010993538153030,
      0.0010993538153030,
      0.0010993538153030,
      0.0010993538153030,
      0.0010993538153030,
      0.0010993538153030,
      0.0010993538153030,
      0.0010993538153030,
      0.0010993538153030,
      0.0010993538153030,
      0.0010993538153030,
      0.0010993538153030,
      0.0010993538153030,
      0.0010993538153030,
      0.0010993538153030,
      0.0010993538153030,
      0.0010993538153030,
      0.0010993538153030,
      0.0010993538153030,
      0.0010993538153030,
      0.0010993538153030,
      0.0010993538153030,
      0.0010993538153030,
      0.0010993538153030,
      0.0001000724844616,
      0.0001000724844616,
      0.0001000724844616,
      0.0001000724844616,
      0.0001000724844616,
      0.0001000724844616,
      0.0001000724844616,
      0.0001000724844616,
      0.0001000724844616,
      0.0001000724844616,
      0.0001000724844616,
      0.0001000724844616,
      0.0001000724844616,
      0.0001000724844616,
      0.0001000724844616,
      0.0001000724844616,
      0.0001000724844616,
      0.0001000724844616,
      0.0001000724844616,
      0.0001000724844616,
      0.0001000724844616,
      0.0001000724844616,
      0.0001000724844616,
      0.0001000724844616,
      0.0003737218328128,
      0.0003737218328128,
      0.0003737218328128,
      0.0003737218328128,
      0.0003737218328128,
      0.0003737218328128,
      0.0003737218328128,
      0.0003737218328128,
      0.0003737218328128,
      0.0003737218328128,
      0.0003737218328128,
      0.0003737218328128,
      0.0003737218328128,
      0.0003737218328128,
      0.0003737218328128,
      0.0003737218328128,
      0.0003737218328128,
      0.0003737218328128,
      0.0003737218328128,
      0.0003737218328128,
      0.0003737218328128,
      0.0003737218328128,
      0.0003737218328128,
      0.0003737218328128,
      0.0001374070393745,
      0.0001374070393745,
      0.0001374070393745,
      0.0001374070393745,
      0.0001374070393745,
      0.0001374070393745,
      0.0001374070393745,
      0.0001374070393745,
      0.0001374070393745,
      0.0001374070393745,
      0.0001374070393745,
      0.0001374070393745,
      0.0001374070393745,
      0.0001374070393745,
      0.0001374070393745,
      0.0001374070393745,
      0.0001374070393745,
      0.0001374070393745,
      0.0001374070393745,
      0.0001374070393745,
      0.0001374070393745,
      0.0001374070393745,
      0.0001374070393745,
      0.0001374070393745,
      0.0007313359725447,
      0.0007313359725447,
      0.0007313359725447,
      0.0007313359725447,
      0.0007313359725447,
      0.0007313359725447,
      0.0007313359725447,
      0.0007313359725447,
      0.0007313359725447,
      0.0007313359725447,
      0.0007313359725447,
      0.0007313359725447,
      0.0007313359725447,
      0.0007313359725447,
      0.0007313359725447,
      0.0007313359725447,
      0.0007313359725447,
      0.0007313359725447,
      0.0007313359725447,
      0.0007313359725447,
      0.0007313359725447,
      0.0007313359725447,
      0.0007313359725447,
      0.0007313359725447,
      0.0005757130709568,
      0.0005757130709568,
      0.0005757130709568,
      0.0005757130709568,
      0.0005757130709568,
      0.0005757130709568,
      0.0005757130709568,
      0.0005757130709568,
      0.0005757130709568,
      0.0005757130709568,
      0.0005757130709568,
      0.0005757130709568,
      0.0005757130709568,
      0.0005757130709568,
      0.0005757130709568,
      0.0005757130709568,
      0.0005757130709568,
      0.0005757130709568,
      0.0005757130709568,
      0.0005757130709568,
      0.0005757130709568,
      0.0005757130709568,
      0.0005757130709568,
      0.0005757130709568,
      0.0006548815788893,
      0.0006548815788893,
      0.0006548815788893,
      0.0006548815788893,
      0.0006548815788893,
      0.0006548815788893,
      0.0006548815788893,
      0.0006548815788893,
      0.0006548815788893,
      0.0006548815788893,
      0.0006548815788893,
      0.0006548815788893,
      0.0006548815788893,
      0.0006548815788893,
      0.0006548815788893,
      0.0006548815788893,
      0.0006548815788893,
      0.0006548815788893,
      0.0006548815788893,
      0.0006548815788893,
      0.0006548815788893,
      0.0006548815788893,
      0.0006548815788893,
      0.0006548815788893}
    },
    {
      364,
      {{0.1092594110391400, 0.1092594110391400, 0.1092594110391400},
      {0.6722217668825801, 0.1092594110391400, 0.1092594110391400},
      {0.1092594110391400, 0.6722217668825801, 0.1092594110391400},
      {0.1092594110391400, 0.1092594110391400, 0.6722217668825801},
      {0.0170690533535099, 0.0170690533535099, 0.0170690533535099},
      {0.9487928399394703, 0.0170690533535099, 0.0170690533535099},
      {0.0170690533535099, 0.9487928399394703, 0.0170690533535099},
      {0.0170690533535099, 0.0170690533535099, 0.9487928399394703},
      {0.0603027644020867, 0.0603027644020867, 0.0603027644020867},
      {0.8190917067937399, 0.0603027644020867, 0.0603027644020867},
      {0.0603027644020867, 0.8190917067937399, 0.0603027644020867},
      {0.0603027644020867, 0.0603027644020867, 0.8190917067937399},
      {0.1802936778752487, 0.1802936778752487, 0.1802936778752487},
      {0.4591189663742538, 0.1802936778752487, 0.1802936778752487},
      {0.1802936778752487, 0.4591189663742538, 0.1802936778752487},
      {0.1802936778752487, 0.1802936778752487, 0.4591189663742538},
      {0.2924357505680077, 0.2075642494319923, 0.2075642494319923},
      {0.2075642494319923, 0.2924357505680077, 0.2075642494319923},
      {0.2924357505680077, 0.2924357505680077, 0.2075642494319923},
      {0.2075642494319923, 0.2075642494319923, 0.2924357505680077},
      {0.2924357505680077, 0.2075642494319923, 0.2924357505680077},
      {0.2075642494319923, 0.2924357505680077, 0.2924357505680077},
      {0.4364943812496382, 0.0635056187503618, 0.0635056187503618},
      {0.0635056187503618, 0.4364943812496382, 0.0635056187503618},
      {0.4364943812496382, 0.4364943812496382, 0.0635056187503618},
      {0.0635056187503618, 0.0635056187503618, 0.4364943812496382},
      {0.4364943812496382, 0.0635056187503618, 0.4364943812496382},
      {0.0635056187503618, 0.4364943812496382, 0.4364943812496382},
      {0.3574259051430166, 0.1425740948569834, 0.1425740948569834},
      {0.1425740948569834, 0.3574259051430166, 0.1425740948569834},
      {0.3574259051430166, 0.3574259051430166, 0.1425740948569834},
      {0.1425740948569834, 0.1425740948569834, 0.3574259051430166},
      {0.3574259051430166, 0.1425740948569834, 0.3574259051430166},
      {0.1425740948569834, 0.3574259051430166, 0.3574259051430166},
      {0.4847450103886418, 0.0152549896113582, 0.0152549896113582},
      {0.0152549896113582, 0.4847450103886418, 0.0152549896113582},
      {0.4847450103886418, 0.4847450103886418, 0.0152549896113582},
      {0.0152549896113582, 0.0152549896113582, 0.4847450103886418},
      {0.4847450103886418, 0.0152549896113582, 0.4847450103886418},
      {0.0152549896113582, 0.4847450103886418, 0.4847450103886418},
      {0.0066056285459364, 0.2817717468661823, 0.2817717468661823},
      {0.2817717468661823, 0.0066056285459364, 0.2817717468661823},
      {0.2817717468661823, 0.2817717468661823, 0.0066056285459364},
      {0.4298508777216989, 0.2817717468661823, 0.2817717468661823},
      {0.4298508777216989, 0.0066056285459364, 0.2817717468661823},
      {0.4298508777216989, 0.2817717468661823, 0.0066056285459364},
      {0.2817717468661823, 0.4298508777216989, 0.2817717468661823},
      {0.0066056285459364, 0.4298508777216989, 0.2817717468661823},
      {0.2817717468661823, 0.4298508777216989, 0.0066056285459364},
      {0.2817717468661823, 0.2817717468661823, 0.4298508777216989},
      {0.0066056285459364, 0.2817717468661823, 0.4298508777216989},
      {0.2817717468661823, 0.0066056285459364, 0.4298508777216989},
      {0.3473076755269789, 0.2532792896616394, 0.2532792896616394},
      {0.2532792896616394, 0.3473076755269789, 0.2532792896616394},
      {0.2532792896616394, 0.2532792896616394, 0.3473076755269789},
      {0.1461337451497424, 0.2532792896616394, 0.2532792896616394},
      {0.1461337451497424, 0.3473076755269789, 0.2532792896616394},
      {0.1461337451497424, 0.2532792896616394, 0.3473076755269789},
      {0.2532792896616394, 0.1461337451497424, 0.2532792896616394},
      {0.3473076755269789, 0.1461337451497424, 0.2532792896616394},
      {0.2532792896616394, 0.1461337451497424, 0.3473076755269789},
      {0.2532792896616394, 0.2532792896616394, 0.1461337451497424},
      {0.3473076755269789, 0.2532792896616394, 0.1461337451497424},
      {0.2532792896616394, 0.3473076755269789, 0.1461337451497424},
      {0.5129631932135305, 0.1194446836887992, 0.1194446836887992},
      {0.1194446836887992, 0.5129631932135305, 0.1194446836887992},
      {0.1194446836887992, 0.1194446836887992, 0.5129631932135305},
      {0.2481474394088712, 0.1194446836887992, 0.1194446836887992},
      {0.2481474394088712, 0.5129631932135305, 0.1194446836887992},
      {0.2481474394088712, 0.1194446836887992, 0.5129631932135305},
      {0.1194446836887992, 0.2481474394088712, 0.1194446836887992},
      {0.5129631932135305, 0.2481474394088712, 0.1194446836887992},
      {0.1194446836887992, 0.2481474394088712, 0.5129631932135305},
      {0.1194446836887992, 0.1194446836887992, 0.2481474394088712},
      {0.5129631932135305, 0.1194446836887992, 0.2481474394088712},
      {0.1194446836887992, 0.5129631932135305, 0.2481474394088712},
      {0.4040135150131379, 0.2796975892179978, 0.2796975892179978},
      {0.2796975892179978, 0.4040135150131379, 0.2796975892179978},
      {0.2796975892179978, 0.2796975892179978, 0.4040135150131379},
      {0.0365913065508666, 0.2796975892179978, 0.2796975892179978},
      {0.0365913065508666, 0.4040135150131379, 0.2796975892179978},
      {0.0365913065508666, 0.2796975892179978, 0.4040135150131379},
      {0.2796975892179978, 0.0365913065508666, 0.2796975892179978},
      {0.4040135150131379, 0.0365913065508666, 0.2796975892179978},
      {0.2796975892179978, 0.0365913065508666, 0.4040135150131379},
      {0.2796975892179978, 0.2796975892179978, 0.0365913065508666},
      {0.4040135150131379, 0.2796975892179978, 0.0365913065508666},
      {0.2796975892179978, 0.4040135150131379, 0.0365913065508666},
      {0.2771505768037979, 0.0591496500175592, 0.0591496500175592},
      {0.0591496500175592, 0.2771505768037979, 0.0591496500175592},
      {0.0591496500175592, 0.0591496500175592, 0.2771505768037979},
      {0.6045501231610838, 0.0591496500175592, 0.0591496500175592},
      {0.6045501231610838, 0.2771505768037979, 0.0591496500175592},
      {0.6045501231610838, 0.0591496500175592, 0.2771505768037979},
      {0.0591496500175592, 0.6045501231610838, 0.0591496500175592},
      {0.2771505768037979, 0.6045501231610838, 0.0591496500175592},
      {0.0591496500175592, 0.6045501231610838, 0.2771505768037979},
      {0.0591496500175592, 0.0591496500175592, 0.6045501231610838},
      {0.2771505768037979, 0.0591496500175592, 0.6045501231610838},
      {0.0591496500175592, 0.2771505768037979, 0.6045501231610838},
      {0.3372693231713962, 0.0115664568797204, 0.0115664568797204},
      {0.0115664568797204, 0.3372693231713962, 0.0115664568797204},
      {0.0115664568797204, 0.0115664568797204, 0.3372693231713962},
      {0.6395977630691629, 0.0115664568797204, 0.0115664568797204},
      {0.6395977630691629, 0.3372693231713962, 0.0115664568797204},
      {0.6395977630691629, 0.0115664568797204, 0.3372693231713962},
      {0.0115664568797204, 0.6395977630691629, 0.0115664568797204},
      {0.3372693231713962, 0.6395977630691629, 0.0115664568797204},
      {0.0115664568797204, 0.6395977630691629, 0.3372693231713962},
      {0.0115664568797204, 0.0115664568797204, 0.6395977630691629},
      {0.3372693231713962, 0.0115664568797204, 0.6395977630691629},
      {0.0115664568797204, 0.3372693231713962, 0.6395977630691629},
      {0.0889921351196739, 0.2546200946118619, 0.2546200946118619},
      {0.2546200946118619, 0.0889921351196739, 0.2546200946118619},
      {0.2546200946118619, 0.2546200946118619, 0.0889921351196739},
      {0.4017676756566024, 0.2546200946118619, 0.2546200946118619},
      {0.4017676756566024, 0.0889921351196739, 0.2546200946118619},
      {0.4017676756566024, 0.2546200946118619, 0.0889921351196739},
      {0.2546200946118619, 0.4017676756566024, 0.2546200946118619},
      {0.0889921351196739, 0.4017676756566024, 0.2546200946118619},
      {0.2546200946118619, 0.4017676756566024, 0.0889921351196739},
      {0.2546200946118619, 0.2546200946118619, 0.4017676756566024},
      {0.0889921351196739, 0.2546200946118619, 0.4017676756566024},
      {0.2546200946118619, 0.0889921351196739, 0.4017676756566024},
      {0.0833537700079728, 0.0066729146778652, 0.0066729146778652},
      {0.0066729146778652, 0.0833537700079728, 0.0066729146778652},
      {0.0066729146778652, 0.0066729146778652, 0.0833537700079728},
      {0.9033004006362969, 0.0066729146778652, 0.0066729146778652},
      {0.9033004006362969, 0.0833537700079728, 0.0066729146778652},
      {0.9033004006362969, 0.0066729146778652, 0.0833537700079728},
      {0.0066729146778652, 0.9033004006362969, 0.0066729146778652},
      {0.0833537700079728, 0.9033004006362969, 0.0066729146778652},
      {0.0066729146778652, 0.9033004006362969, 0.0833537700079728},
      {0.0066729146778652, 0.0066729146778652, 0.9033004006362969},
      {0.0833537700079728, 0.0066729146778652, 0.9033004006362969},
      {0.0066729146778652, 0.0833537700079728, 0.9033004006362969},
      {0.1488738514609226, 0.0595499636617363, 0.0595499636617363},
      {0.0595499636617363, 0.1488738514609226, 0.0595499636617363},
      {0.0595499636617363, 0.0595499636617363, 0.1488738514609226},
      {0.7320262212156049, 0.0595499636617363, 0.0595499636617363},
      {0.7320262212156049, 0.1488738514609226, 0.0595499636617363},
      {0.7320262212156049, 0.0595499636617363, 0.1488738514609226},
      {0.0595499636617363, 0.7320262212156049, 0.0595499636617363},
      {0.1488738514609226, 0.7320262212156049, 0.0595499636617363},
      {0.0595499636617363, 0.7320262212156049, 0.1488738514609226},
      {0.0595499636617363, 0.0595499636617363, 0.7320262212156049},
      {0.1488738514609226, 0.0595499636617363, 0.7320262212156049},
      {0.0595499636617363, 0.1488738514609226, 0.7320262212156049},
      {0.0130804747131276, 0.0956382942582895, 0.7229319246103527},
      {0.0130804747131276, 0.7229319246103527, 0.0956382942582895},
      {0.0956382942582895, 0.0130804747131276, 0.7229319246103527},
      {0.7229319246103527, 0.0130804747131276, 0.0956382942582895},
      {0.0956382942582895, 0.7229319246103527, 0.0130804747131276},
      {0.7229319246103527, 0.0956382942582895, 0.0130804747131276},
      {0.1683493064182302, 0.0956382942582895, 0.7229319246103527},
      {0.1683493064182302, 0.7229319246103527, 0.0956382942582895},
      {0.1683493064182302, 0.0130804747131276, 0.7229319246103527},
      {0.1683493064182302, 0.0130804747131276, 0.0956382942582895},
      {0.1683493064182302, 0.7229319246103527, 0.0130804747131276},
      {0.1683493064182302, 0.0956382942582895, 0.0130804747131276},
      {0.0956382942582895, 0.1683493064182302, 0.7229319246103527},
      {0.7229319246103527, 0.1683493064182302, 0.0956382942582895},
      {0.0130804747131276, 0.1683493064182302, 0.7229319246103527},
      {0.0130804747131276, 0.1683493064182302, 0.0956382942582895},
      {0.7229319246103527, 0.1683493064182302, 0.0130804747131276},
      {0.0956382942582895, 0.1683493064182302, 0.0130804747131276},
      {0.0956382942582895, 0.7229319246103527, 0.1683493064182302},
      {0.7229319246103527, 0.0956382942582895, 0.1683493064182302},
      {0.0130804747131276, 0.7229319246103527, 0.1683493064182302},
      {0.0130804747131276, 0.0956382942582895, 0.1683493064182302},
      {0.7229319246103527, 0.0130804747131276, 0.1683493064182302},
      {0.0956382942582895, 0.0130804747131276, 0.1683493064182302},
      {0.6376551993416253, 0.0603575153489387, 0.2913648223135218},
      {0.6376551993416253, 0.2913648223135218, 0.0603575153489387},
      {0.0603575153489387, 0.6376551993416253, 0.2913648223135218},
      {0.2913648223135218, 0.6376551993416253, 0.0603575153489387},
      {0.0603575153489387, 0.2913648223135218, 0.6376551993416253},
      {0.2913648223135218, 0.0603575153489387, 0.6376551993416253},
      {0.0106224629959142, 0.0603575153489387, 0.2913648223135218},
      {0.0106224629959142, 0.2913648223135218, 0.0603575153489387},
      {0.0106224629959142, 0.6376551993416253, 0.2913648223135218},
      {0.0106224629959142, 0.6376551993416253, 0.0603575153489387},
      {0.0106224629959142, 0.2913648223135218, 0.6376551993416253},
      {0.0106224629959142, 0.0603575153489387, 0.6376551993416253},
      {0.0603575153489387, 0.0106224629959142, 0.2913648223135218},
      {0.2913648223135218, 0.0106224629959142, 0.0603575153489387},
      {0.6376551993416253, 0.0106224629959142, 0.2913648223135218},
      {0.6376551993416253, 0.0106224629959142, 0.0603575153489387},
      {0.2913648223135218, 0.0106224629959142, 0.6376551993416253},
      {0.0603575153489387, 0.0106224629959142, 0.6376551993416253},
      {0.0603575153489387, 0.2913648223135218, 0.0106224629959142},
      {0.2913648223135218, 0.0603575153489387, 0.0106224629959142},
      {0.6376551993416253, 0.2913648223135218, 0.0106224629959142},
      {0.6376551993416253, 0.0603575153489387, 0.0106224629959142},
      {0.2913648223135218, 0.6376551993416253, 0.0106224629959142},
      {0.0603575153489387, 0.6376551993416253, 0.0106224629959142},
      {0.1575062824084584, 0.5720629662753272, 0.2575352925182314},
      {0.1575062824084584, 0.2575352925182314, 0.5720629662753272},
      {0.5720629662753272, 0.1575062824084584, 0.2575352925182314},
      {0.2575352925182314, 0.1575062824084584, 0.5720629662753272},
      {0.5720629662753272, 0.2575352925182314, 0.1575062824084584},
      {0.2575352925182314, 0.5720629662753272, 0.1575062824084584},
      {0.0128954587979830, 0.5720629662753272, 0.2575352925182314},
      {0.0128954587979830, 0.2575352925182314, 0.5720629662753272},
      {0.0128954587979830, 0.1575062824084584, 0.2575352925182314},
      {0.0128954587979830, 0.1575062824084584, 0.5720629662753272},
      {0.0128954587979830, 0.2575352925182314, 0.1575062824084584},
      {0.0128954587979830, 0.5720629662753272, 0.1575062824084584},
      {0.5720629662753272, 0.0128954587979830, 0.2575352925182314},
      {0.2575352925182314, 0.0128954587979830, 0.5720629662753272},
      {0.1575062824084584, 0.0128954587979830, 0.2575352925182314},
      {0.1575062824084584, 0.0128954587979830, 0.5720629662753272},
      {0.2575352925182314, 0.0128954587979830, 0.1575062824084584},
      {0.5720629662753272, 0.0128954587979830, 0.1575062824084584},
      {0.5720629662753272, 0.2575352925182314, 0.0128954587979830},
      {0.2575352925182314, 0.5720629662753272, 0.0128954587979830},
      {0.1575062824084584, 0.2575352925182314, 0.0128954587979830},
      {0.1575062824084584, 0.5720629662753272, 0.0128954587979830},
      {0.2575352925182314, 0.1575062824084584, 0.0128954587979830},
      {0.5720629662753272, 0.1575062824084584, 0.0128954587979830},
      {0.1530963612221368, 0.5982871618499548, 0.1837258256088453},
      {0.1530963612221368, 0.1837258256088453, 0.5982871618499548},
      {0.5982871618499548, 0.1530963612221368, 0.1837258256088453},
      {0.1837258256088453, 0.1530963612221368, 0.5982871618499548},
      {0.5982871618499548, 0.1837258256088453, 0.1530963612221368},
      {0.1837258256088453, 0.5982871618499548, 0.1530963612221368},
      {0.0648906513190632, 0.5982871618499548, 0.1837258256088453},
      {0.0648906513190632, 0.1837258256088453, 0.5982871618499548},
      {0.0648906513190632, 0.1530963612221368, 0.1837258256088453},
      {0.0648906513190632, 0.1530963612221368, 0.5982871618499548},
      {0.0648906513190632, 0.1837258256088453, 0.1530963612221368},
      {0.0648906513190632, 0.5982871618499548, 0.1530963612221368},
      {0.5982871618499548, 0.0648906513190632, 0.1837258256088453},
      {0.1837258256088453, 0.0648906513190632, 0.5982871618499548},
      {0.1530963612221368, 0.0648906513190632, 0.1837258256088453},
      {0.1530963612221368, 0.0648906513190632, 0.5982871618499548},
      {0.1837258256088453, 0.0648906513190632, 0.1530963612221368},
      {0.5982871618499548, 0.0648906513190632, 0.1530963612221368},
      {0.5982871618499548, 0.1837258256088453, 0.0648906513190632},
      {0.1837258256088453, 0.5982871618499548, 0.0648906513190632},
      {0.1530963612221368, 0.1837258256088453, 0.0648906513190632},
      {0.1530963612221368, 0.5982871618499548, 0.0648906513190632},
      {0.1837258256088453, 0.1530963612221368, 0.0648906513190632},
      {0.5982871618499548, 0.1530963612221368, 0.0648906513190632},
      {0.0116550029704661, 0.0872107792177225, 0.8540565555742755},
      {0.0116550029704661, 0.8540565555742755, 0.0872107792177225},
      {0.0872107792177225, 0.0116550029704661, 0.8540565555742755},
      {0.8540565555742755, 0.0116550029704661, 0.0872107792177225},
      {0.0872107792177225, 0.8540565555742755, 0.0116550029704661},
      {0.8540565555742755, 0.0872107792177225, 0.0116550029704661},
      {0.0470776622375358, 0.0872107792177225, 0.8540565555742755},
      {0.0470776622375358, 0.8540565555742755, 0.0872107792177225},
      {0.0470776622375358, 0.0116550029704661, 0.8540565555742755},
      {0.0470776622375358, 0.0116550029704661, 0.0872107792177225},
      {0.0470776622375358, 0.8540565555742755, 0.0116550029704661},
      {0.0470776622375358, 0.0872107792177225, 0.0116550029704661},
      {0.0872107792177225, 0.0470776622375358, 0.8540565555742755},
      {0.8540565555742755, 0.0470776622375358, 0.0872107792177225},
      {0.0116550029704661, 0.0470776622375358, 0.8540565555742755},
      {0.0116550029704661, 0.0470776622375358, 0.0872107792177225},
      {0.8540565555742755, 0.0470776622375358, 0.0116550029704661},
      {0.0872107792177225, 0.0470776622375358, 0.0116550029704661},
      {0.0872107792177225, 0.8540565555742755, 0.0470776622375358},
      {0.8540565555742755, 0.0872107792177225, 0.0470776622375358},
      {0.0116550029704661, 0.8540565555742755, 0.0470776622375358},
      {0.0116550029704661, 0.0872107792177225, 0.0470776622375358},
      {0.8540565555742755, 0.0116550029704661, 0.0470776622375358},
      {0.0872107792177225, 0.0116550029704661, 0.0470776622375358},
      {0.4085328910163256, 0.4137799680661914, 0.1672070477173488},
      {0.4085328910163256, 0.1672070477173488, 0.4137799680661914},
      {0.4137799680661914, 0.4085328910163256, 0.1672070477173488},
      {0.1672070477173488, 0.4085328910163256, 0.4137799680661914},
      {0.4137799680661914, 0.1672070477173488, 0.4085328910163256},
      {0.1672070477173488, 0.4137799680661914, 0.4085328910163256},
      {0.0104800932001341, 0.4137799680661914, 0.1672070477173488},
      {0.0104800932001341, 0.1672070477173488, 0.4137799680661914},
      {0.0104800932001341, 0.4085328910163256, 0.1672070477173488},
      {0.0104800932001341, 0.4085328910163256, 0.4137799680661914},
      {0.0104800932001341, 0.1672070477173488, 0.4085328910163256},
      {0.0104800932001341, 0.4137799680661914, 0.4085328910163256},
      {0.4137799680661914, 0.0104800932001341, 0.1672070477173488},
      {0.1672070477173488, 0.0104800932001341, 0.4137799680661914},
      {0.4085328910163256, 0.0104800932001341, 0.1672070477173488},
      {0.4085328910163256, 0.0104800932001341, 0.4137799680661914},
      {0.1672070477173488, 0.0104800932001341, 0.4085328910163256},
      {0.4137799680661914, 0.0104800932001341, 0.4085328910163256},
      {0.4137799680661914, 0.1672070477173488, 0.0104800932001341},
      {0.1672070477173488, 0.4137799680661914, 0.0104800932001341},
      {0.4085328910163256, 0.1672070477173488, 0.0104800932001341},
      {0.4085328910163256, 0.4137799680661914, 0.0104800932001341},
      {0.1672070477173488, 0.4085328910163256, 0.0104800932001341},
      {0.4137799680661914, 0.4085328910163256, 0.0104800932001341},
      {0.0116793229924696, 0.3904989583719425, 0.5242157619428415},
      {0.0116793229924696, 0.5242157619428415, 0.3904989583719425},
      {0.3904989583719425, 0.0116793229924696, 0.5242157619428415},
      {0.5242157619428415, 0.0116793229924696, 0.3904989583719425},
      {0.3904989583719425, 0.5242157619428415, 0.0116793229924696},
      {0.5242157619428415, 0.3904989583719425, 0.0116793229924696},
      {0.0736059566927464, 0.3904989583719425, 0.5242157619428415},
      {0.0736059566927464, 0.5242157619428415, 0.3904989583719425},
      {0.0736059566927464, 0.0116793229924696, 0.5242157619428415},
      {0.0736059566927464, 0.0116793229924696, 0.3904989583719425},
      {0.0736059566927464, 0.5242157619428415, 0.0116793229924696},
      {0.0736059566927464, 0.3904989583719425, 0.0116793229924696},
      {0.3904989583719425, 0.0736059566927464, 0.5242157619428415},
      {0.5242157619428415, 0.0736059566927464, 0.3904989583719425},
      {0.0116793229924696, 0.0736059566927464, 0.5242157619428415},
      {0.0116793229924696, 0.0736059566927464, 0.3904989583719425},
      {0.5242157619428415, 0.0736059566927464, 0.0116793229924696},
      {0.3904989583719425, 0.0736059566927464, 0.0116793229924696},
      {0.3904989583719425, 0.5242157619428415, 0.0736059566927464},
      {0.5242157619428415, 0.3904989583719425, 0.0736059566927464},
      {0.0116793229924696, 0.5242157619428415, 0.0736059566927464},
      {0.0116793229924696, 0.3904989583719425, 0.0736059566927464},
      {0.5242157619428415, 0.0116793229924696, 0.0736059566927464},
      {0.3904989583719425, 0.0116793229924696, 0.0736059566927464},
      {0.0081636803712330, 0.7735017403534717, 0.1928802696509954},
      {0.0081636803712330, 0.1928802696509954, 0.7735017403534717},
      {0.7735017403534717, 0.0081636803712330, 0.1928802696509954},
      {0.1928802696509954, 0.0081636803712330, 0.7735017403534717},
      {0.7735017403534717, 0.1928802696509954, 0.0081636803712330},
      {0.1928802696509954, 0.7735017403534717, 0.0081636803712330},
      {0.0254543096242999, 0.7735017403534717, 0.1928802696509954},
      {0.0254543096242999, 0.1928802696509954, 0.7735017403534717},
      {0.0254543096242999, 0.0081636803712330, 0.1928802696509954},
      {0.0254543096242999, 0.0081636803712330, 0.7735017403534717},
      {0.0254543096242999, 0.1928802696509954, 0.0081636803712330},
      {0.0254543096242999, 0.7735017403534717, 0.0081636803712330},
      {0.7735017403534717, 0.0254543096242999, 0.1928802696509954},
      {0.1928802696509954, 0.0254543096242999, 0.7735017403534717},
      {0.0081636803712330, 0.0254543096242999, 0.1928802696509954},
      {0.0081636803712330, 0.0254543096242999, 0.7735017403534717},
      {0.1928802696509954, 0.0254543096242999, 0.0081636803712330},
      {0.7735017403534717, 0.0254543096242999, 0.0081636803712330},
      {0.7735017403534717, 0.1928802696509954, 0.0254543096242999},
      {0.1928802696509954, 0.7735017403534717, 0.0254543096242999},
      {0.0081636803712330, 0.1928802696509954, 0.0254543096242999},
      {0.0081636803712330, 0.7735017403534717, 0.0254543096242999},
      {0.1928802696509954, 0.0081636803712330, 0.0254543096242999},
      {0.7735017403534717, 0.0081636803712330, 0.0254543096242999},
      {0.1529886055031752, 0.0560344049440816, 0.4682783426267390},
      {0.1529886055031752, 0.4682783426267390, 0.0560344049440816},
      {0.0560344049440816, 0.1529886055031752, 0.4682783426267390},
      {0.4682783426267390, 0.1529886055031752, 0.0560344049440816},
      {0.0560344049440816, 0.4682783426267390, 0.1529886055031752},
      {0.4682783426267390, 0.0560344049440816, 0.1529886055031752},
      {0.3226986469260043, 0.0560344049440816, 0.4682783426267390},
      {0.3226986469260043, 0.4682783426267390, 0.0560344049440816},
      {0.3226986469260043, 0.1529886055031752, 0.4682783426267390},
      {0.3226986469260043, 0.1529886055031752, 0.0560344049440816},
      {0.3226986469260043, 0.4682783426267390, 0.1529886055031752},
      {0.3226986469260043, 0.0560344049440816, 0.1529886055031752},
      {0.0560344049440816, 0.3226986469260043, 0.4682783426267390},
      {0.4682783426267390, 0.3226986469260043, 0.0560344049440816},
      {0.1529886055031752, 0.3226986469260043, 0.4682783426267390},
      {0.1529886055031752, 0.3226986469260043, 0.0560344049440816},
      {0.4682783426267390, 0.3226986469260043, 0.1529886055031752},
      {0.0560344049440816, 0.3226986469260043, 0.1529886055031752},
      {0.0560344049440816, 0.4682783426267390, 0.3226986469260043},
      {0.4682783426267390, 0.0560344049440816, 0.3226986469260043},
      {0.1529886055031752, 0.4682783426267390, 0.3226986469260043},
      {0.1529886055031752, 0.0560344049440816, 0.3226986469260043},
      {0.4682783426267390, 0.1529886055031752, 0.3226986469260043},
      {0.0560344049440816, 0.1529886055031752, 0.3226986469260043}},
      {0.0004905685091138,
      0.0004905685091138,
      0.0004905685091138,
      0.0004905685091138,
      0.0000803542427022,
      0.0000803542427022,
      0.0000803542427022,
      0.0000803542427022,
      0.0002722353137227,
      0.0002722353137227,
      0.0002722353137227,
      0.0002722353137227,
      0.0014895434241223,
      0.0014895434241223,
      0.0014895434241223,
      0.0014895434241223,
      0.0005898801735219,
      0.0005898801735219,
      0.0005898801735219,
      0.0005898801735219,
      0.0005898801735219,
      0.0005898801735219,
      0.0008546461291872,
      0.0008546461291872,
      0.0008546461291872,
      0.0008546461291872,
      0.0008546461291872,
      0.0008546461291872,
      0.0014176602842961,
      0.0014176602842961,
      0.0014176602842961,
      0.0014176602842961,
      0.0014176602842961,
      0.0014176602842961,
      0.0002270001765942,
      0.0002270001765942,
      0.0002270001765942,
      0.0002270001765942,
      0.0002270001765942,
      0.0002270001765942,
      0.0003254505991165,
      0.0003254505991165,
      0.0003254505991165,
      0.0003254505991165,
      0.0003254505991165,
      0.0003254505991165,
      0.0003254505991165,
      0.0003254505991165,
      0.0003254505991165,
      0.0003254505991165,
      0.0003254505991165,
      0.0003254505991165,
      0.0005920803887837,
      0.0005920803887837,
      0.0005920803887837,
      0.0005920803887837,
      0.0005920803887837,
      0.0005920803887837,
      0.0005920803887837,
      0.0005920803887837,
      0.0005920803887837,
      0.0005920803887837,
      0.0005920803887837,
      0.0005920803887837,
      0.0007891729786813,
      0.0007891729786813,
      0.0007891729786813,
      0.0007891729786813,
      0.0007891729786813,
      0.0007891729786813,
      0.0007891729786813,
      0.0007891729786813,
      0.0007891729786813,
      0.0007891729786813,
      0.0007891729786813,
      0.0007891729786813,
      0.0006793673420873,
      0.0006793673420873,
      0.0006793673420873,
      0.0006793673420873,
      0.0006793673420873,
      0.0006793673420873,
      0.0006793673420873,
      0.0006793673420873,
      0.0006793673420873,
      0.0006793673420873,
      0.0006793673420873,
      0.0006793673420873,
      0.0007329816026731,
      0.0007329816026731,
      0.0007329816026731,
      0.0007329816026731,
      0.0007329816026731,
      0.0007329816026731,
      0.0007329816026731,
      0.0007329816026731,
      0.0007329816026731,
      0.0007329816026731,
      0.0007329816026731,
      0.0007329816026731,
      0.0001359985903007,
      0.0001359985903007,
      0.0001359985903007,
      0.0001359985903007,
      0.0001359985903007,
      0.0001359985903007,
      0.0001359985903007,
      0.0001359985903007,
      0.0001359985903007,
      0.0001359985903007,
      0.0001359985903007,
      0.0001359985903007,
      0.0010728212338927,
      0.0010728212338927,
      0.0010728212338927,
      0.0010728212338927,
      0.0010728212338927,
      0.0010728212338927,
      0.0010728212338927,
      0.0010728212338927,
      0.0010728212338927,
      0.0010728212338927,
      0.0010728212338927,
      0.0010728212338927,
      0.0000381822838330,
      0.0000381822838330,
      0.0000381822838330,
      0.0000381822838330,
      0.0000381822838330,
      0.0000381822838330,
      0.0000381822838330,
      0.0000381822838330,
      0.0000381822838330,
      0.0000381822838330,
      0.0000381822838330,
      0.0000381822838330,
      0.0005314577823675,
      0.0005314577823675,
      0.0005314577823675,
      0.0005314577823675,
      0.0005314577823675,
      0.0005314577823675,
      0.0005314577823675,
      0.0005314577823675,
      0.0005314577823675,
      0.0005314577823675,
      0.0005314577823675,
      0.0005314577823675,
      0.0003318777099950,
      0.0003318777099950,
      0.0003318777099950,
      0.0003318777099950,
      0.0003318777099950,
      0.0003318777099950,
      0.0003318777099950,
      0.0003318777099950,
      0.0003318777099950,
      0.0003318777099950,
      0.0003318777099950,
      0.0003318777099950,
      0.0003318777099950,
      0.0003318777099950,
      0.0003318777099950,
      0.0003318777099950,
      0.0003318777099950,
      0.0003318777099950,
      0.0003318777099950,
      0.0003318777099950,
      0.0003318777099950,
      0.0003318777099950,
      0.0003318777099950,
      0.0003318777099950,
      0.0002016577728115,
      0.0002016577728115,
      0.0002016577728115,
      0.0002016577728115,
      0.0002016577728115,
      0.0002016577728115,
      0.0002016577728115,
      0.0002016577728115,
      0.0002016577728115,
      0.0002016577728115,
      0.0002016577728115,
      0.0002016577728115,
      0.0002016577728115,
      0.0002016577728115,
      0.0002016577728115,
      0.0002016577728115,
      0.0002016577728115,
      0.0002016577728115,
      0.0002016577728115,
      0.0002016577728115,
      0.0002016577728115,
      0.0002016577728115,
      0.0002016577728115,
      0.0002016577728115,
      0.0004905696636973,
      0.0004905696636973,
      0.0004905696636973,
      0.0004905696636973,
      0.0004905696636973,
      0.0004905696636973,
      0.0004905696636973,
      0.0004905696636973,
      0.0004905696636973,
      0.0004905696636973,
      0.0004905696636973,
      0.0004905696636973,
      0.0004905696636973,
      0.0004905696636973,
      0.0004905696636973,
      0.0004905696636973,
      0.0004905696636973,
      0.0004905696636973,
      0.0004905696636973,
      0.0004905696636973,
      0.0004905696636973,
      0.0004905696636973,
      0.0004905696636973,
      0.0004905696636973,
      0.0005756824258018,
      0.0005756824258018,
      0.0005756824258018,
      0.0005756824258018,
      0.0005756824258018,
      0.0005756824258018,
      0.0005756824258018,
      0.0005756824258018,
      0.0005756824258018,
      0.0005756824258018,
      0.0005756824258018,
      0.0005756824258018,
      0.0005756824258018,
      0.0005756824258018,
      0.0005756824258018,
      0.0005756824258018,
      0.0005756824258018,
      0.0005756824258018,
      0.0005756824258018,
      0.0005756824258018,
      0.0005756824258018,
      0.0005756824258018,
      0.0005756824258018,
      0.0005756824258018,
      0.0001373522569130,
      0.0001373522569130,
      0.0001373522569130,
      0.0001373522569130,
      0.0001373522569130,
      0.0001373522569130,
      0.0001373522569130,
      0.0001373522569130,
      0.0001373522569130,
      0.0001373522569130,
      0.0001373522569130,
      0.0001373522569130,
      0.0001373522569130,
      0.0001373522569130,
      0.0001373522569130,
      0.0001373522569130,
      0.0001373522569130,
      0.0001373522569130,
      0.0001373522569130,
      0.0001373522569130,
      0.0001373522569130,
      0.0001373522569130,
      0.0001373522569130,
      0.0001373522569130,
      0.0002305903963617,
      0.0002305903963617,
      0.0002305903963617,
      0.0002305903963617,
      0.0002305903963617,
      0.0002305903963617,
      0.0002305903963617,
      0.0002305903963617,
      0.0002305903963617,
      0.0002305903963617,
      0.0002305903963617,
      0.0002305903963617,
      0.0002305903963617,
      0.0002305903963617,
      0.0002305903963617,
      0.0002305903963617,
      0.0002305903963617,
      0.0002305903963617,
      0.0002305903963617,
      0.0002305903963617,
      0.0002305903963617,
      0.0002305903963617,
      0.0002305903963617,
      0.0002305903963617,
      0.0002969075393211,
      0.0002969075393211,
      0.0002969075393211,
      0.0002969075393211,
      0.0002969075393211,
      0.0002969075393211,
      0.0002969075393211,
      0.0002969075393211,
      0.0002969075393211,
      0.0002969075393211,
      0.0002969075393211,
      0.0002969075393211,
      0.0002969075393211,
      0.0002969075393211,
      0.0002969075393211,
      0.0002969075393211,
      0.0002969075393211,
      0.0002969075393211,
      0.0002969075393211,
      0.0002969075393211,
      0.0002969075393211,
      0.0002969075393211,
      0.0002969075393211,
      0.0002969075393211,
      0.0001161552725167,
      0.0001161552725167,
      0.0001161552725167,
      0.0001161552725167,
      0.0001161552725167,
      0.0001161552725167,
      0.0001161552725167,
      0.0001161552725167,
      0.0001161552725167,
      0.0001161552725167,
      0.0001161552725167,
      0.0001161552725167,
      0.0001161552725167,
      0.0001161552725167,
      0.0001161552725167,
      0.0001161552725167,
      0.0001161552725167,
      0.0001161552725167,
      0.0001161552725167,
      0.0001161552725167,
      0.0001161552725167,
      0.0001161552725167,
      0.0001161552725167,
      0.0001161552725167,
      0.0009538147336486,
      0.0009538147336486,
      0.0009538147336486,
      0.0009538147336486,
      0.0009538147336486,
      0.0009538147336486,
      0.0009538147336486,
      0.0009538147336486,
      0.0009538147336486,
      0.0009538147336486,
      0.0009538147336486,
      0.0009538147336486,
      0.0009538147336486,
      0.0009538147336486,
      0.0009538147336486,
      0.0009538147336486,
      0.0009538147336486,
      0.0009538147336486,
      0.0009538147336486,
      0.0009538147336486,
      0.0009538147336486,
      0.0009538147336486,
      0.0009538147336486,
      0.0009538147336486}
    },
    {
      436,
      {{0.2063721818681210, 0.2063721818681210, 0.2063721818681210},
      {0.3808834543956371, 0.2063721818681210, 0.2063721818681210},
      {0.2063721818681210, 0.3808834543956371, 0.2063721818681210},
      {0.2063721818681210, 0.2063721818681210, 0.3808834543956371},
      {0.1625387945176406, 0.1625387945176406, 0.1625387945176406},
      {0.5123836164470781, 0.1625387945176406, 0.1625387945176406},
      {0.1625387945176406, 0.5123836164470781, 0.1625387945176406},
      {0.1625387945176406, 0.1625387945176406, 0.5123836164470781},
      {0.3116266728255643, 0.3116266728255643, 0.3116266728255643},
      {0.0651199815233071, 0.3116266728255643, 0.3116266728255643},
      {0.3116266728255643, 0.0651199815233071, 0.3116266728255643},
      {0.3116266728255643, 0.3116266728255643, 0.0651199815233071},
      {0.3307279160890313, 0.3307279160890313, 0.3307279160890313},
      {0.0078162517329061, 0.3307279160890313, 0.3307279160890313},
      {0.3307279160890313, 0.0078162517329061, 0.3307279160890313},
      {0.3307279160890313, 0.3307279160890313, 0.0078162517329061},
      {0.0337440820129951, 0.0337440820129951, 0.0337440820129951},
      {0.8987677539610148, 0.0337440820129951, 0.0337440820129951},
      {0.0337440820129951, 0.8987677539610148, 0.0337440820129951},
      {0.0337440820129951, 0.0337440820129951, 0.8987677539610148},
      {0.0656813827842595, 0.0656813827842595, 0.0656813827842595},
      {0.8029558516472214, 0.0656813827842595, 0.0656813827842595},
      {0.0656813827842595, 0.8029558516472214, 0.0656813827842595},
      {0.0656813827842595, 0.0656813827842595, 0.8029558516472214},
      {0.0116907242025971, 0.0116907242025971, 0.0116907242025971},
      {0.9649278273922087, 0.0116907242025971, 0.0116907242025971},
      {0.0116907242025971, 0.9649278273922087, 0.0116907242025971},
      {0.0116907242025971, 0.0116907242025971, 0.9649278273922087},
      {0.1948270858785206, 0.3051729141214794, 0.3051729141214794},
      {0.3051729141214794, 0.1948270858785206, 0.3051729141214794},
      {0.1948270858785206, 0.1948270858785206, 0.3051729141214794},
      {0.3051729141214794, 0.3051729141214794, 0.1948270858785206},
      {0.1948270858785206, 0.3051729141214794, 0.1948270858785206},
      {0.3051729141214794, 0.1948270858785206, 0.1948270858785206},
      {0.2501828346584600, 0.2498171653415399, 0.2498171653415399},
      {0.2498171653415399, 0.2501828346584600, 0.2498171653415399},
      {0.2501828346584600, 0.2501828346584600, 0.2498171653415399},
      {0.2498171653415399, 0.2498171653415399, 0.2501828346584600},
      {0.2501828346584600, 0.2498171653415399, 0.2501828346584600},
      {0.2498171653415399, 0.2501828346584600, 0.2501828346584600},
      {0.4866334770839438, 0.0133665229160562, 0.0133665229160562},
      {0.0133665229160562, 0.4866334770839438, 0.0133665229160562},
      {0.4866334770839438, 0.4866334770839438, 0.0133665229160562},
      {0.0133665229160562, 0.0133665229160562, 0.4866334770839438},
      {0.4866334770839438, 0.0133665229160562, 0.4866334770839438},
      {0.0133665229160562, 0.4866334770839438, 0.4866334770839438},
      {0.2496576564241145, 0.2503423435758855, 0.2503423435758855},
      {0.2503423435758855, 0.2496576564241145, 0.2503423435758855},
      {0.2496576564241145, 0.2496576564241145, 0.2503423435758855},
      {0.2503423435758855, 0.2503423435758855, 0.2496576564241145},
      {0.2496576564241145, 0.2503423435758855, 0.2496576564241145},
      {0.2503423435758855, 0.2496576564241145, 0.2496576564241145},
      {0.1675482277457161, 0.3324517722542840, 0.3324517722542840},
      {0.3324517722542840, 0.1675482277457161, 0.3324517722542840},
      {0.1675482277457161, 0.1675482277457161, 0.3324517722542840},
      {0.3324517722542840, 0.3324517722542840, 0.1675482277457161},
      {0.1675482277457161, 0.3324517722542840, 0.1675482277457161},
      {0.3324517722542840, 0.1675482277457161, 0.1675482277457161},
      {0.1671064025294534, 0.3328935974705466, 0.3328935974705466},
      {0.3328935974705466, 0.1671064025294534, 0.3328935974705466},
      {0.1671064025294534, 0.1671064025294534, 0.3328935974705466},
      {0.3328935974705466, 0.3328935974705466, 0.1671064025294534},
      {0.1671064025294534, 0.3328935974705466, 0.1671064025294534},
      {0.3328935974705466, 0.1671064025294534, 0.1671064025294534},
      {0.3946110017058707, 0.1053889982941293, 0.1053889982941293},
      {0.1053889982941293, 0.3946110017058707, 0.1053889982941293},
      {0.3946110017058707, 0.3946110017058707, 0.1053889982941293},
      {0.1053889982941293, 0.1053889982941293, 0.3946110017058707},
      {0.3946110017058707, 0.1053889982941293, 0.3946110017058707},
      {0.1053889982941293, 0.3946110017058707, 0.3946110017058707},
      {0.4568227805863351, 0.0431772194136649, 0.0431772194136649},
      {0.0431772194136649, 0.4568227805863351, 0.0431772194136649},
      {0.4568227805863351, 0.4568227805863351, 0.0431772194136649},
      {0.0431772194136649, 0.0431772194136649, 0.4568227805863351},
      {0.4568227805863351, 0.0431772194136649, 0.4568227805863351},
      {0.0431772194136649, 0.4568227805863351, 0.4568227805863351},
      {0.1352754021303201, 0.0331030676150443, 0.0331030676150443},
      {0.0331030676150443, 0.1352754021303201, 0.0331030676150443},
      {0.0331030676150443, 0.0331030676150443, 0.1352754021303201},
      {0.7985184626395914, 0.0331030676150443, 0.0331030676150443},
      {0.7985184626395914, 0.1352754021303201, 0.0331030676150443},
      {0.7985184626395914, 0.0331030676150443, 0.1352754021303201},
      {0.0331030676150443, 0.7985184626395914, 0.0331030676150443},
      {0.1352754021303201, 0.7985184626395914, 0.0331030676150443},
      {0.0331030676150443, 0.7985184626395914, 0.1352754021303201},
      {0.0331030676150443, 0.0331030676150443, 0.7985184626395914},
      {0.1352754021303201, 0.0331030676150443, 0.7985184626395914},
      {0.0331030676150443, 0.1352754021303201, 0.7985184626395914},
      {0.0011718790470473, 0.1946185784467675, 0.1946185784467675},
      {0.1946185784467675, 0.0011718790470473, 0.1946185784467675},
      {0.1946185784467675, 0.1946185784467675, 0.0011718790470473},
      {0.6095909640594178, 0.1946185784467675, 0.1946185784467675},
      {0.6095909640594178, 0.0011718790470473, 0.1946185784467675},
      {0.6095909640594178, 0.1946185784467675, 0.0011718790470473},
      {0.1946185784467675, 0.6095909640594178, 0.1946185784467675},
      {0.0011718790470473, 0.6095909640594178, 0.1946185784467675},
      {0.1946185784467675, 0.6095909640594178, 0.0011718790470473},
      {0.1946185784467675, 0.1946185784467675, 0.6095909640594178},
      {0.0011718790470473, 0.1946185784467675, 0.6095909640594178},
      {0.1946185784467675, 0.0011718790470473, 0.6095909640594178},
      {0.3278960098149503, 0.0132125212613112, 0.0132125212613112},
      {0.0132125212613112, 0.3278960098149503, 0.0132125212613112},
      {0.0132125212613112, 0.0132125212613112, 0.3278960098149503},
      {0.6456789476624273, 0.0132125212613112, 0.0132125212613112},
      {0.6456789476624273, 0.3278960098149503, 0.0132125212613112},
      {0.6456789476624273, 0.0132125212613112, 0.3278960098149503},
      {0.0132125212613112, 0.6456789476624273, 0.0132125212613112},
      {0.3278960098149503, 0.6456789476624273, 0.0132125212613112},
      {0.0132125212613112, 0.6456789476624273, 0.3278960098149503},
      {0.0132125212613112, 0.0132125212613112, 0.6456789476624273},
      {0.3278960098149503, 0.0132125212613112, 0.6456789476624273},
      {0.0132125212613112, 0.3278960098149503, 0.6456789476624273},
      {0.0202839984315217, 0.1229320721770598, 0.1229320721770598},
      {0.1229320721770598, 0.0202839984315217, 0.1229320721770598},
      {0.1229320721770598, 0.1229320721770598, 0.0202839984315217},
      {0.7338518572143586, 0.1229320721770598, 0.1229320721770598},
      {0.7338518572143586, 0.0202839984315217, 0.1229320721770598},
      {0.7338518572143586, 0.1229320721770598, 0.0202839984315217},
      {0.1229320721770598, 0.7338518572143586, 0.1229320721770598},
      {0.0202839984315217, 0.7338518572143586, 0.1229320721770598},
      {0.1229320721770598, 0.7338518572143586, 0.0202839984315217},
      {0.1229320721770598, 0.1229320721770598, 0.7338518572143586},
      {0.0202839984315217, 0.1229320721770598, 0.7338518572143586},
      {0.1229320721770598, 0.0202839984315217, 0.7338518572143586},
      {0.3498961219623653, 0.0619950609516838, 0.0619950609516838},
      {0.0619950609516838, 0.3498961219623653, 0.0619950609516838},
      {0.0619950609516838, 0.0619950609516838, 0.3498961219623653},
      {0.5261137561342671, 0.0619950609516838, 0.0619950609516838},
      {0.5261137561342671, 0.3498961219623653, 0.0619950609516838},
      {0.5261137561342671, 0.0619950609516838, 0.3498961219623653},
      {0.0619950609516838, 0.5261137561342671, 0.0619950609516838},
      {0.3498961219623653, 0.5261137561342671, 0.0619950609516838},
      {0.0619950609516838, 0.5261137561342671, 0.3498961219623653},
      {0.0619950609516838, 0.0619950609516838, 0.5261137561342671},
      {0.3498961219623653, 0.0619950609516838, 0.5261137561342671},
      {0.0619950609516838, 0.3498961219623653, 0.5261137561342671},
      {0.1666830796952924, 0.0840718754699170, 0.0840718754699170},
      {0.0840718754699170, 0.1666830796952924, 0.0840718754699170},
      {0.0840718754699170, 0.0840718754699170, 0.1666830796952924},
      {0.6651731693648737, 0.0840718754699170, 0.0840718754699170},
      {0.6651731693648737, 0.1666830796952924, 0.0840718754699170},
      {0.6651731693648737, 0.0840718754699170, 0.1666830796952924},
      {0.0840718754699170, 0.6651731693648737, 0.0840718754699170},
      {0.1666830796952924, 0.6651731693648737, 0.0840718754699170},
      {0.0840718754699170, 0.6651731693648737, 0.1666830796952924},
      {0.0840718754699170, 0.0840718754699170, 0.6651731693648737},
      {0.1666830796952924, 0.0840718754699170, 0.6651731693648737},
      {0.0840718754699170, 0.1666830796952924, 0.6651731693648737},
      {0.1859179591228751, 0.0078670488744539, 0.0078670488744539},
      {0.0078670488744539, 0.1859179591228751, 0.0078670488744539},
      {0.0078670488744539, 0.0078670488744539, 0.1859179591228751},
      {0.7983479431282171, 0.0078670488744539, 0.0078670488744539},
      {0.7983479431282171, 0.1859179591228751, 0.0078670488744539},
      {0.7983479431282171, 0.0078670488744539, 0.1859179591228751},
      {0.0078670488744539, 0.7983479431282171, 0.0078670488744539},
      {0.1859179591228751, 0.7983479431282171, 0.0078670488744539},
      {0.0078670488744539, 0.7983479431282171, 0.1859179591228751},
      {0.0078670488744539, 0.0078670488744539, 0.7983479431282171},
      {0.1859179591228751, 0.0078670488744539, 0.7983479431282171},
      {0.0078670488744539, 0.1859179591228751, 0.7983479431282171},
      {0.2482707411503117, 0.0421758249887124, 0.0421758249887124},
      {0.0421758249887124, 0.2482707411503117, 0.0421758249887124},
      {0.0421758249887124, 0.0421758249887124, 0.2482707411503117},
      {0.6673776088722636, 0.0421758249887124, 0.0421758249887124},
      {0.6673776088722636, 0.2482707411503117, 0.0421758249887124},
      {0.6673776088722636, 0.0421758249887124, 0.2482707411503117},
      {0.0421758249887124, 0.6673776088722636, 0.0421758249887124},
      {0.2482707411503117, 0.6673776088722636, 0.0421758249887124},
      {0.0421758249887124, 0.6673776088722636, 0.2482707411503117},
      {0.0421758249887124, 0.0421758249887124, 0.6673776088722636},
      {0.2482707411503117, 0.0421758249887124, 0.6673776088722636},
      {0.0421758249887124, 0.2482707411503117, 0.6673776088722636},
      {0.0669810222676876, 0.0086834918721388, 0.0086834918721388},
      {0.0086834918721388, 0.0669810222676876, 0.0086834918721388},
      {0.0086834918721388, 0.0086834918721388, 0.0669810222676876},
      {0.9156519939880347, 0.0086834918721388, 0.0086834918721388},
      {0.9156519939880347, 0.0669810222676876, 0.0086834918721388},
      {0.9156519939880347, 0.0086834918721388, 0.0669810222676876},
      {0.0086834918721388, 0.9156519939880347, 0.0086834918721388},
      {0.0669810222676876, 0.9156519939880347, 0.0086834918721388},
      {0.0086834918721388, 0.9156519939880347, 0.0669810222676876},
      {0.0086834918721388, 0.0086834918721388, 0.9156519939880347},
      {0.0669810222676876, 0.0086834918721388, 0.9156519939880347},
      {0.0086834918721388, 0.0669810222676876, 0.9156519939880347},
      {0.2918417077981162, 0.1111185367576208, 0.1111185367576208},
      {0.1111185367576208, 0.2918417077981162, 0.1111185367576208},
      {0.1111185367576208, 0.1111185367576208, 0.2918417077981162},
      {0.4859212186866421, 0.1111185367576208, 0.1111185367576208},
      {0.4859212186866421, 0.2918417077981162, 0.1111185367576208},
      {0.4859212186866421, 0.1111185367576208, 0.2918417077981162},
      {0.1111185367576208, 0.4859212186866421, 0.1111185367576208},
      {0.2918417077981162, 0.4859212186866421, 0.1111185367576208},
      {0.1111185367576208, 0.4859212186866421, 0.2918417077981162},
      {0.1111185367576208, 0.1111185367576208, 0.4859212186866421},
      {0.2918417077981162, 0.1111185367576208, 0.4859212186866421},
      {0.1111185367576208, 0.2918417077981162, 0.4859212186866421},
      {0.0637707896451841, 0.0036854638456060, 0.7163454647076797},
      {0.0637707896451841, 0.7163454647076797, 0.0036854638456060},
      {0.0036854638456060, 0.0637707896451841, 0.7163454647076797},
      {0.7163454647076797, 0.0637707896451841, 0.0036854638456060},
      {0.0036854638456060, 0.7163454647076797, 0.0637707896451841},
      {0.7163454647076797, 0.0036854638456060, 0.0637707896451841},
      {0.2161982818015302, 0.0036854638456060, 0.7163454647076797},
      {0.2161982818015302, 0.7163454647076797, 0.0036854638456060},
      {0.2161982818015302, 0.0637707896451841, 0.7163454647076797},
      {0.2161982818015302, 0.0637707896451841, 0.0036854638456060},
      {0.2161982818015302, 0.7163454647076797, 0.0637707896451841},
      {0.2161982818015302, 0.0036854638456060, 0.0637707896451841},
      {0.0036854638456060, 0.2161982818015302, 0.7163454647076797},
      {0.7163454647076797, 0.2161982818015302, 0.0036854638456060},
      {0.0637707896451841, 0.2161982818015302, 0.7163454647076797},
      {0.0637707896451841, 0.2161982818015302, 0.0036854638456060},
      {0.7163454647076797, 0.2161982818015302, 0.0637707896451841},
      {0.0036854638456060, 0.2161982818015302, 0.0637707896451841},
      {0.0036854638456060, 0.7163454647076797, 0.2161982818015302},
      {0.7163454647076797, 0.0036854638456060, 0.2161982818015302},
      {0.0637707896451841, 0.7163454647076797, 0.2161982818015302},
      {0.0637707896451841, 0.0036854638456060, 0.2161982818015302},
      {0.7163454647076797, 0.0637707896451841, 0.2161982818015302},
      {0.0036854638456060, 0.0637707896451841, 0.2161982818015302},
      {0.0040240981032238, 0.0914756397949307, 0.8558399144225498},
      {0.0040240981032238, 0.8558399144225498, 0.0914756397949307},
      {0.0914756397949307, 0.0040240981032238, 0.8558399144225498},
      {0.8558399144225498, 0.0040240981032238, 0.0914756397949307},
      {0.0914756397949307, 0.8558399144225498, 0.0040240981032238},
      {0.8558399144225498, 0.0914756397949307, 0.0040240981032238},
      {0.0486603476792957, 0.0914756397949307, 0.8558399144225498},
      {0.0486603476792957, 0.8558399144225498, 0.0914756397949307},
      {0.0486603476792957, 0.0040240981032238, 0.8558399144225498},
      {0.0486603476792957, 0.0040240981032238, 0.0914756397949307},
      {0.0486603476792957, 0.8558399144225498, 0.0040240981032238},
      {0.0486603476792957, 0.0914756397949307, 0.0040240981032238},
      {0.0914756397949307, 0.0486603476792957, 0.8558399144225498},
      {0.8558399144225498, 0.0486603476792957, 0.0914756397949307},
      {0.0040240981032238, 0.0486603476792957, 0.8558399144225498},
      {0.0040240981032238, 0.0486603476792957, 0.0914756397949307},
      {0.8558399144225498, 0.0486603476792957, 0.0040240981032238},
      {0.0914756397949307, 0.0486603476792957, 0.0040240981032238},
      {0.0914756397949307, 0.8558399144225498, 0.0486603476792957},
      {0.8558399144225498, 0.0914756397949307, 0.0486603476792957},
      {0.0040240981032238, 0.8558399144225498, 0.0486603476792957},
      {0.0040240981032238, 0.0914756397949307, 0.0486603476792957},
      {0.8558399144225498, 0.0040240981032238, 0.0486603476792957},
      {0.0914756397949307, 0.0040240981032238, 0.0486603476792957},
      {0.4168774597385260, 0.1326563740184036, 0.4223229603801267},
      {0.4168774597385260, 0.4223229603801267, 0.1326563740184036},
      {0.1326563740184036, 0.4168774597385260, 0.4223229603801267},
      {0.4223229603801267, 0.4168774597385260, 0.1326563740184036},
      {0.1326563740184036, 0.4223229603801267, 0.4168774597385260},
      {0.4223229603801267, 0.1326563740184036, 0.4168774597385260},
      {0.0281432058629436, 0.1326563740184036, 0.4223229603801267},
      {0.0281432058629436, 0.4223229603801267, 0.1326563740184036},
      {0.0281432058629436, 0.4168774597385260, 0.4223229603801267},
      {0.0281432058629436, 0.4168774597385260, 0.1326563740184036},
      {0.0281432058629436, 0.4223229603801267, 0.4168774597385260},
      {0.0281432058629436, 0.1326563740184036, 0.4168774597385260},
      {0.1326563740184036, 0.0281432058629436, 0.4223229603801267},
      {0.4223229603801267, 0.0281432058629436, 0.1326563740184036},
      {0.4168774597385260, 0.0281432058629436, 0.4223229603801267},
      {0.4168774597385260, 0.0281432058629436, 0.1326563740184036},
      {0.4223229603801267, 0.0281432058629436, 0.4168774597385260},
      {0.1326563740184036, 0.0281432058629436, 0.4168774597385260},
      {0.1326563740184036, 0.4223229603801267, 0.0281432058629436},
      {0.4223229603801267, 0.1326563740184036, 0.0281432058629436},
      {0.4168774597385260, 0.4223229603801267, 0.0281432058629436},
      {0.4168774597385260, 0.1326563740184036, 0.0281432058629436},
      {0.4223229603801267, 0.4168774597385260, 0.0281432058629436},
      {0.1326563740184036, 0.4168774597385260, 0.0281432058629436},
      {0.2464330224801677, 0.4439198669278809, 0.2804526765357552},
      {0.2464330224801677, 0.2804526765357552, 0.4439198669278809},
      {0.4439198669278809, 0.2464330224801677, 0.2804526765357552},
      {0.2804526765357552, 0.2464330224801677, 0.4439198669278809},
      {0.4439198669278809, 0.2804526765357552, 0.2464330224801677},
      {0.2804526765357552, 0.4439198669278809, 0.2464330224801677},
      {0.0291944340561961, 0.4439198669278809, 0.2804526765357552},
      {0.0291944340561961, 0.2804526765357552, 0.4439198669278809},
      {0.0291944340561961, 0.2464330224801677, 0.2804526765357552},
      {0.0291944340561961, 0.2464330224801677, 0.4439198669278809},
      {0.0291944340561961, 0.2804526765357552, 0.2464330224801677},
      {0.0291944340561961, 0.4439198669278809, 0.2464330224801677},
      {0.4439198669278809, 0.0291944340561961, 0.2804526765357552},
      {0.2804526765357552, 0.0291944340561961, 0.4439198669278809},
      {0.2464330224801677, 0.0291944340561961, 0.2804526765357552},
      {0.2464330224801677, 0.0291944340561961, 0.4439198669278809},
      {0.2804526765357552, 0.0291944340561961, 0.2464330224801677},
      {0.4439198669278809, 0.0291944340561961, 0.2464330224801677},
      {0.4439198669278809, 0.2804526765357552, 0.0291944340561961},
      {0.2804526765357552, 0.4439198669278809, 0.0291944340561961},
      {0.2464330224801677, 0.2804526765357552, 0.0291944340561961},
      {0.2464330224801677, 0.4439198669278809, 0.0291944340561961},
      {0.2804526765357552, 0.2464330224801677, 0.0291944340561961},
      {0.4439198669278809, 0.2464330224801677, 0.0291944340561961},
      {0.1271207488070096, 0.2395450885895190, 0.3875767128917152},
      {0.1271207488070096, 0.3875767128917152, 0.2395450885895190},
      {0.2395450885895190, 0.1271207488070096, 0.3875767128917152},
      {0.3875767128917152, 0.1271207488070096, 0.2395450885895190},
      {0.2395450885895190, 0.3875767128917152, 0.1271207488070096},
      {0.3875767128917152, 0.2395450885895190, 0.1271207488070096},
      {0.2457574497117561, 0.2395450885895190, 0.3875767128917152},
      {0.2457574497117561, 0.3875767128917152, 0.2395450885895190},
      {0.2457574497117561, 0.1271207488070096, 0.3875767128917152},
      {0.2457574497117561, 0.1271207488070096, 0.2395450885895190},
      {0.2457574497117561, 0.3875767128917152, 0.1271207488070096},
      {0.2457574497117561, 0.2395450885895190, 0.1271207488070096},
      {0.2395450885895190, 0.2457574497117561, 0.3875767128917152},
      {0.3875767128917152, 0.2457574497117561, 0.2395450885895190},
      {0.1271207488070096, 0.2457574497117561, 0.3875767128917152},
      {0.1271207488070096, 0.2457574497117561, 0.2395450885895190},
      {0.3875767128917152, 0.2457574497117561, 0.1271207488070096},
      {0.2395450885895190, 0.2457574497117561, 0.1271207488070096},
      {0.2395450885895190, 0.3875767128917152, 0.2457574497117561},
      {0.3875767128917152, 0.2395450885895190, 0.2457574497117561},
      {0.1271207488070096, 0.3875767128917152, 0.2457574497117561},
      {0.1271207488070096, 0.2395450885895190, 0.2457574497117561},
      {0.3875767128917152, 0.1271207488070096, 0.2457574497117561},
      {0.2395450885895190, 0.1271207488070096, 0.2457574497117561},
      {0.2137049061861850, 0.1657130887168779, 0.5503423585599962},
      {0.2137049061861850, 0.5503423585599962, 0.1657130887168779},
      {0.1657130887168779, 0.2137049061861850, 0.5503423585599962},
      {0.5503423585599962, 0.2137049061861850, 0.1657130887168779},
      {0.1657130887168779, 0.5503423585599962, 0.2137049061861850},
      {0.5503423585599962, 0.1657130887168779, 0.2137049061861850},
      {0.0702396465369408, 0.1657130887168779, 0.5503423585599962},
      {0.0702396465369408, 0.5503423585599962, 0.1657130887168779},
      {0.0702396465369408, 0.2137049061861850, 0.5503423585599962},
      {0.0702396465369408, 0.2137049061861850, 0.1657130887168779},
      {0.0702396465369408, 0.5503423585599962, 0.2137049061861850},
      {0.0702396465369408, 0.1657130887168779, 0.2137049061861850},
      {0.1657130887168779, 0.0702396465369408, 0.5503423585599962},
      {0.5503423585599962, 0.0702396465369408, 0.1657130887168779},
      {0.2137049061861850, 0.0702396465369408, 0.5503423585599962},
      {0.2137049061861850, 0.0702396465369408, 0.1657130887168779},
      {0.5503423585599962, 0.0702396465369408, 0.2137049061861850},
      {0.1657130887168779, 0.0702396465369408, 0.2137049061861850},
      {0.1657130887168779, 0.5503423585599962, 0.0702396465369408},
      {0.5503423585599962, 0.1657130887168779, 0.0702396465369408},
      {0.2137049061861850, 0.5503423585599962, 0.0702396465369408},
      {0.2137049061861850, 0.1657130887168779, 0.0702396465369408},
      {0.5503423585599962, 0.2137049061861850, 0.0702396465369408},
      {0.1657130887168779, 0.2137049061861850, 0.0702396465369408},
      {0.3290353014877974, 0.1788776686026366, 0.4254763922480989},
      {0.3290353014877974, 0.4254763922480989, 0.1788776686026366},
      {0.1788776686026366, 0.3290353014877974, 0.4254763922480989},
      {0.4254763922480989, 0.3290353014877974, 0.1788776686026366},
      {0.1788776686026366, 0.4254763922480989, 0.3290353014877974},
      {0.4254763922480989, 0.1788776686026366, 0.3290353014877974},
      {0.0666106376614672, 0.1788776686026366, 0.4254763922480989},
      {0.0666106376614672, 0.4254763922480989, 0.1788776686026366},
      {0.0666106376614672, 0.3290353014877974, 0.4254763922480989},
      {0.0666106376614672, 0.3290353014877974, 0.1788776686026366},
      {0.0666106376614672, 0.4254763922480989, 0.3290353014877974},
      {0.0666106376614672, 0.1788776686026366, 0.3290353014877974},
      {0.1788776686026366, 0.0666106376614672, 0.4254763922480989},
      {0.4254763922480989, 0.0666106376614672, 0.1788776686026366},
      {0.3290353014877974, 0.0666106376614672, 0.4254763922480989},
      {0.3290353014877974, 0.0666106376614672, 0.1788776686026366},
      {0.4254763922480989, 0.0666106376614672, 0.3290353014877974},
      {0.1788776686026366, 0.0666106376614672, 0.3290353014877974},
      {0.1788776686026366, 0.4254763922480989, 0.0666106376614672},
      {0.4254763922480989, 0.1788776686026366, 0.0666106376614672},
      {0.3290353014877974, 0.4254763922480989, 0.0666106376614672},
      {0.3290353014877974, 0.1788776686026366, 0.0666106376614672},
      {0.4254763922480989, 0.3290353014877974, 0.0666106376614672},
      {0.1788776686026366, 0.3290353014877974, 0.0666106376614672},
      {0.0049503159569494, 0.3340176116726106, 0.4722321350880272},
      {0.0049503159569494, 0.4722321350880272, 0.3340176116726106},
      {0.3340176116726106, 0.0049503159569494, 0.4722321350880272},
      {0.4722321350880272, 0.0049503159569494, 0.3340176116726106},
      {0.3340176116726106, 0.4722321350880272, 0.0049503159569494},
      {0.4722321350880272, 0.3340176116726106, 0.0049503159569494},
      {0.1887999372824127, 0.3340176116726106, 0.4722321350880272},
      {0.1887999372824127, 0.4722321350880272, 0.3340176116726106},
      {0.1887999372824127, 0.0049503159569494, 0.4722321350880272},
      {0.1887999372824127, 0.0049503159569494, 0.3340176116726106},
      {0.1887999372824127, 0.4722321350880272, 0.0049503159569494},
      {0.1887999372824127, 0.3340176116726106, 0.0049503159569494},
      {0.3340176116726106, 0.1887999372824127, 0.4722321350880272},
      {0.4722321350880272, 0.1887999372824127, 0.3340176116726106},
      {0.0049503159569494, 0.1887999372824127, 0.4722321350880272},
      {0.0049503159569494, 0.1887999372824127, 0.3340176116726106},
      {0.4722321350880272, 0.1887999372824127, 0.0049503159569494},
      {0.3340176116726106, 0.1887999372824127, 0.0049503159569494},
      {0.3340176116726106, 0.4722321350880272, 0.1887999372824127},
      {0.4722321350880272, 0.3340176116726106, 0.1887999372824127},
      {0.0049503159569494, 0.4722321350880272, 0.1887999372824127},
      {0.0049503159569494, 0.3340176116726106, 0.1887999372824127},
      {0.4722321350880272, 0.0049503159569494, 0.1887999372824127},
      {0.3340176116726106, 0.0049503159569494, 0.1887999372824127},
      {0.0078462883616813, 0.5458773009126385, 0.3752267608204884},
      {0.0078462883616813, 0.3752267608204884, 0.5458773009126385},
      {0.5458773009126385, 0.0078462883616813, 0.3752267608204884},
      {0.3752267608204884, 0.0078462883616813, 0.5458773009126385},
      {0.5458773009126385, 0.3752267608204884, 0.0078462883616813},
      {0.3752267608204884, 0.5458773009126385, 0.0078462883616813},
      {0.0710496499051917, 0.5458773009126385, 0.3752267608204884},
      {0.0710496499051917, 0.3752267608204884, 0.5458773009126385},
      {0.0710496499051917, 0.0078462883616813, 0.3752267608204884},
      {0.0710496499051917, 0.0078462883616813, 0.5458773009126385},
      {0.0710496499051917, 0.3752267608204884, 0.0078462883616813},
      {0.0710496499051917, 0.5458773009126385, 0.0078462883616813},
      {0.5458773009126385, 0.0710496499051917, 0.3752267608204884},
      {0.3752267608204884, 0.0710496499051917, 0.5458773009126385},
      {0.0078462883616813, 0.0710496499051917, 0.3752267608204884},
      {0.0078462883616813, 0.0710496499051917, 0.5458773009126385},
      {0.3752267608204884, 0.0710496499051917, 0.0078462883616813},
      {0.5458773009126385, 0.0710496499051917, 0.0078462883616813},
      {0.5458773009126385, 0.3752267608204884, 0.0710496499051917},
      {0.3752267608204884, 0.5458773009126385, 0.0710496499051917},
      {0.0078462883616813, 0.3752267608204884, 0.0710496499051917},
      {0.0078462883616813, 0.5458773009126385, 0.0710496499051917},
      {0.3752267608204884, 0.0078462883616813, 0.0710496499051917},
      {0.5458773009126385, 0.0078462883616813, 0.0710496499051917},
      {0.1293557534662013, 0.0251043985900785, 0.5925516079062881},
      {0.1293557534662013, 0.5925516079062881, 0.0251043985900785},
      {0.0251043985900785, 0.1293557534662013, 0.5925516079062881},
      {0.5925516079062881, 0.1293557534662013, 0.0251043985900785},
      {0.0251043985900785, 0.5925516079062881, 0.1293557534662013},
      {0.5925516079062881, 0.0251043985900785, 0.1293557534662013},
      {0.2529882400374321, 0.0251043985900785, 0.5925516079062881},
      {0.2529882400374321, 0.5925516079062881, 0.0251043985900785},
      {0.2529882400374321, 0.1293557534662013, 0.5925516079062881},
      {0.2529882400374321, 0.1293557534662013, 0.0251043985900785},
      {0.2529882400374321, 0.5925516079062881, 0.1293557534662013},
      {0.2529882400374321, 0.0251043985900785, 0.1293557534662013},
      {0.0251043985900785, 0.2529882400374321, 0.5925516079062881},
      {0.5925516079062881, 0.2529882400374321, 0.0251043985900785},
      {0.1293557534662013, 0.2529882400374321, 0.5925516079062881},
      {0.1293557534662013, 0.2529882400374321, 0.0251043985900785},
      {0.5925516079062881, 0.2529882400374321, 0.1293557534662013},
      {0.0251043985900785, 0.2529882400374321, 0.1293557534662013},
      {0.0251043985900785, 0.5925516079062881, 0.2529882400374321},
      {0.5925516079062881, 0.0251043985900785, 0.2529882400374321},
      {0.1293557534662013, 0.5925516079062881, 0.2529882400374321},
      {0.1293557534662013, 0.0251043985900785, 0.2529882400374321},
      {0.5925516079062881, 0.1293557534662013, 0.2529882400374321},
      {0.0251043985900785, 0.1293557534662013, 0.2529882400374321}},
      {0.0002508980677918,
      0.0002508980677918,
      0.0002508980677918,
      0.0002508980677918,
      0.0012002402552547,
      0.0012002402552547,
      0.0012002402552547,
      0.0012002402552547,
      0.0009669668722825,
      0.0009669668722825,
      0.0009669668722825,
      0.0009669668722825,
      0.0004108480654750,
      0.0004108480654750,
      0.0004108480654750,
      0.0004108480654750,
      0.0001372545176518,
      0.0001372545176518,
      0.0001372545176518,
      0.0001372545176518,
      0.0003993588993042,
      0.0003993588993042,
      0.0003993588993042,
      0.0003993588993042,
      0.0000277004423010,
      0.0000277004423010,
      0.0000277004423010,
      0.0000277004423010,
      0.0000079127110115,
      0.0000079127110115,
      0.0000079127110115,
      0.0000079127110115,
      0.0000079127110115,
      0.0000079127110115,
      0.0000942493400696,
      0.0000942493400696,
      0.0000942493400696,
      0.0000942493400696,
      0.0000942493400696,
      0.0000942493400696,
      0.0001877741412646,
      0.0001877741412646,
      0.0001877741412646,
      0.0001877741412646,
      0.0001877741412646,
      0.0001877741412646,
      0.0002247076919800,
      0.0002247076919800,
      0.0002247076919800,
      0.0002247076919800,
      0.0002247076919800,
      0.0002247076919800,
      0.0000609511516113,
      0.0000609511516113,
      0.0000609511516113,
      0.0000609511516113,
      0.0000609511516113,
      0.0000609511516113,
      0.0009755286654275,
      0.0009755286654275,
      0.0009755286654275,
      0.0009755286654275,
      0.0009755286654275,
      0.0009755286654275,
      0.0003261151724702,
      0.0003261151724702,
      0.0003261151724702,
      0.0003261151724702,
      0.0003261151724702,
      0.0003261151724702,
      0.0002788740420298,
      0.0002788740420298,
      0.0002788740420298,
      0.0002788740420298,
      0.0002788740420298,
      0.0002788740420298,
      0.0002807349955209,
      0.0002807349955209,
      0.0002807349955209,
      0.0002807349955209,
      0.0002807349955209,
      0.0002807349955209,
      0.0002807349955209,
      0.0002807349955209,
      0.0002807349955209,
      0.0002807349955209,
      0.0002807349955209,
      0.0002807349955209,
      0.0001581622171829,
      0.0001581622171829,
      0.0001581622171829,
      0.0001581622171829,
      0.0001581622171829,
      0.0001581622171829,
      0.0001581622171829,
      0.0001581622171829,
      0.0001581622171829,
      0.0001581622171829,
      0.0001581622171829,
      0.0001581622171829,
      0.0001731805226553,
      0.0001731805226553,
      0.0001731805226553,
      0.0001731805226553,
      0.0001731805226553,
      0.0001731805226553,
      0.0001731805226553,
      0.0001731805226553,
      0.0001731805226553,
      0.0001731805226553,
      0.0001731805226553,
      0.0001731805226553,
      0.0004780930540401,
      0.0004780930540401,
      0.0004780930540401,
      0.0004780930540401,
      0.0004780930540401,
      0.0004780930540401,
      0.0004780930540401,
      0.0004780930540401,
      0.0004780930540401,
      0.0004780930540401,
      0.0004780930540401,
      0.0004780930540401,
      0.0006120337974287,
      0.0006120337974287,
      0.0006120337974287,
      0.0006120337974287,
      0.0006120337974287,
      0.0006120337974287,
      0.0006120337974287,
      0.0006120337974287,
      0.0006120337974287,
      0.0006120337974287,
      0.0006120337974287,
      0.0006120337974287,
      0.0007911925910994,
      0.0007911925910994,
      0.0007911925910994,
      0.0007911925910994,
      0.0007911925910994,
      0.0007911925910994,
      0.0007911925910994,
      0.0007911925910994,
      0.0007911925910994,
      0.0007911925910994,
      0.0007911925910994,
      0.0007911925910994,
      0.0000691814013035,
      0.0000691814013035,
      0.0000691814013035,
      0.0000691814013035,
      0.0000691814013035,
      0.0000691814013035,
      0.0000691814013035,
      0.0000691814013035,
      0.0000691814013035,
      0.0000691814013035,
      0.0000691814013035,
      0.0000691814013035,
      0.0004174983537079,
      0.0004174983537079,
      0.0004174983537079,
      0.0004174983537079,
      0.0004174983537079,
      0.0004174983537079,
      0.0004174983537079,
      0.0004174983537079,
      0.0004174983537079,
      0.0004174983537079,
      0.0004174983537079,
      0.0004174983537079,
      0.0000438319937952,
      0.0000438319937952,
      0.0000438319937952,
      0.0000438319937952,
      0.0000438319937952,
      0.0000438319937952,
      0.0000438319937952,
      0.0000438319937952,
      0.0000438319937952,
      0.0000438319937952,
      0.0000438319937952,
      0.0000438319937952,
      0.0008729914697743,
      0.0008729914697743,
      0.0008729914697743,
      0.0008729914697743,
      0.0008729914697743,
      0.0008729914697743,
      0.0008729914697743,
      0.0008729914697743,
      0.0008729914697743,
      0.0008729914697743,
      0.0008729914697743,
      0.0008729914697743,
      0.0001588419557584,
      0.0001588419557584,
      0.0001588419557584,
      0.0001588419557584,
      0.0001588419557584,
      0.0001588419557584,
      0.0001588419557584,
      0.0001588419557584,
      0.0001588419557584,
      0.0001588419557584,
      0.0001588419557584,
      0.0001588419557584,
      0.0001588419557584,
      0.0001588419557584,
      0.0001588419557584,
      0.0001588419557584,
      0.0001588419557584,
      0.0001588419557584,
      0.0001588419557584,
      0.0001588419557584,
      0.0001588419557584,
      0.0001588419557584,
      0.0001588419557584,
      0.0001588419557584,
      0.0000737346583005,
      0.0000737346583005,
      0.0000737346583005,
      0.0000737346583005,
      0.0000737346583005,
      0.0000737346583005,
      0.0000737346583005,
      0.0000737346583005,
      0.0000737346583005,
      0.0000737346583005,
      0.0000737346583005,
      0.0000737346583005,
      0.0000737346583005,
      0.0000737346583005,
      0.0000737346583005,
      0.0000737346583005,
      0.0000737346583005,
      0.0000737346583005,
      0.0000737346583005,
      0.0000737346583005,
      0.0000737346583005,
      0.0000737346583005,
      0.0000737346583005,
      0.0000737346583005,
      0.0002926965014038,
      0.0002926965014038,
      0.0002926965014038,
      0.0002926965014038,
      0.0002926965014038,
      0.0002926965014038,
      0.0002926965014038,
      0.0002926965014038,
      0.0002926965014038,
      0.0002926965014038,
      0.0002926965014038,
      0.0002926965014038,
      0.0002926965014038,
      0.0002926965014038,
      0.0002926965014038,
      0.0002926965014038,
      0.0002926965014038,
      0.0002926965014038,
      0.0002926965014038,
      0.0002926965014038,
      0.0002926965014038,
      0.0002926965014038,
      0.0002926965014038,
      0.0002926965014038,
      0.0004057111469584,
      0.0004057111469584,
      0.0004057111469584,
      0.0004057111469584,
      0.0004057111469584,
      0.0004057111469584,
      0.0004057111469584,
      0.0004057111469584,
      0.0004057111469584,
      0.0004057111469584,
      0.0004057111469584,
      0.0004057111469584,
      0.0004057111469584,
      0.0004057111469584,
      0.0004057111469584,
      0.0004057111469584,
      0.0004057111469584,
      0.0004057111469584,
      0.0004057111469584,
      0.0004057111469584,
      0.0004057111469584,
      0.0004057111469584,
      0.0004057111469584,
      0.0004057111469584,
      0.0006560853276564,
      0.0006560853276564,
      0.0006560853276564,
      0.0006560853276564,
      0.0006560853276564,
      0.0006560853276564,
      0.0006560853276564,
      0.0006560853276564,
      0.0006560853276564,
      0.0006560853276564,
      0.0006560853276564,
      0.0006560853276564,
      0.0006560853276564,
      0.0006560853276564,
      0.0006560853276564,
      0.0006560853276564,
      0.0006560853276564,
      0.0006560853276564,
      0.0006560853276564,
      0.0006560853276564,
      0.0006560853276564,
      0.0006560853276564,
      0.0006560853276564,
      0.0006560853276564,
      0.0005332985448511,
      0.0005332985448511,
      0.0005332985448511,
      0.0005332985448511,
      0.0005332985448511,
      0.0005332985448511,
      0.0005332985448511,
      0.0005332985448511,
      0.0005332985448511,
      0.0005332985448511,
      0.0005332985448511,
      0.0005332985448511,
      0.0005332985448511,
      0.0005332985448511,
      0.0005332985448511,
      0.0005332985448511,
      0.0005332985448511,
      0.0005332985448511,
      0.0005332985448511,
      0.0005332985448511,
      0.0005332985448511,
      0.0005332985448511,
      0.0005332985448511,
      0.0005332985448511,
      0.0006329350520265,
      0.0006329350520265,
      0.0006329350520265,
      0.0006329350520265,
      0.0006329350520265,
      0.0006329350520265,
      0.0006329350520265,
      0.0006329350520265,
      0.0006329350520265,
      0.0006329350520265,
      0.0006329350520265,
      0.0006329350520265,
      0.0006329350520265,
      0.0006329350520265,
      0.0006329350520265,
      0.0006329350520265,
      0.0006329350520265,
      0.0006329350520265,
      0.0006329350520265,
      0.0006329350520265,
      0.0006329350520265,
      0.0006329350520265,
      0.0006329350520265,
      0.0006329350520265,
      0.0002493230937718,
      0.0002493230937718,
      0.0002493230937718,
      0.0002493230937718,
      0.0002493230937718,
      0.0002493230937718,
      0.0002493230937718,
      0.0002493230937718,
      0.0002493230937718,
      0.0002493230937718,
      0.0002493230937718,
      0.0002493230937718,
      0.0002493230937718,
      0.0002493230937718,
      0.0002493230937718,
      0.0002493230937718,
      0.0002493230937718,
      0.0002493230937718,
      0.0002493230937718,
      0.0002493230937718,
      0.0002493230937718,
      0.0002493230937718,
      0.0002493230937718,
      0.0002493230937718,
      0.0002839993635375,
      0.0002839993635375,
      0.0002839993635375,
      0.0002839993635375,
      0.0002839993635375,
      0.0002839993635375,
      0.0002839993635375,
      0.0002839993635375,
      0.0002839993635375,
      0.0002839993635375,
      0.0002839993635375,
      0.0002839993635375,
      0.0002839993635375,
      0.0002839993635375,
      0.0002839993635375,
      0.0002839993635375,
      0.0002839993635375,
      0.0002839993635375,
      0.0002839993635375,
      0.0002839993635375,
      0.0002839993635375,
      0.0002839993635375,
      0.0002839993635375,
      0.0002839993635375,
      0.0006047958529498,
      0.0006047958529498,
      0.0006047958529498,
      0.0006047958529498,
      0.0006047958529498,
      0.0006047958529498,
      0.0006047958529498,
      0.0006047958529498,
      0.0006047958529498,
      0.0006047958529498,
      0.0006047958529498,
      0.0006047958529498,
      0.0006047958529498,
      0.0006047958529498,
      0.0006047958529498,
      0.0006047958529498,
      0.0006047958529498,
      0.0006047958529498,
      0.0006047958529498,
      0.0006047958529498,
      0.0006047958529498,
      0.0006047958529498,
      0.0006047958529498,
      0.0006047958529498}
    },
    {
      487,
      {{0.2500000000000000, 0.2500000000000000, 0.2500000000000000},
      {0.0962334679417012, 0.0962334679417012, 0.0962334679417012},
      {0.7112995961748966, 0.0962334679417012, 0.0962334679417012},
      {0.0962334679417012, 0.7112995961748966, 0.0962334679417012},
      {0.0962334679417012, 0.0962334679417012, 0.7112995961748966},
      {0.2840710971185555, 0.2840710971185555, 0.2840710971185555},
      {0.1477867086443335, 0.2840710971185555, 0.2840710971185555},
      {0.2840710971185555, 0.1477867086443335, 0.2840710971185555},
      {0.2840710971185555, 0.2840710971185555, 0.1477867086443335},
      {0.0420121804976231, 0.0420121804976231, 0.0420121804976231},
      {0.8739634585071308, 0.0420121804976231, 0.0420121804976231},
      {0.0420121804976231, 0.8739634585071308, 0.0420121804976231},
      {0.0420121804976231, 0.0420121804976231, 0.8739634585071308},
      {0.3706074987820475, 0.1293925012179526, 0.1293925012179526},
      {0.1293925012179526, 0.3706074987820475, 0.1293925012179526},
      {0.3706074987820475, 0.3706074987820475, 0.1293925012179526},
      {0.1293925012179526, 0.1293925012179526, 0.3706074987820475},
      {0.3706074987820475, 0.1293925012179526, 0.3706074987820475},
      {0.1293925012179526, 0.3706074987820475, 0.3706074987820475},
      {0.7679699947393867, 0.0533931212533031, 0.0533931212533031},
      {0.0533931212533031, 0.7679699947393867, 0.0533931212533031},
      {0.0533931212533031, 0.0533931212533031, 0.7679699947393867},
      {0.1252437627540072, 0.0533931212533031, 0.0533931212533031},
      {0.1252437627540072, 0.7679699947393867, 0.0533931212533031},
      {0.1252437627540072, 0.0533931212533031, 0.7679699947393867},
      {0.0533931212533031, 0.1252437627540072, 0.0533931212533031},
      {0.7679699947393867, 0.1252437627540072, 0.0533931212533031},
      {0.0533931212533031, 0.1252437627540072, 0.7679699947393867},
      {0.0533931212533031, 0.0533931212533031, 0.1252437627540072},
      {0.7679699947393867, 0.0533931212533031, 0.1252437627540072},
      {0.0533931212533031, 0.7679699947393867, 0.1252437627540072},
      {0.1660339253205299, 0.0052505539069196, 0.0052505539069196},
      {0.0052505539069196, 0.1660339253205299, 0.0052505539069196},
      {0.0052505539069196, 0.0052505539069196, 0.1660339253205299},
      {0.8234649668656309, 0.0052505539069196, 0.0052505539069196},
      {0.8234649668656309, 0.1660339253205299, 0.0052505539069196},
      {0.8234649668656309, 0.0052505539069196, 0.1660339253205299},
      {0.0052505539069196, 0.8234649668656309, 0.0052505539069196},
      {0.1660339253205299, 0.8234649668656309, 0.0052505539069196},
      {0.0052505539069196, 0.8234649668656309, 0.1660339253205299},
      {0.0052505539069196, 0.0052505539069196, 0.8234649668656309},
      {0.1660339253205299, 0.0052505539069196, 0.8234649668656309},
      {0.0052505539069196, 0.1660339253205299, 0.8234649668656309},
      {0.1034045995093328, 0.2128993323694659, 0.2128993323694659},
      {0.2128993323694659, 0.1034045995093328, 0.2128993323694659},
      {0.2128993323694659, 0.2128993323694659, 0.1034045995093328},
      {0.4707967357517354, 0.2128993323694659, 0.2128993323694659},
      {0.4707967357517354, 0.1034045995093328, 0.2128993323694659},
      {0.4707967357517354, 0.2128993323694659, 0.1034045995093328},
      {0.2128993323694659, 0.4707967357517354, 0.2128993323694659},
      {0.1034045995093328, 0.4707967357517354, 0.2128993323694659},
      {0.2128993323694659, 0.4707967357517354, 0.1034045995093328},
      {0.2128993323694659, 0.2128993323694659, 0.4707967357517354},
      {0.1034045995093328, 0.2128993323694659, 0.4707967357517354},
      {0.2128993323694659, 0.1034045995093328, 0.4707967357517354},
      {0.1333356346432022, 0.4034135325977229, 0.4034135325977229},
      {0.4034135325977229, 0.1333356346432022, 0.4034135325977229},
      {0.4034135325977229, 0.4034135325977229, 0.1333356346432022},
      {0.0598373001613519, 0.4034135325977229, 0.4034135325977229},
      {0.0598373001613519, 0.1333356346432022, 0.4034135325977229},
      {0.0598373001613519, 0.4034135325977229, 0.1333356346432022},
      {0.4034135325977229, 0.0598373001613519, 0.4034135325977229},
      {0.1333356346432022, 0.0598373001613519, 0.4034135325977229},
      {0.4034135325977229, 0.0598373001613519, 0.1333356346432022},
      {0.4034135325977229, 0.4034135325977229, 0.0598373001613519},
      {0.1333356346432022, 0.4034135325977229, 0.0598373001613519},
      {0.4034135325977229, 0.1333356346432022, 0.0598373001613519},
      {0.5532379678070896, 0.0118881193081127, 0.0118881193081127},
      {0.0118881193081127, 0.5532379678070896, 0.0118881193081127},
      {0.0118881193081127, 0.0118881193081127, 0.5532379678070896},
      {0.4229857935766850, 0.0118881193081127, 0.0118881193081127},
      {0.4229857935766850, 0.5532379678070896, 0.0118881193081127},
      {0.4229857935766850, 0.0118881193081127, 0.5532379678070896},
      {0.0118881193081127, 0.4229857935766850, 0.0118881193081127},
      {0.5532379678070896, 0.4229857935766850, 0.0118881193081127},
      {0.0118881193081127, 0.4229857935766850, 0.5532379678070896},
      {0.0118881193081127, 0.0118881193081127, 0.4229857935766850},
      {0.5532379678070896, 0.0118881193081127, 0.4229857935766850},
      {0.0118881193081127, 0.5532379678070896, 0.4229857935766850},
      {0.8080695151909102, 0.0890311655120663, 0.0890311655120663},
      {0.0890311655120663, 0.8080695151909102, 0.0890311655120663},
      {0.0890311655120663, 0.0890311655120663, 0.8080695151909102},
      {0.0138681537849572, 0.0890311655120663, 0.0890311655120663},
      {0.0138681537849572, 0.8080695151909102, 0.0890311655120663},
      {0.0138681537849572, 0.0890311655120663, 0.8080695151909102},
      {0.0890311655120663, 0.0138681537849572, 0.0890311655120663},
      {0.8080695151909102, 0.0138681537849572, 0.0890311655120663},
      {0.0890311655120663, 0.0138681537849572, 0.8080695151909102},
      {0.0890311655120663, 0.0890311655120663, 0.0138681537849572},
      {0.8080695151909102, 0.0890311655120663, 0.0138681537849572},
      {0.0890311655120663, 0.8080695151909102, 0.0138681537849572},
      {0.4873280365294901, 0.0515167869589916, 0.0515167869589916},
      {0.0515167869589916, 0.4873280365294901, 0.0515167869589916},
      {0.0515167869589916, 0.0515167869589916, 0.4873280365294901},
      {0.4096383895525266, 0.0515167869589916, 0.0515167869589916},
      {0.4096383895525266, 0.4873280365294901, 0.0515167869589916},
      {0.4096383895525266, 0.0515167869589916, 0.4873280365294901},
      {0.0515167869589916, 0.4096383895525266, 0.0515167869589916},
      {0.4873280365294901, 0.4096383895525266, 0.0515167869589916},
      {0.0515167869589916, 0.4096383895525266, 0.4873280365294901},
      {0.0515167869589916, 0.0515167869589916, 0.4096383895525266},
      {0.4873280365294901, 0.0515167869589916, 0.4096383895525266},
      {0.0515167869589916, 0.4873280365294901, 0.4096383895525266},
      {0.2466149943644083, 0.1179992852297671, 0.1179992852297671},
      {0.1179992852297671, 0.2466149943644083, 0.1179992852297671},
      {0.1179992852297671, 0.1179992852297671, 0.2466149943644083},
      {0.5173864351760575, 0.1179992852297671, 0.1179992852297671},
      {0.5173864351760575, 0.2466149943644083, 0.1179992852297671},
      {0.5173864351760575, 0.1179992852297671, 0.2466149943644083},
      {0.1179992852297671, 0.5173864351760575, 0.1179992852297671},
      {0.2466149943644083, 0.5173864351760575, 0.1179992852297671},
      {0.1179992852297671, 0.5173864351760575, 0.2466149943644083},
      {0.1179992852297671, 0.1179992852297671, 0.5173864351760575},
      {0.2466149943644083, 0.1179992852297671, 0.5173864351760575},
      {0.1179992852297671, 0.2466149943644083, 0.5173864351760575},
      {0.2781414318856127, 0.0460851233492859, 0.0460851233492859},
      {0.0460851233492859, 0.2781414318856127, 0.0460851233492859},
      {0.0460851233492859, 0.0460851233492859, 0.2781414318856127},
      {0.6296883214158154, 0.0460851233492859, 0.0460851233492859},
      {0.6296883214158154, 0.2781414318856127, 0.0460851233492859},
      {0.6296883214158154, 0.0460851233492859, 0.2781414318856127},
      {0.0460851233492859, 0.6296883214158154, 0.0460851233492859},
      {0.2781414318856127, 0.6296883214158154, 0.0460851233492859},
      {0.0460851233492859, 0.6296883214158154, 0.2781414318856127},
      {0.0460851233492859, 0.0460851233492859, 0.6296883214158154},
      {0.2781414318856127, 0.0460851233492859, 0.6296883214158154},
      {0.0460851233492859, 0.2781414318856127, 0.6296883214158154},
      {0.0081746499483895, 0.1410186708097898, 0.1410186708097898},
      {0.1410186708097898, 0.0081746499483895, 0.1410186708097898},
      {0.1410186708097898, 0.1410186708097898, 0.0081746499483895},
      {0.7097880084320309, 0.1410186708097898, 0.1410186708097898},
      {0.7097880084320309, 0.0081746499483895, 0.1410186708097898},
      {0.7097880084320309, 0.1410186708097898, 0.0081746499483895},
      {0.1410186708097898, 0.7097880084320309, 0.1410186708097898},
      {0.0081746499483895, 0.7097880084320309, 0.1410186708097898},
      {0.1410186708097898, 0.7097880084320309, 0.0081746499483895},
      {0.1410186708097898, 0.1410186708097898, 0.7097880084320309},
      {0.0081746499483895, 0.1410186708097898, 0.7097880084320309},
      {0.1410186708097898, 0.0081746499483895, 0.7097880084320309},
      {0.5855868767598970, 0.1411034408699267, 0.1411034408699267},
      {0.1411034408699267, 0.5855868767598970, 0.1411034408699267},
      {0.1411034408699267, 0.1411034408699267, 0.5855868767598970},
      {0.1322062415002498, 0.1411034408699267, 0.1411034408699267},
      {0.1322062415002498, 0.5855868767598970, 0.1411034408699267},
      {0.1322062415002498, 0.1411034408699267, 0.5855868767598970},
      {0.1411034408699267, 0.1322062415002498, 0.1411034408699267},
      {0.5855868767598970, 0.1322062415002498, 0.1411034408699267},
      {0.1411034408699267, 0.1322062415002498, 0.5855868767598970},
      {0.1411034408699267, 0.1411034408699267, 0.1322062415002498},
      {0.5855868767598970, 0.1411034408699267, 0.1322062415002498},
      {0.1411034408699267, 0.5855868767598970, 0.1322062415002498},
      {0.0303588380906194, 0.0079118622048211, 0.0079118622048211},
      {0.0079118622048211, 0.0303588380906194, 0.0079118622048211},
      {0.0079118622048211, 0.0079118622048211, 0.0303588380906194},
      {0.9538174374997385, 0.0079118622048211, 0.0079118622048211},
      {0.9538174374997385, 0.0303588380906194, 0.0079118622048211},
      {0.9538174374997385, 0.0079118622048211, 0.0303588380906194},
      {0.0079118622048211, 0.9538174374997385, 0.0079118622048211},
      {0.0303588380906194, 0.9538174374997385, 0.0079118622048211},
      {0.0079118622048211, 0.9538174374997385, 0.0303588380906194},
      {0.0079118622048211, 0.0079118622048211, 0.9538174374997385},
      {0.0303588380906194, 0.0079118622048211, 0.9538174374997385},
      {0.0079118622048211, 0.0303588380906194, 0.9538174374997385},
      {0.2461239587816849, 0.1761012114703174, 0.1761012114703174},
      {0.1761012114703174, 0.2461239587816849, 0.1761012114703174},
      {0.1761012114703174, 0.1761012114703174, 0.2461239587816849},
      {0.4016736182776802, 0.1761012114703174, 0.1761012114703174},
      {0.4016736182776802, 0.2461239587816849, 0.1761012114703174},
      {0.4016736182776802, 0.1761012114703174, 0.2461239587816849},
      {0.1761012114703174, 0.4016736182776802, 0.1761012114703174},
      {0.2461239587816849, 0.4016736182776802, 0.1761012114703174},
      {0.1761012114703174, 0.4016736182776802, 0.2461239587816849},
      {0.1761012114703174, 0.1761012114703174, 0.4016736182776802},
      {0.2461239587816849, 0.1761012114703174, 0.4016736182776802},
      {0.1761012114703174, 0.2461239587816849, 0.4016736182776802},
      {0.7694124041149815, 0.0453113744152098, 0.1736495090943957},
      {0.7694124041149815, 0.1736495090943957, 0.0453113744152098},
      {0.0453113744152098, 0.7694124041149815, 0.1736495090943957},
      {0.1736495090943957, 0.7694124041149815, 0.0453113744152098},
      {0.0453113744152098, 0.1736495090943957, 0.7694124041149815},
      {0.1736495090943957, 0.0453113744152098, 0.7694124041149815},
      {0.0116267123754130, 0.0453113744152098, 0.1736495090943957},
      {0.0116267123754130, 0.1736495090943957, 0.0453113744152098},
      {0.0116267123754130, 0.7694124041149815, 0.1736495090943957},
      {0.0116267123754130, 0.7694124041149815, 0.0453113744152098},
      {0.0116267123754130, 0.1736495090943957, 0.7694124041149815},
      {0.0116267123754130, 0.0453113744152098, 0.7694124041149815},
      {0.0453113744152098, 0.0116267123754130, 0.1736495090943957},
      {0.1736495090943957, 0.0116267123754130, 0.0453113744152098},
      {0.7694124041149815, 0.0116267123754130, 0.1736495090943957},
      {0.7694124041149815, 0.0116267123754130, 0.0453113744152098},
      {0.1736495090943957, 0.0116267123754130, 0.7694124041149815},
      {0.0453113744152098, 0.0116267123754130, 0.7694124041149815},
      {0.0453113744152098, 0.1736495090943957, 0.0116267123754130},
      {0.1736495090943957, 0.0453113744152098, 0.0116267123754130},
      {0.7694124041149815, 0.1736495090943957, 0.0116267123754130},
      {0.7694124041149815, 0.0453113744152098, 0.0116267123754130},
      {0.1736495090943957, 0.7694124041149815, 0.0116267123754130},
      {0.0453113744152098, 0.7694124041149815, 0.0116267123754130},
      {0.1139907343389768, 0.1902469770309511, 0.6467333246419940},
      {0.1139907343389768, 0.6467333246419940, 0.1902469770309511},
      {0.1902469770309511, 0.1139907343389768, 0.6467333246419940},
      {0.6467333246419940, 0.1139907343389768, 0.1902469770309511},
      {0.1902469770309511, 0.6467333246419940, 0.1139907343389768},
      {0.6467333246419940, 0.1902469770309511, 0.1139907343389768},
      {0.0490289639880780, 0.1902469770309511, 0.6467333246419940},
      {0.0490289639880780, 0.6467333246419940, 0.1902469770309511},
      {0.0490289639880780, 0.1139907343389768, 0.6467333246419940},
      {0.0490289639880780, 0.1139907343389768, 0.1902469770309511},
      {0.0490289639880780, 0.6467333246419940, 0.1139907343389768},
      {0.0490289639880780, 0.1902469770309511, 0.1139907343389768},
      {0.1902469770309511, 0.0490289639880780, 0.6467333246419940},
      {0.6467333246419940, 0.0490289639880780, 0.1902469770309511},
      {0.1139907343389768, 0.0490289639880780, 0.6467333246419940},
      {0.1139907343389768, 0.0490289639880780, 0.1902469770309511},
      {0.6467333246419940, 0.0490289639880780, 0.1139907343389768},
      {0.1902469770309511, 0.0490289639880780, 0.1139907343389768},
      {0.1902469770309511, 0.6467333246419940, 0.0490289639880780},
      {0.6467333246419940, 0.1902469770309511, 0.0490289639880780},
      {0.1139907343389768, 0.6467333246419940, 0.0490289639880780},
      {0.1139907343389768, 0.1902469770309511, 0.0490289639880780},
      {0.6467333246419940, 0.1139907343389768, 0.0490289639880780},
      {0.1902469770309511, 0.1139907343389768, 0.0490289639880780},
      {0.5045224197189542, 0.2025882107314621, 0.2408858063990418},
      {0.5045224197189542, 0.2408858063990418, 0.2025882107314621},
      {0.2025882107314621, 0.5045224197189542, 0.2408858063990418},
      {0.2408858063990418, 0.5045224197189542, 0.2025882107314621},
      {0.2025882107314621, 0.2408858063990418, 0.5045224197189542},
      {0.2408858063990418, 0.2025882107314621, 0.5045224197189542},
      {0.0520035631505419, 0.2025882107314621, 0.2408858063990418},
      {0.0520035631505419, 0.2408858063990418, 0.2025882107314621},
      {0.0520035631505419, 0.5045224197189542, 0.2408858063990418},
      {0.0520035631505419, 0.5045224197189542, 0.2025882107314621},
      {0.0520035631505419, 0.2408858063990418, 0.5045224197189542},
      {0.0520035631505419, 0.2025882107314621, 0.5045224197189542},
      {0.2025882107314621, 0.0520035631505419, 0.2408858063990418},
      {0.2408858063990418, 0.0520035631505419, 0.2025882107314621},
      {0.5045224197189542, 0.0520035631505419, 0.2408858063990418},
      {0.5045224197189542, 0.0520035631505419, 0.2025882107314621},
      {0.2408858063990418, 0.0520035631505419, 0.5045224197189542},
      {0.2025882107314621, 0.0520035631505419, 0.5045224197189542},
      {0.2025882107314621, 0.2408858063990418, 0.0520035631505419},
      {0.2408858063990418, 0.2025882107314621, 0.0520035631505419},
      {0.5045224197189542, 0.2408858063990418, 0.0520035631505419},
      {0.5045224197189542, 0.2025882107314621, 0.0520035631505419},
      {0.2408858063990418, 0.5045224197189542, 0.0520035631505419},
      {0.2025882107314621, 0.5045224197189542, 0.0520035631505419},
      {0.3899342137938394, 0.0628045986545770, 0.5376101656245691},
      {0.3899342137938394, 0.5376101656245691, 0.0628045986545770},
      {0.0628045986545770, 0.3899342137938394, 0.5376101656245691},
      {0.5376101656245691, 0.3899342137938394, 0.0628045986545770},
      {0.0628045986545770, 0.5376101656245691, 0.3899342137938394},
      {0.5376101656245691, 0.0628045986545770, 0.3899342137938394},
      {0.0096510219270146, 0.0628045986545770, 0.5376101656245691},
      {0.0096510219270146, 0.5376101656245691, 0.0628045986545770},
      {0.0096510219270146, 0.3899342137938394, 0.5376101656245691},
      {0.0096510219270146, 0.3899342137938394, 0.0628045986545770},
      {0.0096510219270146, 0.5376101656245691, 0.3899342137938394},
      {0.0096510219270146, 0.0628045986545770, 0.3899342137938394},
      {0.0628045986545770, 0.0096510219270146, 0.5376101656245691},
      {0.5376101656245691, 0.0096510219270146, 0.0628045986545770},
      {0.3899342137938394, 0.0096510219270146, 0.5376101656245691},
      {0.3899342137938394, 0.0096510219270146, 0.0628045986545770},
      {0.5376101656245691, 0.0096510219270146, 0.3899342137938394},
      {0.0628045986545770, 0.0096510219270146, 0.3899342137938394},
      {0.0628045986545770, 0.5376101656245691, 0.0096510219270146},
      {0.5376101656245691, 0.0628045986545770, 0.0096510219270146},
      {0.3899342137938394, 0.5376101656245691, 0.0096510219270146},
      {0.3899342137938394, 0.0628045986545770, 0.0096510219270146},
      {0.5376101656245691, 0.3899342137938394, 0.0096510219270146},
      {0.0628045986545770, 0.3899342137938394, 0.0096510219270146},
      {0.0102510197711691, 0.1689034611068301, 0.5666194853322549},
      {0.0102510197711691, 0.5666194853322549, 0.1689034611068301},
      {0.1689034611068301, 0.0102510197711691, 0.5666194853322549},
      {0.5666194853322549, 0.0102510197711691, 0.1689034611068301},
      {0.1689034611068301, 0.5666194853322549, 0.0102510197711691},
      {0.5666194853322549, 0.1689034611068301, 0.0102510197711691},
      {0.2542260337897459, 0.1689034611068301, 0.5666194853322549},
      {0.2542260337897459, 0.5666194853322549, 0.1689034611068301},
      {0.2542260337897459, 0.0102510197711691, 0.5666194853322549},
      {0.2542260337897459, 0.0102510197711691, 0.1689034611068301},
      {0.2542260337897459, 0.5666194853322549, 0.0102510197711691},
      {0.2542260337897459, 0.1689034611068301, 0.0102510197711691},
      {0.1689034611068301, 0.2542260337897459, 0.5666194853322549},
      {0.5666194853322549, 0.2542260337897459, 0.1689034611068301},
      {0.0102510197711691, 0.2542260337897459, 0.5666194853322549},
      {0.0102510197711691, 0.2542260337897459, 0.1689034611068301},
      {0.5666194853322549, 0.2542260337897459, 0.0102510197711691},
      {0.1689034611068301, 0.2542260337897459, 0.0102510197711691},
      {0.1689034611068301, 0.5666194853322549, 0.2542260337897459},
      {0.5666194853322549, 0.1689034611068301, 0.2542260337897459},
      {0.0102510197711691, 0.5666194853322549, 0.2542260337897459},
      {0.0102510197711691, 0.1689034611068301, 0.2542260337897459},
      {0.5666194853322549, 0.0102510197711691, 0.2542260337897459},
      {0.1689034611068301, 0.0102510197711691, 0.2542260337897459},
      {0.3714709314621824, 0.1282802298826546, 0.2176247037611105},
      {0.3714709314621824, 0.2176247037611105, 0.1282802298826546},
      {0.1282802298826546, 0.3714709314621824, 0.2176247037611105},
      {0.2176247037611105, 0.3714709314621824, 0.1282802298826546},
      {0.1282802298826546, 0.2176247037611105, 0.3714709314621824},
      {0.2176247037611105, 0.1282802298826546, 0.3714709314621824},
      {0.2826241348940526, 0.1282802298826546, 0.2176247037611105},
      {0.2826241348940526, 0.2176247037611105, 0.1282802298826546},
      {0.2826241348940526, 0.3714709314621824, 0.2176247037611105},
      {0.2826241348940526, 0.3714709314621824, 0.1282802298826546},
      {0.2826241348940526, 0.2176247037611105, 0.3714709314621824},
      {0.2826241348940526, 0.1282802298826546, 0.3714709314621824},
      {0.1282802298826546, 0.2826241348940526, 0.2176247037611105},
      {0.2176247037611105, 0.2826241348940526, 0.1282802298826546},
      {0.3714709314621824, 0.2826241348940526, 0.2176247037611105},
      {0.3714709314621824, 0.2826241348940526, 0.1282802298826546},
      {0.2176247037611105, 0.2826241348940526, 0.3714709314621824},
      {0.1282802298826546, 0.2826241348940526, 0.3714709314621824},
      {0.1282802298826546, 0.2176247037611105, 0.2826241348940526},
      {0.2176247037611105, 0.1282802298826546, 0.2826241348940526},
      {0.3714709314621824, 0.2176247037611105, 0.2826241348940526},
      {0.3714709314621824, 0.1282802298826546, 0.2826241348940526},
      {0.2176247037611105, 0.3714709314621824, 0.2826241348940526},
      {0.1282802298826546, 0.3714709314621824, 0.2826241348940526},
      {0.2323738231259893, 0.3753164302248334, 0.3358240006032013},
      {0.2323738231259893, 0.3358240006032013, 0.3753164302248334},
      {0.3753164302248334, 0.2323738231259893, 0.3358240006032013},
      {0.3358240006032013, 0.2323738231259893, 0.3753164302248334},
      {0.3753164302248334, 0.3358240006032013, 0.2323738231259893},
      {0.3358240006032013, 0.3753164302248334, 0.2323738231259893},
      {0.0564857460459761, 0.3753164302248334, 0.3358240006032013},
      {0.0564857460459761, 0.3358240006032013, 0.3753164302248334},
      {0.0564857460459761, 0.2323738231259893, 0.3358240006032013},
      {0.0564857460459761, 0.2323738231259893, 0.3753164302248334},
      {0.0564857460459761, 0.3358240006032013, 0.2323738231259893},
      {0.0564857460459761, 0.3753164302248334, 0.2323738231259893},
      {0.3753164302248334, 0.0564857460459761, 0.3358240006032013},
      {0.3358240006032013, 0.0564857460459761, 0.3753164302248334},
      {0.2323738231259893, 0.0564857460459761, 0.3358240006032013},
      {0.2323738231259893, 0.0564857460459761, 0.3753164302248334},
      {0.3358240006032013, 0.0564857460459761, 0.2323738231259893},
      {0.3753164302248334, 0.0564857460459761, 0.2323738231259893},
      {0.3753164302248334, 0.3358240006032013, 0.0564857460459761},
      {0.3358240006032013, 0.3753164302248334, 0.0564857460459761},
      {0.2323738231259893, 0.3358240006032013, 0.0564857460459761},
      {0.2323738231259893, 0.3753164302248334, 0.0564857460459761},
      {0.3358240006032013, 0.2323738231259893, 0.0564857460459761},
      {0.3753164302248334, 0.2323738231259893, 0.0564857460459761},
      {0.0282332391047721, 0.0817663469329853, 0.0059682218345955},
      {0.0282332391047721, 0.0059682218345955, 0.0817663469329853},
      {0.0817663469329853, 0.0282332391047721, 0.0059682218345955},
      {0.0059682218345955, 0.0282332391047721, 0.0817663469329853},
      {0.0817663469329853, 0.0059682218345955, 0.0282332391047721},
      {0.0059682218345955, 0.0817663469329853, 0.0282332391047721},
      {0.8840321921276471, 0.0817663469329853, 0.0059682218345955},
      {0.8840321921276471, 0.0059682218345955, 0.0817663469329853},
      {0.8840321921276471, 0.0282332391047721, 0.0059682218345955},
      {0.8840321921276471, 0.0282332391047721, 0.0817663469329853},
      {0.8840321921276471, 0.0059682218345955, 0.0282332391047721},
      {0.8840321921276471, 0.0817663469329853, 0.0282332391047721},
      {0.0817663469329853, 0.8840321921276471, 0.0059682218345955},
      {0.0059682218345955, 0.8840321921276471, 0.0817663469329853},
      {0.0282332391047721, 0.8840321921276471, 0.0059682218345955},
      {0.0282332391047721, 0.8840321921276471, 0.0817663469329853},
      {0.0059682218345955, 0.8840321921276471, 0.0282332391047721},
      {0.0817663469329853, 0.8840321921276471, 0.0282332391047721},
      {0.0817663469329853, 0.0059682218345955, 0.8840321921276471},
      {0.0059682218345955, 0.0817663469329853, 0.8840321921276471},
      {0.0282332391047721, 0.0059682218345955, 0.8840321921276471},
      {0.0282332391047721, 0.0817663469329853, 0.8840321921276471},
      {0.0059682218345955, 0.0282332391047721, 0.8840321921276471},
      {0.0817663469329853, 0.0282332391047721, 0.8840321921276471},
      {0.4691373442926859, 0.3727215008878210, 0.1476210108130195},
      {0.4691373442926859, 0.1476210108130195, 0.3727215008878210},
      {0.3727215008878210, 0.4691373442926859, 0.1476210108130195},
      {0.1476210108130195, 0.4691373442926859, 0.3727215008878210},
      {0.3727215008878210, 0.1476210108130195, 0.4691373442926859},
      {0.1476210108130195, 0.3727215008878210, 0.4691373442926859},
      {0.0105201440064736, 0.3727215008878210, 0.1476210108130195},
      {0.0105201440064736, 0.1476210108130195, 0.3727215008878210},
      {0.0105201440064736, 0.4691373442926859, 0.1476210108130195},
      {0.0105201440064736, 0.4691373442926859, 0.3727215008878210},
      {0.0105201440064736, 0.1476210108130195, 0.4691373442926859},
      {0.0105201440064736, 0.3727215008878210, 0.4691373442926859},
      {0.3727215008878210, 0.0105201440064736, 0.1476210108130195},
      {0.1476210108130195, 0.0105201440064736, 0.3727215008878210},
      {0.4691373442926859, 0.0105201440064736, 0.1476210108130195},
      {0.4691373442926859, 0.0105201440064736, 0.3727215008878210},
      {0.1476210108130195, 0.0105201440064736, 0.4691373442926859},
      {0.3727215008878210, 0.0105201440064736, 0.4691373442926859},
      {0.3727215008878210, 0.1476210108130195, 0.0105201440064736},
      {0.1476210108130195, 0.3727215008878210, 0.0105201440064736},
      {0.4691373442926859, 0.1476210108130195, 0.0105201440064736},
      {0.4691373442926859, 0.3727215008878210, 0.0105201440064736},
      {0.1476210108130195, 0.4691373442926859, 0.0105201440064736},
      {0.3727215008878210, 0.4691373442926859, 0.0105201440064736},
      {0.0062411504342539, 0.2881077012617455, 0.6852432765000537},
      {0.0062411504342539, 0.6852432765000537, 0.2881077012617455},
      {0.2881077012617455, 0.0062411504342539, 0.6852432765000537},
      {0.6852432765000537, 0.0062411504342539, 0.2881077012617455},
      {0.2881077012617455, 0.6852432765000537, 0.0062411504342539},
      {0.6852432765000537, 0.2881077012617455, 0.0062411504342539},
      {0.0204078718039469, 0.2881077012617455, 0.6852432765000537},
      {0.0204078718039469, 0.6852432765000537, 0.2881077012617455},
      {0.0204078718039469, 0.0062411504342539, 0.6852432765000537},
      {0.0204078718039469, 0.0062411504342539, 0.2881077012617455},
      {0.0204078718039469, 0.6852432765000537, 0.0062411504342539},
      {0.0204078718039469, 0.2881077012617455, 0.0062411504342539},
      {0.2881077012617455, 0.0204078718039469, 0.6852432765000537},
      {0.6852432765000537, 0.0204078718039469, 0.2881077012617455},
      {0.0062411504342539, 0.0204078718039469, 0.6852432765000537},
      {0.0062411504342539, 0.0204078718039469, 0.2881077012617455},
      {0.6852432765000537, 0.0204078718039469, 0.0062411504342539},
      {0.2881077012617455, 0.0204078718039469, 0.0062411504342539},
      {0.2881077012617455, 0.6852432765000537, 0.0204078718039469},
      {0.6852432765000537, 0.2881077012617455, 0.0204078718039469},
      {0.0062411504342539, 0.6852432765000537, 0.0204078718039469},
      {0.0062411504342539, 0.2881077012617455, 0.0204078718039469},
      {0.6852432765000537, 0.0062411504342539, 0.0204078718039469},
      {0.2881077012617455, 0.0062411504342539, 0.0204078718039469},
      {0.2548571009364550, 0.0084836770304887, 0.6529603551870550},
      {0.2548571009364550, 0.6529603551870550, 0.0084836770304887},
      {0.0084836770304887, 0.2548571009364550, 0.6529603551870550},
      {0.6529603551870550, 0.2548571009364550, 0.0084836770304887},
      {0.0084836770304887, 0.6529603551870550, 0.2548571009364550},
      {0.6529603551870550, 0.0084836770304887, 0.2548571009364550},
      {0.0836988668460014, 0.0084836770304887, 0.6529603551870550},
      {0.0836988668460014, 0.6529603551870550, 0.0084836770304887},
      {0.0836988668460014, 0.2548571009364550, 0.6529603551870550},
      {0.0836988668460014, 0.2548571009364550, 0.0084836770304887},
      {0.0836988668460014, 0.6529603551870550, 0.2548571009364550},
      {0.0836988668460014, 0.0084836770304887, 0.2548571009364550},
      {0.0084836770304887, 0.0836988668460014, 0.6529603551870550},
      {0.6529603551870550, 0.0836988668460014, 0.0084836770304887},
      {0.2548571009364550, 0.0836988668460014, 0.6529603551870550},
      {0.2548571009364550, 0.0836988668460014, 0.0084836770304887},
      {0.6529603551870550, 0.0836988668460014, 0.2548571009364550},
      {0.0084836770304887, 0.0836988668460014, 0.2548571009364550},
      {0.0084836770304887, 0.6529603551870550, 0.0836988668460014},
      {0.6529603551870550, 0.0084836770304887, 0.0836988668460014},
      {0.2548571009364550, 0.6529603551870550, 0.0836988668460014},
      {0.2548571009364550, 0.0084836770304887, 0.0836988668460014},
      {0.6529603551870550, 0.2548571009364550, 0.0836988668460014},
      {0.0084836770304887, 0.2548571009364550, 0.0836988668460014},
      {0.5081225906571617, 0.1193128036212033, 0.3224332256857841},
      {0.5081225906571617, 0.3224332256857841, 0.1193128036212033},
      {0.1193128036212033, 0.5081225906571617, 0.3224332256857841},
      {0.3224332256857841, 0.5081225906571617, 0.1193128036212033},
      {0.1193128036212033, 0.3224332256857841, 0.5081225906571617},
      {0.3224332256857841, 0.1193128036212033, 0.5081225906571617},
      {0.0501313800358508, 0.1193128036212033, 0.3224332256857841},
      {0.0501313800358508, 0.3224332256857841, 0.1193128036212033},
      {0.0501313800358508, 0.5081225906571617, 0.3224332256857841},
      {0.0501313800358508, 0.5081225906571617, 0.1193128036212033},
      {0.0501313800358508, 0.3224332256857841, 0.5081225906571617},
      {0.0501313800358508, 0.1193128036212033, 0.5081225906571617},
      {0.1193128036212033, 0.0501313800358508, 0.3224332256857841},
      {0.3224332256857841, 0.0501313800358508, 0.1193128036212033},
      {0.5081225906571617, 0.0501313800358508, 0.3224332256857841},
      {0.5081225906571617, 0.0501313800358508, 0.1193128036212033},
      {0.3224332256857841, 0.0501313800358508, 0.5081225906571617},
      {0.1193128036212033, 0.0501313800358508, 0.5081225906571617},
      {0.1193128036212033, 0.3224332256857841, 0.0501313800358508},
      {0.3224332256857841, 0.1193128036212033, 0.0501313800358508},
      {0.5081225906571617, 0.3224332256857841, 0.0501313800358508},
      {0.5081225906571617, 0.1193128036212033, 0.0501313800358508},
      {0.3224332256857841, 0.5081225906571617, 0.0501313800358508},
      {0.1193128036212033, 0.5081225906571617, 0.0501313800358508},
      {0.0109499610637902, 0.4198369201417426, 0.2537102331108258},
      {0.0109499610637902, 0.2537102331108258, 0.4198369201417426},
      {0.4198369201417426, 0.0109499610637902, 0.2537102331108258},
      {0.2537102331108258, 0.0109499610637902, 0.4198369201417426},
      {0.4198369201417426, 0.2537102331108258, 0.0109499610637902},
      {0.2537102331108258, 0.4198369201417426, 0.0109499610637902},
      {0.3155028856836414, 0.4198369201417426, 0.2537102331108258},
      {0.3155028856836414, 0.2537102331108258, 0.4198369201417426},
      {0.3155028856836414, 0.0109499610637902, 0.2537102331108258},
      {0.3155028856836414, 0.0109499610637902, 0.4198369201417426},
      {0.3155028856836414, 0.2537102331108258, 0.0109499610637902},
      {0.3155028856836414, 0.4198369201417426, 0.0109499610637902},
      {0.4198369201417426, 0.3155028856836414, 0.2537102331108258},
      {0.2537102331108258, 0.3155028856836414, 0.4198369201417426},
      {0.0109499610637902, 0.3155028856836414, 0.2537102331108258},
      {0.0109499610637902, 0.3155028856836414, 0.4198369201417426},
      {0.2537102331108258, 0.3155028856836414, 0.0109499610637902},
      {0.4198369201417426, 0.3155028856836414, 0.0109499610637902},
      {0.4198369201417426, 0.2537102331108258, 0.3155028856836414},
      {0.2537102331108258, 0.4198369201417426, 0.3155028856836414},
      {0.0109499610637902, 0.2537102331108258, 0.3155028856836414},
      {0.0109499610637902, 0.4198369201417426, 0.3155028856836414},
      {0.2537102331108258, 0.0109499610637902, 0.3155028856836414},
      {0.4198369201417426, 0.0109499610637902, 0.3155028856836414}},
      {0.0015795708274036,
      0.0003849420077632,
      0.0003849420077632,
      0.0003849420077632,
      0.0003849420077632,
      0.0006819154603735,
      0.0006819154603735,
      0.0006819154603735,
      0.0006819154603735,
      0.0001956091112414,
      0.0001956091112414,
      0.0001956091112414,
      0.0001956091112414,
      0.0009662932171364,
      0.0009662932171364,
      0.0009662932171364,
      0.0009662932171364,
      0.0009662932171364,
      0.0009662932171364,
      0.0003381654398512,
      0.0003381654398512,
      0.0003381654398512,
      0.0003381654398512,
      0.0003381654398512,
      0.0003381654398512,
      0.0003381654398512,
      0.0003381654398512,
      0.0003381654398512,
      0.0003381654398512,
      0.0003381654398512,
      0.0003381654398512,
      0.0000334309747518,
      0.0000334309747518,
      0.0000334309747518,
      0.0000334309747518,
      0.0000334309747518,
      0.0000334309747518,
      0.0000334309747518,
      0.0000334309747518,
      0.0000334309747518,
      0.0000334309747518,
      0.0000334309747518,
      0.0000334309747518,
      0.0004846766620761,
      0.0004846766620761,
      0.0004846766620761,
      0.0004846766620761,
      0.0004846766620761,
      0.0004846766620761,
      0.0004846766620761,
      0.0004846766620761,
      0.0004846766620761,
      0.0004846766620761,
      0.0004846766620761,
      0.0004846766620761,
      0.0004220130772145,
      0.0004220130772145,
      0.0004220130772145,
      0.0004220130772145,
      0.0004220130772145,
      0.0004220130772145,
      0.0004220130772145,
      0.0004220130772145,
      0.0004220130772145,
      0.0004220130772145,
      0.0004220130772145,
      0.0004220130772145,
      0.0001223949289386,
      0.0001223949289386,
      0.0001223949289386,
      0.0001223949289386,
      0.0001223949289386,
      0.0001223949289386,
      0.0001223949289386,
      0.0001223949289386,
      0.0001223949289386,
      0.0001223949289386,
      0.0001223949289386,
      0.0001223949289386,
      0.0001793639290899,
      0.0001793639290899,
      0.0001793639290899,
      0.0001793639290899,
      0.0001793639290899,
      0.0001793639290899,
      0.0001793639290899,
      0.0001793639290899,
      0.0001793639290899,
      0.0001793639290899,
      0.0001793639290899,
      0.0001793639290899,
      0.0003310692795219,
      0.0003310692795219,
      0.0003310692795219,
      0.0003310692795219,
      0.0003310692795219,
      0.0003310692795219,
      0.0003310692795219,
      0.0003310692795219,
      0.0003310692795219,
      0.0003310692795219,
      0.0003310692795219,
      0.0003310692795219,
      0.0008410528940490,
      0.0008410528940490,
      0.0008410528940490,
      0.0008410528940490,
      0.0008410528940490,
      0.0008410528940490,
      0.0008410528940490,
      0.0008410528940490,
      0.0008410528940490,
      0.0008410528940490,
      0.0008410528940490,
      0.0008410528940490,
      0.0004487137434218,
      0.0004487137434218,
      0.0004487137434218,
      0.0004487137434218,
      0.0004487137434218,
      0.0004487137434218,
      0.0004487137434218,
      0.0004487137434218,
      0.0004487137434218,
      0.0004487137434218,
      0.0004487137434218,
      0.0004487137434218,
      0.0002217727147506,
      0.0002217727147506,
      0.0002217727147506,
      0.0002217727147506,
      0.0002217727147506,
      0.0002217727147506,
      0.0002217727147506,
      0.0002217727147506,
      0.0002217727147506,
      0.0002217727147506,
      0.0002217727147506,
      0.0002217727147506,
      0.0002971548168088,
      0.0002971548168088,
      0.0002971548168088,
      0.0002971548168088,
      0.0002971548168088,
      0.0002971548168088,
      0.0002971548168088,
      0.0002971548168088,
      0.0002971548168088,
      0.0002971548168088,
      0.0002971548168088,
      0.0002971548168088,
      0.0000210502637697,
      0.0000210502637697,
      0.0000210502637697,
      0.0000210502637697,
      0.0000210502637697,
      0.0000210502637697,
      0.0000210502637697,
      0.0000210502637697,
      0.0000210502637697,
      0.0000210502637697,
      0.0000210502637697,
      0.0000210502637697,
      0.0005407101573032,
      0.0005407101573032,
      0.0005407101573032,
      0.0005407101573032,
      0.0005407101573032,
      0.0005407101573032,
      0.0005407101573032,
      0.0005407101573032,
      0.0005407101573032,
      0.0005407101573032,
      0.0005407101573032,
      0.0005407101573032,
      0.0001747496976188,
      0.0001747496976188,
      0.0001747496976188,
      0.0001747496976188,
      0.0001747496976188,
      0.0001747496976188,
      0.0001747496976188,
      0.0001747496976188,
      0.0001747496976188,
      0.0001747496976188,
      0.0001747496976188,
      0.0001747496976188,
      0.0001747496976188,
      0.0001747496976188,
      0.0001747496976188,
      0.0001747496976188,
      0.0001747496976188,
      0.0001747496976188,
      0.0001747496976188,
      0.0001747496976188,
      0.0001747496976188,
      0.0001747496976188,
      0.0001747496976188,
      0.0001747496976188,
      0.0006009567909558,
      0.0006009567909558,
      0.0006009567909558,
      0.0006009567909558,
      0.0006009567909558,
      0.0006009567909558,
      0.0006009567909558,
      0.0006009567909558,
      0.0006009567909558,
      0.0006009567909558,
      0.0006009567909558,
      0.0006009567909558,
      0.0006009567909558,
      0.0006009567909558,
      0.0006009567909558,
      0.0006009567909558,
      0.0006009567909558,
      0.0006009567909558,
      0.0006009567909558,
      0.0006009567909558,
      0.0006009567909558,
      0.0006009567909558,
      0.0006009567909558,
      0.0006009567909558,
      0.0004269264013116,
      0.0004269264013116,
      0.0004269264013116,
      0.0004269264013116,
      0.0004269264013116,
      0.0004269264013116,
      0.0004269264013116,
      0.0004269264013116,
      0.0004269264013116,
      0.0004269264013116,
      0.0004269264013116,
      0.0004269264013116,
      0.0004269264013116,
      0.0004269264013116,
      0.0004269264013116,
      0.0004269264013116,
      0.0004269264013116,
      0.0004269264013116,
      0.0004269264013116,
      0.0004269264013116,
      0.0004269264013116,
      0.0004269264013116,
      0.0004269264013116,
      0.0004269264013116,
      0.0002498197446391,
      0.0002498197446391,
      0.0002498197446391,
      0.0002498197446391,
      0.0002498197446391,
      0.0002498197446391,
      0.0002498197446391,
      0.0002498197446391,
      0.0002498197446391,
      0.0002498197446391,
      0.0002498197446391,
      0.0002498197446391,
      0.0002498197446391,
      0.0002498197446391,
      0.0002498197446391,
      0.0002498197446391,
      0.0002498197446391,
      0.0002498197446391,
      0.0002498197446391,
      0.0002498197446391,
      0.0002498197446391,
      0.0002498197446391,
      0.0002498197446391,
      0.0002498197446391,
      0.0003023874287212,
      0.0003023874287212,
      0.0003023874287212,
      0.0003023874287212,
      0.0003023874287212,
      0.0003023874287212,
      0.0003023874287212,
      0.0003023874287212,
      0.0003023874287212,
      0.0003023874287212,
      0.0003023874287212,
      0.0003023874287212,
      0.0003023874287212,
      0.0003023874287212,
      0.0003023874287212,
      0.0003023874287212,
      0.0003023874287212,
      0.0003023874287212,
      0.0003023874287212,
      0.0003023874287212,
      0.0003023874287212,
      0.0003023874287212,
      0.0003023874287212,
      0.0003023874287212,
      0.0004806563456660,
      0.0004806563456660,
      0.0004806563456660,
      0.0004806563456660,
      0.0004806563456660,
      0.0004806563456660,
      0.0004806563456660,
      0.0004806563456660,
      0.0004806563456660,
      0.0004806563456660,
      0.0004806563456660,
      0.0004806563456660,
      0.0004806563456660,
      0.0004806563456660,
      0.0004806563456660,
      0.0004806563456660,
      0.0004806563456660,
      0.0004806563456660,
      0.0004806563456660,
      0.0004806563456660,
      0.0004806563456660,
      0.0004806563456660,
      0.0004806563456660,
      0.0004806563456660,
      0.0005319018842035,
      0.0005319018842035,
      0.0005319018842035,
      0.0005319018842035,
      0.0005319018842035,
      0.0005319018842035,
      0.0005319018842035,
      0.0005319018842035,
      0.0005319018842035,
      0.0005319018842035,
      0.0005319018842035,
      0.0005319018842035,
      0.0005319018842035,
      0.0005319018842035,
      0.0005319018842035,
      0.0005319018842035,
      0.0005319018842035,
      0.0005319018842035,
      0.0005319018842035,
      0.0005319018842035,
      0.0005319018842035,
      0.0005319018842035,
      0.0005319018842035,
      0.0005319018842035,
      0.0000659213722831,
      0.0000659213722831,
      0.0000659213722831,
      0.0000659213722831,
      0.0000659213722831,
      0.0000659213722831,
      0.0000659213722831,
      0.0000659213722831,
      0.0000659213722831,
      0.0000659213722831,
      0.0000659213722831,
      0.0000659213722831,
      0.0000659213722831,
      0.0000659213722831,
      0.0000659213722831,
      0.0000659213722831,
      0.0000659213722831,
      0.0000659213722831,
      0.0000659213722831,
      0.0000659213722831,
      0.0000659213722831,
      0.0000659213722831,
      0.0000659213722831,
      0.0000659213722831,
      0.0002737018719860,
      0.0002737018719860,
      0.0002737018719860,
      0.0002737018719860,
      0.0002737018719860,
      0.0002737018719860,
      0.0002737018719860,
      0.0002737018719860,
      0.0002737018719860,
      0.0002737018719860,
      0.0002737018719860,
      0.0002737018719860,
      0.0002737018719860,
      0.0002737018719860,
      0.0002737018719860,
      0.0002737018719860,
      0.0002737018719860,
      0.0002737018719860,
      0.0002737018719860,
      0.0002737018719860,
      0.0002737018719860,
      0.0002737018719860,
      0.0002737018719860,
      0.0002737018719860,
      0.0000773486816243,
      0.0000773486816243,
      0.0000773486816243,
      0.0000773486816243,
      0.0000773486816243,
      0.0000773486816243,
      0.0000773486816243,
      0.0000773486816243,
      0.0000773486816243,
      0.0000773486816243,
      0.0000773486816243,
      0.0000773486816243,
      0.0000773486816243,
      0.0000773486816243,
      0.0000773486816243,
      0.0000773486816243,
      0.0000773486816243,
      0.0000773486816243,
      0.0000773486816243,
      0.0000773486816243,
      0.0000773486816243,
      0.0000773486816243,
      0.0000773486816243,
      0.0000773486816243,
      0.0002018108899990,
      0.0002018108899990,
      0.0002018108899990,
      0.0002018108899990,
      0.0002018108899990,
      0.0002018108899990,
      0.0002018108899990,
      0.0002018108899990,
      0.0002018108899990,
      0.0002018108899990,
      0.0002018108899990,
      0.0002018108899990,
      0.0002018108899990,
      0.0002018108899990,
      0.0002018108899990,
      0.0002018108899990,
      0.0002018108899990,
      0.0002018108899990,
      0.0002018108899990,
      0.0002018108899990,
      0.0002018108899990,
      0.0002018108899990,
      0.0002018108899990,
      0.0002018108899990,
      0.0006088805028797,
      0.0006088805028797,
      0.0006088805028797,
      0.0006088805028797,
      0.0006088805028797,
      0.0006088805028797,
      0.0006088805028797,
      0.0006088805028797,
      0.0006088805028797,
      0.0006088805028797,
      0.0006088805028797,
      0.0006088805028797,
      0.0006088805028797,
      0.0006088805028797,
      0.0006088805028797,
      0.0006088805028797,
      0.0006088805028797,
      0.0006088805028797,
      0.0006088805028797,
      0.0006088805028797,
      0.0006088805028797,
      0.0006088805028797,
      0.0006088805028797,
      0.0006088805028797,
      0.0002907985397938,
      0.0002907985397938,
      0.0002907985397938,
      0.0002907985397938,
      0.0002907985397938,
      0.0002907985397938,
      0.0002907985397938,
      0.0002907985397938,
      0.0002907985397938,
      0.0002907985397938,
      0.0002907985397938,
      0.0002907985397938,
      0.0002907985397938,
      0.0002907985397938,
      0.0002907985397938,
      0.0002907985397938,
      0.0002907985397938,
      0.0002907985397938,
      0.0002907985397938,
      0.0002907985397938,
      0.0002907985397938,
      0.0002907985397938,
      0.0002907985397938,
      0.0002907985397938}
    },
    {
      552,
      {{0.2967759596934131, 0.2967759596934131, 0.2967759596934131},
      {0.1096721209197606, 0.2967759596934131, 0.2967759596934131},
      {0.2967759596934131, 0.1096721209197606, 0.2967759596934131},
      {0.2967759596934131, 0.2967759596934131, 0.1096721209197606},
      {0.1209047972112185, 0.1209047972112185, 0.1209047972112185},
      {0.6372856083663444, 0.1209047972112185, 0.1209047972112185},
      {0.1209047972112185, 0.6372856083663444, 0.1209047972112185},
      {0.1209047972112185, 0.1209047972112185, 0.6372856083663444},
      {0.3177959071881044, 0.3177959071881044, 0.3177959071881044},
      {0.0466122784356867, 0.3177959071881044, 0.3177959071881044},
      {0.3177959071881044, 0.0466122784356867, 0.3177959071881044},
      {0.3177959071881044, 0.3177959071881044, 0.0466122784356867},
      {0.2012655712414790, 0.2012655712414790, 0.2012655712414790},
      {0.3962032862755631, 0.2012655712414790, 0.2012655712414790},
      {0.2012655712414790, 0.3962032862755631, 0.2012655712414790},
      {0.2012655712414790, 0.2012655712414790, 0.3962032862755631},
      {0.1678838969272885, 0.1678838969272885, 0.1678838969272885},
      {0.4963483092181346, 0.1678838969272885, 0.1678838969272885},
      {0.1678838969272885, 0.4963483092181346, 0.1678838969272885},
      {0.1678838969272885, 0.1678838969272885, 0.4963483092181346},
      {0.0362149396096895, 0.0362149396096895, 0.0362149396096895},
      {0.8913551811709316, 0.0362149396096895, 0.0362149396096895},
      {0.0362149396096895, 0.8913551811709316, 0.0362149396096895},
      {0.0362149396096895, 0.0362149396096895, 0.8913551811709316},
      {0.4473975503825162, 0.0526024496174838, 0.0526024496174838},
      {0.0526024496174838, 0.4473975503825162, 0.0526024496174838},
      {0.4473975503825162, 0.4473975503825162, 0.0526024496174838},
      {0.0526024496174838, 0.0526024496174838, 0.4473975503825162},
      {0.4473975503825162, 0.0526024496174838, 0.4473975503825162},
      {0.0526024496174838, 0.4473975503825162, 0.4473975503825162},
      {0.1954214436686958, 0.3045785563313042, 0.3045785563313042},
      {0.3045785563313042, 0.1954214436686958, 0.3045785563313042},
      {0.1954214436686958, 0.1954214436686958, 0.3045785563313042},
      {0.3045785563313042, 0.3045785563313042, 0.1954214436686958},
      {0.1954214436686958, 0.3045785563313042, 0.1954214436686958},
      {0.3045785563313042, 0.1954214436686958, 0.1954214436686958},
      {0.8181532733687802, 0.0322823054983952, 0.0322823054983952},
      {0.0322823054983952, 0.8181532733687802, 0.0322823054983952},
      {0.0322823054983952, 0.0322823054983952, 0.8181532733687802},
      {0.1172821156344295, 0.0322823054983952, 0.0322823054983952},
      {0.1172821156344295, 0.8181532733687802, 0.0322823054983952},
      {0.1172821156344295, 0.0322823054983952, 0.8181532733687802},
      {0.0322823054983952, 0.1172821156344295, 0.0322823054983952},
      {0.8181532733687802, 0.1172821156344295, 0.0322823054983952},
      {0.0322823054983952, 0.1172821156344295, 0.8181532733687802},
      {0.0322823054983952, 0.0322823054983952, 0.1172821156344295},
      {0.8181532733687802, 0.0322823054983952, 0.1172821156344295},
      {0.0322823054983952, 0.8181532733687802, 0.1172821156344295},
      {0.1671963931828212, 0.0081265144705196, 0.0081265144705196},
      {0.0081265144705196, 0.1671963931828212, 0.0081265144705196},
      {0.0081265144705196, 0.0081265144705196, 0.1671963931828212},
      {0.8165505778761395, 0.0081265144705196, 0.0081265144705196},
      {0.8165505778761395, 0.1671963931828212, 0.0081265144705196},
      {0.8165505778761395, 0.0081265144705196, 0.1671963931828212},
      {0.0081265144705196, 0.8165505778761395, 0.0081265144705196},
      {0.1671963931828212, 0.8165505778761395, 0.0081265144705196},
      {0.0081265144705196, 0.8165505778761395, 0.1671963931828212},
      {0.0081265144705196, 0.0081265144705196, 0.8165505778761395},
      {0.1671963931828212, 0.0081265144705196, 0.8165505778761395},
      {0.0081265144705196, 0.1671963931828212, 0.8165505778761395},
      {0.0661101187666243, 0.1851459522994347, 0.1851459522994347},
      {0.1851459522994347, 0.0661101187666243, 0.1851459522994347},
      {0.1851459522994347, 0.1851459522994347, 0.0661101187666243},
      {0.5635979766345063, 0.1851459522994347, 0.1851459522994347},
      {0.5635979766345063, 0.0661101187666243, 0.1851459522994347},
      {0.5635979766345063, 0.1851459522994347, 0.0661101187666243},
      {0.1851459522994347, 0.5635979766345063, 0.1851459522994347},
      {0.0661101187666243, 0.5635979766345063, 0.1851459522994347},
      {0.1851459522994347, 0.5635979766345063, 0.0661101187666243},
      {0.1851459522994347, 0.1851459522994347, 0.5635979766345063},
      {0.0661101187666243, 0.1851459522994347, 0.5635979766345063},
      {0.1851459522994347, 0.0661101187666243, 0.5635979766345063},
      {0.1281831845252744, 0.4110775382017770, 0.4110775382017770},
      {0.4110775382017770, 0.1281831845252744, 0.4110775382017770},
      {0.4110775382017770, 0.4110775382017770, 0.1281831845252744},
      {0.0496617390711716, 0.4110775382017770, 0.4110775382017770},
      {0.0496617390711716, 0.1281831845252744, 0.4110775382017770},
      {0.0496617390711716, 0.4110775382017770, 0.1281831845252744},
      {0.4110775382017770, 0.0496617390711716, 0.4110775382017770},
      {0.1281831845252744, 0.0496617390711716, 0.4110775382017770},
      {0.4110775382017770, 0.0496617390711716, 0.1281831845252744},
      {0.4110775382017770, 0.4110775382017770, 0.0496617390711716},
      {0.1281831845252744, 0.4110775382017770, 0.0496617390711716},
      {0.4110775382017770, 0.1281831845252744, 0.0496617390711716},
      {0.5566366248307155, 0.0100597974068545, 0.0100597974068545},
      {0.0100597974068545, 0.5566366248307155, 0.0100597974068545},
      {0.0100597974068545, 0.0100597974068545, 0.5566366248307155},
      {0.4232437803555756, 0.0100597974068545, 0.0100597974068545},
      {0.4232437803555756, 0.5566366248307155, 0.0100597974068545},
      {0.4232437803555756, 0.0100597974068545, 0.5566366248307155},
      {0.0100597974068545, 0.4232437803555756, 0.0100597974068545},
      {0.5566366248307155, 0.4232437803555756, 0.0100597974068545},
      {0.0100597974068545, 0.4232437803555756, 0.5566366248307155},
      {0.0100597974068545, 0.0100597974068545, 0.4232437803555756},
      {0.5566366248307155, 0.0100597974068545, 0.4232437803555756},
      {0.0100597974068545, 0.5566366248307155, 0.4232437803555756},
      {0.8242515811428504, 0.0823016150372359, 0.0823016150372359},
      {0.0823016150372359, 0.8242515811428504, 0.0823016150372359},
      {0.0823016150372359, 0.0823016150372359, 0.8242515811428504},
      {0.0111451887826779, 0.0823016150372359, 0.0823016150372359},
      {0.0111451887826779, 0.8242515811428504, 0.0823016150372359},
      {0.0111451887826779, 0.0823016150372359, 0.8242515811428504},
      {0.0823016150372359, 0.0111451887826779, 0.0823016150372359},
      {0.8242515811428504, 0.0111451887826779, 0.0823016150372359},
      {0.0823016150372359, 0.0111451887826779, 0.8242515811428504},
      {0.0823016150372359, 0.0823016150372359, 0.0111451887826779},
      {0.8242515811428504, 0.0823016150372359, 0.0111451887826779},
      {0.0823016150372359, 0.8242515811428504, 0.0111451887826779},
      {0.6072922714391159, 0.0380128286026647, 0.0380128286026647},
      {0.0380128286026647, 0.6072922714391159, 0.0380128286026647},
      {0.0380128286026647, 0.0380128286026647, 0.6072922714391159},
      {0.3166820713555548, 0.0380128286026647, 0.0380128286026647},
      {0.3166820713555548, 0.6072922714391159, 0.0380128286026647},
      {0.3166820713555548, 0.0380128286026647, 0.6072922714391159},
      {0.0380128286026647, 0.3166820713555548, 0.0380128286026647},
      {0.6072922714391159, 0.3166820713555548, 0.0380128286026647},
      {0.0380128286026647, 0.3166820713555548, 0.6072922714391159},
      {0.0380128286026647, 0.0380128286026647, 0.3166820713555548},
      {0.6072922714391159, 0.0380128286026647, 0.3166820713555548},
      {0.0380128286026647, 0.6072922714391159, 0.3166820713555548},
      {0.1927591204214162, 0.0829781690579476, 0.0829781690579476},
      {0.0829781690579476, 0.1927591204214162, 0.0829781690579476},
      {0.0829781690579476, 0.0829781690579476, 0.1927591204214162},
      {0.6412845414626887, 0.0829781690579476, 0.0829781690579476},
      {0.6412845414626887, 0.1927591204214162, 0.0829781690579476},
      {0.6412845414626887, 0.0829781690579476, 0.1927591204214162},
      {0.0829781690579476, 0.6412845414626887, 0.0829781690579476},
      {0.1927591204214162, 0.6412845414626887, 0.0829781690579476},
      {0.0829781690579476, 0.6412845414626887, 0.1927591204214162},
      {0.0829781690579476, 0.0829781690579476, 0.6412845414626887},
      {0.1927591204214162, 0.0829781690579476, 0.6412845414626887},
      {0.0829781690579476, 0.1927591204214162, 0.6412845414626887},
      {0.2290899793113822, 0.0323283014904537, 0.0323283014904537},
      {0.0323283014904537, 0.2290899793113822, 0.0323283014904537},
      {0.0323283014904537, 0.0323283014904537, 0.2290899793113822},
      {0.7062534177077103, 0.0323283014904537, 0.0323283014904537},
      {0.7062534177077103, 0.2290899793113822, 0.0323283014904537},
      {0.7062534177077103, 0.0323283014904537, 0.2290899793113822},
      {0.0323283014904537, 0.7062534177077103, 0.0323283014904537},
      {0.2290899793113822, 0.7062534177077103, 0.0323283014904537},
      {0.0323283014904537, 0.7062534177077103, 0.2290899793113822},
      {0.0323283014904537, 0.0323283014904537, 0.7062534177077103},
      {0.2290899793113822, 0.0323283014904537, 0.7062534177077103},
      {0.0323283014904537, 0.2290899793113822, 0.7062534177077103},
      {0.0040624591785854, 0.1406521940177762, 0.1406521940177762},
      {0.1406521940177762, 0.0040624591785854, 0.1406521940177762},
      {0.1406521940177762, 0.1406521940177762, 0.0040624591785854},
      {0.7146331527858621, 0.1406521940177762, 0.1406521940177762},
      {0.7146331527858621, 0.0040624591785854, 0.1406521940177762},
      {0.7146331527858621, 0.1406521940177762, 0.0040624591785854},
      {0.1406521940177762, 0.7146331527858621, 0.1406521940177762},
      {0.0040624591785854, 0.7146331527858621, 0.1406521940177762},
      {0.1406521940177762, 0.7146331527858621, 0.0040624591785854},
      {0.1406521940177762, 0.1406521940177762, 0.7146331527858621},
      {0.0040624591785854, 0.1406521940177762, 0.7146331527858621},
      {0.1406521940177762, 0.0040624591785854, 0.7146331527858621},
      {0.5168656572693773, 0.1139719439112915, 0.1139719439112915},
      {0.1139719439112915, 0.5168656572693773, 0.1139719439112915},
      {0.1139719439112915, 0.1139719439112915, 0.5168656572693773},
      {0.2551904549080398, 0.1139719439112915, 0.1139719439112915},
      {0.2551904549080398, 0.5168656572693773, 0.1139719439112915},
      {0.2551904549080398, 0.1139719439112915, 0.5168656572693773},
      {0.1139719439112915, 0.2551904549080398, 0.1139719439112915},
      {0.5168656572693773, 0.2551904549080398, 0.1139719439112915},
      {0.1139719439112915, 0.2551904549080398, 0.5168656572693773},
      {0.1139719439112915, 0.1139719439112915, 0.2551904549080398},
      {0.5168656572693773, 0.1139719439112915, 0.2551904549080398},
      {0.1139719439112915, 0.5168656572693773, 0.2551904549080398},
      {0.0251785631681119, 0.0074535713771749, 0.0074535713771749},
      {0.0074535713771749, 0.0251785631681119, 0.0074535713771749},
      {0.0074535713771749, 0.0074535713771749, 0.0251785631681119},
      {0.9599142940775384, 0.0074535713771749, 0.0074535713771749},
      {0.9599142940775384, 0.0251785631681119, 0.0074535713771749},
      {0.9599142940775384, 0.0074535713771749, 0.0251785631681119},
      {0.0074535713771749, 0.9599142940775384, 0.0074535713771749},
      {0.0251785631681119, 0.9599142940775384, 0.0074535713771749},
      {0.0074535713771749, 0.9599142940775384, 0.0251785631681119},
      {0.0074535713771749, 0.0074535713771749, 0.9599142940775384},
      {0.0251785631681119, 0.0074535713771749, 0.9599142940775384},
      {0.0074535713771749, 0.0251785631681119, 0.9599142940775384},
      {0.3907070674791312, 0.1110171267302689, 0.1110171267302689},
      {0.1110171267302689, 0.3907070674791312, 0.1110171267302689},
      {0.1110171267302689, 0.1110171267302689, 0.3907070674791312},
      {0.3872586790603310, 0.1110171267302689, 0.1110171267302689},
      {0.3872586790603310, 0.3907070674791312, 0.1110171267302689},
      {0.3872586790603310, 0.1110171267302689, 0.3907070674791312},
      {0.1110171267302689, 0.3872586790603310, 0.1110171267302689},
      {0.3907070674791312, 0.3872586790603310, 0.1110171267302689},
      {0.1110171267302689, 0.3872586790603310, 0.3907070674791312},
      {0.1110171267302689, 0.1110171267302689, 0.3872586790603310},
      {0.3907070674791312, 0.1110171267302689, 0.3872586790603310},
      {0.1110171267302689, 0.3907070674791312, 0.3872586790603310},
      {0.5333767142737392, 0.0531381050391377, 0.4037166880195431},
      {0.5333767142737392, 0.4037166880195431, 0.0531381050391377},
      {0.0531381050391377, 0.5333767142737392, 0.4037166880195431},
      {0.4037166880195431, 0.5333767142737392, 0.0531381050391377},
      {0.0531381050391377, 0.4037166880195431, 0.5333767142737392},
      {0.4037166880195431, 0.0531381050391377, 0.5333767142737392},
      {0.0097684926675800, 0.0531381050391377, 0.4037166880195431},
      {0.0097684926675800, 0.4037166880195431, 0.0531381050391377},
      {0.0097684926675800, 0.5333767142737392, 0.4037166880195431},
      {0.0097684926675800, 0.5333767142737392, 0.0531381050391377},
      {0.0097684926675800, 0.4037166880195431, 0.5333767142737392},
      {0.0097684926675800, 0.0531381050391377, 0.5333767142737392},
      {0.0531381050391377, 0.0097684926675800, 0.4037166880195431},
      {0.4037166880195431, 0.0097684926675800, 0.0531381050391377},
      {0.5333767142737392, 0.0097684926675800, 0.4037166880195431},
      {0.5333767142737392, 0.0097684926675800, 0.0531381050391377},
      {0.4037166880195431, 0.0097684926675800, 0.5333767142737392},
      {0.0531381050391377, 0.0097684926675800, 0.5333767142737392},
      {0.0531381050391377, 0.4037166880195431, 0.0097684926675800},
      {0.4037166880195431, 0.0531381050391377, 0.0097684926675800},
      {0.5333767142737392, 0.4037166880195431, 0.0097684926675800},
      {0.5333767142737392, 0.0531381050391377, 0.0097684926675800},
      {0.4037166880195431, 0.5333767142737392, 0.0097684926675800},
      {0.0531381050391377, 0.5333767142737392, 0.0097684926675800},
      {0.0945051714627574, 0.0943920742872811, 0.0501546618875249},
      {0.0945051714627574, 0.0501546618875249, 0.0943920742872811},
      {0.0943920742872811, 0.0945051714627574, 0.0501546618875249},
      {0.0501546618875249, 0.0945051714627574, 0.0943920742872811},
      {0.0943920742872811, 0.0501546618875249, 0.0945051714627574},
      {0.0501546618875249, 0.0943920742872811, 0.0945051714627574},
      {0.7609480923624367, 0.0943920742872811, 0.0501546618875249},
      {0.7609480923624367, 0.0501546618875249, 0.0943920742872811},
      {0.7609480923624367, 0.0945051714627574, 0.0501546618875249},
      {0.7609480923624367, 0.0945051714627574, 0.0943920742872811},
      {0.7609480923624367, 0.0501546618875249, 0.0945051714627574},
      {0.7609480923624367, 0.0943920742872811, 0.0945051714627574},
      {0.0943920742872811, 0.7609480923624367, 0.0501546618875249},
      {0.0501546618875249, 0.7609480923624367, 0.0943920742872811},
      {0.0945051714627574, 0.7609480923624367, 0.0501546618875249},
      {0.0945051714627574, 0.7609480923624367, 0.0943920742872811},
      {0.0501546618875249, 0.7609480923624367, 0.0945051714627574},
      {0.0943920742872811, 0.7609480923624367, 0.0945051714627574},
      {0.0943920742872811, 0.0501546618875249, 0.7609480923624367},
      {0.0501546618875249, 0.0943920742872811, 0.7609480923624367},
      {0.0945051714627574, 0.0501546618875249, 0.7609480923624367},
      {0.0945051714627574, 0.0943920742872811, 0.7609480923624367},
      {0.0501546618875249, 0.0945051714627574, 0.7609480923624367},
      {0.0943920742872811, 0.0945051714627574, 0.7609480923624367},
      {0.7653955692270024, 0.0550544516427296, 0.1738892311826489},
      {0.7653955692270024, 0.1738892311826489, 0.0550544516427296},
      {0.0550544516427296, 0.7653955692270024, 0.1738892311826489},
      {0.1738892311826489, 0.7653955692270024, 0.0550544516427296},
      {0.0550544516427296, 0.1738892311826489, 0.7653955692270024},
      {0.1738892311826489, 0.0550544516427296, 0.7653955692270024},
      {0.0056607479476192, 0.0550544516427296, 0.1738892311826489},
      {0.0056607479476192, 0.1738892311826489, 0.0550544516427296},
      {0.0056607479476192, 0.7653955692270024, 0.1738892311826489},
      {0.0056607479476192, 0.7653955692270024, 0.0550544516427296},
      {0.0056607479476192, 0.1738892311826489, 0.7653955692270024},
      {0.0056607479476192, 0.0550544516427296, 0.7653955692270024},
      {0.0550544516427296, 0.0056607479476192, 0.1738892311826489},
      {0.1738892311826489, 0.0056607479476192, 0.0550544516427296},
      {0.7653955692270024, 0.0056607479476192, 0.1738892311826489},
      {0.7653955692270024, 0.0056607479476192, 0.0550544516427296},
      {0.1738892311826489, 0.0056607479476192, 0.7653955692270024},
      {0.0550544516427296, 0.0056607479476192, 0.7653955692270024},
      {0.0550544516427296, 0.1738892311826489, 0.0056607479476192},
      {0.1738892311826489, 0.0550544516427296, 0.0056607479476192},
      {0.7653955692270024, 0.1738892311826489, 0.0056607479476192},
      {0.7653955692270024, 0.0550544516427296, 0.0056607479476192},
      {0.1738892311826489, 0.7653955692270024, 0.0056607479476192},
      {0.0550544516427296, 0.7653955692270024, 0.0056607479476192},
      {0.0975172038222044, 0.1798637301234305, 0.6879571952789530},
      {0.0975172038222044, 0.6879571952789530, 0.1798637301234305},
      {0.1798637301234305, 0.0975172038222044, 0.6879571952789530},
      {0.6879571952789530, 0.0975172038222044, 0.1798637301234305},
      {0.1798637301234305, 0.6879571952789530, 0.0975172038222044},
      {0.6879571952789530, 0.1798637301234305, 0.0975172038222044},
      {0.0346618707754121, 0.1798637301234305, 0.6879571952789530},
      {0.0346618707754121, 0.6879571952789530, 0.1798637301234305},
      {0.0346618707754121, 0.0975172038222044, 0.6879571952789530},
      {0.0346618707754121, 0.0975172038222044, 0.1798637301234305},
      {0.0346618707754121, 0.6879571952789530, 0.0975172038222044},
      {0.0346618707754121, 0.1798637301234305, 0.0975172038222044},
      {0.1798637301234305, 0.0346618707754121, 0.6879571952789530},
      {0.6879571952789530, 0.0346618707754121, 0.1798637301234305},
      {0.0975172038222044, 0.0346618707754121, 0.6879571952789530},
      {0.0975172038222044, 0.0346618707754121, 0.1798637301234305},
      {0.6879571952789530, 0.0346618707754121, 0.0975172038222044},
      {0.1798637301234305, 0.0346618707754121, 0.0975172038222044},
      {0.1798637301234305, 0.6879571952789530, 0.0346618707754121},
      {0.6879571952789530, 0.1798637301234305, 0.0346618707754121},
      {0.0975172038222044, 0.6879571952789530, 0.0346618707754121},
      {0.0975172038222044, 0.1798637301234305, 0.0346618707754121},
      {0.6879571952789530, 0.0975172038222044, 0.0346618707754121},
      {0.1798637301234305, 0.0975172038222044, 0.0346618707754121},
      {0.4095036586170944, 0.2121920547752147, 0.3321735025035050},
      {0.4095036586170944, 0.3321735025035050, 0.2121920547752147},
      {0.2121920547752147, 0.4095036586170944, 0.3321735025035050},
      {0.3321735025035050, 0.4095036586170944, 0.2121920547752147},
      {0.2121920547752147, 0.3321735025035050, 0.4095036586170944},
      {0.3321735025035050, 0.2121920547752147, 0.4095036586170944},
      {0.0461307841041859, 0.2121920547752147, 0.3321735025035050},
      {0.0461307841041859, 0.3321735025035050, 0.2121920547752147},
      {0.0461307841041859, 0.4095036586170944, 0.3321735025035050},
      {0.0461307841041859, 0.4095036586170944, 0.2121920547752147},
      {0.0461307841041859, 0.3321735025035050, 0.4095036586170944},
      {0.0461307841041859, 0.2121920547752147, 0.4095036586170944},
      {0.2121920547752147, 0.0461307841041859, 0.3321735025035050},
      {0.3321735025035050, 0.0461307841041859, 0.2121920547752147},
      {0.4095036586170944, 0.0461307841041859, 0.3321735025035050},
      {0.4095036586170944, 0.0461307841041859, 0.2121920547752147},
      {0.3321735025035050, 0.0461307841041859, 0.4095036586170944},
      {0.2121920547752147, 0.0461307841041859, 0.4095036586170944},
      {0.2121920547752147, 0.3321735025035050, 0.0461307841041859},
      {0.3321735025035050, 0.2121920547752147, 0.0461307841041859},
      {0.4095036586170944, 0.3321735025035050, 0.0461307841041859},
      {0.4095036586170944, 0.2121920547752147, 0.0461307841041859},
      {0.3321735025035050, 0.4095036586170944, 0.0461307841041859},
      {0.2121920547752147, 0.4095036586170944, 0.0461307841041859},
      {0.4765557207500070, 0.1290161512351960, 0.3844720976410142},
      {0.4765557207500070, 0.3844720976410142, 0.1290161512351960},
      {0.1290161512351960, 0.4765557207500070, 0.3844720976410142},
      {0.3844720976410142, 0.4765557207500070, 0.1290161512351960},
      {0.1290161512351960, 0.3844720976410142, 0.4765557207500070},
      {0.3844720976410142, 0.1290161512351960, 0.4765557207500070},
      {0.0099560303737828, 0.1290161512351960, 0.3844720976410142},
      {0.0099560303737828, 0.3844720976410142, 0.1290161512351960},
      {0.0099560303737828, 0.4765557207500070, 0.3844720976410142},
      {0.0099560303737828, 0.4765557207500070, 0.1290161512351960},
      {0.0099560303737828, 0.3844720976410142, 0.4765557207500070},
      {0.0099560303737828, 0.1290161512351960, 0.4765557207500070},
      {0.1290161512351960, 0.0099560303737828, 0.3844720976410142},
      {0.3844720976410142, 0.0099560303737828, 0.1290161512351960},
      {0.4765557207500070, 0.0099560303737828, 0.3844720976410142},
      {0.4765557207500070, 0.0099560303737828, 0.1290161512351960},
      {0.3844720976410142, 0.0099560303737828, 0.4765557207500070},
      {0.1290161512351960, 0.0099560303737828, 0.4765557207500070},
      {0.1290161512351960, 0.3844720976410142, 0.0099560303737828},
      {0.3844720976410142, 0.1290161512351960, 0.0099560303737828},
      {0.4765557207500070, 0.3844720976410142, 0.0099560303737828},
      {0.4765557207500070, 0.1290161512351960, 0.0099560303737828},
      {0.3844720976410142, 0.4765557207500070, 0.0099560303737828},
      {0.1290161512351960, 0.4765557207500070, 0.0099560303737828},
      {0.0137608474867702, 0.1614873223066114, 0.5982705108010641},
      {0.0137608474867702, 0.5982705108010641, 0.1614873223066114},
      {0.1614873223066114, 0.0137608474867702, 0.5982705108010641},
      {0.5982705108010641, 0.0137608474867702, 0.1614873223066114},
      {0.1614873223066114, 0.5982705108010641, 0.0137608474867702},
      {0.5982705108010641, 0.1614873223066114, 0.0137608474867702},
      {0.2264813194055542, 0.1614873223066114, 0.5982705108010641},
      {0.2264813194055542, 0.5982705108010641, 0.1614873223066114},
      {0.2264813194055542, 0.0137608474867702, 0.5982705108010641},
      {0.2264813194055542, 0.0137608474867702, 0.1614873223066114},
      {0.2264813194055542, 0.5982705108010641, 0.0137608474867702},
      {0.2264813194055542, 0.1614873223066114, 0.0137608474867702},
      {0.1614873223066114, 0.2264813194055542, 0.5982705108010641},
      {0.5982705108010641, 0.2264813194055542, 0.1614873223066114},
      {0.0137608474867702, 0.2264813194055542, 0.5982705108010641},
      {0.0137608474867702, 0.2264813194055542, 0.1614873223066114},
      {0.5982705108010641, 0.2264813194055542, 0.0137608474867702},
      {0.1614873223066114, 0.2264813194055542, 0.0137608474867702},
      {0.1614873223066114, 0.5982705108010641, 0.2264813194055542},
      {0.5982705108010641, 0.1614873223066114, 0.2264813194055542},
      {0.0137608474867702, 0.5982705108010641, 0.2264813194055542},
      {0.0137608474867702, 0.1614873223066114, 0.2264813194055542},
      {0.5982705108010641, 0.0137608474867702, 0.2264813194055542},
      {0.1614873223066114, 0.0137608474867702, 0.2264813194055542},
      {0.4143500741402470, 0.1059283078939860, 0.2853013762463641},
      {0.4143500741402470, 0.2853013762463641, 0.1059283078939860},
      {0.1059283078939860, 0.4143500741402470, 0.2853013762463641},
      {0.2853013762463641, 0.4143500741402470, 0.1059283078939860},
      {0.1059283078939860, 0.2853013762463641, 0.4143500741402470},
      {0.2853013762463641, 0.1059283078939860, 0.4143500741402470},
      {0.1944202417194029, 0.1059283078939860, 0.2853013762463641},
      {0.1944202417194029, 0.2853013762463641, 0.1059283078939860},
      {0.1944202417194029, 0.4143500741402470, 0.2853013762463641},
      {0.1944202417194029, 0.4143500741402470, 0.1059283078939860},
      {0.1944202417194029, 0.2853013762463641, 0.4143500741402470},
      {0.1944202417194029, 0.1059283078939860, 0.4143500741402470},
      {0.1059283078939860, 0.1944202417194029, 0.2853013762463641},
      {0.2853013762463641, 0.1944202417194029, 0.1059283078939860},
      {0.4143500741402470, 0.1944202417194029, 0.2853013762463641},
      {0.4143500741402470, 0.1944202417194029, 0.1059283078939860},
      {0.2853013762463641, 0.1944202417194029, 0.4143500741402470},
      {0.1059283078939860, 0.1944202417194029, 0.4143500741402470},
      {0.1059283078939860, 0.2853013762463641, 0.1944202417194029},
      {0.2853013762463641, 0.1059283078939860, 0.1944202417194029},
      {0.4143500741402470, 0.2853013762463641, 0.1944202417194029},
      {0.4143500741402470, 0.1059283078939860, 0.1944202417194029},
      {0.2853013762463641, 0.4143500741402470, 0.1944202417194029},
      {0.1059283078939860, 0.4143500741402470, 0.1944202417194029},
      {0.2864083155254953, 0.2046229328456809, 0.4728908103942965},
      {0.2864083155254953, 0.4728908103942965, 0.2046229328456809},
      {0.2046229328456809, 0.2864083155254953, 0.4728908103942965},
      {0.4728908103942965, 0.2864083155254953, 0.2046229328456809},
      {0.2046229328456809, 0.4728908103942965, 0.2864083155254953},
      {0.4728908103942965, 0.2046229328456809, 0.2864083155254953},
      {0.0360779412345273, 0.2046229328456809, 0.4728908103942965},
      {0.0360779412345273, 0.4728908103942965, 0.2046229328456809},
      {0.0360779412345273, 0.2864083155254953, 0.4728908103942965},
      {0.0360779412345273, 0.2864083155254953, 0.2046229328456809},
      {0.0360779412345273, 0.4728908103942965, 0.2864083155254953},
      {0.0360779412345273, 0.2046229328456809, 0.2864083155254953},
      {0.2046229328456809, 0.0360779412345273, 0.4728908103942965},
      {0.4728908103942965, 0.0360779412345273, 0.2046229328456809},
      {0.2864083155254953, 0.0360779412345273, 0.4728908103942965},
      {0.2864083155254953, 0.0360779412345273, 0.2046229328456809},
      {0.4728908103942965, 0.0360779412345273, 0.2864083155254953},
      {0.2046229328456809, 0.0360779412345273, 0.2864083155254953},
      {0.2046229328456809, 0.4728908103942965, 0.0360779412345273},
      {0.4728908103942965, 0.2046229328456809, 0.0360779412345273},
      {0.2864083155254953, 0.4728908103942965, 0.0360779412345273},
      {0.2864083155254953, 0.2046229328456809, 0.0360779412345273},
      {0.4728908103942965, 0.2864083155254953, 0.0360779412345273},
      {0.2046229328456809, 0.2864083155254953, 0.0360779412345273},
      {0.0236158323301530, 0.0744179686987702, 0.0024421697078358},
      {0.0236158323301530, 0.0024421697078358, 0.0744179686987702},
      {0.0744179686987702, 0.0236158323301530, 0.0024421697078358},
      {0.0024421697078358, 0.0236158323301530, 0.0744179686987702},
      {0.0744179686987702, 0.0024421697078358, 0.0236158323301530},
      {0.0024421697078358, 0.0744179686987702, 0.0236158323301530},
      {0.8995240292632410, 0.0744179686987702, 0.0024421697078358},
      {0.8995240292632410, 0.0024421697078358, 0.0744179686987702},
      {0.8995240292632410, 0.0236158323301530, 0.0024421697078358},
      {0.8995240292632410, 0.0236158323301530, 0.0744179686987702},
      {0.8995240292632410, 0.0024421697078358, 0.0236158323301530},
      {0.8995240292632410, 0.0744179686987702, 0.0236158323301530},
      {0.0744179686987702, 0.8995240292632410, 0.0024421697078358},
      {0.0024421697078358, 0.8995240292632410, 0.0744179686987702},
      {0.0236158323301530, 0.8995240292632410, 0.0024421697078358},
      {0.0236158323301530, 0.8995240292632410, 0.0744179686987702},
      {0.0024421697078358, 0.8995240292632410, 0.0236158323301530},
      {0.0744179686987702, 0.8995240292632410, 0.0236158323301530},
      {0.0744179686987702, 0.0024421697078358, 0.8995240292632410},
      {0.0024421697078358, 0.0744179686987702, 0.8995240292632410},
      {0.0236158323301530, 0.0024421697078358, 0.8995240292632410},
      {0.0236158323301530, 0.0744179686987702, 0.8995240292632410},
      {0.0024421697078358, 0.0236158323301530, 0.8995240292632410},
      {0.0744179686987702, 0.0236158323301530, 0.8995240292632410},
      {0.5078745759286403, 0.2938352359686784, 0.1939104157388909},
      {0.5078745759286403, 0.1939104157388909, 0.2938352359686784},
      {0.2938352359686784, 0.5078745759286403, 0.1939104157388909},
      {0.1939104157388909, 0.5078745759286403, 0.2938352359686784},
      {0.2938352359686784, 0.1939104157388909, 0.5078745759286403},
      {0.1939104157388909, 0.2938352359686784, 0.5078745759286403},
      {0.0043797723637903, 0.2938352359686784, 0.1939104157388909},
      {0.0043797723637903, 0.1939104157388909, 0.2938352359686784},
      {0.0043797723637903, 0.5078745759286403, 0.1939104157388909},
      {0.0043797723637903, 0.5078745759286403, 0.2938352359686784},
      {0.0043797723637903, 0.1939104157388909, 0.5078745759286403},
      {0.0043797723637903, 0.2938352359686784, 0.5078745759286403},
      {0.2938352359686784, 0.0043797723637903, 0.1939104157388909},
      {0.1939104157388909, 0.0043797723637903, 0.2938352359686784},
      {0.5078745759286403, 0.0043797723637903, 0.1939104157388909},
      {0.5078745759286403, 0.0043797723637903, 0.2938352359686784},
      {0.1939104157388909, 0.0043797723637903, 0.5078745759286403},
      {0.2938352359686784, 0.0043797723637903, 0.5078745759286403},
      {0.2938352359686784, 0.1939104157388909, 0.0043797723637903},
      {0.1939104157388909, 0.2938352359686784, 0.0043797723637903},
      {0.5078745759286403, 0.1939104157388909, 0.0043797723637903},
      {0.5078745759286403, 0.2938352359686784, 0.0043797723637903},
      {0.1939104157388909, 0.5078745759286403, 0.0043797723637903},
      {0.2938352359686784, 0.5078745759286403, 0.0043797723637903},
      {0.0023886149405118, 0.2899004919931303, 0.6867525446175153},
      {0.0023886149405118, 0.6867525446175153, 0.2899004919931303},
      {0.2899004919931303, 0.0023886149405118, 0.6867525446175153},
      {0.6867525446175153, 0.0023886149405118, 0.2899004919931303},
      {0.2899004919931303, 0.6867525446175153, 0.0023886149405118},
      {0.6867525446175153, 0.2899004919931303, 0.0023886149405118},
      {0.0209583484488425, 0.2899004919931303, 0.6867525446175153},
      {0.0209583484488425, 0.6867525446175153, 0.2899004919931303},
      {0.0209583484488425, 0.0023886149405118, 0.6867525446175153},
      {0.0209583484488425, 0.0023886149405118, 0.2899004919931303},
      {0.0209583484488425, 0.6867525446175153, 0.0023886149405118},
      {0.0209583484488425, 0.2899004919931303, 0.0023886149405118},
      {0.2899004919931303, 0.0209583484488425, 0.6867525446175153},
      {0.6867525446175153, 0.0209583484488425, 0.2899004919931303},
      {0.0023886149405118, 0.0209583484488425, 0.6867525446175153},
      {0.0023886149405118, 0.0209583484488425, 0.2899004919931303},
      {0.6867525446175153, 0.0209583484488425, 0.0023886149405118},
      {0.2899004919931303, 0.0209583484488425, 0.0023886149405118},
      {0.2899004919931303, 0.6867525446175153, 0.0209583484488425},
      {0.6867525446175153, 0.2899004919931303, 0.0209583484488425},
      {0.0023886149405118, 0.6867525446175153, 0.0209583484488425},
      {0.0023886149405118, 0.2899004919931303, 0.0209583484488425},
      {0.6867525446175153, 0.0023886149405118, 0.0209583484488425},
      {0.2899004919931303, 0.0023886149405118, 0.0209583484488425},
      {0.2780090233488638, 0.0069203955276090, 0.6284083367674260},
      {0.2780090233488638, 0.6284083367674260, 0.0069203955276090},
      {0.0069203955276090, 0.2780090233488638, 0.6284083367674260},
      {0.6284083367674260, 0.2780090233488638, 0.0069203955276090},
      {0.0069203955276090, 0.6284083367674260, 0.2780090233488638},
      {0.6284083367674260, 0.0069203955276090, 0.2780090233488638},
      {0.0866622443561012, 0.0069203955276090, 0.6284083367674260},
      {0.0866622443561012, 0.6284083367674260, 0.0069203955276090},
      {0.0866622443561012, 0.2780090233488638, 0.6284083367674260},
      {0.0866622443561012, 0.2780090233488638, 0.0069203955276090},
      {0.0866622443561012, 0.6284083367674260, 0.2780090233488638},
      {0.0866622443561012, 0.0069203955276090, 0.2780090233488638},
      {0.0069203955276090, 0.0866622443561012, 0.6284083367674260},
      {0.6284083367674260, 0.0866622443561012, 0.0069203955276090},
      {0.2780090233488638, 0.0866622443561012, 0.6284083367674260},
      {0.2780090233488638, 0.0866622443561012, 0.0069203955276090},
      {0.6284083367674260, 0.0866622443561012, 0.2780090233488638},
      {0.0069203955276090, 0.0866622443561012, 0.2780090233488638},
      {0.0069203955276090, 0.6284083367674260, 0.0866622443561012},
      {0.6284083367674260, 0.0069203955276090, 0.0866622443561012},
      {0.2780090233488638, 0.6284083367674260, 0.0866622443561012},
      {0.2780090233488638, 0.0069203955276090, 0.0866622443561012},
      {0.6284083367674260, 0.2780090233488638, 0.0866622443561012},
      {0.0069203955276090, 0.2780090233488638, 0.0866622443561012},
      {0.5431130574905816, 0.1053191068972152, 0.3053177470209279},
      {0.5431130574905816, 0.3053177470209279, 0.1053191068972152},
      {0.1053191068972152, 0.5431130574905816, 0.3053177470209279},
      {0.3053177470209279, 0.5431130574905816, 0.1053191068972152},
      {0.1053191068972152, 0.3053177470209279, 0.5431130574905816},
      {0.3053177470209279, 0.1053191068972152, 0.5431130574905816},
      {0.0462500885912754, 0.1053191068972152, 0.3053177470209279},
      {0.0462500885912754, 0.3053177470209279, 0.1053191068972152},
      {0.0462500885912754, 0.5431130574905816, 0.3053177470209279},
      {0.0462500885912754, 0.5431130574905816, 0.1053191068972152},
      {0.0462500885912754, 0.3053177470209279, 0.5431130574905816},
      {0.0462500885912754, 0.1053191068972152, 0.5431130574905816},
      {0.1053191068972152, 0.0462500885912754, 0.3053177470209279},
      {0.3053177470209279, 0.0462500885912754, 0.1053191068972152},
      {0.5431130574905816, 0.0462500885912754, 0.3053177470209279},
      {0.5431130574905816, 0.0462500885912754, 0.1053191068972152},
      {0.3053177470209279, 0.0462500885912754, 0.5431130574905816},
      {0.1053191068972152, 0.0462500885912754, 0.5431130574905816},
      {0.1053191068972152, 0.3053177470209279, 0.0462500885912754},
      {0.3053177470209279, 0.1053191068972152, 0.0462500885912754},
      {0.5431130574905816, 0.3053177470209279, 0.0462500885912754},
      {0.5431130574905816, 0.1053191068972152, 0.0462500885912754},
      {0.3053177470209279, 0.5431130574905816, 0.0462500885912754},
      {0.1053191068972152, 0.5431130574905816, 0.0462500885912754},
      {0.0087923252192588, 0.3387615700292962, 0.2521955658942761},
      {0.0087923252192588, 0.2521955658942761, 0.3387615700292962},
      {0.3387615700292962, 0.0087923252192588, 0.2521955658942761},
      {0.2521955658942761, 0.0087923252192588, 0.3387615700292962},
      {0.3387615700292962, 0.2521955658942761, 0.0087923252192588},
      {0.2521955658942761, 0.3387615700292962, 0.0087923252192588},
      {0.4002505388571689, 0.3387615700292962, 0.2521955658942761},
      {0.4002505388571689, 0.2521955658942761, 0.3387615700292962},
      {0.4002505388571689, 0.0087923252192588, 0.2521955658942761},
      {0.4002505388571689, 0.0087923252192588, 0.3387615700292962},
      {0.4002505388571689, 0.2521955658942761, 0.0087923252192588},
      {0.4002505388571689, 0.3387615700292962, 0.0087923252192588},
      {0.3387615700292962, 0.4002505388571689, 0.2521955658942761},
      {0.2521955658942761, 0.4002505388571689, 0.3387615700292962},
      {0.0087923252192588, 0.4002505388571689, 0.2521955658942761},
      {0.0087923252192588, 0.4002505388571689, 0.3387615700292962},
      {0.2521955658942761, 0.4002505388571689, 0.0087923252192588},
      {0.3387615700292962, 0.4002505388571689, 0.0087923252192588},
      {0.3387615700292962, 0.2521955658942761, 0.4002505388571689},
      {0.2521955658942761, 0.3387615700292962, 0.4002505388571689},
      {0.0087923252192588, 0.2521955658942761, 0.4002505388571689},
      {0.0087923252192588, 0.3387615700292962, 0.4002505388571689},
      {0.2521955658942761, 0.0087923252192588, 0.4002505388571689},
      {0.3387615700292962, 0.0087923252192588, 0.4002505388571689}},
      {0.0009030757983474,
      0.0009030757983474,
      0.0009030757983474,
      0.0009030757983474,
      0.0006368163313748,
      0.0006368163313748,
      0.0006368163313748,
      0.0006368163313748,
      0.0006762430253890,
      0.0006762430253890,
      0.0006762430253890,
      0.0006762430253890,
      0.0006912783829584,
      0.0006912783829584,
      0.0006912783829584,
      0.0006912783829584,
      0.0008815944349891,
      0.0008815944349891,
      0.0008815944349891,
      0.0008815944349891,
      0.0001620069699771,
      0.0001620069699771,
      0.0001620069699771,
      0.0001620069699771,
      0.0004980293561301,
      0.0004980293561301,
      0.0004980293561301,
      0.0004980293561301,
      0.0004980293561301,
      0.0004980293561301,
      0.0011769578205881,
      0.0011769578205881,
      0.0011769578205881,
      0.0011769578205881,
      0.0011769578205881,
      0.0011769578205881,
      0.0001896690274777,
      0.0001896690274777,
      0.0001896690274777,
      0.0001896690274777,
      0.0001896690274777,
      0.0001896690274777,
      0.0001896690274777,
      0.0001896690274777,
      0.0001896690274777,
      0.0001896690274777,
      0.0001896690274777,
      0.0001896690274777,
      0.0000547388014907,
      0.0000547388014907,
      0.0000547388014907,
      0.0000547388014907,
      0.0000547388014907,
      0.0000547388014907,
      0.0000547388014907,
      0.0000547388014907,
      0.0000547388014907,
      0.0000547388014907,
      0.0000547388014907,
      0.0000547388014907,
      0.0007528841790889,
      0.0007528841790889,
      0.0007528841790889,
      0.0007528841790889,
      0.0007528841790889,
      0.0007528841790889,
      0.0007528841790889,
      0.0007528841790889,
      0.0007528841790889,
      0.0007528841790889,
      0.0007528841790889,
      0.0007528841790889,
      0.0004573261830233,
      0.0004573261830233,
      0.0004573261830233,
      0.0004573261830233,
      0.0004573261830233,
      0.0004573261830233,
      0.0004573261830233,
      0.0004573261830233,
      0.0004573261830233,
      0.0004573261830233,
      0.0004573261830233,
      0.0004573261830233,
      0.0000890850671644,
      0.0000890850671644,
      0.0000890850671644,
      0.0000890850671644,
      0.0000890850671644,
      0.0000890850671644,
      0.0000890850671644,
      0.0000890850671644,
      0.0000890850671644,
      0.0000890850671644,
      0.0000890850671644,
      0.0000890850671644,
      0.0001421267128894,
      0.0001421267128894,
      0.0001421267128894,
      0.0001421267128894,
      0.0001421267128894,
      0.0001421267128894,
      0.0001421267128894,
      0.0001421267128894,
      0.0001421267128894,
      0.0001421267128894,
      0.0001421267128894,
      0.0001421267128894,
      0.0002681122201584,
      0.0002681122201584,
      0.0002681122201584,
      0.0002681122201584,
      0.0002681122201584,
      0.0002681122201584,
      0.0002681122201584,
      0.0002681122201584,
      0.0002681122201584,
      0.0002681122201584,
      0.0002681122201584,
      0.0002681122201584,
      0.0003606702982705,
      0.0003606702982705,
      0.0003606702982705,
      0.0003606702982705,
      0.0003606702982705,
      0.0003606702982705,
      0.0003606702982705,
      0.0003606702982705,
      0.0003606702982705,
      0.0003606702982705,
      0.0003606702982705,
      0.0003606702982705,
      0.0002164940919985,
      0.0002164940919985,
      0.0002164940919985,
      0.0002164940919985,
      0.0002164940919985,
      0.0002164940919985,
      0.0002164940919985,
      0.0002164940919985,
      0.0002164940919985,
      0.0002164940919985,
      0.0002164940919985,
      0.0002164940919985,
      0.0001256703514163,
      0.0001256703514163,
      0.0001256703514163,
      0.0001256703514163,
      0.0001256703514163,
      0.0001256703514163,
      0.0001256703514163,
      0.0001256703514163,
      0.0001256703514163,
      0.0001256703514163,
      0.0001256703514163,
      0.0001256703514163,
      0.0006560653628662,
      0.0006560653628662,
      0.0006560653628662,
      0.0006560653628662,
      0.0006560653628662,
      0.0006560653628662,
      0.0006560653628662,
      0.0006560653628662,
      0.0006560653628662,
      0.0006560653628662,
      0.0006560653628662,
      0.0006560653628662,
      0.0000137673954239,
      0.0000137673954239,
      0.0000137673954239,
      0.0000137673954239,
      0.0000137673954239,
      0.0000137673954239,
      0.0000137673954239,
      0.0000137673954239,
      0.0000137673954239,
      0.0000137673954239,
      0.0000137673954239,
      0.0000137673954239,
      0.0003670989302110,
      0.0003670989302110,
      0.0003670989302110,
      0.0003670989302110,
      0.0003670989302110,
      0.0003670989302110,
      0.0003670989302110,
      0.0003670989302110,
      0.0003670989302110,
      0.0003670989302110,
      0.0003670989302110,
      0.0003670989302110,
      0.0001904853813854,
      0.0001904853813854,
      0.0001904853813854,
      0.0001904853813854,
      0.0001904853813854,
      0.0001904853813854,
      0.0001904853813854,
      0.0001904853813854,
      0.0001904853813854,
      0.0001904853813854,
      0.0001904853813854,
      0.0001904853813854,
      0.0001904853813854,
      0.0001904853813854,
      0.0001904853813854,
      0.0001904853813854,
      0.0001904853813854,
      0.0001904853813854,
      0.0001904853813854,
      0.0001904853813854,
      0.0001904853813854,
      0.0001904853813854,
      0.0001904853813854,
      0.0001904853813854,
      0.0001339556863314,
      0.0001339556863314,
      0.0001339556863314,
      0.0001339556863314,
      0.0001339556863314,
      0.0001339556863314,
      0.0001339556863314,
      0.0001339556863314,
      0.0001339556863314,
      0.0001339556863314,
      0.0001339556863314,
      0.0001339556863314,
      0.0001339556863314,
      0.0001339556863314,
      0.0001339556863314,
      0.0001339556863314,
      0.0001339556863314,
      0.0001339556863314,
      0.0001339556863314,
      0.0001339556863314,
      0.0001339556863314,
      0.0001339556863314,
      0.0001339556863314,
      0.0001339556863314,
      0.0001073509180939,
      0.0001073509180939,
      0.0001073509180939,
      0.0001073509180939,
      0.0001073509180939,
      0.0001073509180939,
      0.0001073509180939,
      0.0001073509180939,
      0.0001073509180939,
      0.0001073509180939,
      0.0001073509180939,
      0.0001073509180939,
      0.0001073509180939,
      0.0001073509180939,
      0.0001073509180939,
      0.0001073509180939,
      0.0001073509180939,
      0.0001073509180939,
      0.0001073509180939,
      0.0001073509180939,
      0.0001073509180939,
      0.0001073509180939,
      0.0001073509180939,
      0.0001073509180939,
      0.0003808005102613,
      0.0003808005102613,
      0.0003808005102613,
      0.0003808005102613,
      0.0003808005102613,
      0.0003808005102613,
      0.0003808005102613,
      0.0003808005102613,
      0.0003808005102613,
      0.0003808005102613,
      0.0003808005102613,
      0.0003808005102613,
      0.0003808005102613,
      0.0003808005102613,
      0.0003808005102613,
      0.0003808005102613,
      0.0003808005102613,
      0.0003808005102613,
      0.0003808005102613,
      0.0003808005102613,
      0.0003808005102613,
      0.0003808005102613,
      0.0003808005102613,
      0.0003808005102613,
      0.0002947025289006,
      0.0002947025289006,
      0.0002947025289006,
      0.0002947025289006,
      0.0002947025289006,
      0.0002947025289006,
      0.0002947025289006,
      0.0002947025289006,
      0.0002947025289006,
      0.0002947025289006,
      0.0002947025289006,
      0.0002947025289006,
      0.0002947025289006,
      0.0002947025289006,
      0.0002947025289006,
      0.0002947025289006,
      0.0002947025289006,
      0.0002947025289006,
      0.0002947025289006,
      0.0002947025289006,
      0.0002947025289006,
      0.0002947025289006,
      0.0002947025289006,
      0.0002947025289006,
      0.0002273488527673,
      0.0002273488527673,
      0.0002273488527673,
      0.0002273488527673,
      0.0002273488527673,
      0.0002273488527673,
      0.0002273488527673,
      0.0002273488527673,
      0.0002273488527673,
      0.0002273488527673,
      0.0002273488527673,
      0.0002273488527673,
      0.0002273488527673,
      0.0002273488527673,
      0.0002273488527673,
      0.0002273488527673,
      0.0002273488527673,
      0.0002273488527673,
      0.0002273488527673,
      0.0002273488527673,
      0.0002273488527673,
      0.0002273488527673,
      0.0002273488527673,
      0.0002273488527673,
      0.0002349188367070,
      0.0002349188367070,
      0.0002349188367070,
      0.0002349188367070,
      0.0002349188367070,
      0.0002349188367070,
      0.0002349188367070,
      0.0002349188367070,
      0.0002349188367070,
      0.0002349188367070,
      0.0002349188367070,
      0.0002349188367070,
      0.0002349188367070,
      0.0002349188367070,
      0.0002349188367070,
      0.0002349188367070,
      0.0002349188367070,
      0.0002349188367070,
      0.0002349188367070,
      0.0002349188367070,
      0.0002349188367070,
      0.0002349188367070,
      0.0002349188367070,
      0.0002349188367070,
      0.0008920326954702,
      0.0008920326954702,
      0.0008920326954702,
      0.0008920326954702,
      0.0008920326954702,
      0.0008920326954702,
      0.0008920326954702,
      0.0008920326954702,
      0.0008920326954702,
      0.0008920326954702,
      0.0008920326954702,
      0.0008920326954702,
      0.0008920326954702,
      0.0008920326954702,
      0.0008920326954702,
      0.0008920326954702,
      0.0008920326954702,
      0.0008920326954702,
      0.0008920326954702,
      0.0008920326954702,
      0.0008920326954702,
      0.0008920326954702,
      0.0008920326954702,
      0.0008920326954702,
      0.0003928752922998,
      0.0003928752922998,
      0.0003928752922998,
      0.0003928752922998,
      0.0003928752922998,
      0.0003928752922998,
      0.0003928752922998,
      0.0003928752922998,
      0.0003928752922998,
      0.0003928752922998,
      0.0003928752922998,
      0.0003928752922998,
      0.0003928752922998,
      0.0003928752922998,
      0.0003928752922998,
      0.0003928752922998,
      0.0003928752922998,
      0.0003928752922998,
      0.0003928752922998,
      0.0003928752922998,
      0.0003928752922998,
      0.0003928752922998,
      0.0003928752922998,
      0.0003928752922998,
      0.0000389250046920,
      0.0000389250046920,
      0.0000389250046920,
      0.0000389250046920,
      0.0000389250046920,
      0.0000389250046920,
      0.0000389250046920,
      0.0000389250046920,
      0.0000389250046920,
      0.0000389250046920,
      0.0000389250046920,
      0.0000389250046920,
      0.0000389250046920,
      0.0000389250046920,
      0.0000389250046920,
      0.0000389250046920,
      0.0000389250046920,
      0.0000389250046920,
      0.0000389250046920,
      0.0000389250046920,
      0.0000389250046920,
      0.0000389250046920,
      0.0000389250046920,
      0.0000389250046920,
      0.0001383198518408,
      0.0001383198518408,
      0.0001383198518408,
      0.0001383198518408,
      0.0001383198518408,
      0.0001383198518408,
      0.0001383198518408,
      0.0001383198518408,
      0.0001383198518408,
      0.0001383198518408,
      0.0001383198518408,
      0.0001383198518408,
      0.0001383198518408,
      0.0001383198518408,
      0.0001383198518408,
      0.0001383198518408,
      0.0001383198518408,
      0.0001383198518408,
      0.0001383198518408,
      0.0001383198518408,
      0.0001383198518408,
      0.0001383198518408,
      0.0001383198518408,
      0.0001383198518408,
      0.0000580054545937,
      0.0000580054545937,
      0.0000580054545937,
      0.0000580054545937,
      0.0000580054545937,
      0.0000580054545937,
      0.0000580054545937,
      0.0000580054545937,
      0.0000580054545937,
      0.0000580054545937,
      0.0000580054545937,
      0.0000580054545937,
      0.0000580054545937,
      0.0000580054545937,
      0.0000580054545937,
      0.0000580054545937,
      0.0000580054545937,
      0.0000580054545937,
      0.0000580054545937,
      0.0000580054545937,
      0.0000580054545937,
      0.0000580054545937,
      0.0000580054545937,
      0.0000580054545937,
      0.0001802177495766,
      0.0001802177495766,
      0.0001802177495766,
      0.0001802177495766,
      0.0001802177495766,
      0.0001802177495766,
      0.0001802177495766,
      0.0001802177495766,
      0.0001802177495766,
      0.0001802177495766,
      0.0001802177495766,
      0.0001802177495766,
      0.0001802177495766,
      0.0001802177495766,
      0.0001802177495766,
      0.0001802177495766,
      0.0001802177495766,
      0.0001802177495766,
      0.0001802177495766,
      0.0001802177495766,
      0.0001802177495766,
      0.0001802177495766,
      0.0001802177495766,
      0.0001802177495766,
      0.0005605988321073,
      0.0005605988321073,
      0.0005605988321073,
      0.0005605988321073,
      0.0005605988321073,
      0.0005605988321073,
      0.0005605988321073,
      0.0005605988321073,
      0.0005605988321073,
      0.0005605988321073,
      0.0005605988321073,
      0.0005605988321073,
      0.0005605988321073,
      0.0005605988321073,
      0.0005605988321073,
      0.0005605988321073,
      0.0005605988321073,
      0.0005605988321073,
      0.0005605988321073,
      0.0005605988321073,
      0.0005605988321073,
      0.0005605988321073,
      0.0005605988321073,
      0.0005605988321073,
      0.0001898032539922,
      0.0001898032539922,
      0.0001898032539922,
      0.0001898032539922,
      0.0001898032539922,
      0.0001898032539922,
      0.0001898032539922,
      0.0001898032539922,
      0.0001898032539922,
      0.0001898032539922,
      0.0001898032539922,
      0.0001898032539922,
      0.0001898032539922,
      0.0001898032539922,
      0.0001898032539922,
      0.0001898032539922,
      0.0001898032539922,
      0.0001898032539922,
      0.0001898032539922,
      0.0001898032539922,
      0.0001898032539922,
      0.0001898032539922,
      0.0001898032539922,
      0.0001898032539922}
    } // end Tet
  };
    
} // end namespace Intrepid2
