/*************************************************************************
 *    Copyright (C) 2008 Christian Kuka <ckuka@madkooky.de>
 *
 *    This file is part of PQStego.
 *
 *    PQStego is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    PQStego is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <jpeglib.h>
#include <string.h>
#include <setjmp.h>
#include <getopt.h>
#include <sys/stat.h>
#include <math.h>
#include "pqstego.h"

#define PROGRAM_NAME "pqstego"

#define AUTHORS "Christian Kuka"




char *program_name;
static char const *infile;

void usage(int status) {
    if (status != EXIT_SUCCESS)
        fprintf(stderr, "Try `%s --help' for more information.\n",
            program_name);
    else {
        printf("\
Usage: %s [OPTION] [MESSAGEFILE] [IMAGEFILE] [OUTFILE]\n\
", program_name);
        fputs("\
Embed/Extract MESSAGEFILE in/from IMAGEFILE.\n\
\n\
  -c, --capacity            print capacity of cover image\n\
  -q, --quality[=QUALITY]   force quality for stego image\n\
  -p, --password[=PASSWORD] use password\n\
  -e, --embed               embed message in cover image\n\
  -x, --extract             extract message from stego image\n\
  -h, --help                show help\n\
  -v, --version             show version\n\
", stdout);

        //fputs(HELP_OPTION_DESCRIPTION, stdout);
        //fputs(VERSION_OPTION_DESCRIPTION, stdout);
        printf("\
\n\
Examples:\n\
  %s -e message.txt cover.jpg stego.jpg\n\
  Embed message.txt in cover.jpg and save it as stego.jpg.\n\
  \n\
  %s -x stego.jpg message.txt\n\
  Extract message from stego.jpg and save it as message.txt.\n\
", program_name, program_name);
        //emit_bug_reporting_address();
    }
    exit(status);

}

static long jround_up(long a, long b) {
    a += b - 1L;
    return a - (a % b);
}

struct pqstego_jpeg_error_mgr {
    struct jpeg_error_mgr pub;
    jmp_buf setjmp_buffer;
};

typedef struct pqstego_jpeg_error_mgr * pqstego_jpeg_error_ptr;

void pqstego_jpeg_error_exit(j_common_ptr cinfo) {
    pqstego_jpeg_error_ptr err = (pqstego_jpeg_error_ptr) cinfo->err;

    /* Always display the message. */
    /* We could postpone this until after returning, if we chose. */
    (*cinfo->err->output_message) (cinfo);

    /* Return control to the setjmp point */
    longjmp(err->setjmp_buffer, 1);
}

uint8_t pqstego_embed(const uint8_t *coverfilename, const uint8_t *stegofilename, const uint8_t *messagefilename, pq_parameter_t *param) {
    struct pqstego_jpeg_error_mgr jerr;
    jvirt_barray_ptr *src_coef_array = NULL;
    jvirt_barray_ptr *dst_coef_array = NULL;
    JBLOCKARRAY coef_buffer;
    jpeg_component_info *compinfo = NULL;
    JQUANT_TBL *quant_table_ptr = NULL;
    JDIMENSION i, ii, iii, iv;
    struct jpeg_compress_struct *cinfo;
    struct jpeg_decompress_struct *dinfo;
    pq_data_t *pq_src_data = NULL;
    pq_data_t *pq_dst_data = NULL;
    FILE *srcfile;
    FILE *dstfile;
    FILE *msgfile;
    uint8_t *message;
    uint32_t msglen;
    uint8_t exit = EXIT_SUCCESS;

    message = (uint8_t*) malloc(sizeof (uint8_t) * 1);
    if ((msgfile = fopen(messagefilename, "rb")) == NULL) {
        return EXIT_FAILURE;
    }
    fseek(msgfile, 0L, SEEK_END);
    msglen = ftell(msgfile);
    rewind(msgfile);
    message = (uint8_t*) malloc(sizeof (uint8_t) * msglen);
    fread(message, 1, msglen, msgfile);
    fclose(msgfile);
    pq_src_data = (pq_data_t*) malloc(sizeof (pq_data_t));
    pq_dst_data = (pq_data_t*) malloc(sizeof (pq_data_t));

    cinfo = malloc(sizeof (struct jpeg_compress_struct));
    dinfo = malloc(sizeof (struct jpeg_decompress_struct));
    if ((!cinfo) || (!dinfo) || (!pq_src_data) || (!pq_dst_data)) {
        return EXIT_FAILURE;
    }
    dinfo->err = jpeg_std_error(&jerr.pub);
    cinfo->err = jpeg_std_error(&jerr.pub);
    if ((srcfile = fopen(coverfilename, "rb")) == NULL) {
        return EXIT_FAILURE;
    }
    jpeg_create_decompress(dinfo);
    jpeg_create_compress(cinfo);
    cinfo->mem->max_memory_to_use = dinfo->mem->max_memory_to_use;
    jpeg_stdio_src(dinfo, srcfile);
    jpeg_read_header(dinfo, TRUE);
    src_coef_array = jpeg_read_coefficients(dinfo);

    pq_src_data->components = dinfo->num_components;
    pq_src_data->component = malloc(sizeof (pq_component_t) * dinfo->num_components);
    if (!pq_src_data->component) {
        return EXIT_FAILURE;
    }
    pq_src_data->size_x = dinfo->image_width;
    pq_src_data->size_y = dinfo->image_height;

    for (i = 0; i < dinfo->num_components; i++) {
        compinfo = (dinfo->comp_info + i);
        JDIMENSION blocks = compinfo->width_in_blocks * compinfo->height_in_blocks;
        (pq_src_data->component + i)->quant = (pq_quant_t*) malloc(sizeof (pq_quant_t));
        if (!(pq_src_data->component + i)->quant) {
            return EXIT_FAILURE;
        }
        (pq_src_data->component + i)->block = (pq_block_t*) malloc(sizeof (pq_block_t) * blocks);
        if (!(pq_src_data->component + i)->block) {
            return EXIT_FAILURE;
        }
        (pq_src_data->component + i)->blocks = blocks;
        memcpy((pq_src_data->component + i)->quant->values, compinfo->quant_table->quantval, sizeof (pq_quant_t));
        for (ii = 0; ii < compinfo->height_in_blocks; ii++) {
            coef_buffer = (dinfo->mem->access_virt_barray)
                    ((j_common_ptr) dinfo, *(src_coef_array + i), ii, (JDIMENSION) 1, FALSE);
            JBLOCKROW block_row = *(coef_buffer);
            for (iii = 0; iii < compinfo->width_in_blocks; iii++) {
                memcpy(((pq_src_data->component + i)->block + (ii * compinfo->width_in_blocks) + iii)->values,
                        (block_row + iii), sizeof (pq_block_t));
            }

        }

    }

    exit = pq_embed(pq_src_data, pq_dst_data, message, msglen, param);
    if (exit == EXIT_SUCCESS) {
        if ((dstfile = fopen(stegofilename, "wb")) == NULL) {
            return EXIT_FAILURE;
        }

        jpeg_copy_critical_parameters(dinfo, cinfo);
        jpeg_stdio_dest(cinfo, dstfile);


        dst_coef_array = (jvirt_barray_ptr *) (cinfo->mem->alloc_small)
                ((j_common_ptr) cinfo, JPOOL_IMAGE, sizeof (jvirt_barray_ptr) * pq_dst_data->components);

        for (i = 0; i < pq_dst_data->components; i++) {
            compinfo = (dinfo->comp_info + i);
            JDIMENSION width = (JDIMENSION) jround_up((long) compinfo->width_in_blocks, (long) compinfo->h_samp_factor);
            JDIMENSION height = (JDIMENSION) jround_up((long) compinfo->height_in_blocks, (long) compinfo->v_samp_factor);
            *(dst_coef_array + i) = (cinfo->mem->request_virt_barray)
                    ((j_common_ptr) cinfo, JPOOL_IMAGE, 0, width, height,
                    (JDIMENSION) compinfo->v_samp_factor);
        }


        for (i = 0; i < NUM_QUANT_TBLS; i++) {
            quant_table_ptr = *(cinfo->quant_tbl_ptrs + i);
            if (quant_table_ptr) {
                for (ii = 0; ii < DCTSIZE2; ii++) {
                    *(quant_table_ptr->quantval + ii) = (unsigned int) *((pq_dst_data->component + i)->quant->values + ii);
                }
            }
        }
        jpeg_write_coefficients(cinfo, dst_coef_array);

        JBLOCKARRAY buf;
        for (i = 0; i < cinfo->num_components; i++) {
            compinfo = (cinfo->comp_info + i);
            for (ii = 0; ii < compinfo->height_in_blocks; ii++) {
                buf = (cinfo->mem->access_virt_barray)
                        ((j_common_ptr) cinfo, *(dst_coef_array + i), ii, (JDIMENSION) 1, 1);
                for (iii = 0; iii < compinfo->width_in_blocks; iii++) {
                    for (iv = 0; iv < DCTSIZE2; iv++) {
                        buf[0][iii][iv] = *(((pq_dst_data->component + i)->block + iii + ii * compinfo->width_in_blocks)->values + iv);
                    }
                }
            }
        }
        jpeg_finish_compress(cinfo);
        jpeg_destroy_compress(cinfo);
        fclose(dstfile);
    }
    jpeg_finish_decompress(dinfo);
    jpeg_destroy_decompress(dinfo);
    fclose(srcfile);


    return exit;

}

uint8_t pqstego_extract(const uint8_t *stegofilename, const uint8_t *messagefilename, pq_parameter_t *param) {
    struct pqstego_jpeg_error_mgr jerr;
    jvirt_barray_ptr *coef_array;
    JBLOCKARRAY coef_buffer;
    JDIMENSION i, ii, iii;
    struct jpeg_decompress_struct *dinfo;
    pq_data_t *pq_src_data = NULL;
    FILE *srcfile;
    FILE *msgfile;
    uint8_t *message;
    uint32_t msglen;
    uint8_t exit = EXIT_SUCCESS;

    pq_src_data = (pq_data_t*) malloc(sizeof (pq_data_t));
    dinfo = malloc(sizeof (struct jpeg_decompress_struct));
    dinfo->err = jpeg_std_error(&jerr.pub);

    if ((srcfile = fopen(stegofilename, "rb")) == NULL) {
        return EXIT_FAILURE;
    }
    jpeg_create_decompress(dinfo);
    jpeg_stdio_src(dinfo, srcfile);
    jpeg_read_header(dinfo, TRUE);

    coef_array = jpeg_read_coefficients(dinfo);
    pq_src_data->components = dinfo->num_components;
    pq_src_data->component = malloc(sizeof (pq_component_t) * dinfo->num_components);
    if (!pq_src_data->component) {
        return EXIT_FAILURE;
    }
    pq_src_data->size_x = dinfo->image_width;
    pq_src_data->size_y = dinfo->image_height;


    for (i = 0; i < dinfo->num_components; i++) {
        jpeg_component_info *compinfo = (dinfo->comp_info + i);
        JDIMENSION blocks = compinfo->width_in_blocks * compinfo->height_in_blocks;
        (pq_src_data->component + i)->quant = (pq_quant_t*) malloc(sizeof (pq_quant_t));
        if (!(pq_src_data->component + i)->quant) {
            return EXIT_FAILURE;
        }
        (pq_src_data->component + i)->block = (pq_block_t*) malloc(sizeof (pq_block_t) * blocks);
        if (!(pq_src_data->component + i)->block) {
            return EXIT_FAILURE;
        }
        (pq_src_data->component + i)->blocks = blocks;
        memcpy((pq_src_data->component + i)->quant->values, compinfo->quant_table->quantval, sizeof (pq_quant_t));
        for (ii = 0; ii < compinfo->height_in_blocks; ii++) {
            coef_buffer = (dinfo->mem->access_virt_barray)
                    ((j_common_ptr) dinfo, *(coef_array + i), ii, (JDIMENSION) 1, FALSE);
            JBLOCKROW block_row = *(coef_buffer);
            for (iii = 0; iii < compinfo->width_in_blocks; iii++) {
                memcpy(((pq_src_data->component + i)->block + (ii * compinfo->width_in_blocks) + iii)->values,
                        (block_row + iii), sizeof (pq_block_t));
            }

        }

    }
    fclose(srcfile);

    exit = pq_extract(pq_src_data, &message, &msglen, param);
    if ((msgfile = fopen(messagefilename, "wb")) == NULL) {
        return EXIT_FAILURE;
    }
    fwrite(message, 1, msglen, msgfile);
    fclose(msgfile);
    return exit;
}

uint32_t pqstego_check_capacity(const uint8_t *stegofilename, pq_parameter_t *param) {
    struct pqstego_jpeg_error_mgr jerr;
    jvirt_barray_ptr *coef_array;
    JBLOCKARRAY coef_buffer;
    JDIMENSION i, ii, iii;
    struct jpeg_decompress_struct *dinfo;
    pq_data_t *pq_src_data = NULL;
    FILE *srcfile;
    uint8_t exit = EXIT_SUCCESS;
    uint32_t capacity = 0;

    pq_src_data = (pq_data_t*) malloc(sizeof (pq_data_t));
    dinfo = malloc(sizeof (struct jpeg_decompress_struct));
    dinfo->err = jpeg_std_error(&jerr.pub);

    if ((srcfile = fopen(stegofilename, "rb")) == NULL) {
        return EXIT_FAILURE;
    }
    jpeg_create_decompress(dinfo);
    jpeg_stdio_src(dinfo, srcfile);
    jpeg_read_header(dinfo, TRUE);

    coef_array = jpeg_read_coefficients(dinfo);
    pq_src_data->components = dinfo->num_components;
    pq_src_data->component = malloc(sizeof (pq_component_t) * dinfo->num_components);
    if (!pq_src_data->component) {
        return EXIT_FAILURE;
    }
    pq_src_data->size_x = dinfo->image_width;
    pq_src_data->size_y = dinfo->image_height;


    for (i = 0; i < dinfo->num_components; i++) {
        jpeg_component_info *compinfo = (dinfo->comp_info + i);
        JDIMENSION blocks = compinfo->width_in_blocks * compinfo->height_in_blocks;
        (pq_src_data->component + i)->quant = (pq_quant_t*) malloc(sizeof (pq_quant_t));
        if (!(pq_src_data->component + i)->quant) {
            return EXIT_FAILURE;
        }
        (pq_src_data->component + i)->block = (pq_block_t*) malloc(sizeof (pq_block_t) * blocks);
        if (!(pq_src_data->component + i)->block) {
            return EXIT_FAILURE;
        }
        (pq_src_data->component + i)->blocks = blocks;
        memcpy((pq_src_data->component + i)->quant->values, compinfo->quant_table->quantval, sizeof (pq_quant_t));
        for (ii = 0; ii < compinfo->height_in_blocks; ii++) {
            coef_buffer = (dinfo->mem->access_virt_barray)
                    ((j_common_ptr) dinfo, *(coef_array + i), ii, (JDIMENSION) 1, FALSE);
            JBLOCKROW block_row = *(coef_buffer);
            for (iii = 0; iii < compinfo->width_in_blocks; iii++) {
                memcpy(((pq_src_data->component + i)->block + (ii * compinfo->width_in_blocks) + iii)->values,
                        (block_row + iii), sizeof (pq_block_t));
            }

        }

    }
    fclose(srcfile);

    pq_check_capacity(pq_src_data, param, &capacity);

    return capacity;
}

int main(int argc, char **argv) {

    static const struct option long_options[] = {
        { "capacity", no_argument, NULL, 'c'},
        { "quality", required_argument, NULL, 'q'},
        { "embed", no_argument, NULL, 'e'},
        { "extract", no_argument, NULL, 'x'},
        { "password", required_argument, NULL, 'p'},
        { "help", no_argument, NULL, 'h'},
        { "version", no_argument, NULL, 'v'},
        { NULL, 0, NULL, 0}
    };
    uint8_t extract = 0;
    uint8_t embed = 0;
    uint8_t capacity = 0;
    uint8_t version = 0;
    uint8_t exit = EXIT_SUCCESS;
    pq_parameter_t *param = NULL;
    uint8_t *password = NULL;
    uint32_t pwlen = 0;
    uint32_t quality = 0;
    uint8_t *coverfile;
    uint8_t *messagefile;
    uint8_t *stegofile;
    int opt;
    int argind;


    program_name = argv[0];
    param = (pq_parameter_t*) malloc(sizeof (pq_parameter_t));
    if (!param) {
        return EXIT_FAILURE;
    }
    if (argc == 1) {
        usage(exit);
        return exit;
    }


    while ((opt = getopt_long(argc, argv, "cq:exp:hv", long_options, NULL)) != -1) {
        switch (opt) {
            case 'c':
                capacity = 1;
                break;
            case 'q':
                quality = (uint32_t) strtol(optarg, NULL, 10);
                break;
            case 'e':
                embed = 1;
                break;

            case 'x':
                extract = 1;
                break;
            case 'p':
                password = optarg;
                pwlen = strlen(optarg) + 1;
                break;

            case 'h':
                usage(EXIT_SUCCESS);
                break;
            case 'v':
                version = 1;
                break;
            default:
                usage(EXIT_FAILURE);
        }

    }
    if (password) {
        param->password = password;
        param->pwlen = pwlen;
    } else {
        param->password = "password";
        param->pwlen = 9;
    }
    param->header_size = 128;
    if (quality > 100) {
        quality = 0;
    }
    param->quality = quality;
    coverfile = "-";
    stegofile = "-";
    messagefile = "-";
    argind = optind;

    do {
        if (argind < argc) {

            if (embed) {
                stegofile = argv[argind];
            } else {
                if (capacity) {
                    coverfile = argv[argind];
                }
            }
            if (extract) {
                messagefile = argv[argind];
            }
        }
        if (argind < argc - 1) {
            if (embed) {
                coverfile = argv[argind];
            }
            if (extract) {
                stegofile = argv[argind];
            }
        }
        if ((argind < argc - 2) && (embed)) {
            messagefile = argv[argind];
        }
    } while (++argind < argc);

    if (version) {
        printf("%s\n\n", PROGRAM_NAME);
        printf("\
Copyright (C) 2008 Christian Kuka\n");
        printf("\
This is free software; see the source for copying conditions.  There is NO\n\
warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n");
        printf("\n");
    }

    if (capacity) {
        printf("Capacity of %s: %d\n", coverfile, pqstego_check_capacity(coverfile, param));
        exit = EXIT_SUCCESS;
    }
    if (embed) {
        exit = pqstego_embed(coverfile, stegofile, messagefile, param);
        if (exit != EXIT_SUCCESS) {
            printf("Error: (%d) %s\n", pq_errno, pq_strerror(pq_errno));
        }
    }
    if (extract) {
        exit = pqstego_extract(stegofile, messagefile, param);
        if (exit != EXIT_SUCCESS) {
            printf("Error: (%d) %s\n", pq_errno, pq_strerror(pq_errno));
        }
    }
    return exit;
}
