/*
This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
~*/

#include "substract_wave.h"


void running_info()
{
fprintf(stderr,\
"    RUNNNING COMMANDS\n\
ESC       exits.\n\
'['  ']'  1%% level up / down.\n\
'{'  '}'  10%% level up / down.\n\
','  '.'  1 sample early / late.\n\
'<'  '>'  10 samples early / late.\n\
");

} /* end function running info */


void usage()
{
fprintf(stderr,\
"Usage:\
substract_wave [-a] [-b offset] [-d] [-f mult_factor] [-k] -m main_filename -s subtract_filename [-r] -o output_filename.\n\
-a  play to /dev/dsp (not to output filename or stdout).\n\
-b  samples offset for substract channel.\n\
-d  debug_mode.\n\
-f  float multiplication factor, 0-100, default 100.\n\
-k  offset in both files from start in bytes.\n\
-m  main channel filename.\n\
-o  output filename (if none specified then stdout is used).\n\
-r  reverse, add channels instead of substracting.\n\
-s  channel to substract filename.\n\
 TYPE '?' during running to view commands.\n\
");

running_info();
} /* end function usage */


char *strsave(char *s)
/* save string s somewhere */
{
char *p;

p = malloc(strlen(s) + 1);
if(!p) return 0;
strcpy(p, s);

return p;
} /* end function strsave */


int main(int argc, char **argv)
{
int a, b, i, j, l, r;
double da, dm, ds, dout;
int cm, cs;
int error;
int le, re;
char *main_input_filename, *substract_input_filename, *output_filename;
wave_header *main_header, *substract_header, *out_header;
FILE *pfmain, *pfsubstract, *pfout;
int dsp_speed;
int sample_size;
int dsp_stereo;
int header_size;
unsigned long out_bytes;
unsigned char *main_buffer, *substract_buffer, *out_buffer;
unsigned char *ptr, *pmbptr, *psbptr, *pobptr, *pend;
int in_buffer_size, out_buffer_size;
int bytes_passed;
int keep_main_reference;
int adjust_flag;
struct pollfd pfd[1];
int total;
int length;
int audiofd;
char *device;
int tmp;
int abuf_size;  
char *audiobuf;
char *pabuf;
char ca;
fd_set readfs;
struct timeval timeout;
double substract_factor;
off_t start_offset;
long shift_samples;
long shifted_bytes;
long la;
long samples_offset;
int silence_inserted_flag;
long main_offset;
int have_reported_substract_file_eof_flag;
int substract_flag;

/*select con io 1 byte at the time*/
setbuf(stdout, NULL);
setbuf(stdin, NULL);


fprintf(stderr, "\nsubstract_wave-%s Copyright Jan Panteltje 2004.\n\n", VERSION);


/* defaults */
debug_flag = 0;
substract_flag = 1;
main_input_filename = NULL;
substract_input_filename = NULL;
output_filename = NULL;
keep_main_reference = 1;
adjust_flag = 0;
device = strsave("/dev/dsp");
if(! device)
	{
	fprintf(stderr, "could not allocate memory for device, aborting.\n");

	exit(1);
	}

substract_factor = 100.0;
start_offset = 0;
shift_samples = 0;
samples_offset = 0;
silence_inserted_flag = 0;
audiofd = -1; /* for gcc -Wall */
audiobuf = 0; /* for gcc -Wall */
main_offset = 0;
have_reported_substract_file_eof_flag = 0;
/* end defaults */

while(1)
	{
	a = getopt(argc, argv, "ab:df:k:m:o:rs:");
	if(a == -1) break;

	switch(a)
		{
        case -1:
        	break;
		case 'a':
			adjust_flag = 1;
			break;
		case 'b':
			samples_offset = atol(optarg);
			break;
		case 'd':
			debug_flag = 1;
			break;
		case 'f':
			substract_factor = atof(optarg);
			break;
		case 'k':
			start_offset = atol(optarg);

			if(start_offset < 0)
				{
				fprintf(stderr,\
				"substract_wave: only positive values for -k start_offset allowed (now %ld), aborting.\n",\
				(long) start_offset);

				exit(1);
				}
			break;
		case 'm':
			main_input_filename = strsave(optarg);
			if(! main_input_filename)
				{
				fprintf(stderr,\
				"substract_wave: could not allocate space for main_input_filename.\n");

				exit(1);
				}
			break;
		case 'o':
			output_filename = strsave(optarg);
			if(! output_filename)
				{
				fprintf(stderr,\
		"substract_wave: could not allocate space for output_filename.\n");

				exit(1);
				}
			break;
		case 'r':
			substract_flag = 0;
			break;
		case 's':
			substract_input_filename = strsave(optarg);
			if(! substract_input_filename)
				{
				fprintf(stderr,\
				"substract_wave: could not allocate space for substract_input_filename.\n");

				exit(1);
				}
			break;
		default:
			fprintf(stderr, "substract_wave: unknown option.\n");
			usage();
			exit(1);
			break;
		}/* end switch a */
	}/* end while getopt() */

error = 0;
if(! main_input_filename)
	{
	fprintf(stderr,\
	"substract_wave: No main filename specified.\n");
	error = 1;
	}

if(! substract_input_filename)
	{
	fprintf(stderr,\
	"substract_wave: No substract filename specified.\n");
	error = 1;
	}

if(! output_filename && !adjust_flag)
	{
	fprintf(stderr,\
	"substract_wave: No output filename specified, using stdout\n");
//	error = 1;
	}

if(error)
	{
	usage();
	exit(1);
	}

/* allocate space for main input wave header */
main_header = malloc( sizeof(wave_header) );
if(! main_header)
	{
	fprintf(stderr,\
	"substract_wave: could not allocate space for main_header.\n");

	exit(1);
	}

/* allocate space for substract input wave header */
substract_header = malloc( sizeof(wave_header) );
if(! substract_header)
	{
	fprintf(stderr,\
	"substract_wave: could not allocate space for substract_header.\n");

	exit(1);
	}

/* allocate space for output wave header */
out_header = malloc( sizeof(wave_header) );
if(! out_header)
	{
	fprintf(stderr,\
	"substract_wave: could not allocate space for out_header.\n");

	exit(1);
	}

pfmain = fopen(main_input_filename, "r");
if(! pfmain)
	{
	fprintf(stderr,\
	"substract_wave: could not open main input file %s for read, aborting.\n", 
	main_input_filename);

	exit(1);
	}

pfsubstract = fopen(substract_input_filename, "r");
if(! pfsubstract)
	{
	fprintf(stderr,\
	"substract_wave: could not open substract input file %s for read, aborting.\n", 
	substract_input_filename);

	exit(1);
	}

if(! output_filename)
	{
	pfout = stdout;
	}
else
	{
	pfout = fopen(output_filename, "w");
	if(! pfout)
		{
		fprintf(stderr,\
		"substract_wave: could not open output file %s for read, aborting.\n", 
		output_filename);

		exit(1);
		}
	}


header_size = sizeof(wave_header);

//fprintf(stderr, "substract_wave: header_size=%d\n", header_size);
//fprintf(stderr, "substract_wave: sizof(char)=%d\n", sizeof(char) );

/* read main input file header */
a = fread(main_header, sizeof(char), header_size, pfmain);

if(a != header_size)
	{
	fprintf(stderr,\
	"substract_wave: could only read %d bytes of main input file header, aborting.\n", a);
	exit(1);
	}

/* set up audio if in adjust mode */
if(adjust_flag)
	{
	dsp_speed = main_header -> sample_fq;   
	dsp_stereo = main_header -> modus - 1;
	sample_size = (main_header -> byte_p_spl * 8) / main_header -> modus;

	audiofd = open (device, O_WRONLY, 0);
	if(audiofd < 0)
		{
		fprintf(stderr,\
		" could not open device %s for read, aborted\n",\
		device);

		return 0;
		}

	tmp = sample_size;
	ioctl(audiofd, SNDCTL_DSP_SAMPLESIZE, &tmp);
	if (tmp != sample_size)
		{
		close(audiofd);

		fprintf(stderr,\
		"play_file(): could not set sample_size %d for %s aborted\n",\
		sample_size, device);

		return 0;
		}

	if (ioctl (audiofd, SNDCTL_DSP_STEREO, &dsp_stereo) == -1)
		{
		close(audiofd);

		fprintf(stderr,\
		"play_file(): Could not set stereo / mono mode for %s aborted\n",\
		device);

		return 0;
		}

	tmp = dsp_speed;
	if (ioctl (audiofd, SNDCTL_DSP_SPEED, &tmp) == -1)
		{
		close(audiofd);

		fprintf(stderr,\
		"play_file(): could not set sample speed %d for %s aborted\n",\
		dsp_speed, device);

		return 0;
		}

	ioctl (audiofd, SNDCTL_DSP_GETBLKSIZE, &abuf_size);
	if (abuf_size < 1)
		{
		close(audiofd);

		fprintf(stderr,\
		"play_file(): could not get blocksize for %s aborted\n",\
		device);

		return 0;
		}

	if ((audiobuf = malloc (abuf_size)) == NULL)
		{
		close(audiofd);

		fprintf(stderr,\
		"play_file(): could not allocate buffer, aborted\n");

		return 0;
		}
	} /* end if adjust_flag */

/* read substract input file header */
a = fread(substract_header, sizeof(char), header_size, pfsubstract);

if(a != header_size)
	{
	fprintf(stderr,\
	"substract_wave: could only read %d bytes of substract input file header, aborting.\n", a);
	exit(1);
	}

/* check if main channel mono */
if(main_header -> modus != 1)
	{
	fprintf(stderr,\
	"substract_wave: main input channel is not mono, aborting.\n");

	exit(1);
	}

/* check if substract channel mono */
if(substract_header -> modus != 1)
	{
	fprintf(stderr,\
	"substract_wave: substract input channel is not mono, aborting.\n");

	exit(1);
	}

/* check if sample rates are the same */
if(main_header -> sample_fq != substract_header -> sample_fq)
	{
	fprintf(stderr,\
	"substract_wave: input sample rates differ! (l=%u r=%u) aborting.\n",\
	main_header -> sample_fq, substract_header -> sample_fq);

	exit(1);
	}

/* check if bytes per sample is 2 (16 bit mono) */
if(main_header -> byte_p_spl != 2)
	{
	fprintf(stderr,\
	"substract_wave: main channel not 2 bytes per sample.\n");

	exit(1);
	}

if(substract_header -> byte_p_spl != 2)
	{
	fprintf(stderr,\
	"substract_wave: substract channel not 2 bytes per sample.\n");

	exit(1);
	}

/* check if bits per sample is 16 */
if(main_header -> bit_p_spl != 16)
	{
	fprintf(stderr,\
	"substract_wave: main channel not 16 bits per sample.\n");

	exit(1);
	}

if(substract_header -> bit_p_spl != 16)
	{
	fprintf(stderr,\
	"substract_wave: substract channel not 16 bits per sample.\n");

	exit(1);
	}

/* check if bytes per sample the same */
if(main_header -> byte_p_spl != substract_header -> byte_p_spl)
	{
	fprintf(stderr,\
"substract_wave: input bytes per sample differ! (l=%d r=%d) aborting.\n",\
	main_header -> byte_p_spl, substract_header -> byte_p_spl);

	exit(1);
	}

/* check if bits per sample the same */
if(main_header -> bit_p_spl != substract_header -> bit_p_spl)
	{
	fprintf(stderr,\
"substract_wave: input bytes per sample differ! (l=%d r=%d) aborting.\n",\
	main_header -> bit_p_spl, substract_header -> bit_p_spl);

	exit(1);
	}

/* allocate space for buffers */
/* this must be a multiple of 4 */
#define IN_BUFFER_SIZE 4*32768

in_buffer_size = IN_BUFFER_SIZE;

out_buffer_size = in_buffer_size;

main_buffer = malloc(in_buffer_size);
if(! main_buffer)
	{
	fprintf(stderr,\
	"substract_wave: could allocate main_buffer, aborting.\n");

	exit(1);
	}

substract_buffer = malloc(in_buffer_size);
if(! substract_buffer)
	{
	fprintf(stderr,\
	"substract_wave: could allocate substract_buffer, aborting.\n");

	exit(1);
	}

out_buffer = malloc(out_buffer_size);
if(! out_buffer)
	{
	fprintf(stderr,\
	"substract_wave: could allocate out_buffer, aborting.\n");

	exit(1);
	}

/*
copy main wave header parameters to out wave header, but change 4 things:
file length is updated,
modus = 2, stereo,
bytes_p_spl *= 2,
and data_length is the longest of the two input files.
*/
ptr = (char *)out_header;
for(i = 0; i < header_size; i++)
	{
	*ptr = 0;
	ptr++;
	}
strcpy(out_header -> main_chunk, "RIFF");
strcpy(out_header -> chunk_type, "WAVE"); 
strcpy(out_header -> sub_chunk, "fmt ");
out_header -> length_chunk = 16; //always 16
out_header -> format = 1; //PCM

out_header -> modus = 1; // mono
out_header -> byte_p_sec = main_header -> byte_p_sec;
out_header -> byte_p_spl = main_header -> byte_p_spl;

out_header -> sample_fq = main_header -> sample_fq;
out_header -> bit_p_spl = 16; // main_header -> bit_p_spl;
strcpy(out_header -> data_chunk, "data");

main_offset = start_offset + (samples_offset * 2);

//fprintf(stderr, "main_offset=%ld\n", main_offset);

//fprintf(stderr, "main_header-> data_length=%lu\n", main_header -> data_length);

/* output file has always the same length as - or smaller then the input file */
out_header -> data_length = main_header -> data_length;

out_header -> data_length -= start_offset;

/*
fprintf(stderr, "out_header-> data_length=%lu diff=%ld\n",\
out_header -> data_length, out_header -> data_length - main_header-> data_length);
*/

out_header -> length = out_header -> data_length + 36; // FIXME ?

/* seek start_offset bytes in main channel from header end */
a = fseek(pfmain, start_offset, SEEK_CUR);	
if(a)
	{
	fprintf(stderr,\
	"substract_wave: could not set main channel offset %lu, aborting.\n",\
	(unsigned long) start_offset);

	exit(1);
	}

if(main_offset < 0)
	{
	/* handled later by adding empty bytes in buffer */
	}
else
	{
	/* seek start_offset bytes in substract channel from header end */
	a = fseek(pfsubstract, start_offset + (samples_offset * 2), SEEK_CUR);	
	if(a)
		{
		fprintf(stderr, "start_offset=%ld samples_offset=%ld\n", (long) start_offset, samples_offset);

		fprintf(stderr,\
		"substract_wave: could not set substract channel offset bytes %ld, aborting.\n",\
		(long) (start_offset + (samples_offset * 2) ) );
		
		exit(1);
		}
	}

if(! adjust_flag)
	{
	/* write out header */
	a = fwrite(out_header, sizeof(char), header_size, pfout);
	if(a != header_size)
		{
		fprintf(stderr,\
		"substract_wave: could only write %d of %d bytes of out header, aborting.\n",\
		a, header_size);   

		exit(1); 
		}
	} /* end if ! adjust_flag */

pfd[0].fd = fileno(stdin);
pfd[0].events = POLLOUT;

bytes_passed = 0;
out_bytes = 0;
le = 0;
re = 0;
r = 0;
l = 0; /* -Wall */
dout = 0; /* -Wall */
while(1)
	{
	if(adjust_flag)
		{
//		substract_factor = 0.0;

		/* read a user key */
		system("stty raw");
	
		FD_ZERO(&readfs);
		FD_SET(fileno(stdin), &readfs);
		timeout.tv_sec = 0;
		timeout.tv_usec = 0;
		b = select(FD_SETSIZE, &readfs, NULL, NULL, &timeout);
		if(b == -1)
			{
			fprintf(stderr, "select returned -1, aborting\n");

			exit(1);
			}

		if( FD_ISSET(fileno(stdin), &readfs) )
			{
			a = read( fileno(stdin), &ca, 1);

			FD_CLR(fileno(stdin), &readfs);

			system("stty sane");

			if(ca == 27)
				{
				fprintf(stderr, "user abort\n");

				exit(1);
				}
			else if(
			(ca == '[') || (ca == ']') || \
			(ca == '{') || (ca == '}') || \
			(ca == ',') || (ca == '.') || \
			(ca == '<') || (ca == '>')
			)
				{
				if(      (ca == '[') && (substract_factor -1.0 >= 0.0) ) substract_factor -= 1.0;
				else if( (ca == ']') && (substract_factor +1.0 <= 100.0) ) substract_factor += 1.0;
				else if( (ca == '{') && (substract_factor - 10.0 >= 0.0) ) substract_factor -= 10.0;
				else if( (ca == '}') && (substract_factor + 10.0 <= 100.0) ) substract_factor += 10.0;
				else if(ca == ',') shift_samples = -1;
				else if(ca == '.') shift_samples = 1;
				else if(ca == '<') shift_samples = -10;
				else if(ca == '>') shift_samples = 10;

//				fprintf(stderr, "ca=%d(%c)\n", ca, ca);
	
				fprintf(stderr, "substract_factor=%.2f\n", substract_factor);

				la = ftell(pfsubstract);

				/* seek in substract channel from position */
				a = fseek(pfsubstract, shift_samples * 2, SEEK_CUR);	
				if( (a) || ferror(pfsubstract) )
					{
					fprintf(stderr,\
					"substract_wave: could not set substract channel shift_samples %ld, aborting.\n",\
					shift_samples);
		
					exit(1);
					}

				shifted_bytes = ftell(pfsubstract) - la;				

				samples_offset += shifted_bytes / 2;

//				fprintf(stderr, "shifted_samples=%ld\n", shifted_bytes / 2);

				fprintf(stderr, "samples_offset=%ld\n", samples_offset);

				shift_samples = 0;

				} /* end if manual substract factor change */
			else
				{
				running_info();
				}
			} /* end if key */
		else /* a = 0 */
			{
			system("stty sane");
			/* no data available */
			}

		} /* end if adjust_flag */

	if(! le)
		{
		l = fread(main_buffer, sizeof(char), in_buffer_size, pfmain);
		if(l != in_buffer_size)
			{
			if(feof(pfmain) )
				{
//				fclose(pfmain);
				le = 1;
				}
			else
				{
				perror("main read: ");
				fprintf(stderr,\
				"substract_wave: error reading main file, aborting.\n"); 

				exit(1);
				}
			} /* end if l != requested */
		} /* end if ! le */

	if(! re)
		{
		if( (!silence_inserted_flag) && (main_offset < 0) )
			{
			for(i = 0; i < -main_offset; i++) substract_buffer[i] = 0;

			r = fread(substract_buffer, sizeof(char), in_buffer_size + main_offset, pfsubstract);

			r += -main_offset;

			silence_inserted_flag = 1;
			}
		else
			{
			r = fread(substract_buffer, sizeof(char), in_buffer_size, pfsubstract);
			}

		if(r < in_buffer_size)
			{
			if(feof(pfsubstract) )
				{
				fclose(pfsubstract);
				re = 1;

				/* clear rest buffer */
				for(i = r; i < in_buffer_size; i++) substract_buffer[i] = 0;

				r = in_buffer_size;
				}
			else
				{
				perror("substract read: ");
				fprintf(stderr,\
				"substract_wave: error reading substract file, aborting.\n"); 
				exit(1);
				}
			} /* end if r != requested */
		} /* end if ! re */
	else /* use silence */
		{
	
		if(! have_reported_substract_file_eof_flag)
			{
			fprintf(stderr, "Early EOF in substract file, assuming silence.\n");
			have_reported_substract_file_eof_flag = 1;
			}

		for(i = 0; i < in_buffer_size; i++) substract_buffer[i] = 0;
		r = in_buffer_size;
		}

//fprintf(stderr, "l=%d r=%d\n", l, r);

	pmbptr = main_buffer;
	psbptr = substract_buffer;
	pobptr = out_buffer;

	if(feof(pfmain) )
		{
//fprintf(stderr, "EOF in main l=%d\n", l);

		pend = main_buffer + l;

		out_buffer_size = l;
		}
	else
		{
		pend = main_buffer + in_buffer_size;
		}

	/* one sample at the time */
	/* low byte first !!! */
	i = 0;
	da = substract_factor / 100.0;
	while(1)
		{
		if(i < r)
			{
			unsigned long p = 1 << 15;
			unsigned long l, ll;

			/* get main buffer as double */
			cm = *pmbptr + (256 * *(pmbptr + 1) );
//fprintf(stderr, "cm=%d  ", cm);
			l = cm;
//fprintf(stderr, "old l=%lu ", l);
			if(l & p) /* sign bit */
				{
				l = p - (l ^ p);

				dm = -(double) l / (double) p; 	
				}
			else /* positive */
				{
				dm = l / ( (double) p );
				}
//fprintf(stderr, "dm=%.10f  ", dx);

			/* get substract_buffer as double */
			cs = *psbptr + (256 * *(psbptr + 1) );
//fprintf(stderr, "cs=%d  ", cs);
			l = cs;
//fprintf(stderr, "old l=%lu ", l);
			if(l & p) /* sign bit */
				{
				l = p - (l ^ p);

				ds = -(double) l / (double) p; 	
				}
			else /* positive */
				{
				ds = l / ( (double) p );
				}
//fprintf(stderr, "ds=%.10f  ", dx);

			/* multiply substract channel by substract_factor */
			ds *= da;

			/* substract from main buffer */
			if(substract_flag)
				{
				dout = dm - ds;
				}
			else
				{
				dout = dm + ds;
				}

			/* double data to output buffer */
			ll = floor(dout * p);
			if( dout < 0) ll += 2 * p;
//fprintf(stderr, "new ll=%lu\n", ll);

			*pobptr = ll % 256;
			*(pobptr + 1) = ll / 256;

			}
		else
			{
			for(j = 0; j < main_header -> byte_p_spl; j++) out_buffer[i + j] = 0;
			}

		if(pmbptr == pend) break;

		pmbptr += main_header -> byte_p_spl;	
		psbptr += main_header -> byte_p_spl;
		pobptr += main_header -> byte_p_spl;

		i += main_header -> byte_p_spl;
		} /* end for i */

//fprintf(stderr, "abuf_size=%d\n", abuf_size);

	/* test if to dsp device */
	if(adjust_flag)
		{
//fprintf(stderr, "requested=%d\n", out_buffer_size);
		total = out_buffer_size;
		ptr = out_buffer;
		while(1)
			{
			memcpy(audiobuf, ptr, abuf_size);

			pabuf = audiobuf;
			a = abuf_size;
			while(1)
				{
				if(total <= 0) break;
				errno = 0;
				length = write (audiofd, pabuf, a);

				if(length < 0)
					{
					if(errno == EAGAIN)
						{
						usleep(10000);					

						continue;
						}
					else if(errno == EINTR)
						{
						usleep(10000);

						continue;
						}	
					else
						{
						perror("write(): ");
						fprintf(stderr, "write failed to %s, aborted\n",  device);
	
						close(audiofd);

						exit(1);
						}
					} /* end if read returned < 0 */
				else if(length == 0) break;
				else
					{
					a -= length;
					if(a == 0) break;

					pabuf += length;
					}
				} /* end while try write  */

			
			total -= abuf_size;
			if(total == 0) break;
//fprintf(stderr, "main to send=%d\n", total);

			ptr += abuf_size;
			} /* end while output out_buffer to audio device */ 
		
		} /* end if adjust_flag (to dsp device) */
	else /* to file or stdout */
		{
		a = fwrite(out_buffer, sizeof(char), out_buffer_size, pfout);
		if(a != out_buffer_size)
			{
			fprintf(stderr,\
			"substract_wave: could only write %d of %d bytes, aborting.\n",\
			a, out_buffer_size);			

			exit(1);
			}
		} /* end if to file or stdout */

	out_bytes += out_buffer_size;

	/* print progress after each MB */
	bytes_passed += out_buffer_size;
	if(bytes_passed > 1000000)
		{
		fprintf(stderr,\
		"substract_wave:                                                 %lu bytes (%lu MB) written\r",\
		out_bytes, out_bytes / 1000000);

		bytes_passed = 0;
		}

	if(feof(pfmain) ) break;
	} /* end while */

fclose(pfmain);

if(adjust_flag)
	{
	close(audiofd);
	}

fprintf(stderr, "\n%lu bytes written.\n", out_bytes + 44);

fprintf(stderr, "Ready.\n");

exit(0);
} /* end function main */


