/*
 * test_moves_order.c --- Order the moves of the current position.
 *
 * Copyright (c) 2001, 2002, 2003, 2004 by Pascal Wassong All Rights Reserved.
 *
 * Time-stamp: <2004-07-01 13:19:21 pascal>
 *
 * This file is part of Natch.
 *
 * Natch is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Natch is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include	"test_moves_order.h"
#include	"main.h"
#include	"distance.h"
#include	"pcpj.h"
#include	"twin_squares.h"
#include	"piece.h"

#include	<stdlib.h>	/* for exit(...) */
#include	<assert.h>

/*--------------------------------------------------------------------------*/

/* #define	DEBUG_TEST_MOVE_ORDER */
#include	"pcpjtool.h"

/*--------------------------------------------------------------------------*/

typedef enum
{
    e_Initial_Square,
    e_Destination_Square,
    e_Captured_Without_Moving
} constraint_type_t ;

/*--------------------------------------------------------------------------*/

static bool_t add_capture_constraints(
    struct moves_order_t*	moves_order,
    const exploration_t*	exploration );

static bool_t add_square_constraints(
    struct moves_order_t*	moves_order,
    const exploration_t*	exploration );

static bool_t add_castling_constraints(
    const piece_t*		king,
    const piece_t*		rook,
    struct moves_order_t*	moves_order ,
    const exploration_t*	exploration );

static bool_t add_straigth_line_constraints(
    struct moves_order_t*	moves_order ,
    const exploration_t*	exploration );

/*--------------------------------------------------------------------------*/

static bool_t add_one_order( struct moves_order_t*	moves_order,
			     piece_index_t		before_piece_index,
			     unsigned int		before_move_index,
			     move_type_t		before_type,
			     piece_index_t		after_piece_index,
			     unsigned int		after_move_index,
			     move_type_t		after_type,
			     const exploration_t*	exploration,
			     const char*		description );

static bool_t add_constraint(
    struct moves_order_t*	moves_order,
    const exploration_t*	exploration,
    square_t			square,
    piece_index_t		piece_index,
    constraint_type_t		constraint_type );

static void print_one_move( FILE*			fd,
			    piece_index_t		piece_index,
			    unsigned int		move_index,
			    move_type_t			type,
			    const exploration_t*	exploration );

static void add_square_to_list( square_t	square,
				square_t*	squares,
				unsigned int*	nb_squares );

/*--------------------------------------------------------------------------*/

#define	MAX_DESTINATION_PER_TYPE	128

static square_t		King_Squares  [ MAX_DESTINATION_PER_TYPE ];
static unsigned int	Nb_King_Squares ;
static square_t		Pawn_Squares  [ MAX_DESTINATION_PER_TYPE ];
static unsigned int	Nb_Pawn_Squares ;
static square_t		Queen_Squares [ MAX_DESTINATION_PER_TYPE ];
static unsigned int	Nb_Queen_Squares ;
static square_t		Rook_Squares  [ MAX_DESTINATION_PER_TYPE ];
static unsigned int	Nb_Rook_Squares ;
static square_t		Bishop_Squares[ MAX_DESTINATION_PER_TYPE ];
static unsigned int	Nb_Bishop_Squares ;
static square_t		Knight_Squares[ MAX_DESTINATION_PER_TYPE ];
static unsigned int 	Nb_Knight_Squares ;

/*--------------------------------------------------------------------------*/

bool_t
test_moves_order( const exploration_t* exploration )
{
    struct moves_order_t*	moves_order = create_moves_order( exploration );
    piece_index_t		i ;
    square_t			square ;
    bool_t			ok 	    = TRUE ;
    distance_t			distance_table[ 6 * SIZE_PIECE ];
    const piece_t*		wK 	    = NULL ;
    const piece_t*		bK 	    = NULL ;
    const piece_t*		wR 	    = NULL ;
    const piece_t*		bR 	    = NULL ;

    if ( !add_capture_constraints( moves_order, exploration )
	 || !add_square_constraints( moves_order, exploration ) )
    {
	return FALSE ;
    }

    /* initialisations for distance computation */
    create_twin_squares( &exploration->pieces[ INDEX_ROI_BLANC ],
			 exploration->nb_moving_pieces );
    Nb_King_Squares   = 0 ;
    Nb_Pawn_Squares   = 0 ;
    Nb_Queen_Squares  = 0 ;
    Nb_Rook_Squares   =	0 ;
    Nb_Bishop_Squares =	0 ;
    Nb_Knight_Squares =	0 ;
    for ( i = INDEX_ROI_BLANC ; i <= INDEX_PION_NOIR_H ; i++ )
    {
	const piece_t*	piece = &exploration->pieces[ i ];
	unsigned int	j ;
	piece_type_t	piece_type = piece->typePiece ;

	if ( piece->castling != NO_CASTLING )
	{
	    if ( piece_type == ROI )
	    {
		if ( piece->camp == BLANC )
		{
		    wK = piece ;
		}
		else
		{
		    bK = piece ;
		}
	    }
	    else
	    {
		if ( piece->camp == BLANC )
		{
		    wR = piece ;
		}
		else
		{
		    bR = piece ;
		}
	    }
	}

	for ( j = 0 ; j < piece->nbDestination ; j++ )
	{
	    square_t	square ;
	    if ( j == 0 )
	    {
		if ( piece->typePiece == ROI && piece->castling != NO_CASTLING )
		{
		    continue ;
		}

		square = piece->caseInitiale ;
	    }
	    else
	    {
		square = piece->destination[ j - 1 ];
	    }

	    if ( square == CASE_ARRIVEE_QUELCONQUE )
	    {
		continue ;
	    }

	    if ( square == piece->casePromotion )
	    {
		piece_type = piece->typePromotion ;
	    }

	    switch( piece_type )
	    {
	    case ROI:
		add_square_to_list( square, King_Squares, &Nb_King_Squares  );
		break ;
	    case PION:
		add_square_to_list( square, Pawn_Squares, &Nb_Pawn_Squares  );
		break ;
	    case DAME:
		add_square_to_list( square, Queen_Squares,&Nb_Queen_Squares );
		break ;
	    case TOUR:
		add_square_to_list( square, Rook_Squares, &Nb_Rook_Squares  );
		break ;
	    case FOU:
		add_square_to_list( square,
				    Bishop_Squares,
				    &Nb_Bishop_Squares );
		break ;
	    case CAVALIER:
		add_square_to_list( square,
				    Knight_Squares,
				    &Nb_Knight_Squares );
		break ;
	    default:
		assert( FALSE );
		exit( EXIT_FAILURE );
	    }
	}
    }

    if ( wK != NULL )
    {
	ok = add_castling_constraints( wK, wR, moves_order, exploration );
	if ( !ok )
	{
	    return FALSE ;
	}
    }
    if ( bK != NULL )
    {
	ok = add_castling_constraints( bK, bR, moves_order, exploration );
	if ( !ok )
	{
	    return FALSE ;
	}
    }

    for ( square = a1 ; square <= h8 ; square++ )
    {
	bool_t	something_happens = FALSE ;

	if ( CasesInterdites[ square ] )
	{
	    continue ;
	}

	for ( i = INDEX_ROI_BLANC ; i <= INDEX_PION_NOIR_H ; i++ )
	{
	    const piece_t*	piece = &exploration->pieces[ i ];

	    if ( piece->typePiece == TOUR && piece->castling != NO_CASTLING )
	    {
		continue ;
	    }

	    /* Look if pieces on their original square block other pieces.  */
	    if ( square == piece->caseInitiale && piece->distance[ 0 ] > 0 )
	    {
		something_happens = TRUE ;
		break ;
	    }

	    /* Piece captured without having moved may block other pieces.  */
	    if ( square == piece->caseInitiale
		 && piece->pieceCapturante != PIECE_PAS_CAPTUREE
		 && piece->distance[ 0 ] == 0 )
	    {
		something_happens = TRUE ;
		break ;
	    }

	    /* Look if piece that have reached their destination square block
	     * other pieces.  */
	    if ( square == piece->destination[ piece->nbDestination - 1 ]
		 && piece->distance[ 0 ] > 0
		 && piece->pieceCapturante == PIECE_PAS_CAPTUREE
		 && ( ( piece->camp == BLANC && NbCoupsBlancsRestants < 2 )
		      || ( piece->camp == NOIR && NbCoupsNoirsRestants < 2 )
		      || ( piece->pieceCapturee[ piece->nbDestination - 1 ]
			   == PAS_DE_CAPTURE
			   && !( piece->castling != NO_CASTLING	/* No OO/OOO */
				 /* && piece->typePiece == ROI : already done */
				 && piece->nbDestination == 1 )
			   /* and not just promoted */
			   && !( square == piece->casePromotion
				 && ( promotion_index( piece )
				      == piece->nbDestination - 1 ) ) ) ) )
	    {
		something_happens = TRUE ;
		break ;
	    }
	}

	if ( !something_happens )
	{
	    continue ;
	}

	CasesInterdites[ square ] = TRUE;

	distance_create_table_for_squares(
	    distance_table,
	    King_Squares,
	    Nb_King_Squares,
	    Pawn_Squares,
	    Nb_Pawn_Squares,
	    Queen_Squares,
	    Nb_Queen_Squares,
	    Rook_Squares,
	    Nb_Rook_Squares,
	    Bishop_Squares,
	    Nb_Bishop_Squares,
	    Knight_Squares,
	    Nb_Knight_Squares );

	distance_push( distance_table );

	for ( i = INDEX_ROI_BLANC ; i <= INDEX_PION_NOIR_H ; i++ )
	{
	    const piece_t*	piece = &exploration->pieces[ i ];

	    if ( piece->typePiece == TOUR && piece->castling != NO_CASTLING )
	    {
		continue ;
	    }

	    /* Look if pieces on their original square block other pieces. */
	    if ( square == piece->caseInitiale && piece->distance[ 0 ] > 0 )
	    {
		ok = add_constraint( moves_order,
				     exploration,
				     square,
				     i,
				     e_Initial_Square );
		if ( !ok )
		{
		    break ;
		}
	    }

	    /* Piece captured without having moved may block other pieces. */
	    if ( square == piece->caseInitiale
		 && piece->pieceCapturante != PIECE_PAS_CAPTUREE
		 && piece->distance[ 0 ] == 0 )
	    {
		ok = add_constraint( moves_order,
				     exploration,
				     square,
				     i,
				     e_Captured_Without_Moving );
		if ( !ok )
		{
		    break ;
		}
	    }

	    /* Look if piece that have reached their destination square block
	     * other pieces.  */
	    if ( square == piece->destination[ piece->nbDestination - 1 ]
		 && piece->distance[ 0 ] > 0
		 && piece->pieceCapturante == PIECE_PAS_CAPTUREE
		 && ( ( piece->camp == BLANC && NbCoupsBlancsRestants < 2 )
		      || ( piece->camp == NOIR && NbCoupsNoirsRestants < 2 )
		      || ( piece->pieceCapturee[ piece->nbDestination - 1 ]
			   == PAS_DE_CAPTURE
			   && !( piece->castling != NO_CASTLING	/* No OO/OOO */
				 /* && piece->typePiece == ROI : already done */
				 && piece->nbDestination == 1 )
			   /* and not just promoted */
			   && !( square == piece->casePromotion
				 && ( promotion_index( piece )
				      == piece->nbDestination - 1 ) ) ) ) )
	    {
		ok = add_constraint( moves_order,
				     exploration,
				     square,
				     i,
				     e_Destination_Square );
		if ( !ok )
		{
		    break ;
		}
	    }
	}

	distance_pop();
	CasesInterdites[ square ] = FALSE ;

	if ( !ok )
	{
	    break ;
	}
    }

/*     if ( ok ) */
/*     { */
/* 	ok = add_straigth_line_constraints( moves_order, exploration ); */
/*     } */

    if ( ok && MainPrintPositionLevel > 1 )
    {
	print_moves_order( MainFD, moves_order, exploration );
    }

    delete_moves_order( moves_order );

    return ok ;
}

/*--------------------------------------------------------------------------*/

static bool_t add_capture_constraints(
    struct moves_order_t*	moves_order,
    const exploration_t*	exploration )
{
    piece_index_t	i ;

    /* Captured pieces must reach there destination before the capture. */
    for ( i = INDEX_ROI_BLANC ; i <= INDEX_PION_NOIR_H ; i++ )
    {
	const piece_t*	piece = &exploration->pieces[ i ];
	unsigned int	j ;

	for ( j = 0 ; j < piece->nbDestination ; j++ )
	{
	    piece_index_t index = piece->pieceCapturee[ j ];
	    if ( index != PAS_DE_CAPTURE )
	    {
		const piece_t* captured_piece = &exploration->pieces[ index ];
		if ( captured_piece->distance[ 0 ] > 0 )
		{
		    /* Captured piece has moved */
		    if ( !add_one_order(
			     moves_order,
			     index,
			     captured_piece->nbDestination - 1,
			     e_Last_Move,
			     i,
			     j,
			     e_Last_Move,
			     exploration,
			     "captured piece must reach destination" ) )
		    {
			return FALSE ;
		    }
		}
	    }
	}
    }
    return TRUE ;
}

/*--------------------------------------------------------------------------*/

static bool_t add_square_constraints(
    struct moves_order_t*	moves_order,
    const exploration_t*	exploration )
{
    square_t		square ;
    const piece_t*	wK = &exploration->pieces[ exploration->board[ e1 ] ];
    const piece_t*	bK = &exploration->pieces[ exploration->board[ e8 ] ];

    for ( square = a1 ; square <= h8 ; square++ )
    {
	const piece_t*	orig = NULL ;
	const piece_t*	dest = NULL ;
	piece_index_t	i ;

	if ( square == h1 && wK->castling == KING_SIDE )
	{
	    orig = wK ;
	}
	else if ( square == a1 && wK->castling == QUEEN_SIDE )
	{
	    orig = wK ;
	}
	else if ( square == h8 && bK->castling == KING_SIDE )
	{
	    orig = bK ;
	}
	else if ( square == a8 && bK->castling == QUEEN_SIDE )
	{
	    orig = bK ;
	}
	else if ( exploration->board[ square ] != INDEX_CASE_VIDE )
	{
	    orig = &exploration->pieces[ exploration->board[ square ] ];
	    if ( orig->distance[ 0 ] == 0 )
	    {
		orig = NULL ;
	    }
	}

	for ( i = INDEX_ROI_BLANC ;
	      dest == NULL && i <= INDEX_PION_NOIR_H ;
	      i++ )
	{
	    const piece_t* piece = &exploration->pieces[ i ];
	    if ( piece->destination[ piece->nbDestination - 1 ] == square
		 && piece->distance[ 0 ] != 0
		 && piece->pieceCapturante == PIECE_PAS_CAPTUREE

		 /* If the piece made a capture (or promoted) on the last move
		  * and may swithback, then we don't make any assumption on the
		  * move order.  See tests `promotee_must_switchback2.test' for
		  * capture, `promotee_must_switchback3.test' for promotion, and
		  * also `promotee_must_switchback4.test'.
		  */
		 && !( ( ( piece->pieceCapturee[ piece->nbDestination - 1 ]
			   != PAS_DE_CAPTURE )
			 || ( square == piece->casePromotion
			      && ( promotion_index( piece )
				   == piece->nbDestination - 1 ) ) )
		       && ( ( piece->camp == BLANC
			      && NbCoupsBlancsRestants > 1 )
			    || ( piece->camp == NOIR
				 && NbCoupsNoirsRestants > 1 ) ) ) )
	    {
		dest = piece ;
	    }
	}

	if ( orig == NULL && dest == NULL )
	{
	    continue ;
	}

	for ( i = INDEX_ROI_BLANC ; i <= INDEX_PION_NOIR_H ; i++ )
	{
	    const piece_t*	piece = &exploration->pieces[ i ];
	    int			move_index ;

	    if ( piece->distance[ 0 ] == 0 )
	    {
		continue ;
	    }

	    if ( orig != NULL && piece != orig )
	    {
		for ( move_index = 0 ;
		      move_index < piece->nbDestination ;
		      move_index++ )
		{
		    if ( piece->destination[ move_index ] == square )
		    {
			if ( !add_one_order( moves_order,
					     orig->index,
					     0,
					     e_First_Move,
					     piece->index,
					     move_index,
					     e_Last_Move,
					     exploration,
					     "square constraint 1" ) )
			{
			    return FALSE ;
			}
		    }
		}
	    }

	    if ( dest != NULL
		 && piece != dest
		 && piece->pieceCapturante == PIECE_PAS_CAPTUREE
		 && piece->nbDestination >= 2 )
	    {
		for ( move_index = piece->nbDestination - 2 ;
		      move_index >= 0 ;
		      move_index-- )
		{
		    if ( piece->destination[ move_index ] == square )
		    {
			if ( !add_one_order( moves_order,
					     piece->index,
					     move_index + 1,
					     e_First_Move,
					     dest->index,
					     dest->nbDestination - 1,
					     e_Last_Move,
					     exploration,
					     "square constraint 2" ) )
			{
			    return FALSE ;
			}
		    }
		}
	    }
	}
    }

    return TRUE ;
}

/*--------------------------------------------------------------------------*/

static bool_t add_castling_constraints(
    const piece_t*		king,
    const piece_t*		rook,
    struct moves_order_t*	moves_order ,
    const exploration_t*	exploration )
{
    square_t	intermediates[ 3 ];
    bool_t	queen_side ;
    int		i ;
    square_t	destination ;

    /* Castling occurs before first move of the rook */
    if ( rook->distance[ 0 ] != 0 )
    {
	if ( !add_one_order( moves_order,
			     king->index,
			     0,
			     e_Last_Move,
			     rook->index,
			     0,
			     e_First_Move,
			     exploration,
			     "castling : occurs before 1st move of rook" ) )
	{
	    return FALSE ;
	}
    }

    if ( king->castling == KING_SIDE )
    {
	intermediates[ 0 ] = king->caseInitiale + 1 ;
	intermediates[ 1 ] = king->caseInitiale + 2 ;
	queen_side = FALSE ;
    }
    else
    {
	intermediates[ 0 ] = king->caseInitiale - 1 ;
	intermediates[ 1 ] = king->caseInitiale - 2 ;
	intermediates[ 2 ] = king->caseInitiale - 3 ;
	queen_side = TRUE ;
    }

    for ( i = INDEX_ROI_BLANC ; i <= INDEX_PION_NOIR_H ; i++ )
    {
	const piece_t*	piece = &exploration->pieces[ i ];

	/* Castling occurs after intermediate squares have been emptied */
	if ( piece != rook
	     && ( piece->caseInitiale == intermediates[ 0 ]
		  || piece->caseInitiale == intermediates[ 1 ]
		  || ( queen_side
		       && piece->caseInitiale == intermediates[ 2 ] ) ) )
	{
	    bool_t	ok ;

	    if ( piece->distance[ 0 ] == 0
		 && NbCoupsBlancsRestants < 2
		 && NbCoupsNoirsRestants  < 2 )
	    {
		bool_t		captured_on_square = TRUE;
		piece_index_t	capturer_index 	   = piece->pieceCapturante;
		piece_index_t	captured_index     = piece->index;
		int		nbCaptures 	   = 0;
		int		move_index ;
		while ( captured_on_square )
		{
		    const piece_t* capturer
			= &exploration->pieces[ capturer_index ];

		    nbCaptures++;

		    move_index = 0 ;
		    while ( capturer->pieceCapturee[ move_index ]
			    != captured_index )
		    {
			move_index++;
		    }
		    if ( move_index == capturer->nbDestination - 1 )
		    {
			if ( capturer->pieceCapturante == PIECE_PAS_CAPTUREE )
			{
#ifdef DEBUG_TEST_MOVE_ORDER
			    fprintf( MainFD,
				     "Impossible move order.  See file %s:%d\n",
				     __FILE__, __LINE__ );
#endif /* DEBUG_TEST_MOVE_ORDER */
			    return FALSE ;
			}
			capturer_index = capturer->pieceCapturante ;
			captured_index = capturer->index ;
		    }
		    else
		    {
			captured_on_square = FALSE;
		    }
		}
		if ( nbCaptures == 1 )
		{
		    ok = add_one_order(
			moves_order,
			capturer_index,
			move_index + 1,
			e_First_Move,
			king->index,
			0,
			e_First_Move,
			exploration,
			"castling : blocking piece must be captured" );
		}
		else
		{
		    ok = TRUE;
		}
	    }
	    else if ( piece->distance[ 0 ] > 0 )
	    {
		ok = add_one_order(
		    moves_order,
		    piece->index,
		    0,
		    e_First_Move,
		    king->index,
		    0,
		    e_First_Move,
		    exploration,
		    "castling : blocking piece must move before castling" );
	    }
	    else
	    {
		ok = TRUE;
	    }
	    if ( !ok )
	    {
		return FALSE;
	    }
	}

	/* Castling occurs before intermediate squares are definitively
	 * occupied */
	destination = piece->destination[ piece->nbDestination - 1 ];
	if ( piece != king
	     && piece != rook
	     && piece->pieceCapturante == PIECE_PAS_CAPTUREE
	     && ( destination == intermediates[ 0 ]
		  || destination == intermediates[ 1 ]
		  || ( queen_side && destination == intermediates[ 2 ] ) ) )
	{
	    if ( !add_one_order(
		     moves_order,
		     king->index,
		     0,
		     e_Last_Move,
		     piece->index,
		     piece->nbDestination - 1,
		     e_Last_Move,
		     exploration,
	"castling : intermediate square cannot be definitively occupied" ) )
	    {
		return FALSE ;
	    }
	}
    }

    /* Castling occures before origin squares of King or Rook are needed */
    /* This is done for the King but not for the Rook in the general algotithm.
     */

    return TRUE ;
}

/*--------------------------------------------------------------------------*/
static bool_t add_straigth_line_constraints(
    struct moves_order_t*	moves_order ,
    const exploration_t*	exploration )
{
    /* A la 1re destination
     *	- si la distance vaut 1 et pas un cavalier et si nbCoupsRestants == 0
     *	- ou si c'est un pion, tant qu'il avance en ligne droite
     * Crer une case couple entre arrive et dpart
     * Vrifier toutes les trajectoires
     * Si une trajectoire est impossible, elle est faite APRES.
     *
     * idem avec la dernire destination, mais la trajectoire est faite AVANT.
     */

    return TRUE ;
}

/*--------------------------------------------------------------------------*/

static void
print_one_move( FILE*			fd,
		piece_index_t		piece_index,
		unsigned int		move_index,
		move_type_t		type,
		const exploration_t*	exploration )
{
    square_t	from ;
    piece_t	piece = exploration->pieces[ piece_index ];
    static char	move_type[] = { ' ', 'F', 'L', 'B' };

    if ( move_index == 0 )
    {
	from = piece.caseInitiale ;
    }
    else
    {
	from = piece.destination[ move_index - 1 ];
    }

    fprintf( fd,
	     "%c%c%c-%c%c",
	     InitialePieces[ piece.typePiece ],
	     'a' + column( from ),
	     '1' + ( row( from ) >> 4 ),
	     'a' + column( piece.destination[ move_index ] ),
	     '1' + ( row( piece.destination[ move_index ] ) >> 4 ) );
    fprintf( fd, "(%d,%d", piece_index, move_index );
    if ( type != e_First_And_Last_Move )
    {
	fprintf( fd, ",%c)", move_type[ type ] );
    }
    else
    {
	fprintf( fd, ")  " );
    }

}

/*--------------------------------------------------------------------------*/

static void
add_square_to_list(
    square_t		square,
    square_t*		squares,
    unsigned int*	nb_squares )
{
    unsigned int	i ;

    assert( *nb_squares < MAX_DESTINATION_PER_TYPE );

    for ( i = 0 ; i < *nb_squares ; i++ )
    {
	if ( squares[ i ] == square )
	{
	    break ;
	}
    }

    if ( i == *nb_squares )
    {
	squares[ i ] = square ;
	(*nb_squares)++ ;
    }
}

/*--------------------------------------------------------------------------*/

static bool_t
add_one_order( struct moves_order_t*	moves_order,
	       piece_index_t		before_piece_index,
	       unsigned int		before_move_index,
	       move_type_t		before_type,
	       piece_index_t		after_piece_index,
	       unsigned int		after_move_index,
	       move_type_t		after_type,
	       const exploration_t*	exploration,
	       const char*		description )
{
    bool_t	ok = add_move_order( moves_order,
				     before_piece_index,
				     before_move_index,
				     before_type,
				     after_piece_index,
				     after_move_index,
				     after_type );

    if ( !ok && MainPrintPositionLevel > 1 )
    {
	writeAssociation(
	    "Position eliminated because of impossible order :" );
	print_moves_order( MainFD, moves_order, exploration );
	fprintf( MainFD, "Impossible order (%s) : ", description );
	print_one_move( MainFD,
			before_piece_index,
			before_move_index,
			before_type,
			exploration );
	fprintf( MainFD, " -> " );
	print_one_move( MainFD,
			after_piece_index,
			after_move_index,
			after_type,
			exploration );
	fprintf( MainFD, "\n" );
    }

    return ok ;
}

/*--------------------------------------------------------------------------*/

static bool_t
add_constraint(
    struct moves_order_t*	moves_order,
    const exploration_t*	exploration,
    square_t			square,
    piece_index_t		piece_index,
    constraint_type_t		constraint_type )
{
    int			i ;
    const piece_t*	p ;

    /* Used if captured_without_moving : we initialise them to avoid a compiler
     * warning. */
    piece_index_t	capturing_piece_index = 0 ;
    unsigned int	capturing_move_index  = 0 ;

    if ( constraint_type == e_Captured_Without_Moving )
    {
	const piece_t*	capturing_piece ;
	capturing_piece_index =
	    exploration->pieces[ piece_index ].pieceCapturante ;
	capturing_piece = &exploration->pieces[ capturing_piece_index ];
	for ( capturing_move_index = 0 ;
	      capturing_move_index < capturing_piece->nbDestination ;
	      capturing_move_index++ )
	{
	    if ( capturing_piece->pieceCapturee[ capturing_move_index ] ==
		 piece_index )
	    {
		break ;
	    }
	}
    }

    for ( i = 0, p = &( exploration->pieces[ INDEX_ROI_BLANC ] ) ;
	  i < exploration->nb_moving_pieces ;
	  i++, p++ )
    {
	int		move_index ;
	piece_type_t	piece_type = p->typePiece ;

	/* Skip the piece that must not be tested */
	if ( p->distance[ 0 ] == 0 )
	{
	    continue ;
	}
	if ( constraint_type == e_Initial_Square
	     && p->caseInitiale == square )
	{
	    continue ;
	}
	if ( constraint_type == e_Captured_Without_Moving &&
	     ( i + INDEX_ROI_BLANC == capturing_piece_index ) )
	{
	    continue ;
	}
	if ( constraint_type == e_Destination_Square &&
	     i + INDEX_ROI_BLANC == piece_index )
	{
	    continue ;
	}

	if ( constraint_type == e_Destination_Square )
	{
	    int index_promotion = p->nbDestination;

	    if ( p->casePromotion != CASE_PAS_DE_PROMOTION )
	    {
		piece_type = p->typePromotion ;
		index_promotion = promotion_index( p );
	    }

	    for ( move_index = p->nbDestination - 1 ;
		  move_index >= 0 ;
		  move_index-- )
	    {
		square_t current_square;

		if ( move_index == 0 && p->typePiece == ROI
		     && p->castling != NO_CASTLING )
		{
		    continue ;
		}

		if ( move_index == 0 )
		{
		    current_square = p->caseInitiale ;
		}
		else
		{
		    current_square = p->destination[ move_index - 1 ];
		}

		if ( move_index == index_promotion )
		{
		    piece_type = PION ;
		}

		if ( current_square != CASE_ARRIVEE_QUELCONQUE
		     && current_square != square
		     && p->destination[ move_index ]
		     != CASE_ARRIVEE_QUELCONQUE )
		{
		    distance_t	delta = distance_delta(
			piece_type,
			p->camp,
			current_square,
			p->destination[ move_index ] );
		    if ( delta > 0 )
		    {
			if ( ( p->camp == BLANC
			       && delta > NbCoupsBlancsRestants )
			     || ( p->camp == NOIR
				  && delta > NbCoupsNoirsRestants ) )
			{
			    if ( !add_one_order(
				     moves_order,
				     i + INDEX_ROI_BLANC,
				     move_index,
				     e_First_Move,
				     piece_index,
				     exploration->pieces[
					 piece_index ].nbDestination - 1,
				     e_Last_Move,
				     exploration,
				     "add_constraint 1" ) )
			    {
				return FALSE ;
			    }
			    break ;
			}
		    }
		}
	    }
	}
	else
	{
	    square_t	current_square 	= p->caseInitiale ;

	    for ( move_index = 0 ;
		  move_index < p->nbDestination ;
		  move_index++ )
	    {
		if ( move_index == 0 && p->typePiece == ROI
		     && p->castling != NO_CASTLING )
		{
		    current_square = p->destination[ 0 ];
		    continue ;
		}

		/* Useless : if ( p->casePromotion != CASE_PAS_DE_PROMOTION */
		if ( current_square == p->casePromotion )
		{
		    /* If the piece return to its promotion square, this is
		     * still correct. */
		    piece_type = p->typePromotion ;
		}

		if ( current_square != CASE_ARRIVEE_QUELCONQUE
		     && current_square != square
		     && p->destination[ move_index ]
		     != CASE_ARRIVEE_QUELCONQUE )
		{
		    distance_t	delta = distance_delta(
			piece_type,
			p->camp,
			current_square,
			p->destination[ move_index ] );
		    if ( delta > 0 )
		    {
			if ( ( p->camp == BLANC
			       && delta > NbCoupsBlancsRestants )
			     || ( p->camp == NOIR
				  && delta > NbCoupsNoirsRestants ) )
			{
			    piece_index_t	before_piece_index ;
			    unsigned int	before_move_index  ;
			    move_type_t		before_type ;
			    switch( constraint_type )
			    {
			    case e_Captured_Without_Moving:
			    {
				before_piece_index = capturing_piece_index ;
				before_move_index  = capturing_move_index  ;
				before_type        = e_Last_Move ;
			    }
			    break;
			    case e_Initial_Square:
			    {
				before_piece_index = piece_index ;
				before_move_index  = 0 ;
				before_type        = e_First_Move ;
			    }
			    break;
			    default:
				assert( FALSE );
				exit( 1 );
				break;
			    }

			    if ( !add_one_order(
				     moves_order,
				     before_piece_index,
				     before_move_index,
				     before_type,
				     i + INDEX_ROI_BLANC,
				     move_index,
				     e_Last_Move,
				     exploration,
				     "add_constraint 2" ) )
			    {
				return FALSE ;
			    }
			    break ;
			}
		    }
		}
		current_square = p->destination[ move_index ];
	    }
	}
    }

    return TRUE ;
}
