/* ScummVM - Graphic Adventure Engine
*
* ScummVM is the legal property of its developers, whose names
* are too numerous to list here. Please refer to the COPYRIGHT
* file distributed with this source distribution.
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <http://www.gnu.org/licenses/>.
*
*/

#ifndef DARKSEED_LANGTEXT_H
#define DARKSEED_LANGTEXT_H

#include "common/language.h"

namespace Darkseed {

enum class KoreanObjectSuffixType {
	None,
	Object,
	Topic,
	Linking,
	Location
};

struct I18nText {
	const char *en;
	const char *es;
	const char *fr;
	const char *de;
	const char *ko;
	const char *zh;
};

struct TextWithPosition {
	int x;
	int y;
	const char *text;
};

struct I18NTextWithPosition {
	const TextWithPosition en;
	const TextWithPosition es;
	const TextWithPosition fr;
	const TextWithPosition de;
	const TextWithPosition ko;
	const TextWithPosition zh;
};

Common::U32String getI18NText(const I18nText &text);
const TextWithPosition &getI18NTextWithPosition(const I18NTextWithPosition &i18nTextWithPosition);

Common::U32String convertToU32String(const char *text, Common::Language language);

Common::U32String formatInjectStrings(const Common::u32char_type_t *format, ...);

constexpr I18nText kI18N_CarKeysIgnitionText = {
	"You see the car keys in the ignition.",
	"VES LAS LLAVES DEL COCHE EN EL CONTACTO.",
	"VOUS VOYEZ LES CLEFS DE LA VOTTURE DAUS LE STARTER.",
	"DU SIEHSTDIE AUTOSCHLUSSEL IM ANLASSER.",
	"\xb8\x61\x95\xb7\xc0\x61\x20\xb5\x69\xae\x41\x88\x61\x20\xb7\xa1\x8b\x61\x93\xa1\xad\x65\xb5\x41\x20\x8d\xb9\xd1\x61\xb7\xb6\x93\x65\x88\xf5\xb7\xa1\x20\xa5\xa1\xb7\xb3\x93\xa1\x94\x61\x2e",
	"\xa7\x41\xac\xdd\xa8\xec\xa4\x46\xa8\xae\xc6\x5f\xb0\xcd\xa6\x62\xc2\x49\xa4\xf5\xb6\x7d\xc3\xf6\xa4\x57\xa1\x43"
};

constexpr I18nText kI18N_YouSeeIronBarsText = {
	"You see the iron bars of your cell.",
	"VES LAS BARRAS DE HIERRO DE TU CELDA.",
	"VOUS VOYEZ LES BARREAUX DE FER DE VOTRE CELLULE.",
	"DU SIEHST DIE EISENSTANGEN IN DER ZELLE.",
	"\x88\x71\xa4\x77\xb7\x81\x20\xb8\x61\xa2\x89\xae\x41\xc9\xb7\x20\xb7\xb3\x93\xa1\x94\x61\x2e",
	"\xa7\x41\xac\xdd\xa8\xec\xa4\x46\xba\xbb\xc5\xa2\xb8\xcc\xaa\xba\xc5\x4b\xb1\xf8\xa1\x43"
};

constexpr I18nText kI18N_YouSeeDelbertText = {
	"You see Delbert, not much to look at.",
	"VES A DELBERT, NO HAY MUCHO QUE VER.",
	"VOUS VOYEZ DELBERT, PAS GRAND CHOSE A VOIR.",
	"DU SIEHST DELBERT, NICHT VIEL ZU SEHEN",
	"\xd0\x65\x88\x61\x9d\xb3\x89\x41\x20\xb4\x81\x89\x65\x89\xc1\xb7\x81\x20\x91\xa9\xb7\xa1\xb5\x41\x20\xb5\x69\xba\x97\xd0\x61\x89\xa1\xb7\xb6\xaf\x73\x93\xa1\x94\x61\x2e",
	"\xa7\x41\xac\xdd\xa8\xec\xa4\x46\xc0\xb9\xba\xb8\xa7\x42\xa1\x41\xa5\x4c\xa6\xfc\xa5\x47\xa6\xa3\xb5\xdb\xb3\x72\xaa\xaf\xaa\xb1\xa1\x43"
};

constexpr I18nText kI18N_YouSeeTheClerkText = {
	"You see the clerk.",
	"VES AL EMPLEADO.",
	"VOUS VOYEZ L'HOMME.",
	"DU SIEHST DEN MANN.",
	"\xb7\xa1\x20\x88\x61\x89\x41\xb7\x81\x20\xb8\xf1\xb6\xa5\xb7\xb3\x93\xa1\x94\x61\x2e",
	"\xa7\x41\xac\xdd\xa8\xec\xa4\x46\xa9\xb1\xad\xfb\xa1\x43"
};

constexpr I18nText kI18N_YouSeeTheOpenGloveBoxText = {
	"You see the open glove box.",
	"VES LA CAJA DE LOS GUANTES ABIERTA.",
	"VOUS VOYEZ LA BOITE DE GANT OUVERTE.",
	"DU SIEHST DAS OFFENE HANDSCHUHFACH.",
	"\xb8\x77\x88\x73\xc9\xb7\xb7\xa1\x20\xb5\x69\x9d\x61\xb7\xb6\x93\x65\x88\xf5\xb7\xa1\x20\xa5\xa1\xb7\xb3\x93\xa1\x94\x61\x2e",
	"\xa7\x41\xac\xdd\xa8\xec\xa5\xb4\xb6\x7d\xa4\x46\xaa\xba\xa4\xe2\xae\x4d\xb2\xb0\xa1\x43"
};

constexpr I18nText kI18N_youSeeTheText = {
  "You see the %s.",
  "VES %s.",
  "VOUS VOYEZ %s.",
  "DU SIEHST %s.",
  "%s \xac\x69\xcd\x61\xa5\xb3\x93\xa1\x94\x61",
	"\xa7\x41\xac\xdd\xa8\xec\xa4\x46%s\xa1\x43"
};

constexpr I18nText kI18N_TheCopsIgnoreYourDemandsText = {
  "The cops ignore your demands for attention.",
  "LOS POLICIAS IGNORAN TUS LLAMADAS DE ATENCION.",
  "LES FLICS IGNORENT VOTRE DEMANDE D'ATTENTION.",
  "DIE POLIZISTEN BEACHTEN NICHT DEINE FORDERUNGEN.",
  "\xb4\x61\xa2\x81\x9f\xa1\x20\xaf\xa5\xd1\xa1\x9f\x69\x20\xa5\xa1\x90\x81\x95\xa1\x20\x89\x77\xc0\x69\x97\x69\xb7\x65\x20\xc1\x61\x94\x61\xa5\xa9\x20\xac\x97\x88\x62\x95\xa1\x20\xd0\x61\xbb\xa1\x20\xb4\x67\x93\x65\x8a\x85\xb6\x61\x2e",
	nullptr // Untranslated in this version
};

constexpr I18nText kI18N_ThePhoneIsRingingText = {
  "The phone is ringing.",
  "EL TELEFONO ESTA SONANDO.",
  "LE TELEPHONE SONNE.",
  "DAS TELEFON KLINGELT.",
  "\xb8\xe5\xd1\xc1\xa5\x49\xb7\xa1\x20\xb6\x89\x9f\xb3\x93\xa1\x94\x61\x2e",
  "\xb9\x71\xb8\xdc\xc1\x6e\xc5\x54\xb0\x5f\x2e\x0a"
};

constexpr I18nText kI18N_TheDoorbellIsRingingText = {
  "The doorbell is ringing.",
  "EL TIMBRE DE LA PUERTA ESTA SONANDO.",
  "LA SONETTE DE LA PORTE SONNE.",
  "DIE TUERKLINGEL LAEUTET.",
  "\xd1\x65\x89\xc5\x20\xc1\xa1\xb7\xa5\xb9\xb7\xb7\xa1\x20\xb6\x89\x9f\xb3\x93\xa1\x94\x61\x2e",
  "\xaa\xf9\xb9\x61\xc1\x6e\xc5\x54\xb0\x5f\x2e\x0a"
};

constexpr I18nText kI18N_ChooseAnItemBeforeText = {
  "Choose an item before giving clerk more money.",
  "ELIGE UN OBJETO ANTES DE DARLE AL EMPLEADO MAS DINERO.",
  "CHOISISSEZ QUELQUE CHOSE AVANT DE REMETTRE L'ARGENT A VENDEUR.",
  "SUCHE ETWAS AUS BEVOR DU DEN MANN BEZAHLST.",
  "\xb8\xf1\xb6\xa5\xb5\x41\x89\x41\x20\x95\xa5\xb7\x69\x20\x94\xe1\x20\xba\x81\x8b\xa1\xb8\xe5\xb5\x41\x20\xa0\xe5\xb8\xe1\x20\x8a\x81\xb7\xb3\xd0\x69\x20\xa2\x89\x88\xe5\xb7\x69\x20\x89\xa1\x9f\x61\xad\x41\xb6\x61\x2e",
  "\xad\x6e\xae\xb3\xb3\x66\xab\x7e\xab\x65\xa1\x41\xb0\x4f\xb1\x6f\xbd\xd0\xa5\xfd\xa5\x49\xbf\xfa\xb5\xb9\xa9\xb1\xad\xfb\xa1\x43"
};

constexpr I18nText kI18N_YouTouchDelbertText = {
  "You touch Delbert...",
  "TOCAS A DELBERT...",
  "VOUS TOUCHEZ DELBERT.",
  "GREIFE DELBERT AN...",
  "\xae\x89\xb7\x69\x20\xd0\x65\xa4\xe5\x20\x8a\xa5\xd0\x81\x20\xa5\xa1\x93\x65\x88\xf5\xb7\xa1\x20\xb4\xe1\x98\xe9\x8c\x61\xb6\x61\x3f",
  "\xa7\x41\xb8\x49\xc4\xb2\xc0\xb9\xba\xb8\xa7\x42\xa4\x40\xa4\x55\xa1\x44\xa1\x44"
};

constexpr I18nText kI18N_YouTouchTheColdIronBarsText = {
  "You touch the cold iron bars.",
  "TOCAS LAS FRIAS BARRAS DE HIERRO.",
  "VOUS TOUCHEZ LES BARREAUX DE FER.",
  "GREIFE DIE KALTEN EISEN STANGEN AN.",
  "\xc0\x77\xac\x69\xb7\x81\x20\x88\x71\xc1\xa2\xb7\x65\x20\xb5\x62\xaf\xa1\x20\xc0\x61\x88\x73\x8a\x85\xb6\x61\x21",
  "\xa7\x41\xba\x4e\xb5\xdb\xa6\x42\xa7\x4e\xaa\xba\xc5\x4b\xb1\xf8\xa1\x43"
};

constexpr I18nText kI18N_TheSergeantSaysNiceGunText = {
  "The sergeant says 'Nice gun eh? It's a Browning'",
  "EL SARGENTO DICE: 'BUENA PISTOLA, EH? ES UNA BROWNING.'",
  "LE SERGENT DIT: BEAU REVOLVER HEIN, C'EST UN BROWNING.",
  "DER SEARGENT SAGT 'SCHOENE PISTOLE, EH? ES IST EIN BROWNING.'",
  "\xc1\xb7\x20\x8b\x65\xc0\xe1\xb5\x41\xac\xe1\x20\x98\xe9\xb4\xe1\xbb\xa1\xaf\xa1\xb6\x61\x21",
  "\xc4\xb5\xa9\x78\xbb\xa1\xa1\x47\xac\x4f\xa7\xe2\xa6\x6e\xba\x6a\xa7\x63\xa1\x48\xb3\x6f\xac\x4f\xa5\xd5\xae\xd4\xb9\xe7\xa4\xe2\xba\x6a\xa9\x4f\xa1\x49"
};

constexpr I18nText kI18N_YouTurnOnTheMusicText = {
  "You turn on the music.",
  "PONES MUSICA.",
  "VOUS METTEZ LA MUSIQUE.",
  "SCHALTE DIE MUSIK AN.",
  "\xc0\x61\x20\x20\x9c\x61\x97\xa1\xb5\xa1\x9f\x69\x20\xc5\x73\x93\xa1\x94\x61\x2e",
  "\xa7\x41\xa5\xb4\xb6\x7d\xad\xb5\xbc\xd6\xa1\x43"
};

constexpr I18nText kI18N_YouTurnOffTheMusicText = {
  "You turn off the music.",
  "QUITAS LA MUSICA.",
  "VOUS ARRETEZ LA MUSIQUE.",
  "SCHALTE DIE MUSIK AB.",
  "\xc0\x61\x20\x9c\x61\x97\xa1\xb5\xa1\x9f\x69\x20\x8f\x73\x93\xa1\x94\x61\x2e",
  "\xa7\x41\xc3\xf6\xb1\xbc\xad\xb5\xbc\xd6\xa1\x43"
};

constexpr I18nText kI18N_YouTouchTheOrnateSignalText = {
  "You touch the surface of the ornate sigil.",
  "TOCAS LA SUPERFICIE DE LA FIGURA ADORNADA.",
  "VOUS TOUCHEZ LA SURFACE DE LA PIERRE MAGIQUE.",
  "GREIFE DIE VERZAUBERTEN STEINE AN.",
  "\xac\xe2\xa2\x61\xb7\x81\x20\xa5\xb7\xb7\xa5\xb7\x69\x20\xa0\x65\xb9\x61\xa5\xb3\x93\xa1\x94\x61\x2e",
  "\xa7\x41\xc4\xb2\xba\x4e\xb5\xdb\xc0\x4a\xb9\xa2\xba\xeb\xac\xfc\xaa\xba\xab\xca\xa6\x4c\xaa\xed\xad\xb1\xa1\x43"
};

constexpr I18nText kI18N_ThisSentryCannotBeStoppedWithText = {
	"This sentry cannot be stopped with a %s.",
	"NO PUEDES PARAR A ESTE CENTINELA CON %s.",
	"VOUS N'ARRETEREZ PAS LA SENTINELLE AVEC %s.",
	"DIESE WACHE KANN NICHT AUFGEHALTEN WERDEN MIT %s.",
	"%s\x9d\xa1\x93\x65\x20\x88\x65\xae\x81\x9f\x69\x20\xb8\xe1\xbb\xa1\xd0\x69\x20\xae\x81\x20\xb4\xf4\xaf\x73\x93\xa1\x94\x61\x2e",
	nullptr // not translated
};

constexpr I18nText kI18N_HasNoEffectOnTheAlienTubesText = {
	"The %s has no effect on the alien tubes.",
	"%s NO TIENE EFECTO SOBRE LOS TUBOS ALIENIGENAS.",
	"%s RESTE SANS EFFET SUR LES TUBES EXTRA-TERRESTRES.",
	"%s BEWIRKT NICHTS AN AUSSERIRDISCHEN TUNNELS.",
	"%s \x95\xb7\x9d\x62\xac\xe5\xb5\x41\x20\xb4\x61\xa2\x81\x9c\xe5\x20\xb5\x77\xd0\xb7\xb7\x69\x20\xa3\xa1\xc3\xa1\xbb\xa1\x20\xa1\xb5\xd0\x73\x93\xa1\x94\x61\x2e",
	"\xa5\xce\x25\x73\xa6\x62\xb3\x6f\xa8\xc7\xb2\xa7\xa7\xce\xb8\xd5\xba\xde\xa4\x57\xa1\x41\xa4\x40\xc2\x49\xae\xc4\xaa\x47\xa4\x5d\xa8\x53\xa6\xb3\xa1\x43"
};

constexpr I18nText kI18N_YouIncinerateTheText = {
	"You incinerate the %s inside the power nexus!.",
	"INCINERAS %s DENTRO DE LA FUENTA DE ENERGIA.",
	"VOUS INCINEREZ %s A L'INTERIEUR DU LIEN DE PUISSANCE.",
	"DU VERNICHTEST %s MIT DER STROMVERSORGUNG!",
	"%s \xb5\xc5\xb8\xe5\xd3\xa1\x20\xcc\x61\x8a\x41\xaf\xa1\xc5\x76\xaf\x73\x93\xa1\x94\x61\x2e",
	"\xa7\x41\xb1\x4e\x25\x73\xa1\x41\xa9\xf1\xa6\x62\xaf\xe0\xb6\x71\xc1\x70\xb5\xb2\xa8\x74\xb2\xce\xa4\xa4\xa4\xdf\xa4\xba\xbf\x4e\xa6\xa8\xa6\xc7\xc2\x75\xa1\x49"
};

constexpr I18nText kI18N_HasNoEffectOnTheAlienMonstrosityText = {
	"The %s has no effect on this alien monstrosity.",
	"%s NO TIENE EFECTO SOBRE ESTA MONSTRUOSIDAD ALIENIGENA.",
	"%s RESTE SANS EFFET SUR CETTE MONSTRUOSITE EXTRA-TERRESTRE.",
	"%s BEWIRKT NICHTS AM AUSSERIRDISCHEN MONSTRUM.",
	"%s \xb7\x41\x8b\xa1\xc1\x41\x20\xc4\xf1\xcf\x41\xc8\xe1\xb5\x41\x20\xb5\x77\xd0\xb7\xb7\x69\x20\xa3\xa1\xc3\xa9\xae\x85\x20\xb4\xf4\xaf\x73\x93\xa1\x94\x61\x2e",
	"\xa5\xce%s\xb9\xef\xa5\x49\xb3\x6f\xa8\xc7\xb2\xa7\xa7\xce\xa9\xc7\xaa\xab\xa1\x41\xae\xda\xa5\xbb\xa8\x53\xa6\xb3\xa5\xce\xa1\x43"
};

constexpr I18nText kI18N_cutTheWiresText = {
	"The %s is not strong enough to cut the cables.\nPerhaps if you had some wire cutters?",
	"%s NO ES LO BASTANTE RESISTENTE COMO PARA CORTAR LOS CABLES.\nY SI TUVIERAS UN CORTACABLES?",
	"%s NE POURRA PAS COUPER LES CABLES.\nET SI VOUS AVIEZ DES PINCES COUPANTES?",
	"%s IST NICHT STARK GENUG, UM DIE KABEL ABZUSCHNEIDEN.\nHAST DU VIELLEICHT EINEN KABELSCHNEIDER?",
	"%s \xc5\x41\xb7\xa1\xa7\x69\xb7\x69\x20\xb8\xe9\x94\x65\xd0\x69\x20\xae\x81\x20\xb7\xb6\xb7\x69\x20\xb8\xf7\x95\xa1\x9d\xa1\x20\x88\x77\xd0\x61\xbb\xa5\x20\xa1\xb5\xd0\x73\x93\xa1\x94\x61\x2e\x20\x89\xc1\xb5\x65\x20\xc5\x41\xb7\xa1\xa7\x69\xb7\x69\x20\xb8\xe9\x94\x65\xd0\x69\x20\xae\x81\x20\xb7\xb6\xb7\x69\x8c\x61\xb6\x61\x3f",
	"%s\xae\xda\xa5\xbb\xa8\x53\xbf\xec\xaa\x6b\xa4\xc1\xc2\x5f\xb3\x6f\xa8\xc7\xb9\x71\xc6\x6c\xbd\x75\xa1\x43 \xa9\xce\xb3\xa7\x41\xa6\xb3\xaa\xf7\xc4\xdd\xa4\xc1\xb3\xce\xbe\xb9\xb6\xdc\xa1\x48"
};

constexpr I18nText kI18N_NoEffectOnTheProtectedAncientText = {
	"The %s has no effect on the protected ancient.",
	"%s NO TIENE EFECTO SOBRE EL ANTEPASADO PROTEGIDO.",
	"%s N'A AUCUN EFFET SUR L'ANCIEN PROTEGE.",
	"%s BEWIRKT NICHTS BEI DEM GESCHUETZTEN ALTEN.",
	"%s \xc4\x93\xaf\x49\xad\xa2\xb7\x81\x20\xa3\xa1\xb7\xa1\x9c\x61\xb5\x41\x89\x41\x20\xb5\x77\xd0\xb7\xb7\x69\x20\xa3\xa1\xc3\xa1\xbb\xa1\x20\xa1\xb5\xd0\x73\x93\xa1\x94\x61\x2e",
	"\x25\x73\xb9\xef\xb3\x6f\xa8\xc7\xa8\xfc\xa8\xec\xab\x4f\xc5\x40\xaa\xba\xa5\x6a\xa5\x4e\xa4\x48\xa1\x41\xa8\xc3\xa8\x53\xa6\xb3\xa5\xf4\xa6\xf3\xae\xc4\xa5\xce\xa1\x43"
};

constexpr I18nText kI18N_YouHideTheObjectUnderThePillowText = {
	"You hide the %s under the pillow.",
	"OCULTAS %s BAJO LA ALMOHADA.",
	"VOUS CACHEZ %s SOUS L'OREILLER.",
	"DU VERSTECKST %s UNTER DEM KISSEN.",
	"%s \xa5\x41\x88\x81\xa3\xbb\xb5\x41\x20\xae\x91\x8b\xb3\x93\xa1\x94\x61\x2e",
	"\xa7\x41\xa7\xe2%s\xc2\xc3\xa6\x62\xaa\x45\xc0\x59\xa9\xb3\xa4\x55\xa1\x43"
};

constexpr I18nText kI18N_DontPutTheObjectInTheTrunkText = {
	"Don't put the %s in the trunk, you'll forget it.",
	"NO PONGAS %s EN EL BAUL, TE OLVIDARAS.",
	"NE METTEZ PAS %s DANS LE COFFRE, VOUS L'OUBLIEREZ.",
	"WENN DU %s IN DEN TRUHE LEGST, VERGISST DU ES.", // TODO check this is correct.
	"%s \xcb\x61\x9c\xf7\xc7\x61\xb5\x41\x20\x90\xfd\xbb\xa1\x20\xa0\x61\xaf\xb3\xaf\xa1\xb6\x61\x2e",
	"\xa4\xa3\xad\x6e\xb1\x4e\x25\x73\xa9\xf1\xa6\x62\xa8\x54\xa8\xae\xa6\xe6\xa7\xf5\xbd\x63\xa4\xa4\xa1\x41\xa7\x41\xb1\x4e\xab\xdc\xae\x65\xa9\xf6\xa7\xd1\xb0\x4f\xa5\xa6\xb4\xbf\xb8\x67\xa9\xf1\xa6\x62\xa6\xf3\xb3\x42\xa1\x43"
};

constexpr I18nText kI18N_TheCarWontStartWithTheText = {
	"The car won't start with the %s.",
	"EL COCHE NO SE PONDRA EN MARCHA CON %s.",
	"LA VOITURE NE DEMARRERA PAS AVEC %s.",
	"DAS AUTO SPRINGT NICHT AN MIT %s.",
	"%s\x9d\xa1\x93\x65\x20\xb8\x61\x95\xb7\xc0\x61\xb7\x81\x20\xaf\xa1\x95\xb7\xb7\x69\x20\x88\xe9\x20\xae\x81\x20\xb4\xf4\xaf\x73\x93\xa1\x94\x61\x2e",
	"\xa8\xcf\xa5\xce%s\xa8\xc3\xa4\xa3\xaf\xe0\xc5\xfd\xa8\x54\xa8\xae\xb5\x6f\xb0\xca\xa1\x43"
};

constexpr I18nText kI18N_IfYouPutTheObjectInTheTrunkText = {
	"If you put the %s in the trunk, you'll forget it.",
	"SI PONES %s EN EL BAUL, TE OLVIDARAS.",
	"SI VOUS METTEZ %s DANS LE COFFRE, VOUS L'OUBLIEREZ.",
	"WENN DU %s IN DEN TRUHE LEGST, VERGISST DU ES.",
	"%s \xb7\xb0\xb4\xe1\xa4\xe1\x9f\xa1\x89\xa1\x20\xaf\xbc\xb7\x61\xaf\xb3\x93\xa1\x8c\x61\x3f",
	"\xad\x59\xac\x4f\xa7\x41\xb1\x4e\x25\x73\xa9\xf1\xa6\x62\xa8\x54\xa8\xae\xa6\xe6\xa7\xf5\xbd\x63\xa4\xa4\xa1\x41\xa7\x41\xb1\x4e\xab\xdc\xae\x65\xa9\xf6\xa7\xd1\xb0\x4f\xa5\xa6\xb4\xbf\xb8\x67\xa9\xf1\xa6\x62\xa6\xf3\xb3\x42\xa1\x43"
};

constexpr I18nText kI18N_TheObjectIsYoursYouHaventLostItText = {
	"The %s is yours, you haven't lost it.",
	"%s ES TUYA, NO LA HASA PERDIDO.",
	"%s EST A VOUS, VOUS NE L'AVEZ PAS PERDUE.",
	"%s GEHOERT DIR, DU HAST ES NICHT VERLOREN.",
	"%s \xb7\xa1\xa3\xa1\x20\x94\x77\xaf\xa5\xb7\x81\x20\x88\xf5\xb7\xb3\x93\xa1\x94\x61\x2e",
	"%s\xa4\x77\xb8\x67\xac\x4f\xa7\x41\xaa\xba\xa4\x46\xa1\x41\xa7\x41\xa8\xc3\xa8\x53\xa6\xb3\xbf\xf2\xa5\xa2\xb1\xbc\xb0\xda\xa1\x48"
};

constexpr I18nText kI18N_notAGoodPlaceToHideTheText = {
	"Not a good place to hide the %s.",
	"NO ES UN BUEN SITIO PARA OCULTAR %S.",
	"VOUS NE POURREZ PAS CACHER %S.",
	"KEIN GUTES VERSTECK FUER %S.",
	"%s \x91\xbd\xb4\x61\x96\x89\xa0\x65\xd0\x65\x20\xb8\x77\xad\xa1\x93\x65\x20\xb4\x61\x93\xb3\x93\xa1\x94\x61\x2e",
	"\xb3\x6f\xa4\xa3\xac\x4f\xa4\x40\xad\xd3\xc2\xc3%s\xaa\xba\xa6\x6e\xa6\x61\xa4\xe8\xa1\x43"
};

constexpr I18nText kI18N_youTryToPutTheObjectInsideButTheDoorWontOpenText = {
	"You try to put the %s inside, but the door won't open",
	"INTENTAS PONER %s DENTRO, PERO LA PUERTA NO SE ABRIRA.",
	"VOUS ESSAYEZ DE METTRE %s A L'INTERIEUR, MAIS LA PORTE NE S'OUVRIRA PAS.",
	"DU VERSUCHST, %s HINEINZUSTELLEN, ABER DIE TUER OEFFNET SICH NICHT.",
	"%s \x8b\x61\x89\xb5\xb5\x41\x20\xac\x61\xb6\x77\xd0\x69\x20\xcf\xa9\xb6\x61\x88\x61\x20\xb7\xb6\xb7\x69\x8c\x61\xb6\x61\x3f",
	"\xa7\x41\xb9\xc1\xb8\xd5\xb1\x4e\x25\x73\xa9\xf1\xb8\x6d\xa6\x62\xb8\xcc\xad\xb1\xa1\x41\xa6\xfd\xac\x4f\xaa\xf9\xa8\xc3\xa8\x53\xa6\xb3\xa5\xb4\xb6\x7d\xa1\x43"
};

constexpr I18nText kI18N_theKitchenIsNoPlaceToKeepTheText = {
	"The kitchen is no place to keep the %s.",
	"LA COCINA NO ES LUGAR PARA GUARDAR %s.",
	"LA CUISINE NE CONVIENT PAS A %s.",
	"DIE KUECHE IST KEIN PLATZ FUER %s.",
	"\xa6\x81\xb4\xfa\xb5\x41\x93\x65 %s \x91\xbd\xb4\x61\x96\x89\xa0\x65\xd0\x65\x20\xb8\x77\xad\xa1\x88\x61\x20\xb4\x61\x93\xb3\x93\xa1\x94\x61\x2e",
	"\xbc\x70\xa9\xd0\xa8\xc3\xa8\x53\xa6\xb3\xa6\x68\xa4\xd6\xaa\xc5\xb6\xa1\xa8\xd3\xa6\x73\xa9\xf1%s"
};

constexpr I18nText kI18N_youllForgetTheObjectHereText = {
	"You'll forget the %s here.",
	"OLIVIDARAS %s AQUI.",
	"VOUS OUBLIEREZ %s ICI.",
	"DU WIRST NOCH %s HIER VERGESSEN.",
	"\xb5\x61\x8b\xa1\x94\x61 %s \x91\xbd\xb4\x61\x96\x81\xa1\x65\x20\xb7\xb0\xb4\xe1\xa4\xe1\x9f\xa9\x20\x88\xf5\xb7\xb3\x93\xa1\x94\x61\x2e",
	"\xa7\x41\xb1\x4e\xb7\x7c\xa7\xd1\xa4\x46\xa7\xe2\x25\x73\xa9\xf1\xa6\x62\xb3\x6f\xb8\xcc\xa1\x43"
};

constexpr I18nText kI18N_youdRatherHaveTheObjectWithYouText = {
	"You'd rather have the %s with you.",
	"SERIA MEJOR TENER %s CONTIGO.",
	"IL VAUDRAIT MIEUX AVOIR %s SUR VOUS.",
	"DU HAETTEST LIEBER %s BEI DIR.",
	"\xc0\x61\x9c\x61\x9f\xa1 %s \x88\x61\xbb\xa1\x93\x65\x89\x41\x20\xb9\xbd\xb7\x69\x88\xf5\x20\xb7\xb3\x93\xa1\x94\x61\x2e",
	"\xa7\x41\xb9\xe7\xc4\x40\xb1\x4e\x25\x73\xaf\x64\xa6\x62\xa8\xad\xa4\x57\xa1\x43"
};

constexpr I18nText kI18N_theObjectHasNoEffectText = {
	"The %s has no effect.",
	"%s NO TIENE EFECTO.",
	"%s N'A AUCUN EFFET.",
	"%s BEWIRKT NICHTS.",
	"%s \xb5\x77\xd0\xb7\xb7\x69\x20\xa3\xa1\xc3\xa1\xbb\xa1\x20\xa1\xb5\xd0\x73\x93\xa1\x94\x61\x2e",
	"%s\xa8\x53\xa6\xb3\xae\xc4\xaa\x47\xa1\x43"
};

constexpr I18nText kI18N_thisIsNotAGoodPlaceForTheText = {
	"This is not a good place for the %s.",
	"NO ES UN BUEN SITIO PARA %s.",
	"CE N'EST PAS UN BON ENDROIT POUR %s.",
	"DAS IST KEIN GUTER PLATZ FUER %s.",
	"%s \x91\xbd\xb4\x61\x96\x89\x20\xb8\x77\xad\xa1\x9d\xa1\xac\xe1\x93\x65\x20\xb8\xe2\xd0\x73\xd0\x61\xbb\xa1\x20\xb4\x67\xb7\x65\x88\xf5\x20\x88\x7b\x8a\x85\xb6\x61\x2e",
	"\xb3\x6f\xb8\xcc\xa4\xa3\xac\x4f\xa9\xf1\xb8\x6d\x25\x73\xaa\xba\xa6\x6e\xa6\x61\xa4\xe8\xa1\x43"
};

constexpr I18nText kI18N_youSeeAReflectionOfTheText = {
	"You see a reflection of the %s.",
	"VES UN REFLEJO DE %s.",
	"VOUS VOYEZ UN REFLET REPRESENTANT %s.",
	"DU SIEHST EIN SPIEGELBILD VON %s.",
	"%s \x88\xe1\xb6\x89\xb9\xa1\x88\x62\xb5\x41\x20\xa7\xa1\xc2\xa1\x95\xa1\x20\xb4\x61\xa2\x81\x9c\xe5\x20\xa4\x65\xb7\x77\xb7\xa1\x20\xb4\xf4\xaf\x73\x93\xa1\x94\x61\x2e",
	"\xa7\x41\xac\xdd\xa8\xec\xa4\x46\x25\x73\xaa\xba\xa4\xcf\xae\x67\xac\x4d\xb9\xb3\xa1\x43"
};

constexpr I18nText kI18N_youDontWantToLeaveTheObjectUnderTheBedText = {
	"You don't want to leave the %s under the bed.",
	"NO QUIERES DEJAR %s DEBAJO DE LA CAMA.",
	"POURQUOI LAISSERIEZ-VOUS %s SOUS LE LIT?",
	"DU WILLST %s NICHT UNTER DEM BETT LASSEN.",
	"%s \xc3\xb1\x94\x81\xa3\xbb\xb5\x41\x20\x90\xfd\x93\x65\x88\xf5\xb7\x65\x20\xa5\x69\x9d\xa1\x20\xb9\xbd\xb7\x65\x20\xac\x97\x88\x62\xb7\xa1\x20\xb4\x61\x93\xa5\x88\xf5\x20\x88\x7b\xaf\x73\x93\xa1\x94\x61\x2e",
	"\xa7\x41\xa4\xa3\xad\x6e\xb1\x4e\x25\x73\xaf\x64\xa6\x62\xa7\xc9\xa9\xb3\xa4\x55\xa1\x43"
};

constexpr I18nText kI18N_genResponse0_usingTheObjectOnTheObjectItMustBeYourHeadachesText = {
	"Using the %s on the %s doesn't make any sense, it must be your headaches!",
	"UTILIZAR %s EN %s NO TIENE SENTIDO, TE PRODUCIRA DOLOR DE CABEZA",
	"POURQUOI UTILISER %s SUR %s PRENEZ DONC DE L'ASPIRINE!",
	"GEBRAUCHEN VON %s AUF %s MACHT KEINEN SINN, DAS MUSS AN DEINEN KOPFSCHMERZEN LIEGEN!",
	"%s %s\xb5\x41\x20\xac\x61\xb6\x77\xd0\x61\x93\x65\x88\xf5\xb7\x65\x20\xa0\x69\x95\xa1\x96\x41\xbb\xa1\x20\xb4\x67\xaf\x73\x93\xa1\x94\x61\x2e\x20\x8b\x61\x20\xaf\xb1\xd0\x65\x20\x96\x81\xc9\xb7\x98\x81\xa2\x85\xb5\x41\x20\x8b\x61\x9c\xe1\x93\x65\x88\xf5\xb7\xa1\x20\xb4\x61\x93\xa9\x8c\x61\xb6\x61\x3f",
	"\xa8\xcf\xa5\xce\x25\x73\xa9\xf3\x25\x73\xb2\x40\xb5\x4c\xb7\x4e\xb8\x71\xa1\x41\xa8\xba\xa4\x40\xa9\x77\xa5\xd1\xa9\xf3\xa7\x41\xaa\xba\xc0\x59\xb5\x68\xa1\x49"
};

constexpr I18nText kI18N_genResponse1_theObjectWillDoNothingToTheText = {
	"The %s will do nothing to the %s.",
	"%s NO TIENE NADA QUE VER CON %s.",
	"%s NE FERONT RIEN A %s.",
	"%s HAT NICHTS ZU TUN MIT %s",
	"%s %s\xb5\x41\x20\xac\x61\xb6\x77\xd0\x61\x93\x65\x88\xf5\xb7\x65\x20\xb4\x61\xa2\x81\x9c\xe5\x20\xad\xa1\xb6\x77\xb7\xa1\x20\xb4\xf4\xaf\x73\x93\xa1\x94\x61\x2e",
	"\xa7\xe2\x25\x73\xa5\xce\xa6\x62\x25\x73\xa4\x57\xa8\x53\xa6\xb3\xa5\xce\xa1\x43"
};

constexpr I18nText kI18N_genResponse2_theObjectDoesntHaveAnyEffectOnTheText = {
	"The %s doesn't have any effect on the %s.",
	"%s NO LE CAUSARA EFECTO A %s.",
	"%s N'AURA AUCUN EFFET SUR %s.",
	"%s BEWIRKT NICHTS MIT %s.",
	"%s %s\xb5\x41\x20\xb4\x61\xa2\x81\xb5\x77\xd0\xb7\x95\xa1\x20\xa3\xa1\xc3\xa1\xbb\xa1\xa1\xb5\xd0\x73\x93\xa1\x94\x61\x2e",
	"\xa7\xe2\x25\x73\xa5\xce\xa6\x62\x25\x73\xa4\x57\xb2\x40\xb5\x4c\xae\xc4\xaa\x47\xa1\x43"
};

constexpr I18nText kI18N_genResponse3_theObjectHasNothingToDoWithTheText = {
	"The %s has nothing to do with %s.",
	"%s NO TIENE NADA QUE VER CON %s.",
	"%s N'A AUCUN RAPPORT AVEC %s.",
	"%s HAT NICHTS ZU TUN MIT %s.",
	"%s %s \xb8\xe5\xd1\x61\x20\x89\xc5\x89\x81\x88\x61\x20\xb4\xf4\xaf\x73\x93\xa1\x94\x61\x2e",
	"\x25\x73\xb9\xef\x25\x73\xb0\x5f\xa4\xa3\xa4\x46\xa4\xb0\xbb\xf2\xa7\x40\xa5\xce\xa1\x43"
};

constexpr I18nText kI18N_genResponse4_areYouFeelingAlrightText = {
	"Are you feeling alright?",
	"TE ENCUENTRAS BIEN?",
	"VOUS ETES SUR QUE CA VA BIEN?",
	"FUEHLST DU DICH GUT?",
	"\xbb\xa1\x8b\x71\x20\xb8\xf7\xaf\xa5\xb7\xa1\x20\xb5\xa5\xb8\xe5\xd0\x73\x93\xa1\x8c\x61\x3f\x0a",
	"\xa7\x41\xc1\xd9\xa6\x6e\xa7\x61\xa1\x48"
};

constexpr I18nText kI18N_blankText = {
	"",
	"",
	"",
	"",
	"",
	""
};
} // namespace Darkseed

#endif //DARKSEED_LANGTEXT_H
